# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QCopycanvas
                                 A QGIS plugin
 This tool can be used to copy the map canvas and place it in the clipboard.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-02-10
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Marios S. Kyriakou, KIOS Research and Innovation Center of Excellence (KIOS CoE)
        email                : mariosmsk@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import (QSettings, QTranslator, qVersion, QCoreApplication, Qt)
from PyQt5.QtGui import (QIcon, QImage, QFont, QKeySequence)
from PyQt5.QtWidgets import (QAction, QApplication, QWidget, QToolButton, QMenu)
from qgis.core import Qgis, QgsProject, QgsLayerTreeModel, QgsLayerTreeLayer, QgsLayerTreeGroup, QgsLayerTree
from qgis.gui import *

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
import os.path
try:
    from PIL import Image, ImageDraw, ImageFont
except:
    pass

class QCopycanvas:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'QCopycanvas_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&QCopycanvas')

        self.toolbar = self.iface.addToolBar(u'QCopycanvas')
        self.toolbar.setObjectName(u'QCopycanvas')
        self.toolButton = QToolButton()
        self.toolButton.setMenu(QMenu())
        self.toolButton.setPopupMode(QToolButton.MenuButtonPopup)
        self.toolbar.addWidget(self.toolButton)

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('QCopycanvas', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        shortcut=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if shortcut is not None:
            action.setShortcut(QKeySequence(shortcut))

        #if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
        #    self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/QCopycanvas/icon.png'
        self.mainButton = self.add_action(
            icon_path,
            text=self.tr(u'Copy Map Canvas'),
            shortcut="Ctrl+Space",
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

        self.with_legend_btn = QAction(QIcon(''), "Copy Map Canvas with Legend", self.iface.mainWindow())
        self.with_legend_btn.setText("Copy Map Canvas with Legend")
        self.with_legend_btn.triggered.connect(self.with_legend_btn_run)

        menu = self.toolButton.menu()
        menu.addAction(self.mainButton)
        menu.addAction(self.with_legend_btn)
        self.toolButton.setDefaultAction(self.mainButton)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&QCopycanvas'),
                action)
            self.iface.removeToolBarIcon(action)

    def get_group_layers(self, group):
        for child in group.children():
            if isinstance(child, QgsLayerTreeGroup):
                self.get_group_layers(child)
            elif child.isVisible():
                self.checked_layers.append(child.name())

    def with_legend_btn_run(self):

        project = QgsProject.instance()
        self.checked_layers = []
        root = project.layerTreeRoot()
        for child in root.children():
            if child.isVisible() and isinstance(child, QgsLayerTreeLayer):
                self.checked_layers.append(child.name())

            if child.isVisible() and isinstance(child, QgsLayerTreeGroup):
                self.get_group_layers(child)

        layersToAdd = [layer for layer in project.mapLayers().values() if
                       layer.name() in sorted(self.checked_layers, reverse=False)]
        layers_names_length = [len(layer.name()) for layer in project.mapLayers().values() if
                       layer.name() in sorted(self.checked_layers, reverse=False)]
        maxlen = max(layers_names_length)

        root = QgsLayerTree()
        for layer in layersToAdd:
           root.addLayer(layer)

        model = QgsLayerTreeModel(root)
        lenlen = model.rowCount()
        view = QgsLayerTreeView()
        view.setModel(model)

        view.setFixedHeight(lenlen * 20)
        view.setFixedWidth(maxlen * 10)

        legendIm = QImage(QWidget.grab(view))
        legendpath = self.plugin_dir + "\\legend.png"
        legendIm.save(legendpath)

        legendIm = Image.open(legendpath)
        #legendIm = legendIm.imageData()
        legendWidth, legendHeight = legendIm.size

        main_image = QImage(QWidget.grab(self.iface.mapCanvas()))
        mainpath = self.plugin_dir + "\\main.png"
        main_image.save(mainpath)

        main_image = Image.open(mainpath).convert("RGBA")
        width, height = main_image.size

        d = ImageDraw.Draw(main_image)
        font = ImageFont.truetype("arial.ttf", 16)
        d.text(((width/2.5) + len(project.title()), 10), project.title(), fill='black', font=font)

        if abs(height - width) < 150:
            sq_fit_size = legendWidth
            height = legendHeight
        else:
            sq_fit_size = width - legendWidth

        if width > sq_fit_size and height > sq_fit_size:
            if width > height:
                height = int((sq_fit_size / width) * height)
                width = sq_fit_size
            else:
                width = int((sq_fit_size / height) * width)
                height = sq_fit_size
                main_image = main_image.resize((width, height))

        main_image.paste(legendIm, (width - legendWidth, height - legendHeight))
        finalpath = self.plugin_dir + "\\main.png"
        main_image.save(finalpath)

        QApplication.clipboard().setImage(QImage(finalpath))
        self.iface.messageBar().pushMessage('QCopycanvas', 'Copied map canvas to clipboard with legend', level=Qgis.Success,
                                            duration=2)
    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started

        QApplication.clipboard().setImage(QImage(QWidget.grab(self.iface.mapCanvas())))
        self.iface.messageBar().pushMessage('QCopycanvas', 'Copied map canvas to clipboard', level=Qgis.Success,
                                            duration=2)