"""!
Unittest for slf.volume module
"""

import numpy as np
import os
import unittest

from pyteltools.geom.geometry import Polyline
from pyteltools.slf import Serafin
from pyteltools.slf.volume import VolumeCalculator
from . import TestHeader


HOME = os.path.expanduser('~')
FMT_FLOAT = '{0:.6f}'
VAR_ID = 'U'


class VolumeTestCase(unittest.TestCase):
    def setUp(self):
        self.path = os.path.join(HOME, 'dummpy.slf')

        # create the test Serafin
        test_values = np.array([[0.93675635, -2.4558125, -2.71423527, 1.24962901],
                                [2.16546061, -0.18057363, -0.39374323, 2.7258734],
                                [0.64561824, -1.34825002, 1.90541639, 0.03538929],
                                [1.46941875, -2.26513048, 2.55197906, 2.13117339],
                                [-1.37949687, 2.71902714, -1.0330546, 1.82773433],
                                [1.03057091, -0.21637649, -0.82214216, -1.18631312],
                                [0.82630353, -0.06540417, 1.26278836, 0.05753716],
                                [0.63359244, 1.1448635, 2.07245032, -2.49180871],
                                [-0.01851896, -0.22335092, 2.92214001, 2.00274332],
                                [0.66866341, -0.37513442, 0.58383205, -1.39967382],
                                [2.52950947, -0.55057747, 1.90459775, 1.79967672],
                                [0.78326658, -2.28666747, 2.46506052, 0.77379966],
                                [-0.97687252, 1.9139335, -0.79238172, -2.25499896],
                                [-2.95909051, -1.06862173, -2.30647652, 0.29276768],
                                [1.03608311, 0.76064327, 0.37804945, 0.46144437],
                                [2.00526375, 0.023449, 2.91267964, -2.95186793],
                                [0.69720909, -0.42387534, -2.85544755, -2.22470644],
                                [-0.57729711, 2.13273224, 2.62563128, -1.84145277],
                                [0.61796565, -2.09566578, -0.97022197, -2.26105382],
                                [1.76528317, -0.10687991, 1.63679687, 1.88596746],
                                [-1.44908768, -1.96090935, -1.67910651, 1.56702068],
                                [-1.27269229, -1.02396383, -0.52661424, 2.32236962],
                                [-2.36801095, 2.42495565, -2.29053738, 0.415255],
                                [-2.46169841, -2.38972665, -0.52584325, 0.24340676],
                                [-0.03344924, -0.65679025, -2.18116144, 0.24829135],
                                [1.43239688, 2.00417893, 2.69705007, -2.12932573],
                                [1.37469906, -0.29791054, 1.37391837, 0.23294384],
                                [1.76797037, 0.43578566, -1.79598241, -0.75486298],
                                [1.73661732, -2.35347814, -2.19024393, 1.63727437],
                                [1.87608204, 1.77958726, -1.55296534, -2.92199993]], dtype=np.float64)
        header = TestHeader()
        header.add_variable_from_ID(VAR_ID)
        with Serafin.Write(self.path, 'fr', overwrite=True) as f:
            f.write_header(header)
            for time, vals in enumerate(test_values):
                vals = vals.reshape(1, 4)
                f.write_entire_frame(header, time, vals)

        # create the test polygons
        coords = np.array([[11.42096771, 7.87339648, 3.20590729, 2.86295535,
                            5.18551938, 9.47457804],
                           [6.25671364, -0.69153983, 11.48992183, 5.16910945,
                            4.11304455, 5.1608558],
                           [3.1159781, 7.08005171, 6.77515377, 7.87338014,
                            5.4238916, 1.49341813],
                           [1.37045714, 7.47959328, 4.71295752, 9.78513066,
                            7.07497822, 11.88102676],
                           [1.86568554, 3.7942655, 8.19081358, -1.28744285,
                            0.16955305, 3.68175815],
                           [1.6639868, 0.01768842, 11.13287521, 1.63237544,
                            6.31783879, 11.8248328],
                           [4.13465814, 8.61579313, 4.72725379, 4.52686932,
                            4.44880083, -0.34837116],
                           [10.32511922, 6.93168821, -0.67370794, 2.47492875,
                            4.32771996, -0.96254095],
                           [4.46322661, 3.84433754, 8.69920778, 4.47458586,
                            10.94450627, 2.17421674],
                           [7.07106505, 10.38882295, 9.44523426, 8.56329147,
                            8.18248026, 10.83269414],
                           [2.31264257, -1.97142674, 8.32181997, 2.05090799,
                            0.59012525, 5.9545952],
                           [-1.60169163, 7.34844228, 11.12994393, 10.68383049,
                            10.53809365, 4.03401194],
                           [6.35587925, 8.99341385, 10.4780118, 5.30421444,
                            1.72616816, 11.07985125],
                           [-1.75219598, 11.48380621, 9.50645953, -0.27203797,
                            1.07833667, 4.69030569],
                           [10.72983826, 2.38345748, 11.97805685, -1.1482228,
                            11.70948613, 10.0107903],
                           [7.74200462, 9.74679228, 4.6840044, 4.9592865,
                            5.35513912, 6.5655731],
                           [11.37096847, 11.89701708, -1.94252355, 0.5357545,
                            8.68565988, -0.91168636],
                           [9.2928836, 3.54140075, -1.86760055, 3.68613375,
                            8.44797587, 3.58492468],
                           [9.21124011, 5.61718686, 0.4761447, -0.38495452,
                            7.7991579, 6.36700276],
                           [3.60732119, 10.81322106, 6.90955092, 0.01383031,
                            -0.99885568, -1.10082674],
                           [2.89052125, 7.98854841, 10.04078827, 10.25812434,
                            8.99534648, 6.435721],
                           [0.73206347, 0.06312857, 11.80711648, 8.51012199,
                            11.69033869, 8.30534341],
                           [-0.6753033, 5.1667621, 9.66774498, 2.20052837,
                            8.19661689, 4.14254003],
                           [-1.32405758, 0.40672828, 1.12503513, 7.23921172,
                            3.11903613, 6.60118722],
                           [3.50130738, 5.39703658, 0.49642509, 3.66275043,
                            3.43803927, 11.60049302],
                           [9.00318601, 5.58316908, 10.42232448, 7.8490614,
                            -0.8828602, 8.32860412],
                           [7.66740036, 5.45391401, -0.53368001, 2.94663767,
                            7.61921932, -1.25766693],
                           [1.17969157, 8.72298138, 11.00751148, -1.99633748,
                            3.88149075, 10.67069305],
                           [0.25815905, 4.92600082, 6.79109514, 2.36807772,
                            6.8914301, 8.22753112],
                           [8.0508814, 2.01813477, 6.44897507, 0.61051512,
                            8.81231419, 3.44038985]])

        self.polygons = []
        for coord in coords:
            coord = coord.reshape(3, 2)
            first = coord[0]
            coord = list(map(tuple, list(coord) + [first]))
            self.polygons.append(Polyline(coord))
        self.polynames = ['Polygon %d' % (i + 1) for i in range(len(self.polygons))]

    def tearDown(self):
        # remove the test Serafin
        os.remove(self.path)

    def test_volume_net(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = VolumeCalculator(VolumeCalculator.NET, VAR_ID, None, f,
                                          self.polynames, self.polygons, 1)
            calculator.construct_triangles()
            calculator.construct_weights()
            result = calculator.run(FMT_FLOAT)
        self.assertEqual(result, [['0.0', '0.092088', '0.000000', '0.000000', '0.000000', '-0.359371', '-3.591970', '-0.463912', '-5.929864', '0.000000', '0.000000', '-5.954653', '0.000000', '0.000000', '1.372925', '0.000000', '0.000000', '-9.439319', '0.011981', '-0.427459', '-9.435392', '0.000000', '-0.003377', '0.514093', '0.000000', '0.329062', '0.000000', '-2.565312', '0.000000', '1.277600', '0.000000'], ['1.0', '1.218428', '0.000000', '0.000000', '0.000000', '5.400734', '12.384716', '0.507500', '17.765214', '0.000000', '0.000000', '19.691849', '0.000000', '0.000000', '6.114878', '0.000000', '0.000000', '20.856086', '0.067190', '2.940341', '22.719815', '0.000000', '0.126393', '1.977474', '0.000000', '0.897523', '0.000000', '15.550621', '0.000000', '4.663002', '0.000000'], ['2.0', '0.589999', '0.000000', '0.000000', '0.000000', '1.535705', '5.210586', '0.579576', '5.098278', '0.000000', '0.000000', '6.201419', '0.000000', '0.000000', '1.972024', '0.000000', '0.000000', '4.863228', '0.016339', '0.649073', '5.023474', '0.000000', '0.026815', '0.565876', '0.000000', '0.229276', '0.000000', '5.608270', '0.000000', '1.362367', '0.000000'], ['3.0', '1.368243', '0.000000', '0.000000', '0.000000', '5.639429', '15.398459', '1.105707', '18.169817', '0.000000', '0.000000', '20.493930', '0.000000', '0.000000', '5.221833', '0.000000', '0.000000', '19.052486', '0.054167', '2.873405', '19.812110', '0.000000', '0.121881', '1.569827', '0.000000', '0.593104', '0.000000', '17.410446', '0.000000', '3.678430', '0.000000'], ['4.0', '-0.412808', '0.000000', '0.000000', '0.000000', '2.130115', '4.407969', '-0.203701', '8.972456', '0.000000', '0.000000', '8.043625', '0.000000', '0.000000', '-1.737051', '0.000000', '0.000000', '12.261883', '-0.002000', '1.677220', '12.192309', '0.000000', '0.064363', '-0.538573', '0.000000', '-0.413534', '0.000000', '4.513491', '0.000000', '-1.463567', '0.000000'], ['5.0', '0.058145', '0.000000', '0.000000', '0.000000', '-1.963846', '-5.421368', '-0.231728', '-7.013583', '0.000000', '0.000000', '-6.823992', '0.000000', '0.000000', '0.988606', '0.000000', '0.000000', '-7.704800', '0.002096', '-1.259682', '-7.149670', '0.000000', '-0.049679', '0.370264', '0.000000', '0.340631', '0.000000', '-5.168406', '0.000000', '1.005848', '0.000000'], ['6.0', '0.550693', '0.000000', '0.000000', '0.000000', '1.605083', '4.931735', '0.446556', '6.397066', '0.000000', '0.000000', '7.412804', '0.000000', '0.000000', '2.255068', '0.000000', '0.000000', '7.780292', '0.021019', '0.802846', '8.439974', '0.000000', '0.031233', '0.691266', '0.000000', '0.319216', '0.000000', '5.672928', '0.000000', '1.658523', '0.000000'], ['7.0', '0.197194', '0.000000', '0.000000', '0.000000', '-1.567218', '-1.701367', '0.333307', '-1.959999', '0.000000', '0.000000', '-1.323190', '0.000000', '0.000000', '0.638366', '0.000000', '0.000000', '-0.179246', '-0.003024', '-0.970139', '0.452773', '0.000000', '-0.049033', '0.177677', '0.000000', '0.192741', '0.000000', '-2.198003', '0.000000', '0.533286', '0.000000'], ['8.0', '0.836718', '0.000000', '0.000000', '0.000000', '5.446512', '15.512176', '1.013285', '19.811410', '0.000000', '0.000000', '21.207498', '0.000000', '0.000000', '2.449624', '0.000000', '0.000000', '22.490857', '0.032967', '3.085716', '22.737540', '0.000000', '0.123032', '0.664080', '0.000000', '0.070708', '0.000000', '16.490280', '0.000000', '1.428706', '0.000000'], ['9.0', '0.129815', '0.000000', '0.000000', '0.000000', '-1.477632', '-3.009803', '0.065635', '-4.621265', '0.000000', '0.000000', '-4.246621', '0.000000', '0.000000', '0.648447', '0.000000', '0.000000', '-5.152726', '-0.001414', '-0.997537', '-4.888599', '0.000000', '-0.041941', '0.208895', '0.000000', '0.202732', '0.000000', '-3.176716', '0.000000', '0.597328', '0.000000'], ['10.0', '1.609696', '0.000000', '0.000000', '0.000000', '5.848193', '15.487863', '1.044967', '20.690724', '0.000000', '0.000000', '23.463546', '0.000000', '0.000000', '7.142679', '0.000000', '0.000000', '24.316820', '0.072529', '3.049046', '26.332179', '0.000000', '0.126459', '2.240539', '0.000000', '1.011136', '0.000000', '18.482922', '0.000000', '5.323039', '0.000000'], ['11.0', '0.863457', '0.000000', '0.000000', '0.000000', '2.941523', '8.939612', '0.827797', '9.057666', '0.000000', '0.000000', '10.597447', '0.000000', '0.000000', '2.861254', '0.000000', '0.000000', '8.405413', '0.026419', '1.361468', '8.489436', '0.000000', '0.058197', '0.819089', '0.000000', '0.289979', '0.000000', '9.690487', '0.000000', '1.936465', '0.000000'], ['12.0', '-0.889331', '0.000000', '0.000000', '0.000000', '-4.241909', '-10.536150', '-0.595511', '-12.500047', '0.000000', '0.000000', '-13.907930', '0.000000', '0.000000', '-3.611617', '0.000000', '0.000000', '-12.512413', '-0.040018', '-2.198251', '-12.951277', '0.000000', '-0.096619', '-1.107648', '0.000000', '-0.408034', '0.000000', '-12.216965', '0.000000', '-2.573557', '0.000000'], ['13.0', '-1.500685', '0.000000', '0.000000', '0.000000', '-3.623521', '-10.767535', '-0.963572', '-15.779886', '0.000000', '0.000000', '-18.560550', '0.000000', '0.000000', '-7.002588', '0.000000', '0.000000', '-20.965149', '-0.064399', '-1.815067', '-23.580149', '0.000000', '-0.069036', '-2.222222', '0.000000', '-1.139276', '0.000000', '-13.336501', '0.000000', '-5.379402', '0.000000'], ['14.0', '0.543194', '0.000000', '0.000000', '0.000000', '1.962099', '5.062669', '0.296278', '7.913858', '0.000000', '0.000000', '8.857358', '0.000000', '0.000000', '2.698421', '0.000000', '0.000000', '10.391583', '0.027971', '1.092497', '11.467770', '0.000000', '0.043336', '0.869231', '0.000000', '0.422094', '0.000000', '6.291376', '0.000000', '2.071357', '0.000000'], ['15.0', '0.799056', '0.000000', '0.000000', '0.000000', '-1.127882', '-0.030451', '0.666137', '-0.794211', '0.000000', '0.000000', '0.983961', '0.000000', '0.000000', '3.381261', '0.000000', '0.000000', '0.738908', '0.017540', '-0.989391', '2.054362', '0.000000', '-0.047891', '1.044443', '0.000000', '0.677492', '0.000000', '0.265816', '0.000000', '2.683461', '0.000000'], ['16.0', '-0.599267', '0.000000', '0.000000', '0.000000', '-5.527583', '-15.718596', '-0.940903', '-20.592013', '0.000000', '0.000000', '-21.541541', '0.000000', '0.000000', '-1.241485', '0.000000', '0.000000', '-23.727806', '-0.024482', '-3.266346', '-23.676694', '0.000000', '-0.128510', '-0.271929', '0.000000', '0.165451', '0.000000', '-16.319432', '0.000000', '-0.445508', '0.000000'], ['17.0', '-0.066429', '0.000000', '0.000000', '0.000000', '-0.536702', '1.354360', '0.411533', '2.654897', '0.000000', '0.000000', '2.776462', '0.000000', '0.000000', '-1.095707', '0.000000', '0.000000', '5.571840', '-0.013071', '-0.164581', '5.675549', '0.000000', '-0.020143', '-0.413579', '0.000000', '-0.219650', '0.000000', '0.294553', '0.000000', '-0.984844', '0.000000'], ['18.0', '-0.304925', '0.000000', '0.000000', '0.000000', '-4.698296', '-12.229001', '-0.471530', '-18.414790', '0.000000', '0.000000', '-18.785892', '0.000000', '0.000000', '-0.866388', '0.000000', '0.000000', '-22.915470', '-0.023348', '-2.980099', '-23.358011', '0.000000', '-0.116359', '-0.230023', '0.000000', '0.113050', '0.000000', '-13.134347', '0.000000', '-0.352515', '0.000000'], ['19.0', '1.281452', '0.000000', '0.000000', '0.000000', '5.429049', '14.306440', '0.890094', '19.413105', '0.000000', '0.000000', '21.583170', '0.000000', '0.000000', '5.561093', '0.000000', '0.000000', '22.927114', '0.059544', '2.934346', '24.496605', '0.000000', '0.120575', '1.733930', '0.000000', '0.733226', '0.000000', '16.751996', '0.000000', '4.077289', '0.000000'], ['20.0', '-0.596209', '0.000000', '0.000000', '0.000000', '-0.403144', '-2.687492', '-0.454636', '-5.379312', '0.000000', '0.000000', '-6.638612', '0.000000', '0.000000', '-2.880812', '0.000000', '0.000000', '-9.375555', '-0.022779', '-0.184211', '-10.954290', '0.000000', '0.001438', '-0.919490', '0.000000', '-0.543234', '0.000000', '-3.458131', '0.000000', '-2.278598', '0.000000'], ['21.0', '-0.200887', '0.000000', '0.000000', '0.000000', '2.149134', '4.498759', '-0.026958', '5.024331', '0.000000', '0.000000', '4.437669', '0.000000', '0.000000', '-1.433313', '0.000000', '0.000000', '3.679891', '-0.003704', '1.338206', '2.641136', '0.000000', '0.059894', '-0.491909', '0.000000', '-0.427310', '0.000000', '4.402701', '0.000000', '-1.326407', '0.000000'], ['22.0', '-1.249116', '0.000000', '0.000000', '0.000000', '-1.870331', '-6.223462', '-0.838864', '-5.722582', '0.000000', '0.000000', '-8.056844', '0.000000', '0.000000', '-5.268663', '0.000000', '0.000000', '-5.336728', '-0.041687', '-0.549572', '-6.442841', '0.000000', '-0.025798', '-1.631121', '0.000000', '-0.844556', '0.000000', '-7.719156', '0.000000', '-4.006239', '0.000000'], ['23.0', '-1.003904', '0.000000', '0.000000', '0.000000', '-2.409419', '-6.427160', '-0.446215', '-11.841477', '0.000000', '0.000000', '-13.681736', '0.000000', '0.000000', '-5.441338', '0.000000', '0.000000', '-17.475985', '-0.052158', '-1.332201', '-20.048633', '0.000000', '-0.049058', '-1.784944', '0.000000', '-0.968761', '0.000000', '-8.764630', '0.000000', '-4.323367', '0.000000'], ['24.0', '-0.354859', '0.000000', '0.000000', '0.000000', '-1.571338', '-5.770265', '-0.555659', '-7.626980', '0.000000', '0.000000', '-8.345034', '0.000000', '0.000000', '-0.868868', '0.000000', '0.000000', '-9.717466', '-0.009119', '-0.899348', '-9.995856', '0.000000', '-0.030403', '-0.212713', '0.000000', '-0.030343', '0.000000', '-5.799340', '0.000000', '-0.476058', '0.000000'], ['25.0', '0.675584', '0.000000', '0.000000', '0.000000', '0.311317', '3.399805', '0.652863', '5.920508', '0.000000', '0.000000', '7.400338', '0.000000', '0.000000', '2.891357', '0.000000', '0.000000', '10.201566', '0.020836', '0.098940', '11.758549', '0.000000', '-0.007729', '0.893976', '0.000000', '0.525685', '0.000000', '3.897669', '0.000000', '2.230256', '0.000000'], ['26.0', '0.811212', '0.000000', '0.000000', '0.000000', '2.251186', '6.532186', '0.566094', '8.480969', '0.000000', '0.000000', '9.950482', '0.000000', '0.000000', '3.524959', '0.000000', '0.000000', '10.172517', '0.032811', '1.098837', '11.200491', '0.000000', '0.044393', '1.099159', '0.000000', '0.527495', '0.000000', '7.796359', '0.000000', '2.645323', '0.000000'], ['27.0', '0.243552', '0.000000', '0.000000', '0.000000', '-1.350807', '-4.787912', '-0.335680', '-4.780680', '0.000000', '0.000000', '-4.322609', '0.000000', '0.000000', '2.486483', '0.000000', '0.000000', '-4.216275', '0.018190', '-0.889275', '-2.898083', '0.000000', '-0.033146', '0.895512', '0.000000', '0.639144', '0.000000', '-3.609278', '0.000000', '2.270597', '0.000000'], ['28.0', '0.557380', '0.000000', '0.000000', '0.000000', '1.228537', '0.646746', '-0.174077', '0.025221', '0.000000', '0.000000', '0.813701', '0.000000', '0.000000', '3.512692', '0.000000', '0.000000', '-2.106301', '0.033450', '0.409200', '-1.404773', '0.000000', '0.030635', '1.191172', '0.000000', '0.647861', '0.000000', '2.570015', '0.000000', '2.892898', '0.000000'], ['29.0', '-0.029982', '0.000000', '0.000000', '0.000000', '-4.120393', '-10.970081', '-0.511919', '-12.017387', '0.000000', '0.000000', '-11.844404', '0.000000', '0.000000', '1.532974', '0.000000', '0.000000', '-10.699114', '0.001827', '-2.426472', '-9.121184', '0.000000', '-0.102291', '0.615548', '0.000000', '0.637516', '0.000000', '-10.571201', '0.000000', '1.701030', '0.000000']])

    def test_volume_strict(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = VolumeCalculator(VolumeCalculator.NET_STRICT, VAR_ID, None, f,
                                          self.polynames, self.polygons, 1)
            calculator.construct_triangles()
            calculator.construct_weights()
            result = calculator.run(FMT_FLOAT)
        self.assertEqual(result, [['0.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.435392', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['1.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '22.719815', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['2.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.023474', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['3.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '19.812110', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['4.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '12.192309', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['5.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-7.149670', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['6.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '8.439974', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['7.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.452773', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['8.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '22.737540', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['9.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.888599', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['10.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '26.332179', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['11.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '8.489436', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['12.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-12.951277', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['13.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-23.580149', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['14.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '11.467770', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['15.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.054362', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['16.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-23.676694', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['17.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.675549', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['18.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-23.358011', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['19.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '24.496605', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['20.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-10.954290', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['21.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.641136', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['22.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-6.442841', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['23.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-20.048633', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['24.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.995856', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['25.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '11.758549', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['26.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '11.200491', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['27.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-2.898083', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['28.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.404773', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000'], ['29.0', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.121184', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000']])

    def test_volume_positive(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = VolumeCalculator(VolumeCalculator.POSITIVE, VAR_ID, None, f,
                                          self.polynames, self.polygons, 1)
            calculator.construct_triangles()
            calculator.construct_weights()
            result = calculator.run(FMT_FLOAT)
        self.assertEqual(result, [['0.0', '0.092088', '0.142337', '-0.050250', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.359371', '1.061117', '-1.420488', '-3.591970', '1.877503', '-5.469473', '-0.463912', '-0.000000', '-0.463912', '-5.929864', '2.547872', '-8.477736', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.954653', '3.208758', '-9.163412', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.372925', '1.605798', '-0.232872', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.439319', '3.192789', '-12.632108', '0.011981', '0.014505', '-0.002524', '-0.427459', '0.383078', '-0.810537', '-9.435392', '3.790943', '-13.226335', '0.000000', '0.000000', '0.000000', '-0.003377', '0.019672', '-0.023049', '0.514093', '0.525534', '-0.011441', '0.000000', '0.000000', '0.000000', '0.329062', '0.329062', '0.000000', '0.000000', '0.000000', '0.000000', '-2.565312', '2.597370', '-5.162682', '0.000000', '0.000000', '0.000000', '1.277600', '1.331182', '-0.053583', '0.000000', '0.000000', '0.000000'], ['1.0', '1.218428', '1.218428', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.400734', '5.411802', '-0.011068', '12.384716', '12.389593', '-0.004876', '0.507500', '0.508987', '-0.001486', '17.765214', '17.844357', '-0.079143', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '19.691849', '19.810420', '-0.118571', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '6.114878', '6.114878', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '20.856086', '21.013056', '-0.156970', '0.067190', '0.067190', '0.000000', '2.940341', '2.942883', '-0.002542', '22.719815', '22.901415', '-0.181600', '0.000000', '0.000000', '0.000000', '0.126393', '0.126395', '-0.000002', '1.977474', '1.977474', '0.000000', '0.000000', '0.000000', '0.000000', '0.897523', '0.897523', '0.000000', '0.000000', '0.000000', '0.000000', '15.550621', '15.605151', '-0.054530', '0.000000', '0.000000', '0.000000', '4.663002', '4.663002', '0.000000', '0.000000', '0.000000', '0.000000'], ['2.0', '0.589999', '5.172848', '-4.582849', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.535705', '5.214803', '-3.679098', '5.210586', '6.232261', '-1.021675', '0.579576', '5.253684', '-4.674108', '5.098278', '7.224884', '-2.126605', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '6.201419', '7.583796', '-1.382377', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.972024', '5.532608', '-3.560584', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '4.863228', '7.657433', '-2.794205', '0.016339', '5.174488', '-5.158149', '0.649073', '5.202324', '-4.553251', '5.023474', '7.888999', '-2.865525', '0.000000', '0.000000', '0.000000', '0.026815', '5.173318', '-5.146503', '0.565876', '5.290321', '-4.724445', '0.000000', '0.000000', '0.000000', '0.229276', '5.301917', '-5.072641', '0.000000', '0.000000', '0.000000', '5.608270', '6.739825', '-1.131555', '0.000000', '0.000000', '0.000000', '1.362367', '5.498246', '-4.135879', '0.000000', '0.000000', '0.000000'], ['3.0', '1.368243', '12.305142', '-10.936899', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.639429', '13.719700', '-8.080271', '15.398459', '17.255269', '-1.856810', '1.105707', '12.461583', '-11.355876', '18.169817', '20.511837', '-2.342020', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '20.493930', '21.686366', '-1.192437', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.221833', '13.700970', '-8.479137', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '19.052486', '21.588934', '-2.536448', '0.054167', '12.320654', '-12.266487', '2.873405', '13.106544', '-10.233139', '19.812110', '22.325429', '-2.513319', '0.000000', '0.000000', '0.000000', '0.121881', '12.335978', '-12.214097', '1.569827', '12.780886', '-11.211059', '0.000000', '0.000000', '0.000000', '0.593104', '12.657229', '-12.064125', '0.000000', '0.000000', '0.000000', '17.410446', '18.999772', '-1.589326', '0.000000', '0.000000', '0.000000', '3.678430', '13.476897', '-9.798467', '0.000000', '0.000000', '0.000000'], ['4.0', '-0.412808', '0.031515', '-0.444323', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.130115', '2.444996', '-0.314881', '4.407969', '6.237762', '-1.829793', '-0.203701', '0.035851', '-0.239552', '8.972456', '10.727495', '-1.755039', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '8.043625', '10.240265', '-2.196640', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.737051', '0.095989', '-1.833040', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '12.261883', '14.713105', '-2.451222', '-0.002000', '0.007513', '-0.009513', '1.677220', '1.985803', '-0.308583', '12.192309', '15.297710', '-3.105402', '0.000000', '0.000000', '0.000000', '0.064363', '0.077510', '-0.013147', '-0.538573', '0.001459', '-0.540031', '0.000000', '0.000000', '0.000000', '-0.413534', '0.000000', '-0.413534', '0.000000', '0.000000', '0.000000', '4.513491', '6.504863', '-1.991373', '0.000000', '0.000000', '0.000000', '-1.463567', '0.022244', '-1.485811', '0.000000', '0.000000', '0.000000'], ['5.0', '0.058145', '0.110558', '-0.052413', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.963846', '0.123783', '-2.087629', '-5.421368', '0.200863', '-5.622232', '-0.231728', '0.004709', '-0.236437', '-7.013583', '0.288383', '-7.301966', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-6.823992', '0.732099', '-7.556091', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.988606', '1.059585', '-0.070978', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-7.704800', '0.691323', '-8.396123', '0.002096', '0.004266', '-0.002170', '-1.259682', '0.004070', '-1.263752', '-7.149670', '1.324886', '-8.474556', '0.000000', '0.000000', '0.000000', '-0.049679', '0.000160', '-0.049839', '0.370264', '0.370269', '-0.000005', '0.000000', '0.000000', '0.000000', '0.340631', '0.340631', '-0.000000', '0.000000', '0.000000', '0.000000', '-5.168406', '0.319470', '-5.487876', '0.000000', '0.000000', '0.000000', '1.005848', '1.011660', '-0.005813', '0.000000', '0.000000', '0.000000'], ['6.0', '0.550693', '4.293258', '-3.742565', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.605083', '4.714946', '-3.109863', '4.931735', '5.677887', '-0.746152', '0.446556', '4.354641', '-3.908084', '6.397066', '7.212963', '-0.815898', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '7.412804', '7.887279', '-0.474474', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.255068', '4.956770', '-2.701702', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '7.780292', '7.994745', '-0.214452', '0.021019', '4.299959', '-4.278940', '0.802846', '4.429857', '-3.627011', '8.439974', '8.448505', '-0.008531', '0.000000', '0.000000', '0.000000', '0.031233', '4.296543', '-4.265310', '0.691266', '4.522198', '-3.830932', '0.000000', '0.000000', '0.000000', '0.319216', '4.491070', '-4.171854', '0.000000', '0.000000', '0.000000', '5.672928', '6.298798', '-0.625870', '0.000000', '0.000000', '0.000000', '1.658523', '4.871553', '-3.213030', '0.000000', '0.000000', '0.000000'], ['7.0', '0.197194', '0.303061', '-0.105868', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.567218', '0.873294', '-2.440512', '-1.701367', '2.952306', '-4.653672', '0.333307', '0.342085', '-0.008778', '-1.959999', '4.527399', '-6.487398', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.323190', '5.292697', '-6.615887', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.638366', '1.118003', '-0.479637', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.179246', '6.565664', '-6.744910', '-0.003024', '0.009036', '-0.012059', '-0.970139', '0.431905', '-1.402044', '0.452773', '7.208672', '-6.755900', '0.000000', '0.000000', '0.000000', '-0.049033', '0.014880', '-0.063914', '0.177677', '0.302678', '-0.125001', '0.000000', '0.000000', '0.000000', '0.192741', '0.192805', '-0.000065', '0.000000', '0.000000', '0.000000', '-2.198003', '3.529406', '-5.727409', '0.000000', '0.000000', '0.000000', '0.533286', '0.792398', '-0.259112', '0.000000', '0.000000', '0.000000'], ['8.0', '0.836718', '0.836718', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.446512', '5.451527', '-0.005015', '15.512176', '15.512176', '0.000000', '1.013285', '1.013285', '0.000000', '19.811410', '19.829649', '-0.018240', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '21.207498', '21.218126', '-0.010629', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.449624', '2.452211', '-0.002587', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '22.490857', '22.532363', '-0.041507', '0.032967', '0.033020', '-0.000053', '3.085716', '3.085716', '0.000000', '22.737540', '22.789565', '-0.052025', '0.000000', '0.000000', '0.000000', '0.123032', '0.123032', '0.000000', '0.664080', '0.665056', '-0.000976', '0.000000', '0.000000', '0.000000', '0.070708', '0.071697', '-0.000990', '0.000000', '0.000000', '0.000000', '16.490280', '16.501801', '-0.011521', '0.000000', '0.000000', '0.000000', '1.428706', '1.431003', '-0.002298', '0.000000', '0.000000', '0.000000'], ['9.0', '0.129815', '0.182198', '-0.052383', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.477632', '0.161420', '-1.639053', '-3.009803', '0.758349', '-3.768152', '0.065635', '0.106518', '-0.040883', '-4.621265', '0.839333', '-5.460598', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.246621', '1.101338', '-5.347959', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.648447', '0.812619', '-0.164172', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.152726', '1.184134', '-6.336860', '-0.001414', '0.004304', '-0.005718', '-0.997537', '0.126847', '-1.124385', '-4.888599', '1.528615', '-6.417213', '0.000000', '0.000000', '0.000000', '-0.041941', '0.005391', '-0.047332', '0.208895', '0.230991', '-0.022095', '0.000000', '0.000000', '0.000000', '0.202732', '0.202732', '-0.000000', '0.000000', '0.000000', '0.000000', '-3.176716', '0.905976', '-4.082691', '0.000000', '0.000000', '0.000000', '0.597328', '0.663199', '-0.065871', '0.000000', '0.000000', '0.000000'], ['10.0', '1.609696', '12.467568', '-10.857872', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.848193', '14.708047', '-8.859854', '15.487863', '17.618966', '-2.131103', '1.044967', '12.599124', '-11.554157', '20.690724', '23.013178', '-2.322454', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '23.463546', '24.827237', '-1.363691', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '7.142679', '14.836615', '-7.693935', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '24.316820', '25.022731', '-0.705910', '0.072529', '12.495039', '-12.422510', '3.049046', '13.394687', '-10.345641', '26.332179', '26.436139', '-0.103959', '0.000000', '0.000000', '0.000000', '0.126459', '12.502631', '-12.376172', '2.240539', '13.287291', '-11.046752', '0.000000', '0.000000', '0.000000', '1.011136', '13.100064', '-12.088928', '0.000000', '0.000000', '0.000000', '18.482922', '20.466797', '-1.983875', '0.000000', '0.000000', '0.000000', '5.323039', '14.484317', '-9.161278', '0.000000', '0.000000', '0.000000'], ['11.0', '0.863457', '8.044254', '-7.180797', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.941523', '8.397103', '-5.455580', '8.939612', '10.315739', '-1.376127', '0.827797', '8.162454', '-7.334657', '9.057666', '11.781146', '-2.723480', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '10.597447', '12.282627', '-1.685180', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.861254', '8.575824', '-5.714570', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '8.405413', '12.393707', '-3.988294', '0.026419', '8.048577', '-8.022158', '1.361468', '8.276080', '-6.914613', '8.489436', '12.679009', '-4.189573', '0.000000', '0.000000', '0.000000', '0.058197', '8.052246', '-7.994049', '0.819089', '8.217328', '-7.398238', '0.000000', '0.000000', '0.000000', '0.289979', '8.203355', '-7.913376', '0.000000', '0.000000', '0.000000', '9.690487', '11.162862', '-1.472374', '0.000000', '0.000000', '0.000000', '1.936465', '8.503474', '-6.567009', '0.000000', '0.000000', '0.000000'], ['12.0', '-0.889331', '0.000000', '-0.889331', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.241909', '0.035351', '-4.277260', '-10.536150', '0.209333', '-10.745483', '-0.595511', '0.000000', '-0.595511', '-12.500047', '0.532327', '-13.032374', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-13.907930', '0.441134', '-14.349064', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-3.611617', '0.000000', '-3.611617', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-12.512413', '2.030246', '-14.542659', '-0.040018', '0.000092', '-0.040109', '-2.198251', '0.130012', '-2.328263', '-12.951277', '2.406318', '-15.357595', '0.000000', '0.000000', '0.000000', '-0.096619', '0.002553', '-0.099173', '-1.107648', '0.000000', '-1.107648', '0.000000', '0.000000', '0.000000', '-0.408034', '0.000000', '-0.408034', '0.000000', '0.000000', '0.000000', '-12.216965', '0.173351', '-12.390316', '0.000000', '0.000000', '0.000000', '-2.573557', '0.000000', '-2.573557', '0.000000', '0.000000', '0.000000'], ['13.0', '-1.500685', '0.000000', '-1.500685', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-3.623521', '0.014403', '-3.637925', '-10.767535', '0.031457', '-10.798992', '-0.963572', '0.000000', '-0.963572', '-15.779886', '0.031457', '-15.811343', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-18.560550', '0.031457', '-18.592007', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-7.002588', '0.000000', '-7.002588', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-20.965149', '0.031457', '-20.996606', '-0.064399', '0.000000', '-0.064399', '-1.815067', '0.010229', '-1.825296', '-23.580149', '0.031457', '-23.611606', '0.000000', '0.000000', '0.000000', '-0.069036', '0.000737', '-0.069772', '-2.222222', '0.000000', '-2.222222', '0.000000', '0.000000', '0.000000', '-1.139276', '0.000000', '-1.139276', '0.000000', '0.000000', '0.000000', '-13.336501', '0.031457', '-13.367958', '0.000000', '0.000000', '0.000000', '-5.379402', '0.000000', '-5.379402', '0.000000', '0.000000', '0.000000'], ['14.0', '0.543194', '3.751154', '-3.207960', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.962099', '11.467770', '-9.505670', '5.062669', '11.467770', '-6.405101', '0.296278', '6.951428', '-6.655150', '7.913858', '11.467770', '-3.553912', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '8.857358', '11.467770', '-2.610412', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.698421', '8.267495', '-5.569074', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '10.391583', '11.467770', '-1.076187', '0.027971', '8.267495', '-8.239524', '1.092497', '6.951428', '-5.858931', '11.467770', '11.467770', '0.000000', '0.000000', '0.000000', '0.000000', '0.043336', '6.951428', '-6.908092', '0.869231', '8.267495', '-7.398264', '0.000000', '0.000000', '0.000000', '0.422094', '8.267495', '-7.845402', '0.000000', '0.000000', '0.000000', '6.291376', '11.467770', '-5.176393', '0.000000', '0.000000', '0.000000', '2.071357', '8.267495', '-6.196139', '0.000000', '0.000000', '0.000000'], ['15.0', '0.799056', '0.840826', '-0.041771', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.127882', '1.418524', '-2.546406', '-0.030451', '5.112425', '-5.142876', '0.666137', '0.667259', '-0.001122', '-0.794211', '6.784475', '-7.578685', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.983961', '8.353248', '-7.369287', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '3.381261', '3.506852', '-0.125591', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.738908', '8.937178', '-8.198270', '0.017540', '0.023511', '-0.005971', '-0.989391', '0.750637', '-1.740029', '2.054362', '10.259901', '-8.205539', '0.000000', '0.000000', '0.000000', '-0.047891', '0.030942', '-0.078833', '1.044443', '1.048799', '-0.004355', '0.000000', '0.000000', '0.000000', '0.677492', '0.677492', '-0.000000', '0.000000', '0.000000', '0.000000', '0.265816', '6.241919', '-5.976103', '0.000000', '0.000000', '0.000000', '2.683461', '2.716506', '-0.033044', '0.000000', '0.000000', '0.000000'], ['16.0', '-0.599267', '0.000000', '-0.599267', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.527583', '0.015501', '-5.543084', '-15.718596', '0.000003', '-15.718599', '-0.940903', '0.000000', '-0.940903', '-20.592013', '0.009574', '-20.601587', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-21.541541', '0.100931', '-21.642471', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.241485', '0.241024', '-1.482509', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-23.727806', '0.029756', '-23.757562', '-0.024482', '0.000359', '-0.024841', '-3.266346', '0.000000', '-3.266346', '-23.676694', '0.272223', '-23.948917', '0.000000', '0.000000', '0.000000', '-0.128510', '0.000000', '-0.128510', '-0.271929', '0.040115', '-0.312044', '0.000000', '0.000000', '0.000000', '0.165451', '0.165451', '0.000000', '0.000000', '0.000000', '0.000000', '-16.319432', '0.015736', '-16.335168', '0.000000', '0.000000', '0.000000', '-0.445508', '0.235488', '-0.680995', '0.000000', '0.000000', '0.000000'], ['17.0', '-0.066429', '0.091531', '-0.157960', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.536702', '1.189438', '-1.726140', '1.354360', '4.388491', '-3.034131', '0.411533', '0.411533', '-0.000000', '2.654897', '6.848621', '-4.193724', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.776462', '7.568736', '-4.792273', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.095707', '0.303535', '-1.399241', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.571840', '10.345240', '-4.773400', '-0.013071', '0.003583', '-0.016654', '-0.164581', '0.619081', '-0.783661', '5.675549', '10.868939', '-5.193390', '0.000000', '0.000000', '0.000000', '-0.020143', '0.017675', '-0.037818', '-0.413579', '0.036486', '-0.450066', '0.000000', '0.000000', '0.000000', '-0.219650', '0.000000', '-0.219650', '0.000000', '0.000000', '0.000000', '0.294553', '4.415474', '-4.120921', '0.000000', '0.000000', '0.000000', '-0.984844', '0.100619', '-1.085463', '0.000000', '0.000000', '0.000000'], ['18.0', '-0.304925', '0.006906', '-0.311831', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.698296', '0.000000', '-4.698296', '-12.229001', '0.016385', '-12.245386', '-0.471530', '0.000000', '-0.471530', '-18.414790', '0.003081', '-18.417871', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-18.785892', '0.006502', '-18.792394', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.866388', '0.163636', '-1.030023', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-22.915470', '0.034078', '-22.949548', '-0.023348', '0.000000', '-0.023348', '-2.980099', '0.000000', '-2.980099', '-23.358011', '0.163636', '-23.521647', '0.000000', '0.000000', '0.000000', '-0.116359', '0.000000', '-0.116359', '-0.230023', '0.013600', '-0.243624', '0.000000', '0.000000', '0.000000', '0.113050', '0.113080', '-0.000030', '0.000000', '0.000000', '0.000000', '-13.134347', '0.002883', '-13.137230', '0.000000', '0.000000', '0.000000', '-0.352515', '0.163636', '-0.516151', '0.000000', '0.000000', '0.000000'], ['19.0', '1.281452', '10.576095', '-9.294643', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.429049', '12.655913', '-7.226865', '14.306440', '15.977994', '-1.671553', '0.890094', '10.700705', '-9.810610', '19.413105', '21.200360', '-1.787254', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '21.583170', '22.600041', '-1.016871', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '5.561093', '12.410352', '-6.849259', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '22.927114', '23.409455', '-0.482341', '0.059544', '10.599198', '-10.539654', '2.934346', '11.642090', '-8.707744', '24.496605', '24.497962', '-0.001357', '0.000000', '0.000000', '0.000000', '0.120575', '10.616222', '-10.495648', '1.733930', '11.213605', '-9.479675', '0.000000', '0.000000', '0.000000', '0.733226', '11.036013', '-10.302787', '0.000000', '0.000000', '0.000000', '16.751996', '18.248629', '-1.496633', '0.000000', '0.000000', '0.000000', '4.077289', '12.118934', '-8.041645', '0.000000', '0.000000', '0.000000'], ['20.0', '-0.596209', '0.011140', '-0.607349', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.403144', '1.041958', '-1.445102', '-2.687492', '1.916659', '-4.604151', '-0.454636', '0.000045', '-0.454682', '-5.379312', '2.181102', '-7.560414', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-6.638612', '2.181282', '-8.819894', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-2.880812', '0.017889', '-2.898700', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.375555', '2.181282', '-11.556837', '-0.022779', '0.001075', '-0.023854', '-0.184211', '0.613479', '-0.797690', '-10.954290', '2.181282', '-13.135572', '0.000000', '0.000000', '0.000000', '0.001438', '0.028801', '-0.027364', '-0.919490', '0.000000', '-0.919490', '0.000000', '0.000000', '0.000000', '-0.543234', '0.000000', '-0.543234', '0.000000', '0.000000', '0.000000', '-3.458131', '2.085520', '-5.543651', '0.000000', '0.000000', '0.000000', '-2.278598', '0.001398', '-2.279996', '0.000000', '0.000000', '0.000000'], ['21.0', '-0.200887', '0.088263', '-0.289150', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.149134', '2.575134', '-0.426000', '4.498759', '5.628420', '-1.129662', '-0.026958', '0.106182', '-0.133140', '5.024331', '7.040643', '-2.016312', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '4.437669', '7.024033', '-2.586364', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.433313', '0.206858', '-1.640171', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '3.679891', '7.155796', '-3.475905', '-0.003704', '0.006964', '-0.010668', '1.338206', '1.555773', '-0.217567', '2.641136', '7.155796', '-4.514660', '0.000000', '0.000000', '0.000000', '0.059894', '0.067525', '-0.007631', '-0.491909', '0.018052', '-0.509962', '0.000000', '0.000000', '0.000000', '-0.427310', '0.000000', '-0.427310', '0.000000', '0.000000', '0.000000', '4.402701', '6.028357', '-1.625656', '0.000000', '0.000000', '0.000000', '-1.326407', '0.069527', '-1.395934', '0.000000', '0.000000', '0.000000'], ['22.0', '-1.249116', '0.000000', '-1.249116', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.870331', '0.130250', '-2.000581', '-6.223462', '1.230007', '-7.453469', '-0.838864', '-0.000000', '-0.838864', '-5.722582', '2.757624', '-8.480206', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-8.056844', '2.363914', '-10.420758', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.268663', '-0.000000', '-5.268663', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.336728', '5.462178', '-10.798906', '-0.041687', '-0.000000', '-0.041687', '-0.549572', '0.666545', '-1.216117', '-6.442841', '5.937275', '-12.380115', '0.000000', '0.000000', '0.000000', '-0.025798', '0.025439', '-0.051237', '-1.631121', '-0.000000', '-1.631121', '0.000000', '0.000000', '0.000000', '-0.844556', '-0.000000', '-0.844556', '0.000000', '0.000000', '0.000000', '-7.719156', '1.014925', '-8.734081', '0.000000', '0.000000', '0.000000', '-4.006239', '-0.000000', '-4.006239', '0.000000', '0.000000', '0.000000'], ['23.0', '-1.003904', '0.000000', '-1.003904', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-2.409419', '0.022817', '-2.432236', '-6.427160', '0.032149', '-6.459308', '-0.446215', '0.000000', '-0.446215', '-11.841477', '0.032149', '-11.873626', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-13.681736', '0.032149', '-13.713885', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-5.441338', '0.000000', '-5.441338', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-17.475985', '0.032149', '-17.508134', '-0.052158', '0.000000', '-0.052158', '-1.332201', '0.019035', '-1.351236', '-20.048633', '0.032149', '-20.080782', '0.000000', '0.000000', '0.000000', '-0.049058', '0.000987', '-0.050045', '-1.784944', '0.000000', '-1.784944', '0.000000', '0.000000', '0.000000', '-0.968761', '0.000000', '-0.968761', '0.000000', '0.000000', '0.000000', '-8.764630', '0.032149', '-8.796779', '0.000000', '0.000000', '0.000000', '-4.323367', '0.000000', '-4.323367', '0.000000', '0.000000', '0.000000'], ['24.0', '-0.354859', '0.000001', '-0.354859', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.571338', '0.088108', '-1.659446', '-5.770265', '0.124954', '-5.895219', '-0.555659', '0.000000', '-0.555659', '-7.626980', '0.165310', '-7.792290', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-8.345034', '0.177161', '-8.522195', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-0.868868', '0.023821', '-0.892689', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-9.717466', '0.176972', '-9.894438', '-0.009119', '0.000543', '-0.009662', '-0.899348', '0.010367', '-0.909716', '-9.995856', '0.178702', '-10.174558', '0.000000', '0.000000', '0.000000', '-0.030403', '0.000715', '-0.031118', '-0.212713', '0.007522', '-0.220236', '0.000000', '0.000000', '0.000000', '-0.030343', '0.000262', '-0.030606', '0.000000', '0.000000', '0.000000', '-5.799340', '0.166997', '-5.966337', '0.000000', '0.000000', '0.000000', '-0.476058', '0.014951', '-0.491009', '0.000000', '0.000000', '0.000000'], ['25.0', '0.675584', '0.699507', '-0.023922', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.311317', '1.812056', '-1.500739', '3.399805', '6.132872', '-2.733066', '0.652863', '0.652863', '-0.000000', '5.920508', '9.312678', '-3.392170', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '7.400338', '10.799946', '-3.399608', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.891357', '2.967335', '-0.075978', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '10.201566', '13.604211', '-3.402644', '0.020836', '0.024215', '-0.003379', '0.098940', '0.961793', '-0.862854', '11.758549', '15.161194', '-3.402644', '0.000000', '0.000000', '0.000000', '-0.007729', '0.033373', '-0.041102', '0.893976', '0.896727', '-0.002751', '0.000000', '0.000000', '0.000000', '0.525685', '0.525685', '-0.000000', '0.000000', '0.000000', '0.000000', '3.897669', '7.060226', '-3.162557', '0.000000', '0.000000', '0.000000', '2.230256', '2.250325', '-0.020069', '0.000000', '0.000000', '0.000000'], ['26.0', '0.811212', '5.963123', '-5.151911', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.251186', '6.686498', '-4.435312', '6.532186', '7.632624', '-1.100438', '0.566094', '6.032255', '-5.466161', '8.480969', '9.700821', '-1.219852', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '9.950482', '10.680025', '-0.729543', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '3.524959', '7.076314', '-3.551355', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '10.172517', '10.600876', '-0.428359', '0.032811', '5.974384', '-5.941572', '1.098837', '6.132547', '-5.033710', '11.200491', '11.319628', '-0.119138', '0.000000', '0.000000', '0.000000', '0.044393', '5.968144', '-5.923751', '1.099159', '6.346599', '-5.247441', '0.000000', '0.000000', '0.000000', '0.527495', '6.291386', '-5.763890', '0.000000', '0.000000', '0.000000', '7.796359', '8.798300', '-1.001942', '0.000000', '0.000000', '0.000000', '2.645323', '6.930703', '-4.285380', '0.000000', '0.000000', '0.000000'], ['27.0', '0.243552', '0.246955', '-0.003403', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-1.350807', '0.625349', '-1.976156', '-4.787912', '0.491725', '-5.279637', '-0.335680', '0.001694', '-0.337374', '-4.780680', '1.789777', '-6.570457', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.322609', '2.736457', '-7.059066', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '2.486483', '2.502196', '-0.015713', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.216275', '3.152795', '-7.369070', '0.018190', '0.018190', '-0.000000', '-0.889275', '0.002058', '-0.891332', '-2.898083', '4.475702', '-7.373786', '0.000000', '0.000000', '0.000000', '-0.033146', '0.000046', '-0.033192', '0.895512', '0.895512', '0.000000', '0.000000', '0.000000', '0.000000', '0.639144', '0.639144', '0.000000', '0.000000', '0.000000', '0.000000', '-3.609278', '1.687721', '-5.296999', '0.000000', '0.000000', '0.000000', '2.270597', '2.270597', '0.000000', '0.000000', '0.000000', '0.000000'], ['28.0', '0.557380', '0.557380', '-0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.228537', '2.006025', '-0.777488', '0.646746', '3.938370', '-3.291623', '-0.174077', '0.000002', '-0.174079', '0.025221', '5.332454', '-5.307233', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.813701', '6.778732', '-5.965031', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '3.512692', '3.533411', '-0.020719', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-2.106301', '6.725198', '-8.831500', '0.033450', '0.033450', '-0.000000', '0.409200', '0.840931', '-0.431731', '-1.404773', '7.978166', '-9.382939', '0.000000', '0.000000', '0.000000', '0.030635', '0.040119', '-0.009484', '1.191172', '1.191172', '-0.000000', '0.000000', '0.000000', '0.000000', '0.647861', '0.647861', '-0.000000', '0.000000', '0.000000', '0.000000', '2.570015', '5.399355', '-2.829340', '0.000000', '0.000000', '0.000000', '2.892898', '2.892898', '-0.000000', '0.000000', '0.000000', '0.000000'], ['29.0', '-0.029982', '0.157800', '-0.187782', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-4.120393', '0.607840', '-4.728233', '-10.970081', '0.343944', '-11.314024', '-0.511919', '0.005882', '-0.517801', '-12.017387', '1.697792', '-13.715179', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-11.844404', '2.190787', '-14.035192', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '1.532974', '1.879584', '-0.346610', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '0.000000', '-10.699114', '3.623349', '-14.322463', '0.001827', '0.010833', '-0.009006', '-2.426472', '0.078284', '-2.504756', '-9.121184', '5.201280', '-14.322463', '0.000000', '0.000000', '0.000000', '-0.102291', '0.001411', '-0.103701', '0.615548', '0.627416', '-0.011867', '0.000000', '0.000000', '0.000000', '0.637516', '0.637516', '0.000000', '0.000000', '0.000000', '0.000000', '-10.571201', '1.379475', '-11.950676', '0.000000', '0.000000', '0.000000', '1.701030', '1.778419', '-0.077389', '0.000000', '0.000000', '0.000000']])
