"""!
Unittest for slf.variables module
"""

import numpy as np
import os
import unittest

from pyteltools.geom.geometry import Polyline
from pyteltools.slf.flux import FluxCalculator
from pyteltools.slf import Serafin
from . import TestHeader


HOME = os.path.expanduser('~')
FMT_FLOAT = '{0:.6f}'


class FluxTestCase(unittest.TestCase):
    def setUp(self):
        self.path = os.path.join(HOME, 'dummpy.slf')

        # create the test Serafin
        test_values = [
            np.array([[ 0.38349591,  0.28184575,  0.82131011,  0.44144203], [ 0.01539982,  0.94318509,  0.37354463,  0.29029064], [ 0.95649501,  0.42545964,  0.71934098,  0.90633775], [ 0.91877206,  0.84460727,  0.86020649,  0.51518127]]),
            np.array([[ 0.8834196 ,  0.11747724,  0.7443704 ,  0.18205073], [ 0.63082923,  0.15845509,  0.40364192,  0.58075115], [ 0.95807   ,  0.92086884,  0.36064879,  0.04418103], [ 0.12337765,  0.4802664 ,  0.97878443,  0.48208562]]),
            np.array([[ 0.72436517,  0.61577218,  0.96423117,  0.48696964], [ 0.53834509,  0.03411994,  0.24254472,  0.38556598], [ 0.21807547,  0.24359492,  0.67922177,  0.80593002], [ 0.67306665,  0.68544011,  0.88005295,  0.48822952]]),
            np.array([[ 0.4266814 ,  0.04112674,  0.9751324 ,  0.89211481], [ 0.82700639,  0.26816616,  0.47189482,  0.48683202], [ 0.62045743,  0.67811216,  0.25319826,  0.58580643], [ 0.649816  ,  0.3202664 ,  0.66340659,  0.33334621]]),
            np.array([[ 0.97596404,  0.93659027,  0.76142286,  0.58538529], [ 0.52142683,  0.67156646,  0.44521961,  0.95003345], [ 0.64410477,  0.04232678,  0.74991078,  0.56150637], [ 0.77494019,  0.93531219,  0.85723383,  0.64954979]]),
            np.array([[ 0.75918213,  0.57037281,  0.1823303 ,  0.63376985], [ 0.71374718,  0.37418838,  0.99893839,  0.03951073], [ 0.7794488 ,  0.43852008,  0.47891494,  0.89329253], [ 0.97323672,  0.79065161,  0.28634436,  0.43179421]]),
            np.array([[ 0.99968489,  0.68308622,  0.47463641,  0.24550367], [ 0.11440212,  0.50180714,  0.2568784 ,  0.92023353], [ 0.25696521,  0.89712184,  0.20472891,  0.90875594], [ 0.31967462,  0.02949244,  0.36033003,  0.69904339]]),
            np.array([[ 0.54281142,  0.66098924,  0.4149728 ,  0.91846564], [ 0.237657  ,  0.38495761,  0.63251488,  0.05476437], [ 0.67620745,  0.96333286,  0.75031009,  0.87225518], [ 0.26194027,  0.98453379,  0.30568016,  0.35503823]]),
            np.array([[ 0.41436019,  0.72452591,  0.90901117,  0.16116732], [ 0.25034048,  0.37672312,  0.87244555,  0.12765039], [ 0.90845104,  0.60487605,  0.90527204,  0.34513181], [ 0.23093057,  0.89318517,  0.16678644,  0.26758509]]),
            np.array([[ 0.35634827,  0.55111418,  0.42012225,  0.79099202], [ 0.91161338,  0.53207439,  0.22908794,  0.81284158], [ 0.90841946,  0.95788289,  0.66069332,  0.2040349 ], [ 0.75599161,  0.9001013 ,  0.65526525,  0.91310287]]),
            np.array([[ 0.3764112 ,  0.38985541,  0.05099266,  0.77503252], [ 0.79573943,  0.69245179,  0.16970622,  0.06244361], [ 0.28953677,  0.76840003,  0.83995792,  0.96428801], [ 0.78521681,  0.08375228,  0.29135521,  0.38921955]]),
            np.array([[ 0.75413895,  0.04269859,  0.91790778,  0.63899474], [ 0.80022646,  0.63410013,  0.31118819,  0.17954582], [ 0.68971431,  0.61910832,  0.15826529,  0.96853187], [ 0.04154995,  0.98974203,  0.97081504,  0.38760663]]),
            np.array([[ 0.66745926,  0.60750215,  0.68134103,  0.62826551], [ 0.85042564,  0.36022665,  0.95148049,  0.18717025], [ 0.99335372,  0.37630344,  0.8377429 ,  0.75061677], [ 0.3261356 ,  0.8277549 ,  0.56952539,  0.80531943]]),
            np.array([[ 0.88357202,  0.02153313,  0.18212082,  0.51299134], [ 0.59512007,  0.06815293,  0.38473985,  0.64628018], [ 0.26339498,  0.24181991,  0.12389131,  0.42526214], [ 0.4563927 ,  0.04295343,  0.03932699,  0.02801154]]),
            np.array([[ 0.45702848,  0.815847  ,  0.16776277,  0.4272676 ], [ 0.72506628,  0.84708967,  0.87085608,  0.11994325], [ 0.45944565,  0.78291774,  0.08821093,  0.86300512], [ 0.66304638,  0.79926807,  0.27185172,  0.10918613]]),
            np.array([[ 0.85910071,  0.24295702,  0.34468638,  0.87223485], [ 0.14970224,  0.96194458,  0.09386879,  0.63735934], [ 0.67075182,  0.17109016,  0.63395692,  0.42396658], [ 0.80629822,  0.02946476,  0.67986736,  0.09836991]]),
            np.array([[ 0.5439189 ,  0.18774825,  0.80606559,  0.91923317], [ 0.05124375,  0.62560636,  0.17210656,  0.99270152], [ 0.63688775,  0.90882296,  0.68855399,  0.16320009], [ 0.41456404,  0.65842399,  0.12674771,  0.72161136]]),
            np.array([[ 0.380624  ,  0.60065918,  0.53671119,  0.70267577], [ 0.5946558 ,  0.94934849,  0.48774351,  0.22266659], [ 0.58608867,  0.90501109,  0.49181942,  0.29218863], [ 0.92443523,  0.10821503,  0.04640351,  0.88464552]]),
            np.array([[ 0.46466689,  0.17915849,  0.98603722,  0.15737033], [ 0.6273254 ,  0.6936598 ,  0.74723218,  0.48373246], [ 0.98085997,  0.25126396,  0.92974446,  0.11992768], [ 0.7965158 ,  0.25355006,  0.66147963,  0.89280903]]),
            np.array([[ 0.80122616,  0.60390765,  0.40715564,  0.07244639], [ 0.07190884,  0.15900124,  0.6191638 ,  0.25791523], [ 0.05607264,  0.42564605,  0.64313823,  0.88604954], [ 0.45922158,  0.99356688,  0.93066546,  0.32445746]]),
            np.array([[ 0.75010116,  0.51001168,  0.28518772,  0.3383823 ], [ 0.82723333,  0.25065044,  0.45501072,  0.03700784], [ 0.97794956,  0.43337004,  0.30973652,  0.15415813], [ 0.73419326,  0.24596219,  0.95960246,  0.17060384]]),
            np.array([[ 0.44228902,  0.77611109,  0.60703901,  0.43638402], [ 0.85255282,  0.11540885,  0.21085168,  0.17389563], [ 0.85954179,  0.84297558,  0.72334092,  0.01533289], [ 0.76241282,  0.80291587,  0.4825426 ,  0.56971382]]),
            np.array([[ 0.2125425 ,  0.15317314,  0.81263665,  0.53678568], [ 0.0374813 ,  0.19408153,  0.66138014,  0.35400696], [ 0.0484195 ,  0.08342311,  0.2510757 ,  0.24646198], [ 0.29006191,  0.41855452,  0.08272338,  0.33746253]]),
            np.array([[ 0.8083028 ,  0.52077958,  0.18066442,  0.77366625], [ 0.35033558,  0.21554811,  0.84865907,  0.98138635], [ 0.62603975,  0.00295888,  0.42658333,  0.71167046], [ 0.44213297,  0.33092185,  0.00418637,  0.51434004]]),
            np.array([[ 0.73393336,  0.47600865,  0.61869032,  0.77847503], [ 0.92765712,  0.87776387,  0.78372421,  0.09138112], [ 0.0355477 ,  0.78524731,  0.45985249,  0.20993098], [ 0.16087317,  0.04529427,  0.5073861 ,  0.09809498]]),
            np.array([[ 0.18100058,  0.03897991,  0.84324002,  0.77060709], [ 0.71142567,  0.35858666,  0.35301731,  0.67264405], [ 0.01677446,  0.56535169,  0.32949704,  0.17599686], [ 0.69972369,  0.49167021,  0.05696451,  0.53038728]]),
            np.array([[ 0.31951308,  0.83190817,  0.38766072,  0.15687814], [ 0.51477907,  0.26300675,  0.16267741,  0.59089223], [ 0.36850031,  0.19151521,  0.69364139,  0.15943386], [ 0.02686085,  0.14346006,  0.79225742,  0.83226549]]),
            np.array([[ 0.04339084,  0.16120337,  0.22403014,  0.41756213], [ 0.76451083,  0.98377409,  0.84243657,  0.5300914 ], [ 0.32652207,  0.65437784,  0.16184502,  0.66166233], [ 0.99290626,  0.31116712,  0.45263879,  0.12808324]]),
            np.array([[ 0.11982914,  0.79115526,  0.3545459 ,  0.00713988], [ 0.6994817 ,  0.25667014,  0.53218665,  0.06291636], [ 0.64345868,  0.45583228,  0.96024733,  0.53476051], [ 0.24901628,  0.82377853,  0.53754479,  0.08947586]]),
            np.array([[ 0.839994  ,  0.54840571,  0.31428023,  0.8393961 ], [ 0.58210453,  0.89363652,  0.25273033,  0.78891088], [ 0.2054574 ,  0.81293273,  0.70006029,  0.40613581], [ 0.46434694,  0.65174283,  0.30786446,  0.34987846]]),
            np.array([[ 0.52493016,  0.14057393,  0.32215758,  0.63633855], [ 0.17678125,  0.71122811,  0.32610112,  0.72541517], [ 0.1144667 ,  0.52874223,  0.3153468 ,  0.51140573], [ 0.42885969,  0.12981366,  0.96514766,  0.84219602]]),
            np.array([[ 0.77563702,  0.82235278,  0.14396114,  0.91403095], [ 0.42545689,  0.40414622,  0.42723717,  0.04181317], [ 0.95521682,  0.8521327 ,  0.48233156,  0.58139662], [ 0.80576318,  0.90203305,  0.53721483,  0.92889833]]),
            np.array([[ 0.07107774,  0.24920402,  0.37291948,  0.99975011], [ 0.35646242,  0.91486961,  0.44374282,  0.80039632], [ 0.12014315,  0.9787527 ,  0.93533124,  0.53350572], [ 0.04190089,  0.67785967,  0.09988663,  0.045246  ]]),
            np.array([[ 0.49083488,  0.06198431,  0.04431377,  0.35977193], [ 0.47123454,  0.01158896,  0.97320346,  0.37319324], [ 0.54184916,  0.94492721,  0.97364154,  0.77345031], [ 0.00243901,  0.18637228,  0.90610595,  0.7039006 ]]),
            np.array([[ 0.05823386,  0.49904627,  0.3121271 ,  0.99303485], [ 0.71962799,  0.90520214,  0.87191338,  0.33935328], [ 0.694913  ,  0.27374491,  0.49503294,  0.96327468], [ 0.21859903,  0.12077599,  0.74199905,  0.24159476]]),
            np.array([[ 0.61558118,  0.52908435,  0.41403691,  0.30899264], [ 0.2203235 ,  0.79704203,  0.94338344,  0.46194543], [ 0.64239215,  0.8149627 ,  0.18993373,  0.4283352 ], [ 0.0234214 ,  0.22388328,  0.33269977,  0.578147  ]]),
            np.array([[ 0.77704244,  0.92685241,  0.32150208,  0.77908883], [ 0.63614327,  0.30791896,  0.9029831 ,  0.76816949], [ 0.55978961,  0.06818406,  0.29327571,  0.15439419], [ 0.66534166,  0.97704166,  0.58008716,  0.5741167 ]]),
            np.array([[ 0.26373396,  0.58572532,  0.87021662,  0.83473499], [ 0.49744521,  0.36881997,  0.42342179,  0.46387461], [ 0.46349518,  0.93080337,  0.03884231,  0.54897874], [ 0.18167533,  0.73088342,  0.29395683,  0.33912486]]),
            np.array([[ 0.9096747 ,  0.58869004,  0.89636449,  0.25293954], [ 0.25871395,  0.62741256,  0.72082718,  0.26088372], [ 0.56624826,  0.60778881,  0.3656397 ,  0.59554513], [ 0.58259937,  0.11039406,  0.95365015,  0.83271268]]),
            np.array([[ 0.79780641,  0.7715427 ,  0.12404283,  0.09811476], [ 0.54102751,  0.31476696,  0.40490336,  0.89759896], [ 0.27465404,  0.6913    ,  0.36581295,  0.68421642], [ 0.16374261,  0.68782208,  0.77777459,  0.27683339]])
        ]

        header = TestHeader()
        for var_ID in ['U', 'V', 'H', 'M']:
            header.add_variable_from_ID(var_ID)
        with Serafin.Write(self.path, 'fr', overwrite=True) as f:
            f.write_header(header)
            for i in range(len(test_values)):
                f.write_entire_frame(header, i, test_values[i])

        # create the test polylines
        coords = [
            np.array([[ 0.33516516,  0.14775744], [ 4.80317728,  6.53057925], [ 5.7079121 ,  6.63488869], [ 4.56838111,  6.68305044]]),
            np.array([[ 7.16778771,  0.55599878], [ 5.85858821, -0.48346468], [ 0.90030813,  3.63023465]]),
            np.array([[ 2.80105777,  2.02054732], [ 1.83738635,  6.10318025]]),
            np.array([[ 7.91444069, -0.16992875], [ 3.14468973,  0.48291762], [ 1.63276453,  3.95552059], [-0.71028473, -0.34632502], [ 3.07477872, -0.39815476], [ 1.67879386, -0.73113959], [ 2.7777812 , -0.58639942]]),
            np.array([[ 1.94631661,  5.80397466], [-0.66534917,  0.98038665], [ 7.67676173,  3.39723144], [ 2.15811876,  1.94477331], [ 5.69775358,  6.10071847], [ 5.60909705,  5.41824527], [ 5.32994224,  3.46729671], [ 5.62454035,  5.82524893], [ 0.81650446,  4.24341911], [ 2.69514967,  2.93222578]]),
            np.array([[ 0.4112632 ,  7.35911993], [ 5.78783224, -0.53952112], [ 3.1314806 ,  1.29475498], [ 7.19009737,  6.18461839], [-0.86513537,  0.37042227]]),
            np.array([[ 5.6647955 ,  5.82430319], [-0.06633732,  0.97147435], [ 5.33314178,  6.21182902], [ 5.17198408, -0.5260646 ], [ 0.60782029,  2.98914304], [ 6.16447888,  4.49995109], [ 6.75962118,  2.46247507], [ 5.61123594,  4.05962306], [ 7.04903154,  1.71083704]]),
            np.array([[ 3.94162583,  5.85486057], [ 5.64116511,  1.28147204], [-0.03616107,  7.16348401], [ 4.97232931, -0.08758067]]),
            np.array([[ 7.65397187,  7.08617393], [ 3.34961149,  0.27459418], [-0.38428159,  3.73414406]]),
            np.array([[-0.56254266,  3.32622177], [ 2.70954267,  0.15436384], [ 3.36715729, -0.23689889], [-0.04920449,  3.26182403], [ 7.56172038,  7.66800133]]),
            np.array([[ 7.90262065,  7.23266968], [ 0.11310021,  3.96239076], [ 1.36894758,  6.90374244], [ 5.55539891,  1.08345161]]),
            np.array([[ 0.36552206,  6.94990425], [ 1.75148827,  4.24083333], [ 0.11404859,  5.46592372], [ 3.90100947,  6.23521917], [ 7.87119458,  5.12494087], [ 6.26190492,  2.27587642], [ 3.67088539, -0.35428948], [ 3.49065578,  7.10497498]]),
            np.array([[ 1.49252541,  5.68399448], [ 2.02563478,  2.04538145], [ 5.77418371,  2.11232819], [-0.85168539,  5.0050098 ], [ 7.49894319,  0.68341616], [ 7.15007817,  0.24349379]]),
            np.array([[ 3.61649849,  7.68677758], [ 2.03439843,  6.54862102], [-0.94272542,  1.72822117], [ 2.49713093,  0.54639136], [ 3.42996212,  1.01209602], [-0.76380308,  3.53376436], [ 4.53112907, -0.15162511], [ 0.92091672,  4.36176484]]),
            np.array([[ 4.67583898,  0.97272905], [ 5.32702372,  2.79367611], [ 1.95923888,  7.10437185], [-0.7928374 , -0.27465712]]),
            np.array([[ 0.18725367,  4.15428264], [ 1.72042163,  1.88630745], [ 0.05130969,  5.47055058], [-0.49317242,  0.97379607], [ 2.53373556, -0.28187753], [ 3.42080572,  4.99631783]]),
            np.array([[ 3.93818962,  6.55366264], [ 6.94210475,  4.06445501], [-0.49667675,  4.90206002], [ 4.56100806,  5.43507218], [ 2.61170676, -0.22533858]]),
            np.array([[ 4.41419369,  0.96091219], [-0.84847669,  2.88517475]]),
            np.array([[ 0.10530324,  0.78285491], [ 3.71576491,  3.84673394], [ 7.01425929,  2.61441371], [ 3.213055  ,  6.26791352], [ 5.45222433,  0.20512947], [ 5.05734931,  3.60919133], [-0.75857325,  1.29736883], [ 1.79810256,  5.60790027], [ 6.01224993,  0.76235463]]),
            np.array([[ 7.85715386,  6.31215236], [ 3.83382456,  4.5563913 ], [ 4.91233911,  4.42359593], [ 7.33086176,  2.85596184]]),
            np.array([[ 7.83351641,  4.3821358 ], [ 1.05582499, -0.13725898], [ 5.85266178,  5.12600643], [-0.13362487,  5.68293408], [ 7.86479393,  4.33502349], [ 5.38104354,  3.9369318 ], [ 7.74827371, -0.55866116], [-0.60660468,  1.79112249]]),
            np.array([[ 7.55362153,  6.86522918], [ 0.04387461,  3.49003747], [ 3.51545942,  7.51471191], [ 0.49739629,  3.37986262], [ 6.75037627, -0.74089793], [ 7.76109531, -0.4255701 ]]),
            np.array([[ 7.69058968,  0.36180708], [-0.42939881,  2.21113472], [ 1.52885129,  0.5757281 ], [ 7.48247604,  1.37394834], [ 2.70024117,  0.28950217]]),
            np.array([[-0.43028257,  0.17357834], [-0.38698925,  2.03403134], [ 7.52459744,  4.40274596], [ 0.70548419,  3.08830309], [-0.53028785,  0.13606776], [ 5.74150638,  0.80980262]]),
            np.array([[ 3.83914453,  2.93130618], [ 1.02748009,  4.23837827], [ 1.46235823,  1.65017826]]),
            np.array([[ 6.13898919,  4.95606311], [ 1.82989372,  6.71890027], [ 0.19434703,  7.58918449], [ 5.69726422,  7.89863721], [ 6.6307677 ,  3.17044702], [ 1.84314463,  0.97232777], [ 5.78155332,  3.03541808], [ 0.32231054,  1.505471  ], [ 7.88716187,  2.87964526], [-0.71775007,  4.62105355]]),
            np.array([[ 6.78884531,  3.54953309], [ 4.97335114,  3.37301133], [ 2.5964572 ,  0.6831734 ], [ 5.25533761,  1.27449381], [ 1.82575043,  6.18220284], [ 5.25184666,  1.88694295], [ 4.03218289,  0.44283122]]),
            np.array([[ 5.69637295,  0.99403255], [ 1.15037014,  6.66151517], [ 5.3191899 ,  2.66799767], [ 2.50889166,  4.5067131 ], [ 0.38456718,  1.59240081], [ 1.86758687,  4.26676202], [ 6.86929981,  3.52322122], [ 3.43252512,  4.16491964], [ 1.08318859,  6.38340178]]),
            np.array([[  1.57154422e+00,   7.80484281e+00], [ -7.27625169e-01,   1.85132435e+00], [  2.38046591e+00,   5.46283635e+00], [  6.91906577e+00,  -6.03760227e-01], [  6.94432915e+00,   5.31403678e+00], [  5.67499070e+00,   5.89586036e+00], [  9.37816734e-01,   4.98834994e+00], [  1.29188050e+00,   3.95422069e+00], [  2.19952542e+00,  -7.59687755e-05]]),
            np.array([[ 4.18141281,  7.54497087], [ 5.62953865,  0.58605707], [ 0.65691631,  0.97833203], [-0.05816246, -0.34953473]]),
            np.array([[ 4.03494423,  2.02329985], [ 3.66877668, -0.6109489 ], [ 5.10996455, -0.95662148], [ 1.74459023,  0.70997636], [ 7.40040892,  7.89444939], [ 4.2027693 ,  6.80086867], [ 0.57830731,  6.36669212]]),
            np.array([[ 4.09587703,  5.07066962], [ 5.20329629,  7.6518374 ], [ 1.51038912,  1.9541105 ], [ 4.28720395,  2.42865213], [ 6.14189371,  5.1248026 ]]),
            np.array([[ 3.8954852 , -0.85530739], [ 6.47553342,  6.96140376], [ 2.40789266,  4.03558543], [ 6.93888969,  5.40109088], [ 4.71455006,  0.01888135], [ 2.00754725,  3.46642803]]),
            np.array([[ 0.00971732, -0.30030379], [ 0.46184538,  2.76096037], [ 4.01762499, -0.3213378 ], [ 1.42882533,  3.17809532], [ 1.072632  ,  4.06861429], [ 4.76329772,  1.00848224], [-0.22386363,  4.37813646], [ 4.23630186,  2.66543495], [ 2.14144946,  4.8364954 ]]),
            np.array([[ 4.21130041, -0.05378894], [ 1.26368087,  0.85713535], [ 3.82500242, -0.80915999], [ 7.96289457,  0.54861529], [ 6.7410688 ,  2.65976054]]),
            np.array([[ 4.49699289,  4.13714046], [ 5.35463478,  2.15409997], [-0.06445185,  0.74478093], [ 6.40035708,  5.51181133], [ 2.17274545,  2.81982293], [-0.92125902, -0.9804321 ], [ 0.32266591,  6.49591342], [-0.41982644,  6.31622335], [ 2.43382197,  2.80514807]]),
            np.array([[ 1.21334714,  3.06341096], [ 7.45246201, -0.04556055], [ 0.42089979,  0.92067826], [ 2.45265439,  5.28947263], [ 6.06841405,  4.29497371]]),
            np.array([[ 1.26553746,  6.72934805], [ 0.98162253,  3.65856255], [ 1.8511666 ,  1.24394985]]),
            np.array([[ 7.47647265,  3.52699719], [ 6.73870926,  2.68218775], [ 7.58848464,  3.19572378], [-0.15009991,  4.26565027], [ 7.17037576,  5.3469049 ], [ 5.94091953,  7.39578488], [ 2.09295335,  2.90389635]]),
            np.array([[ 3.54081856,  2.10910272], [-0.67816514,  4.44587544], [ 6.9594918 ,  0.77152679], [ 1.57382738,  3.44784716], [ 1.40997486,  2.52381853]]),
            np.array([[ 6.58635694,  6.25679774], [ 0.1498538 ,  4.60976191], [-0.2403225 ,  6.19189456]]),
            np.array([[ 1.28238221,  7.37001018], [ 2.59426515,  7.40984215], [ 2.54261491,  2.30882387], [ 1.21981801,  4.78973502], [ 1.33470352,  7.3698538 ], [ 4.64406713,  1.56684878], [ 5.43602634,  4.46990109], [ 3.2893599 ,  6.511502  ], [ 6.34391991,  4.88558133], [ 2.62756057,  5.08351818]]),
            np.array([[ 5.45564241,  3.96555179], [ 1.43886865,  7.71383624], [ 4.25557516,  2.25393755], [ 0.44764547,  5.59746392], [ 4.88740458,  5.86081002], [ 2.19356936, -0.50870256], [ 1.51647308,  5.37675932], [ 1.67201364,  6.3631736 ]]),
            np.array([[ 0.14433655,  1.93468028], [ 5.36738181,  4.77293945], [ 4.00924827, -0.33848857], [ 1.52944231,  7.26499904], [ 4.32226952,  0.36630442]]),
            np.array([[ 2.04606078, -0.8725981 ], [ 0.08886986, -0.99531724], [ 4.59431559,  7.40429235], [ 2.3329605 ,  3.93532287], [ 6.33444877,  2.75446423], [ 5.08726694,  0.01017429], [ 5.37008761,  6.04194346], [ 2.89323478,  3.23204465], [ 7.9940126 ,  6.25642016], [ 5.44244461,  1.37407028]]),
            np.array([[ 4.81540002,  1.94830201], [ 0.55167434,  7.26872853]]),
            np.array([[ 5.68142955,  2.97698974], [ 2.38503138,  0.94440594], [-0.25423939,  0.79009496], [ 4.87264772,  5.152948  ], [ 0.95759799,  4.71189823], [ 4.88453146,  6.51443782]]),
            np.array([[ 2.35822376,  1.56934561], [ 7.31268651,  4.99966641], [ 3.19137098,  4.18602248], [ 3.87637395,  2.26882767], [ 4.51580954,  3.06387788], [ 5.68040692,  5.99340555], [ 2.14292478,  4.76438291]]),
            np.array([[ 2.81977388,  5.00264389], [ 1.52217918,  4.60743519], [ 4.06738751,  5.69670084], [ 7.6498814 ,  3.95621587], [ 1.50636546,  0.58677499], [ 1.30562149,  2.54085268], [ 1.10421663,  3.17472567], [-0.39305874,  0.19024148], [ 1.22921679, -0.49408541], [ 1.59801336,  6.97763946]]),
            np.array([[ 4.85454119,  7.78990984], [ 3.06902414,  4.70583399], [ 2.29676791,  2.85675661], [ 3.45949365,  5.79115992], [ 7.91385961,  1.53862616]]),
            np.array([[ 6.05234784,  1.44960972], [ 3.87164326,  3.21801021], [ 2.38585056,  3.90163533], [ 7.38676759,  2.9213207 ], [ 1.70154381,  1.39079061], [-0.16993793,  3.64622817], [ 4.39855521,  7.03035026]]),
            np.array([[ 7.89412053,  5.10829733], [ 7.43128757,  4.17066784], [ 6.84103279,  4.55658653], [ 6.02690217,  7.82216114], [ 3.71230457,  4.26576928], [ 6.35485769,  3.01370065], [ 1.70080858,  2.60931264], [ 5.34287544,  1.0554355 ], [ 1.08000558,  5.49994586]]),
            np.array([[-0.34457742,  6.60831292], [ 0.12216126,  0.98967261], [ 0.27893884,  6.58445844], [ 0.7225065 ,  6.55710732], [ 0.9016516 ,  7.62709878], [ 6.29928512,  2.13364715], [-0.65900116, -0.77613937], [ 2.71900391,  1.30587532]]),
            np.array([[ 1.50938623,  4.46648937], [ 3.78110477,  2.89623097], [ 7.64395317,  0.97168703], [ 2.23588958,  6.79348277]]),
            np.array([[ 6.524952  ,  0.60701927], [-0.44850876,  5.82679923], [-0.08808957,  3.83343075], [ 0.87467454,  0.46621152], [ 1.622586  , -0.30082337]]),
            np.array([[ 6.44492777,  0.48225661], [ 2.67745763,  7.20880836], [ 4.29385011,  3.5284241 ], [ 1.23773019,  1.81835999], [ 7.26026813,  6.78375912], [ 3.88404771,  4.86679959], [ 1.52322878,  4.87899863], [ 4.55476708,  1.59382601], [ 4.45778479,  2.67458774], [ 5.55153743,  4.01912042]]),
            np.array([[ 2.71025271,  2.77854912], [ 7.80391138,  4.59059006], [ 0.97796588,  2.2544419 ], [ 4.67557882,  7.81869051], [ 1.20507559,  5.27080256], [ 4.47711888,  3.94238493], [ 4.3736829 ,  3.75780946]]),
            np.array([[ 4.43124795,  0.79194253], [ 5.82670695,  5.53199585], [ 3.02336677,  3.43570801], [ 6.14827356,  6.55537044]]),
            np.array([[ 7.90517408, -0.09284805], [ 7.9176939 ,  2.32923785], [ 1.82119525, -0.30610949], [ 1.07923351,  4.84538868], [ 3.15177748,  1.12336528], [ 3.96020375,  1.37910063]]),
            np.array([[ 0.36863136,  3.55570486], [ 0.08739766,  0.1344606 ]])]

        self.sections = []
        for coord in coords:
            coord = list(map(list, coord))
            self.sections.append(Polyline(coord))
        self.names = [poly.id for poly in self.sections]

    def tearDown(self):
        # remove the test Serafin
        os.remove(self.path)

    def test_line_integral(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = FluxCalculator(FluxCalculator.LINE_INTEGRAL, ['U'], f,
                                        self.names, self.sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '2.118614', '2.513977', '0.982362', '3.335296', '5.269674', '7.202070', '7.118954', '4.546073', '2.821008', '2.890381', '2.767453', '3.792860', '5.262250', '6.323353', '0.943729', '3.483942', '3.018129', '1.612145', '8.674304', '0.000000', '7.644223', '3.244732', '6.805922', '5.263244', '1.226458', '7.874018', '8.436014', '6.276322', '3.584781', '3.152800', '3.702001', '2.997686', '5.970491', '9.403420', '2.119086', '6.253568', '5.709236', '0.537184', '2.472293', '4.946116', '0.273253', '4.577606', '6.486770', '7.612072', '7.099950', '1.884733', '4.339029', '3.530728', '4.044479', '2.149606', '4.200997', '5.979241', '3.663470', '1.585879', '2.543439', '5.901258', '4.041108', '2.081090', '4.521504', '0.013812'], ['1.0', '2.198227', '1.917233', '1.113221', '2.715294', '5.196455', '6.811219', '7.631914', '4.746612', '2.191290', '2.538208', '3.937309', '3.639222', '5.547679', '4.456546', '1.211458', '3.050702', '3.449845', '1.068226', '9.681210', '0.000000', '6.344556', '3.019418', '5.014748', '4.962170', '1.427322', '6.772028', '9.116702', '9.311620', '4.375669', '2.216070', '2.785384', '3.216984', '5.845705', '8.387176', '1.406742', '5.740958', '4.891077', '0.469169', '3.635083', '4.926511', '0.551545', '5.946369', '6.849682', '7.449584', '8.301710', '2.458149', '4.513022', '4.074616', '3.623253', '3.148035', '4.344585', '5.919701', '3.005514', '1.944112', '2.570950', '6.615702', '5.573958', '2.428437', '3.514954', '0.006624'], ['2.0', '3.504423', '3.197379', '1.501134', '4.403715', '7.408855', '9.609133', '10.554758', '6.116169', '3.789151', '4.516009', '4.083706', '4.917466', '7.363733', '8.800571', '1.309484', '5.243476', '4.310838', '2.133582', '12.200215', '0.000000', '10.498994', '4.351369', '8.858982', '7.635261', '1.946343', '10.632851', '11.309509', '9.697307', '5.453069', '4.386238', '4.881630', '4.384945', '7.855487', '13.016016', '3.087277', '9.515143', '7.684855', '0.912145', '3.911455', '6.840593', '0.489650', '6.859885', '9.363240', '10.294286', '10.456199', '2.705090', '6.538687', '4.878469', '6.246962', '3.434166', '5.850562', '7.947167', '5.269914', '2.273138', '3.725094', '8.508468', '6.387193', '2.816546', '6.294837', '0.029923'], ['3.0', '2.503628', '3.641794', '1.431373', '4.295869', '7.731241', '9.993597', '9.044933', '6.444994', '3.709366', '3.237329', '3.489654', '5.170325', '7.427849', '8.362447', '1.184843', '4.460525', '4.142060', '2.393959', '11.142747', '0.000000', '9.951642', '4.511269', '8.902761', '6.438018', '1.582646', '11.510076', '11.328051', '7.962182', '4.366689', '3.689339', '5.177351', '4.256983', '8.206117', '13.298101', '2.220294', '8.032340', '7.360197', '0.590855', '3.302962', '7.050421', '0.321173', '6.136598', '8.856633', '10.928342', '9.127027', '2.528782', '5.642390', '5.150815', '4.863361', '2.983048', '5.968281', '8.641879', '4.344801', '2.185200', '3.167677', '8.242757', '5.233602', '2.854381', '5.800454', '0.002662'], ['4.0', '4.762058', '3.491352', '1.986692', '4.741897', '9.067381', '10.682738', '12.843201', '6.848278', '4.263235', '5.765658', '4.671785', '5.155455', '8.698256', '10.569502', '1.389136', '6.702464', '5.099870', '2.549039', '13.751145', '0.000000', '12.044862', '4.893379', '9.530113', '9.054790', '2.559214', '12.544167', '12.157517', '11.756043', '6.613893', '4.994997', '5.536163', '5.492582', '8.447606', '15.548903', '3.625471', '12.268797', '8.472870', '1.272089', '4.992633', '8.060727', '0.650373', '8.377412', '11.358674', '11.823647', '12.491411', '3.110443', '8.235535', '5.663420', '8.033644', '4.501162', '6.908116', '8.909526', '6.215310', '2.673323', '4.462994', '10.311753', '8.144548', '3.061790', '7.377091', '0.045324'], ['5.0', '3.647201', '2.479492', '1.681008', '2.929596', '7.263017', '7.464297', '9.218961', '4.948319', '2.835863', '3.987870', '3.179596', '3.310059', '6.577420', '7.562362', '0.848858', '5.028195', '3.788700', '1.998951', '9.172102', '0.000000', '8.150419', '3.463432', '5.942362', '6.098733', '2.003909', '9.683417', '7.980465', '8.572853', '4.632335', '3.024323', '3.935118', '4.382339', '5.648371', '11.804618', '2.106071', '9.316426', '5.423169', '0.960190', '3.940285', '6.117364', '0.506173', '6.337369', '8.563609', '8.822965', '8.917798', '2.224644', '6.202607', '4.361542', '5.840628', '3.709100', '5.245810', '6.550370', '4.029152', '1.989424', '3.095189', '7.927515', '6.278575', '2.091469', '5.035197', '0.027867'], ['6.0', '3.654451', '2.142322', '1.526767', '3.083540', '6.496364', '7.404867', '9.738856', '4.965904', '2.753020', '4.179840', '4.009455', '3.551950', '6.409941', '6.767983', '1.135626', '4.762804', '3.915189', '1.540218', '10.488037', '0.000000', '8.077874', '3.376409', '5.894012', '6.531662', '2.014496', '8.395952', '8.995630', '10.227606', '5.312252', '3.205559', '3.481932', '4.116553', '5.913169', '10.696120', '2.364808', '8.872697', '5.689768', '0.941030', '4.288081', '5.779007', '0.634557', '6.837291', '8.464777', '8.342232', '9.783379', '2.542440', '6.184542', '4.314225', '5.815434', '3.839561', '5.041143', '6.270247', '4.268541', '2.116308', '3.307009', '7.851426', '6.833140', '2.352351', '4.877017', '0.033357'], ['7.0', '3.976131', '3.425990', '1.825238', '4.018050', '8.537106', '9.364370', '10.400359', '5.913973', '3.813712', '4.698551', '3.113534', '4.289706', '7.629629', '9.969313', '0.906525', '5.970787', '4.224743', '2.689400', '10.495308', '0.000000', '10.542660', '4.344400', '8.477007', '7.334129', '2.134079', '12.171103', '9.544913', '8.170058', '4.903513', '4.178189', '5.309842', '4.931373', '7.120636', '14.529703', '2.883184', '10.855536', '7.138285', '1.083978', '3.805838', '7.278671', '0.399030', '6.559694', '9.790079', '10.717253', '9.742025', '2.325101', '6.998649', '4.901536', '6.839335', '3.606602', '6.138675', '8.023795', '5.122701', '2.147447', '3.600306', '8.855757', '6.241388', '2.383425', '6.559965', '0.031935'], ['8.0', '2.605401', '2.284217', '0.878104', '3.576247', '4.637561', '6.818573', '7.691174', '4.080160', '2.992918', '3.757027', '2.787751', '3.627567', '4.830461', '6.859752', '0.960380', '3.910986', '2.853924', '1.468824', '8.959289', '0.000000', '8.187593', '3.081198', '7.242578', '6.039777', '1.314185', '7.064829', '8.119759', '6.320653', '3.981021', '3.901019', '3.610961', '2.771501', '5.651416', '8.913389', '2.943039', '6.965194', '6.104306', '0.732030', '2.358923', '4.512732', '0.287844', '4.409176', '6.339385', '6.826023', '7.384447', '1.774286', '4.622636', '2.974027', '4.885291', '1.952995', '3.828579', '5.245778', '4.429470', '1.463161', '2.833363', '5.490575', '4.103468', '1.910248', '5.009353', '0.034582'], ['9.0', '3.197397', '2.971134', '1.456149', '3.507645', '7.021449', '7.923602', '8.503277', '4.923158', '3.297109', '3.903477', '2.432278', '3.683155', '6.272549', '8.534734', '0.738011', '4.945549', '3.433101', '2.303228', '8.669667', '0.000000', '9.012463', '3.672617', '7.464720', '6.143931', '1.699600', '10.238930', '8.002102', '6.272475', '3.931085', '3.657482', '4.576350', '3.999180', '6.051538', '12.160142', '2.524932', '8.931334', '6.153267', '0.882813', '2.917694', '6.030395', '0.277808', '5.185225', '8.011818', '8.945528', '7.884412', '1.853111', '5.702346', '3.983957', '5.651925', '2.762877', '5.059436', '6.720997', '4.370561', '1.725703', '2.979734', '7.182838', '4.849802', '1.963244', '5.614943', '0.026517'], ['10.0', '2.832948', '2.337624', '1.392544', '2.450493', '6.271605', '6.233228', '7.010349', '4.015354', '2.473520', '3.101436', '1.929198', '2.662886', '5.365542', '6.839391', '0.503080', '4.205514', '2.918559', '1.961418', '6.598401', '0.000000', '6.892362', '2.923486', '5.263382', '4.722599', '1.540345', '8.743401', '5.945154', '5.443774', '3.199231', '2.519358', '3.623172', '3.609760', '4.562605', '10.311382', '1.671036', '7.704473', '4.455341', '0.767381', '2.739686', '5.143858', '0.279432', '4.571739', '6.872029', '7.484455', '6.494881', '1.536607', '4.914678', '3.479972', '4.699426', '2.692666', '4.334000', '5.538473', '3.225634', '1.479190', '2.365400', '6.294373', '4.415601', '1.526145', '4.330395', '0.018960'], ['11.0', '2.634837', '3.119277', '1.468202', '3.852821', '7.386570', '9.433747', '9.343038', '6.328463', '3.260816', '3.158134', '4.184307', '4.913356', '7.363532', '7.109467', '1.336230', '4.201623', '4.302232', '1.958242', '11.629987', '0.000000', '9.026620', '4.230165', '7.640029', '6.298174', '1.708642', '10.369916', '11.444350', '9.799795', '4.893067', '3.178173', '4.449752', '4.268457', '7.867019', '12.258059', '1.906344', '7.729494', '6.760150', '0.585233', '3.976715', '6.793921', '0.501289', '6.872928', '8.888396', '10.410732', '9.799200', '2.824193', '5.721943', '5.268617', '4.705566', '3.537889', '5.854588', '8.246296', '4.010480', '2.345870', '3.192567', '8.452745', '6.156866', '2.977568', '5.142803', '0.003067'], ['12.0', '3.604885', '3.119405', '1.600399', '4.019311', '7.617871', '9.107032', '10.168174', '5.814867', '3.605585', '4.418750', '3.622254', '4.443850', '7.233096', '8.828818', '1.110711', '5.342613', '4.155416', '2.245643', '11.111157', '0.000000', '10.024518', '4.165384', '8.205480', '7.214652', '1.980483', '10.802103', '10.180843', '8.969429', '5.099447', '4.061521', '4.802896', '4.484034', '7.227744', '13.103904', '2.834836', '9.729359', '7.095015', '0.952849', '3.834269', '6.775116', '0.459403', '6.609073', '9.245422', '10.075412', '9.886623', '2.492949', '6.528915', '4.752774', '6.260807', '3.479095', '5.771931', '7.671401', '4.967710', '2.167639', '3.544644', '8.414710', '6.242864', '2.553507', '6.097384', '0.029500'], ['13.0', '2.521277', '1.878998', '1.444888', '2.014355', '6.153753', '6.316444', '7.395425', '4.587664', '1.883985', '2.362252', '3.400821', '2.938751', '5.819717', '4.590781', '0.914217', '3.393986', '3.472586', '1.356482', '8.219030', '0.000000', '5.625299', '2.871091', '3.771317', '4.316364', '1.623421', '7.689326', '7.584554', '8.843890', '4.002994', '1.486658', '2.759327', '3.747812', '5.024574', '9.221534', '0.777397', '6.493030', '3.884730', '0.555385', '3.910046', '5.253666', '0.559926', '6.071342', '7.217888', '7.696203', '7.851365', '2.286908', '4.832277', '4.285444', '3.767725', '3.638046', '4.604644', '5.924305', '2.415304', '1.936144', '2.353564', '7.095778', '5.875944', '2.131835', '3.194677', '0.002196'], ['14.0', '3.257791', '1.999119', '1.254535', '2.625664', '5.448187', '5.723297', '7.531817', '3.507001', '2.551003', '3.888496', '2.082971', '2.478308', '4.847167', '7.034001', '0.561074', '4.485935', '2.747382', '1.700485', '6.977608', '0.000000', '7.215054', '2.690312', '5.488598', '5.427148', '1.609811', '7.583335', '5.699597', '5.763578', '3.653935', '3.169537', '3.363744', '3.305131', '4.200169', '9.458957', '2.431397', '8.170466', '4.747270', '0.935770', '2.688324', '4.589730', '0.319866', '4.465024', '6.614368', '6.522308', '6.828749', '1.473893', '5.108129', '2.903751', '5.401774', '2.525283', '3.878949', '4.692792', '3.902829', '1.369790', '2.631648', '5.814587', '4.550320', '1.385875', '4.650013', '0.038989'], ['15.0', '3.582181', '3.084851', '1.912757', '3.392943', '8.618663', '9.172164', '10.152464', '6.241134', '3.196405', '3.777373', '3.894580', '4.256350', '7.912917', '8.056191', '1.088805', '5.208999', '4.534801', '2.314542', '10.896491', '0.000000', '9.087153', '4.207310', '6.846353', '6.490252', '2.147333', '11.569228', '10.149199', '10.114405', '5.110328', '2.990803', '4.606416', '5.070563', '7.154688', '13.726162', '1.823436', '9.694499', '6.225939', '0.872386', '4.579932', '7.352790', '0.573776', '7.463585', '9.877283', '10.853669', '10.203996', '2.761156', '6.735675', '5.519382', '5.806896', '4.306210', '6.319314', '8.283080', '4.083267', '2.430239', '3.360575', '9.376865', '7.096747', '2.711956', '5.403857', '0.012660'], ['16.0', '3.043040', '3.632433', '1.635493', '4.246037', '8.299723', '10.102806', '9.817588', '6.553148', '3.769939', '3.708276', '3.639103', '5.047934', '7.818898', '8.877981', '1.167810', '5.014388', '4.380589', '2.510703', '11.417188', '0.000000', '10.262098', '4.591767', '8.786008', '6.845266', '1.837227', '12.034748', '11.240949', '8.703374', '4.764093', '3.786774', '5.279922', '4.670707', '8.135308', '14.055936', '2.349342', '9.104512', '7.392714', '0.747787', '3.745516', '7.396590', '0.393819', '6.680707', '9.523513', '11.270725', '9.775181', '2.633249', '6.278920', '5.361707', '5.551413', '3.439335', '6.275794', '8.782685', '4.595460', '2.304784', '3.396568', '8.852529', '5.942611', '2.856490', '6.070350', '0.009699'], ['17.0', '3.200014', '2.963883', '1.401060', '3.667693', '6.844576', '8.043390', '8.682191', '4.968844', '3.373249', '4.020678', '2.603284', '3.832538', '6.251303', '8.548345', '0.813545', '4.934946', '3.461849', '2.235115', '9.086560', '0.000000', '9.218520', '3.711716', '7.733381', '6.366802', '1.689729', '10.044616', '8.370469', '6.535132', '4.095885', '3.849497', '4.578173', '3.921020', '6.234098', '12.024790', '2.702065', '8.892348', '6.398766', '0.885123', '2.924566', '5.984636', '0.290180', '5.244140', '8.010241', '8.908567', '8.104944', '1.915761', '5.715384', '3.956136', '5.716814', '2.714172', '5.029307', '6.720332', '4.553359', '1.746366', '3.069402', '7.146582', '4.894606', '2.038196', '5.721869', '0.028853'], ['18.0', '1.569116', '2.031930', '0.702036', '3.030786', '4.025814', '6.374039', '6.263826', '4.098263', '2.375315', '2.329973', '2.997133', '3.617205', '4.481801', '4.700827', '1.048421', '2.585262', '2.709981', '1.077783', '8.476212', '0.000000', '6.475495', '2.813606', '5.936481', '4.650887', '0.977612', '6.004463', '8.313934', '6.478597', '3.433920', '2.721834', '2.925429', '2.351477', '5.575757', '7.330866', '1.835102', '4.646969', '5.149478', '0.359096', '2.300768', '4.093990', '0.305314', '4.249442', '5.448933', '6.428273', '6.611056', '1.886053', '3.523804', '3.133078', '3.115672', '1.868871', '3.533760', '5.194936', '3.154565', '1.478480', '2.256076', '5.026976', '3.701429', '2.070146', '3.676351', '0.009014'], ['19.0', '2.830437', '1.497540', '1.107392', '2.353453', '4.642005', '5.450209', '7.480969', '3.638041', '2.062095', '3.305251', '3.149833', '2.653954', '4.704080', '5.004126', '0.898861', '3.597076', '2.929843', '1.049074', '8.106597', '0.000000', '6.097531', '2.478004', '4.430222', '5.081484', '1.532164', '5.930154', '6.827019', '7.975214', '4.164156', '2.533611', '2.501073', '3.011304', '4.396293', '7.737803', '1.935775', '6.701247', '4.341462', '0.736362', '3.270586', '4.199628', '0.503250', '5.194615', '6.307008', '6.028851', '7.526986', '1.941761', '4.685738', '3.135766', '4.494305', '2.885799', '3.681342', '4.517131', '3.349059', '1.588878', '2.558162', '5.812579', '5.247257', '1.773507', '3.689530', '0.029356'], ['20.0', '2.999597', '1.874673', '1.317109', '2.467579', '5.632856', '6.121479', '7.841269', '4.100096', '2.279362', '3.349866', '3.039082', '2.842119', '5.355760', '5.800502', '0.841025', '3.997947', '3.194879', '1.420392', '8.197510', '0.000000', '6.639881', '2.810874', '4.823379', '5.206443', '1.658684', '7.360909', '7.078906', '7.915238', '4.148195', '2.540354', '3.008772', '3.494571', '4.783595', '9.188426', '1.823909', '7.438560', '4.567700', '0.776249', '3.440014', '4.888866', '0.483856', '5.493002', '7.020300', '7.055085', '7.778815', '1.998930', '5.099628', '3.595554', '4.772443', '3.144609', '4.236492', '5.278919', '3.401704', '1.710036', '2.644923', '6.520645', '5.469102', '1.858463', '4.040353', '0.024948'], ['21.0', '3.211851', '2.576760', '1.234262', '3.555246', '5.950663', '7.337652', '8.516247', '4.460149', '3.193474', '4.175123', '2.678773', '3.568532', '5.634793', '7.990281', '0.844975', '4.729746', '3.213321', '1.905048', '8.973197', '0.000000', '8.808802', '3.376419', '7.362056', '6.390930', '1.629022', '8.714462', '7.960239', '6.623943', '4.192289', '3.942316', '4.110974', '3.517141', '5.756497', '10.752015', '2.926307', '8.514066', '6.199484', '0.906720', '2.807696', '5.334502', '0.317097', '4.997865', '7.423856', '7.880721', '7.954022', '1.838702', '5.475621', '3.468130', '5.698575', '2.518123', '4.503229', '5.905338', '4.631583', '1.622973', '3.050543', '6.517317', '4.788991', '1.906508', '5.500013', '0.037107'], ['22.0', '1.717485', '2.561330', '0.870162', '3.232830', '4.993808', '6.941351', '6.251737', '4.328358', '2.739180', '2.462877', '2.304812', '3.681214', '4.902845', '6.104988', '0.830161', '3.129028', '2.727969', '1.645576', '7.808482', '0.000000', '7.275112', '3.127790', '6.731146', '4.726596', '1.023488', '7.731757', '7.900352', '5.038980', '3.000750', '2.965858', '3.688080', '2.726133', '5.752112', '9.013726', '1.914861', '5.551387', '5.453886', '0.431917', '1.986122', '4.684555', '0.173385', '3.911949', '5.897053', '7.317858', '6.185403', '1.649639', '3.814807', '3.291702', '3.516173', '1.735820', '3.940914', '5.795206', '3.341265', '1.414906', '2.258868', '5.348142', '3.269575', '1.926356', '4.291895', '0.007584'], ['23.0', '3.876429', '2.796770', '1.865413', '3.173899', '8.126329', '8.295661', '9.951907', '5.536859', '3.088644', '4.169823', '3.443305', '3.676850', '7.312399', '8.223351', '0.917940', '5.423168', '4.182061', '2.246416', '9.955501', '0.000000', '8.847608', '3.847068', '6.468718', '6.505799', '2.169393', '10.859590', '8.809187', '9.290496', '4.955708', '3.161975', '4.369468', '4.855758', '6.278932', '13.103792', '2.128446', '10.058401', '5.877123', '1.006892', '4.311251', '6.818387', '0.542254', '6.947277', '9.417463', '9.878621', '9.670252', '2.448413', '6.726754', '4.904441', '6.210752', '4.083927', '5.841790', '7.369408', '4.250368', '2.200674', '3.321724', '8.773648', '6.819811', '2.324936', '5.434804', '0.025630'], ['24.0', '3.707225', '3.331606', '1.780103', '4.049011', '8.389915', '9.685231', '10.597139', '6.308212', '3.680739', '4.353135', '3.851539', '4.665166', '7.859150', '9.060520', '1.154631', '5.532852', '4.491056', '2.421669', '11.575233', '0.000000', '10.261926', '4.432155', '8.268574', '7.291029', '2.108928', '11.741176', '10.752293', '9.655109', '5.286983', '3.902144', '5.048435', '4.906303', '7.652332', '14.072614', '2.606465', '10.133153', '7.205951', '0.951960', '4.218895', '7.358771', '0.503508', '7.173735', '9.916563', '10.950100', '10.416549', '2.698525', '6.878114', '5.282949', '6.347770', '3.883975', '6.279675', '8.363435', '4.894166', '2.362091', '3.637093', '9.161429', '6.744491', '2.750319', '6.189213', '0.023438'], ['25.0', '1.839009', '3.062748', '1.054822', '3.603980', '6.038633', '8.048736', '6.888686', '5.062912', '3.108879', '2.556118', '2.444988', '4.198769', '5.751521', '7.025258', '0.881415', '3.529558', '3.131961', '2.020193', '8.565506', '0.000000', '8.201599', '3.637790', '7.602751', '5.114530', '1.145250', '9.331590', '8.889555', '5.378977', '3.179733', '3.165368', '4.342517', '3.232169', '6.600654', '10.677322', '1.926714', '6.266906', '6.080712', '0.449025', '2.220058', '5.540691', '0.165589', '4.402219', '6.801191', '8.672749', '6.807739', '1.842915', '4.293975', '3.913628', '3.814920', '2.003027', '4.645107', '6.873991', '3.576559', '1.620100', '2.463199', '6.234047', '3.596806', '2.187784', '4.822175', '0.002237'], ['26.0', '2.597599', '1.603058', '0.824263', '2.499915', '3.785624', '4.679926', '6.197260', '2.697836', '2.267109', '3.450666', '1.750030', '2.215949', '3.596002', '5.897678', '0.539599', '3.604229', '2.084391', '1.256527', '6.102226', '0.000000', '6.318666', '2.176021', '5.171886', '4.828224', '1.208105', '5.558637', '4.974011', '4.461179', '3.084440', '3.123664', '2.758172', '2.333476', '3.586243', '7.166711', '2.494970', '6.466387', '4.373651', '0.774211', '1.875470', '3.401390', '0.225948', '3.310573', '5.038343', '4.884543', '5.565735', '1.147531', '4.003077', '2.010016', '4.529056', '1.654750', '2.862328', '3.527814', '3.628137', '1.019878', '2.260270', '4.244796', '3.358049', '1.133277', '4.073636', '0.039513'], ['27.0', '1.164894', '1.416637', '0.574953', '1.593313', '3.005511', '3.525521', '3.327445', '2.145988', '1.485642', '1.498426', '0.834251', '1.676628', '2.641102', '3.777110', '0.280842', '2.006958', '1.386305', '1.073932', '3.493258', '0.000000', '3.948419', '1.631152', '3.477203', '2.491842', '0.628447', '4.596244', '3.435094', '2.059203', '1.423511', '1.594118', '2.112816', '1.629016', '2.709047', '5.309567', '1.049212', '3.573560', '2.727271', '0.324436', '0.991088', '2.594559', '0.057111', '1.925207', '3.271083', '3.934076', '3.027734', '0.705473', '2.242225', '1.673291', '2.215882', '0.958033', '2.149205', '2.997019', '1.818945', '0.681217', '1.186761', '2.911230', '1.678317', '0.822956', '2.448521', '0.007729'], ['28.0', '1.895274', '1.085012', '0.445355', '1.944910', '2.161256', '3.041255', '4.264705', '1.543729', '1.732161', '2.758979', '0.960371', '1.476044', '2.102397', '4.523684', '0.330359', '2.640396', '1.211515', '0.858971', '4.061336', '0.000000', '4.750315', '1.423734', '4.087302', '3.653986', '0.781066', '3.473475', '3.118797', '2.365532', '2.079246', '2.654498', '1.977599', '1.354572', '2.315840', '4.668521', '2.218095', '4.646557', '3.317947', '0.607367', '0.923195', '2.033088', '0.096702', '1.815191', '3.160408', '2.891111', '3.588777', '0.601155', '2.713447', '0.947804', '3.450606', '0.758137', '1.674841', '2.014428', '2.926775', '0.544197', '1.635526', '2.434733', '1.902502', '0.621652', '3.149097', '0.037322'], ['29.0', '4.118634', '3.164841', '1.981084', '3.660813', '8.815306', '9.307862', '10.865441', '6.162948', '3.492719', '4.535613', '3.793052', '4.226376', '8.009583', '9.101119', '1.046667', '5.867216', '4.574584', '2.473824', '11.127069', '0.000000', '9.919283', '4.301046', '7.475167', '7.210935', '2.312468', '11.941834', '9.991268', '10.036006', '5.403245', '3.614191', '4.900214', '5.232890', '7.129643', '14.375711', '2.427085', '10.842269', '6.693880', '1.068023', '4.591157', '7.482010', '0.567588', '7.509240', '10.257488', '10.921777', '10.579372', '2.691432', '7.270418', '5.370439', '6.719176', '4.320798', '6.400702', '8.200236', '4.753397', '2.404778', '3.655849', '9.530052', '7.285965', '2.610633', '6.064867', '0.026982'], ['30.0', '2.365439', '2.273578', '1.281251', '2.520710', '5.969882', '6.580364', '6.957139', '4.416734', '2.336351', '2.579594', '2.630221', '3.118431', '5.500847', '5.775814', '0.763992', '3.576725', '3.122782', '1.667383', '7.633766', '0.000000', '6.542694', '3.011204', '5.143103', '4.547738', '1.426761', '8.202224', '7.258183', '6.693783', '3.449846', '2.212938', '3.356305', '3.458920', '5.176141', '9.658960', '1.341617', '6.611331', '4.546148', '0.575695', '3.007381', '5.144657', '0.357967', '5.019937', '6.802819', '7.669942', '6.972384', '1.887581', '4.581801', '3.824000', '3.959998', '2.819555', '4.402428', '5.891866', '2.912388', '1.664109', '2.328495', '6.427626', '4.680464', '1.912452', '3.882525', '0.007390'], ['31.0', '4.683868', '3.268573', '2.112787', '3.767778', '9.221168', '9.301915', '11.385590', '6.005640', '3.730915', '5.214260', '3.441200', '4.011056', '8.114592', '10.242280', '0.902010', '6.589261', '4.562233', '2.735755', '10.829541', '0.000000', '10.585005', '4.352323', '7.864857', '7.716128', '2.492334', '12.578527', '9.342922', '9.538010', '5.491121', '4.075345', '5.228614', '5.484265', '6.873144', '15.233788', '2.881357', '12.118106', '6.913782', '1.276003', '4.554945', '7.660435', '0.537710', '7.436601', '10.638403', '11.020051', '10.668099', '2.520830', '7.812157', '5.217011', '7.623542', '4.359249', '6.506946', '8.095468', '5.266290', '2.338180', '3.859355', '9.699526', '7.381361', '2.397322', '6.664497', '0.039796'], ['32.0', '2.431527', '3.064357', '1.289179', '3.247301', '6.655941', '7.494129', '6.907162', '4.614347', '3.094047', '3.003068', '1.647571', '3.486289', '5.725074', '8.000478', '0.538231', '4.250227', '2.965749', '2.370178', '7.104481', '0.000000', '8.210820', '3.478793', '7.162211', '5.054239', '1.343209', '10.093156', '7.092713', '4.198612', '2.866122', '3.159020', '4.528762', '3.579689', '5.688102', '11.532657', '1.996337', '7.591362', '5.584772', '0.668254', '2.134200', '5.644570', '0.111366', '4.098633', '7.022626', '8.547596', '6.269668', '1.474558', '4.754461', '3.675256', '4.584562', '2.118622', '4.671576', '6.505991', '3.646673', '1.457728', '2.431131', '6.314599', '3.553853', '1.722552', '5.089446', '0.012040'], ['33.0', '1.629045', '1.178091', '0.906627', '1.213368', '3.838665', '3.779967', '4.483546', '2.705862', '1.186343', '1.541714', '1.876448', '1.686955', '3.516644', '3.069035', '0.487586', '2.211055', '2.060381', '0.911607', '4.722311', '0.000000', '3.515196', '1.735610', '2.345360', '2.649130', '1.008170', '4.856955', '4.295572', '5.017918', '2.346549', '0.972771', '1.764409', '2.319487', '2.922460', '5.803874', '0.539795', '4.200629', '2.345401', '0.379050', '2.294941', '3.210762', '0.315728', '3.564192', '4.405728', '4.666788', '4.628838', '1.299100', '3.007457', '2.531812', '2.451391', '2.169305', '2.794358', '3.544561', '1.533250', '1.130332', '1.430904', '4.284226', '3.478601', '1.202756', '2.055843', '0.003589'], ['34.0', '2.906765', '3.130424', '1.370331', '3.429571', '6.891681', '7.586696', '7.505341', '4.547414', '3.324671', '3.631404', '1.538418', '3.421743', '5.834421', '8.887871', '0.486657', '4.828783', '3.008059', '2.522577', '7.223789', '0.000000', '8.869631', '3.552430', '7.612572', '5.625107', '1.497635', '10.473390', '6.879355', '4.128826', '3.098148', '3.635937', '4.775542', '3.756573', '5.623852', '12.143086', '2.462471', '8.611671', '5.918395', '0.838626', '2.171552', '5.782715', '0.108590', '4.159911', '7.378701', '8.633191', '6.574100', '1.411152', '5.232376', '3.572772', '5.364559', '2.171299', '4.764758', '6.446914', '4.174109', '1.440418', '2.669509', '6.488100', '3.746950', '1.623734', '5.626414', '0.023763'], ['35.0', '2.763736', '1.929091', '1.160701', '2.637661', '5.202731', '6.058229', '7.432038', '3.931590', '2.371794', '3.288037', '2.792598', '2.914335', '5.015799', '5.876535', '0.817767', '3.819174', '2.968355', '1.406673', '7.962189', '0.000000', '6.759786', '2.772659', '5.244748', '5.163499', '1.498279', '7.077478', '6.992827', '7.063427', '3.880625', '2.761877', '3.071122', '3.183607', '4.797649', '8.817023', '2.006221', '7.023429', '4.751133', '0.730343', '2.996160', '4.618232', '0.404265', '4.954045', '6.561288', '6.748120', '7.291233', '1.838671', '4.760300', '3.298135', '4.590047', '2.700221', '3.974853', '5.082359', '3.497073', '1.568835', '2.564670', '5.995158', '4.850477', '1.780443', '4.125098', '0.025675'], ['36.0', '4.684337', '3.259746', '2.005462', '4.028412', '8.872768', '9.416159', '11.577193', '5.992072', '3.866979', '5.427299', '3.595722', '4.200358', '7.988247', '10.346553', '0.989914', '6.586458', '4.538209', '2.644723', '11.319275', '0.000000', '10.930288', '4.384948', '8.340872', '8.057059', '2.455732', '12.235933', '9.753859', '9.673497', '5.671005', '4.431807', '5.249065', '5.309402', '7.080857', '14.974603', '3.214846', '12.062713', '7.299928', '1.287894', '4.457730', '7.520216', '0.536020', '7.380269', '10.537923', '10.857901', '10.863641', '2.553178', '7.789392', '5.075507', '7.748664', '4.187918', '6.390330', '8.000977', '5.586009', '2.323559', '3.981240', '9.516602', '7.309588', '2.458242', '6.869808', '0.044675'], ['37.0', '3.229481', '3.659579', '1.428767', '4.585466', '7.578973', '9.641245', '9.543939', '5.847217', '4.091218', '4.372422', '2.843802', '4.805001', '7.046752', '9.958054', '0.978133', '5.368518', '3.848496', '2.619573', '10.578050', '0.000000', '10.959606', '4.417690', '9.733101', '7.284554', '1.717447', '11.617008', '10.163197', '6.678801', '4.407052', '4.682374', '5.520239', '4.210860', '7.642588', '13.735600', '3.231434', '9.550539', '7.823243', '0.896364', '2.868617', '6.812171', '0.232544', '5.530329', '8.844251', '10.361029', '8.908397', '2.125738', '6.139763', '4.451904', '6.168397', '2.611251', '5.686359', '7.946999', '5.317567', '1.925085', '3.449161', '7.817185', '4.896440', '2.421973', '6.736513', '0.028043'], ['38.0', '3.358048', '2.620591', '1.409037', '3.851185', '6.621073', '8.617407', '10.106024', '5.642344', '3.247891', '4.190375', '4.343991', '4.458069', '6.820924', '7.344998', '1.350101', '4.706849', '4.142373', '1.668431', '11.848801', '0.000000', '9.217078', '3.878357', '7.474034', '7.092359', '1.899238', '9.074234', '10.749471', '10.400054', '5.501013', '3.794998', '4.052117', '4.075218', '7.153552', '11.383553', '2.714407', '8.650904', '6.829781', '0.846672', '4.117668', '6.192259', '0.585161', '6.928318', '8.757297', '9.238487', '10.287527', '2.748294', '6.157467', '4.626965', '5.734435', '3.573074', '5.371993', '7.144410', '4.738636', '2.238947', '3.519079', '8.073635', '6.610803', '2.737433', '5.448816', '0.028815'], ['39.0', '3.185897', '1.290152', '1.200683', '2.003311', '4.673136', '4.769065', '7.416034', '3.170146', '1.905304', '3.567461', '2.767430', '2.075806', '4.462608', '5.180498', '0.713663', '3.897958', '2.758073', '1.089261', '7.178127', '0.000000', '5.767496', '2.216808', '3.796197', '4.969564', '1.641597', '5.783312', '5.566325', '7.505657', '4.041974', '2.437140', '2.338686', '3.085827', '3.606370', '7.660110', '1.960840', '7.282772', '3.834097', '0.866528', '3.267163', '4.001384', '0.502640', '5.013606', '6.223221', '5.526346', '7.190435', '1.725961', '4.865000', '2.847701', '4.881314', '2.968410', '3.498807', '3.960198', '3.316796', '1.467371', '2.509126', '5.667965', '5.293786', '1.456445', '3.639429', '0.037235']])

    def test_double_line_integral(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = FluxCalculator(FluxCalculator.DOUBLE_LINE_INTEGRAL, ['M', 'H'], f,
                                        self.names, self.sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '3.235374', '2.640099', '1.535066', '3.426236', '7.055411', '8.359176', '9.545241', '5.592207', '3.034898', '3.753379', '4.002081', '4.144828', '6.889049', '7.143791', '1.197029', '4.562218', '4.110415', '1.822641', '10.847684', '0.000000', '8.610576', '3.802598', '6.761009', '6.394817', '1.881821', '9.551545', '10.029510', '9.883077', '5.086047', '3.201944', '4.010600', '4.263686', '6.777729', '11.637944', '2.138523', '8.490537', '6.172875', '0.797235', '4.139693', '6.290060', '0.567494', '6.846193', '8.751650', '9.373334', '9.748581', '2.640598', '6.057292', '4.780412', '5.403551', '3.731163', '5.477742', '7.223929', '4.170678', '2.211102', '3.229103', '8.192258', '6.497028', '2.610029', '5.063315', '0.018045'], ['1.0', '1.426988', '0.895235', '0.421143', '1.598155', '2.201363', '3.010006', '3.775030', '1.813321', '1.377138', '1.952620', '1.366702', '1.577500', '2.333588', '3.239777', '0.395705', '2.033106', '1.303884', '0.596059', '4.444949', '0.000000', '3.815321', '1.223595', '3.129474', '3.031429', '0.677856', '3.237594', '3.672047', '3.130613', '2.030359', '1.836376', '1.691683', '1.371633', '2.401486', '4.095459', '1.595002', '3.589169', '2.732782', '0.390574', '1.178335', '2.127717', '0.110700', '2.103295', '3.090740', '2.984324', '3.614504', '0.881957', '2.235120', '1.376536', '2.414855', '0.965094', '1.864808', '2.373840', '2.166436', '0.730295', '1.469728', '2.712310', '1.994813', '0.893116', '2.358253', '0.020947'], ['2.0', '1.471851', '1.967688', '0.700488', '2.500199', '3.874425', '5.324424', '4.938404', '3.269133', '2.141176', '2.029944', '1.679548', '2.752081', '3.763050', '4.899439', '0.601053', '2.517039', '2.061158', '1.306346', '5.908039', '0.000000', '5.664362', '2.403072', '5.237254', '3.732306', '0.834000', '5.984704', '5.879885', '3.753230', '2.310007', '2.381835', '2.845925', '2.154634', '4.341247', '7.064475', '1.541814', '4.537945', '4.219940', '0.382819', '1.537686', '3.609582', '0.127343', '3.015333', '4.590587', '5.608958', '4.819373', '1.230926', '3.044937', '2.456261', '2.886906', '1.382172', '3.019201', '4.397766', '2.643226', '1.071388', '1.771070', '4.152527', '2.529072', '1.425387', '3.398572', '0.008145'], ['3.0', '1.522359', '1.087208', '0.700236', '1.404841', '3.072271', '3.491816', '4.221877', '2.335554', '1.293002', '1.760099', '1.677738', '1.682757', '2.954796', '3.172625', '0.479474', '2.117362', '1.793569', '0.792252', '4.494222', '0.000000', '3.703173', '1.593091', '2.781013', '2.832180', '0.873048', '4.081179', '4.081056', '4.311509', '2.230584', '1.374388', '1.719215', '1.884018', '2.764822', '5.099981', '0.993424', '3.879302', '2.569663', '0.385631', '1.854181', '2.700993', '0.259677', '2.998141', '3.863406', '4.008447', '4.203199', '1.114682', '2.725153', '2.024373', '2.494456', '1.685457', '2.334732', '2.990075', '1.869067', '0.946202', '1.417955', '3.563723', '2.934884', '1.063654', '2.228569', '0.010759'], ['4.0', '1.770046', '2.020498', '0.947614', '2.613180', '4.722771', '6.127430', '6.235992', '4.077508', '2.180555', '2.204188', '2.763319', '3.207691', '4.759625', '4.772697', '0.887662', '2.760686', '2.786952', '1.261213', '7.712164', '0.000000', '6.007727', '2.765453', '5.108237', '4.220529', '1.123309', '6.654419', '7.495251', '6.443622', '3.283015', '2.183050', '2.898769', '2.760174', '5.141131', '7.938994', '1.376503', '5.108466', '4.519409', '0.414316', '2.584556', '4.377743', '0.330858', '4.487406', '5.783129', '6.686859', '6.473973', '1.835140', '3.772225', '3.352014', '3.188576', '2.281695', '3.775194', '5.292425', '2.718744', '1.517808', '2.108228', '5.465792', '4.036743', '1.927416', '3.457246', '0.002695'], ['5.0', '2.831376', '1.695866', '1.343233', '2.076759', '5.495332', '5.538541', '7.219475', '3.811319', '1.949508', '2.960987', '2.775301', '2.446056', '5.057077', '5.210276', '0.725714', '3.678645', '3.018326', '1.344512', '7.217266', '0.000000', '5.804916', '2.581529', '3.939643', '4.538189', '1.591932', '6.948262', '6.201332', '7.487869', '3.774245', '1.997020', '2.696452', '3.433303', '4.271782', '8.669100', '1.397376', '6.899370', '3.885338', '0.712294', '3.414437', '4.606442', '0.484589', '5.300874', '6.648397', '6.669909', '7.253651', '1.866594', '4.786735', '3.461091', '4.305181', '3.227486', '4.010683', '4.920987', '2.848368', '1.621634', '2.331900', '6.271280', '5.276332', '1.678606', '3.515839', '0.017263'], ['6.0', '1.226546', '1.568268', '0.759737', '1.525621', '3.648897', '3.596108', '3.472028', '2.303343', '1.468895', '1.484202', '0.761612', '1.589915', '2.982148', '4.094363', '0.219815', '2.216173', '1.549482', '1.267189', '3.261632', '0.000000', '3.979850', '1.764395', '3.218756', '2.315786', '0.721868', '5.344836', '3.205031', '2.143989', '1.406925', '1.304193', '2.302374', '1.941620', '2.714992', '6.028445', '0.850733', '3.908052', '2.571892', '0.355783', '1.174337', '2.951496', '0.078869', '2.130398', '3.629034', '4.391036', '3.017270', '0.696172', '2.459204', '1.934598', '2.315201', '1.191460', '2.431250', '3.271457', '1.639772', '0.734205', '1.117177', '3.278856', '1.968778', '0.782769', '2.500913', '0.001710'], ['7.0', '2.732233', '1.696659', '0.851014', '2.410331', '3.988016', '4.483162', '5.872752', '2.462965', '2.333819', '3.562550', '1.202211', '1.964986', '3.454954', '6.461906', '0.359489', '3.878064', '1.908689', '1.475820', '5.254393', '0.000000', '6.482609', '2.120455', '5.283635', '4.743265', '1.188220', '6.022588', '4.148525', '3.310693', '2.732469', '3.263066', '2.975898', '2.365787', '3.226544', '7.522173', '2.577107', '6.899257', '4.271467', '0.831915', '1.559071', '3.367135', '0.144841', '2.836495', '4.945950', '4.788577', '5.025046', '0.886283', '4.050042', '1.825504', '4.773979', '1.456568', '2.821830', '3.386723', '3.722503', '0.870779', '2.205972', '4.052998', '2.912565', '0.897492', '4.249804', '0.044600'], ['8.0', '1.653995', '0.929782', '0.546936', '1.449741', '2.306925', '2.757916', '3.900003', '1.596245', '1.327738', '2.169072', '1.022859', '1.254579', '2.199257', '3.564697', '0.291188', '2.297469', '1.280237', '0.744557', '3.548251', '0.000000', '3.714310', '1.281784', '2.934089', '2.976662', '0.809658', '3.268147', '2.776188', '2.800992', '1.882040', '1.816083', '1.626013', '1.453663', '2.054583', '4.422444', '1.537412', '4.062665', '2.535425', '0.501184', '1.222366', '2.099651', '0.151174', '2.080962', '3.120360', '2.984037', '3.355446', '0.682082', '2.487499', '1.184686', '2.821599', '1.084894', '1.691594', '2.039598', '2.187796', '0.624613', '1.385634', '2.614598', '2.214372', '0.631841', '2.483397', '0.025857'], ['9.0', '3.431937', '1.984072', '1.261664', '2.964382', '5.474823', '6.463561', '8.579826', '4.060729', '2.699040', '4.141395', '3.022630', '3.043828', '5.335069', '6.834361', '0.872771', '4.530746', '3.178107', '1.504865', '8.811389', '0.000000', '7.701209', '2.947582', '5.950490', '6.148213', '1.752996', '7.434288', '7.370623', '7.780594', '4.479176', '3.454455', '3.319017', '3.462899', '5.015535', '9.594097', '2.633124', '8.353249', '5.378558', '0.943013', '3.294445', '4.904072', '0.453256', '5.422914', '7.240025', '7.049934', '8.229092', '1.957524', '5.500447', '3.330321', '5.626995', '2.951924', '4.202343', '5.197140', '4.272095', '1.669182', '3.012031', '6.482051', '5.448780', '1.839034', '4.825448', '0.041378'], ['10.0', '1.465736', '1.433907', '0.803363', '1.606575', '3.768561', '4.295208', '4.506337', '2.877822', '1.469426', '1.507358', '1.693076', '2.016309', '3.633196', '3.503499', '0.472706', '2.189006', '1.907575', '1.013387', '5.154870', '0.000000', '3.962484', '1.866147', '3.228481', '2.905740', '0.933691', '5.189745', '4.737390', '4.253966', '2.164952', '1.354028', '2.067937', '2.202146', '3.327021', '6.115137', '0.780912', '4.196145', '2.861449', '0.361583', '1.879652', '3.412747', '0.186015', '3.197059', '4.274030', '4.903726', '4.436418', '1.246165', '2.808586', '2.433405', '2.375000', '1.726442', '2.838772', '3.881428', '1.702183', '1.103094', '1.532609', '4.170108', '2.947405', '1.268391', '2.419627', '0.003766'], ['11.0', '2.109879', '1.773568', '0.709933', '2.220558', '3.802394', '4.490851', '4.946000', '2.491688', '2.321054', '2.880856', '1.012925', '2.082369', '3.348091', '6.009444', '0.339445', '3.225658', '1.779997', '1.528395', '4.797096', '0.000000', '6.014073', '2.064807', '5.287632', '4.054745', '0.952085', '6.146565', '4.394627', '2.587503', '2.271271', '2.887139', '2.946118', '2.124041', '3.215787', '7.199197', '2.102090', '5.900943', '3.957225', '0.669016', '1.165538', '3.269490', '0.059820', '2.399481', '4.514067', '4.717368', '4.122116', '0.825429', '3.509782', '1.877450', '4.057383', '1.032203', '2.768517', '3.550525', '3.258274', '0.809042', '1.886850', '3.723027', '2.283565', '0.952943', '3.893779', '0.029047'], ['12.0', '2.496392', '2.519548', '1.213139', '3.042088', '5.838878', '6.953755', '7.314310', '4.395926', '2.742708', '3.053324', '2.339573', '3.332251', '5.465412', '6.781246', '0.720610', '3.900268', '2.977792', '1.828080', '7.856698', '0.000000', '7.402705', '3.169573', '6.267475', '5.146684', '1.432823', '8.449384', '7.342661', '5.852959', '3.414403', '2.914783', '3.748987', '3.358095', '5.436595', '10.139525', '1.969385', '7.121401', '5.252860', '0.676352', '2.603584', '5.220189', '0.255498', '4.634107', '6.781370', '7.764352', '6.885275', '1.729960', '4.666970', '3.513258', '4.428131', '2.406460', '4.325545', '5.911906', '3.485276', '1.563741', '2.511662', '6.212445', '4.276211', '1.831022', '4.594342', '0.015536'], ['13.0', '0.289851', '0.108376', '0.154742', '0.134660', '0.549136', '0.535372', '0.774120', '0.417692', '0.114438', '0.229049', '0.385974', '0.228746', '0.546939', '0.290689', '0.092865', '0.314386', '0.342887', '0.072808', '0.802534', '0.000000', '0.403319', '0.233046', '0.178248', '0.412412', '0.186013', '0.567416', '0.670266', '1.053281', '0.433638', '0.077980', '0.165558', '0.377979', '0.422491', '0.779684', '0.042493', '0.620073', '0.285668', '0.057768', '0.476596', '0.474743', '0.075198', '0.687413', '0.709386', '0.687285', '0.861416', '0.251050', '0.481935', '0.409070', '0.346820', '0.453081', '0.416157', '0.504133', '0.184279', '0.205887', '0.232786', '0.725169', '0.687169', '0.206121', '0.225366', '0.000626'], ['14.0', '2.134515', '0.871396', '0.724256', '1.249650', '2.827989', '2.838634', '4.392826', '1.692463', '1.276006', '2.332336', '1.151236', '1.063417', '2.542005', '3.707631', '0.276593', '2.576942', '1.413467', '0.835000', '3.765811', '0.000000', '3.673644', '1.316198', '2.659792', '3.109194', '1.006868', '3.725801', '2.681118', '3.414734', '2.134810', '1.807501', '1.539181', '1.856685', '1.927095', '4.917698', '1.339342', '4.918754', '2.400574', '0.612706', '1.644137', '2.382727', '0.211280', '2.603113', '3.612560', '3.212078', '3.998781', '0.804451', '2.997582', '1.411898', '3.266271', '1.574054', '1.996914', '2.241759', '2.162431', '0.745508', '1.540750', '3.227745', '2.753419', '0.669437', '2.466226', '0.029756'], ['15.0', '0.847834', '0.751487', '0.448202', '1.109247', '2.268760', '3.022280', '3.275868', '2.239564', '0.843666', '0.906493', '2.166574', '1.735100', '2.498024', '1.404029', '0.684053', '1.116794', '1.670841', '0.333314', '4.799440', '0.000000', '2.656374', '1.342503', '1.996694', '2.062145', '0.567406', '2.835880', '4.627132', '4.911463', '2.156681', '0.835960', '1.068275', '1.395302', '2.762616', '3.295076', '0.405555', '2.157044', '2.122453', '0.125599', '1.813261', '2.118903', '0.314676', '2.899586', '3.041564', '3.253956', '4.019801', '1.275382', '1.956802', '2.047316', '1.370860', '1.533420', '2.025052', '2.750783', '1.212940', '0.952389', '1.142984', '3.034733', '2.675639', '1.266664', '1.248926', '0.000439'], ['16.0', '2.012737', '1.068701', '0.670722', '1.514506', '2.907376', '3.287855', '4.457495', '1.929388', '1.530134', '2.413275', '1.218243', '1.426808', '2.680111', '4.097903', '0.336550', '2.633425', '1.545504', '0.934329', '4.098986', '0.000000', '4.245745', '1.495272', '3.343717', '3.354591', '0.952144', '4.108915', '3.334411', '3.370425', '2.194302', '2.056600', '1.867128', '1.832834', '2.346775', '5.304647', '1.561538', '4.874171', '2.821582', '0.579484', '1.535449', '2.520249', '0.186592', '2.571962', '3.798030', '3.584241', '4.040519', '0.856835', '3.026460', '1.546829', '3.315555', '1.420599', '2.133747', '2.550267', '2.462372', '0.774090', '1.613490', '3.282766', '2.626435', '0.797356', '2.766207', '0.028399'], ['17.0', '1.935617', '1.180180', '0.971733', '1.354121', '3.874673', '3.809564', '5.011340', '2.635566', '1.348501', '1.964917', '1.914210', '1.621813', '3.571541', '3.600286', '0.498393', '2.390817', '2.089023', '0.989009', '4.904147', '0.000000', '3.826232', '1.813720', '2.634272', '2.949991', '1.124034', '4.925861', '4.280302', '5.249610', '2.605022', '1.233374', '1.771529', '2.443487', '2.883479', '6.068000', '0.747648', '4.767821', '2.543772', '0.515653', '2.413156', '3.240288', '0.345583', '3.765248', '4.620067', '4.617705', '4.945559', '1.295249', '3.357840', '2.421599', '2.987161', '2.280841', '2.814428', '3.439896', '1.789013', '1.132898', '1.521964', '4.419636', '3.746223', '1.153762', '2.366660', '0.005730'], ['18.0', '1.733593', '1.503546', '0.871677', '2.090144', '4.237672', '5.643421', '6.062038', '3.981633', '1.729620', '1.875978', '3.407602', '3.061872', '4.556657', '3.279123', '1.056327', '2.316191', '2.898828', '0.814359', '8.081929', '0.000000', '5.061659', '2.449923', '4.010892', '3.950860', '1.115848', '5.496801', '7.824393', '7.953534', '3.628127', '1.734652', '2.186668', '2.631014', '4.860135', '6.665788', '0.960612', '4.453721', '3.922956', '0.317212', '3.084219', '3.981968', '0.478849', '5.011103', '5.643987', '6.122035', '6.967076', '2.130574', '3.661756', '3.504439', '2.780139', '2.684780', '3.631806', '4.971328', '2.380465', '1.648701', '2.099077', '5.521086', '4.585666', '2.118815', '2.645285', '0.003516'], ['19.0', '1.634536', '1.908115', '0.560436', '2.683254', '3.278584', '4.857337', '4.693086', '2.691175', '2.326889', '2.554694', '1.144825', '2.519664', '3.090472', '5.605094', '0.472369', '2.732597', '1.721624', '1.345165', '5.096733', '0.000000', '6.116543', '2.241726', '5.771276', '3.997969', '0.739616', '5.481134', '5.010841', '2.457245', '2.093840', '2.963406', '2.955465', '1.826062', '3.902635', '6.699092', '2.181274', '4.742823', '4.435778', '0.481775', '1.013060', '3.036237', '0.049330', '2.274459', '4.132927', '4.866506', '4.266652', '0.876499', '2.973740', '1.788923', '3.325215', '0.924932', '2.532444', '3.647290', '3.177647', '0.791181', '1.751631', '3.389671', '1.850048', '1.092751', '3.797226', '0.019818'], ['20.0', '1.208163', '0.706644', '0.578731', '1.136069', '2.590850', '3.023913', '3.965739', '2.303582', '0.934064', '1.423727', '2.460234', '1.712802', '2.750969', '1.835050', '0.735154', '1.557770', '1.958242', '0.371662', '5.174052', '0.000000', '3.052310', '1.411580', '1.941165', '2.466375', '0.752269', '3.083721', '4.780723', '5.853298', '2.659227', '0.923935', '1.169117', '1.640979', '2.748261', '3.740538', '0.667807', '2.866965', '2.218446', '0.250626', '2.202322', '2.294882', '0.410542', '3.382358', '3.591530', '3.424913', '4.546627', '1.392232', '2.481429', '2.219265', '1.984736', '1.849663', '2.225547', '2.744699', '1.561809', '1.049938', '1.323988', '3.421200', '3.382049', '1.294142', '1.530871', '0.005559'], ['21.0', '2.407811', '1.202818', '0.845607', '1.999447', '3.549536', '4.323900', '6.169915', '2.792409', '1.761369', '2.959601', '2.477396', '2.125202', '3.631967', '4.313522', '0.729916', '3.095795', '2.317168', '0.852804', '6.527227', '0.000000', '5.203895', '2.001825', '3.867653', '4.374380', '1.253987', '4.622341', '5.394225', '6.280506', '3.429760', '2.361919', '2.055046', '2.313202', '3.481583', '6.180034', '1.852369', '5.671484', '3.701935', '0.657971', '2.515765', '3.263109', '0.404836', '4.039998', '5.012848', '4.670265', '6.029542', '1.484615', '3.881103', '2.334905', '3.958673', '2.166849', '2.826670', '3.414730', '3.025550', '1.211153', '2.160209', '4.472079', '4.199007', '1.353732', '3.216210', '0.032547'], ['22.0', '0.248080', '0.262841', '0.119150', '0.295647', '0.606742', '0.658348', '0.661835', '0.406180', '0.290608', '0.323517', '0.167639', '0.310064', '0.520212', '0.763183', '0.049716', '0.419992', '0.283133', '0.211158', '0.665530', '0.000000', '0.782667', '0.303948', '0.646035', '0.495512', '0.129589', '0.912291', '0.645979', '0.431977', '0.299504', '0.301633', '0.421638', '0.332031', '0.496431', '1.050732', '0.224018', '0.733750', '0.510355', '0.069757', '0.210446', '0.503700', '0.014328', '0.388032', '0.667786', '0.755558', '0.594638', '0.139158', '0.461675', '0.333695', '0.462654', '0.202394', '0.429782', '0.568679', '0.371969', '0.135101', '0.235344', '0.583507', '0.353028', '0.154907', '0.484510', '0.001676'], ['23.0', '1.029569', '0.905853', '0.646389', '0.841445', '2.779089', '2.609061', '2.866455', '1.872003', '0.804556', '0.961183', '1.079246', '1.143120', '2.408989', '2.224143', '0.266356', '1.565556', '1.386811', '0.702815', '2.871444', '0.000000', '2.433432', '1.214705', '1.591045', '1.686970', '0.656934', '3.556029', '2.688769', '2.986172', '1.391755', '0.580693', '1.356869', '1.612497', '2.008471', '4.192627', '0.345103', '2.799006', '1.567137', '0.226868', '1.466339', '2.253608', '0.183007', '2.272488', '2.996292', '3.356277', '2.924266', '0.806618', '1.963615', '1.748727', '1.543866', '1.443344', '1.929666', '2.492670', '0.976299', '0.737160', '0.902613', '2.892433', '2.187019', '0.761845', '1.421519', '0.000356'], ['24.0', '0.181362', '0.384192', '0.066931', '0.657831', '0.445050', '0.982657', '0.915551', '0.546767', '0.449931', '0.453078', '0.288129', '0.590367', '0.530921', '0.969030', '0.131710', '0.410298', '0.312634', '0.185607', '1.138579', '0.000000', '1.186324', '0.461752', '1.201776', '0.788366', '0.103479', '0.802413', '1.123246', '0.521793', '0.406578', '0.577881', '0.549707', '0.248011', '0.908327', '1.106627', '0.487983', '0.618411', '0.977180', '0.058539', '0.153002', '0.526787', '0.009342', '0.406326', '0.672293', '0.917657', '0.832163', '0.181122', '0.425795', '0.288246', '0.476998', '0.110575', '0.398708', '0.675488', '0.599668', '0.147891', '0.313363', '0.530598', '0.304965', '0.236824', '0.734546', '0.001791'], ['25.0', '0.835290', '0.484329', '0.260190', '0.639824', '1.221052', '1.291945', '1.745041', '0.703097', '0.687622', '1.024594', '0.320272', '0.518421', '1.078714', '1.907192', '0.082465', '1.137075', '0.523478', '0.458597', '1.568212', '0.000000', '1.807808', '0.587788', '1.498346', '1.362114', '0.374679', '1.870395', '1.168535', '0.923994', '0.794411', '0.930996', '0.839405', '0.723956', '0.851248', '2.250151', '0.688762', '2.160014', '1.171097', '0.269749', '0.435686', '1.053664', '0.025931', '0.819795', '1.450384', '1.363839', '1.404160', '0.254046', '1.216141', '0.533020', '1.452835', '0.388098', '0.865485', '1.020498', '1.032755', '0.261339', '0.661755', '1.233082', '0.865479', '0.254483', '1.238631', '0.013235'], ['26.0', '0.462740', '1.078835', '0.227787', '1.466731', '1.613491', '2.945836', '2.311353', '1.756694', '1.114265', '0.760492', '0.891896', '1.638014', '1.791354', '2.190486', '0.380900', '0.935330', '0.931275', '0.580885', '3.288453', '0.000000', '2.834678', '1.298671', '3.032175', '1.847169', '0.301265', '2.724886', '3.413030', '1.685306', '1.027922', '1.323774', '1.380941', '0.867525', '2.538923', '3.237649', '0.729004', '1.740425', '2.396539', '0.104694', '0.571288', '1.746180', '0.027776', '1.356137', '1.998509', '2.837817', '2.386476', '0.640102', '1.220202', '1.159779', '1.091683', '0.470089', '1.398292', '2.332447', '1.259508', '0.523099', '0.863345', '1.866296', '0.949481', '0.818082', '1.660193', '0.001365'], ['27.0', '1.384599', '0.743523', '0.623969', '0.986859', '2.415545', '2.676726', '3.592076', '1.799680', '0.907097', '1.415679', '1.325619', '1.164997', '2.392368', '2.323312', '0.337316', '1.716968', '1.378754', '0.557484', '3.523323', '0.000000', '2.593020', '1.175152', '1.844377', '2.278769', '0.811348', '2.984455', '2.937532', '3.642414', '1.797713', '0.973553', '1.156756', '1.589714', '2.011132', '4.016841', '0.698696', '3.289233', '1.833520', '0.353206', '1.651972', '2.191326', '0.222330', '2.574384', '3.119514', '3.136424', '3.506848', '0.912867', '2.235762', '1.547416', '2.019963', '1.544094', '1.807091', '2.271357', '1.338082', '0.788891', '1.163156', '2.980507', '2.575420', '0.802624', '1.664241', '0.010239'], ['28.0', '1.240628', '1.096700', '0.385192', '1.870573', '2.015447', '3.146992', '3.699459', '1.813936', '1.478284', '1.946478', '1.229540', '1.704925', '2.137774', '3.467019', '0.439778', '1.898017', '1.257809', '0.679299', '4.210203', '0.000000', '4.048987', '1.455536', '3.619144', '2.975083', '0.600210', '3.152918', '3.685066', '2.721551', '1.871117', '2.005319', '1.757021', '1.217254', '2.671789', '4.116098', '1.628621', '3.290928', '3.052990', '0.373116', '0.981509', '2.018204', '0.115448', '1.912889', '2.858979', '3.064184', '3.424040', '0.755623', '2.129354', '1.210580', '2.388700', '0.792603', '1.676597', '2.305092', '2.228613', '0.627778', '1.340188', '2.389475', '1.792534', '0.825034', '2.530880', '0.017988'], ['29.0', '1.453241', '0.985891', '0.434657', '1.461306', '2.181584', '2.720243', '3.365658', '1.517576', '1.362569', '1.966433', '0.824856', '1.292171', '2.009718', '3.553165', '0.263648', '2.128094', '1.126452', '0.785275', '3.322205', '0.000000', '3.762903', '1.240764', '3.165675', '2.781531', '0.645017', '3.354533', '2.751829', '2.070958', '1.615660', '1.904451', '1.714576', '1.285167', '2.041723', '4.201968', '1.506537', '3.739802', '2.571996', '0.433129', '0.881393', '1.942365', '0.080899', '1.652799', '2.811471', '2.807155', '2.960643', '0.578998', '2.228598', '1.100538', '2.596213', '0.776954', '1.628489', '2.046177', '2.160284', '0.533510', '1.300074', '2.315225', '1.646638', '0.611742', '2.421586', '0.024884'], ['30.0', '0.957601', '1.524436', '0.532469', '1.749264', '2.895804', '3.663900', '3.237271', '2.220039', '1.555951', '1.428894', '0.839255', '1.843444', '2.591141', '3.814467', '0.322940', '1.873000', '1.408751', '1.089756', '3.495658', '0.000000', '4.075845', '1.740813', '3.741008', '2.449953', '0.560843', '4.563846', '3.667892', '1.954217', '1.362143', '1.613396', '2.191971', '1.524246', '2.933039', '5.302922', '1.077923', '3.235610', '2.895046', '0.278593', '0.887395', '2.566440', '0.053428', '1.840056', '3.179525', '4.042466', '2.893205', '0.688792', '2.110561', '1.662323', '2.068599', '0.832646', '2.091556', '3.046360', '1.836904', '0.657906', '1.112923', '2.766125', '1.559957', '0.849313', '2.513244', '0.003535'], ['31.0', '3.942810', '2.477743', '1.683650', '3.133757', '7.176472', '7.525061', '9.683410', '4.876662', '2.976287', '4.416812', '3.205398', '3.313358', '6.550134', '7.993706', '0.865601', '5.313483', '3.769333', '2.005881', '9.478732', '0.000000', '8.560521', '3.507633', '6.322622', '6.556515', '2.089606', '9.593428', '7.977404', '8.660502', '4.848218', '3.438110', '3.985596', '4.396537', '5.679611', '11.920066', '2.471581', '9.855044', '5.754794', '1.067583', '3.953970', '6.102336', '0.513886', '6.386442', '8.664917', '8.734553', '9.245827', '2.207210', '6.452099', '4.198728', '6.318423', '3.706115', '5.199891', '6.420692', '4.411220', '1.973446', '3.282628', '7.932952', '6.400258', '2.053041', '5.372335', '0.037187'], ['32.0', '1.108057', '0.470970', '0.157570', '0.910686', '0.921664', '1.197111', '1.843849', '0.460853', '0.888680', '1.641042', '0.159108', '0.558706', '0.700445', '2.556885', '0.062510', '1.623376', '0.473639', '0.452070', '1.428355', '0.000000', '2.621980', '0.553543', '2.124066', '1.977519', '0.342378', '1.610075', '0.934820', '0.458475', '0.885070', '1.608227', '1.112936', '0.544970', '0.811072', '2.198441', '1.451533', '2.570331', '1.649551', '0.335041', '0.216291', '0.722994', '0.009464', '0.480938', '1.425746', '1.095347', '1.486616', '0.112780', '1.364749', '0.222178', '1.971120', '0.197185', '0.645111', '0.626201', '1.801612', '0.129674', '0.862243', '0.871023', '0.584881', '0.128453', '1.709236', '0.030486'], ['33.0', '1.313091', '2.556317', '0.631431', '3.280509', '4.094593', '6.226565', '5.099231', '3.626184', '2.712820', '2.315967', '1.453650', '3.370539', '3.903194', '6.182938', '0.651128', '2.774512', '2.136744', '1.652063', '6.239383', '0.000000', '7.061555', '2.913278', '6.954733', '4.237967', '0.713803', '6.924834', '6.633924', '2.830623', '2.192394', '3.134518', '3.622510', '2.101057', '5.222495', '8.071066', '2.088399', '4.705937', '5.329001', '0.374264', '1.075977', '3.879350', '0.034874', '2.643869', '4.751200', '6.348728', '4.763448', '1.115179', '3.105109', '2.469087', '3.164642', '0.941136', '3.175551', '4.923688', '3.290060', '0.997771', '1.870204', '4.038263', '1.973824', '1.491703', '4.249146', '0.008344'], ['34.0', '0.733863', '1.194220', '0.400367', '1.418823', '2.406415', '3.441451', '2.938625', '2.210316', '1.254654', '0.974781', '1.291122', '1.857172', '2.454216', '2.559333', '0.460078', '1.345748', '1.397919', '0.742287', '3.951556', '0.000000', '3.290264', '1.507767', '3.106244', '2.174010', '0.486732', '3.695957', '4.153300', '2.820416', '1.493742', '1.290117', '1.660701', '1.311612', '2.847497', '4.230815', '0.696968', '2.480471', '2.516484', '0.159334', '1.069330', '2.308086', '0.110492', '2.054864', '2.883338', '3.616488', '3.061854', '0.908167', '1.811480', '1.752086', '1.529425', '0.904254', '1.967439', '2.936405', '1.479735', '0.749152', '1.089805', '2.694487', '1.714400', '1.047837', '1.871815', '0.001725'], ['35.0', '0.983914', '0.828569', '0.415516', '0.979992', '2.026647', '2.113017', '2.318369', '1.265002', '0.970203', '1.199243', '0.498929', '0.925187', '1.717741', '2.661122', '0.128990', '1.497499', '0.875781', '0.703955', '2.220094', '0.000000', '2.618495', '0.957296', '2.126669', '1.741308', '0.471288', '3.014823', '1.947537', '1.350261', '1.023786', '1.091827', '1.377272', '1.150070', '1.526414', '3.508233', '0.820726', '2.702833', '1.691585', '0.281221', '0.704619', '1.664383', '0.035626', '1.280425', '2.233768', '2.397937', '2.064754', '0.437069', '1.614972', '1.025847', '1.691517', '0.703746', '1.421420', '1.813373', '1.284621', '0.438376', '0.836009', '1.971980', '1.171499', '0.466969', '1.663700', '0.008645'], ['36.0', '1.010790', '0.691532', '0.511849', '0.943386', '2.126372', '2.455235', '3.097996', '1.733888', '0.798569', '1.109369', '1.443878', '1.213093', '2.166793', '1.837025', '0.413038', '1.321267', '1.335870', '0.446655', '3.495404', '0.000000', '2.342438', '1.125947', '1.678849', '1.904052', '0.639507', '2.652238', '3.129873', '3.645007', '1.719220', '0.770198', '1.023861', '1.348893', '2.032004', '3.374661', '0.524321', '2.517463', '1.716683', '0.245972', '1.508354', '1.934421', '0.232718', '2.380597', '2.740038', '2.830929', '3.199478', '0.907153', '1.893239', '1.533994', '1.593244', '1.342481', '1.673654', '2.156258', '1.126986', '0.743037', '0.984199', '2.634165', '2.349414', '0.841377', '1.375374', '0.003785'], ['37.0', '1.716972', '0.821837', '0.492399', '1.111668', '2.227864', '2.147409', '3.165332', '1.128875', '1.196004', '2.054959', '0.495058', '0.791975', '1.797609', '3.578350', '0.119039', '2.299358', '0.940583', '0.831734', '2.565132', '0.000000', '3.401404', '1.026280', '2.626048', '2.576336', '0.701662', '3.302587', '1.723229', '1.580430', '1.452623', '1.822121', '1.530268', '1.328633', '1.375680', '4.046110', '1.387880', '4.184331', '2.117288', '0.524827', '0.825178', '1.771647', '0.072179', '1.443261', '2.646898', '2.339091', '2.640188', '0.392306', '2.338558', '0.881880', '2.858896', '0.799552', '1.496414', '1.634953', '2.046954', '0.422819', '1.245096', '2.166994', '1.590515', '0.363144', '2.285195', '0.031780'], ['38.0', '1.638983', '1.878085', '0.906753', '2.165403', '4.487651', '5.296959', '5.248060', '3.490710', '1.952546', '1.974831', '1.991262', '2.637736', '4.203883', '4.640042', '0.613896', '2.691248', '2.396936', '1.314309', '6.005651', '0.000000', '5.356607', '2.401561', '4.437780', '3.595391', '1.007255', '6.392654', '5.928328', '4.856457', '2.587434', '1.869776', '2.782605', '2.544283', '4.250707', '7.499053', '1.209202', '4.841005', '3.788126', '0.396104', '2.114148', '3.950822', '0.233163', '3.673370', '5.170151', '6.022733', '5.235208', '1.436481', '3.381896', '2.923882', '2.935793', '1.948878', '3.374744', '4.654111', '2.393074', '1.256469', '1.789212', '4.805781', '3.316594', '1.523459', '3.156726', '0.003694'], ['39.0', '1.325604', '1.204913', '0.393240', '1.708630', '2.228602', '3.066394', '3.278311', '1.654303', '1.579249', '1.995792', '0.712602', '1.544283', '2.017760', '3.980882', '0.279886', '2.127913', '1.165433', '0.924843', '3.283189', '0.000000', '4.240688', '1.426765', '3.814965', '2.877277', '0.565211', '3.694188', '3.068123', '1.657744', '1.508623', '2.133989', '2.004011', '1.245545', '2.367323', '4.552246', '1.649289', '3.638442', '2.931525', '0.401476', '0.695764', '1.988750', '0.046135', '1.495483', '2.860188', '3.101122', '2.854308', '0.536061', '2.203133', '1.124204', '2.606572', '0.617409', '1.671059', '2.245848', '2.367150', '0.498843', '1.277490', '2.247921', '1.365108', '0.646520', '2.679912', '0.022189']])

    def test_line_flux(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = FluxCalculator(FluxCalculator.LINE_FLUX, ['U', 'V'], f,
                                        self.names, self.sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '-0.288575', '-3.016929', '-1.121940', '-3.413331', '0.772482', '0.957126', '-1.497095', '1.029324', '-1.225930', '-0.182532', '2.490960', '-2.190128', '1.920657', '0.107126', '-0.762759', '-0.571430', '1.259515', '-1.979045', '3.580892', '0.000000', '-4.004627', '3.181440', '-0.917082', '3.033691', '-0.438827', '-0.457691', '0.765444', '-2.334918', '3.699124', '-2.659348', '-0.773482', '1.181739', '-2.881225', '-0.254352', '-0.857121', '-0.999113', '-0.810434', '0.780613', '-0.050731', '-0.956255', '0.014806', '1.988684', '0.110619', '1.531774', '-0.055183', '-1.922112', '-1.376733', '0.642140', '-2.174062', '-0.067922', '-1.984724', '-1.076432', '0.099794', '1.415096', '-1.048049', '0.647956', '-0.534026', '-0.902973', '-0.164259', '0.010127'], ['1.0', '-0.367014', '-3.004068', '-1.388895', '-2.975302', '0.840975', '2.090281', '-0.561201', '-0.047490', '-1.021284', '-0.000190', '3.816637', '-2.410013', '2.654193', '-0.895826', '-1.129240', '-1.146147', '0.446310', '-1.909527', '5.421408', '0.000000', '-2.634289', '2.786015', '-1.528486', '1.723360', '-1.109966', '-1.315098', '0.157118', '-3.703158', '4.599357', '-1.840049', '0.219999', '1.647691', '-2.573091', '-0.716249', '-0.586312', '-0.749485', '1.369321', '0.644203', '-0.638324', '-1.231343', '-0.269798', '2.989958', '0.406764', '2.143485', '0.855174', '-3.230529', '-1.417151', '1.251527', '-2.142740', '-0.162298', '-2.694520', '-2.258599', '0.490921', '2.414747', '-2.632207', '0.735526', '0.073088', '-0.824853', '0.368832', '0.005918'], ['2.0', '-1.883802', '-3.202337', '-1.683951', '-3.246725', '-0.084750', '2.664104', '-0.746075', '0.553942', '-0.603372', '-0.133419', '3.705999', '-2.665478', '2.450239', '-1.545227', '-1.250501', '-2.276231', '1.656508', '-1.721881', '4.939249', '0.000000', '-2.509292', '2.959612', '-1.053910', '0.486884', '-0.708542', '-1.068448', '0.750954', '-3.435356', '5.313743', '-0.624638', '-0.338493', '1.625410', '-3.235306', '-1.027068', '-0.330429', '0.790094', '1.654274', '0.989457', '-0.164050', '-1.066939', '-0.215556', '3.088614', '0.592957', '1.974743', '-0.878027', '-3.101114', '-1.025596', '0.676809', '-2.700739', '-0.140589', '-2.086955', '-2.001817', '0.653938', '2.214327', '-2.031313', '0.728899', '-0.018857', '-1.157261', '0.467061', '0.029630'], ['3.0', '-0.379096', '-4.132009', '-1.728569', '-4.054310', '0.192862', '2.281983', '-0.587598', '1.020918', '-0.948122', '0.245508', '3.713953', '-3.155931', '2.734437', '-1.705847', '-1.192558', '-2.340474', '1.513832', '-2.356547', '5.789250', '0.000000', '-3.613398', '3.509936', '-1.808988', '1.771182', '-1.364778', '-1.566116', '0.789382', '-4.422217', '5.002722', '-2.133813', '-0.306206', '1.318567', '-4.151371', '-0.867258', '-0.687652', '-0.638944', '1.728351', '0.792917', '-0.722041', '-1.659108', '-0.438720', '3.067775', '0.822607', '2.814259', '0.438510', '-3.494128', '-0.781551', '0.630860', '-1.634038', '-0.118861', '-2.453493', '-2.432249', '0.470544', '2.766480', '-3.063755', '1.155826', '0.976638', '-1.178845', '0.811699', '0.001530'], ['4.0', '-1.596598', '-4.952529', '-2.357174', '-5.242667', '1.350962', '2.800079', '-2.389165', '1.105286', '-2.487868', '-0.176866', '4.438671', '-3.571791', '4.294143', '-1.050493', '-1.223681', '-1.559204', '1.234158', '-3.476290', '5.689628', '0.000000', '-5.105486', '4.821681', '-1.892570', '3.227771', '-1.119657', '-1.412598', '0.572424', '-4.174897', '6.939909', '-2.950702', '-0.552892', '2.882931', '-4.453255', '-0.890118', '-0.900016', '-0.649842', '1.041485', '1.569197', '-0.622850', '-1.923545', '-0.215420', '3.936414', '-0.777609', '2.185503', '-0.147486', '-3.893633', '-2.665613', '1.821817', '-5.225913', '-0.193494', '-4.133811', '-2.415815', '0.150107', '3.116675', '-2.388715', '1.352303', '-0.523685', '-1.019397', '-0.027399', '0.042516'], ['5.0', '-1.722775', '-3.136385', '-1.840999', '-4.546961', '-0.123788', '2.348276', '-1.580670', '0.162064', '-1.724959', '0.085915', '3.556007', '-3.254244', '2.534421', '-0.655475', '-0.739976', '-1.807681', '0.931002', '-1.697571', '4.710515', '0.000000', '-4.335704', '2.890616', '-1.509990', '2.476085', '-0.717072', '-1.365532', '0.367915', '-3.604641', '5.254578', '-2.058357', '-1.416181', '1.726357', '-2.803748', '-0.801347', '-0.804317', '0.368899', '0.242362', '1.082183', '-0.224954', '-1.026180', '-0.312622', '3.345956', '0.049522', '0.931912', '-0.502656', '-3.143296', '-1.543273', '0.672601', '-3.076723', '-0.155778', '-2.162652', '-2.128472', '-0.287782', '2.439020', '-2.055330', '1.002764', '0.308282', '-0.446750', '-1.046329', '0.026262'], ['6.0', '-1.343319', '-3.609057', '-1.794866', '-3.707792', '1.414911', '2.245929', '-2.209293', '0.745754', '-2.022727', '-0.440380', '3.502231', '-2.628304', '3.485499', '-0.595624', '-0.903455', '-0.918404', '0.670602', '-2.753985', '4.043581', '0.000000', '-3.727986', '3.860059', '-1.467348', '2.653714', '-0.762042', '-0.777281', '0.252414', '-2.775196', '5.206111', '-2.409955', '-0.224314', '2.667801', '-3.057280', '-0.678400', '-0.681017', '-0.778160', '0.618614', '1.165138', '-0.285387', '-1.408893', '0.024398', '3.260754', '-0.767019', '1.280979', '0.025857', '-2.854971', '-2.574860', '1.941746', '-4.385528', '-0.169289', '-3.473392', '-1.579057', '0.082558', '2.212719', '-1.683645', '1.017975', '-1.183804', '-0.691220', '0.008845', '0.031283'], ['7.0', '-2.467825', '-3.226227', '-1.879589', '-4.231846', '-0.618869', '2.467007', '-1.906761', '1.137285', '-1.163166', '-0.275425', '2.969739', '-3.198762', '2.157371', '-1.106444', '-0.742477', '-2.591225', '2.024130', '-1.675192', '3.532681', '0.000000', '-3.995091', '3.241501', '-1.063061', '1.634064', '-0.334435', '-0.670949', '0.959598', '-2.726983', '4.921526', '-1.397655', '-1.655151', '1.705205', '-3.365699', '-0.924171', '-0.592100', '0.931438', '0.190405', '1.148779', '0.379842', '-0.906061', '-0.059786', '3.159701', '0.354637', '0.699618', '-1.629638', '-2.486772', '-1.257242', '0.405906', '-2.912578', '-0.108046', '-1.560538', '-1.315526', '-0.105524', '1.817902', '-1.335209', '1.023971', '-0.304363', '-0.798453', '-0.435013', '0.030321'], ['8.0', '-1.254091', '-2.877861', '-0.978602', '-3.856655', '0.252210', '1.530618', '-1.132207', '0.375028', '-1.190302', '-0.067610', '3.002315', '-2.214918', '2.091481', '-0.039099', '-0.756166', '-0.659906', '0.992242', '-1.501586', '4.259730', '0.000000', '-3.980497', '3.009747', '-1.239267', '2.325577', '-0.251987', '-0.686599', '0.416315', '-2.704447', '4.371433', '-1.777852', '-0.658276', '1.172885', '-2.555690', '-0.418814', '-0.810757', '0.114161', '-0.560142', '0.828651', '-0.202884', '-0.559846', '-0.101964', '2.044330', '-0.136788', '1.359028', '-0.922214', '-2.248267', '-1.284143', '0.652269', '-2.905801', '-0.083010', '-2.014122', '-1.488307', '-0.364285', '1.645633', '-1.235085', '0.419311', '-0.143501', '-0.758628', '-1.041521', '0.032988'], ['9.0', '-0.236228', '-4.213411', '-1.880021', '-4.103276', '1.127809', '1.766104', '-1.356232', '1.129398', '-2.150361', '0.427124', '2.844293', '-2.679205', '3.426390', '-1.296680', '-0.833883', '-1.292542', '0.738163', '-3.030845', '4.293710', '0.000000', '-3.829905', '3.411462', '-1.672235', '2.440904', '-1.405835', '-1.732649', '0.429259', '-3.753485', '4.931340', '-2.499814', '-0.250661', '1.760622', '-3.933124', '-0.582011', '-0.652646', '-0.978768', '1.680138', '1.192578', '-1.173738', '-2.012167', '-0.519326', '2.523310', '-0.820941', '2.352235', '0.715903', '-3.146769', '-1.474515', '1.026457', '-3.438752', '-0.124795', '-3.249661', '-2.273893', '0.176120', '2.848136', '-2.363170', '1.327213', '0.923166', '-0.667926', '0.443762', '0.024274'], ['10.0', '-0.622708', '-2.425832', '-1.614138', '-2.950371', '0.147534', '1.241229', '-1.017108', '0.506163', '-1.344306', '0.370990', '2.042899', '-2.221790', '1.732312', '-0.678894', '-0.636719', '-1.368946', '0.966950', '-1.592587', '2.995196', '0.000000', '-2.793535', '1.733845', '-0.556300', '1.685546', '-0.799394', '-1.326803', '0.627516', '-2.700434', '3.749647', '-1.529878', '-1.156511', '0.971958', '-2.448000', '-0.503877', '-0.438197', '0.037812', '0.708838', '0.990628', '-0.407508', '-1.227787', '-0.413264', '2.109540', '-0.021315', '1.221286', '0.142205', '-2.274900', '-0.778700', '-0.025102', '-2.000253', '-0.103329', '-1.456364', '-1.674068', '0.427862', '1.912262', '-1.313131', '0.890362', '0.748028', '-0.449467', '-0.019943', '0.016151'], ['11.0', '-0.398764', '-3.255170', '-1.713456', '-3.458947', '0.262323', '1.985860', '-0.509497', '0.324620', '-0.746445', '0.163233', '3.916605', '-2.942767', '2.137298', '-0.952881', '-1.350255', '-1.848419', '1.444557', '-1.832004', '5.910432', '0.000000', '-3.232471', '2.762175', '-1.059466', '1.911666', '-1.102971', '-1.481156', '0.849697', '-4.184005', '5.095825', '-1.990863', '-0.649151', '1.182250', '-3.241132', '-0.790272', '-0.663538', '-0.409466', '1.048446', '0.824215', '-0.379818', '-1.343015', '-0.367987', '3.180396', '1.169588', '2.537221', '0.549932', '-3.432588', '-0.893635', '0.336082', '-1.456117', '-0.148199', '-2.021695', '-2.427300', '0.951342', '2.485659', '-2.528462', '0.817080', '0.486533', '-1.187965', '0.541619', '0.000533'], ['12.0', '-1.420835', '-3.805669', '-1.829844', '-4.858100', '0.063013', '2.448057', '-1.212650', '0.358930', '-1.598371', '0.205709', '4.039771', '-3.355028', '2.870754', '-0.922883', '-1.033432', '-1.848333', '1.183789', '-2.056337', '5.799352', '0.000000', '-4.546260', '3.400584', '-1.724141', '2.446860', '-0.930252', '-1.586571', '0.521086', '-4.287714', '5.843700', '-2.157433', '-1.001657', '1.629862', '-3.511842', '-0.836270', '-0.860772', '0.167040', '0.674018', '1.111016', '-0.528498', '-1.248991', '-0.410422', '3.330548', '0.184856', '1.902329', '-0.382716', '-3.574079', '-1.390647', '0.682399', '-3.123962', '-0.151511', '-2.560040', '-2.525572', '-0.068724', '2.780601', '-2.491899', '0.987886', '0.633288', '-0.819709', '-0.676478', '0.027927'], ['13.0', '-0.710026', '-3.019417', '-1.712133', '-3.063453', '0.610495', '2.353034', '-1.109023', '0.284800', '-1.324531', '-0.068222', '3.362336', '-2.701021', '2.788841', '-1.308601', '-0.839813', '-1.771663', '0.429282', '-2.043398', '4.380972', '0.000000', '-2.625210', '2.768016', '-1.669136', '1.637911', '-1.201587', '-1.284735', '0.114270', '-3.383666', '4.270468', '-1.892275', '-0.159593', '1.990551', '-2.655567', '-0.843140', '-0.501158', '-0.691458', '1.664457', '0.709784', '-0.499121', '-1.420975', '-0.248281', '3.307776', '0.213236', '1.418950', '0.823625', '-3.082057', '-1.558833', '1.382925', '-2.213209', '-0.168089', '-2.608058', '-1.998888', '0.251796', '2.430725', '-2.664708', '1.121288', '0.043599', '-0.508415', '0.472069', '0.001855'], ['14.0', '-0.759033', '-3.094451', '-1.493152', '-4.481901', '0.731624', '1.196482', '-1.712572', '0.243506', '-2.359971', '0.360353', '2.690925', '-2.526977', '2.556952', '0.246771', '-0.520710', '-0.304318', '0.397391', '-2.040627', '3.897826', '0.000000', '-4.783210', '2.723812', '-1.249746', '3.515735', '-0.646958', '-1.383595', '0.230135', '-3.074333', '4.694105', '-2.732407', '-1.260454', '1.392489', '-2.572220', '-0.276745', '-0.954508', '-0.373477', '-0.578091', '1.187307', '-0.659424', '-1.115293', '-0.379302', '2.211640', '-0.949049', '1.109146', '-0.041262', '-2.574039', '-1.692018', '0.588701', '-3.675775', '-0.117034', '-2.484404', '-1.928925', '-0.462932', '2.219843', '-1.308304', '0.835732', '0.484225', '-0.260079', '-1.433561', '0.035523'], ['15.0', '-0.997513', '-3.780921', '-2.148531', '-3.911734', '0.903328', '2.063635', '-2.362382', '1.364845', '-1.779128', '-0.373401', '3.218069', '-3.198534', '2.916254', '-0.810798', '-0.931653', '-1.815869', '1.511389', '-2.775502', '3.868199', '0.000000', '-4.151133', '3.824496', '-1.163957', '2.994783', '-0.905479', '-0.805167', '0.875080', '-2.963732', '4.953120', '-2.918726', '-1.086762', '2.318438', '-3.669274', '-0.736741', '-0.748238', '-0.993988', '0.391297', '1.170883', '0.048333', '-1.663777', '-0.004057', '3.533730', '0.093849', '1.350280', '0.226561', '-2.823189', '-2.100672', '1.230530', '-3.098939', '-0.149692', '-2.734113', '-1.454981', '0.454674', '2.165066', '-1.746720', '1.327449', '-0.892839', '-0.883275', '0.535175', '0.008886'], ['16.0', '-0.691952', '-4.646851', '-1.916679', '-4.233120', '1.037851', '2.207267', '-2.128710', '2.006983', '-1.646075', '-0.375470', '3.246041', '-3.175046', '3.403150', '-1.383700', '-0.964973', '-1.982281', '1.663104', '-3.257454', '4.240236', '0.000000', '-4.352476', '4.701477', '-1.834848', '2.815585', '-1.102328', '-0.731193', '0.877994', '-3.225690', '4.840372', '-3.040644', '-0.366495', '2.303906', '-4.564465', '-0.744612', '-0.796296', '-1.434724', '0.929656', '1.017369', '-0.282767', '-1.865423', '-0.004538', '3.237167', '-0.011888', '2.077463', '0.193613', '-2.858581', '-1.924283', '1.571970', '-3.029868', '-0.111712', '-3.214569', '-1.442704', '0.180845', '2.308062', '-2.261872', '1.436607', '-0.623953', '-1.137409', '1.002704', '0.007233'], ['17.0', '-0.642329', '-3.508344', '-1.642456', '-4.252816', '0.600944', '1.333927', '-1.571373', '0.902413', '-1.824030', '0.231052', '2.770643', '-2.697576', '2.441964', '-0.294631', '-0.794178', '-1.004926', '1.219134', '-2.274973', '4.109602', '0.000000', '-4.491859', '3.110626', '-1.064170', '3.046525', '-0.754434', '-1.221913', '0.744865', '-3.201189', '4.763194', '-2.607918', '-1.168671', '1.314214', '-3.346659', '-0.420109', '-0.850233', '-0.489321', '-0.111098', '1.165536', '-0.467164', '-1.342946', '-0.313931', '2.364869', '-0.294321', '1.718550', '-0.094103', '-2.628704', '-1.368278', '0.400719', '-3.012508', '-0.100997', '-2.301517', '-1.810639', '0.082459', '2.167592', '-1.445760', '0.940302', '0.363089', '-0.745825', '-0.443348', '0.025039'], ['18.0', '0.701967', '-3.534188', '-1.012779', '-3.920020', '1.353454', '1.052575', '-0.733337', '0.253368', '-1.620730', '0.294572', '3.420598', '-2.285633', '2.672472', '0.068270', '-0.949217', '0.034053', '0.265445', '-2.298540', '5.600338', '0.000000', '-4.302646', '3.274654', '-1.758505', '3.550703', '-1.074793', '-1.359987', '0.188496', '-3.802015', '4.343723', '-3.234012', '0.032655', '1.138280', '-2.945578', '-0.205602', '-1.063073', '-1.720454', '-0.051791', '0.653997', '-1.010763', '-1.281077', '-0.341153', '2.116591', '-0.122955', '2.584191', '1.191234', '-2.935299', '-1.445794', '1.033923', '-2.272759', '-0.107497', '-2.967925', '-2.227545', '-0.025014', '2.366967', '-2.342492', '0.660750', '0.501195', '-0.810439', '-0.363483', '0.006242'], ['19.0', '-1.761164', '-2.173547', '-1.174999', '-2.969784', '0.268161', '1.937118', '-1.430502', '0.063842', '-1.115474', '-0.388351', '2.914799', '-2.105588', '2.066547', '-0.246980', '-0.655258', '-0.951920', '0.664347', '-1.268078', '3.365802', '0.000000', '-2.883993', '2.532782', '-1.030886', '1.638442', '-0.200945', '-0.457860', '0.173427', '-2.086009', '4.021661', '-1.250363', '-0.597271', '1.726773', '-1.787540', '-0.609150', '-0.539654', '0.331249', '-0.087761', '0.780240', '0.139482', '-0.484160', '0.056202', '2.563329', '-0.122768', '0.485700', '-0.799569', '-2.106329', '-1.664040', '1.092289', '-2.984254', '-0.125674', '-1.931762', '-1.185346', '-0.196975', '1.446768', '-1.123996', '0.494795', '-0.874744', '-0.477319', '-0.756062', '0.028633'], ['20.0', '-1.259061', '-2.159359', '-1.500129', '-2.901188', '0.008824', '1.911135', '-0.641647', '-0.322708', '-1.038179', '0.206259', '3.085952', '-2.284611', '1.902491', '-0.684360', '-0.875664', '-1.369865', '0.671658', '-1.171706', '4.232729', '0.000000', '-2.421896', '1.705565', '-0.791774', '1.146693', '-0.691981', '-1.372531', '0.300394', '-3.137824', '4.467904', '-0.941343', '-0.759450', '1.230078', '-1.936684', '-0.718164', '-0.407156', '0.598534', '0.950237', '0.892007', '-0.347587', '-0.861844', '-0.372891', '2.645619', '0.238909', '1.233058', '-0.136499', '-2.790208', '-0.993652', '0.351180', '-2.324246', '-0.149307', '-1.673543', '-2.072759', '0.403180', '2.093189', '-1.722925', '0.603602', '0.448758', '-0.507115', '-0.397751', '0.023817'], ['21.0', '-1.444653', '-2.542584', '-1.447783', '-2.821911', '-0.089485', '1.916078', '-0.404666', '0.208910', '-0.833418', '0.299493', '2.750281', '-2.037747', '2.005419', '-1.181605', '-0.952592', '-1.595727', '1.123906', '-1.379259', '3.947187', '0.000000', '-2.091744', '1.898230', '-0.682073', '0.366130', '-0.669870', '-1.348071', '0.541608', '-3.037996', '4.547173', '-0.323277', '-0.458313', '1.043799', '-2.543338', '-0.749451', '-0.230463', '0.974537', '1.538963', '1.006705', '-0.510967', '-0.978549', '-0.429739', '2.155669', '0.032418', '1.731914', '-0.679895', '-2.638538', '-0.615264', '0.141642', '-2.589318', '-0.114490', '-1.669521', '-1.995415', '0.529733', '2.060235', '-1.533409', '0.582253', '0.750148', '-0.759318', '0.037835', '0.036450'], ['22.0', '-0.743548', '-3.077202', '-0.964367', '-3.390369', '0.090163', '1.567135', '-1.092961', '1.129206', '-0.797532', '-0.205235', '2.455089', '-2.277743', '1.956058', '-0.768012', '-0.615624', '-1.418390', '1.249886', '-1.687748', '3.517592', '0.000000', '-3.473638', '3.230040', '-1.527220', '1.927097', '-0.514143', '-0.429485', '0.564000', '-2.431422', '3.286062', '-1.922863', '-0.417655', '1.140522', '-3.014054', '-0.496082', '-0.703441', '-0.526475', '0.044518', '0.513493', '-0.104448', '-0.820493', '-0.012445', '2.034921', '0.337675', '1.380768', '-0.470974', '-1.923948', '-0.943506', '0.776297', '-1.626154', '-0.050661', '-1.766910', '-1.049887', '-0.332182', '1.463828', '-1.650476', '0.750900', '-0.107455', '-0.814097', '-0.029093', '0.006800'], ['23.0', '-1.528462', '-4.684070', '-2.166420', '-5.217868', '0.940138', '2.890817', '-2.615483', '1.265387', '-2.567273', '-0.266277', '3.744862', '-3.644956', '4.165932', '-1.350551', '-0.663654', '-1.956088', '0.735940', '-3.235823', '4.517596', '0.000000', '-5.072212', '4.719640', '-2.569185', '3.236365', '-1.177442', '-1.183033', '0.169186', '-3.660463', '5.586910', '-3.167183', '-0.615059', '2.944572', '-4.109099', '-0.890367', '-0.927069', '-0.966667', '1.015608', '1.194921', '-0.545887', '-1.805086', '-0.139902', '3.887232', '-0.840681', '1.202167', '0.037218', '-3.399798', '-2.592350', '2.135061', '-4.416533', '-0.168667', '-3.840110', '-1.934572', '-0.682815', '2.887496', '-2.720716', '1.641780', '-0.381964', '-0.481883', '-0.241515', '0.024660'], ['24.0', '-0.919916', '-3.891301', '-2.045581', '-4.985149', '0.321497', '2.095467', '-1.407437', '0.427986', '-1.802390', '0.293135', '4.068326', '-3.542701', '2.760159', '-0.570209', '-1.131372', '-1.677411', '1.332865', '-2.268897', '5.988363', '0.000000', '-4.986957', '3.366318', '-1.422055', '3.219364', '-1.044911', '-1.729664', '0.758191', '-4.484300', '6.019026', '-2.822235', '-1.394316', '1.555911', '-3.652859', '-0.733400', '-0.986708', '-0.286413', '0.278034', '1.228775', '-0.499477', '-1.471997', '-0.448810', '3.462634', '0.346944', '2.100692', '0.120548', '-3.696904', '-1.480510', '0.441325', '-2.939830', '-0.161018', '-2.547066', '-2.635672', '0.272922', '2.869065', '-2.396024', '1.072905', '0.630656', '-0.899548', '-0.556312', '0.019887'], ['25.0', '0.335163', '-3.983376', '-1.392370', '-3.660994', '0.738350', '1.568577', '-0.658097', '1.202256', '-1.251579', '0.332336', '2.849432', '-2.528721', '2.735709', '-1.352432', '-0.907497', '-1.478389', '1.003340', '-2.552703', '4.722428', '0.000000', '-3.485512', '3.363160', '-1.806252', '2.143636', '-1.358064', '-1.415515', '0.566250', '-3.758700', '4.044390', '-2.453372', '0.043213', '1.167059', '-3.880594', '-0.538326', '-0.700144', '-1.295241', '1.456123', '0.697351', '-0.989875', '-1.712118', '-0.421242', '2.233755', '0.149882', '2.711369', '0.866178', '-2.882077', '-0.840881', '0.791137', '-1.712932', '-0.083515', '-2.623400', '-2.053446', '0.220303', '2.476070', '-2.667128', '1.116402', '1.004858', '-0.944388', '0.802103', '0.000766'], ['26.0', '-0.692067', '-2.586221', '-1.093991', '-2.638042', '0.973922', '1.185416', '-0.998113', '0.445180', '-1.586467', '0.198703', '1.948216', '-1.400960', '2.454940', '-0.523559', '-0.561227', '-0.263730', '0.261206', '-1.927233', '2.683637', '0.000000', '-2.408161', '2.225055', '-0.937072', '1.429638', '-0.631193', '-1.009286', '0.094958', '-2.129633', '3.623669', '-1.201551', '0.048377', '1.343709', '-2.220242', '-0.327821', '-0.374116', '-0.177979', '0.896605', '0.944649', '-0.788711', '-1.080148', '-0.283036', '1.432212', '-1.125982', '1.371259', '-0.081010', '-1.954509', '-1.290218', '0.888064', '-3.383039', '-0.089959', '-2.413800', '-1.439334', '-0.010028', '1.749103', '-1.054811', '0.607388', '0.231781', '-0.391277', '-0.185549', '0.038305'], ['27.0', '1.567859', '-3.524208', '-0.958757', '-4.197089', '1.754943', '0.119429', '-1.292214', '0.611660', '-2.646312', '0.674810', '1.972726', '-2.002786', '2.755168', '0.469544', '-0.285595', '0.764006', '-0.474710', '-2.680920', '3.787269', '0.000000', '-4.977701', '2.952707', '-1.928267', '4.686471', '-1.241925', '-1.547162', '-0.108451', '-3.187411', '3.163233', '-4.183632', '-0.330757', '0.962150', '-2.791936', '0.211217', '-1.205949', '-2.531436', '-0.556636', '0.714515', '-1.454294', '-1.594218', '-0.492877', '1.265566', '-1.253207', '1.951062', '1.860795', '-2.266549', '-1.531519', '0.958028', '-2.481834', '-0.066784', '-3.065617', '-1.896706', '-0.707877', '2.299232', '-1.955475', '1.000970', '1.155781', '-0.125871', '-0.836440', '0.003830'], ['28.0', '-0.445773', '-1.739752', '-0.630279', '-2.475832', '0.440875', '0.619249', '-0.364822', '-0.202560', '-1.272804', '0.483597', '1.604596', '-1.051875', '1.544452', '0.080575', '-0.396719', '0.198424', '0.008432', '-1.035326', '2.673069', '0.000000', '-2.357055', '1.262046', '-0.696006', '1.426932', '-0.376962', '-1.091519', '-0.022726', '-2.031444', '2.960284', '-0.931044', '-0.278659', '0.492813', '-1.369942', '-0.115267', '-0.446138', '0.229755', '0.112607', '0.723938', '-0.790741', '-0.553233', '-0.397692', '0.768423', '-0.859743', '1.133798', '-0.220485', '-1.612427', '-0.668693', '0.165104', '-2.474702', '-0.060144', '-1.538682', '-1.475644', '-0.247350', '1.440671', '-0.769566', '0.228893', '0.853212', '-0.191953', '-1.005038', '0.036139'], ['29.0', '-0.918608', '-4.389697', '-2.344555', '-4.630768', '1.233608', '2.237805', '-2.317812', '1.176278', '-2.415883', '-0.029191', '3.589969', '-3.379024', '3.672237', '-0.959652', '-0.981040', '-1.589263', '1.107904', '-3.244148', '4.641616', '0.000000', '-4.663377', '4.057170', '-1.565291', '3.299405', '-1.226929', '-1.430551', '0.631749', '-3.751872', '5.821627', '-3.137463', '-0.888734', '2.497998', '-4.049793', '-0.748715', '-0.829437', '-1.023179', '0.936544', '1.412283', '-0.564077', '-2.004936', '-0.264641', '3.614124', '-0.560699', '1.825292', '0.480107', '-3.440930', '-2.299525', '1.404862', '-4.103229', '-0.174188', '-3.496865', '-2.137280', '0.291070', '2.853185', '-2.211899', '1.471732', '-0.217208', '-0.771698', '0.248083', '0.023378'], ['30.0', '-0.167804', '-3.541722', '-1.539294', '-3.625174', '1.164483', '1.447087', '-1.917197', '1.206177', '-1.859480', '-0.153770', '2.555231', '-2.542346', '2.814471', '-0.548102', '-0.612542', '-0.962352', '0.733358', '-2.644619', '3.383155', '0.000000', '-3.971415', '3.534678', '-1.546866', '3.146572', '-0.953123', '-0.803373', '0.420316', '-2.653025', '3.819503', '-3.068377', '-0.495811', '1.889130', '-3.217823', '-0.419487', '-0.809682', '-1.548365', '0.240093', '0.848309', '-0.407864', '-1.523599', '-0.075422', '2.575439', '-0.389318', '1.386403', '0.690578', '-2.342268', '-1.856433', '1.353651', '-2.688994', '-0.106647', '-2.793039', '-1.314939', '-0.082959', '1.979318', '-1.794074', '1.175196', '-0.387998', '-0.585094', '0.242493', '0.004596'], ['31.0', '-2.860312', '-2.966781', '-2.199994', '-3.984322', '-0.606552', '2.719192', '-1.988511', '0.819201', '-1.306809', '-0.236885', '3.102825', '-3.241792', '2.261114', '-1.233217', '-0.815920', '-2.754819', '1.924633', '-1.631911', '3.430351', '0.000000', '-3.511725', '2.821850', '-0.751563', '1.227998', '-0.385588', '-0.902230', '0.916558', '-2.782027', '5.368804', '-0.978049', '-1.800236', '1.955158', '-3.091279', '-1.063035', '-0.419801', '1.370517', '0.665276', '1.353675', '0.392287', '-1.002012', '-0.130790', '3.505465', '0.225475', '0.491816', '-1.618654', '-2.734113', '-1.414328', '0.383254', '-3.393845', '-0.147327', '-1.600365', '-1.531195', '0.202332', '2.002367', '-1.260996', '1.068167', '-0.343858', '-0.687137', '-0.378443', '0.038063'], ['32.0', '0.283476', '-4.474718', '-1.625510', '-4.587857', '1.257789', '1.122305', '-2.130291', '1.954074', '-2.386349', '0.175441', '2.118472', '-2.774665', '3.183296', '-0.708699', '-0.464607', '-0.976179', '0.953084', '-3.278984', '3.277521', '0.000000', '-5.087833', '4.126212', '-1.901589', '3.883700', '-1.178069', '-1.074791', '0.605731', '-2.991317', '3.903811', '-3.780288', '-0.687796', '1.676894', '-4.237198', '-0.267400', '-1.004392', '-1.962669', '0.174777', '1.017087', '-0.826680', '-1.935020', '-0.251160', '2.146258', '-0.871708', '1.858586', '0.726411', '-2.346821', '-1.677189', '1.129150', '-2.962044', '-0.064565', '-3.084709', '-1.421994', '-0.424060', '2.271989', '-1.908055', '1.475948', '0.389476', '-0.605080', '0.194337', '0.008426'], ['33.0', '-0.410421', '-2.609979', '-1.085026', '-3.403611', '0.412168', '1.627389', '-1.112122', '0.148827', '-1.541590', '0.076180', '2.565056', '-2.304426', '2.323894', '-0.545231', '-0.337284', '-0.935458', '-0.077947', '-1.592902', '3.573998', '0.000000', '-3.360384', '2.518278', '-1.937228', '2.388982', '-0.865112', '-1.044164', '-0.206445', '-2.785242', '3.180252', '-2.272586', '-0.373666', '1.436068', '-2.068922', '-0.459259', '-0.750081', '-0.828136', '0.367846', '0.472766', '-0.570518', '-0.938770', '-0.229098', '2.351215', '-0.243041', '0.801310', '0.560826', '-2.319805', '-1.363607', '1.183094', '-1.948898', '-0.108437', '-2.207305', '-1.581944', '-0.698801', '1.962657', '-2.203944', '0.909222', '0.366224', '-0.097436', '-0.671295', '0.003479'], ['34.0', '-0.187537', '-4.245206', '-1.666451', '-5.219139', '0.660101', '1.216482', '-1.883130', '1.375398', '-2.451489', '0.463192', '2.436909', '-3.038227', '2.904461', '-0.459183', '-0.468680', '-1.052764', '0.961841', '-2.770259', '3.953967', '0.000000', '-5.598113', '3.669560', '-1.829983', '3.938010', '-1.043467', '-1.458917', '0.581589', '-3.512142', '4.508010', '-3.518635', '-1.280741', '1.323079', '-3.978684', '-0.306837', '-1.100498', '-1.123531', '-0.193990', '1.144000', '-0.864722', '-1.661495', '-0.442119', '2.212090', '-0.772977', '1.724032', '0.229019', '-2.659227', '-1.403049', '0.557738', '-3.085482', '-0.074587', '-2.687165', '-1.885727', '-0.619152', '2.489342', '-1.908957', '1.326987', '1.026594', '-0.485670', '-0.726487', '0.020121'], ['35.0', '-0.576028', '-3.540966', '-1.371973', '-4.609882', '1.170569', '1.404928', '-2.142981', '0.612705', '-2.324601', '-0.099887', '3.094958', '-2.704758', '2.954242', '0.304823', '-0.562180', '-0.230842', '0.401479', '-2.438473', '4.177600', '0.000000', '-5.208754', '3.734714', '-1.760008', '4.142395', '-0.645756', '-0.876236', '0.167911', '-2.922264', '4.559485', '-3.475744', '-0.844271', '1.900388', '-2.877438', '-0.280769', '-1.138695', '-1.246021', '-0.927575', '0.977863', '-0.435341', '-1.135529', '-0.085379', '2.596279', '-0.788894', '1.126958', '0.198944', '-2.532888', '-2.251082', '1.397810', '-3.653030', '-0.118408', '-3.033529', '-1.574496', '-0.664125', '2.076324', '-1.590021', '0.921181', '-0.446065', '-0.432462', '-1.142390', '0.022490'], ['36.0', '-2.002832', '-4.797985', '-2.320557', '-5.664197', '0.696366', '2.988560', '-2.475682', '1.073834', '-2.655655', '-0.044534', '4.011257', '-3.765955', '4.180601', '-1.307442', '-0.820845', '-2.005178', '1.005341', '-3.147606', '5.049264', '0.000000', '-5.351588', '4.577478', '-2.310721', '3.062602', '-1.085883', '-1.498045', '0.329426', '-4.068931', '6.461939', '-2.778451', '-0.924937', '2.774758', '-4.280649', '-0.945580', '-0.914021', '-0.216845', '1.034360', '1.480809', '-0.644943', '-1.766070', '-0.297844', '3.869893', '-0.957547', '1.456900', '-0.518379', '-3.709460', '-2.433344', '1.693985', '-5.035758', '-0.176610', '-3.769063', '-2.328014', '-0.555110', '3.123599', '-2.543493', '1.526860', '0.022260', '-0.591027', '-0.628345', '0.043262'], ['37.0', '-1.213876', '-3.986968', '-1.655631', '-4.173369', '0.158124', '2.071034', '-1.273859', '1.398092', '-1.264654', '0.068312', '2.981263', '-2.832757', '2.700628', '-1.360007', '-0.926312', '-2.006373', '1.724187', '-2.360209', '4.313231', '0.000000', '-3.892131', '3.621036', '-1.443081', '1.771483', '-0.834225', '-1.079300', '0.871869', '-3.358477', '4.889806', '-1.793928', '-0.653085', '1.460241', '-4.041008', '-0.739504', '-0.631593', '-0.036521', '0.996340', '1.063016', '-0.433455', '-1.399365', '-0.270591', '2.556875', '0.027938', '2.066417', '-0.682224', '-2.767383', '-1.043712', '0.598187', '-2.826712', '-0.087249', '-2.333844', '-1.764439', '0.095211', '2.246828', '-1.928868', '1.050203', '0.422307', '-1.041088', '0.258556', '0.026471'], ['38.0', '-1.440494', '-3.312038', '-1.554794', '-4.190537', '0.542842', '2.126167', '-1.574665', '0.311414', '-1.372959', '-0.286850', '4.028088', '-2.925749', '2.593493', '-0.164385', '-1.103348', '-1.118013', '1.218286', '-1.934148', '5.334590', '0.000000', '-4.333447', '3.543343', '-1.272963', '2.800509', '-0.493439', '-0.847172', '0.572488', '-3.373476', '5.495466', '-2.311453', '-0.847569', '1.846338', '-2.937600', '-0.672429', '-0.891616', '-0.163531', '-0.314600', '1.019020', '-0.012424', '-0.894809', '-0.044326', '3.232767', '0.258026', '1.555881', '-0.505009', '-3.030344', '-1.887784', '1.039097', '-3.279199', '-0.150024', '-2.546218', '-1.871720', '0.115076', '2.105524', '-1.722998', '0.658230', '-0.708661', '-0.972144', '-0.661418', '0.026272'], ['39.0', '-0.763292', '-3.230713', '-1.549899', '-3.489277', '1.593708', '1.796143', '-1.742732', '0.217736', '-2.402106', '0.030235', '2.947073', '-2.125231', '3.497940', '-0.458843', '-0.624359', '-0.234046', '-0.257352', '-2.596080', '3.663220', '0.000000', '-3.361151', '3.054813', '-1.617836', '2.583306', '-0.963784', '-1.269785', '-0.250990', '-2.825569', '4.637333', '-2.273063', '0.062685', '2.313178', '-2.449169', '-0.469151', '-0.631123', '-0.864034', '0.956859', '1.095523', '-0.915576', '-1.444414', '-0.251349', '2.569635', '-1.446035', '1.239771', '0.589816', '-2.762811', '-2.335598', '1.827366', '-4.421148', '-0.162433', '-3.547907', '-1.886499', '-0.230530', '2.405106', '-1.793635', '0.964258', '-0.242390', '-0.218519', '-0.417005', '0.035827']])

    def test_area_flux(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = FluxCalculator(FluxCalculator.AREA_FLUX, ['U', 'V', 'H'], f,
                                        self.names, self.sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '-0.357471', '-2.370076', '-0.952408', '-2.516463', '0.469718', '1.033709', '-1.026548', '0.759259', '-0.772134', '-0.200879', '2.141473', '-1.855451', '1.546296', '-0.321656', '-0.640871', '-0.940936', '1.015600', '-1.523654', '2.997603', '0.000000', '-2.776522', '2.484670', '-0.888176', '1.840175', '-0.462375', '-0.414255', '0.579630', '-2.023641', '2.902411', '-1.789072', '-0.459141', '0.997571', '-2.292034', '-0.346236', '-0.583716', '-0.623551', '-0.106587', '0.540727', '-0.029657', '-0.813382', '0.014057', '1.808749', '0.351511', '1.260574', '0.018971', '-1.674920', '-0.940351', '0.588631', '-1.432061', '-0.062472', '-1.543730', '-0.936352', '0.134803', '1.220180', '-1.193624', '0.592935', '-0.386144', '-0.738953', '0.170200', '0.004466'], ['1.0', '-0.237362', '-1.040859', '-0.831427', '-1.367619', '0.404240', '0.847680', '-0.175987', '-0.691025', '-0.557858', '0.083243', '2.289969', '-1.210743', '1.059190', '-0.123974', '-0.792672', '-0.480366', '0.112445', '-0.649185', '3.242240', '0.000000', '-1.141616', '0.794061', '-0.334287', '0.755627', '-0.490835', '-0.948510', '0.152046', '-1.976890', '2.846575', '-0.723441', '-0.185169', '0.680621', '-0.759406', '-0.410203', '-0.276142', '0.034080', '0.566519', '0.443663', '-0.313339', '-0.469164', '-0.233940', '1.642089', '0.395673', '0.944835', '0.614091', '-1.871829', '-0.726906', '0.233375', '-1.192188', '-0.128122', '-1.126103', '-1.397991', '0.699399', '1.275906', '-1.169548', '0.111277', '0.005691', '-0.355828', '-0.055369', '0.005429'], ['2.0', '-0.766018', '-2.037321', '-0.796093', '-1.831076', '-0.086314', '1.428873', '-0.358186', '0.775758', '-0.185629', '-0.114411', '1.757451', '-1.453720', '1.349091', '-1.075716', '-0.572874', '-1.383988', '1.052056', '-1.075027', '2.448709', '0.000000', '-1.484341', '1.964716', '-0.828529', '0.346192', '-0.456328', '-0.422255', '0.455861', '-1.809976', '2.433979', '-0.548565', '-0.031730', '0.809535', '-2.146001', '-0.509760', '-0.210624', '0.076311', '0.940070', '0.400830', '-0.110853', '-0.666963', '-0.059990', '1.530313', '0.385634', '1.256152', '-0.451271', '-1.494145', '-0.359413', '0.467911', '-1.012908', '-0.044543', '-1.106225', '-0.912165', '0.154356', '1.126710', '-1.267750', '0.549510', '0.069058', '-0.710717', '0.514135', '0.007283'], ['3.0', '-0.230635', '-2.075464', '-1.058050', '-2.039781', '0.156680', '1.124600', '-0.505258', '0.519450', '-0.675333', '0.169286', '1.778625', '-1.686096', '1.466789', '-0.941222', '-0.578110', '-1.301929', '0.762676', '-1.307313', '2.699813', '0.000000', '-1.753749', '1.636816', '-0.816492', '0.863948', '-0.781097', '-0.908235', '0.423959', '-2.224389', '2.649578', '-1.096036', '-0.305449', '0.782595', '-2.120261', '-0.471011', '-0.304982', '-0.295397', '1.050364', '0.503799', '-0.391959', '-0.991489', '-0.268391', '1.652472', '0.296278', '1.289793', '0.336830', '-1.827895', '-0.469690', '0.301224', '-1.016281', '-0.071841', '-1.294848', '-1.260618', '0.342677', '1.489889', '-1.510799', '0.678431', '0.547663', '-0.501548', '0.497909', '0.001034'], ['4.0', '-0.639531', '-2.694350', '-1.199104', '-2.650454', '0.430543', '1.878607', '-0.742251', '0.418709', '-0.819063', '-0.137483', '2.916634', '-2.147832', '2.219935', '-1.050891', '-0.815836', '-1.400723', '0.662801', '-1.657244', '3.973598', '0.000000', '-2.336905', '2.618649', '-1.413511', '1.288960', '-0.861045', '-0.867649', '0.252659', '-2.810051', '3.634116', '-1.475328', '-0.005868', '1.468004', '-2.458833', '-0.660901', '-0.475222', '-0.496097', '1.155271', '0.527360', '-0.355209', '-1.038139', '-0.133151', '2.514979', '0.319675', '1.515815', '0.292284', '-2.461982', '-1.094330', '1.086206', '-1.738766', '-0.116383', '-2.093375', '-1.556115', '0.154628', '1.861566', '-2.124029', '0.760489', '-0.057945', '-0.684472', '0.392001', '0.002544'], ['5.0', '-1.183724', '-2.147898', '-1.420206', '-2.847176', '-0.200545', '1.882733', '-0.988116', '0.167347', '-1.099694', '0.042935', '2.481107', '-2.301885', '1.744559', '-0.768956', '-0.553748', '-1.680238', '0.678999', '-1.205320', '3.197534', '0.000000', '-2.630834', '1.894430', '-1.012918', '1.356972', '-0.653260', '-1.032896', '0.275591', '-2.590686', '3.568549', '-1.309898', '-0.932214', '1.278625', '-1.974566', '-0.676204', '-0.434020', '0.217003', '0.679222', '0.705787', '-0.137473', '-0.822614', '-0.240551', '2.574034', '0.237829', '0.659614', '-0.132549', '-2.302897', '-1.052734', '0.484814', '-1.859040', '-0.120396', '-1.429792', '-1.540804', '-0.028388', '1.783716', '-1.665190', '0.831483', '0.283508', '-0.306346', '-0.302524', '0.011798'], ['6.0', '-0.845492', '-2.419708', '-1.127961', '-2.313121', '1.113467', '1.148643', '-1.867072', '1.021490', '-1.655692', '-0.206403', '1.274379', '-1.329688', '2.397466', '-0.348558', '-0.272365', '-0.253512', '0.378859', '-2.101927', '1.149175', '0.000000', '-2.397027', '2.523970', '-0.849239', '1.813139', '-0.367141', '-0.240358', '0.124217', '-1.098813', '2.706265', '-1.642639', '-0.044096', '1.844693', '-2.117198', '-0.286712', '-0.384669', '-0.752030', '0.275101', '0.873551', '-0.169137', '-1.044370', '0.007529', '1.498887', '-1.204045', '0.487158', '-0.224971', '-1.241037', '-1.876305', '1.324282', '-3.312725', '-0.067844', '-2.401239', '-0.508821', '-0.114021', '1.154405', '-0.510277', '0.724637', '-0.672268', '-0.270358', '0.087510', '0.027571'], ['7.0', '-2.077559', '-2.681677', '-1.529969', '-3.524628', '-0.503220', '1.897702', '-1.688735', '1.090017', '-1.072691', '-0.210076', '2.185756', '-2.543719', '1.765038', '-0.858958', '-0.539378', '-2.043727', '1.695421', '-1.438634', '2.532712', '0.000000', '-3.395647', '2.683290', '-0.809010', '1.411673', '-0.214590', '-0.497583', '0.830608', '-2.027845', '3.948540', '-1.151602', '-1.466089', '1.365398', '-2.816547', '-0.705480', '-0.484633', '0.810361', '0.038902', '1.001991', '0.302472', '-0.750799', '-0.042023', '2.398919', '0.077126', '0.471467', '-1.494176', '-1.868262', '-1.028988', '0.267319', '-2.589083', '-0.077342', '-1.247846', '-0.939705', '-0.186232', '1.404449', '-0.871152', '0.852127', '-0.246454', '-0.617579', '-0.434871', '0.028980'], ['8.0', '-0.771126', '-1.869308', '-0.622826', '-2.681245', '0.143975', '1.051292', '-0.478969', '-0.096782', '-0.643581', '-0.032389', '2.469018', '-1.497910', '1.371755', '0.027137', '-0.631440', '-0.410009', '0.598862', '-0.816911', '3.611216', '0.000000', '-2.687264', '1.976649', '-0.890412', '1.549123', '-0.165982', '-0.540935', '0.189981', '-2.141049', '3.200335', '-1.110403', '-0.404285', '0.710414', '-1.582836', '-0.316364', '-0.593755', '0.116061', '-0.416845', '0.499973', '-0.157270', '-0.293301', '-0.086999', '1.445318', '0.133711', '0.995938', '-0.522458', '-1.720449', '-0.809032', '0.379720', '-1.756307', '-0.066701', '-1.371642', '-1.208864', '-0.136484', '1.200050', '-1.047293', '0.126178', '-0.079814', '-0.573371', '-0.782070', '0.020158'], ['9.0', '-0.184987', '-2.038802', '-1.084860', '-2.362869', '0.693042', '0.893321', '-0.542117', '-0.037286', '-1.352550', '0.459452', '1.947242', '-1.442030', '1.753302', '-0.134171', '-0.644486', '-0.216652', '0.239461', '-1.487462', '3.081469', '0.000000', '-2.303740', '1.431910', '-0.566855', '1.614121', '-0.718426', '-1.289754', '0.214501', '-2.504694', '3.378603', '-1.415262', '-0.330162', '0.812646', '-1.744801', '-0.257648', '-0.408747', '-0.236786', '0.589582', '0.840501', '-0.710458', '-0.990205', '-0.439588', '1.458362', '-0.479124', '1.541020', '0.384924', '-2.093268', '-0.928185', '0.254748', '-2.313688', '-0.095399', '-1.795269', '-1.727727', '0.337709', '1.758845', '-1.150203', '0.488374', '0.771404', '-0.381260', '-0.263707', '0.023130'], ['10.0', '-0.429310', '-1.974844', '-1.101398', '-2.351713', '0.137429', '0.710506', '-1.045402', '0.797116', '-1.048402', '0.114976', '1.228905', '-1.654590', '1.294302', '-0.510503', '-0.288920', '-1.088711', '0.937490', '-1.301481', '1.750443', '0.000000', '-2.414478', '1.641898', '-0.473846', '1.455351', '-0.496564', '-0.735372', '0.559480', '-1.634151', '2.530208', '-1.299981', '-0.983115', '0.683917', '-2.093907', '-0.266118', '-0.374435', '-0.103190', '0.184114', '0.718015', '-0.292744', '-0.901297', '-0.154274', '1.344475', '-0.188569', '0.746264', '-0.061092', '-1.346916', '-0.504933', '0.053559', '-1.522295', '-0.044656', '-1.033294', '-0.942126', '0.018388', '1.232792', '-0.786688', '0.793240', '0.258116', '-0.347071', '-0.020622', '0.012235'], ['11.0', '-0.323820', '-2.121070', '-1.371242', '-2.123280', '0.181950', '1.393354', '-0.683274', '0.476748', '-0.668611', '0.100566', '2.173204', '-2.095711', '1.479997', '-0.923302', '-0.737174', '-1.603677', '0.990325', '-1.367512', '3.078623', '0.000000', '-1.878322', '1.688077', '-0.644581', '1.056632', '-0.866885', '-1.036177', '0.589759', '-2.494905', '3.119808', '-1.291026', '-0.544744', '0.965213', '-2.217896', '-0.590190', '-0.303216', '-0.257273', '1.051011', '0.610158', '-0.243425', '-1.093081', '-0.257006', '2.280031', '0.668300', '1.372465', '0.545296', '-2.186312', '-0.649760', '0.281233', '-1.026081', '-0.106165', '-1.289206', '-1.491384', '0.692091', '1.676425', '-1.620403', '0.833030', '0.296623', '-0.594106', '0.631978', '0.000332'], ['12.0', '-0.938078', '-2.846174', '-1.438731', '-3.538922', '-0.102463', '2.198505', '-0.546726', '-0.068173', '-0.942144', '0.194028', '3.603787', '-2.745444', '2.186606', '-0.951491', '-0.960160', '-1.728446', '0.834205', '-1.437439', '5.274844', '0.000000', '-3.123312', '2.464277', '-1.474280', '1.580706', '-0.897500', '-1.399428', '0.325047', '-3.805114', '4.627475', '-1.585508', '-0.534874', '1.255988', '-2.608854', '-0.803949', '-0.604685', '0.005531', '1.001879', '0.700100', '-0.436397', '-0.967450', '-0.389278', '2.933909', '0.672855', '1.684103', '0.086088', '-3.144758', '-1.016076', '0.563227', '-1.833033', '-0.140156', '-1.980996', '-2.243381', '0.206707', '2.355867', '-2.473727', '0.728468', '0.699547', '-0.693094', '-0.243560', '0.010988'], ['13.0', '-0.219707', '-0.951335', '-0.565061', '-0.846742', '0.204296', '0.736182', '-0.383577', '0.243162', '-0.413012', '-0.030211', '0.850616', '-0.774523', '0.885880', '-0.529095', '-0.209378', '-0.620463', '0.137957', '-0.679027', '1.015984', '0.000000', '-0.669192', '0.849867', '-0.500098', '0.397368', '-0.403967', '-0.359073', '0.024608', '-0.912000', '1.146241', '-0.505032', '-0.023437', '0.668606', '-0.874051', '-0.268316', '-0.113748', '-0.241069', '0.650971', '0.214639', '-0.148624', '-0.509584', '-0.067857', '0.984693', '-0.008629', '0.365752', '0.269928', '-0.856104', '-0.468168', '0.471967', '-0.660710', '-0.048086', '-0.795837', '-0.528997', '0.081762', '0.716935', '-0.779641', '0.417086', '0.003204', '-0.125071', '0.294453', '0.000451'], ['14.0', '-0.563122', '-1.775248', '-1.020165', '-2.264637', '0.560453', '0.652671', '-1.209125', '0.463835', '-1.591222', '0.184596', '1.080525', '-1.240939', '1.518425', '0.120779', '-0.235648', '-0.148214', '0.289088', '-1.425900', '1.376493', '0.000000', '-2.594008', '1.513159', '-0.380316', '1.956890', '-0.358711', '-0.742414', '0.243723', '-1.360345', '2.586205', '-1.496592', '-0.885207', '0.912762', '-1.520878', '-0.115084', '-0.402245', '-0.168767', '-0.165752', '0.867686', '-0.357263', '-0.800005', '-0.187845', '1.147037', '-0.947150', '0.515628', '-0.134493', '-1.253036', '-1.104001', '0.296118', '-2.590624', '-0.061552', '-1.427814', '-0.904638', '-0.207822', '1.178432', '-0.343603', '0.623016', '0.122769', '-0.115173', '-0.583988', '0.027542'], ['15.0', '-0.596786', '-1.732606', '-1.024825', '-1.727985', '0.247433', '1.284146', '-0.819245', '0.435987', '-0.534457', '-0.224851', '1.960869', '-1.716499', '1.311108', '-0.631887', '-0.586437', '-1.245702', '0.737658', '-1.137594', '2.410220', '0.000000', '-1.667882', '1.756065', '-0.692534', '0.998330', '-0.559800', '-0.491359', '0.399928', '-1.781240', '2.496543', '-1.070610', '-0.403739', '1.095739', '-1.689559', '-0.495765', '-0.335762', '-0.246830', '0.555877', '0.421007', '0.038570', '-0.768562', '-0.002184', '2.050850', '0.516934', '0.851384', '0.201604', '-1.648276', '-0.797493', '0.658297', '-1.061641', '-0.091083', '-1.207335', '-0.931421', '0.295402', '1.171812', '-1.238588', '0.647676', '-0.385689', '-0.512848', '0.387022', '0.001642'], ['16.0', '-0.313518', '-1.960072', '-0.842961', '-2.186640', '0.635858', '0.840337', '-0.974526', '0.476296', '-0.928809', '-0.146872', '1.860796', '-1.405167', '1.437623', '0.071829', '-0.567365', '-0.367057', '0.717532', '-1.366676', '2.555719', '0.000000', '-2.505416', '2.058550', '-0.526767', '1.860069', '-0.377795', '-0.452761', '0.429292', '-1.689392', '2.818181', '-1.732731', '-0.402812', '0.945433', '-1.798666', '-0.188340', '-0.497889', '-0.568149', '-0.321355', '0.610842', '-0.136965', '-0.671918', '-0.002386', '1.537609', '-0.113968', '1.143010', '0.007572', '-1.498174', '-1.053595', '0.563874', '-1.808562', '-0.064825', '-1.474972', '-0.976989', '0.145564', '1.106195', '-0.827590', '0.464491', '-0.419961', '-0.600119', '-0.068916', '0.006498'], ['17.0', '-0.357844', '-1.651399', '-0.821422', '-2.232046', '0.515994', '0.447091', '-0.902673', '0.257103', '-1.158956', '0.198993', '1.390424', '-1.239030', '1.237503', '0.319985', '-0.414177', '0.088432', '0.515564', '-1.147229', '2.077458', '0.000000', '-2.509042', '1.437822', '-0.270461', '2.006345', '-0.247948', '-0.648532', '0.349836', '-1.567537', '2.686324', '-1.442170', '-0.779082', '0.662786', '-1.481189', '-0.088263', '-0.467703', '-0.218358', '-0.483558', '0.750504', '-0.238398', '-0.629976', '-0.179786', '1.053933', '-0.454325', '0.836859', '-0.163184', '-1.299352', '-0.868155', '0.130843', '-2.054289', '-0.055986', '-1.260105', '-0.949627', '0.065347', '1.067508', '-0.305011', '0.342576', '0.131653', '-0.337118', '-0.631950', '0.022385'], ['18.0', '0.234897', '-1.737990', '-0.515792', '-2.235290', '0.322713', '0.861599', '0.354487', '-0.632013', '-0.501487', '0.275018', '2.800350', '-1.366768', '1.262431', '-0.026541', '-0.822662', '-0.223842', '0.076362', '-0.785684', '4.639137', '0.000000', '-2.243384', '1.508096', '-1.052956', '1.560077', '-0.632331', '-1.038201', '0.011919', '-2.926526', '2.994920', '-1.331771', '-0.051834', '0.362715', '-1.319350', '-0.282767', '-0.596701', '-0.517864', '0.216003', '0.253261', '-0.586040', '-0.514845', '-0.301130', '1.414015', '0.543746', '1.667331', '0.606105', '-2.160693', '-0.499304', '0.265716', '-0.765037', '-0.084939', '-1.451593', '-1.736229', '0.155501', '1.555955', '-1.809466', '0.066699', '0.657803', '-0.587397', '-0.390146', '0.001677'], ['19.0', '-0.788429', '-1.428477', '-0.493860', '-1.772438', '0.081925', '0.947079', '-0.823271', '0.445843', '-0.672025', '-0.122875', '1.099466', '-1.052327', '1.138316', '-0.178697', '-0.179795', '-0.457652', '0.424648', '-0.874783', '1.361207', '0.000000', '-1.847340', '1.600189', '-0.785109', '1.059015', '-0.143286', '-0.105000', '0.091695', '-1.040142', '1.693535', '-0.957359', '-0.252653', '0.828386', '-1.268682', '-0.225727', '-0.341395', '-0.121484', '-0.173269', '0.357785', '0.058059', '-0.346921', '0.008304', '1.026234', '-0.243206', '0.347929', '-0.606570', '-0.881247', '-0.823979', '0.596187', '-1.457858', '-0.024748', '-1.072229', '-0.426440', '-0.419256', '0.677003', '-0.589857', '0.339071', '-0.142138', '-0.252219', '-0.364064', '0.011992'], ['20.0', '-0.623704', '-0.771539', '-0.840963', '-1.112481', '-0.033281', '1.061818', '-0.053306', '-0.660937', '-0.415863', '0.217051', '1.869933', '-1.114563', '0.849571', '-0.327146', '-0.694453', '-0.637737', '0.104857', '-0.404239', '2.590449', '0.000000', '-0.691529', '0.379546', '-0.229324', '0.280753', '-0.389691', '-0.909107', '0.125008', '-1.769815', '2.466106', '-0.225007', '-0.267010', '0.632725', '-0.567220', '-0.512359', '-0.125383', '0.430311', '0.851937', '0.436103', '-0.149000', '-0.372502', '-0.327014', '1.540771', '0.423350', '0.634257', '0.219930', '-1.687723', '-0.601016', '0.037180', '-1.048411', '-0.120884', '-0.800688', '-1.216371', '0.672072', '1.143047', '-0.952386', '0.079530', '0.375238', '-0.222960', '-0.123836', '0.010609'], ['21.0', '-0.895522', '-1.111074', '-0.720722', '-1.433994', '-0.026035', '0.989057', '0.004716', '-0.430000', '-0.462805', '0.296743', '1.806459', '-0.954745', '1.000313', '-0.273843', '-0.691776', '-0.458807', '0.421911', '-0.554395', '2.697897', '0.000000', '-1.054007', '0.686081', '-0.169611', '0.179823', '-0.218192', '-0.848239', '0.219959', '-1.834889', '2.839419', '0.033401', '-0.210516', '0.474100', '-0.974149', '-0.420069', '-0.134037', '0.772302', '0.703903', '0.606499', '-0.256711', '-0.343027', '-0.338064', '1.082174', '0.060941', '0.972772', '-0.446743', '-1.604462', '-0.501186', '-0.103422', '-1.717593', '-0.081182', '-0.968378', '-1.248418', '0.492614', '1.141870', '-0.653915', '-0.023846', '0.529841', '-0.415344', '-0.304521', '0.030590'], ['22.0', '-0.095936', '-0.677719', '-0.150173', '-0.691557', '-0.004366', '0.300884', '-0.186966', '0.351419', '-0.115923', '-0.039404', '0.385737', '-0.426498', '0.372491', '-0.209427', '-0.109675', '-0.295905', '0.262206', '-0.354709', '0.593399', '0.000000', '-0.711558', '0.720621', '-0.377692', '0.375608', '-0.093713', '-0.041051', '0.129336', '-0.406850', '0.495268', '-0.404650', '-0.026489', '0.181960', '-0.667471', '-0.095193', '-0.141369', '-0.146237', '0.016175', '0.068257', '-0.018315', '-0.155268', '-0.000904', '0.322720', '0.079651', '0.296145', '-0.116496', '-0.304511', '-0.141184', '0.136433', '-0.226748', '-0.003801', '-0.315919', '-0.126756', '-0.103278', '0.239773', '-0.327697', '0.156489', '-0.000868', '-0.170391', '0.033768', '0.000567'], ['23.0', '-0.646937', '-2.534043', '-1.227098', '-2.371922', '0.376350', '1.908299', '-1.063736', '0.738580', '-1.026215', '-0.174949', '2.153930', '-2.059446', '2.278992', '-1.338640', '-0.397953', '-1.572788', '0.390079', '-1.696577', '2.595693', '0.000000', '-2.035984', '2.480722', '-1.589750', '1.213778', '-0.902755', '-0.703326', '-0.001418', '-2.222362', '2.748875', '-1.455584', '-0.042455', '1.683744', '-2.305126', '-0.658129', '-0.329051', '-0.739264', '1.407441', '0.433879', '-0.317546', '-1.136026', '-0.085418', '2.417710', '-0.017809', '0.772116', '0.424823', '-2.013475', '-1.188269', '1.350774', '-1.640423', '-0.102270', '-2.037448', '-1.121467', '-0.202068', '1.697595', '-1.988279', '1.057810', '-0.064253', '-0.272974', '0.485568', '0.000541'], ['24.0', '-0.243556', '-1.375159', '-0.480680', '-2.018821', '0.393795', '0.141573', '-0.848509', '0.423318', '-0.903856', '0.064687', '0.888179', '-0.917131', '0.911373', '0.386968', '-0.184780', '0.225256', '0.468071', '-0.851563', '1.372463', '0.000000', '-2.340576', '1.375434', '-0.332715', '1.866235', '-0.049892', '-0.296946', '0.287938', '-0.941757', '1.850265', '-1.308962', '-0.694413', '0.416227', '-1.253406', '0.054724', '-0.489243', '-0.215046', '-0.883111', '0.541598', '-0.150450', '-0.370935', '-0.038824', '0.569447', '-0.513474', '0.528046', '-0.346770', '-0.725521', '-0.583943', '0.115973', '-1.556072', '-0.017018', '-0.884126', '-0.526937', '-0.272431', '0.644228', '-0.096428', '0.264228', '-0.082012', '-0.265004', '-0.742335', '0.015189'], ['25.0', '0.110681', '-1.014828', '-0.250656', '-1.079835', '0.324832', '0.082841', '-0.420564', '0.458383', '-0.473552', '0.001620', '0.473595', '-0.486947', '0.622022', '-0.037206', '-0.142398', '-0.094946', '0.291640', '-0.687685', '0.804401', '0.000000', '-1.231253', '1.005136', '-0.375427', '0.870428', '-0.159875', '-0.163876', '0.193159', '-0.587512', '0.901603', '-0.866372', '-0.090836', '0.260805', '-0.950439', '0.003023', '-0.257832', '-0.416551', '-0.186130', '0.225305', '-0.168641', '-0.330036', '-0.023647', '0.300487', '-0.219908', '0.529164', '0.023669', '-0.417755', '-0.288309', '0.178221', '-0.658831', '-0.007014', '-0.627607', '-0.265517', '-0.083269', '0.396748', '-0.319115', '0.210278', '0.006280', '-0.212742', '-0.008736', '0.000421'], ['26.0', '-0.132821', '-0.818504', '-0.276090', '-0.890724', '0.191232', '0.425514', '-0.133978', '0.016586', '-0.370026', '0.068792', '0.859215', '-0.499831', '0.688640', '-0.120331', '-0.243508', '-0.137091', '0.109764', '-0.488365', '1.315207', '0.000000', '-0.859268', '0.719772', '-0.363987', '0.503594', '-0.204392', '-0.356587', '0.030785', '-0.889102', '1.230151', '-0.452035', '0.025949', '0.314917', '-0.693689', '-0.119763', '-0.173686', '-0.091763', '0.211459', '0.210866', '-0.243480', '-0.246526', '-0.099497', '0.523264', '-0.120979', '0.550353', '0.035591', '-0.742866', '-0.347964', '0.225158', '-0.803918', '-0.029464', '-0.693806', '-0.573378', '-0.016860', '0.596043', '-0.521506', '0.130667', '0.146925', '-0.183965', '-0.105304', '0.007498'], ['27.0', '0.863664', '-1.707690', '-0.520014', '-1.774058', '1.073276', '-0.130599', '-0.859258', '0.555591', '-1.405192', '0.256613', '0.627259', '-0.815949', '1.360051', '0.256757', '-0.097717', '0.465971', '-0.162737', '-1.506624', '1.248897', '0.000000', '-2.286422', '1.458672', '-0.719836', '2.325028', '-0.582892', '-0.622168', '0.041760', '-1.165595', '1.354057', '-2.108549', '-0.183955', '0.549272', '-1.403157', '0.160617', '-0.508919', '-1.418782', '-0.262021', '0.419266', '-0.670552', '-0.883774', '-0.178524', '0.469945', '-0.812300', '0.847532', '0.979550', '-0.864462', '-0.861372', '0.500366', '-1.398080', '-0.026496', '-1.531244', '-0.667725', '-0.230193', '0.969913', '-0.640964', '0.544800', '0.310091', '-0.052401', '-0.145433', '0.002475'], ['28.0', '-0.178039', '-1.188358', '-0.360008', '-1.749912', '0.228501', '0.431630', '-0.154012', '-0.189686', '-0.736010', '0.292270', '1.224937', '-0.744178', '0.999048', '0.024616', '-0.271267', '0.046396', '0.007768', '-0.604067', '2.071779', '0.000000', '-1.645475', '0.925344', '-0.610379', '0.989579', '-0.261624', '-0.695792', '-0.063519', '-1.475227', '1.929583', '-0.730763', '-0.112762', '0.297828', '-0.930984', '-0.092301', '-0.350086', '0.008531', '0.000923', '0.379205', '-0.504259', '-0.312508', '-0.252566', '0.556885', '-0.394413', '0.758596', '-0.061381', '-1.113796', '-0.412484', '0.141011', '-1.341584', '-0.038404', '-1.007361', '-1.009383', '-0.184650', '0.964630', '-0.715755', '0.115709', '0.584493', '-0.146419', '-0.649267', '0.016655'], ['29.0', '-0.364307', '-2.221828', '-0.924210', '-2.540862', '0.745240', '0.648809', '-1.431094', '0.769963', '-1.428989', '-0.006430', '1.407298', '-1.474366', '1.747518', '0.078840', '-0.342771', '-0.148516', '0.613967', '-1.697861', '1.929802', '0.000000', '-2.938769', '2.191917', '-0.657210', '2.305942', '-0.321074', '-0.459875', '0.396247', '-1.453094', '2.730367', '-1.950373', '-0.603773', '1.059805', '-2.037846', '-0.126516', '-0.545265', '-0.677126', '-0.389517', '0.779556', '-0.247574', '-0.824175', '-0.067353', '1.241389', '-0.680822', '0.807195', '-0.099624', '-1.292953', '-1.268791', '0.582496', '-2.450203', '-0.047921', '-1.749591', '-0.739181', '-0.160498', '1.130281', '-0.526581', '0.584540', '-0.226386', '-0.383391', '-0.344105', '0.018605'], ['30.0', '0.017170', '-1.525557', '-0.567910', '-1.509047', '0.531356', '0.376016', '-0.993263', '0.773054', '-0.890092', '-0.059192', '0.639932', '-0.891674', '1.164054', '-0.157626', '-0.134835', '-0.237564', '0.333267', '-1.217230', '0.841949', '0.000000', '-1.765169', '1.544214', '-0.614235', '1.454309', '-0.291805', '-0.160671', '0.215052', '-0.724766', '1.254799', '-1.414159', '-0.214678', '0.751856', '-1.423230', '-0.091682', '-0.346272', '-0.815571', '-0.099343', '0.369890', '-0.137444', '-0.626878', '-0.012701', '0.723261', '-0.404853', '0.393394', '0.177139', '-0.645207', '-0.846063', '0.520082', '-1.222912', '-0.021407', '-1.159503', '-0.231730', '-0.135238', '0.626693', '-0.462739', '0.465766', '-0.182164', '-0.190543', '0.111857', '0.002382'], ['31.0', '-2.264735', '-1.912023', '-1.694314', '-2.665806', '-0.339688', '2.089621', '-1.412848', '0.317239', '-1.027344', '-0.164547', '2.334998', '-2.207345', '1.669030', '-0.778875', '-0.656052', '-1.810578', '1.173349', '-1.099938', '2.420328', '0.000000', '-2.257475', '1.786402', '-0.344736', '0.778871', '-0.238030', '-0.725845', '0.559128', '-1.963609', '4.061156', '-0.502918', '-1.291835', '1.572671', '-1.889536', '-0.806038', '-0.234237', '1.185019', '0.593410', '1.072310', '0.326561', '-0.690959', '-0.119202', '2.673443', '0.017940', '0.227007', '-1.167329', '-2.062645', '-1.214965', '0.321622', '-2.789838', '-0.132603', '-1.210531', '-1.196014', '0.304482', '1.478509', '-0.783814', '0.699888', '-0.367271', '-0.426503', '-0.352857', '0.032480'], ['32.0', '0.262169', '-2.929084', '-0.758380', '-3.309187', '0.911791', '0.212966', '-1.782383', '1.507928', '-1.739821', '0.022961', '1.031842', '-1.653576', '1.868145', '0.223840', '-0.223850', '0.028481', '0.694563', '-2.156839', '1.700772', '0.000000', '-4.079073', '2.971974', '-1.148235', '3.362770', '-0.407174', '-0.329151', '0.552522', '-1.350564', '2.290173', '-3.018213', '-0.643927', '0.933929', '-2.720493', '0.052329', '-0.858596', '-1.523412', '-1.033861', '0.668393', '-0.374612', '-0.966920', '-0.050898', '0.934348', '-0.804286', '0.997477', '0.111166', '-1.023721', '-1.284029', '0.644320', '-2.204465', '-0.017939', '-1.898146', '-0.429706', '-0.548294', '1.034339', '-0.569974', '0.740617', '-0.073825', '-0.394733', '-0.438221', '0.008049'], ['33.0', '-0.311975', '-2.199244', '-0.770357', '-3.001576', '0.352802', '1.137041', '-1.064029', '0.286951', '-1.313979', '0.026342', '1.881698', '-1.846702', '1.858717', '-0.301687', '-0.168020', '-0.608968', '-0.021164', '-1.322323', '2.674333', '0.000000', '-3.077446', '2.235625', '-1.683865', '2.193664', '-0.587857', '-0.685765', '-0.146969', '-2.043573', '2.383423', '-2.036598', '-0.376592', '1.085949', '-1.749693', '-0.277668', '-0.698205', '-0.718783', '-0.107832', '0.379132', '-0.418023', '-0.694226', '-0.133871', '1.668244', '-0.338184', '0.543035', '0.258763', '-1.655525', '-1.092636', '0.933427', '-1.630979', '-0.065877', '-1.762300', '-1.082729', '-0.799739', '1.439472', '-1.635134', '0.711425', '0.205411', '-0.065797', '-0.750038', '0.003249'], ['34.0', '-0.131387', '-3.027581', '-1.287728', '-3.265377', '0.240030', '1.180995', '-1.158516', '1.193598', '-1.458286', '0.247701', '1.625738', '-2.176815', '2.067686', '-1.021942', '-0.320065', '-1.517873', '0.774633', '-1.976705', '2.563812', '0.000000', '-3.301283', '2.554460', '-1.440916', '1.967433', '-0.906183', '-1.034102', '0.430796', '-2.413963', '2.833908', '-2.106557', '-0.682715', '0.977933', '-2.966723', '-0.411496', '-0.582762', '-0.745451', '0.720466', '0.674878', '-0.614794', '-1.288609', '-0.308005', '1.706597', '-0.261292', '1.138602', '0.366609', '-1.889876', '-0.819890', '0.424919', '-1.667191', '-0.052455', '-1.772740', '-1.251778', '-0.314698', '1.796546', '-1.754795', '1.117907', '0.757474', '-0.339491', '0.181241', '0.005788'], ['35.0', '-0.331114', '-1.505663', '-0.798840', '-1.970120', '0.768085', '0.561377', '-1.158854', '0.174953', '-1.297957', '-0.017058', '1.404076', '-1.168083', '1.431020', '0.355684', '-0.297385', '0.159010', '0.135208', '-1.208961', '1.745594', '0.000000', '-2.314799', '1.514896', '-0.430827', '2.073090', '-0.260336', '-0.504054', '0.114341', '-1.250710', '2.405071', '-1.573961', '-0.546585', '1.003307', '-1.185793', '-0.091552', '-0.478208', '-0.495482', '-0.454401', '0.645934', '-0.221244', '-0.592547', '-0.053075', '1.259537', '-0.625525', '0.463220', '0.176012', '-1.211696', '-1.243871', '0.617792', '-2.200321', '-0.072391', '-1.497154', '-0.805267', '-0.097761', '1.011038', '-0.420109', '0.435870', '-0.361366', '-0.164993', '-0.574964', '0.018188'], ['36.0', '-0.435004', '-1.043305', '-0.692149', '-1.275403', '0.105348', '1.065229', '-0.274589', '-0.200066', '-0.450310', '0.027995', '1.593366', '-1.120092', '1.040102', '-0.452481', '-0.395173', '-0.698205', '0.131451', '-0.581167', '2.121319', '0.000000', '-0.982821', '0.896612', '-0.591016', '0.540747', '-0.439293', '-0.652296', '-0.008087', '-1.546014', '1.984649', '-0.573530', '-0.098145', '0.739506', '-0.873422', '-0.379011', '-0.188257', '0.007274', '0.653093', '0.302760', '-0.214841', '-0.434736', '-0.149979', '1.453493', '0.182042', '0.582217', '0.254474', '-1.413704', '-0.579322', '0.452626', '-0.913661', '-0.081848', '-0.944846', '-1.034703', '0.166870', '1.067675', '-1.122037', '0.365007', '0.116756', '-0.189768', '-0.038607', '0.003495'], ['37.0', '-0.837184', '-1.764758', '-0.954925', '-1.961504', '0.231024', '0.732541', '-1.115554', '0.742197', '-1.053683', '0.056428', '0.961463', '-1.266932', '1.361834', '-0.491476', '-0.282661', '-0.802690', '0.801840', '-1.293181', '1.165679', '0.000000', '-1.893532', '1.518514', '-0.372759', '0.926792', '-0.309523', '-0.500073', '0.445883', '-1.167291', '2.397084', '-0.801631', '-0.652102', '0.853448', '-1.811536', '-0.286033', '-0.223185', '0.155678', '0.348280', '0.752491', '-0.188504', '-0.775161', '-0.129363', '1.050289', '-0.528078', '0.536158', '-0.519712', '-1.076017', '-0.698636', '0.237779', '-2.098521', '-0.042898', '-1.169824', '-0.622921', '0.020158', '0.977806', '-0.355023', '0.559760', '0.090932', '-0.287267', '-0.041982', '0.024282'], ['38.0', '-0.834468', '-1.705457', '-0.909245', '-2.119049', '0.343544', '1.148387', '-0.973839', '0.194361', '-0.826123', '-0.157154', '2.026620', '-1.555876', '1.420620', '-0.101412', '-0.554309', '-0.599175', '0.631302', '-1.084977', '2.563006', '0.000000', '-2.184516', '1.796256', '-0.570598', '1.454121', '-0.265215', '-0.467537', '0.313999', '-1.677690', '2.928212', '-1.198240', '-0.486530', '1.077999', '-1.521885', '-0.366613', '-0.422730', '-0.052801', '-0.076336', '0.605553', '0.004670', '-0.521498', '-0.025085', '1.774705', '0.026743', '0.730844', '-0.241112', '-1.587342', '-1.088542', '0.587807', '-1.920749', '-0.085901', '-1.378007', '-0.965967', '0.110903', '1.124540', '-0.811852', '0.410300', '-0.434307', '-0.460228', '-0.310479', '0.015939'], ['39.0', '-0.458109', '-1.830905', '-0.783328', '-1.860910', '0.986735', '0.811520', '-1.186103', '0.450653', '-1.447164', '-0.012675', '1.127460', '-0.953444', '1.958990', '-0.190340', '-0.183785', '0.057995', '-0.108483', '-1.588827', '1.252332', '0.000000', '-1.902385', '1.796151', '-0.840638', '1.486255', '-0.424193', '-0.463708', '-0.133131', '-1.144352', '2.180050', '-1.272971', '0.058881', '1.318184', '-1.430742', '-0.155284', '-0.334838', '-0.583129', '0.346025', '0.645574', '-0.441739', '-0.835400', '-0.081877', '1.068155', '-1.137577', '0.523653', '0.088776', '-1.143236', '-1.360882', '1.070974', '-2.668442', '-0.060546', '-2.007473', '-0.701987', '-0.286620', '1.107340', '-0.634682', '0.551695', '-0.220030', '-0.084239', '-0.197006', '0.024397']])

    def test_mass_flux(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            calculator = FluxCalculator(FluxCalculator.MASS_FLUX, ['U', 'V', 'H', 'M'], f,
                                        self.names, self.sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '-0.274313', '-1.652278', '-0.684448', '-1.867955', '0.372551', '0.746949', '-0.653220', '0.343606', '-0.543767', '-0.145061', '1.785679', '-1.351363', '1.095366', '-0.090347', '-0.533630', '-0.566228', '0.698715', '-1.020388', '2.523243', '0.000000', '-2.073888', '1.755359', '-0.587313', '1.424664', '-0.324333', '-0.355981', '0.396478', '-1.627649', '2.332502', '-1.300096', '-0.384759', '0.703085', '-1.548257', '-0.237325', '-0.455650', '-0.390769', '-0.194893', '0.408926', '-0.025892', '-0.546575', '0.012302', '1.385813', '0.298171', '0.990620', '0.019868', '-1.336574', '-0.685807', '0.408816', '-1.081772', '-0.053861', '-1.130079', '-0.816861', '0.135542', '0.936695', '-0.891779', '0.359112', '-0.314760', '-0.573652', '-0.006791', '0.003773'], ['1.0', '-0.073925', '-0.573058', '-0.247339', '-0.770693', '0.181372', '0.289168', '-0.088695', '-0.249566', '-0.250149', '0.005148', '1.044077', '-0.497874', '0.506131', '0.045862', '-0.241894', '-0.124936', '0.127493', '-0.312414', '1.580744', '0.000000', '-0.800762', '0.551207', '-0.231059', '0.528695', '-0.161789', '-0.308881', '0.030796', '-0.963643', '1.256579', '-0.441748', '-0.094334', '0.248642', '-0.470129', '-0.093312', '-0.203759', '-0.094608', '-0.029261', '0.169590', '-0.153706', '-0.176981', '-0.043017', '0.555212', '0.081049', '0.460145', '0.126851', '-0.754187', '-0.297678', '0.162128', '-0.539173', '-0.027148', '-0.580658', '-0.603714', '0.052058', '0.545740', '-0.540998', '0.040476', '0.010270', '-0.217325', '-0.124353', '0.002559'], ['2.0', '-0.478501', '-1.314759', '-0.462583', '-1.215347', '-0.054762', '0.932601', '-0.110837', '0.413456', '-0.069253', '-0.070417', '1.264107', '-0.890276', '0.837034', '-0.603543', '-0.423904', '-0.813487', '0.673034', '-0.645223', '1.820816', '0.000000', '-1.027438', '1.296399', '-0.529490', '0.252985', '-0.267508', '-0.278878', '0.283414', '-1.265812', '1.661415', '-0.363409', '0.020612', '0.466813', '-1.347294', '-0.314381', '-0.160537', '0.061898', '0.513711', '0.246673', '-0.074605', '-0.371122', '-0.039697', '0.973362', '0.296814', '0.923252', '-0.329875', '-1.014310', '-0.225001', '0.271614', '-0.641896', '-0.028874', '-0.714611', '-0.661387', '0.112954', '0.732229', '-0.844744', '0.282755', '0.049579', '-0.515845', '0.275142', '0.004983'], ['3.0', '-0.109128', '-0.912257', '-0.470291', '-0.911799', '0.009341', '0.626357', '-0.065450', '0.052054', '-0.215063', '0.120414', '1.052137', '-0.831546', '0.646067', '-0.406332', '-0.357086', '-0.593909', '0.319171', '-0.521735', '1.644656', '0.000000', '-0.750118', '0.675724', '-0.386633', '0.354736', '-0.394784', '-0.503555', '0.179190', '-1.263644', '1.366001', '-0.460016', '-0.079988', '0.328629', '-0.914811', '-0.257862', '-0.150814', '-0.098595', '0.553562', '0.197588', '-0.192280', '-0.419533', '-0.164278', '0.880888', '0.316827', '0.742888', '0.208963', '-1.022624', '-0.208308', '0.120445', '-0.362968', '-0.042076', '-0.602917', '-0.741877', '0.235093', '0.773577', '-0.836944', '0.252157', '0.365847', '-0.280630', '0.200083', '0.000340'], ['4.0', '-0.497275', '-2.037566', '-0.895737', '-2.106710', '0.338992', '1.367063', '-0.570519', '0.248268', '-0.656862', '-0.101520', '2.280219', '-1.616008', '1.662580', '-0.682569', '-0.642517', '-0.992527', '0.507234', '-1.234173', '3.143709', '0.000000', '-1.908359', '1.996320', '-1.044842', '1.058140', '-0.607676', '-0.668123', '0.206620', '-2.154397', '2.869096', '-1.172050', '-0.051316', '1.073147', '-1.832847', '-0.475529', '-0.399923', '-0.331844', '0.720093', '0.427895', '-0.267577', '-0.739110', '-0.102679', '1.872054', '0.216002', '1.160891', '0.168579', '-1.882026', '-0.848499', '0.764261', '-1.401727', '-0.088706', '-1.584371', '-1.213219', '0.112372', '1.406378', '-1.577182', '0.518106', '-0.066412', '-0.535832', '0.187528', '0.002366'], ['5.0', '-0.853637', '-1.082450', '-0.998756', '-1.491949', '-0.081520', '1.240448', '-0.503934', '-0.185485', '-0.690380', '0.080680', '1.639767', '-1.292146', '1.035857', '-0.419634', '-0.479919', '-0.949154', '0.254673', '-0.643049', '2.006477', '0.000000', '-1.231520', '0.815586', '-0.365625', '0.605466', '-0.413447', '-0.775192', '0.136845', '-1.633567', '2.400621', '-0.552177', '-0.554794', '0.881949', '-0.916353', '-0.486003', '-0.196985', '0.333863', '0.638646', '0.509636', '-0.056791', '-0.503276', '-0.218253', '1.733038', '0.137841', '0.382245', '-0.001066', '-1.561188', '-0.751293', '0.244004', '-1.301309', '-0.105232', '-0.862042', '-1.084454', '0.290187', '1.152546', '-0.950813', '0.407911', '0.190046', '-0.160990', '-0.141114', '0.009360'], ['6.0', '-0.273537', '-1.220497', '-0.502228', '-0.958580', '0.436158', '0.595096', '-0.767319', '0.628785', '-0.614494', '-0.120447', '0.507095', '-0.620562', '1.162459', '-0.557614', '-0.101835', '-0.407355', '0.159428', '-1.014407', '0.448994', '0.000000', '-0.802340', '1.241387', '-0.650809', '0.533305', '-0.294412', '-0.048149', '0.016445', '-0.530327', '0.942337', '-0.684909', '0.181244', '0.900195', '-1.121336', '-0.210307', '-0.145875', '-0.540541', '0.540348', '0.263761', '-0.075013', '-0.590742', '0.002412', '0.688897', '-0.426535', '0.229419', '0.017078', '-0.530612', '-0.706771', '0.747620', '-1.072455', '-0.024968', '-1.125649', '-0.110099', '-0.075568', '0.524830', '-0.531893', '0.387224', '-0.181994', '-0.120542', '0.385725', '0.001104'], ['7.0', '-1.168028', '-1.121527', '-0.640196', '-1.650719', '-0.113698', '0.603024', '-0.981015', '0.522056', '-0.772359', '-0.045229', '0.652783', '-0.910901', '0.805889', '-0.050922', '-0.156709', '-0.444371', '0.702640', '-0.695667', '0.594230', '0.000000', '-1.724003', '1.122537', '-0.108734', '0.855073', '0.088902', '-0.152663', '0.379538', '-0.555976', '1.844631', '-0.432844', '-0.859849', '0.607480', '-1.141112', '-0.188367', '-0.218194', '0.572515', '-0.347540', '0.628925', '0.117335', '-0.280768', '-0.012483', '0.731243', '-0.489766', '0.019363', '-1.058016', '-0.581573', '-0.580803', '0.044292', '-1.874664', '-0.024889', '-0.592755', '-0.253523', '-0.232127', '0.481578', '0.198345', '0.289369', '-0.169583', '-0.194902', '-0.587207', '0.027922'], ['8.0', '-0.399015', '-0.574647', '-0.227976', '-0.904964', '0.126718', '0.191620', '-0.412364', '0.090851', '-0.401216', '0.015345', '0.528395', '-0.425921', '0.453166', '0.194222', '-0.134178', '0.087580', '0.219124', '-0.318462', '0.679324', '0.000000', '-0.984738', '0.575859', '-0.095277', '0.657156', '0.049306', '-0.140152', '0.107819', '-0.412028', '1.052323', '-0.301612', '-0.303452', '0.259149', '-0.508764', '-0.037481', '-0.217659', '0.201064', '-0.360861', '0.279706', '-0.040025', '-0.085036', '-0.022160', '0.327265', '-0.246646', '0.164504', '-0.337172', '-0.399742', '-0.362362', '0.089099', '-1.014726', '-0.017623', '-0.443721', '-0.280238', '-0.111671', '0.307586', '0.063079', '0.055970', '-0.114357', '-0.119831', '-0.464175', '0.017611'], ['9.0', '-0.171062', '-1.686314', '-0.915240', '-1.932781', '0.633232', '0.694019', '-0.546167', '0.065642', '-1.181156', '0.358188', '1.465696', '-1.153705', '1.465855', '-0.099736', '-0.482118', '-0.146172', '0.205281', '-1.284284', '2.282906', '0.000000', '-1.911286', '1.205460', '-0.428390', '1.354523', '-0.579697', '-1.024131', '0.183060', '-1.930763', '2.718198', '-1.181281', '-0.299208', '0.703425', '-1.450588', '-0.182461', '-0.316279', '-0.205792', '0.450066', '0.731052', '-0.580848', '-0.849383', '-0.339329', '1.154019', '-0.518917', '1.212254', '0.281275', '-1.633649', '-0.791737', '0.232833', '-2.024114', '-0.074969', '-1.490791', '-1.352186', '0.254588', '1.408860', '-0.839240', '0.449150', '0.561213', '-0.287307', '-0.204399', '0.020738'], ['10.0', '-0.183229', '-0.728095', '-0.523106', '-0.799196', '-0.033505', '0.514903', '-0.202104', '0.163456', '-0.286294', '0.080309', '0.669883', '-0.713128', '0.517580', '-0.415621', '-0.168316', '-0.684397', '0.298476', '-0.438487', '0.950755', '0.000000', '-0.656974', '0.507276', '-0.245681', '0.249734', '-0.321324', '-0.431101', '0.156834', '-0.898387', '1.051509', '-0.337559', '-0.272324', '0.298005', '-0.771341', '-0.207412', '-0.100079', '0.044957', '0.439240', '0.236052', '-0.143884', '-0.405142', '-0.110572', '0.744351', '0.114257', '0.381044', '0.145308', '-0.749012', '-0.152154', '0.037824', '-0.372502', '-0.029890', '-0.400115', '-0.573798', '0.128848', '0.640800', '-0.622982', '0.354178', '0.246990', '-0.142179', '0.186133', '0.001256'], ['11.0', '-0.005051', '-1.178076', '-0.483019', '-1.258881', '0.282733', '0.313862', '-0.550752', '0.454746', '-0.451563', '-0.008395', '0.829224', '-0.863831', '0.766025', '-0.148127', '-0.239494', '-0.322962', '0.581711', '-0.789903', '1.329811', '0.000000', '-1.356993', '1.111799', '-0.294322', '1.002907', '-0.209055', '-0.291606', '0.368221', '-0.916465', '1.445801', '-0.984899', '-0.274464', '0.362109', '-1.236879', '-0.082483', '-0.249813', '-0.315101', '-0.062767', '0.361708', '-0.138680', '-0.433425', '-0.031926', '0.682183', '-0.046326', '0.716439', '0.042045', '-0.749178', '-0.334731', '0.128589', '-0.768462', '-0.016277', '-0.703548', '-0.500317', '0.082000', '0.629359', '-0.485794', '0.333669', '-0.046413', '-0.353890', '0.070408', '0.000320'], ['12.0', '-0.622697', '-1.956013', '-0.882262', '-2.368706', '0.015431', '1.230529', '-0.626576', '0.415447', '-0.727581', '0.043056', '1.799258', '-1.642032', '1.416810', '-0.610694', '-0.410636', '-1.058179', '0.638842', '-1.055635', '2.568197', '0.000000', '-2.212525', '1.801896', '-0.950843', '1.144104', '-0.490689', '-0.697710', '0.259654', '-1.977564', '2.600690', '-1.114102', '-0.436810', '0.792097', '-1.860913', '-0.392933', '-0.412720', '-0.004864', '0.379692', '0.493152', '-0.259512', '-0.649840', '-0.150982', '1.598422', '0.066696', '0.922043', '-0.168393', '-1.633527', '-0.599232', '0.397875', '-1.349230', '-0.059533', '-1.204150', '-1.147201', '-0.152503', '1.319844', '-1.308348', '0.611078', '0.254951', '-0.402589', '-0.177729', '0.008958'], ['13.0', '-0.048740', '-0.070079', '-0.107904', '-0.079066', '0.012830', '0.155206', '0.008589', '-0.072274', '-0.027029', '0.007073', '0.220752', '-0.132182', '0.091658', '-0.063288', '-0.076082', '-0.112670', '-0.007707', '-0.038363', '0.263148', '0.000000', '-0.018880', '0.033415', '-0.027292', '0.004019', '-0.088033', '-0.115352', '-0.006126', '-0.214461', '0.246584', '-0.019554', '-0.007121', '0.099721', '-0.049160', '-0.063969', '-0.004315', '0.021455', '0.139604', '0.027186', '-0.025231', '-0.064819', '-0.026936', '0.244284', '0.078290', '0.085888', '0.093839', '-0.212266', '-0.059189', '0.054208', '-0.056791', '-0.017419', '-0.082619', '-0.169562', '0.091762', '0.146583', '-0.153617', '0.050178', '0.016493', '-0.019609', '0.037948', '0.000024'], ['14.0', '-0.314466', '-0.602433', '-0.443290', '-0.884847', '0.358767', '0.216232', '-0.451947', '0.004456', '-0.735806', '0.160335', '0.471883', '-0.409727', '0.617836', '0.309354', '-0.144745', '0.340548', '0.021818', '-0.539027', '0.599231', '0.000000', '-1.055007', '0.447975', '0.071953', '0.983554', '-0.059152', '-0.372187', '0.072508', '-0.566084', '1.264404', '-0.535812', '-0.417880', '0.368579', '-0.433242', '-0.006167', '-0.137736', '0.081599', '-0.209881', '0.483155', '-0.152077', '-0.293560', '-0.115514', '0.425318', '-0.544139', '0.255709', '-0.140999', '-0.555523', '-0.547414', '0.044677', '-1.429659', '-0.034511', '-0.631466', '-0.459333', '0.020396', '0.499829', '0.140015', '0.163049', '0.042888', '-0.038051', '-0.478138', '0.021825'], ['15.0', '-0.315609', '-0.511639', '-0.374158', '-0.547517', '0.004130', '0.679086', '-0.054321', '-0.199371', '-0.000054', '-0.115253', '1.242785', '-0.800105', '0.396815', '-0.216153', '-0.403881', '-0.571864', '0.254694', '-0.246172', '1.593912', '0.000000', '-0.467227', '0.548324', '-0.252968', '0.222668', '-0.271257', '-0.301288', '0.139527', '-1.039079', '1.267756', '-0.245378', '-0.128835', '0.401021', '-0.452287', '-0.279330', '-0.116932', '0.051180', '0.326451', '0.094292', '0.033953', '-0.217725', '-0.001200', '1.111772', '0.588504', '0.510524', '0.167607', '-0.938008', '-0.245723', '0.276539', '-0.220888', '-0.061226', '-0.411489', '-0.599689', '0.271924', '0.560544', '-0.682621', '0.166463', '-0.159624', '-0.280729', '0.101979', '0.000081'], ['16.0', '-0.150006', '-0.979777', '-0.493811', '-1.068078', '0.421861', '0.361098', '-0.735944', '0.359892', '-0.640433', '-0.070338', '0.691482', '-0.706709', '0.800358', '0.043535', '-0.190546', '-0.146822', '0.352067', '-0.819399', '0.855316', '0.000000', '-1.249258', '1.008416', '-0.199104', '0.998163', '-0.190870', '-0.209922', '0.219448', '-0.657200', '1.335021', '-0.973447', '-0.229658', '0.581398', '-0.923139', '-0.061732', '-0.205089', '-0.385961', '-0.152223', '0.381512', '-0.072224', '-0.414147', '-0.001352', '0.739220', '-0.262950', '0.436264', '0.040800', '-0.641682', '-0.656139', '0.334193', '-1.127781', '-0.031212', '-0.812058', '-0.389844', '0.089628', '0.532530', '-0.266746', '0.321080', '-0.251304', '-0.209720', '0.005862', '0.004241'], ['17.0', '-0.236676', '-0.864793', '-0.632609', '-0.978185', '0.172326', '0.533195', '-0.370930', '0.074903', '-0.541119', '0.101182', '0.833954', '-0.755119', '0.733490', '-0.280862', '-0.278124', '-0.522393', '0.242294', '-0.651585', '1.172603', '0.000000', '-0.874165', '0.633896', '-0.232676', '0.459885', '-0.312331', '-0.503790', '0.181355', '-1.008242', '1.444431', '-0.577582', '-0.262372', '0.461375', '-0.790746', '-0.210344', '-0.119433', '0.000812', '0.425505', '0.395345', '-0.143878', '-0.472156', '-0.156554', '0.835089', '-0.068723', '0.478948', '0.120216', '-0.901043', '-0.400435', '0.072876', '-0.856348', '-0.047645', '-0.661754', '-0.643426', '0.243278', '0.733406', '-0.598165', '0.291284', '0.196374', '-0.160849', '0.112765', '0.002962'], ['18.0', '0.131110', '-1.196546', '-0.372895', '-1.430550', '0.146578', '0.812094', '0.378564', '-0.542258', '-0.260395', '0.201428', '2.109514', '-1.029164', '0.913611', '-0.157115', '-0.628741', '-0.310217', '0.041363', '-0.544285', '3.465461', '0.000000', '-1.373717', '1.012409', '-0.786354', '0.896194', '-0.519967', '-0.793198', '-0.009883', '-2.223667', '2.162756', '-0.849603', '0.077353', '0.287277', '-0.911724', '-0.272653', '-0.341301', '-0.361903', '0.439362', '0.140219', '-0.441457', '-0.378294', '-0.235435', '1.153452', '0.541635', '1.277897', '0.493986', '-1.667567', '-0.348510', '0.238979', '-0.443628', '-0.067587', '-1.048975', '-1.322093', '0.179872', '1.181879', '-1.448126', '0.070682', '0.556949', '-0.439439', '-0.152058', '0.000453'], ['19.0', '-0.501787', '-0.901413', '-0.237842', '-1.298228', '0.052591', '0.446162', '-0.522668', '0.257272', '-0.455276', '-0.053632', '0.676524', '-0.625396', '0.637676', '0.083121', '-0.119850', '-0.132338', '0.277554', '-0.491282', '0.892276', '0.000000', '-1.416330', '1.034851', '-0.467194', '0.828043', '-0.013163', '-0.052917', '0.077758', '-0.608968', '1.097716', '-0.653645', '-0.235630', '0.409354', '-0.779520', '-0.083778', '-0.286156', '0.005240', '-0.436689', '0.237835', '0.022757', '-0.137758', '0.003952', '0.505484', '-0.192198', '0.209195', '-0.504594', '-0.494568', '-0.498145', '0.264493', '-1.009481', '-0.011904', '-0.615943', '-0.257651', '-0.316246', '0.364705', '-0.225184', '0.128917', '-0.089380', '-0.166213', '-0.453701', '0.011716'], ['20.0', '-0.240063', '-0.365412', '-0.391404', '-0.567048', '-0.090686', '0.536468', '0.097599', '-0.571168', '-0.127279', '0.145155', '1.282899', '-0.682535', '0.407429', '-0.152653', '-0.508441', '-0.365273', '0.047833', '-0.144681', '1.915155', '0.000000', '-0.331309', '0.135853', '-0.172212', '0.131990', '-0.216634', '-0.535627', '0.095505', '-1.141199', '1.460116', '-0.130832', '-0.088107', '0.285140', '-0.241040', '-0.324461', '-0.086037', '0.136619', '0.525591', '0.158073', '-0.077849', '-0.145308', '-0.221288', '0.867028', '0.509793', '0.401102', '0.251773', '-1.038594', '-0.321494', '-0.036369', '-0.354511', '-0.077170', '-0.448607', '-0.694328', '0.510256', '0.642845', '-0.639094', '-0.106201', '0.302799', '-0.154760', '-0.063902', '0.002734'], ['21.0', '-0.658853', '-0.690402', '-0.511553', '-0.912936', '-0.010511', '0.650814', '-0.038961', '-0.300601', '-0.355788', '0.229016', '1.126965', '-0.621908', '0.667189', '-0.166795', '-0.466864', '-0.258697', '0.247341', '-0.371702', '1.662244', '0.000000', '-0.644982', '0.369923', '-0.059279', '0.091661', '-0.119667', '-0.584204', '0.155086', '-1.137217', '1.881981', '0.095384', '-0.174052', '0.327458', '-0.587766', '-0.290472', '-0.064820', '0.611263', '0.506995', '0.444894', '-0.157985', '-0.226618', '-0.250927', '0.692417', '-0.013951', '0.575969', '-0.326067', '-1.046293', '-0.372832', '-0.115868', '-1.268684', '-0.058425', '-0.635759', '-0.798009', '0.364518', '0.744877', '-0.330222', '-0.032239', '0.359146', '-0.237255', '-0.245814', '0.024494'], ['22.0', '-0.032018', '-0.166778', '-0.050304', '-0.161499', '0.001516', '0.063502', '-0.083129', '0.095904', '-0.049549', '-0.011192', '0.079853', '-0.116284', '0.101657', '-0.059766', '-0.020054', '-0.088955', '0.073820', '-0.104318', '0.112223', '0.000000', '-0.168085', '0.167816', '-0.080187', '0.089869', '-0.026435', '-0.011041', '0.039984', '-0.088304', '0.130013', '-0.100906', '-0.023173', '0.059144', '-0.175661', '-0.026552', '-0.032893', '-0.039268', '0.018892', '0.024967', '-0.003815', '-0.050635', '-0.000262', '0.084988', '0.007295', '0.051103', '-0.022062', '-0.071931', '-0.046525', '0.038398', '-0.078944', '-0.001175', '-0.086141', '-0.020944', '-0.023599', '0.061530', '-0.069430', '0.048485', '-0.002716', '-0.033588', '0.020574', '0.000235'], ['23.0', '-0.276730', '-0.934430', '-0.564742', '-0.739113', '0.167247', '0.755434', '-0.410284', '0.356993', '-0.389024', '-0.070378', '0.712709', '-0.706325', '0.893362', '-0.655096', '-0.153360', '-0.711394', '0.129720', '-0.666095', '0.719805', '0.000000', '-0.517842', '0.847602', '-0.502496', '0.272285', '-0.403084', '-0.283909', '-0.011312', '-0.763171', '0.978872', '-0.377248', '-0.017611', '0.732634', '-0.884096', '-0.288029', '-0.072173', '-0.250529', '0.766287', '0.180330', '-0.119071', '-0.531835', '-0.037190', '0.954328', '-0.061841', '0.211354', '0.233461', '-0.739311', '-0.447850', '0.546900', '-0.618856', '-0.045253', '-0.764604', '-0.400365', '0.034563', '0.653943', '-0.727051', '0.458917', '-0.041847', '-0.078847', '0.389471', '0.000181'], ['24.0', '-0.025667', '-0.327982', '-0.054882', '-0.519965', '0.028343', '0.053099', '-0.100644', '0.082775', '-0.100140', '0.002871', '0.266022', '-0.213312', '0.166897', '0.044180', '-0.058546', '-0.016611', '0.096452', '-0.118285', '0.438467', '0.000000', '-0.537602', '0.351712', '-0.181468', '0.347114', '-0.024079', '-0.050576', '0.039313', '-0.266452', '0.348630', '-0.260578', '-0.087192', '0.050276', '-0.295659', '-0.001608', '-0.156117', '-0.039692', '-0.223230', '0.047490', '-0.029971', '-0.042577', '-0.006106', '0.130419', '-0.005529', '0.155067', '-0.073967', '-0.177043', '-0.059410', '0.034527', '-0.152294', '-0.002484', '-0.157068', '-0.139741', '-0.088930', '0.136495', '-0.146049', '0.030297', '0.008704', '-0.084585', '-0.164455', '0.000735'], ['25.0', '0.058431', '-0.388865', '-0.145193', '-0.390491', '0.176450', '0.017635', '-0.211498', '0.152597', '-0.239118', '0.010379', '0.183806', '-0.215101', '0.291771', '-0.044973', '-0.043547', '-0.051344', '0.109628', '-0.320759', '0.306074', '0.000000', '-0.433407', '0.354582', '-0.125697', '0.326814', '-0.092267', '-0.100408', '0.071852', '-0.257252', '0.410322', '-0.346705', '-0.056582', '0.134560', '-0.375675', '0.000698', '-0.073043', '-0.182907', '0.013148', '0.122890', '-0.093390', '-0.185202', '-0.015754', '0.139564', '-0.132919', '0.200546', '0.063916', '-0.193352', '-0.127308', '0.080944', '-0.323127', '-0.004360', '-0.297392', '-0.128553', '-0.003966', '0.198180', '-0.135991', '0.113784', '0.011042', '-0.065941', '0.039149', '0.000209'], ['26.0', '-0.035064', '-0.533379', '-0.094872', '-0.518280', '0.104368', '0.247450', '-0.087749', '0.156960', '-0.170768', '-0.013036', '0.395654', '-0.257076', '0.388268', '-0.109996', '-0.083873', '-0.103651', '0.083984', '-0.304368', '0.604980', '0.000000', '-0.543711', '0.552296', '-0.305941', '0.321886', '-0.113950', '-0.102398', '0.002161', '-0.432742', '0.510621', '-0.336442', '0.074294', '0.186590', '-0.465102', '-0.044510', '-0.112297', '-0.172092', '0.069050', '0.063762', '-0.110567', '-0.136649', '-0.011450', '0.258372', '-0.071157', '0.315159', '-0.017414', '-0.319470', '-0.167753', '0.208095', '-0.332179', '-0.006521', '-0.383237', '-0.228952', '-0.122714', '0.272609', '-0.325816', '0.115950', '0.037108', '-0.113615', '-0.010242', '0.001080'], ['27.0', '0.374701', '-0.539074', '-0.227842', '-0.618281', '0.352181', '0.018404', '-0.029661', '-0.038076', '-0.453142', '0.227078', '0.381399', '-0.314382', '0.454735', '0.057352', '-0.095400', '0.168928', '-0.156181', '-0.431958', '0.803311', '0.000000', '-0.677497', '0.291063', '-0.238769', '0.705106', '-0.321931', '-0.465030', '-0.040365', '-0.750624', '0.648328', '-0.626474', '-0.046682', '0.088066', '-0.408861', '0.046695', '-0.166848', '-0.354168', '0.114459', '0.163043', '-0.365351', '-0.357283', '-0.158008', '0.243679', '-0.222872', '0.535384', '0.484125', '-0.541851', '-0.142395', '0.072372', '-0.380102', '-0.018658', '-0.511147', '-0.558284', '-0.000491', '0.537558', '-0.435427', '0.197865', '0.397918', '-0.023772', '-0.094112', '0.000810'], ['28.0', '-0.177279', '-0.480569', '-0.108229', '-0.816273', '0.109608', '0.065829', '-0.160687', '-0.021853', '-0.340207', '0.096696', '0.434814', '-0.282840', '0.361645', '0.153316', '-0.078206', '0.118571', '0.062528', '-0.219359', '0.694987', '0.000000', '-0.851429', '0.429950', '-0.193886', '0.514009', '-0.005608', '-0.193479', '-0.014731', '-0.482683', '0.792796', '-0.250470', '-0.161096', '0.104637', '-0.373374', '0.005284', '-0.186099', '0.093651', '-0.267529', '0.182050', '-0.143252', '-0.079446', '-0.066302', '0.125232', '-0.223903', '0.200517', '-0.201704', '-0.335222', '-0.189237', '0.035376', '-0.713385', '-0.009693', '-0.388212', '-0.308093', '-0.144610', '0.299114', '-0.062732', '0.009906', '0.132971', '-0.050909', '-0.434289', '0.013422'], ['29.0', '-0.167969', '-0.876901', '-0.407202', '-1.066295', '0.397676', '0.197842', '-0.668195', '0.271815', '-0.703305', '0.021358', '0.552627', '-0.572689', '0.736982', '0.200147', '-0.133973', '0.134819', '0.233487', '-0.726678', '0.733978', '0.000000', '-1.305797', '0.859630', '-0.139072', '1.135416', '-0.086600', '-0.203280', '0.163154', '-0.554822', '1.236941', '-0.866137', '-0.337659', '0.458006', '-0.778659', '-0.007037', '-0.220793', '-0.280971', '-0.309991', '0.402292', '-0.105276', '-0.340721', '-0.030938', '0.477784', '-0.413851', '0.296541', '-0.068361', '-0.517789', '-0.622019', '0.215300', '-1.256609', '-0.021812', '-0.766688', '-0.313648', '-0.036600', '0.462462', '-0.035000', '0.222834', '-0.139266', '-0.135952', '-0.263800', '0.012011'], ['30.0', '-0.035800', '-1.144852', '-0.344517', '-1.028807', '0.229953', '0.358209', '-0.601131', '0.659115', '-0.473294', '-0.062930', '0.427566', '-0.664532', '0.798630', '-0.348927', '-0.097185', '-0.394171', '0.269226', '-0.828960', '0.578020', '0.000000', '-1.115353', '1.161682', '-0.610722', '0.763884', '-0.226901', '-0.066910', '0.164747', '-0.500861', '0.701035', '-0.832856', '-0.039724', '0.490227', '-1.103117', '-0.136609', '-0.234817', '-0.506223', '0.141826', '0.160987', '-0.082398', '-0.423726', '-0.005856', '0.496934', '-0.134036', '0.291707', '0.069427', '-0.426074', '-0.463042', '0.386452', '-0.616732', '-0.010135', '-0.750484', '-0.079848', '-0.172957', '0.410576', '-0.469624', '0.339849', '-0.062488', '-0.155132', '0.197229', '0.000335'], ['31.0', '-1.984400', '-1.581903', '-1.480245', '-2.182427', '-0.289440', '1.770337', '-1.234276', '0.356169', '-0.938523', '-0.137975', '1.779194', '-1.752674', '1.405786', '-0.685376', '-0.502385', '-1.536610', '0.951591', '-0.951714', '1.692856', '0.000000', '-1.850767', '1.461393', '-0.218058', '0.600564', '-0.183971', '-0.590293', '0.455874', '-1.497762', '3.314138', '-0.358236', '-1.150719', '1.357495', '-1.556620', '-0.670112', '-0.153469', '1.059651', '0.541844', '0.951538', '0.293745', '-0.611122', '-0.096861', '2.188742', '-0.149507', '0.065022', '-1.061303', '-1.631804', '-1.050125', '0.240863', '-2.468646', '-0.109331', '-0.979495', '-0.930188', '0.230500', '1.199929', '-0.544561', '0.629915', '-0.329843', '-0.308522', '-0.261864', '0.029225'], ['32.0', '0.143002', '-0.399049', '-0.098485', '-0.660299', '0.326721', '-0.265601', '-0.522350', '0.183304', '-0.551630', '0.054824', '0.068719', '-0.197776', '0.247838', '0.576716', '-0.016349', '0.541101', '0.098949', '-0.406919', '0.162086', '0.000000', '-1.077201', '0.427994', '0.103527', '1.216734', '0.033716', '-0.021767', '0.134577', '-0.100263', '0.529946', '-0.878215', '-0.369655', '0.135694', '-0.336280', '0.163938', '-0.217883', '-0.466436', '-0.783041', '0.221962', '-0.041372', '-0.126337', '-0.003491', '0.019858', '-0.363400', '0.083574', '-0.002787', '-0.067709', '-0.444073', '0.037972', '-0.745033', '-0.001808', '-0.360867', '-0.016290', '-0.072643', '0.085767', '0.352242', '0.038475', '-0.085845', '-0.029969', '-0.412618', '0.005304'], ['33.0', '-0.097208', '-1.463510', '-0.270733', '-1.994183', '0.144593', '0.584492', '-0.718551', '0.482997', '-0.705711', '-0.043813', '0.850382', '-1.116957', '1.055528', '-0.202481', '0.034343', '-0.333566', '0.080769', '-0.818217', '1.309179', '0.000000', '-2.135394', '1.657716', '-1.298271', '1.492492', '-0.260546', '-0.159719', '-0.087999', '-0.962470', '0.976901', '-1.436312', '-0.168973', '0.536154', '-1.205403', '-0.108307', '-0.441237', '-0.615095', '-0.362282', '0.116477', '-0.175017', '-0.335723', '-0.011710', '0.741934', '-0.182297', '0.223685', '-0.052923', '-0.693235', '-0.574161', '0.577047', '-0.763863', '-0.009959', '-0.979986', '-0.321527', '-0.756629', '0.644416', '-0.891870', '0.434617', '0.061402', '-0.022534', '-0.537637', '0.000599'], ['34.0', '-0.019041', '-1.056146', '-0.280047', '-1.214086', '0.028729', '0.446141', '-0.322724', '0.388218', '-0.346654', '0.042535', '0.673882', '-0.824899', '0.662939', '-0.286211', '-0.105239', '-0.439083', '0.270432', '-0.573419', '1.106093', '0.000000', '-1.235425', '1.008192', '-0.668179', '0.775816', '-0.251981', '-0.276442', '0.127139', '-0.865032', '0.883189', '-0.800734', '-0.082617', '0.273365', '-1.015239', '-0.127920', '-0.258045', '-0.289304', '0.061410', '0.122913', '-0.189521', '-0.299873', '-0.070368', '0.605184', '0.076539', '0.488234', '0.074455', '-0.644878', '-0.226821', '0.229270', '-0.383858', '-0.013216', '-0.578361', '-0.425562', '-0.235212', '0.571419', '-0.691669', '0.341948', '0.214740', '-0.156218', '-0.106803', '0.000717'], ['35.0', '-0.084991', '-0.579281', '-0.218837', '-0.681082', '0.221176', '0.163373', '-0.457960', '0.218096', '-0.455278', '-0.016592', '0.301964', '-0.363597', '0.493436', '0.077009', '-0.029901', '-0.004218', '0.085340', '-0.478676', '0.369446', '0.000000', '-0.843419', '0.603307', '-0.221295', '0.691066', '-0.096282', '-0.078722', '0.041697', '-0.336954', '0.611992', '-0.606126', '-0.146519', '0.338823', '-0.505316', '-0.007381', '-0.173958', '-0.284647', '-0.181349', '0.178166', '-0.059197', '-0.223649', '-0.004338', '0.315270', '-0.254839', '0.118393', '-0.001744', '-0.290000', '-0.425264', '0.247136', '-0.678973', '-0.009333', '-0.517125', '-0.149433', '-0.139274', '0.276617', '-0.143440', '0.175434', '-0.066833', '-0.049196', '-0.122311', '0.004014'], ['36.0', '-0.319111', '-0.676949', '-0.474417', '-0.862769', '0.097601', '0.657623', '-0.229066', '-0.121012', '-0.341410', '0.019729', '0.996555', '-0.698887', '0.689232', '-0.265949', '-0.256766', '-0.415402', '0.078381', '-0.389642', '1.302732', '0.000000', '-0.663634', '0.573590', '-0.345852', '0.360815', '-0.253555', '-0.425759', '0.008517', '-0.939311', '1.323833', '-0.362326', '-0.095182', '0.488181', '-0.555488', '-0.236268', '-0.131430', '0.070482', '0.382438', '0.239345', '-0.135597', '-0.276640', '-0.099885', '0.906676', '0.032486', '0.341404', '0.122230', '-0.891797', '-0.399559', '0.263143', '-0.708463', '-0.054052', '-0.614403', '-0.657156', '0.115325', '0.679729', '-0.664624', '0.223084', '0.038385', '-0.114843', '-0.062288', '0.003381'], ['37.0', '-0.414567', '-0.654513', '-0.327593', '-0.796076', '0.160669', '0.150812', '-0.550806', '0.312305', '-0.542452', '0.024497', '0.240442', '-0.402225', '0.518873', '0.008652', '-0.066573', '-0.086621', '0.317655', '-0.541079', '0.228164', '0.000000', '-0.892506', '0.601825', '-0.031915', '0.515928', '-0.016500', '-0.125528', '0.182533', '-0.291393', '0.963058', '-0.322972', '-0.351925', '0.322722', '-0.649169', '-0.041549', '-0.079760', '0.094791', '-0.100743', '0.366884', '-0.060157', '-0.265240', '-0.028151', '0.252297', '-0.404648', '0.122410', '-0.388202', '-0.280397', '-0.344251', '0.067724', '-1.141765', '-0.010568', '-0.482443', '-0.132822', '-0.061287', '0.286915', '0.169472', '0.172725', '-0.036679', '-0.082873', '-0.206363', '0.017370'], ['38.0', '-0.458885', '-1.248434', '-0.542441', '-1.435939', '0.087465', '0.889013', '-0.472619', '0.173413', '-0.357815', '-0.128709', '1.491161', '-1.139457', '0.921842', '-0.288640', '-0.399223', '-0.689896', '0.476753', '-0.705986', '2.013335', '0.000000', '-1.419648', '1.329050', '-0.625378', '0.791642', '-0.285029', '-0.301710', '0.207789', '-1.343895', '1.832495', '-0.821963', '-0.182568', '0.665717', '-1.155492', '-0.301355', '-0.303975', '-0.155425', '0.134999', '0.259361', '-0.000916', '-0.369715', '-0.014823', '1.270330', '0.305764', '0.655311', '-0.086546', '-1.148395', '-0.568808', '0.443154', '-0.819068', '-0.051284', '-0.889173', '-0.686551', '0.026499', '0.794893', '-0.900049', '0.301406', '-0.150242', '-0.381525', '-0.014594', '0.002004'], ['39.0', '-0.236277', '-0.777441', '-0.234422', '-0.924011', '0.376682', '0.214238', '-0.613625', '0.215834', '-0.699946', '0.002258', '0.411798', '-0.416520', '0.744267', '0.183037', '-0.041437', '0.183536', '0.018001', '-0.675969', '0.491747', '0.000000', '-1.157615', '0.823386', '-0.321752', '0.894791', '-0.058762', '-0.112398', '-0.006123', '-0.401635', '0.918405', '-0.684372', '-0.116949', '0.458111', '-0.601986', '0.001228', '-0.208844', '-0.250770', '-0.245158', '0.286872', '-0.138000', '-0.249379', '-0.016656', '0.312123', '-0.511072', '0.157218', '-0.137108', '-0.377223', '-0.652116', '0.367894', '-1.286131', '-0.013656', '-0.819207', '-0.192657', '-0.263067', '0.367845', '-0.049202', '0.170379', '-0.108356', '-0.032960', '-0.336947', '0.016414']])

    def test_mass_flux_on_resampled_lines(self):
        with Serafin.Read(self.path, 'fr') as f:
            f.read_header()
            f.get_time()

            # Resample sections and overwrite them
            resampled_sections = []
            for section in self.sections:
                resampled_sections.append(section.resample(0.1))
            sections = resampled_sections

            # Compute with resampled lines
            calculator = FluxCalculator(FluxCalculator.MASS_FLUX, ['U', 'V', 'H', 'M'], f,
                                        self.names, sections, 1)
            calculator.construct_triangles()
            calculator.construct_intersections()
            result = calculator.run(fmt_float=FMT_FLOAT)
        self.assertEqual(result, [['0.0', '-0.274313', '-1.652278', '-0.684448', '-1.867955', '0.372551', '0.746949', '-0.653220', '0.343606', '-0.543767', '-0.145061', '1.785679', '-1.351363', '1.095366', '-0.090347', '-0.533630', '-0.566228', '0.698715', '-1.020388', '2.523243', '0.000000', '-2.073888', '1.755359', '-0.587313', '1.424664', '-0.324333', '-0.355981', '0.396478', '-1.627649', '2.332502', '-1.300096', '-0.384759', '0.703085', '-1.548257', '-0.237325', '-0.455650', '-0.390769', '-0.194893', '0.408926', '-0.025892', '-0.546575', '0.012302', '1.385813', '0.298171', '0.990620', '0.019868', '-1.336574', '-0.685807', '0.408816', '-1.081772', '-0.053861', '-1.130079', '-0.816861', '0.135542', '0.936695', '-0.891779', '0.359112', '-0.314760', '-0.573652', '-0.006791', '0.003773'], ['1.0', '-0.073925', '-0.573058', '-0.247339', '-0.770693', '0.181372', '0.289168', '-0.088695', '-0.249566', '-0.250149', '0.005148', '1.044077', '-0.497874', '0.506131', '0.045862', '-0.241894', '-0.124936', '0.127493', '-0.312414', '1.580744', '0.000000', '-0.800762', '0.551207', '-0.231059', '0.528695', '-0.161789', '-0.308881', '0.030796', '-0.963643', '1.256579', '-0.441748', '-0.094334', '0.248642', '-0.470129', '-0.093312', '-0.203759', '-0.094608', '-0.029261', '0.169590', '-0.153706', '-0.176981', '-0.043017', '0.555212', '0.081049', '0.460145', '0.126851', '-0.754187', '-0.297678', '0.162128', '-0.539173', '-0.027148', '-0.580658', '-0.603714', '0.052058', '0.545740', '-0.540998', '0.040476', '0.010270', '-0.217325', '-0.124353', '0.002559'], ['2.0', '-0.478501', '-1.314759', '-0.462583', '-1.215347', '-0.054762', '0.932601', '-0.110837', '0.413456', '-0.069253', '-0.070417', '1.264107', '-0.890276', '0.837034', '-0.603543', '-0.423904', '-0.813487', '0.673034', '-0.645223', '1.820816', '0.000000', '-1.027438', '1.296399', '-0.529490', '0.252985', '-0.267508', '-0.278878', '0.283414', '-1.265812', '1.661415', '-0.363409', '0.020612', '0.466813', '-1.347294', '-0.314381', '-0.160537', '0.061898', '0.513711', '0.246673', '-0.074605', '-0.371122', '-0.039697', '0.973362', '0.296814', '0.923252', '-0.329875', '-1.014310', '-0.225001', '0.271614', '-0.641896', '-0.028874', '-0.714611', '-0.661387', '0.112954', '0.732229', '-0.844744', '0.282755', '0.049579', '-0.515845', '0.275142', '0.004983'], ['3.0', '-0.109128', '-0.912257', '-0.470291', '-0.911799', '0.009341', '0.626357', '-0.065450', '0.052054', '-0.215063', '0.120414', '1.052137', '-0.831546', '0.646067', '-0.406332', '-0.357086', '-0.593909', '0.319171', '-0.521735', '1.644656', '0.000000', '-0.750118', '0.675724', '-0.386633', '0.354736', '-0.394784', '-0.503555', '0.179190', '-1.263644', '1.366001', '-0.460016', '-0.079988', '0.328629', '-0.914811', '-0.257862', '-0.150814', '-0.098595', '0.553562', '0.197588', '-0.192280', '-0.419533', '-0.164278', '0.880888', '0.316827', '0.742888', '0.208963', '-1.022624', '-0.208308', '0.120445', '-0.362968', '-0.042076', '-0.602917', '-0.741877', '0.235093', '0.773577', '-0.836944', '0.252157', '0.365847', '-0.280630', '0.200083', '0.000340'], ['4.0', '-0.497275', '-2.037566', '-0.895737', '-2.106710', '0.338992', '1.367063', '-0.570519', '0.248268', '-0.656862', '-0.101520', '2.280219', '-1.616008', '1.662580', '-0.682569', '-0.642517', '-0.992527', '0.507234', '-1.234173', '3.143709', '0.000000', '-1.908359', '1.996320', '-1.044842', '1.058140', '-0.607676', '-0.668123', '0.206620', '-2.154397', '2.869096', '-1.172050', '-0.051316', '1.073147', '-1.832847', '-0.475529', '-0.399923', '-0.331844', '0.720093', '0.427895', '-0.267577', '-0.739110', '-0.102679', '1.872054', '0.216002', '1.160891', '0.168579', '-1.882026', '-0.848499', '0.764261', '-1.401727', '-0.088706', '-1.584371', '-1.213219', '0.112372', '1.406378', '-1.577182', '0.518106', '-0.066412', '-0.535832', '0.187528', '0.002366'], ['5.0', '-0.853637', '-1.082450', '-0.998756', '-1.491949', '-0.081520', '1.240448', '-0.503934', '-0.185485', '-0.690380', '0.080680', '1.639767', '-1.292146', '1.035857', '-0.419634', '-0.479919', '-0.949154', '0.254673', '-0.643049', '2.006477', '0.000000', '-1.231520', '0.815586', '-0.365625', '0.605466', '-0.413447', '-0.775192', '0.136845', '-1.633567', '2.400621', '-0.552177', '-0.554794', '0.881949', '-0.916353', '-0.486003', '-0.196985', '0.333863', '0.638646', '0.509636', '-0.056791', '-0.503276', '-0.218253', '1.733038', '0.137841', '0.382245', '-0.001066', '-1.561188', '-0.751293', '0.244004', '-1.301309', '-0.105232', '-0.862042', '-1.084454', '0.290187', '1.152546', '-0.950813', '0.407911', '0.190046', '-0.160990', '-0.141114', '0.009360'], ['6.0', '-0.273537', '-1.220497', '-0.502228', '-0.958580', '0.436158', '0.595096', '-0.767319', '0.628785', '-0.614494', '-0.120447', '0.507095', '-0.620562', '1.162459', '-0.557614', '-0.101835', '-0.407355', '0.159428', '-1.014407', '0.448994', '0.000000', '-0.802340', '1.241387', '-0.650809', '0.533305', '-0.294412', '-0.048149', '0.016445', '-0.530327', '0.942337', '-0.684909', '0.181244', '0.900195', '-1.121336', '-0.210307', '-0.145875', '-0.540541', '0.540348', '0.263761', '-0.075013', '-0.590742', '0.002412', '0.688897', '-0.426535', '0.229419', '0.017078', '-0.530612', '-0.706771', '0.747620', '-1.072455', '-0.024968', '-1.125649', '-0.110099', '-0.075568', '0.524830', '-0.531893', '0.387224', '-0.181994', '-0.120542', '0.385725', '0.001104'], ['7.0', '-1.168028', '-1.121527', '-0.640196', '-1.650719', '-0.113698', '0.603024', '-0.981015', '0.522056', '-0.772359', '-0.045229', '0.652783', '-0.910901', '0.805889', '-0.050922', '-0.156709', '-0.444371', '0.702640', '-0.695667', '0.594230', '0.000000', '-1.724003', '1.122537', '-0.108734', '0.855073', '0.088902', '-0.152663', '0.379538', '-0.555976', '1.844631', '-0.432844', '-0.859849', '0.607480', '-1.141112', '-0.188367', '-0.218194', '0.572515', '-0.347540', '0.628925', '0.117335', '-0.280768', '-0.012483', '0.731243', '-0.489766', '0.019363', '-1.058016', '-0.581573', '-0.580803', '0.044292', '-1.874664', '-0.024889', '-0.592755', '-0.253523', '-0.232127', '0.481578', '0.198345', '0.289369', '-0.169583', '-0.194902', '-0.587207', '0.027922'], ['8.0', '-0.399015', '-0.574647', '-0.227976', '-0.904964', '0.126718', '0.191620', '-0.412364', '0.090851', '-0.401216', '0.015345', '0.528395', '-0.425921', '0.453166', '0.194222', '-0.134178', '0.087580', '0.219124', '-0.318462', '0.679324', '0.000000', '-0.984738', '0.575859', '-0.095277', '0.657156', '0.049306', '-0.140152', '0.107819', '-0.412028', '1.052323', '-0.301612', '-0.303452', '0.259149', '-0.508764', '-0.037481', '-0.217659', '0.201064', '-0.360861', '0.279706', '-0.040025', '-0.085036', '-0.022160', '0.327265', '-0.246646', '0.164504', '-0.337172', '-0.399742', '-0.362362', '0.089099', '-1.014726', '-0.017623', '-0.443721', '-0.280238', '-0.111671', '0.307586', '0.063079', '0.055970', '-0.114357', '-0.119831', '-0.464175', '0.017611'], ['9.0', '-0.171062', '-1.686314', '-0.915240', '-1.932781', '0.633232', '0.694019', '-0.546167', '0.065642', '-1.181156', '0.358188', '1.465696', '-1.153705', '1.465855', '-0.099736', '-0.482118', '-0.146172', '0.205281', '-1.284284', '2.282906', '0.000000', '-1.911286', '1.205460', '-0.428390', '1.354523', '-0.579697', '-1.024131', '0.183060', '-1.930763', '2.718198', '-1.181281', '-0.299208', '0.703425', '-1.450588', '-0.182461', '-0.316279', '-0.205792', '0.450066', '0.731052', '-0.580848', '-0.849383', '-0.339329', '1.154019', '-0.518917', '1.212254', '0.281275', '-1.633649', '-0.791737', '0.232833', '-2.024114', '-0.074969', '-1.490791', '-1.352186', '0.254588', '1.408860', '-0.839240', '0.449150', '0.561213', '-0.287307', '-0.204399', '0.020738'], ['10.0', '-0.183229', '-0.728095', '-0.523106', '-0.799196', '-0.033505', '0.514903', '-0.202104', '0.163456', '-0.286294', '0.080309', '0.669883', '-0.713128', '0.517580', '-0.415621', '-0.168316', '-0.684397', '0.298476', '-0.438487', '0.950755', '0.000000', '-0.656974', '0.507276', '-0.245681', '0.249734', '-0.321324', '-0.431101', '0.156834', '-0.898387', '1.051509', '-0.337559', '-0.272324', '0.298005', '-0.771341', '-0.207412', '-0.100079', '0.044957', '0.439240', '0.236052', '-0.143884', '-0.405142', '-0.110572', '0.744351', '0.114257', '0.381044', '0.145308', '-0.749012', '-0.152154', '0.037824', '-0.372502', '-0.029890', '-0.400115', '-0.573798', '0.128848', '0.640800', '-0.622982', '0.354178', '0.246990', '-0.142179', '0.186133', '0.001256'], ['11.0', '-0.005051', '-1.178076', '-0.483019', '-1.258881', '0.282733', '0.313862', '-0.550752', '0.454746', '-0.451563', '-0.008395', '0.829224', '-0.863831', '0.766025', '-0.148127', '-0.239494', '-0.322962', '0.581711', '-0.789903', '1.329811', '0.000000', '-1.356993', '1.111799', '-0.294322', '1.002907', '-0.209055', '-0.291606', '0.368221', '-0.916465', '1.445801', '-0.984899', '-0.274464', '0.362109', '-1.236879', '-0.082483', '-0.249813', '-0.315101', '-0.062767', '0.361708', '-0.138680', '-0.433425', '-0.031926', '0.682183', '-0.046326', '0.716439', '0.042045', '-0.749178', '-0.334731', '0.128589', '-0.768462', '-0.016277', '-0.703548', '-0.500317', '0.082000', '0.629359', '-0.485794', '0.333669', '-0.046413', '-0.353890', '0.070408', '0.000320'], ['12.0', '-0.622697', '-1.956013', '-0.882262', '-2.368706', '0.015431', '1.230529', '-0.626576', '0.415447', '-0.727581', '0.043056', '1.799258', '-1.642032', '1.416810', '-0.610694', '-0.410636', '-1.058179', '0.638842', '-1.055635', '2.568197', '0.000000', '-2.212525', '1.801896', '-0.950843', '1.144104', '-0.490689', '-0.697710', '0.259654', '-1.977564', '2.600690', '-1.114102', '-0.436810', '0.792097', '-1.860913', '-0.392933', '-0.412720', '-0.004864', '0.379692', '0.493152', '-0.259512', '-0.649840', '-0.150982', '1.598422', '0.066696', '0.922043', '-0.168393', '-1.633527', '-0.599232', '0.397875', '-1.349230', '-0.059533', '-1.204150', '-1.147201', '-0.152503', '1.319844', '-1.308348', '0.611078', '0.254951', '-0.402589', '-0.177729', '0.008958'], ['13.0', '-0.048740', '-0.070079', '-0.107904', '-0.079066', '0.012830', '0.155206', '0.008589', '-0.072274', '-0.027029', '0.007073', '0.220752', '-0.132182', '0.091658', '-0.063288', '-0.076082', '-0.112670', '-0.007707', '-0.038363', '0.263148', '0.000000', '-0.018880', '0.033415', '-0.027292', '0.004019', '-0.088033', '-0.115352', '-0.006126', '-0.214461', '0.246584', '-0.019554', '-0.007121', '0.099721', '-0.049160', '-0.063969', '-0.004315', '0.021455', '0.139604', '0.027186', '-0.025231', '-0.064819', '-0.026936', '0.244284', '0.078290', '0.085888', '0.093839', '-0.212266', '-0.059189', '0.054208', '-0.056791', '-0.017419', '-0.082619', '-0.169562', '0.091762', '0.146583', '-0.153617', '0.050178', '0.016493', '-0.019609', '0.037948', '0.000024'], ['14.0', '-0.314466', '-0.602433', '-0.443290', '-0.884847', '0.358767', '0.216232', '-0.451947', '0.004456', '-0.735806', '0.160335', '0.471883', '-0.409727', '0.617836', '0.309354', '-0.144745', '0.340548', '0.021818', '-0.539027', '0.599231', '0.000000', '-1.055007', '0.447975', '0.071953', '0.983554', '-0.059152', '-0.372187', '0.072508', '-0.566084', '1.264404', '-0.535812', '-0.417880', '0.368579', '-0.433242', '-0.006167', '-0.137736', '0.081599', '-0.209881', '0.483155', '-0.152077', '-0.293560', '-0.115514', '0.425318', '-0.544139', '0.255709', '-0.140999', '-0.555523', '-0.547414', '0.044677', '-1.429659', '-0.034511', '-0.631466', '-0.459333', '0.020396', '0.499829', '0.140015', '0.163049', '0.042888', '-0.038051', '-0.478138', '0.021825'], ['15.0', '-0.315609', '-0.511639', '-0.374158', '-0.547517', '0.004130', '0.679086', '-0.054321', '-0.199371', '-0.000054', '-0.115253', '1.242785', '-0.800105', '0.396815', '-0.216153', '-0.403881', '-0.571864', '0.254694', '-0.246172', '1.593912', '0.000000', '-0.467227', '0.548324', '-0.252968', '0.222668', '-0.271257', '-0.301288', '0.139527', '-1.039079', '1.267756', '-0.245378', '-0.128835', '0.401021', '-0.452287', '-0.279330', '-0.116932', '0.051180', '0.326451', '0.094292', '0.033953', '-0.217725', '-0.001200', '1.111772', '0.588504', '0.510524', '0.167607', '-0.938008', '-0.245723', '0.276539', '-0.220888', '-0.061226', '-0.411489', '-0.599689', '0.271924', '0.560544', '-0.682621', '0.166463', '-0.159624', '-0.280729', '0.101979', '0.000081'], ['16.0', '-0.150006', '-0.979777', '-0.493811', '-1.068078', '0.421861', '0.361098', '-0.735944', '0.359892', '-0.640433', '-0.070338', '0.691482', '-0.706709', '0.800358', '0.043535', '-0.190546', '-0.146822', '0.352067', '-0.819399', '0.855316', '0.000000', '-1.249258', '1.008416', '-0.199104', '0.998163', '-0.190870', '-0.209922', '0.219448', '-0.657200', '1.335021', '-0.973447', '-0.229658', '0.581398', '-0.923139', '-0.061732', '-0.205089', '-0.385961', '-0.152223', '0.381512', '-0.072224', '-0.414147', '-0.001352', '0.739220', '-0.262950', '0.436264', '0.040800', '-0.641682', '-0.656139', '0.334193', '-1.127781', '-0.031212', '-0.812058', '-0.389844', '0.089628', '0.532530', '-0.266746', '0.321080', '-0.251304', '-0.209720', '0.005862', '0.004241'], ['17.0', '-0.236676', '-0.864793', '-0.632609', '-0.978185', '0.172326', '0.533195', '-0.370930', '0.074903', '-0.541119', '0.101182', '0.833954', '-0.755119', '0.733490', '-0.280862', '-0.278124', '-0.522393', '0.242294', '-0.651585', '1.172603', '0.000000', '-0.874165', '0.633896', '-0.232676', '0.459885', '-0.312331', '-0.503790', '0.181355', '-1.008242', '1.444431', '-0.577582', '-0.262372', '0.461375', '-0.790746', '-0.210344', '-0.119433', '0.000812', '0.425505', '0.395345', '-0.143878', '-0.472156', '-0.156554', '0.835089', '-0.068723', '0.478948', '0.120216', '-0.901043', '-0.400435', '0.072876', '-0.856348', '-0.047645', '-0.661754', '-0.643426', '0.243278', '0.733406', '-0.598165', '0.291284', '0.196374', '-0.160849', '0.112765', '0.002962'], ['18.0', '0.131110', '-1.196546', '-0.372895', '-1.430550', '0.146578', '0.812094', '0.378564', '-0.542258', '-0.260395', '0.201428', '2.109514', '-1.029164', '0.913611', '-0.157115', '-0.628741', '-0.310217', '0.041363', '-0.544285', '3.465461', '0.000000', '-1.373717', '1.012409', '-0.786354', '0.896194', '-0.519967', '-0.793198', '-0.009883', '-2.223667', '2.162756', '-0.849603', '0.077353', '0.287277', '-0.911724', '-0.272653', '-0.341301', '-0.361903', '0.439362', '0.140219', '-0.441457', '-0.378294', '-0.235435', '1.153452', '0.541635', '1.277897', '0.493986', '-1.667567', '-0.348510', '0.238979', '-0.443628', '-0.067587', '-1.048975', '-1.322093', '0.179872', '1.181879', '-1.448126', '0.070682', '0.556949', '-0.439439', '-0.152058', '0.000453'], ['19.0', '-0.501787', '-0.901413', '-0.237842', '-1.298228', '0.052591', '0.446162', '-0.522668', '0.257272', '-0.455276', '-0.053632', '0.676524', '-0.625396', '0.637676', '0.083121', '-0.119850', '-0.132338', '0.277554', '-0.491282', '0.892276', '0.000000', '-1.416330', '1.034851', '-0.467194', '0.828043', '-0.013163', '-0.052917', '0.077758', '-0.608968', '1.097716', '-0.653645', '-0.235630', '0.409354', '-0.779520', '-0.083778', '-0.286156', '0.005240', '-0.436689', '0.237835', '0.022757', '-0.137758', '0.003952', '0.505484', '-0.192198', '0.209195', '-0.504594', '-0.494568', '-0.498145', '0.264493', '-1.009481', '-0.011904', '-0.615943', '-0.257651', '-0.316246', '0.364705', '-0.225184', '0.128917', '-0.089380', '-0.166213', '-0.453701', '0.011716'], ['20.0', '-0.240063', '-0.365412', '-0.391404', '-0.567048', '-0.090686', '0.536468', '0.097599', '-0.571168', '-0.127279', '0.145155', '1.282899', '-0.682535', '0.407429', '-0.152653', '-0.508441', '-0.365273', '0.047833', '-0.144681', '1.915155', '0.000000', '-0.331309', '0.135853', '-0.172212', '0.131990', '-0.216634', '-0.535627', '0.095505', '-1.141199', '1.460116', '-0.130832', '-0.088107', '0.285140', '-0.241040', '-0.324461', '-0.086037', '0.136619', '0.525591', '0.158073', '-0.077849', '-0.145308', '-0.221288', '0.867028', '0.509793', '0.401102', '0.251773', '-1.038594', '-0.321494', '-0.036369', '-0.354511', '-0.077170', '-0.448607', '-0.694328', '0.510256', '0.642845', '-0.639094', '-0.106201', '0.302799', '-0.154760', '-0.063902', '0.002734'], ['21.0', '-0.658853', '-0.690402', '-0.511553', '-0.912936', '-0.010511', '0.650814', '-0.038961', '-0.300601', '-0.355788', '0.229016', '1.126965', '-0.621908', '0.667189', '-0.166795', '-0.466864', '-0.258697', '0.247341', '-0.371702', '1.662244', '0.000000', '-0.644982', '0.369923', '-0.059279', '0.091661', '-0.119667', '-0.584204', '0.155086', '-1.137217', '1.881981', '0.095384', '-0.174052', '0.327458', '-0.587766', '-0.290472', '-0.064820', '0.611263', '0.506995', '0.444894', '-0.157985', '-0.226618', '-0.250927', '0.692417', '-0.013951', '0.575969', '-0.326067', '-1.046293', '-0.372832', '-0.115868', '-1.268684', '-0.058425', '-0.635759', '-0.798009', '0.364518', '0.744877', '-0.330222', '-0.032239', '0.359146', '-0.237255', '-0.245814', '0.024494'], ['22.0', '-0.032018', '-0.166778', '-0.050304', '-0.161499', '0.001516', '0.063502', '-0.083129', '0.095904', '-0.049549', '-0.011192', '0.079853', '-0.116284', '0.101657', '-0.059766', '-0.020054', '-0.088955', '0.073820', '-0.104318', '0.112223', '0.000000', '-0.168085', '0.167816', '-0.080187', '0.089869', '-0.026435', '-0.011041', '0.039984', '-0.088304', '0.130013', '-0.100906', '-0.023173', '0.059144', '-0.175661', '-0.026552', '-0.032893', '-0.039268', '0.018892', '0.024967', '-0.003815', '-0.050635', '-0.000262', '0.084988', '0.007295', '0.051103', '-0.022062', '-0.071931', '-0.046525', '0.038398', '-0.078944', '-0.001175', '-0.086141', '-0.020944', '-0.023599', '0.061530', '-0.069430', '0.048485', '-0.002716', '-0.033588', '0.020574', '0.000235'], ['23.0', '-0.276730', '-0.934430', '-0.564742', '-0.739113', '0.167247', '0.755434', '-0.410284', '0.356993', '-0.389024', '-0.070378', '0.712709', '-0.706325', '0.893362', '-0.655096', '-0.153360', '-0.711394', '0.129720', '-0.666095', '0.719805', '0.000000', '-0.517842', '0.847602', '-0.502496', '0.272285', '-0.403084', '-0.283909', '-0.011312', '-0.763171', '0.978872', '-0.377248', '-0.017611', '0.732634', '-0.884096', '-0.288029', '-0.072173', '-0.250529', '0.766287', '0.180330', '-0.119071', '-0.531835', '-0.037190', '0.954328', '-0.061841', '0.211354', '0.233461', '-0.739311', '-0.447850', '0.546900', '-0.618856', '-0.045253', '-0.764604', '-0.400365', '0.034563', '0.653943', '-0.727051', '0.458917', '-0.041847', '-0.078847', '0.389471', '0.000181'], ['24.0', '-0.025667', '-0.327982', '-0.054882', '-0.519965', '0.028343', '0.053099', '-0.100644', '0.082775', '-0.100140', '0.002871', '0.266022', '-0.213312', '0.166897', '0.044180', '-0.058546', '-0.016611', '0.096452', '-0.118285', '0.438467', '0.000000', '-0.537602', '0.351712', '-0.181468', '0.347114', '-0.024079', '-0.050576', '0.039313', '-0.266452', '0.348630', '-0.260578', '-0.087192', '0.050276', '-0.295659', '-0.001608', '-0.156117', '-0.039692', '-0.223230', '0.047490', '-0.029971', '-0.042577', '-0.006106', '0.130419', '-0.005529', '0.155067', '-0.073967', '-0.177043', '-0.059410', '0.034527', '-0.152294', '-0.002484', '-0.157068', '-0.139741', '-0.088930', '0.136495', '-0.146049', '0.030297', '0.008704', '-0.084585', '-0.164455', '0.000735'], ['25.0', '0.058431', '-0.388865', '-0.145193', '-0.390491', '0.176450', '0.017635', '-0.211498', '0.152597', '-0.239118', '0.010379', '0.183806', '-0.215101', '0.291771', '-0.044973', '-0.043547', '-0.051344', '0.109628', '-0.320759', '0.306074', '0.000000', '-0.433407', '0.354582', '-0.125697', '0.326814', '-0.092267', '-0.100408', '0.071852', '-0.257252', '0.410322', '-0.346705', '-0.056582', '0.134560', '-0.375675', '0.000698', '-0.073043', '-0.182907', '0.013148', '0.122890', '-0.093390', '-0.185202', '-0.015754', '0.139564', '-0.132919', '0.200546', '0.063916', '-0.193352', '-0.127308', '0.080944', '-0.323127', '-0.004360', '-0.297392', '-0.128553', '-0.003966', '0.198180', '-0.135991', '0.113784', '0.011042', '-0.065941', '0.039149', '0.000209'], ['26.0', '-0.035064', '-0.533379', '-0.094872', '-0.518280', '0.104368', '0.247450', '-0.087749', '0.156960', '-0.170768', '-0.013036', '0.395654', '-0.257076', '0.388268', '-0.109996', '-0.083873', '-0.103651', '0.083984', '-0.304368', '0.604980', '0.000000', '-0.543711', '0.552296', '-0.305941', '0.321886', '-0.113950', '-0.102398', '0.002161', '-0.432742', '0.510621', '-0.336442', '0.074294', '0.186590', '-0.465102', '-0.044510', '-0.112297', '-0.172092', '0.069050', '0.063762', '-0.110567', '-0.136649', '-0.011450', '0.258372', '-0.071157', '0.315159', '-0.017414', '-0.319470', '-0.167753', '0.208095', '-0.332179', '-0.006521', '-0.383237', '-0.228952', '-0.122714', '0.272609', '-0.325816', '0.115950', '0.037108', '-0.113615', '-0.010242', '0.001080'], ['27.0', '0.374701', '-0.539074', '-0.227842', '-0.618281', '0.352181', '0.018404', '-0.029661', '-0.038076', '-0.453142', '0.227078', '0.381399', '-0.314382', '0.454735', '0.057352', '-0.095400', '0.168928', '-0.156181', '-0.431958', '0.803311', '0.000000', '-0.677497', '0.291063', '-0.238769', '0.705106', '-0.321931', '-0.465030', '-0.040365', '-0.750624', '0.648328', '-0.626474', '-0.046682', '0.088066', '-0.408861', '0.046695', '-0.166848', '-0.354168', '0.114459', '0.163043', '-0.365351', '-0.357283', '-0.158008', '0.243679', '-0.222872', '0.535384', '0.484125', '-0.541851', '-0.142395', '0.072372', '-0.380102', '-0.018658', '-0.511147', '-0.558284', '-0.000491', '0.537558', '-0.435427', '0.197865', '0.397918', '-0.023772', '-0.094112', '0.000810'], ['28.0', '-0.177279', '-0.480569', '-0.108229', '-0.816273', '0.109608', '0.065829', '-0.160687', '-0.021853', '-0.340207', '0.096696', '0.434814', '-0.282840', '0.361645', '0.153316', '-0.078206', '0.118571', '0.062528', '-0.219359', '0.694987', '0.000000', '-0.851429', '0.429950', '-0.193886', '0.514009', '-0.005608', '-0.193479', '-0.014731', '-0.482683', '0.792796', '-0.250470', '-0.161096', '0.104637', '-0.373374', '0.005284', '-0.186099', '0.093651', '-0.267529', '0.182050', '-0.143252', '-0.079446', '-0.066302', '0.125232', '-0.223903', '0.200517', '-0.201704', '-0.335222', '-0.189237', '0.035376', '-0.713385', '-0.009693', '-0.388212', '-0.308093', '-0.144610', '0.299114', '-0.062732', '0.009906', '0.132971', '-0.050909', '-0.434289', '0.013422'], ['29.0', '-0.167969', '-0.876901', '-0.407202', '-1.066295', '0.397676', '0.197842', '-0.668195', '0.271815', '-0.703305', '0.021358', '0.552627', '-0.572689', '0.736982', '0.200147', '-0.133973', '0.134819', '0.233487', '-0.726678', '0.733978', '0.000000', '-1.305797', '0.859630', '-0.139072', '1.135416', '-0.086600', '-0.203280', '0.163154', '-0.554822', '1.236941', '-0.866137', '-0.337659', '0.458006', '-0.778659', '-0.007037', '-0.220793', '-0.280971', '-0.309991', '0.402292', '-0.105276', '-0.340721', '-0.030938', '0.477784', '-0.413851', '0.296541', '-0.068361', '-0.517789', '-0.622019', '0.215300', '-1.256609', '-0.021812', '-0.766688', '-0.313648', '-0.036600', '0.462462', '-0.035000', '0.222834', '-0.139266', '-0.135952', '-0.263800', '0.012011'], ['30.0', '-0.035800', '-1.144852', '-0.344517', '-1.028807', '0.229953', '0.358209', '-0.601131', '0.659115', '-0.473294', '-0.062930', '0.427566', '-0.664532', '0.798630', '-0.348927', '-0.097185', '-0.394171', '0.269226', '-0.828960', '0.578020', '0.000000', '-1.115353', '1.161682', '-0.610722', '0.763884', '-0.226901', '-0.066910', '0.164747', '-0.500861', '0.701035', '-0.832856', '-0.039724', '0.490227', '-1.103117', '-0.136609', '-0.234817', '-0.506223', '0.141826', '0.160987', '-0.082398', '-0.423726', '-0.005856', '0.496934', '-0.134036', '0.291707', '0.069427', '-0.426074', '-0.463042', '0.386452', '-0.616732', '-0.010135', '-0.750484', '-0.079848', '-0.172957', '0.410576', '-0.469624', '0.339849', '-0.062488', '-0.155132', '0.197229', '0.000335'], ['31.0', '-1.984400', '-1.581903', '-1.480245', '-2.182427', '-0.289440', '1.770337', '-1.234276', '0.356169', '-0.938523', '-0.137975', '1.779194', '-1.752674', '1.405786', '-0.685376', '-0.502385', '-1.536610', '0.951591', '-0.951714', '1.692856', '0.000000', '-1.850767', '1.461393', '-0.218058', '0.600564', '-0.183971', '-0.590293', '0.455874', '-1.497762', '3.314138', '-0.358236', '-1.150719', '1.357495', '-1.556620', '-0.670112', '-0.153469', '1.059651', '0.541844', '0.951538', '0.293745', '-0.611122', '-0.096861', '2.188742', '-0.149507', '0.065022', '-1.061303', '-1.631804', '-1.050125', '0.240863', '-2.468646', '-0.109331', '-0.979495', '-0.930188', '0.230500', '1.199929', '-0.544561', '0.629915', '-0.329843', '-0.308522', '-0.261864', '0.029225'], ['32.0', '0.143002', '-0.399049', '-0.098485', '-0.660299', '0.326721', '-0.265601', '-0.522350', '0.183304', '-0.551630', '0.054824', '0.068719', '-0.197776', '0.247838', '0.576716', '-0.016349', '0.541101', '0.098949', '-0.406919', '0.162086', '0.000000', '-1.077201', '0.427994', '0.103527', '1.216734', '0.033716', '-0.021767', '0.134577', '-0.100263', '0.529946', '-0.878215', '-0.369655', '0.135694', '-0.336280', '0.163938', '-0.217883', '-0.466436', '-0.783041', '0.221962', '-0.041372', '-0.126337', '-0.003491', '0.019858', '-0.363400', '0.083574', '-0.002787', '-0.067709', '-0.444073', '0.037972', '-0.745033', '-0.001808', '-0.360867', '-0.016290', '-0.072643', '0.085767', '0.352242', '0.038475', '-0.085845', '-0.029969', '-0.412618', '0.005304'], ['33.0', '-0.097208', '-1.463510', '-0.270733', '-1.994183', '0.144593', '0.584492', '-0.718551', '0.482997', '-0.705711', '-0.043813', '0.850382', '-1.116957', '1.055528', '-0.202481', '0.034343', '-0.333566', '0.080769', '-0.818217', '1.309179', '0.000000', '-2.135394', '1.657716', '-1.298271', '1.492492', '-0.260546', '-0.159719', '-0.087999', '-0.962470', '0.976901', '-1.436312', '-0.168973', '0.536154', '-1.205403', '-0.108307', '-0.441237', '-0.615095', '-0.362282', '0.116477', '-0.175017', '-0.335723', '-0.011710', '0.741934', '-0.182297', '0.223685', '-0.052923', '-0.693235', '-0.574161', '0.577047', '-0.763863', '-0.009959', '-0.979986', '-0.321527', '-0.756629', '0.644416', '-0.891870', '0.434617', '0.061402', '-0.022534', '-0.537637', '0.000599'], ['34.0', '-0.019041', '-1.056146', '-0.280047', '-1.214086', '0.028729', '0.446141', '-0.322724', '0.388218', '-0.346654', '0.042535', '0.673882', '-0.824899', '0.662939', '-0.286211', '-0.105239', '-0.439083', '0.270432', '-0.573419', '1.106093', '0.000000', '-1.235425', '1.008192', '-0.668179', '0.775816', '-0.251981', '-0.276442', '0.127139', '-0.865032', '0.883189', '-0.800734', '-0.082617', '0.273365', '-1.015239', '-0.127920', '-0.258045', '-0.289304', '0.061410', '0.122913', '-0.189521', '-0.299873', '-0.070368', '0.605184', '0.076539', '0.488234', '0.074455', '-0.644878', '-0.226821', '0.229270', '-0.383858', '-0.013216', '-0.578361', '-0.425562', '-0.235212', '0.571419', '-0.691669', '0.341948', '0.214740', '-0.156218', '-0.106803', '0.000717'], ['35.0', '-0.084991', '-0.579281', '-0.218837', '-0.681082', '0.221176', '0.163373', '-0.457960', '0.218096', '-0.455278', '-0.016592', '0.301964', '-0.363597', '0.493436', '0.077009', '-0.029901', '-0.004218', '0.085340', '-0.478676', '0.369446', '0.000000', '-0.843419', '0.603307', '-0.221295', '0.691066', '-0.096282', '-0.078722', '0.041697', '-0.336954', '0.611992', '-0.606126', '-0.146519', '0.338823', '-0.505316', '-0.007381', '-0.173958', '-0.284647', '-0.181349', '0.178166', '-0.059197', '-0.223649', '-0.004338', '0.315270', '-0.254839', '0.118393', '-0.001744', '-0.290000', '-0.425264', '0.247136', '-0.678973', '-0.009333', '-0.517125', '-0.149433', '-0.139274', '0.276617', '-0.143440', '0.175434', '-0.066833', '-0.049196', '-0.122311', '0.004014'], ['36.0', '-0.319111', '-0.676949', '-0.474417', '-0.862769', '0.097601', '0.657623', '-0.229066', '-0.121012', '-0.341410', '0.019729', '0.996555', '-0.698887', '0.689232', '-0.265949', '-0.256766', '-0.415402', '0.078381', '-0.389642', '1.302732', '0.000000', '-0.663634', '0.573590', '-0.345852', '0.360815', '-0.253555', '-0.425759', '0.008517', '-0.939311', '1.323833', '-0.362326', '-0.095182', '0.488181', '-0.555488', '-0.236268', '-0.131430', '0.070482', '0.382438', '0.239345', '-0.135597', '-0.276640', '-0.099885', '0.906676', '0.032486', '0.341404', '0.122230', '-0.891797', '-0.399559', '0.263143', '-0.708463', '-0.054052', '-0.614403', '-0.657156', '0.115325', '0.679729', '-0.664624', '0.223084', '0.038385', '-0.114843', '-0.062288', '0.003381'], ['37.0', '-0.414567', '-0.654513', '-0.327593', '-0.796076', '0.160669', '0.150812', '-0.550806', '0.312305', '-0.542452', '0.024497', '0.240442', '-0.402225', '0.518873', '0.008652', '-0.066573', '-0.086621', '0.317655', '-0.541079', '0.228164', '0.000000', '-0.892506', '0.601825', '-0.031915', '0.515928', '-0.016500', '-0.125528', '0.182533', '-0.291393', '0.963058', '-0.322972', '-0.351925', '0.322722', '-0.649169', '-0.041549', '-0.079760', '0.094791', '-0.100743', '0.366884', '-0.060157', '-0.265240', '-0.028151', '0.252297', '-0.404648', '0.122410', '-0.388202', '-0.280397', '-0.344251', '0.067724', '-1.141765', '-0.010568', '-0.482443', '-0.132822', '-0.061287', '0.286915', '0.169472', '0.172725', '-0.036679', '-0.082873', '-0.206363', '0.017370'], ['38.0', '-0.458885', '-1.248434', '-0.542441', '-1.435939', '0.087465', '0.889013', '-0.472619', '0.173413', '-0.357815', '-0.128709', '1.491161', '-1.139457', '0.921842', '-0.288640', '-0.399223', '-0.689896', '0.476753', '-0.705986', '2.013335', '0.000000', '-1.419648', '1.329050', '-0.625378', '0.791642', '-0.285029', '-0.301710', '0.207789', '-1.343895', '1.832495', '-0.821963', '-0.182568', '0.665717', '-1.155492', '-0.301355', '-0.303975', '-0.155425', '0.134999', '0.259361', '-0.000916', '-0.369715', '-0.014823', '1.270330', '0.305764', '0.655311', '-0.086546', '-1.148395', '-0.568808', '0.443154', '-0.819068', '-0.051284', '-0.889173', '-0.686551', '0.026499', '0.794893', '-0.900049', '0.301406', '-0.150242', '-0.381525', '-0.014594', '0.002004'], ['39.0', '-0.236277', '-0.777441', '-0.234422', '-0.924011', '0.376682', '0.214238', '-0.613625', '0.215834', '-0.699946', '0.002258', '0.411798', '-0.416520', '0.744267', '0.183037', '-0.041437', '0.183536', '0.018001', '-0.675969', '0.491747', '0.000000', '-1.157615', '0.823386', '-0.321752', '0.894791', '-0.058762', '-0.112398', '-0.006123', '-0.401635', '0.918405', '-0.684372', '-0.116949', '0.458111', '-0.601986', '0.001228', '-0.208844', '-0.250770', '-0.245158', '0.286872', '-0.138000', '-0.249379', '-0.016656', '0.312123', '-0.511072', '0.157218', '-0.137108', '-0.377223', '-0.652116', '0.367894', '-1.286131', '-0.013656', '-0.819207', '-0.192657', '-0.263067', '0.367845', '-0.049202', '0.170379', '-0.108356', '-0.032960', '-0.336947', '0.016414']])
