/**
 * SERWERCALC - Manual de Utilização
 * Script principal para funcionalidades de navegação e interação
 * Versão: 1.1
 * Autor: Eng. Wanderilo Lima
 */

// =============================================
// 1. INICIALIZAÇÃO E VARIÁVEIS GLOBAIS
// =============================================

// Elementos globais
const backToTopButton = document.querySelector('.back-to-top');
const sections = document.querySelectorAll('.content-section');
const navLinks = document.querySelectorAll('.nav-links a, .sub-nav a');

// =============================================
// 2. CONTROLE DE NAVEGAÇÃO
// =============================================

/**
 * 2.1 Scroll suave e atualização de navegação ativa
 * Controla o comportamento de clique nos links de navegação
 */
function setupSmoothScroll() {
  document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function(e) {
      e.preventDefault();
      const target = document.querySelector(this.getAttribute('href'));
      if(target) {
        target.scrollIntoView({ behavior: 'smooth', block: 'start' });
        updateActiveNav(this);
      }
    });
  });
}

/**
 * 2.2 Atualiza o estado ativo da navegação
 * @param {HTMLElement} clickedLink - O elemento de link que foi clicado
 */
function updateActiveNav(clickedLink) {
  // Remove todas as classes ativas
  document.querySelectorAll('.nav-links a, .sub-nav a').forEach(link => {
    link.classList.remove('active');
  });
  
  // Adiciona ativo ao link clicado
  clickedLink.classList.add('active');
  
  // Ativa todos os pais na hierarquia
  let parent = clickedLink.parentElement;
  while(parent) {
    if(parent.classList.contains('sub-nav')) {
      const parentLink = parent.previousElementSibling;
      if(parentLink && parentLink.tagName === 'A') {
        parentLink.classList.add('active');
        parent.parentElement.classList.add('active');
      }
    }
    parent = parent.parentElement;
  }
}

/**
 * 2.3 Controle de expandir/recolher para menus
 * Gerencia o comportamento de clique nos itens principais
 */
function setupMenuToggle() {
  document.querySelectorAll('.nav-links > li > a, .sub-nav > li > a').forEach(link => {
    link.addEventListener('click', function(e) {
      const parentLi = this.parentElement;
      const subNav = parentLi.querySelector('.sub-nav');
      
      if(subNav) {
        // Só previne default se for apenas para expandir/recolher
        if(this.getAttribute('href') === '#') {
          e.preventDefault();
        }
        
        // Fecha todos os submenus do mesmo nível
        const siblings = Array.from(parentLi.parentElement.children);
        siblings.forEach(li => {
          if(li !== parentLi) {
            li.classList.remove('active');
          }
        });
        
        // Alterna o estado do item clicado
        parentLi.classList.toggle('active');
      }
    });
  });
}

/**
 * 2.4 Atualização de navegação durante o scroll
 * Mantém a navegação sincronizada com a posição da página
 */
function setupScrollSpy() {
  window.addEventListener('scroll', () => {
    let current = '';
    
    sections.forEach(section => {
      const sectionTop = section.offsetTop;
      const sectionHeight = section.clientHeight;
      
      if(pageYOffset >= (sectionTop - 200)) {
        current = section.getAttribute('id');
      }
    });
    
    navLinks.forEach(link => {
      link.classList.remove('active');
      if(link.getAttribute('href') === `#${current}`) {
        link.classList.add('active');
        
        // Ativar todos os pais na hierarquia
        let parent = link.parentElement;
        while(parent) {
          if(parent.classList.contains('sub-nav')) {
            const parentLink = parent.previousElementSibling;
            if(parentLink && parentLink.tagName === 'A') {
              parentLink.classList.add('active');
              parent.parentElement.classList.add('active');
            }
          }
          parent = parent.parentElement;
        }
      }
    });
  });
}

/**
 * 2.5 Mantém os menus pais abertos quando um filho está ativo
 */
function maintainOpenParents() {
  document.querySelectorAll('.sub-nav a.active').forEach(activeLink => {
    let parent = activeLink.parentElement;
    while(parent) {
      if(parent.classList.contains('sub-nav')) {
        const parentLi = parent.closest('li');
        if(parentLi) {
          parentLi.classList.add('active');
        }
      }
      parent = parent.parentElement;
    }
  });
}

/** */
document.addEventListener('DOMContentLoaded', function() {
  // Seleciona todos os itens do menu que possuem submenus
  const menuItemsWithSubmenus = document.querySelectorAll('.nav-links > li > a, .sub-nav > li > a');
  
  // Adiciona evento de clique a cada item
  menuItemsWithSubmenus.forEach(item => {
    // Verifica se o item tem um submenu
    const submenu = item.nextElementSibling;
    if (submenu && submenu.classList.contains('sub-nav')) {
      // Adiciona ícone de seta
      const icon = document.createElement('i');
      icon.className = 'fas fa-chevron-down arrow-icon';
      item.appendChild(icon);
      
      // Adiciona evento de clique
      item.addEventListener('click', function(e) {
        // Previne o comportamento padrão apenas se for um item com submenu
        if (this.parentElement.querySelector('.sub-nav')) {
          e.preventDefault();
        }
        
        // Fecha todos os submenus do mesmo nível
        const siblingItems = this.parentElement.parentElement.querySelectorAll('li');
        siblingItems.forEach(sibling => {
          if (sibling !== this.parentElement) {
            const siblingSubmenu = sibling.querySelector('.sub-nav');
            if (siblingSubmenu) {
              siblingSubmenu.style.display = 'none';
              const siblingArrow = sibling.querySelector('.arrow-icon');
              if (siblingArrow) {
                siblingArrow.classList.remove('fa-chevron-up');
                siblingArrow.classList.add('fa-chevron-down');
              }
            }
          }
        });
        
        // Alterna o submenu atual
        const currentSubmenu = this.nextElementSibling;
        if (currentSubmenu && currentSubmenu.classList.contains('sub-nav')) {
          const isHidden = currentSubmenu.style.display === 'none';
          currentSubmenu.style.display = isHidden ? 'block' : 'none';
          
          // Alterna o ícone da seta
          const arrow = this.querySelector('.arrow-icon');
          if (arrow) {
            arrow.classList.toggle('fa-chevron-down');
            arrow.classList.toggle('fa-chevron-up');
          }
        }
      });
    }
  });
  
  // Fecha todos os submenus inicialmente (exceto o ativo)
  const allSubmenus = document.querySelectorAll('.sub-nav');
  allSubmenus.forEach(submenu => {
    if (!submenu.closest('.active')) {
      submenu.style.display = 'none';
    }
  });
});



// =============================================
// 3. BOTÃO "VOLTAR AO TOPO"
// =============================================

/**
 * 3.1 Mostrar/ocultar botão "Voltar ao topo"
 */
function setupBackToTopButton() {
  window.addEventListener('scroll', () => {
    if (window.pageYOffset > 300) {
      backToTopButton.classList.add('visible');
    } else {
      backToTopButton.classList.remove('visible');
    }
  });

  // 3.2 Comportamento de clique
  backToTopButton.addEventListener('click', () => {
    window.scrollTo({ top: 0, behavior: 'smooth' });
  });
}

// =============================================
// 4. TOOLTIPS PARA ACESSIBILIDADE
// =============================================

/**
 * 4.1 Adiciona tooltips aos ícones de navegação
 */
function setupNavigationTooltips() {
  document.querySelectorAll('.nav-links a i, .sub-nav a i').forEach(icon => {
    const parentLink = icon.parentElement;
    const tooltip = document.createElement('span');
    tooltip.className = 'tooltiptext';
    tooltip.textContent = parentLink.textContent.trim();
    
    const tooltipContainer = document.createElement('span');
    tooltipContainer.className = 'tooltip';
    tooltipContainer.appendChild(tooltip);
    
    parentLink.insertBefore(tooltipContainer, icon);
    tooltipContainer.appendChild(icon);
  });
}

// =============================================
// 5. SISTEMA DE ZOOM PARA IMAGENS
// =============================================

/**
 * 5.1 Configura o sistema de zoom para imagens
 */
function setupImageZoom() {
  document.querySelectorAll('.figure img').forEach(img => {
    // Cria o container do zoom
    const zoomContainer = document.createElement('div');
    zoomContainer.className = 'zoom-container';
    
    // Configurações do container
    Object.assign(zoomContainer.style, {
      display: 'none',
      position: 'fixed',
      top: '0',
      left: '0',
      width: '100%',
      height: '100%',
      backgroundColor: 'rgba(0,0,0,0.9)',
      zIndex: '1000',
      justifyContent: 'center',
      alignItems: 'center',
      cursor: 'zoom-out'
    });
    
    // Cria a imagem ampliada
    const zoomedImg = document.createElement('img');
    zoomedImg.src = img.src;
    Object.assign(zoomedImg.style, {
      maxWidth: '90%',
      maxHeight: '90%',
      objectFit: 'contain',
      cursor: 'grab',
      transition: 'transform 0.2s ease-out'
    });
    zoomedImg.draggable = false;
    
    // Botão de fechar estilizado
    const closeBtn = document.createElement('button');
    closeBtn.innerHTML = '&times;';
    Object.assign(closeBtn.style, {
      position: 'absolute',
      top: '20px',
      right: '20px',
      background: 'var(--ocre-dark)',
      color: 'white',
      border: 'none',
      borderRadius: '50%',
      width: '40px',
      height: '40px',
      fontSize: '1.5rem',
      cursor: 'pointer',
      zIndex: '1001'
    });
    
    // Monta a estrutura
    zoomContainer.appendChild(zoomedImg);
    zoomContainer.appendChild(closeBtn);
    document.body.appendChild(zoomContainer);
    
    // Controle do arraste (pan)
    let isDragging = false;
    let startX, startY, translateX = 0, translateY = 0;
    
    // Abre o zoom
    img.addEventListener('click', function() {
      zoomContainer.style.display = 'flex';
      document.body.style.overflow = 'hidden';
      zoomedImg.style.transform = 'translate(0, 0)';
      translateX = 0;
      translateY = 0;
    });
    
    // Fecha o zoom
    const closeZoom = () => {
      zoomContainer.style.display = 'none';
      document.body.style.overflow = 'auto';
    };
    
    closeBtn.addEventListener('click', closeZoom);
    zoomContainer.addEventListener('click', closeZoom);
    
    // Previne que o clique na imagem feche o zoom
    zoomedImg.addEventListener('click', (e) => e.stopPropagation());
    
    // Controles de arraste com mouse
    zoomedImg.addEventListener('mousedown', (e) => {
      isDragging = true;
      startX = e.clientX - translateX;
      startY = e.clientY - translateY;
      zoomedImg.style.cursor = 'grabbing';
    });
    
    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;
      translateX = e.clientX - startX;
      translateY = e.clientY - startY;
      zoomedImg.style.transform = `translate(${translateX}px, ${translateY}px)`;
    });
    
    document.addEventListener('mouseup', () => {
      isDragging = false;
      zoomedImg.style.cursor = 'grab';
    });
    
    // Controles para touch (mobile)
    zoomedImg.addEventListener('touchstart', (e) => {
      isDragging = true;
      startX = e.touches[0].clientX - translateX;
      startY = e.touches[0].clientY - translateY;
    }, { passive: false });
    
    document.addEventListener('touchmove', (e) => {
      if (!isDragging) return;
      e.preventDefault();
      translateX = e.touches[0].clientX - startX;
      translateY = e.touches[0].clientY - startY;
      zoomedImg.style.transform = `translate(${translateX}px, ${translateY}px)`;
    }, { passive: false });
    
    document.addEventListener('touchend', () => {
      isDragging = false;
    });
  });
}

// =============================================
// 6. INICIALIZAÇÃO DA APLICAÇÃO
// =============================================

/**
 * 6.1 Inicializa todas as funcionalidades
 */
function init() {
  setupSmoothScroll();
  setupMenuToggle();
  setupScrollSpy();
  setupBackToTopButton();
  setupNavigationTooltips();
  setupImageZoom();
  
  // Mantém os menus pais abertos ao carregar
  document.addEventListener('DOMContentLoaded', () => {
    maintainOpenParents();
    
    // Ativa o item "Configuração" por padrão se estiver na seção
    if(window.location.hash === '#pocos-de-visita' || window.location.hash === '#rede-de-esgoto') {
      const configItem = document.querySelector('a[href="#configuracao-desenho"]');
      if(configItem) {
        const parentLi = configItem.closest('li');
        if(parentLi) {
          parentLi.classList.add('active');
          configItem.classList.add('active');
        }
      }
    }
  });
  
  // Mantém os menus pais abertos durante o scroll
  window.addEventListener('scroll', maintainOpenParents);
}

// Inicia a aplicação
init();