# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PlotAllocation
                                 A QGIS plugin
 Plot allocation for forest inventory
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-28
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Vinicius Richter and Renato Souza Santos
        email                : vinicius00rich@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""


import pathlib
import sys
import os
import subprocess
import platform


def install_dependencies():
    plugin_dir = os.path.dirname(os.path.realpath(__file__))
    operating_system = platform.system()
    try:
        import pip
    except ImportError:
        exec(open(str(pathlib.Path(plugin_dir, "scripts", "get_pip.py"))).read())
        import pip

        if operating_system == "Darwin":
            pip.main(["install", "--upgrade", "pip"])
        elif operating_system == "Linux":
            subprocess.check_call(
                [sys.executable, "-m", "pip", "install", "upgrade", "pip"]
            )
        elif operating_system == "Windows":
            subprocess.check_call(
                ["python3", "-m", "pip", "install", "--upgrade", "pip"]
            )
    sys.path.append(plugin_dir)

    with open(os.path.join(plugin_dir, "requirements.txt"), "r") as requirements:
        for dep in requirements.readlines():
            dep = dep.replace("\n", "")
            dep_noversion = dep.strip().split("==")[0]
            try:
                __import__(dep_noversion)
            except ImportError:
                print("{} not available, installing".format(dep))
                if operating_system == "Darwin":
                    pip.main(["install", dep])
                elif operating_system == "Linux":
                    subprocess.check_call([sys.executable, "-m", "pip", "install", dep])
                elif operating_system == "Windows":
                    subprocess.check_call(["python3", "-m", "pip", "install", dep])
try:
    install_dependencies()
except Exception:
    pass


debbug_mode = False
import os
import os.path
import shapely
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.utils import iface
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QDialogButtonBox
from qgis.core import *
from PyQt5.QtWidgets import QMessageBox, QProgressDialog
from .resources import *
from .plot_allocation_dialog import PlotAllocationDialog
from sklearn.cluster import KMeans

from shapely.ops import nearest_points

import geopandas as gpd
from shapely.geometry import shape, mapping, GeometryCollection, Point, Polygon
from shapely.ops import unary_union
from shapely import wkt
import math
import numpy as np
import traceback
import pandas as pd
from shapely.affinity import rotate
from shapely.geometry import MultiPolygon

class PlotAllocation:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PlotAllocation_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FPT Plot Allocation')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.select_button_connected = False
        self.select_save_button_connected = False
        self.dialog_open = False

    def get_reduced_polygons_to_min_border_distance(self, layer):
        """
        Essa função cria uma versão dos polígonos que seja menor que `min_border_distance` do que os polígonos reais,
        considerando a distância de borda. Se o polígono reduzido ficar inválido ou muito pequeno, ele será excluído.
        """
        reduced_polygons = []

        for feature in layer.getFeatures():
            geom = feature.geometry()
            shapely_geom = wkt.loads(geom.asWkt())

            reduced_poly = shapely_geom.buffer(-self.get_distances()[0])
            if reduced_poly.is_valid and not reduced_poly.is_empty and reduced_poly.area > 0:
                reduced_polygons.append(reduced_poly)

        # Convertendo a lista de polígonos reduzidos para GeoDataFrame
        reduced_polygons_gdf = gpd.GeoDataFrame(geometry=reduced_polygons, crs=layer.crs().toWkt())

        # Criando uma nova camada no QGIS para exibir os polígonos reduzidos
        reduced_layer = QgsVectorLayer("Polygon?crs={}".format(layer.crs().authid()),
                                       "Reduced Polygons", "memory")
        provider = reduced_layer.dataProvider()

        for poly in reduced_polygons_gdf.geometry:
            feature = QgsFeature()
            feature.setGeometry(QgsGeometry.fromWkt(poly.wkt))
            provider.addFeature(feature)

        reduced_layer.updateExtents()
        # QgsProject.instance().addMapLayer(reduced_layer) adiciona o layer no qgis

        return reduced_layer

    def get_selected_epsg(self):
        """Obter o EPSG selecionado pelo usuário."""
        selected_epsg_text = self.dlg.epsg_selector.currentText()
        epsg_code = selected_epsg_text.split('-')[-1].strip()  # Extrair apenas o código EPSG
        return epsg_code

    ##### FUNÇÕES CRIADAS #####
    def load_vectors(self):
        """Preenche a box com o boundary, excluindo layers de pontos"""
        self.dlg.shp.clear()
        layers_list = [layer for layer in QgsProject.instance().mapLayers().values()]
        vectors_layers_list = []
        for layer in layers_list:
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                vectors_layers_list.append(layer.name())
        self.dlg.shp.addItems(vectors_layers_list)

        # Preencher a QComboBox epsg_selector com os EPSGs e descrições das zonas
        self.update_epsg_selector()

    def update_epsg_selector(self):
        """Atualiza a QComboBox epsg_selector com base na seleção dos botões de rádio"""
        self.dlg.epsg_selector.clear()
        if self.dlg.radio_button_south.isChecked():
            epsg_codes = [
                "Zone 1S - EPSG:32701", "Zone 2S - EPSG:32702", "Zone 3S - EPSG:32703", "Zone 4S - EPSG:32704",
                "Zone 5S - EPSG:32705", "Zone 6S - EPSG:32706", "Zone 7S - EPSG:32707", "Zone 8S - EPSG:32708",
                "Zone 9S - EPSG:32709", "Zone 10S - EPSG:32710", "Zone 11S - EPSG:32711", "Zone 12S - EPSG:32712",
                "Zone 13S - EPSG:32713", "Zone 14S - EPSG:32714", "Zone 15S - EPSG:32715", "Zone 16S - EPSG:32716",
                "Zone 17S - EPSG:32717", "Zone 18S - EPSG:32718", "Zone 19S - EPSG:32719", "Zone 20S - EPSG:32720",
                "Zone 21S - EPSG:32721", "Zone 22S - EPSG:32722", "Zone 23S - EPSG:32723", "Zone 24S - EPSG:32724",
                "Zone 25S - EPSG:32725", "Zone 26S - EPSG:32726", "Zone 27S - EPSG:32727", "Zone 28S - EPSG:32728",
                "Zone 29S - EPSG:32729", "Zone 30S - EPSG:32730", "Zone 31S - EPSG:32731", "Zone 32S - EPSG:32732",
                "Zone 33S - EPSG:32733", "Zone 34S - EPSG:32734", "Zone 35S - EPSG:32735", "Zone 36S - EPSG:32736",
                "Zone 37S - EPSG:32737", "Zone 38S - EPSG:32738", "Zone 39S - EPSG:32739", "Zone 40S - EPSG:32740",
                "Zone 41S - EPSG:32741", "Zone 42S - EPSG:32742", "Zone 43S - EPSG:32743", "Zone 44S - EPSG:32744",
                "Zone 45S - EPSG:32745", "Zone 46S - EPSG:32746", "Zone 47S - EPSG:32747", "Zone 48S - EPSG:32748",
                "Zone 49S - EPSG:32749", "Zone 50S - EPSG:32750", "Zone 51S - EPSG:32751", "Zone 52S - EPSG:32752",
                "Zone 53S - EPSG:32753", "Zone 54S - EPSG:32754", "Zone 55S - EPSG:32755", "Zone 56S - EPSG:32756",
                "Zone 57S - EPSG:32757", "Zone 58S - EPSG:32758", "Zone 59S - EPSG:32759", "Zone 60S - EPSG:32760"
            ]
        else:
            epsg_codes = [
                "Zone 1N - EPSG:32601", "Zone 2N - EPSG:32602", "Zone 3N - EPSG:32603", "Zone 4N - EPSG:32604",
                "Zone 5N - EPSG:32605", "Zone 6N - EPSG:32606", "Zone 7N - EPSG:32607", "Zone 8N - EPSG:32608",
                "Zone 9N - EPSG:32609", "Zone 10N - EPSG:32610", "Zone 11N - EPSG:32611", "Zone 12N - EPSG:32612",
                "Zone 13N - EPSG:32613", "Zone 14N - EPSG:32614", "Zone 15N - EPSG:32615", "Zone 16N - EPSG:32616",
                "Zone 17N - EPSG:32617", "Zone 18N - EPSG:32618", "Zone 19N - EPSG:32619", "Zone 20N - EPSG:32620",
                "Zone 21N - EPSG:32621", "Zone 22N - EPSG:32622", "Zone 23N - EPSG:32623", "Zone 24N - EPSG:32624",
                "Zone 25N - EPSG:32625", "Zone 26N - EPSG:32626", "Zone 27N - EPSG:32627", "Zone 28N - EPSG:32628",
                "Zone 29N - EPSG:32629", "Zone 30N - EPSG:32630", "Zone 31N - EPSG:32631", "Zone 32N - EPSG:32632",
                "Zone 33N - EPSG:32633", "Zone 34N - EPSG:32634", "Zone 35N - EPSG:32635", "Zone 36N - EPSG:32636",
                "Zone 37N - EPSG:32637", "Zone 38N - EPSG:32638", "Zone 39N - EPSG:32639", "Zone 40N - EPSG:32640",
                "Zone 41N - EPSG:32641", "Zone 42N - EPSG:32642", "Zone 43N - EPSG:32643", "Zone 44N - EPSG:32644",
                "Zone 45N - EPSG:32645", "Zone 46N - EPSG:32646", "Zone 47N - EPSG:32647", "Zone 48N - EPSG:32648",
                "Zone 49N - EPSG:32649", "Zone 50N - EPSG:32650", "Zone 51N - EPSG:32651", "Zone 52N - EPSG:32652",
                "Zone 53N - EPSG:32653", "Zone 54N - EPSG:32654", "Zone 55N - EPSG:32655", "Zone 56N - EPSG:32656",
                "Zone 57N - EPSG:32657", "Zone 58N - EPSG:32658", "Zone 59N - EPSG:32659", "Zone 60N - EPSG:32660"
            ]

        self.dlg.epsg_selector.addItems(epsg_codes)
        # Seleciona por padrão o EPSG 32722 (Zone 22S)
        index = self.dlg.epsg_selector.findText("Zone 22S - EPSG:32722")
        if index != -1:
            self.dlg.epsg_selector.setCurrentIndex(index)

    def open_vector(self):

        """Caixa de diálogo para carregar vetores locais"""
        selected_layer, _ = QFileDialog.getOpenFileName(caption="Select layer",
                                                        filter="Shapefiles (*.shp)")
        # Se a selected_layer não for vazia
        if selected_layer:
            self.iface.addVectorLayer(selected_layer,
                                      os.path.splitext(os.path.basename(selected_layer))[0], "ogr")
            self.load_vectors()

    def transform_to_layer_crs(self, geometry, source_crs, target_crs):
        """Transformar a geometria para o CRS do layer alvo"""
        try:
            transform_context = QgsProject.instance().transformContext()
            coord_transform = QgsCoordinateTransform(source_crs, target_crs, transform_context)
            transformed_geometry = QgsGeometry(geometry)
            if transformed_geometry.transform(coord_transform) == 0:
                return transformed_geometry
            else:
                QgsMessageLog.logMessage("Falha na transformação de geometria.", 'Your Plugin Name', Qgis.Critical)
                return None
        except Exception as e:
            QgsMessageLog.logMessage(f"Erro na transformação de CRS: {e}", 'Your Plugin Name', Qgis.Critical)
            return None

    def set_input_layer(self):
        """Obter a layer definida no combobox shp e verificar se o CRS é conhecido."""
        layer = None
        layer_name = self.dlg.shp.currentText()
        for layer_ in QgsProject.instance().mapLayers().values():
            if layer_.name() == layer_name:
                layer = layer_
                break

        if not layer:
            if debbug_mode:
                QgsMessageLog.logMessage("Layer not found", 'Your Plugin Name', Qgis.Critical)
            return None, None

        # Verificar se o CRS do layer é desconhecido
        if not layer.crs().isValid() or layer.crs().authid() == '':
            QMessageBox.warning(self.iface.mainWindow(), "Unknown CRS",
                                "The selected layer's CRS is unknown. Please set a valid CRS before proceeding.")
            return None, None

        if debbug_mode:
            QgsMessageLog.logMessage(f"Selected Layer CRS: {layer.crs().authid()}", 'Your Plugin Name', Qgis.Info)

        # Obter o EPSG selecionado pelo usuário
        selected_epsg = self.get_selected_epsg()
        if debbug_mode:
            QgsMessageLog.logMessage(f"selected_epsg: {selected_epsg}", 'Your Plugin Name', Qgis.Info)
        dest_crs = QgsCoordinateReferenceSystem(selected_epsg)
        transform_context = QgsProject.instance().transformContext()
        coord_transform = QgsCoordinateTransform(layer.crs(), dest_crs, transform_context)

        # Criar uma nova camada para as geometrias transformadas
        transformed_layer = QgsVectorLayer("Polygon?crs={}".format(dest_crs.authid()), "Transformed Layer", "memory")
        provider = transformed_layer.dataProvider()

        # Adicionar campos da camada original à nova camada
        provider.addAttributes(layer.fields())
        transformed_layer.updateFields()
        if debbug_mode:
            QgsMessageLog.logMessage(f"Transformed Layer CRS: {transformed_layer.crs().authid()}", 'Your Plugin Name',
                                 Qgis.Info)

        # Transformar as geometrias do layer original e adicionar à nova camada
        transformed_features = []
        attributes = []  # Lista para armazenar atributos das feições
        field_names = [field.name() for field in layer.fields()]

        for feature in layer.getFeatures():
            transformed_geometry = QgsGeometry(feature.geometry())
            if transformed_geometry.transform(coord_transform) == 0:
                transformed_feature = QgsFeature()
                transformed_feature.setGeometry(transformed_geometry)
                transformed_feature.setAttributes(
                    feature.attributes())  # Garantir que os atributos sejam copiados corretamente
                transformed_features.append(transformed_feature)
            else:
                QgsMessageLog.logMessage(f"Failed to transform feature ID {feature.id()}", 'Your Plugin Name',
                                         Qgis.Warning)

        provider.addFeatures(transformed_features)
        transformed_layer.updateExtents()
        if debbug_mode:
            QgsMessageLog.logMessage(f"Layer CRS after transformation: {transformed_layer.crs().authid()}",
                                     'Your Plugin Name', Qgis.Info)

        # Calcular a área total dos polígonos (em metros)
        self.total_area = self.calculate_total_area(transformed_layer)
        if debbug_mode:
            QgsMessageLog.logMessage(f"Total Area: {self.total_area}", 'Your Plugin Name', Qgis.Info)

        return transformed_layer

    def calculate_total_area(self, layer):
        """Calcular a área total dos polígonos no layer fornecido"""
        total_area = 0
        for feature in layer.getFeatures():
            area = feature.geometry().area()
            if debbug_mode:
                QgsMessageLog.logMessage(f"Feature ID {feature.id()} Area: {area}", 'Your Plugin Name', Qgis.Info)
            total_area += area
        return total_area

    def _check_rectangle(self, point, existing_points, width, height):
        """
        Verifica se um retângulo centrado em 'point' é válido.

        :param point: QgsGeometry centrada no ponto do retângulo.
        :param existing_points: Lista de QgsGeometry dos centros dos retângulos existentes.
        :param width: Largura do retângulo.
        :param height: Altura do retângulo.
        :return: True se o retângulo estiver completamente contido em uma feature, não intersectar nenhum retângulo existente, e estiver a pelo menos self.min_border_distance dos limites dos polígonos. False caso contrário.
        """
        point_ = point.asPoint()

        half_width = width / 2
        half_height = height / 2

        # Define os cantos do retângulo ao redor do ponto central
        rectangle_points = [
            QgsPointXY(point_.x() - half_width, point_.y() - half_height),
            QgsPointXY(point_.x() + half_width, point_.y() - half_height),
            QgsPointXY(point_.x() + half_width, point_.y() + half_height),
            QgsPointXY(point_.x() - half_width, point_.y() + half_height),
            QgsPointXY(point_.x() - half_width, point_.y() - half_height)  # Fecha o polígono
        ]

        # Cria a geometria do retângulo como um polígono
        rectangle_geom = QgsGeometry.fromPolygonXY([rectangle_points])

        # Verifica se o retângulo está completamente dentro de pelo menos uma feature na camada
        for feature in self.shp_layer.getFeatures():
            feature_geom = feature.geometry()

            # Buffer negativo da feature para garantir distância mínima dos limites
            buffered_feature_geom = feature_geom.buffer(-self.min_border_distance, 1)

            contains_all_points = all(
                buffered_feature_geom.contains(QgsGeometry.fromPointXY(pt)) for pt in rectangle_points)

            if contains_all_points:
                # Verifica se o retângulo não intersecta nenhum retângulo dos existing_points
                for existing_point in existing_points:
                    existing_point_ = existing_point.asPoint()
                    existing_rectangle_points = [
                        QgsPointXY(existing_point_.x() - half_width, existing_point_.y() - half_height),
                        QgsPointXY(existing_point_.x() + half_width, existing_point_.y() - half_height),
                        QgsPointXY(existing_point_.x() + half_width, existing_point_.y() + half_height),
                        QgsPointXY(existing_point_.x() - half_width, existing_point_.y() + half_height),
                        QgsPointXY(existing_point_.x() - half_width, existing_point_.y() - half_height)
                        # Fecha o polígono
                    ]
                    existing_rectangle_geom = QgsGeometry.fromPolygonXY([existing_rectangle_points])
                    if rectangle_geom.intersects(existing_rectangle_geom):
                        return False  # Se intersecta com um retângulo existente, é inválido

                return True  # Se está contido em uma feature e não intersecta retângulos existentes, é válido

        return False
    def _check_points_distance(self, point, plot_area, existing_points, buffer_type):
        """Check if the point is at a valid distance from existing points and polygon boundaries."""
        if buffer_type == "round":
            min_distance = self.get_distances()[1]  # Obtém a distância mínima permitida
            for existing_point in existing_points:
                if point.distance(existing_point) < min_distance:
                    return False

            return True


        elif buffer_type == "squared":
            min_distance = self.get_distances()[1]  # Obtém a distância mínima permitida
            for existing_point in existing_points:
                if point.distance(existing_point) < min_distance:
                    return False

            return True
        elif buffer_type == "rectangle":
            min_distance = self.get_distances()[1]  # Obtém a distância mínima permitida
            for existing_point in existing_points:
                if point.distance(existing_point) < min_distance:
                    return False
        return True

    def get_distances(self):
        """obtem a distancia que os pontos devem manter da borda e também a distancia que um ponto deve ter de outro"""
        if self.plot_format == 'round':
            add_value = math.sqrt(self.plot_area/math.pi) * 2
        if self.plot_format == 'squared':
            add_value = (math.sqrt(self.plot_area)) * math.sqrt(2)
        if self.plot_format == 'rectangle':
            distance_from_each_other = math.sqrt((self.custom_plot_format_x_value) ** 2 + (self.custom_plot_format_y_value ** 2))
            distance_from_border = max(self.custom_plot_format_x_value, self.custom_plot_format_y_value)
            return distance_from_border, distance_from_each_other
        distance_from_each_other = add_value
        distance_from_border = self.min_border_distance + add_value
        return distance_from_border, distance_from_each_other


    def create_all_possible_points(self, with_index_flag=False):
        """Essa função cria todos os pontos possíveis, de acordo com grid_spacing.
        Se with_index_flag==True, retorna uma lista de tuplas contendo o número do polígono e as coordenadas do ponto.
        Se with_index_flag==False, retorna apenas a lista de geometrias de pontos."""

        # Obtém a extensão (bounding box) da camada
        layer_extent = self.reduced_shp.extent()

        x_min = layer_extent.xMinimum()
        y_min = layer_extent.yMinimum()
        x_max = layer_extent.xMaximum()
        y_max = layer_extent.yMaximum()

        if not (np.isfinite(x_min) and np.isfinite(y_min) and np.isfinite(y_max) and np.isfinite(y_max)):
            print("Extensão do layer contém valores infinitos ou NaN.")
            return None

        if x_min >= x_max or y_min >= y_max:
            print("Intervalo de valores de x ou y inválido.")
            return None

        # Obtém a distância que uma parcela deve ter da outra
        grid_spacing = self.get_distances()[1]
        if debbug_mode:
            QgsMessageLog.logMessage(f"GRID SPACING: {grid_spacing}", 'Your Plugin Name', Qgis.Info)
        valid_points = []

        x_coords = np.arange(x_min + grid_spacing / 2, x_max, grid_spacing)
        y_coords = np.arange(y_min + grid_spacing / 2, y_max, grid_spacing)

        if with_index_flag:
            for x in x_coords:
                for y in y_coords:
                    point = QgsPointXY(x, y)
                    point_geom = QgsGeometry.fromPointXY(point)
                    for index, poly in enumerate(self.reduced_shp.getFeatures()):
                        geom = poly.geometry()
                        if geom.contains(point_geom):
                            valid_points.append(
                                (index, point_geom))  # Salva o índice do polígono junto com a geometria do ponto
                            break
        else:
            for x in x_coords:
                for y in y_coords:
                    point = QgsPointXY(x, y)
                    point_geom = QgsGeometry.fromPointXY(point)
                    for poly in self.reduced_shp.getFeatures():
                        geom = poly.geometry()
                        if geom.contains(point_geom):
                            valid_points.append(point_geom)
            return valid_points

        return valid_points

    # =================================================
    """Best allocation"""

    def update_plots_column(self):
        """Atualiza o QComboBox plots_column com as colunas da camada selecionada"""
        # Limpar os itens anteriores do plots_column
        self.dlg.plots_column.clear()

        # Obter o nome da camada selecionada no QComboBox shp
        layer_name = self.dlg.shp.currentText()

        # Buscar a camada correspondente no projeto
        layer = None
        for lyr in QgsProject.instance().mapLayers().values():
            if lyr.name() == layer_name:
                layer = lyr
                break

        # Se não encontrar a camada, sair da função
        if not layer:
            return

        # Obter os nomes das colunas da tabela de atributos
        field_names = [field.name() for field in layer.fields()]

        # Adicionar os nomes das colunas ao QComboBox plots_column
        self.dlg.plots_column.addItems(field_names)


    def _generate_systematic_best_sampling_sample_points(self):
        valid_points = self.create_all_possible_points()

        if len(valid_points) < self.max_number_of_points:
            QMessageBox.warning(self.dlg, "Warning!",
                                "Unable to generate plots with the established criteria. Only the possible plots were generated.")
            return self.create_point_layer(valid_points, self.crs)

        points_gdf = gpd.GeoDataFrame(geometry=valid_points, crs=self.crs)

        final_points = []

        for feature in self.reduced_shp.getFeatures():
            num_parcelas = feature['plots']
            if num_parcelas <= 0:
                continue

            geometry = feature.geometry()

            if geometry.isMultipart():
                polygons = geometry.asMultiPolygon()
            else:
                polygons = [geometry.asPolygon()]

            for polygon in polygons:
                if not polygon:
                    continue

                shapely_polygon = shapely.geometry.Polygon(polygon[0])  # Apenas o anel externo
                shapely_full_polygon = shapely.geometry.Polygon(polygon[0],
                                                                holes=polygon[1:])  # Polígono completo (com buracos)

                points_within_polygon = points_gdf[points_gdf.within(shapely_polygon)]

                if len(points_within_polygon) <= num_parcelas:
                    final_points.append(points_within_polygon)
                    continue

                kmeans = KMeans(n_clusters=num_parcelas, random_state=0).fit(
                    np.array(list(points_within_polygon.geometry.apply(lambda point: (point.x, point.y))))
                )

                selected_points = gpd.GeoDataFrame(
                    geometry=[shapely.geometry.Point(xy) for xy in kmeans.cluster_centers_],
                    crs=points_gdf.crs)

                # Corrigir pontos que caíram fora do polígono válido
                corrected_points = []
                for point in selected_points.geometry:
                    if not shapely_full_polygon.contains(point):  # Se o ponto está fora (inclusive em buracos)
                        if debbug_mode:
                            QgsMessageLog.logMessage(f"❗ Ponto fora do polígono detectado: {point}", 'Your Plugin Name',
                                                     Qgis.Info)

                        # Obter o ponto mais próximo na borda externa do polígono
                        nearest_point = nearest_points(point, shapely_polygon.exterior)[1]
                        if debbug_mode:
                            QgsMessageLog.logMessage(f"✅ Ponto movido para: {nearest_point}", 'Your Plugin Name', Qgis.Info)

                        corrected_points.append(nearest_point)
                    else:
                        corrected_points.append(point)

                selected_points.geometry = corrected_points
                final_points.append(selected_points)

        total_parcelas = sum(len(points) for points in final_points)

        if total_parcelas > self.max_number_of_points:
            all_points_gdf = gpd.GeoDataFrame(pd.concat(final_points, ignore_index=True), crs=self.crs)
            sampled_points_gdf = all_points_gdf.sample(n=int(self.max_number_of_points), random_state=0).reset_index(
                drop=True)
            final_points = [sampled_points_gdf]

        final_points_gdf = self.create_point_layer(final_points, self.crs)

        return final_points_gdf

    """FIM best allocation"""
    #=================================================
    """Distribuição randomica"""

    def _generate_random_sample_points(self, shp, plot_area, max_attempts=3000):
        extent = shp.extent()
        x_min, y_min, x_max, y_max = extent.xMinimum(), extent.yMinimum(), extent.xMaximum(), extent.yMaximum()

        if not (np.isfinite(x_min) and np.isfinite(y_min) and np.isfinite(x_max) and np.isfinite(y_max)):
            if debbug_mode:
                QgsMessageLog.logMessage("Extensão do layer contém valores infinitos ou NaN.", 'Your Plugin Name',
                                         Qgis.Critical)
            return None

        if x_min >= x_max or y_min >= y_max:
            if debbug_mode:
                QgsMessageLog.logMessage("Intervalo de valores de x ou y inválido.", 'Your Plugin Name', Qgis.Critical)
            return None

        valid_points = []
        source_crs = shp.crs()
        target_crs = shp.crs()

        # ✅ Itera sobre cada feição (talhão)
        for feature in shp.getFeatures():
            feature_id = feature.id()
            num_points = self.plots_list.get(feature_id, 0)
            if num_points <= 0:
                continue  # Pula talhões sem parcelas atribuídas

            geometry = feature.geometry()

            if geometry.isMultipart():
                polygons = geometry.asMultiPolygon()
            else:
                polygons = [geometry.asPolygon()]

            for polygon in polygons:
                if not polygon:
                    continue

                shapely_polygon = shapely.geometry.Polygon(polygon[0])
                attempts = 0
                talhao_points = []

                while len(talhao_points) < num_points and attempts < max_attempts:
                    try:
                        # Gera um ponto aleatório dentro dos limites do polígono
                        min_x, min_y, max_x, max_y = shapely_polygon.bounds
                        x = np.random.uniform(min_x, max_x)
                        y = np.random.uniform(min_y, max_y)
                    except OverflowError as e:
                        if debbug_mode:
                            QgsMessageLog.logMessage(f"Erro ao gerar valores x ou y: {e}", 'Your Plugin Name',
                                                 Qgis.Critical)
                        break

                    point = QgsGeometry.fromPointXY(QgsPointXY(x, y))
                    transformed_point = self.transform_to_layer_crs(point, source_crs, target_crs)

                    if geometry.contains(transformed_point):
                        if attempts==0:
                            talhao_points.append(transformed_point)
                            attempts += 1
                        else:
                            attempts += 1
                            if (self._check_points_distance(transformed_point, plot_area, talhao_points, self.plot_format)):
                                talhao_points.append(transformed_point)
                            if debbug_mode:
                                if attempts == max_attempts:
                                    QgsMessageLog.logMessage("CHEGOU NO MÁXIMO DE TENTATIVAS.",
                                                             'Your Plugin Name', Qgis.Critical)

                valid_points.extend(talhao_points)

        # ✅ Criar a camada de pontos gerados
        if valid_points:
            point_layer = self.create_point_layer(valid_points, shp.crs())
            return point_layer
        else:
            if debbug_mode:
                QgsMessageLog.logMessage("Não foi possível gerar parcelas com os critérios estabelecidos.",
                                     'Your Plugin Name', Qgis.Critical)
            return None

    """FIM distribuição randomica"""
    #=====================================================================================
    """Distribuição sistematica"""
    def _generate_systematic_sample_points(self, shp, plot_area):
        valid_points = self.create_all_possible_points()

        valid_points = self.create_point_layer(valid_points, self.shp_layer.crs())
        if len(valid_points) < self.max_number_of_points:
            QMessageBox.warning(self.dlg, "Warning!",
                                "Unable to generate plots with the established criteria. Only the possible plots were generated.")
            return valid_points
        return valid_points
    """FIM Distribuição sistematica"""
    # =====================================================================================
    """Distribuição systematic custom"""

    def _generate_systematic_custom_sample_points(self, shp, plot_area):
        x_spacing, y_spacing, rotation_angle = self.systematic_custom_x_d, self.systematic_custom_y_d, self.systematic_custom_degree

        # Extensão da camada
        layer_extent = self.reduced_shp.extent()
        x_min, x_max = layer_extent.xMinimum(), layer_extent.xMaximum()
        y_min, y_max = layer_extent.yMinimum(), layer_extent.yMaximum()

        if not (np.isfinite(x_min) and np.isfinite(y_min) and np.isfinite(x_max) and np.isfinite(y_max)):
            print("Layer extent contains infinite or NaN values.")
            return None

        if x_min >= x_max or y_min >= y_max:
            print("Invalid range of x or y values.")
            return None

        # Calcular a expansão da extensão para cobrir rotação
        center_x = (x_min + x_max) / 2
        center_y = (y_min + y_max) / 2
        diagonal = np.sqrt((x_max - x_min) ** 2 + (y_max - y_min) ** 2)
        half_diagonal = diagonal / 2

        x_min_expanded = center_x - half_diagonal
        x_max_expanded = center_x + half_diagonal
        y_min_expanded = center_y - half_diagonal
        y_max_expanded = center_y + half_diagonal

        # Geração dos pontos do grid com a extensão expandida
        x_coords = np.arange(x_min_expanded + x_spacing / 2, x_max_expanded, x_spacing)
        y_coords = np.arange(y_min_expanded + y_spacing / 2, y_max_expanded, y_spacing)

        points = []
        for x in x_coords:
            for y in y_coords:
                points.append(Point(x, y))

        # Criando uma GeoDataFrame a partir dos pontos gerados
        grid_gdf = gpd.GeoDataFrame(geometry=points, crs=self.crs)

        # Rotacionando a grade
        rotated_grid = grid_gdf.copy()
        rotated_grid['geometry'] = [rotate(point, rotation_angle, origin=(center_x, center_y)) for point in
                                    grid_gdf['geometry']]

        # Converter a camada reduzida (reduced_shp) para GeoDataFrame
        features = [feat for feat in self.reduced_shp.getFeatures()]
        geometries = [feat.geometry().asWkt() for feat in features]
        geom_gdf = gpd.GeoDataFrame(geometry=gpd.GeoSeries.from_wkt(geometries), crs=self.reduced_shp.crs().authid())

        # Unificar todas as geometrias (unary_union)
        reduced_shp_union = geom_gdf.geometry.union_all()

        # Filtrando os pontos que estão dentro da forma reduzida (reduced_shp)
        valid_points = rotated_grid[rotated_grid.intersects(reduced_shp_union)]

        if not valid_points.empty:
            # Convertendo para uma lista de QgsGeometry
            valid_points_list = [QgsGeometry.fromWkt(geom.wkt) for geom in valid_points['geometry']]
            valid_points_layer = self.create_point_layer(valid_points_list, self.crs)
            return valid_points_layer
        else:
            print("Unable to generate plots with the established criteria.")
            return None
    """FIM Distribuição systematic custom"""

    # =====================================================================================
    def create_point_layer(self, points, crs):
        # Verifique se crs é uma string ou um objeto QgsCoordinateReferenceSystem
        if isinstance(crs, str):
            crs_id = crs  # Se for uma string, use diretamente
        else:
            crs_id = crs.authid()  # Se for um objeto QgsCoordinateReferenceSystem, use authid()

        # Cria uma camada de pontos em memória
        point_layer = QgsVectorLayer(f"Point?crs={crs_id}", "sample_points", "memory")
        pr = point_layer.dataProvider()

        # Adiciona o campo "plot_n" que irá armazenar a numeração
        pr.addAttributes([QgsField("plot_n", QVariant.Int)])
        point_layer.updateFields()

        point_layer.startEditing()
        try:
            plot_number = 1  # Inicializa a numeração a partir de 1

            for point in points:
                for geom in point.geometry:
                    feature = QgsFeature()
                    feature.setGeometry(QgsGeometry.fromWkt(geom.wkt))

                    # Define o valor da numeração para o campo "plot_n"
                    feature.setAttributes([plot_number])
                    pr.addFeature(feature)

                    plot_number += 1  # Incrementa o número para o próximo ponto

            point_layer.commitChanges()
            point_layer.updateExtents()
        except:
            plot_number = 1  # Reinicializa a numeração em caso de erro

            for point in points:
                feature = QgsFeature()
                feature.setGeometry(point)

                # Define o valor da numeração para o campo "plot_n"
                feature.setAttributes([plot_number])
                pr.addFeature(feature)

                plot_number += 1  # Incrementa o número para o próximo ponto

            point_layer.commitChanges()
            point_layer.updateExtents()

        return point_layer

    ##### FIM FUNÇÕES CRIADAS #####
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('PlotAllocation', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def select_output_file(self):
        """Abre um diálogo para selecionar o caminho do arquivo de saída"""
        file_dialog = QFileDialog()
        output_path, _ = file_dialog.getSaveFileName(
            self.dlg,
            "Select Output File",
            "",
            "Shapefiles (*.shp);;All Files (*)"
        )
        if output_path:
            self.dlg.output_name.setText(output_path)

    def create_buffer_layer(self, point_layer, buffer_distance, buffer_type):
        # Cria uma camada de polígonos em memória para armazenar os buffers
        buffer_layer_name = f"{point_layer.name()}_buffer"  # Nome da camada de buffer
        buffer_layer = QgsVectorLayer("Polygon?crs={}".format(point_layer.crs().authid()), buffer_layer_name, "memory")
        provider = buffer_layer.dataProvider()

        # Adiciona a coluna "buffer_n" para numerar os buffers
        provider.addAttributes([QgsField("buffer_n", QVariant.Int)])
        buffer_layer.updateFields()

        # Itera sobre as feições da camada de pontos
        for feature in point_layer.getFeatures():
            geom = feature.geometry()
            plot_number = feature["plot_n"]  # Captura a numeração do ponto

            # Cria o buffer com base no tipo especificado
            if buffer_type == "round":
                buffer_geom = geom.buffer(buffer_distance, 30)  # 30 segmentos para aproximar um círculo
            elif buffer_type == "squared":
                # Cria um buffer quadrado
                center = geom.asPoint()
                x, y = center.x(), center.y()
                half_side = buffer_distance / 2
                coords = [
                    QgsPointXY(x - half_side, y - half_side),
                    QgsPointXY(x + half_side, y - half_side),
                    QgsPointXY(x + half_side, y + half_side),
                    QgsPointXY(x - half_side, y + half_side),
                    QgsPointXY(x - half_side, y - half_side)
                ]
                buffer_geom = QgsGeometry.fromPolygonXY([coords])
            elif buffer_type == "rectangle":
                point_ = geom.asPoint()

                half_width = self.custom_plot_format_x_value / 2
                half_height = self.custom_plot_format_y_value / 2

                # Define os cantos do retângulo ao redor do ponto central
                rectangle_points = [
                    QgsPointXY(point_.x() - half_width, point_.y() - half_height),
                    QgsPointXY(point_.x() + half_width, point_.y() - half_height),
                    QgsPointXY(point_.x() + half_width, point_.y() + half_height),
                    QgsPointXY(point_.x() - half_width, point_.y() + half_height)
                ]

                buffer_geom = QgsGeometry.fromPolygonXY([rectangle_points])

            # Cria a feição do buffer e define sua geometria e numeração
            buffer_feature = QgsFeature()
            buffer_feature.setGeometry(buffer_geom)

            # Define o valor da numeração para o campo "buffer_n"
            buffer_feature.setAttributes([plot_number])

            provider.addFeatures([buffer_feature])

        # Atualiza a camada e adiciona a opacidade
        buffer_layer.updateExtents()
        self.set_layer_opacity(buffer_layer, 0.5)
        QgsProject.instance().addMapLayer(buffer_layer)
        return buffer_layer

    def set_layer_opacity(self, layer, opacity):
        """Set the layer opacity."""
        renderer = layer.renderer()
        if renderer is not None:
            symbol = renderer.symbol()
            symbol.setOpacity(opacity)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""


        icon_path = ':/plugins/plot_allocation/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Plot Alocation FPTools'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&FPT Plot Allocation'),
                action)
            self.iface.removeToolBarIcon(action)

    def update_ok_button_state(self):
        """Activate the OK button if all required fields are filled."""
        is_shp_selected = bool(self.dlg.shp.currentText())
        plot_format = self.dlg.plot_format_selector.currentText()
        distribution = self.dlg.distribution.currentText()

        if plot_format == "rectangle":
            # Para o formato 'rectangle', é necessário que os valores X e Y sejam preenchidos e diferentes de zero
            is_x_filled = self.dlg.custom_plot_format_x_value.value() > 0
            is_y_filled = self.dlg.custom_plot_format_y_value.value() > 0
            is_sample_number_filled = self.dlg.sample_number.value() > 0
            if self.dlg.use_column_n_plots_checkbox:
                is_valid = is_shp_selected and is_x_filled and is_y_filled
            else:
                is_valid = is_shp_selected and is_x_filled and is_y_filled and is_sample_number_filled
            if distribution == "systematic custom":
                # Para a distribuição 'systematic custom', os valores X e Y devem ser preenchidos e diferentes de zero
                is_x_filled = self.dlg.systematic_custom_x_d.value() > 0
                is_y_filled = self.dlg.systematic_custom_y_d.value() > 0
                # Exigir 'plot_area' se o formato não for 'rectangle'
                is_plot_area_filled = plot_format == "rectangle" or self.dlg.plot_area.value() > 0
                is_valid = is_shp_selected and is_x_filled and is_y_filled and is_plot_area_filled and is_x_filled

        elif distribution == "systematic custom":
            # Para a distribuição 'systematic custom', os valores X e Y devem ser preenchidos e diferentes de zero
            is_x_filled = self.dlg.systematic_custom_x_d.value() > 0
            is_y_filled = self.dlg.systematic_custom_y_d.value() > 0
            # Exigir 'plot_area' se o formato não for 'rectangle'
            is_plot_area_filled = plot_format == "rectangle" or self.dlg.plot_area.value() > 0
            is_valid = is_shp_selected and is_x_filled and is_y_filled and is_plot_area_filled
        else:
            # Para outros formatos, o campo 'plot_area' é necessário
            is_plot_area_filled = self.dlg.plot_area.value() > 0
            is_sample_number_filled = self.dlg.sample_number.value() > 0 if self.dlg.sample_number.isEnabled() else True
            is_valid = is_shp_selected and is_plot_area_filled and is_sample_number_filled

        self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(is_valid)

    def update_sample_number_state(self):
        """Ativa ou desativa o campo sample_number dependendo da distribuição selecionada."""
        distribution = self.dlg.distribution.currentText()
        use_n_plots_col = self.dlg.use_column_n_plots_checkbox.isChecked()

        if distribution == "systematic" or distribution == "systematic custom":
            self.dlg.sample_number.setEnabled(False)
            self.dlg.plots_column.setEnabled(False)
            self.dlg.label_9.setEnabled(False)
            self.dlg.label_10.setEnabled(False)
            self.dlg.use_column_n_plots_checkbox.setEnabled(False)
            self.dlg.by_hectare_checkbox.setEnabled(False)
            self.dlg.label_11.setEnabled(False)
        else:
            self.dlg.label_10.setEnabled(True)
            self.dlg.use_column_n_plots_checkbox.setEnabled(True)
            self.dlg.by_hectare_checkbox.setEnabled(True)
            self.dlg.label_11.setEnabled(True)

            if use_n_plots_col:
                self.dlg.sample_number.setEnabled(False)
                self.dlg.label_3.setEnabled(False)
                self.dlg.plots_column.setEnabled(True)
                self.dlg.label_9.setEnabled(True)
                self.dlg.by_hectare_checkbox.setEnabled(False)
                self.dlg.label_11.setEnabled(False)

            else:
                self.dlg.plots_column.setEnabled(False)
                self.dlg.sample_number.setEnabled(True)
                self.dlg.label_9.setEnabled(False)
                self.dlg.by_hectare_checkbox.setEnabled(True)
                self.dlg.label_11.setEnabled(True)



    def update_systematic_custom_parameters_state(self):
        """Enable or disable systematic custom parameters based on the distribution type."""
        is_custom_systematic = self.dlg.distribution.currentText() == "systematic custom"
        self.dlg.systematic_custom_x_d.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_y_d.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_x_label.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_y_label.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_title.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_degree_label.setEnabled(is_custom_systematic)
        self.dlg.systematic_custom_degree.setEnabled(is_custom_systematic)

    def update_rectangle_plot_format_parameters_state(self):
        is_custom_plot_format = self.dlg.plot_format_selector.currentText() == "rectangle"
        self.dlg.custom_plot_format_label_x.setEnabled(is_custom_plot_format)
        self.dlg.custom_plot_format_label_y.setEnabled(is_custom_plot_format)
        self.dlg.custom_plot_format_x_value.setEnabled(is_custom_plot_format)
        self.dlg.custom_plot_format_y_value.setEnabled(is_custom_plot_format)

        # Desativa o campo 'Plot area' se o formato selecionado é 'rectangle'
        self.dlg.plot_area.setEnabled(not is_custom_plot_format)

    def toggle_sample_number_fields(self):
        """Habilita ou desabilita os campos 'label_3' e 'sample_number' dependendo da opção selecionada."""
        use_n_plots_col = self.dlg.use_column_n_plots_checkbox.isChecked()

        # Se use_column_n_plots_checkbox estiver marcado, desabilita os campos
        self.dlg.label_3.setEnabled(not use_n_plots_col)
        self.dlg.sample_number.setEnabled(not use_n_plots_col)

    def run(self):
        iface.messageBar().pushMessage("Atention", "Large areas may take some time to compute...", level=Qgis.Info)
        """Run method that performs all the real work"""
        if self.dialog_open:  # Check if the dialog is already open
            QMessageBox.warning(self.iface.mainWindow(), "Plugin Already Open", "The plugin is already open.")
            return

        if self.first_start:
            self.first_start = False
            self.dlg = PlotAllocationDialog()
            self.dlg.distribution.addItems(["random", "best sampling", "systematic", "systematic custom"])

            # Define o botão de usar coluna como n plots como padrão
            self.dlg.use_column_n_plots_checkbox.setChecked(True)
            self.dlg.radio_button_south.setChecked(True)
            # Initialize the plot_format_selector options
            self.dlg.plot_format_selector.addItems(["round", "squared", "rectangle"])

            # Conectar os botões de rádio à função de ativação/desativação dos campos
            self.dlg.use_column_n_plots_checkbox.toggled.connect(self.update_sample_number_state)



            # Conectar o fechamento da janela para resetar o estado do plugin
            self.dlg.finished.connect(self.on_dialog_closed)

        self.dialog_open = True  # Set the attribute to True when opening the dialog
        self.dlg.show()

        self.load_vectors()




        if not self.select_button_connected:
            self.dlg.select_button.clicked.connect(self.open_vector)
            self.dlg.shp.currentIndexChanged.connect(self.update_ok_button_state)
            self.select_button_connected = True

        if not self.select_save_button_connected:
            self.dlg.select_save_button.clicked.connect(self.select_output_file)
            self.dlg.output_name.textChanged.connect(self.update_ok_button_state)
            self.select_save_button_connected = True

        # Conectar os botões de rádio à função de ativação/desativação dos campos
        self.dlg.use_column_n_plots_checkbox.toggled.connect(self.toggle_sample_number_fields)


        # Chamar a função ao iniciar o diálogo para garantir que os estados estejam corretos
        self.toggle_sample_number_fields()

        self.dlg.shp.currentIndexChanged.connect(self.update_plots_column)

        self.dlg.radio_button_south.toggled.connect(self.update_epsg_selector)
        self.dlg.radio_button_north.toggled.connect(self.update_epsg_selector)

        self.dlg.plot_area.valueChanged.connect(self.update_ok_button_state)
        self.dlg.sample_number.valueChanged.connect(self.update_ok_button_state)
        self.dlg.min_border_distance.valueChanged.connect(self.update_ok_button_state)
        self.dlg.epsg_selector.currentIndexChanged.connect(self.update_ok_button_state)

        self.dlg.distribution.currentIndexChanged.connect(self.update_sample_number_state)
        self.dlg.distribution.currentIndexChanged.connect(self.update_systematic_custom_parameters_state)
        self.dlg.plot_format_selector.currentIndexChanged.connect(self.update_systematic_custom_parameters_state)
        self.dlg.distribution.currentIndexChanged.connect(self.update_rectangle_plot_format_parameters_state)
        self.dlg.plot_format_selector.currentIndexChanged.connect(self.update_rectangle_plot_format_parameters_state)
        self.dlg.distribution.currentIndexChanged.connect(self.update_ok_button_state)
        self.dlg.plot_format_selector.currentIndexChanged.connect(self.update_ok_button_state)
        self.dlg.custom_plot_format_x_value.valueChanged.connect(self.update_ok_button_state)
        self.dlg.custom_plot_format_y_value.valueChanged.connect(self.update_ok_button_state)
        self.dlg.custom_plot_format_y_value.valueChanged.connect(self.update_systematic_custom_parameters_state)
        self.dlg.custom_plot_format_x_value.valueChanged.connect(self.update_systematic_custom_parameters_state)
        self.dlg.systematic_custom_x_d.valueChanged.connect(self.update_ok_button_state)
        self.dlg.systematic_custom_y_d.valueChanged.connect(self.update_ok_button_state)

        self.update_ok_button_state()
        self.update_sample_number_state()
        self.update_plots_column()
        self.update_systematic_custom_parameters_state()

        result = self.dlg.exec_()
        if result:
            self.shp_layer = self.set_input_layer()

            self.crs = self.shp_layer.crs().authid()
            self.min_border_distance = self.dlg.min_border_distance.value()
            if self.min_border_distance < 0:
                raise ValueError("Minimum border distance must be non-negative.")
            if debbug_mode:
                QgsMessageLog.logMessage(f"Min Border Distance: {self.min_border_distance}", 'Your Plugin Name', Qgis.Info)

            self.plot_area = self.dlg.plot_area.value()
            self.plot_format = self.dlg.plot_format_selector.currentText()

            self.layer_name = self.dlg.shp.currentText()
            if self.dlg.plot_format_selector.currentText() == "rectangle":
                self.custom_plot_format_x_value = float(
                    self.dlg.custom_plot_format_x_value.value())
                self.custom_plot_format_y_value = float(
                    self.dlg.custom_plot_format_y_value.value())
                self.plot_area = self.custom_plot_format_x_value * self.custom_plot_format_y_value

            self.reduced_shp = self.get_reduced_polygons_to_min_border_distance(self.shp_layer)

            if debbug_mode:
                QgsProject.instance().addMapLayer(self.reduced_shp, True)

            self.reduced_shp_total_area = self.calculate_total_area(self.reduced_shp)

            if self.dlg.by_hectare_checkbox.isChecked() and self.dlg.sample_number.value() < 1:
                QMessageBox.warning(self.dlg, "Warning",
                                    "Sample number must be at least 1 when using 'by hectare' option.")
                return  # Interrompe a execução caso a condição seja atendida


            if not self.shp_layer:
                return
            try:

                if self.plot_area <= 0:
                    raise ValueError("Plot area must be greater than zero.")

                use_column_for_plots = self.dlg.use_column_n_plots_checkbox.isChecked()
                selected_plots_column = self.dlg.plots_column.currentText()


                self.sample_number = self.dlg.sample_number.value()
                if self.sample_number < 1:
                    self.max_number_of_points = math.ceil((self.sample_number * self.total_area) / self.plot_area)
                else:
                    self.max_number_of_points = math.floor(self.sample_number)



                """CRIAÇÃO DAS PARCELAS PROPORCIONAIS POR TALHÃO OU USANDO UMA COLUNA ESPECIFICADA"""

                self.plots_list = {}  #  Agora será um dicionário {id_talhao: num_parcelas}
                area_prop_list = {}

                #  Passo 1: Criar dicionário com os valores de 'selected_plots_column' da camada original
                plots_dict = {}

                if use_column_for_plots and selected_plots_column:
                    if debbug_mode:
                        QgsMessageLog.logMessage(f"self.shp_layer.fields(): {self.shp_layer.fields()}", 'Your Plugin Name',
                                             Qgis.Info)
                    for feature in self.shp_layer.getFeatures():
                        value = feature[selected_plots_column]
                        if value is not None and isinstance(value, (int, float)):  # Certifica-se de que é numérico
                            plots_dict[feature.id()] = int(value)  # Armazena como inteiro
                        else:
                            plots_dict[feature.id()] = 0  # Caso contrário, assume zero

                #  Passo 2: Iterar sobre `reduced_shp` e usar os valores do dicionário `plots_dict`
                for feature in self.reduced_shp.getFeatures():
                    feature_id = feature.id()

                    if use_column_for_plots and feature_id in plots_dict:
                        num_plots = plots_dict[feature_id]  # Obtém os valores da camada original
                        if debbug_mode:
                            QgsMessageLog.logMessage(f"num_plots: {num_plots}", 'Your Plugin Name',
                                                     Qgis.Info)
                        geom = feature.geometry()
                        area = geom.area()
                        area_prop = area / self.reduced_shp_total_area

                        if num_plots is None or num_plots < 0:
                            num_plots = 0  # Evita valores negativos ou nulos
                    else:
                        if self.dlg.by_hectare_checkbox.isChecked():

                            # Calcula proporcionalmente pela área se não estiver usando uma coluna específica
                            geom = feature.geometry()
                            area = geom.area()
                            area_prop = area / self.reduced_shp_total_area
                            num_plots = round((area/10000) * self.sample_number)
                            if debbug_mode:
                                QgsMessageLog.logMessage(f"self.sample_number: {self.sample_number}", 'Your Plugin Name',
                                                         Qgis.Info)

                                QgsMessageLog.logMessage(f"area do poligono: {area}", 'Your Plugin Name',
                                                         Qgis.Info)
                                QgsMessageLog.logMessage(f"plots do poligono: {num_plots}", 'Your Plugin Name',
                                                         Qgis.Info)
                        else:
                            # Calcula proporcionalmente pela área se não estiver usando uma coluna específica
                            geom = feature.geometry()
                            area = geom.area()
                            area_prop = area / self.reduced_shp_total_area
                            num_plots = round(area_prop * self.max_number_of_points)  # Converte para número de parcelas

                    self.plots_list[feature_id] = math.floor(num_plots)  # Adiciona no dicionário
                    area_prop_list[feature_id] = area_prop  # Guarda proporção da área

                self.max_number_of_points = sum(self.plots_list.values())
                self.sample_number = self.max_number_of_points

                # Ajuste `self.plots_list` se for proporcional
                if not use_column_for_plots and not self.dlg.by_hectare_checkbox.isChecked():
                    difference = self.max_number_of_points - sum(self.plots_list.values())

                    # Distribui a diferença para garantir que a soma seja igual a max_number_of_points
                    while difference != 0:
                        if difference > 0:
                            for feature_id in sorted(area_prop_list, key=area_prop_list.get, reverse=True):  # Ordena em ordem decrescente de área
                                self.plots_list[feature_id] += 1
                                difference = self.max_number_of_points - sum(self.plots_list.values())
                                if difference == 0:
                                    break
                        elif difference < 0:
                            for feature_id in sorted(area_prop_list, key=area_prop_list.get):  # Ordena em ordem crescente de área
                                if self.plots_list[feature_id] > 0:
                                    self.plots_list[feature_id] -= 1
                                    difference = self.max_number_of_points - sum(self.plots_list.values())
                                    if difference == 0:
                                        break
                if debbug_mode:
                    QgsMessageLog.logMessage(f"self.plots_list: {self.plots_list}", 'Your Plugin Name',
                                         Qgis.Info)
                """FIM CRIAÇÃO DAS PARCELAS PROPORCIONAIS OU USANDO UMA COLUNA ESPECIFICADA"""

                # Adiciona os novos atributos à tabela de atributos, se ainda não existirem
                if self.reduced_shp.fields().indexFromName('area_prop') == -1:
                    self.reduced_shp.dataProvider().addAttributes([QgsField('area_prop', QVariant.Double)])
                if self.reduced_shp.fields().indexFromName('plots') == -1:
                    self.reduced_shp.dataProvider().addAttributes([QgsField('plots', QVariant.Int)])

                self.reduced_shp.updateFields()
                # Inicia a edição da camada
                self.reduced_shp.startEditing()

                # Atualiza os valores dos atributos
                for feature in self.reduced_shp.getFeatures():
                    feature_id = feature.id()

                    area_prop_value = area_prop_list.get(feature_id, 0)  # Obtém o valor da proporção de área
                    plots_value = self.plots_list.get(feature_id, 0)  # Obtém o número de parcelas do dicionário

                    self.reduced_shp.changeAttributeValue(feature_id,
                                                          self.reduced_shp.fields().indexFromName('area_prop'),
                                                          area_prop_value)
                    self.reduced_shp.changeAttributeValue(feature_id,
                                                          self.reduced_shp.fields().indexFromName('plots'),
                                                          int(plots_value))

                # Confirma as alterações
                self.reduced_shp.commitChanges()

                # Confirma as alterações
                self.reduced_shp.commitChanges()


                if self.sample_number < 0:
                    raise ValueError("Sample number must be non-negative.")


                self.distribution = self.dlg.distribution.currentText()
                if debbug_mode:
                    QgsMessageLog.logMessage(f"Distribution: {self.distribution}", 'Your Plugin Name', Qgis.Info)
                self.buffer_check_box = self.dlg.buffer_check_box.isChecked()
                if debbug_mode:
                    QgsMessageLog.logMessage(f"Buffer Check Box: {self.buffer_check_box}", 'Your Plugin Name', Qgis.Info)

                self.total_area = self.calculate_total_area(self.shp_layer)
                if self.total_area <= 0:
                    raise ValueError("Total area must be greater than zero.")
                if debbug_mode:
                    QgsMessageLog.logMessage(f"Total Area: {self.total_area}", 'Your Plugin Name', Qgis.Info)

                if self.distribution =="systematic custom" or self.distribution=="systematic":
                    self.sample_number=0
                if self.sample_number * self.plot_area >= self.total_area:
                    QMessageBox.warning(self.dlg, "Aviso", "The number of plots exceeds the total boundary area.")
                    return

                if self.distribution == "random":
                    point_layer = self._generate_random_sample_points(self.reduced_shp, self.plot_area)
                elif self.distribution == "systematic":
                    point_layer = self._generate_systematic_sample_points(self.reduced_shp, self.plot_area)
                elif self.distribution == "best sampling":
                    point_layer = self._generate_systematic_best_sampling_sample_points()
                elif self.distribution == "systematic custom":
                    self.systematic_custom_x_d = self.dlg.systematic_custom_x_d.value()
                    self.systematic_custom_y_d = self.dlg.systematic_custom_y_d.value()
                    self.systematic_custom_degree = self.dlg.systematic_custom_degree.value()

                    point_layer = self._generate_systematic_custom_sample_points(self.reduced_shp, self.plot_area)
                else:
                    QMessageBox.warning(self.dlg, "Aviso", "Distribuição não suportada.")
                    return

                if point_layer:
                    output_path = self.dlg.output_name.text()
                    if output_path:
                        output_dir = os.path.dirname(output_path)
                        if os.path.isdir(output_dir):
                            options = QgsVectorFileWriter.SaveVectorOptions()
                            options.driverName = "ESRI Shapefile"
                            transform_context = QgsProject.instance().transformContext()
                            error = QgsVectorFileWriter.writeAsVectorFormatV3(
                                point_layer,
                                output_path,
                                QgsCoordinateTransformContext(transform_context),
                                options
                            )
                            if error[0] == QgsVectorFileWriter.NoError:
                                saved_layer = QgsVectorLayer(output_path,
                                                             os.path.splitext(os.path.basename(output_path))[0], "ogr")
                                if saved_layer.isValid():
                                    QgsProject.instance().addMapLayer(saved_layer)

                                    if self.buffer_check_box:
                                        buffer_distance = math.sqrt(
                                            self.plot_area / math.pi) if self.plot_format == "round" else math.sqrt(
                                            self.plot_area)

                                        # Criar a camada de buffer
                                        buffer_layer = self.create_buffer_layer(saved_layer, buffer_distance,
                                                                                self.plot_format)

                                        if buffer_layer.isValid():
                                            # Definir o caminho de saída para o buffer
                                            buffer_output_path = os.path.join(output_dir,
                                                                              os.path.splitext(
                                                                                  os.path.basename(output_path))[
                                                                                  0] + "_buffer.shp")  # Adicione a extensão .shp
                                            buffer_options = QgsVectorFileWriter.SaveVectorOptions()
                                            buffer_options.driverName = "ESRI Shapefile"

                                            buffer_error = QgsVectorFileWriter.writeAsVectorFormatV3(
                                                buffer_layer,
                                                buffer_output_path,
                                                QgsCoordinateTransformContext(transform_context),
                                                buffer_options
                                            )

                                            if buffer_error[0] == QgsVectorFileWriter.NoError:
                                                # Remover a camada de buffer existente, se houver
                                                existing_buffer_layer = QgsProject.instance().mapLayersByName(
                                                    f"{saved_layer.name()}_buffer")
                                                if existing_buffer_layer:
                                                    QgsProject.instance().removeMapLayer(existing_buffer_layer[
                                                                                             0].id())  # Remove a primeira camada encontrada

                                                # Adicionar a nova camada de buffer ao projeto
                                                new_buffer_layer = QgsVectorLayer(buffer_output_path,
                                                                                  f"{saved_layer.name()}_buffer", "ogr")
                                                QgsProject.instance().addMapLayer(new_buffer_layer)

                                                # Definir a opacidade do buffer
                                                self.set_layer_opacity(new_buffer_layer, 0.5)


                                            else:
                                                QMessageBox.critical(self.dlg, "Erro",
                                                                     f"Error saving buffer layer: {buffer_error[1]}")
                                                return
                                        else:
                                            QMessageBox.critical(self.dlg, "Erro", "Failed to create buffer layer.")
                                            return
                                else:
                                    QMessageBox.critical(self.dlg, "Erro", "Failed to load the saved layer.")
                                    return
                            else:
                                QMessageBox.critical(self.dlg, "Erro", f"Error saving point layer: {error[1]}")
                                return
                        else:
                            QMessageBox.critical(self.dlg, "Erro", "Output directory does not exist.")
                            return
                    else:
                        QgsProject.instance().addMapLayer(point_layer)

                        if self.buffer_check_box:
                            buffer_distance = math.sqrt(
                                self.plot_area / math.pi) if self.plot_format == "round" else math.sqrt(self.plot_area)
                            self.create_buffer_layer(point_layer, buffer_distance, self.plot_format)
                else:
                    QMessageBox.critical(self.dlg, "Erro", "Não foi possível gerar a camada de pontos.")
                    return


            except Exception as e:
                if debbug_mode:
                    QgsMessageLog.logMessage(f"Erro ao acessar um campo: {e}", 'Your Plugin Name', Qgis.Critical)
                    QgsMessageLog.logMessage(traceback.format_exc(), 'Your Plugin Name', Qgis.Critical)
                QMessageBox.critical(self.dlg, "Erro", f"Erro ao acessar um campo: {e}")
                return

    def on_dialog_closed(self):
        self.dialog_open = False









