# -*- coding: utf-8 -*-
from qgis.PyQt.QtCore import QVariant
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterEnum,
    QgsProcessingParameterNumber,
    QgsProcessingParameterFeatureSink,
    QgsFields,
    QgsField,
    QgsFeature,
    QgsWkbTypes,
)

class PercentRatesAlgorithm(QgsProcessingAlgorithm):
    入力レイヤ = "INPUT"
    分母 = "DENOM"
    分子 = "NUMS"
    基数 = "BASE"
    桁数 = "DECIMALS"
    出力 = "OUTPUT"

    基数ラベル = ["100分率", "1,000分率", "10,000分率", "100,000分率"]
    基数値 = [100, 1000, 10000, 100000]

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.入力レイヤ,
                "入力ベクターレイヤ",
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.分母,
                "分母にするフィールド（例：総人口）",
                parentLayerParameterName=self.入力レイヤ,
                type=QgsProcessingParameterField.Numeric
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.分子,
                "分子にするフィールド（複数選択可）",
                parentLayerParameterName=self.入力レイヤ,
                type=QgsProcessingParameterField.Numeric,
                allowMultiple=True
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.基数,
                "分率の基数",
                options=self.基数ラベル,
                defaultValue=0
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.桁数,
                "四捨五入する小数点以下の桁数",
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=1,
                minValue=0,
                maxValue=10
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.出力,
                "出力（一時レイヤ）",
                type=QgsProcessing.TypeVectorAnyGeometry
            )
        )

    def name(self):
        return "割合計算"

    def displayName(self):
        return "割合計算（複数分子）"

    def group(self):
        return "GISPHN計算"

    def groupId(self):
        return "gisphn_keisan_group"

    def createInstance(self):
        return PercentRatesAlgorithm()

    def processAlgorithm(self, parameters, context, feedback):
        source = self.parameterAsSource(parameters, self.入力レイヤ, context)
        if source is None:
            raise QgsProcessingException("入力レイヤが取得できません。")

        denom_field = self.parameterAsString(parameters, self.分母, context)
        num_fields = self.parameterAsFields(parameters, self.分子, context)
        if not num_fields:
            raise QgsProcessingException("分子フィールドを少なくとも1つ選択してください。")

        base_index = self.parameterAsEnum(parameters, self.基数, context)
        base_value = self.基数値[base_index]
        decimals = int(self.parameterAsInt(parameters, self.桁数, context))

        out_fields = QgsFields()
        for f in source.fields():
            out_fields.append(f)

        for nf in num_fields:
            proposed = f"{nf}_割合{base_value}"
            out_fields.append(QgsField(proposed, QVariant.Double, len=20, prec=decimals))

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.出力,
            context,
            out_fields,
            source.wkbType() if source.wkbType() != QgsWkbTypes.Unknown else QgsWkbTypes.NoGeometry,
            source.sourceCrs()
        )

        denom_idx = source.fields().indexOf(denom_field)
        num_idxs = [source.fields().indexOf(nf) for nf in num_fields]
        if denom_idx < 0 or any(idx < 0 for idx in num_idxs):
            raise QgsProcessingException("選択したフィールド名が見つかりません。")

        for feat in source.getFeatures():
            attrs = list(feat.attributes())
            denom_val = attrs[denom_idx]
            try:
                safe_den = (denom_val is not None) and (float(denom_val) != 0.0)
            except Exception:
                safe_den = False

        # compute
            for nidx in num_idxs:
                out_val = None
                if safe_den:
                    num_val = attrs[nidx]
                    if num_val is not None:
                        try:
                            out_val = round((float(num_val) / float(denom_val)) * base_value, decimals)
                        except Exception:
                            out_val = None
                attrs.append(out_val)

            new_feat = QgsFeature(out_fields)
            new_feat.setGeometry(feat.geometry())
            new_feat.setAttributes(attrs)
            sink.addFeature(new_feat)

        return {self.出力: dest_id}
