import os
import sys
from dataclasses import dataclass
from functools import cached_property, lru_cache
from importlib import invalidate_caches
from pathlib import Path

from qgis.core import Qgis, QgsApplication
from qgis.utils import iface

DEFAULT_PREFIX = Path(QgsApplication.qgisSettingsDirPath()) / "python" / "dependencies"


def clear_cache():
    sys.path_importer_cache.clear()
    invalidate_caches()


@dataclass
class PipPrefix:
    path: str = DEFAULT_PREFIX

    def __post_init__(self):
        self.path = Path(self.path).resolve()

    def __setattr__(self, name, value):
        """Invalidate cached properties (scheme, size)"""
        if name == "name":
            self.__dict__.pop("scheme", None)
            self.__dict__.pop("size", None)
        super().__setattr__(name, value)

    def __str__(self) -> str:
        return str(self.path)

    def __repr__(self) -> str:
        return f"{self} ({self.sizestr})"

    def __enter__(self):
        if not self:
            self._temp = True
            self.register(False)

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.__dict__.pop("_temp", False):
            self.unregister(False)

    def __bool__(self) -> bool:
        lib, bin = str(self.lib.resolve()), str(self.bin.resolve())
        path = sys.path

        return lib in path and bin in path

    @classmethod
    @lru_cache(maxsize=1)
    def plat_scheme(cls) -> dict[str, Path]:
        """https://docs.python.org/3/library/sysconfig.html#sysconfig-prefix-scheme"""

        if os.name == "nt":
            return {
                "prefix": Path(),
                "stdlib": Path() / "Lib",
                "platstdlib": Path() / "Lib",
                "platlib": Path() / "Lib" / "site-packages",
                "purelib": Path() / "Lib" / "site-packages",
                "include": Path() / "Include",
                "platinclude": Path() / "Include",
                "scripts": Path() / "Scripts",
                "data": Path(),
            }
        py_ver = f"python{sys.version_info.major}.{sys.version_info.minor}"
        return {
            "prefix": Path(),
            "stdlib": Path() / "lib" / f"{py_ver}",
            "platstdlib": Path() / "lib" / f"{py_ver}",
            "platlib": Path() / "lib" / f"{py_ver}" / "site-packages",
            "purelib": Path() / "lib" / f"{py_ver}" / "site-packages",
            "include": Path() / "include" / f"{py_ver}",
            "platinclude": Path() / "include" / f"{py_ver}",
            "scripts": Path() / "bin",
            "data": Path(),
        }

    @cached_property
    def scheme(self) -> dict[str, Path]:
        return {k: self.path / v for k, v in self.plat_scheme().items()}

    @property
    def lib(self) -> Path:
        return self.scheme["platlib"]

    @property
    def bin(self) -> Path:
        return self.scheme["scripts"]

    @cached_property
    def size(self) -> int:
        """Prefix size in Bytes"""
        size = 0
        for path in set(self.scheme.values()):
            size += sum([f.stat().st_size for f in path.glob("**/*")])
        return size

    @property
    def sizestr(self) -> str:
        """Prefix size as formated string"""
        size = self.size
        for unit in ("B", "KB", "MB", "GB", "TB", "PB"):
            if size < 1024:
                return f"{size:.1f}{unit}"
            size /= 1024
        return None

    def register(self, notify: bool = True):
        self.lib.mkdir(parents=True, exist_ok=True)
        self.bin.mkdir(parents=True, exist_ok=True)

        lib, bin = str(self.lib.resolve()), str(self.bin.resolve())

        if lib not in sys.path:
            # TODO: is it necessary ?
            # os.environ["PYTHONPATH"] = lib + ";" + os.environ.get("PYTHONPATH", "")
            sys.path.insert(1, lib)
            if notify:
                iface.messageBar().pushMessage(
                    f"{lib} was added to PATH",
                    level=Qgis.Info,
                )

        if bin not in sys.path:
            sys.path.insert(1, bin)
            if notify:
                iface.messageBar().pushMessage(
                    f"{bin} was added to PATH",
                    level=Qgis.Info,
                )
        clear_cache()

    def unregister(self, notify: bool = True):
        lib, bin = str(self.lib.resolve()), str(self.bin.resolve())

        if lib not in sys.path:
            # TODO: is it necessary ?
            # os.environ["PYTHONPATH"] = (
            #     os.environ.get("PYTHONPATH", "").replace(lib, "").replace(";;", ";")
            # )
            sys.path.remove(lib)
            if notify:
                iface.messageBar().pushMessage(
                    f"{lib} was removed from PATH",
                    level=Qgis.Info,
                )

        if bin not in sys.path:
            sys.path.remove(bin)
            if notify:
                iface.messageBar().pushMessage(
                    f"{bin} was removed from PATH",
                    level=Qgis.Info,
                )

        sys.path_importer_cache.clear()
