from __future__ import annotations

import os
import subprocess
from datetime import datetime
from pathlib import Path

from qgis.PyQt.QtCore import QStandardPaths, Qt
from qgis.PyQt.QtGui import QCursor, QIcon
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QProgressBar,
    QPushButton,
    QTextEdit,
    QVBoxLayout,
    QWidget,
)

INTERPRETER = Path(QStandardPaths.findExecutable("python")).resolve()


class QDialogSubprocess(QDialog):
    def __init__(
        self,
        *args,
        parent: QWidget = None,
        title: str = "",
        icon: QIcon = None,
    ):
        super().__init__(
            parent, Qt.WindowType.CustomizeWindowHint | Qt.WindowType.WindowTitleHint
        )
        self.setWindowTitle(title or "subprocess")
        self.setWindowIcon(QIcon(icon))
        # self.setWindowModality(Qt.WindowModality.WindowModal)

        self.progress = QProgressBar(parent=self)
        self.progress.setTextVisible(False)

        self.logger = QTextEdit(parent=parent)
        palette = self.logger.palette()
        palette.setColor(palette.Base, Qt.GlobalColor.black)
        palette.setColor(palette.Text, Qt.GlobalColor.white)
        self.logger.setPalette(palette)
        self.logger.setText(" ".join([str(_) for _ in args]).strip())

        self.button = QPushButton("Run", parent=self)
        self.button.clicked.connect(self.run)

        layout = QVBoxLayout()
        layout.addWidget(self.progress)
        layout.addWidget(self.logger)
        layout.addWidget(self.button)
        self.setLayout(layout)

    def _setup_close_button(self):
        self.button.setText("Close")
        self.button.clicked.disconnect()
        self.button.clicked.connect(self.close)

    @property
    def startupinfo(self):
        if os.name == "nt":
            from subprocess import (  # noqa: PLC0415
                STARTF_USESHOWWINDOW,
                STARTF_USESTDHANDLES,
                STARTUPINFO,
                SW_HIDE,
            )

            startupinfo = STARTUPINFO()
            startupinfo.dwFlags |= STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = SW_HIDE
            return startupinfo
        return None

    def run(self, *args) -> int:
        cmd = " ".join([str(_) for _ in args]).strip() or self.logger.toPlainText()
        if not cmd:
            self.reject()  # return 0

        # Reset UI
        self.logger.clear()
        self.progress.setRange(0, 0)
        # Log command
        self.logger.append(f"{datetime.now().strftime('%Y-%m-%d %H:%M')} > {cmd}\n")

        self.open()
        QApplication.setOverrideCursor(QCursor(Qt.CursorShape.WaitCursor))

        try:
            process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                startupinfo=self.startupinfo,
                text=True,
            )
        except (OSError, subprocess.SubprocessError) as err:
            self.logger.append(f"\nError: {err}")
            self._setup_close_button()
            returncode = -1
        else:
            self.button.setText("Cancel")
            self.button.clicked.disconnect()
            self.button.clicked.connect(process.kill)

            # Read output line by line
            while process.poll() is None:
                QApplication.processEvents()
                output = process.stdout.readline()
                if output:
                    self.logger.append(output)

            # Finalize: Process completed, but keep dialog open
            self.progress.setRange(0, 1)
            self.progress.setValue(1)
            self.logger.append(process.stdout.read())

        finally:
            returncode = process.returncode
            QApplication.restoreOverrideCursor()
            # Change button to "Close" and only close on click
            self._setup_close_button()
            self.activateWindow()
            self.setResult(returncode)

    def exec(self):
        self.run()
        return super().exec()


def dlg_run(
    *args,
    parent: QWidget = None,
    title: str | None = None,
    icon: QIcon = None,
) -> int:
    dlg = QDialogSubprocess(*args, parent=parent, title=title, icon=icon)
    return dlg.exec()
