import os
import shutil
from collections import namedtuple
from functools import cached_property
from pathlib import Path

import qgis
from qgis.core import Qgis, QgsMessageLog
from qgis.PyQt.QtCore import QObject, Qt
from qgis.PyQt.QtGui import QCursor
from qgis.PyQt.QtWidgets import QAction, QApplication, QFileDialog, QMessageBox, QWidget

from .loader import PluginLoader
from .prefix import DEFAULT_PREFIX, PipPrefix
from .settings import PluginSettings
from .ui import DependenciesDialog, qicon


class PackageInstallerQgis(QObject):
    """QGIS Plugin Implementation."""

    def __init__(self, iface):

        super().__init__()
        self.iface = iface
        # register prefix
        self.prefix.register()
        # patch qgis.utils.loadPlugin
        QgsMessageLog.logMessage(
            "Patch `qgis.utils.loadPlugin`", tag=self.name, level=Qgis.Info
        )
        self._loadPlugin = qgis.utils.loadPlugin
        qgis.utils.loadPlugin = self.loadPlugin
        # store previsouly loaded plugins
        self._defered_plugins = list(qgis.utils.plugins)
        if self.__class__.__name__ in self._defered_plugins:
            self._defered_plugins.remove(self.__class__.__name__, None)

        self.iface.initializationCompleted.connect(self.__post_init__)
        self.iface.messageBar().pushMessage(
            f"{self.name}",
            "ready to roll",
            level=Qgis.Success,
        )

    def __post_init__(self):

        # reload deferred plugins
        for plugin in self._defered_plugins:
            self.unloadPlugin(plugin)
            self.loadPlugin(plugin, start=True)

    @cached_property
    def settings(self):
        return PluginSettings()

    @property
    def name(self):
        return self.settings.get("name", self.__class__.__name__)

    @property
    def prefix(self):
        return PipPrefix(self.settings.get("prefix", DEFAULT_PREFIX))

    @prefix.setter
    def prefix(self, value: str):
        value = PipPrefix(value)
        if value.path != self.prefix.path:
            self.prefix.unregister()
            self.settings.set("prefix", str(value.path))
            self.prefix.register()

    # required by QGIS at plugin start
    def initGui(self):
        self.gui = DependenciesDialog(self)
        # add action into plugins menu
        self.menu = self.iface.pluginMenu().addMenu(qicon(), self.name)
        self.menu.addAction(self.show)
        self.menu.addSeparator()
        self.menu.addAction(self.about)
        # add launcher into plugins toolbar
        self.iface.pluginToolBar().addAction(self.show)

    # required by QGIS at plugin stop
    def unload(self):
        # remove the menu entries
        self.iface.pluginMenu().removeAction(self.menu.menuAction())
        # remove launcher into plugins toolbar
        self.iface.pluginToolBar().removeAction(self.show)
        # restore original qgis.utils.loadPlugin
        QgsMessageLog.logMessage(
            "Restore original `qgis.utils.loadPlugin`", tag=self.name, level=Qgis.Info
        )
        qgis.utils.loadPlugin = self._loadPlugin

    def loadPlugin(self, plugin: str, start: bool = False) -> bool:

        # handle per-plugin isolation
        return PluginLoader.loadPlugin(plugin, start)

    def unloadPlugin(self, plugin: str):
        qgis.utils.unloadPlugin(plugin)

    def listPlugins(self) -> list[namedtuple]:
        Plugin = namedtuple(
            "Plugin", ["name", "path", "prefix", "dependencies", "requirements"]
        )
        plugins = []
        for p in qgis.utils.plugins:
            with PluginLoader(p) as l:
                if not l.needs_third_parties:
                    continue
                plugins.append(Plugin(l.name, l.path, l.prefix.path, *l.dependencies))

    @cached_property
    def about(self) -> QAction:
        """Display plugin details"""

        def callback():
            dlg = QWidget(self.iface.mainWindow())
            dlg.setWindowIcon(qicon())

            content = f"<b>{self.name}</b><br>"
            content += f"<i>{self.settings.get('description')}</i><br><br>"
            content += f"{self.settings.get('about')}<br><br>"

            content += f"<b>Author</b> : <a href=mailto:{self.settings.get('email')}>{self.settings.get('author')}</a><br>"
            if homepage := self.settings.get("homepage", None):
                content += f"<b>Homepage</b> : <a href={homepage}>GitLab</a><br>"
            content += f"<b>Source code</b> : <a href={self.settings.get('repository')}>GitLab</a><br>"
            if tracker := self.settings.get("tracker", None):
                content += f"<b>Repport issue</b> : <a href={tracker}>GitLab</a><br>"
            if doc := self.settings.get("documentation", None):
                content += f"<b>Documentation</b> : <a href={doc}>GitLab</a><br>"

            content += f"<br><b>Version</b> : {self.settings.get('version')}<br>"
            content += f"<i><b>Requires</b> QGIS &#x02265; {self.settings.get('qgisMinimumVersion')}</i><br>"
            if plugins := self.settings.get("plugin_dependencies"):
                content += f"<i><b>Depends on</b> : {plugins.strip(',')}</i><br>"
            if deps := self.settings.get("pip_dependencies"):
                content += f"<i><b>Requirements</b> : {deps.strip(',')}</i><br>"

            QMessageBox.about(dlg, f"About", content)
            dlg.show()

        action = QAction(
            qicon("SP_FileDialogInfoView"), f"About {self.name}", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def show(self) -> QAction:
        """Show the PIQ dependencies dialog"""

        def callback():
            if not self.gui:
                self.gui = DependenciesDialog(self)
            self.gui.show()

        action = QAction(icon=qicon(), text="Package Installer QGIS", parent=self)
        action.triggered.connect(callback)
        return action

    @cached_property
    def purge(self) -> QAction:
        """Purge the prefix content"""

        @self.busy
        def callback(*args) -> bool:

            if (
                QMessageBox.warning(
                    None,
                    "Warning",
                    f"You are about to purge the content of:\n\n"
                    + f"`{DEFAULT_PREFIX}`\n\n"
                    + f"This will remove {self.prefix.sizestr} of python packages.",
                    buttons=QMessageBox.StandardButtons(
                        QMessageBox.StandardButton.Apply
                        | QMessageBox.StandardButton.Cancel
                    ),
                    defaultButton=QMessageBox.StandardButton.Cancel,
                )
                == QMessageBox.StandardButton.Apply
                or not self.prefix.size
            ):
                try:
                    for child in self.prefix.scheme.values():
                        if child.is_dir():
                            # remove prefix content
                            shutil.rmtree(child, ignore_errors=True)
                            # WARNING: we MUST recreate it !
                            child.mkdir(exist_ok=True)
                except Exception as err:
                    self.iface.messageBar().pushMessage(
                        type(err).__name__,
                        str(err),
                        level=Qgis.Critical,
                    )

                    return False
            return True

        action = QAction(
            icon=qicon("SP_TrashIcon"), text="Purge all dependencies", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def restore(self) -> QAction:
        """Reset the prefix to its default location

        Note: this does NOT alter dependencies in any way !
        """

        @self.busy
        def callback(*args):

            new, old = Path(DEFAULT_PREFIX).resolve(), Path(self.prefix.path).resolve()
            if old == new:
                return True

            response = (
                QMessageBox.warning(
                    None,
                    "Warning",
                    f"You are about to reset the prefix to:\n\n"
                    + f"`{new}`\n\n"
                    + f"Your current prefix contains {self.prefix.sizestr} of python packages, do you want to move them along ?",
                    buttons=QMessageBox.StandardButtons(
                        QMessageBox.StandardButton.Yes
                        | QMessageBox.StandardButton.No
                        | QMessageBox.StandardButton.Cancel
                    ),
                    defaultButton=QMessageBox.StandardButton.Cancel,
                )
                if self.prefix.size
                else QMessageBox.StandardButton.No
            )
            if response == QMessageBox.StandardButton.Yes:
                try:
                    shutil.copytree(old, new, symlinks=True, dirs_exist_ok=True)
                    self.prefix = str(new)
                except Exception as err:
                    self.iface.messageBar().pushMessage(
                        type(err).__name__,
                        str(err),
                        level=Qgis.Critical,
                    )
                    return False
            elif response == QMessageBox.StandardButton.No:
                self.prefix = str(new)
            return True

        action = QAction(
            icon=qicon("SP_ComputerIcon"), text="Reset default values", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def reveal(self) -> QAction:
        """Reveal the prefix path in explorer"""

        def callback(*args):
            os.startfile(self.prefix.path)

        action = QAction(
            icon=qicon("SP_DirOpenIcon"), text="Reveal in explorer", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def move(self) -> QAction:
        """Move the prefix content to a new location"""

        @self.busy
        def callback(*args) -> None:
            destination = QFileDialog.getExistingDirectory(
                None,
                "Open Directory",
                str(self.prefix.path),
                QFileDialog.Option.ShowDirsOnly,
            )
            if not destination:
                return False

            new, old = Path(destination).resolve(), Path(self.prefix.path).resolve()
            if old == new:
                return True

            if (
                QMessageBox.warning(
                    None,
                    "Warning",
                    f"You are about to move the prefix content to:\n\n"
                    + f"`{old}`\n\n"
                    + f"This will move {self.prefix.sizestr} of python packages.",
                    buttons=QMessageBox.StandardButtons(
                        QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No
                    ),
                    defaultButton=QMessageBox.StandardButton.No,
                )
                == QMessageBox.StandardButton.Yes
                or not self.prefix.size
            ):
                try:
                    shutil.copytree(old, new, symlinks=True, dirs_exist_ok=True)
                except Exception as err:
                    self.iface.messageBar().pushMessage(
                        type(err).__name__,
                        str(err),
                        level=Qgis.Critical,
                    )
                    shutil.rmtree(new)
                    return False
                else:
                    shutil.rmtree(old, ignore_errors=True)
                    self.prefix = str(new)
                finally:
                    QApplication.restoreOverrideCursor()
            return True

        action = QAction(
            icon=qicon("SP_ArrowForward"), text="Move dependencies to...", parent=self
        )
        action.triggered.connect(callback)
        return action

    def busy(self, foo):
        def wrapper(*args, **kwargs):
            QApplication.setOverrideCursor(QCursor(Qt.WaitCursor))
            try:
                return foo(*args, **kwargs)
            except Exception as e:
                raise e
            finally:
                QApplication.restoreOverrideCursor()

        return wrapper
