from __future__ import annotations

import os
import subprocess
from datetime import datetime
from pathlib import Path

from qgis.PyQt.QtCore import QStandardPaths, Qt
from qgis.PyQt.QtGui import QCursor, QIcon
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QProgressBar,
    QPushButton,
    QTextEdit,
    QVBoxLayout,
    QWidget,
)

INTERPRETER = Path(QStandardPaths.findExecutable("python")).resolve()


class QDialogSubprocess(QDialog):

    def __init__(
        self,
        *args,
        parent: QWidget = None,
        title: str = "",
        icon: QIcon = None,
    ):
        super().__init__(parent, Qt.CustomizeWindowHint | Qt.WindowTitleHint)
        self.setWindowTitle(title or "subprocess")
        self.setWindowIcon(QIcon(icon))
        self.setWindowModality(Qt.WindowModality.WindowModal)

        self.progress = QProgressBar(parent=self)
        self.logger = QTextEdit(parent=parent)
        self.button = QPushButton("Run", parent=self)

        palette = self.logger.palette()
        palette.setColor(palette.Base, Qt.black)
        palette.setColor(palette.Text, Qt.white)
        self.logger.setPalette(palette)

        self.button.clicked.connect(self.run)
        self.progress.setTextVisible(False)

        layout = QVBoxLayout()
        layout.addWidget(self.progress)
        layout.addWidget(self.logger)
        layout.addWidget(self.button)
        self.setLayout(layout)

        if args:
            self.run(*args)

    @property
    def startupinfo(self):
        if os.name == "nt":
            from subprocess import (
                STARTF_USESHOWWINDOW,
                STARTF_USESTDHANDLES,
                STARTUPINFO,
                SW_HIDE,
            )

            startupinfo = STARTUPINFO()
            startupinfo.dwFlags |= STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = SW_HIDE
            return startupinfo
        else:
            return None

    def run(self, *args) -> int:

        cmd = " ".join([str(_) for _ in args]).strip() or self.logger.toPlainText()

        # reset & display logger
        self.logger.clear()
        self.progress.setRange(0, 0)
        self.open()

        # log the command in the logger
        self.logger.append(f"{datetime.now().strftime("%Y-%m-%d %H:%M")} > {cmd}\n")
        QApplication.setOverrideCursor(QCursor(Qt.WaitCursor))
        try:
            # launch the process
            process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                startupinfo=self.startupinfo,
                text=True,
            )
        except (OSError, subprocess.CalledProcessError) as err:
            self.logger.append(f"\n{err}")
            self.button.setText("Close")
            self.button.clicked.connect(self.close)
            returncode = -1
        else:
            # transform the run button into a cancel one
            self.button.setText("Cancel")
            self.button.clicked.connect(process.kill)

            # log the process outputs (stdout/stderr)
            while process and process.poll() is None:
                QApplication.processEvents()
                print(process.stdout)
                self.logger.append(process.stdout.read())

            # set the progressbar to "complete"
            self.progress.setRange(0, 1)
            self.progress.setValue(1)
            # flush remaining stdout
            self.logger.append(process.stdout.read())
            # propagate the returncode
            returncode = process.returncode
            self.setResult(QDialog.Accepted if returncode == 0 else QDialog.Rejected)
            # allow closing from UI
            self.button.setText("Close")
            self.button.clicked.connect(self.close)
        finally:
            QApplication.restoreOverrideCursor()
            return returncode


def dlg_run(
    *args,
    parent: QWidget = None,
    title: str = None,
    icon: QIcon = None,
) -> int:
    return QDialogSubprocess(parent=parent, title=title, icon=icon).run(*args)
