from __future__ import annotations

import sys
from dataclasses import dataclass
from functools import cached_property
from importlib import invalidate_caches
from pathlib import Path

from pip._internal.req.req_file import parse_requirements
from pip._vendor.pkg_resources import DistributionNotFound, VersionConflict, WorkingSet
from qgis.core import Qgis, QgsSettings
from qgis.utils import iface, loadPlugin, plugin_paths, pluginMetadata, startPlugin

from .cmd import INTERPRETER, dlg_run
from .prefix import PipPrefix
from .ui import qicon


def clear_cache():
    sys.path_importer_cache.clear()
    invalidate_caches()


@dataclass
class PluginLoader:

    name: str

    def __setattr__(self, name, value):
        if name == "name":
            self.__dict__.clear()
        super().__setattr__(name, value)

    def __post_init__(self):
        for p in plugin_paths:
            if first := next(Path(p).glob(self.name), None):
                self.path = first.resolve()
                return
        raise KeyError(f"Could not find {self.name}")

    @cached_property
    def prefix(self) -> PipPrefix:
        """Return the PIP prefix"""
        # find manifest's `general:pip_prefix` field
        prefix = pluginMetadata(self.name, "pip_prefix")
        # WARNING: not found defaults to "__error__" not None
        if prefix in ("", None, "__error__", "__default__"):
            return PipPrefix()
        elif prefix[0] in ("/", "\\") or prefix[1] == ":":
            return PipPrefix(prefix)
        else:
            return PipPrefix(self.path / prefix)

    @cached_property
    def dependencies(self) -> tuple[tuple[str]]:
        """Look for dependencies, both:
            - pip-like packages requirements
            - frozen environments requirements file(s)

        These dependencies are inferred from metadata's `pip_dependencies` field,
        with a default rollback to `requirements.txt` at plugin root directory.
        """

        pkgs, reqs = set(), set()

        # find manifest's `general:pip_dependencies` field
        pip_dependencies = pluginMetadata(self.name, "pip_dependencies")
        # not found defaults to "__error__" ...
        pip_dependencies = pip_dependencies.replace("__error__", "")
        # rollback to standard `requirements.txt` file
        if not pip_dependencies:
            requirements_txt = self.path / "requirements.txt"
            if requirements_txt.is_file():
                reqs.add(requirements_txt.resolve().as_posix())

        # parse `pip_dependencies` (may be files !)
        dependencies = [el.strip() for el in pip_dependencies.split(",")]
        for dep in filter(None, dependencies):
            file = self.path / dep
            if file.is_file():
                reqs.add(file.resolve().as_posix())
            else:
                pkgs.add(dep)

        return tuple(pkgs), tuple(reqs)

    @cached_property
    def has_dependencies(self) -> bool:
        return any(self.dependencies)

    @cached_property
    def requires_extra(self) -> bool:
        with self.prefix:

            if not self.has_dependencies:
                return False

            pkgs, reqs = self.dependencies
            env = WorkingSet()
            try:
                for p in pkgs:
                    env.require(p)
                for file in reqs:
                    for p in [el.requirement for el in parse_requirements(file, False)]:
                        env.require(p)
            except (VersionConflict, DistributionNotFound):
                return True
            return False

    def load(self, start: bool = False):

        if self.requires_extra:
            iface.messageBar().pushMessage(
                f"Installing {self.name} dependencies", level=Qgis.Info
            )
            # python -um pip install ...
            args = f"{INTERPRETER} -um pip install"
            pkgs, reqs = self.dependencies
            if pkgs:  # ... expand dependencies
                args += f" {' '.join(pkgs)}"
            if reqs:  # ... include frozen requirements
                args += " " + " ".join([f"--requirement {r}" for r in reqs])
            args += f" --prefix {self.prefix.path}"
            if not self.prefix:  # isolated plugins should ignore installed libraries
                "--ignore-installed"
            dlg_run(args, title=f"Installing {self.name} dependencies", icon=qicon())
            clear_cache()
            iface.messageBar().pushMessage(
                f"{self.name} dependencies are now available", level=Qgis.Success
            )
        loaded = loadPlugin(self.name)
        if loaded and start:
            started = startPlugin(self.name)
            if started:  # Sanitize plugin startup
                QgsSettings().setValue(f"/PythonPlugins/{self.name}", True)
                QgsSettings().remove(f"/PythonPlugins/watchDog/{self.name}")
        return loaded

    @classmethod
    def loadPlugin(cls, name: str, start: bool = False) -> bool:
        return cls(name).load(start)
