from __future__ import annotations

from functools import cached_property
from pathlib import Path

from pip._vendor.pkg_resources import WorkingSet
from qgis.PyQt.QtCore import QStringListModel, Qt
from qgis.PyQt.QtWidgets import QDialogButtonBox
from qgis.PyQt.uic import loadUiType

from ..settings import PluginSettings
from .icons import qicon

Class, BaseClass = loadUiType(Path(__file__).with_suffix(".ui"))


class DependenciesDialog(BaseClass, Class):

    def __init__(self, instance: PackageInstallerQgis):
        super().__init__()
        self.setupUi(self)
        self.instance = instance
        self.setWindowIcon(qicon())
        self.setWindowTitle(self.instance.name)

        # connect signals
        self.settings.valueChanged.connect(self.settingChanged)
        self.buttonBox.clicked.connect(self.buttonBoxCallback)
        self.prefixButton.clicked.connect(self.instance.move.trigger)

        self.update()

    @cached_property
    def settings(self):
        return PluginSettings()

    @property
    def prefix(self):
        return str(self.settings.get("prefix"))

    @prefix.setter
    def prefix(self, value: str):
        self.settings.set("prefix", str(Path(value).resolve()))

    def show(self):
        self.update()
        self.setEnabled(True)
        self.setWindowState(Qt.WindowActive)
        super().show()

    def update(self):
        model = self.dependenciesList.model() or QStringListModel()
        deps = [
            f"{d.project_name} ({d.version})"
            for d in WorkingSet()
            if Path(d.location).is_relative_to(self.prefix)
        ]
        self.dependenciesList.setModel(model)
        model.setStringList(deps)
        self.prefixLine.setText(self.prefix)
        super().update()

    def settingChanged(self, key: str, value: str):
        self.update()

    def buttonBoxCallback(self, btn: int):
        match self.buttonBox.standardButton(btn):
            case QDialogButtonBox.Reset:
                self.instance.purge.trigger()
            case QDialogButtonBox.RestoreDefaults:
                self.instance.restore.trigger()
            case QDialogButtonBox.Open:
                self.instance.reveal.trigger()
            case _:
                return
        self.update()
