from __future__ import annotations

import sys
from dataclasses import dataclass
from functools import cached_property
from pathlib import Path

from pip._internal.req.req_file import parse_requirements
from pip._vendor.pkg_resources import DistributionNotFound, VersionConflict, WorkingSet
from qgis.utils import plugin_paths, pluginMetadata

from .prefix import register_prefix, unregister_prefix


@dataclass
class PluginLoader:

    name: str

    def __enter__(self):
        if self.prefix:
            register_prefix(self.prefix)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.prefix:
            unregister_prefix(self.prefix)
        return

    @cached_property
    def prefix(self) -> Path:
        """Return the PIP prefix absolute path"""
        # find manifest's `general:pip_prefix` field
        pip_prefix = pluginMetadata(self.name, "pip_prefix")
        # not found defaults to "__error__" ...
        pip_prefix = pip_prefix.replace("__error__", "").replace("__default__", "")
        if pip_prefix:
            return (self.directory / pip_prefix).resolve()
        else:
            return None

    @cached_property
    def directory(self) -> Path:
        """Return the plugin root directory absolute path"""
        try:  # requires plugin to be loaded
            path = sys.modules[self.name].__path__
            assert len(path) == 1
            return Path(path[0]).resolve()
        except KeyError:  # loop through known paths
            for p in plugin_paths:
                if first := next(Path(p).glob(self.name), None):
                    return first.resolve()
            raise KeyError(self.name)

    @cached_property
    def dependencies(self) -> tuple[tuple[str]]:
        """Look for dependencies, both:
            - pip-like packages requirements
            - frozen environments requirements file(s)

        These dependencies are inferred from metadata's `pip_dependencies` field,
        with a default rollback to `requirements.txt` at plugin root directory.
        """

        pkgs, reqs = set(), set()

        # find manifest's `general:pip_dependencies` field
        pip_dependencies = pluginMetadata(self.name, "pip_dependencies")
        # not found defaults to "__error__" ...
        pip_dependencies = pip_dependencies.replace("__error__", "")
        # rollback to standard `requirements.txt` file
        if not pip_dependencies:
            requirements_txt = self.directory / "requirements.txt"
            if requirements_txt.is_file():
                reqs.add(requirements_txt.resolve().as_posix())

        # parse `pip_dependencies` (may be files !)
        dependencies = [el.strip() for el in pip_dependencies.split(",")]
        for dep in filter(None, dependencies):
            file = self.directory / dep
            if file.is_file():
                reqs.add(file.resolve().as_posix())
            else:
                pkgs.add(dep)

        return tuple(pkgs), tuple(reqs)

    @cached_property
    def has_dependencies(self) -> bool:
        return any(self.dependencies)

    @cached_property
    def needs_third_parties(self) -> bool:
        if not self.has_dependencies:
            return False
        pkgs, reqs = self.dependencies
        env = WorkingSet()
        try:
            for p in pkgs:
                env.require(p)
            for file in reqs:
                for p in [el.requirement for el in parse_requirements(file, False)]:
                    env.require(p)
        except (VersionConflict, DistributionNotFound):
            return True
        return False
