from __future__ import annotations

import os
import shutil
from pathlib import Path

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QDialog,
    QFileDialog,
    QGridLayout,
    QLabel,
    QMessageBox,
    QProgressDialog,
    QPushButton,
)

from .utils import folder_size, qicon


class PackageInstallerQgisDlg(QDialog):
    def __init__(self, instance: PackageInstallerQgis):

        super().__init__()

        self.setWindowModality(Qt.WindowModality.ApplicationModal)
        self.setModal(True)
        self.setWindowIcon(qicon("icon")),
        self.setWindowTitle(f"{instance.__class__.__name__}")

        self.instance = instance
        self.__init_components__()
        self.__init_layout__()

    def __init_components__(self):

        # prefix widgets
        self.prefix_label = QLabel("Default prefix: ")
        self.prefix_line = QLabel(str(self.instance.prefix_path))
        self.prefix_move = QPushButton(self.tr("Change default prefix ..."))
        self.prefix_move.clicked.connect(self.move_prefix)
        self.prefix_open = QPushButton(self.tr("Reveal in file explorer"))
        self.prefix_open.clicked.connect(
            lambda: os.startfile(self.instance.prefix_path)
        )
        self.prefix_purge = QPushButton(self.tr("Purge installed packages"))
        self.prefix_purge.clicked.connect(self.purge_prefix)

        # TODO: plugins widgets
        # TODO: dependencies widgets

    def __init_layout__(self) -> None:
        layout = QGridLayout()
        # layout.addWidget(widget: QWidget, fromRow: int, fromColumn: int, rowSpan: int, columnSpan: int, alignment: Qt.Alignment = Qt.Alignment())

        layout.addWidget(self.prefix_label, 0, 0)
        layout.addWidget(self.prefix_line, 0, 1, 1, -1)
        layout.addWidget(self.prefix_move, 1, 0)
        layout.addWidget(self.prefix_open, 1, 1)
        layout.addWidget(self.prefix_purge, 1, 2)

        self.setLayout(layout)

    def move_prefix(self) -> bool:
        directory = QFileDialog.getExistingDirectory(
            self,
            "Open Directory",
            str(self.prefix_path.parent),
            QFileDialog.Option.ShowDirsOnly,
        )

        if not directory:
            return False

        new, old = Path(directory).resolve(), self.instance.prefix_path.resolve()
        if old == new:
            return False

        choice = QMessageBox.warning(
            None,
            "Warning",
            "Are you sure you want to move all the plugin's dependencies ?\n"
            + f"There are {folder_size(old)} of python packages, this could take a while...",
            buttons=QMessageBox.StandardButtons(
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No
            ),
            defaultButton=QMessageBox.StandardButton.No,
        )

        if choice == QMessageBox.StandardButton.Yes:
            content = list(old.glob("*"))
            n = len(content)
            progress = QProgressDialog("Moving files...", "Abort Copy", 0, n)
            progress.setWindowModality(Qt.WindowModality.WindowModal)

            for i, chunk in enumerate(content):
                progress.setValue(i)
                if progress.wasCanceled():  # roll-back
                    [shutil.rmtree(chunk) for chunk in content]
                shutil.copy2(chunk, new)
            progress.close()
            self.prefix_line.setText(str(new))
            self.instance.prefix_path = new
            return True

        return False

    def purge_prefix(self) -> bool:
        prefix = self.instance.prefix_path

        choice = QMessageBox.warning(
            None,
            "Warning",
            "Are you sure you want to delete all the plugin's dependencies ?\n"
            + f"All python packages will be permanently be erased, this might break loaded plugins !",
            buttons=QMessageBox.StandardButtons(
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No
            ),
            defaultButton=QMessageBox.StandardButton.No,
        )

        if choice == QMessageBox.StandardButton.Yes:
            content = list(prefix.glob("*"))
            n, N = 0, folder_size(prefix, as_string=False)
            progress = QProgressDialog(f"Purging {prefix}...", None, n, N)
            progress.setWindowModality(Qt.WindowModality.WindowModal)

            for chunk in content:
                n += folder_size(prefix / chunk, as_string=False)
                shutil.rmtree(prefix / chunk)
                progress.setValue(n)
            progress.close()

            return True

        return False
