import os
import site
import sys
from pathlib import Path

from qgis.core import QgsApplication, QgsSettings
from qgis.utils import iface

DEFAULT_PREFIX = Path(QgsApplication.qgisSettingsDirPath()) / "python" / "dependencies"


def prefix_scheme(prefix: str = None) -> dict:
    """https://docs.python.org/3/library/sysconfig.html#sysconfig-prefix-scheme"""

    prefix = Path(prefix or site.getuserbase()).resolve()

    if os.name == "nt":
        return {
            "prefix": prefix,
            "stdlib": prefix / "Lib",
            "platstdlib": prefix / "Lib",
            "platlib": prefix / "Lib" / "site-packages",
            "purelib": prefix / "Lib" / "site-packages",
            "include": prefix / "Include",
            "platinclude": prefix / "Include",
            "scripts": prefix / "Scripts",
            "data": prefix,
        }
    else:
        py_ver = f"python{sys.version_info.major}.{sys.version_info.minor}"
        return {
            "prefix": prefix,
            "stdlib": prefix / "lib" / f"{py_ver}",
            "platstdlib": prefix / "lib" / f"{py_ver}",
            "platlib": prefix / "lib" / f"{py_ver}" / "site-packages",
            "purelib": prefix / "lib" / f"{py_ver}" / "site-packages",
            "include": prefix / "include" / f"{py_ver}",
            "platinclude": prefix / "include" / f"{py_ver}",
            "scripts": prefix / "bin",
            "data": prefix,
        }


def register_prefix(prefix: str) -> Path:

    scheme = prefix_scheme(prefix)
    lib, bin = str(scheme["platlib"]), str(scheme["scripts"])

    if lib not in sys.path:
        iface.messageBar().pushMessage(
            "PackageInstallerQgis", f"{lib} was added to PYTHONPATH", duration=5
        )
        os.environ["PYTHONPATH"] = lib + ";" + os.environ.get("PYTHONPATH", "")
        sys.path.insert(1, lib)

    if bin not in sys.path:
        iface.messageBar().pushMessage(
            "PackageInstallerQgis", f"{bin} was added to PATH", duration=5
        )
        sys.path.insert(1, bin)

    sys.path_importer_cache.clear()

    return scheme["prefix"]


def unregister_prefix(prefix: str) -> Path:

    scheme = prefix_scheme(prefix)
    lib, bin = str(scheme["platlib"]), str(scheme["scripts"])

    if lib in sys.path:
        iface.messageBar().pushMessage(
            "PackageInstallerQgis", f"{lib} was removed from PYTHONPATH", duration=5
        )
        os.environ["PYTHONPATH"] = (
            os.environ.get("PYTHONPATH", "").replace(lib, "").replace(";;", ";")
        )
        sys.path.remove(lib)

    if bin in sys.path:
        iface.messageBar().pushMessage(
            "PackageInstallerQgis", f"{bin} was removed from PATH", duration=5
        )
        sys.path.remove(bin)

    sys.path_importer_cache.clear()

    return scheme["prefix"]


if __name__ == "startup":
    prefix = QgsSettings().value("PackageInstallerQgis/prefix_path", DEFAULT_PREFIX)
    register_prefix(prefix)
