from __future__ import annotations

from collections import namedtuple
from functools import cached_property
from pathlib import Path
from shutil import rmtree

import qgis
from qgis.core import Qgis, QgsSettings
from qgis.PyQt.QtCore import QObject
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QWidget

from .loader import PluginLoader
from .plugin_dlg import PackageInstallerQgisDlg
from .prefix import DEFAULT_PREFIX, register_prefix, unregister_prefix
from .settings import PluginSettings
from .utils import log, qicon, warn
from .utils.cmd import INTERPRETER, dlg_run


# define plugin
class PackageInstallerQgis(QObject):
    """QGIS Plugin Implementation."""

    def __init__(self, iface):

        super().__init__()
        self.iface = iface
        # register prefix
        register_prefix(self.prefix_path)
        # patch qgis.utils.loadPlugin
        self._loadPlugin = qgis.utils.loadPlugin
        log("Patch `qgis.utils.loadPlugin`")
        qgis.utils.loadPlugin = self.loadPlugin
        # store previsouly loaded plugins
        self._defered_plugins = list(qgis.utils.plugins)
        if self.__class__.__name__ in self._defered_plugins:
            self._defered_plugins.remove(self.__class__.__name__, None)

        self.iface.initializationCompleted.connect(self.__post_init__)
        self.iface.messageBar().pushMessage(
            "PackageInstallerQgis loaded !",
            f"PackageInstallerQgis is ready to go !",
            level=Qgis.Success,
            duration=5,
        )

    def __post_init__(self):

        # reload deferred plugins
        for plugin in self._defered_plugins:
            self.unloadPlugin(plugin)
            self.loadPlugin(plugin, start=True)

    @property
    def settings(self):
        return PluginSettings()

    @property
    def name(self):
        return self.settings.get("name", self.__class__.__name__)

    @property
    def prefix_path(self) -> Path:
        return Path(
            QgsSettings().value("PackageInstallerQgis/prefix_path", DEFAULT_PREFIX)
        ).resolve()

    @prefix_path.setter
    def prefix_path(self, path: str):
        path = Path(path).resolve()
        if path != self.prefix_path:
            unregister_prefix(self.prefix_path)
            QgsSettings().setValue("PackageInstallerQgis/prefix_path", str(path))
            register_prefix(self.prefix_path)

    # required by QGIS at plugin start
    def initGui(self):

        # add action into plugins menu
        self.menu = self.iface.pluginMenu().addMenu(qicon(), self.name)
        self.menu.addAction(self.show)
        self.menu.addSeparator()
        self.menu.addAction(self.about)
        # add launcher into plugins toolbar
        self.iface.pluginToolBar().addAction(self.show)

    # required by QGIS at plugin stop
    def unload(self):
        # remove the menu entries
        self.iface.pluginMenu().removeAction(self.menu.menuAction())
        # remove launcher into plugins toolbar
        self.iface.pluginToolBar().removeAction(self.show)
        # restore original qgis.utils.loadPlugin
        log("Restore original `qgis.utils.loadPlugin`")
        qgis.utils.loadPlugin = self._loadPlugin

    def loadPlugin(self, plugin: str, start: bool = False) -> bool:

        if plugin not in qgis.utils.available_plugins:
            return False
            raise KeyError(plugin)

        # handle per-plugin isolation
        with PluginLoader(plugin) as loader:
            if loader.needs_third_parties:
                self.iface.messageBar().pushMessage(
                    f"Installing {plugin}",
                    f"PackageInstallerQgis is installing {plugin} dependencies",
                    level=Qgis.Info,
                    duration=5,
                )
                args = f"{INTERPRETER} -um pip install"
                pkgs, reqs = loader.dependencies
                if pkgs:
                    args += f" {' '.join(pkgs)}"
                if reqs:
                    args += " " + " ".join([f"--requirement {r}" for r in reqs])
                if loader.prefix:  # isolated plugins should ignore installed libraries
                    args += f" --prefix {loader.prefix} --ignore-installed"
                else:
                    args += f" --prefix {self.prefix_path}"
                returncode = dlg_run(
                    args,
                    parent=self.iface.mainWindow(),
                    title=f"installing {plugin} dependencies ...",
                    icon=qicon(),
                )
                if returncode != 0:
                    warn(f"Could not install {plugin} dependencies !")

            loaded = self._loadPlugin(plugin)
            if loaded and start:
                started = qgis.utils.startPlugin(plugin)
                if started:  # Do we really need this ?!
                    QgsSettings().setValue(f"/PythonPlugins/{plugin}", True)
                    QgsSettings().remove(f"/PythonPlugins/watchDog/{plugin}")

            return loaded

    def unloadPlugin(self, plugin: str):
        qgis.utils.unloadPlugin(plugin)

    def purgeDependencies(self) -> bool:
        if not self.prefix_path.is_dir():
            return False
        if (
            QMessageBox.critical(
                None,
                "Purge QGIS python dependencies",
                "Warning: This operation cannot be undone, "
                + "all PackageInstallerQGIS installed dependencies will be removed !\n"
                + "Are you sure you want to purge the QGIS's python dependencies ?",
            )
            == QMessageBox.StandardButton.Ok
        ):
            rmtree(self.prefix_path)
            return True
        return False

    def listPlugins(self) -> List[namedtuple]:
        Plugin = namedtuple(
            "Plugin", ["name", "path", "dependencies", "requirements", "isolation"]
        )
        plugins = []
        for p in qgis.utils.plugins:
            with PluginLoader(p) as l:
                if not l.needs_third_parties:
                    continue
                plugins.append(Plugin(l.plugin, l.directory, *l.dependencies, l.prefix))

    @cached_property
    def about(self) -> QAction:
        """Display plugin details"""

        def callback():
            dlg = QWidget(self.iface.mainWindow())
            dlg.setWindowIcon(qicon())

            content = f"<b>{self.name}</b><br>"
            content += f"<i>{self.settings.get('description')}</i><br><br>"
            content += f"{self.settings.get('about')}<br><br>"

            content += f"<b>Author</b> : <a href=mailto:{self.settings.get('email')}>{self.settings.get('author')}</a><br>"
            if homepage := self.settings.get("homepage", None):
                content += f"<b>Homepage</b> : <a href={homepage}>GitLab</a><br>"
            content += f"<b>Source code</b> : <a href={self.settings.get('repository')}>GitLab</a><br>"
            if tracker := self.settings.get("tracker", None):
                content += f"<b>Repport issue</b> : <a href={tracker}>GitLab</a><br>"
            if doc := self.settings.get("documentation", None):
                content += f"<b>Documentation</b> : <a href={doc}>GitLab</a><br>"

            content += f"<br><b>Version</b> : {self.settings.get('version')}<br>"
            content += f"<i><b>Requires</b> QGIS &#x02265; {self.settings.get('qgisMinimumVersion')}</i><br>"
            if plugins := self.settings.get("plugin_dependencies"):
                content += f"<i><b>Depends on</b> : {plugins.strip(',')}</i><br>"
            if deps := self.settings.get("pip_dependencies"):
                content += f"<i><b>Requirements</b> : {deps.strip(',')}</i><br>"

            QMessageBox.about(dlg, f"About", content)
            dlg.show()

        action = QAction(qicon("about"), f"About {self.name}", parent=self)
        action.triggered.connect(callback)
        return action

    @cached_property
    def show(self) -> QAction:
        def callback():
            PackageInstallerQgisDlg(self).open()

        action = QAction(icon=qicon("icon"), text="Package Installer QGIS", parent=self)
        action.triggered.connect(callback)
        return action


# make sure its loaded ASAP
qgis.utils.loadPlugin("PackageInstallerQgis")
