# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PRESTODialog
                                 A QGIS plugin
 PRESTO
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-06-26
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Unicampania
        email                : savino.giacobbe@unicampania.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QMessageBox
from qgis.PyQt.QtCore import QSettings

from itertools import product

from scipy.interpolate import make_interp_spline
import numpy as np

import pandas as pd

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'PRESTO_dialog_base.ui'))

def generate_scenarios(
    dialog_instance,
    energy_data,
    df_hub,
    res_df,
    hub_df,
    annual_prod_df,
    buildings_path,
    selected_hub_id
):
    def energy_class_order(val):
        order = ["G", "F", "E", "D", "C", "B", "A1", "A2", "A3", "A4"]
        return order.index(val) if val in order else len(order)

    df_buildings = pd.read_csv(buildings_path)

    if isinstance(selected_hub_id, str):
        if selected_hub_id.lower() == "all":
            selected_hub_ids = df_hub["HUB_ID"].unique()
        else:
            try:
                selected_hub_ids = [int(h.strip()) for h in selected_hub_id.split(",")]
            except ValueError:
                QMessageBox.warning(dialog_instance, "Error",
                                    "Invalid HUB_ID format. Use comma-separated numbers.")
                return
    else:
        selected_hub_ids = [selected_hub_id]

    scenarios = []
    scenario_id = 1
    
    for hub_id in selected_hub_ids:
        df_hub_selected = df_hub[df_hub["HUB_ID"] == hub_id]
        if df_hub_selected.empty:
            print(f"⚠️ HUB {hub_id} not found.")
            continue

        selected_cabin = df_hub_selected["Cabin_ID"].values[0]
        filtered_apartments = df_buildings[df_buildings["Cabin_ID"] == selected_cabin]

        if filtered_apartments.empty:
            print("⚠️ No apartments found with the same Cabin_ID as the selected HUB.")
            return

        # Estrai i dati energetici specifici per HUB
        hub_energy = hub_df[hub_df["HUB_ID"] == hub_id]
        prod_energy = annual_prod_df[annual_prod_df["HUB_ID"] == hub_id]

        if hub_energy.empty or prod_energy.empty:
            print("⚠️ Energy data for the HUB not found.")
            return

        hub_energy_vals = hub_energy.iloc[0, 1:].values
        prod_energy_vals = prod_energy.iloc[0, 1:].values

        self_consumption = []
        input_energy = []
        for he, pe in zip(hub_energy_vals, prod_energy_vals):
            if pe == 0:
                self_consumption.append(0)
                input_energy.append(0)
            elif he >= pe:
                self_consumption.append(pe)
                input_energy.append(0)
            else:
                self_consumption.append(he)
                input_energy.append(pe - he)

        # DIZIONARIO SCENARI CONFIGURATI
        scenario_profiles = {
            "Social Scenario": {
                "order": [
                    ("Family Income", True),
                    ("Family Density (100 sqm)", False),
                    ("Yearly Energy Demand", True),
                    ("Average Family Age", False),
                    ("Energy Class", None)
                ],
                "thresholds": {}
            },
            "Technical Scenario": {
                "order": [
                    ("Energy Class", "best_efficiency"),
                    ("Yearly Energy Demand", False),
                    ("Family Density (100 sqm)", False),
                    ("Family Income", None)
                ],
                "thresholds": {
                    "Yearly Energy Demand": ("<=", 7000)
                }
            },
            "Youth Scenario": {
                "order": [
                    ("Average Family Age", True),
                    ("Family Income", True),
                    ("Energy Class", "best_efficiency")
                ],
                "thresholds": {
                    "Average Family Age": ("<", 50)
                }
            },
            "Fairness Scenario": {
                "order": [
                    ("Family Income", True),
                    ("Yearly Energy Demand", "mid"),
                    ("Energy Class", "best_efficiency")
                ],
                "thresholds": {}
            },
            "Elderly Scenario": {
                "order": [
                    ("Average Family Age", False),
                    ("Family Income", True),
                    ("Energy Class", "best_efficiency")
                ],
                "thresholds": {
                    "Average Family Age": (">=", 50)
                }
            }
        }

        # ITERAZIONE SCENARI DEFINITI
        alpha = 0.9
        while alpha >= 0.2:
            for scenario_label, profile in scenario_profiles.items():
                sorted_apts = filtered_apartments.copy()

                # Applica soglie
                for field, (op, value) in profile.get("thresholds", {}).items():
                    if op == "<=":
                        sorted_apts = sorted_apts[sorted_apts[field] <= value]
                    elif op == ">=":
                        sorted_apts = sorted_apts[sorted_apts[field] >= value]
                    elif op == "<":
                        sorted_apts = sorted_apts[sorted_apts[field] < value]
                    elif op == ">":
                        sorted_apts = sorted_apts[sorted_apts[field] > value]

                # Gestione ordinamento
                sort_fields = []
                sort_orders = []

                for field, order in profile["order"]:
                    if order is None:
                        continue
                    elif order == "best_efficiency":
                        sort_fields.append(field)
                        sort_orders.append(True)  # A4 prima
                        sorted_apts[field] = sorted_apts[field].map(energy_class_order)
                    elif order == "mid":
                        media = sorted_apts[field].mean()
                        sort_fields.append(f"|{field}-media|")
                        sorted_apts[f"|{field}-media|"] = (sorted_apts[field] - media).abs()
                        sort_orders.append(True)
                    else:
                        sort_fields.append(field)
                        sort_orders.append(order)

                sorted_apts = sorted_apts.sort_values(
                    by=sort_fields,
                    ascending=sort_orders,
                    na_position="last"
                )

                # Il resto del codice (included_apts = [], self_consumption, ecc.) resta invariato

                included_apts = []
                shared_energy_used = [0] * len(prod_energy_vals)

                for _, apt in sorted_apts.iterrows():
                    apt_energy_row = res_df[res_df["Apartment_ID"] == apt["Apartment_ID"]]
                    if apt_energy_row.empty:
                        continue

                    apt_vals = apt_energy_row.iloc[0, 1:].values

                    exceeds = False

                    for i in range(len(shared_energy_used)):
                        if input_energy[i] == 0:
                            continue  # salta confronto se non c'è energia in quella fascia oraria

                        tolleranza = 1.5  # fino al 15% in più consentito
                        if shared_energy_used[i] + apt_vals[i] > tolleranza * alpha * input_energy[i]:
                            exceeds = True
                            break

                    if not exceeds:
                        shared_energy_used = [shared_energy_used[i] + apt_vals[i] for i in range(len(shared_energy_used))]
                        included_apts.append(apt)

                if not included_apts:
                    continue

                sci_num = 0
                sci_den = 0

                for i in range(len(prod_energy_vals)):
                    if prod_energy_vals[i] == 0:
                        continue

                    if self_consumption[i] >= prod_energy_vals[i]:
                        sci_num += prod_energy_vals[i]
                    else:
                        available_for_sharing = prod_energy_vals[i] - self_consumption[i]
                        sci_num += self_consumption[i] + min(shared_energy_used[i], available_for_sharing)

                    sci_den += prod_energy_vals[i]

                sci = ((sci_num / sci_den) * 100) if sci_den > 0 else 0

                total_yearly = sum(a["Yearly Energy Demand"] for _, a in enumerate(included_apts))
                ssi = sci_num / total_yearly * 100 if total_yearly > 0 else 0

                if abs(sci - ssi) <= 45:
                    scenario = {
                        "Scenario_ID": f"Scenario {scenario_id}",
                        "Produced_Energy (kWh/y)": sci_den,
                        "HUB_SelfConsumptionEnergy (kWh/y)": sum(self_consumption),
                        "Energy_Input (kWh/y)": sum(input_energy),
                        "Residential_TotalEnergyConsumption (kWh/y)": sum(shared_energy_used),
                        "SCI (%)": sci,
                        "SSI (%)": ssi,
                        "Family Number": len(included_apts)
                    }

                    scenario["Scenario_Type"] = scenario_label
                    
                    # Salva anche i dati orari
                    scenario["Hourly_SelfConsumption"] = self_consumption
                    scenario["Hourly_SharedEnergy"] = shared_energy_used
                    scenario["Hourly_Production"] = list(prod_energy_vals)

                    # Dati richiesti
                    energia_consumata = scenario["HUB_SelfConsumptionEnergy (kWh/y)"] + scenario[
                        "Residential_TotalEnergyConsumption (kWh/y)"]
                    energia_prod = scenario["Produced_Energy (kWh/y)"]
                    energia_autoconsumata = scenario["HUB_SelfConsumptionEnergy (kWh/y)"]
                    energia_condivisa = scenario["Residential_TotalEnergyConsumption (kWh/y)"]

                    # Indicatori
                    try:
                        co2_index = (1 - (((0.276 * (energia_consumata - (energia_autoconsumata + energia_condivisa))) + (0.05 * energia_prod)) / (0.276 * energia_consumata))) * 100
                    except ZeroDivisionError:
                        co2_index = 0

                    co2_storage = 0.430 * energia_prod / 1000
                    km_auto_evitati = -1000 * co2_storage / 0.125
                    numero_alberi = energia_prod / 0.2
                    barili_petrolio = energia_prod * 1000 * 0.000589

                    # Aggiunta al dizionario dello scenario
                    scenario["CO2_Index (%)"] = co2_index
                    scenario["CO2_Storage (t/y)"] = co2_storage
                    scenario["CarKmAvoided (Km/y)"] = km_auto_evitati
                    scenario["PlantedTree"] = numero_alberi
                    scenario["OilBarrelsAvoided (n/y)"] = barili_petrolio

                    scenarios.append((scenario, included_apts))
                    scenario_id += 1

                    # 🧮 Parametri economici
                    try:
                        pv_surface = sum(item["PV_Surface_sqm"] for item in energy_data if item["HUB_ID"] == hub_id)
                    except StopIteration:
                        pv_surface = 0  # default se non trovato

                    try:
                        costo_pv_mq = float(dialog_instance.PVCost_lineEdit.text())
                    except:
                        costo_pv_mq = 0

                    try:
                        costo_manutenzione_mq = float(dialog_instance.MaintenanceCost_lineEdit.text())
                    except:
                        costo_manutenzione_mq = 0

                    try:
                        simulation_years = int(dialog_instance.SimulationTime_lineEdit.text())
                    except:
                        simulation_years = 20  # default

                    # Variabili base
                    costo_pv_totale = pv_surface * costo_pv_mq
                    manutenzione_annua = pv_surface * costo_manutenzione_mq
                    ricavo_annuo = 0.12 * energia_condivisa
                    beneficio_co2 = 71 * co2_storage

                    # Calcolo VAN e B/C
                    van = 0
                    payback = 0
                    van_final = 0
                    bc_final = 0

                    van_values = []

                    for i in range(simulation_years + 1):
                        benefici = (ricavo_annuo + beneficio_co2) * i

                        manutenzione_effettiva = manutenzione_annua * (i // 5)
                        costi = costo_pv_totale + manutenzione_effettiva

                        attualizzazione = (1 + 0.03) ** i

                        van_i = (benefici - costi) / attualizzazione
                        van_values.append(van_i)

                        van_final = van_i

                        if i > 0:
                            beneficio_att = abs(benefici / attualizzazione)
                            costo_att = abs(costi / attualizzazione)
                            if costo_att > 0:
                                bc_final = beneficio_att / costo_att

                        if van <= 0 and van_i > 0:
                            payback = i

                        van = van_i

                    # Salva indicatori nel dizionario scenario
                    scenario["PV_Cost (€)"] = costo_pv_totale
                    scenario["VAN (€)"] = van_final
                    scenario["B/C"] = bc_final
                    scenario["Payback (y)"] = payback
                    scenario["VAN_values"] = van_values

            alpha -= 0.05

    import matplotlib.pyplot as plt
    import os

    # Cartella di salvataggio
    save_folder = QtWidgets.QFileDialog.getExistingDirectory(
        dialog_instance,
        "Select a folder to save the hourly charts"
    )

    if save_folder:
        for scenario, _ in scenarios:
            scenario_id = scenario["Scenario_ID"]
            hours = np.arange(24)

            # Dati
            self_consumption = scenario["Hourly_SelfConsumption"]
            shared_energy_used = scenario["Hourly_SharedEnergy"]
            prod_energy_vals = scenario["Hourly_Production"]

            # Smooth production curve
            hours_smooth = np.linspace(0, 23, 200)
            spline = make_interp_spline(hours, prod_energy_vals, k=3)
            prod_smooth = spline(hours_smooth)

            # Plot
            plt.figure(figsize=(10, 6))
            bar_width = 0.3  # più sottile

            # Colori personalizzati (RGB normalizzati 0–1)
            colore_autoconsumo = (237 / 255, 125 / 255, 49 / 255)
            colore_consumo = (165 / 255, 165 / 255, 165 / 255)

            # Istogrammi e linea

            # Prima le barre arancioni (base)
            plt.bar(hours, self_consumption, width=bar_width, color=colore_autoconsumo,
                    label='Self-Consumed Energy')

            # Poi le barre grigie sopra (impilate)
            plt.bar(hours, shared_energy_used, width=bar_width, color=colore_consumo,
                    label='Energy Consumed by Users', bottom=self_consumption)

            plt.plot(hours_smooth, prod_smooth, color='blue', linewidth=2, label='Produced Energy')

            plt.xlabel("Hours of the Day")
            plt.ylabel("Energy (kWh)")
            plt.title(f"{scenario_id} - Hourly Profile")
            plt.legend()
            plt.grid(True)
            plt.tight_layout()


            file_path = os.path.join(save_folder, f"{scenario_id.replace(' ', '_')}_hourly_profile.png")
            plt.savefig(file_path)
            plt.close()

            # Grafico VAN anno per anno (solo se VAN_values presenti)
            if "VAN_values" in scenario:
                van_values = scenario["VAN_values"]
                anni = list(range(len(van_values)))

                plt.figure(figsize=(10, 4))
                bar_width = 0.3  # stesso spessore dei grafici orari
                plt.bar(anni, van_values, width=bar_width, color=(0.2, 0.4, 0.6), label="VAN (€)")

                plt.xlabel("Year")
                plt.ylabel("VAN (€)")
                plt.title(f"{scenario_id} - Annual VAN")
                plt.legend()
                plt.grid(True)
                plt.tight_layout()

                file_path_van = os.path.join(save_folder, f"{scenario_id.replace(' ', '_')}_profilo_VAN.png")
                plt.savefig(file_path_van)
                plt.close()

    # Popola la tabella nella GUI (se esiste)
    if hasattr(dialog_instance, "Scenarios_tableWidget"):
        table = dialog_instance.Scenarios_tableWidget
        table.setRowCount(0)
        table.setColumnCount(8)
        table.setHorizontalHeaderLabels([
            "Scenario_ID",
            "Produced_Energy (kWh/y)",
            "HUB_SelfConsumptionEnergy (kWh/y)",
            "Energy_Input (kWh/y)",
            "Residential_TotalEnergyConsumption (kWh/y)",
            "SCI (%)",
            "SSI (%)",
            "Family Number",
            "CO2_Index (%)",
            "CO2_Storage (t/y)",
            "CarKmAvoided (Km/y)",
            "PlantedTree",
            "OilBarrelsAvoided (n/y)",
            "PV_Cost (€)",
            "VAN (€)",
            "B/C",
            "Payback (y)"
        ])

        for row_idx, (scenario, _) in enumerate(scenarios):
            table.insertRow(row_idx)
            for col_idx, key in enumerate([
                "Scenario_ID",
                "Produced_Energy (kWh/y)",
                "HUB_SelfConsumptionEnergy (kWh/y)",
                "Energy_Input (kWh/y)",
                "Residential_TotalEnergyConsumption (kWh/y)",
                "SCI (%)",
                "SSI (%)",
                "Family Number",
                "CO2_Index (%)",
                "CO2_Storage (t/y)",
                "CarKmAvoided (Km/y)",
                "PlantedTree",
                "OilBarrelsAvoided (n/y)",
                "PV_Cost (€)",
                "VAN (€)",
                "B/C",
                "Payback (y)"
            ]):
                item = QtWidgets.QTableWidgetItem(str(round(scenario[key], 2)) if isinstance(scenario[key], float) else str(scenario[key]))
                table.setItem(row_idx, col_idx, item)

        table.resizeColumnsToContents()

        # Esporta in CSV su richiesta dell’utente
        save_path, _ = QtWidgets.QFileDialog.getSaveFileName(
            dialog_instance,
            "Save scenarios to CSV",
            "",
            "CSV files (*.csv);;All files (*)"
        )


        if save_path:
            try:
                df_scenarios = pd.DataFrame([s for s, _ in scenarios])
                df_scenarios.to_csv(save_path, index=False)
                QMessageBox.information(dialog_instance, "Export completed", f"Scenarios saved to:\n{save_path}")
            except Exception as e:
                QMessageBox.critical(dialog_instance, "Error", f"Error while saving the CSV:\n{e}")


        # Esporta CSV dettagliato per appartamenti
        all_apartments = []

        for scenario, included_apts in scenarios:
            scenario_id = scenario["Scenario_ID"]
            for apt in included_apts:
                apartment_id = apt["Apartment_ID"]
                res_energy_row = res_df[res_df["Apartment_ID"] == apartment_id]
                res_total_energy = res_energy_row.iloc[0, 1:].sum() if not res_energy_row.empty else 0
                apt_info = apt.to_dict()
                apt_info["Scenario_ID"] = scenario_id
                apt_info["Residential_EnergyConsumption (kWh/y)"] = res_total_energy
                all_apartments.append(apt_info)

        if all_apartments:
            df_apts = pd.DataFrame(all_apartments)

            # Sort by Scenario_ID
            df_apts.sort_values(by="Scenario_ID", inplace=True)

            # Export CSV
            save_path, _ = QtWidgets.QFileDialog.getSaveFileName(
                dialog_instance,
                "Save apartment data by scenario",
                "",
                "CSV files (*.csv);;All files (*)"
            )

            if save_path:
                try:
                    df_apts.to_csv(save_path, index=False)
                    QMessageBox.information(
                        dialog_instance,
                        "Export completed",
                        f"Apartment data saved to:\n{save_path}"
                    )
                except Exception as e:
                    QMessageBox.critical(
                        dialog_instance,
                        "Error",
                        f"Error while saving the detailed CSV:\n{e}"
                    )


        # Terzo file: riepilogo aggregato per edificio e scenario
        if all_apartments:
            df_apts = pd.DataFrame(all_apartments)

            # Rimuovi le colonne non richieste
            df_apts.drop(columns=[
                "Energy Class",
                "Family Density (100 sqm)",
                "Density Category",
                "Family Description",
                "Family Income",
                "Apartment_Surface_sqm",
                "Yearly Energy Demand",
                "Apartment_ID"
            ], inplace=True, errors="ignore")

            # Campi da sommare
            sum_fields = [
                "People_Number",
                "People Number (0 - 10)",
                "People Number (11 - 18)",
                "People Number (19 - 34)",
                "People Number (35 - 49)",
                "People Number (50 - 69)",
                "People Number (70+)",
                "Residential_EnergyConsumption (kWh/y)"
            ]

            # Campo da mediare
            mean_fields = ["Average Family Age"]

            # Campi costanti (da prendere dal primo record del gruppo)
            constant_fields = [
                "ID", "Street_Name", "Dettagli Edificio", "Propriety",
                "Building_State", "Cabin_ID", "Lot_ID", "X_m", "Y_m"
            ]

            grouped = df_apts.groupby(["Scenario_ID", "Building_ID"])
            records = []

            for (scenario_id, building_id), group in grouped:
                record = {
                    "Scenario_ID": scenario_id,
                    "Building_ID": building_id,
                    "Apartment_Number": len(group)
                }

                # Somme
                for field in sum_fields:
                    record[field] = group[field].sum()

                # Medie
                for field in mean_fields:
                    record[field] = group[field].mean()

                # Campi costanti (prendi il primo valore del gruppo)
                for field in constant_fields:
                    record[field] = group[field].iloc[0]

                records.append(record)

            df_building_summary = pd.DataFrame(records)

            # Ordina per Scenario_ID e Building_ID
            df_building_summary.sort_values(by=["Scenario_ID", "Building_ID"], inplace=True)

            # Esporta il CSV
            save_path, _ = QtWidgets.QFileDialog.getSaveFileName(
                dialog_instance,
                "Save building summary by scenario",
                "",
                "CSV files (*.csv);;All files (*)"
            )


            if save_path:
                try:
                    df_building_summary.to_csv(save_path, index=False)
                    QMessageBox.information(
                        dialog_instance,
                        "Export completed",
                        f"Building summary saved to:\n{save_path}"
                    )
                except Exception as e:
                    QMessageBox.critical(
                        dialog_instance,
                        "Error",
                        f"Error while saving the third CSV:\n{e}"
                    )


        return scenarios

class PRESTODialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(PRESTODialog, self).__init__(parent)
        self.setupUi(self)

        # Percorso al CSV nella cartella del plugin
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        csv_path = os.path.join(plugin_dir, 'Data', 'GHI_Italy.csv')

        # Carica il CSV e costruisci il dizionario
        try:
            df = pd.read_csv(csv_path)
            self.italy_dict = self.build_admin_dict(df)
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error loading the CSV:\n{e}")
            return

        # Popola le regioni
        self.Regions_comboBox.addItems(sorted(self.italy_dict.keys()))
        self.Province_comboBox.setEnabled(False)
        self.Municipality_comboBox.setEnabled(False)

        # Collega i segnali
        self.Regions_comboBox.currentTextChanged.connect(self.update_provinces)
        self.Province_comboBox.currentTextChanged.connect(self.update_municipalities)

        self.Run_pushButton.clicked.connect(self.on_run_clicked)

        self.load_gui_state()
        self.Save_pushButton.clicked.connect(self.save_gui_state)


    def save_gui_state(self):
        settings = QSettings()

        settings.setValue("BuildingsPath", self.Buildings_mQgsFileWidget.filePath())
        settings.setValue("HUBPath", self.HUB_mQgsFileWidget.filePath())
        settings.setValue("HubConsumptionPath", self.HubConsumption_mQgsFileWidget.filePath())
        settings.setValue("ConsumptionYear", self.ConsumptionYear_lineEdit.text())
        settings.setValue("SimulationTime", self.SimulationTime_lineEdit.text())
        settings.setValue("AvailableBudget", self.AvailableBudget_lineEdit.text())
        settings.setValue("HubID", self.HubID_lineEdit.text())
        settings.setValue("PVCoverage", self.PVCoverage_lineEdit.text())
        settings.setValue("Region", self.Regions_comboBox.currentText())
        settings.setValue("Province", self.Province_comboBox.currentText())
        settings.setValue("Municipality", self.Municipality_comboBox.currentText())

        QMessageBox.information(self, "State saved", "The parameters have been successfully saved.")

    def load_gui_state(self):
        settings = QSettings()

        self.Buildings_mQgsFileWidget.setFilePath(settings.value("BuildingsPath", ""))
        self.HUB_mQgsFileWidget.setFilePath(settings.value("HUBPath", ""))
        self.HubConsumption_mQgsFileWidget.setFilePath(settings.value("HubConsumptionPath", ""))
        self.ConsumptionYear_lineEdit.setText(settings.value("ConsumptionYear", ""))
        self.SimulationTime_lineEdit.setText(settings.value("SimulationTime", ""))
        self.AvailableBudget_lineEdit.setText(settings.value("AvailableBudget", ""))
        self.HubID_lineEdit.setText(settings.value("HubID", ""))
        self.PVCoverage_lineEdit.setText(settings.value("PVCoverage", ""))

        region = settings.value("Region", "")
        province = settings.value("Province", "")
        municipality = settings.value("Municipality", "")

        if region and region in self.italy_dict:
            self.Regions_comboBox.setCurrentText(region)
            self.update_provinces(region)
            if province and province in self.italy_dict[region]:
                self.Province_comboBox.setCurrentText(province)
                self.update_municipalities(province)
                if municipality and municipality in self.italy_dict[region][province]:
                    self.Municipality_comboBox.setCurrentText(municipality)

    def build_admin_dict(self, df):
        structure = {}
        grouped = df.groupby(['Region', 'Province'])
        for (region, province), group in grouped:
            comuni = sorted(group['Municipality'].unique())
            if region not in structure:
                structure[region] = {}
            structure[region][province] = comuni
        # Ordina regioni e province
        return {
            region: {
                prov: structure[region][prov]
                for prov in sorted(structure[region])
            }
            for region in sorted(structure)
        }

    def update_provinces(self, selected_region):
        self.Province_comboBox.clear()
        self.Municipality_comboBox.clear()
        self.Municipality_comboBox.setEnabled(False)

        if selected_region in self.italy_dict:
            provinces = sorted(self.italy_dict[selected_region].keys())
            self.Province_comboBox.addItems(provinces)
            self.Province_comboBox.setEnabled(True)
        else:
            self.Province_comboBox.setEnabled(False)

    def update_municipalities(self, selected_province):
        selected_region = self.Regions_comboBox.currentText()
        self.Municipality_comboBox.clear()

        if selected_region in self.italy_dict and selected_province in self.italy_dict[selected_region]:
            comuni = self.italy_dict[selected_region][selected_province]
            self.Municipality_comboBox.addItems(comuni)
            self.Municipality_comboBox.setEnabled(True)
        else:
            self.Municipality_comboBox.setEnabled(False)

    def on_run_clicked(self):
        # 1. Ottieni i path dei file caricati
        buildings_path = self.Buildings_mQgsFileWidget.filePath()
        hub_path = self.HUB_mQgsFileWidget.filePath()

        # 2. Ottieni i valori inseriti dall'utente
        sim_time = self.SimulationTime_lineEdit.text()
        budget = self.AvailableBudget_lineEdit.text()
        hub_id = self.HubID_lineEdit.text().strip() or "All"
        pv_coverage = self.PVCoverage_lineEdit.text()
        selected_municipality = self.Municipality_comboBox.currentText()

        # Percorso al file GHI
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        ghi_path = os.path.join(plugin_dir, 'Data', 'GHI_Italy.csv')

        result = self.calculate_energy_production(
            hub_path, ghi_path, hub_id, pv_coverage, selected_municipality
        )

        if result:
            energy_data, df_hub = result

            output_lines = []
            header = f"{'HUB_ID'.ljust(10)}{'Name'.ljust(40)}{'Energy_Production (kWh/year)'.ljust(30)}{'PV_Surface (sqm)'.ljust(20)}"

            output_lines.append(header)
            output_lines.append("-" * len(header))

            for item in energy_data:
                hub_id = str(item["HUB_ID"]).ljust(10)
                name_row = df_hub[df_hub["HUB_ID"] == item["HUB_ID"]]
                name = name_row["Name"].values[0] if not name_row.empty else "N/A"
                name = name[:38].ljust(40)
                energy = f"{round(item['EnergyProduction_kWh'], 2):,.2f}".rjust(25)
                pv_surface = f"{round(item['PV_Surface_sqm'], 2):,.2f}".rjust(20)

                # Aggiungi la riga formattata all'output
                output_lines.append(f"{hub_id}{name}{energy}{pv_surface}")


            self.HubsEnergyProduction_textEdit.clear()
            self.HubsEnergyProduction_textEdit.setPlainText("\n".join(output_lines))
        
        res_df = self.calculate_residential_energy_consumption(
            self.Buildings_mQgsFileWidget.filePath(),
            os.path.join(plugin_dir, "Data", "profili GSE_prelievo.csv")
        )

        hub_df = self.calculate_hub_energy_consumption(
            self.HubConsumption_mQgsFileWidget.filePath(),
            os.path.join(plugin_dir, "Data", "profiliHUB.csv"),
            self.ConsumptionYear_lineEdit.text()
        )

        # Calcolo AnnualHUBEnergyProduction
        annual_prod_df = self.calculate_annual_hub_energy_production(
            energy_data,
            os.path.join(plugin_dir, "Data", "profilo_produzione.csv")
        )

        generate_scenarios(
            self,
            energy_data,
            df_hub,
            res_df,
            hub_df,
            annual_prod_df,
            self.Buildings_mQgsFileWidget.filePath(),
            self.HubID_lineEdit.text().strip() or "All"
        )

    def calculate_energy_production(self, hub_path, ghi_path, hub_id_input, pv_coverage, selected_municipality):
        # 1. Carica i CSV
        try:
            df_hub = pd.read_csv(hub_path)
            df_ghi = pd.read_csv(ghi_path)
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error loading CSV files:\n{e}")
            return None


        # 2. Filtra i GHI per comune selezionato
        ghi_row = df_ghi[df_ghi["Municipality"] == selected_municipality]
        if ghi_row.empty:
            QMessageBox.warning(self, "Warning", f"No radiation data available for the municipality '{selected_municipality}'")
            return None
        radiation = ghi_row["Radiation (kWh/smq)"].values[0]

        # 3. Interpreta gli hub_id
        if hub_id_input.lower() == "all":
            selected_hubs = df_hub
        else:
            try:
                ids = [int(h.strip()) for h in hub_id_input.split(',')]
                selected_hubs = df_hub[df_hub["HUB_ID"].isin(ids)]
            except ValueError:
                QMessageBox.warning(self, "Error", "Invalid HUB ID format. Use comma-separated numbers.")
                return None

        if selected_hubs.empty:
            QMessageBox.warning(self, "Warning", "No HUBs found with the specified IDs.")
            return None

        # 4. Calcola EnergyProduction
        YIELD_MAP = {"Flat": 0.12, "Pitched": 0.16}
        results = []
        for _, row in selected_hubs.iterrows():
            yield_value = YIELD_MAP.get(row["Roof_Type"], 0)
            surface = row["Covered_Surface_sqm"]

            pv_surface = (surface * float(pv_coverage)) / 100
            energy = (pv_surface * float(radiation) * yield_value)

            results.append({
                "HUB_ID": row["HUB_ID"],
                "Covered_Surface_sqm": surface,
                "Roof_Type": row["Roof_Type"],
                "Radiation": radiation,
                "Yield": yield_value,
                "PV_Surface_sqm": pv_surface,
                "EnergyProduction_kWh": energy
            })

        return results, df_hub
    
    def calculate_residential_energy_consumption(self, buildings_path, gse_profile_path):
        df_buildings = pd.read_csv(buildings_path)
        df_profile = pd.read_csv(gse_profile_path)

        # Crea intestazioni E0, E1, ..., E23
        columns = [f"E{i}" for i in range(len(df_profile))]
        result = pd.DataFrame(columns=["Apartment_ID"] + columns)

        for _, row in df_buildings.iterrows():
            energy = row["Yearly Energy Demand"]
            hourly = (df_profile["somma normalizzata"] * energy / 100).tolist()
            result_row = [row["Apartment_ID"]] + hourly
            result.loc[len(result)] = result_row

        return result

    def calculate_hub_energy_consumption(self, hub_consumption_path, hub_profile_path, year_filter):
        # Load CSV files
        df_consumption = pd.read_csv(hub_consumption_path)
        df_profile = pd.read_csv(hub_profile_path)

        # --- Rename Italian columns to English ---
        column_mapping = {
            "Inizio Periodo Consumi": "Start Consumption Period",
            "Fine Periodo Consumi": "End Consumption Period",
            "Consumo Kwh": "Energy Consumption (kWh)",
            "HUB_ID": "HUB_ID"
        }

        # Rename only the existing columns
        df_consumption.rename(
            columns={k: v for k, v in column_mapping.items() if k in df_consumption.columns},
            inplace=True
        )

        # --- Convert dates (using the newly renamed English names) ---
        df_consumption["Start Consumption Period"] = pd.to_datetime(
            df_consumption["Start Consumption Period"],
            dayfirst=False,
            format="%m/%d/%Y",
            errors="coerce"
        )
        df_consumption["End Consumption Period"] = pd.to_datetime(
            df_consumption["End Consumption Period"],
            dayfirst=False,
            format="%m/%d/%Y",
            errors="coerce"
        )

        # --- Filter by year ---
        df_consumption = df_consumption[
            df_consumption["Start Consumption Period"].dt.year == int(year_filter)
        ]

        # --- Sum energy consumption by HUB_ID ---
        hub_totals = df_consumption.groupby("HUB_ID")["Energy Consumption (kWh)"].sum().reset_index()

        # --- Create H0, H1, ..., H23 labels ---
        columns = [f"H{i}" for i in range(len(df_profile))]
        result = pd.DataFrame(columns=["HUB_ID"] + columns)

        # --- Expand annual consumption into hourly using the profile ---
        for _, row in hub_totals.iterrows():
            hub_id = row["HUB_ID"]
            annual_kwh = row["Energy Consumption (kWh)"]
            hourly = (df_profile["%Consumi"] * annual_kwh).tolist()

            result_row = [hub_id] + hourly
            result.loc[len(result)] = result_row

        return result

    
    def calculate_annual_hub_energy_production(self, energy_data, production_profile_path):
        # Carica il profilo di produzione
        try:
            df_profile = pd.read_csv(production_profile_path)
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error loading the production profile:\n{e}")
            return None

        # Intestazioni: P0, P1, ..., P23
        columns = [f"P{i}" for i in range(len(df_profile))]
        result = pd.DataFrame(columns=["HUB_ID"] + columns)

        # Raggruppa EnergyProduction_kWh per HUB_ID
        from collections import defaultdict
        totals = defaultdict(float)
        for item in energy_data:
            hub_id = item["HUB_ID"]
            totals[hub_id] += item["EnergyProduction_kWh"]

        # Costruisci la tabella
        for hub_id, total_production in totals.items():
            hourly_values = (df_profile["%production"] * total_production).tolist()
            row = [hub_id] + hourly_values
            result.loc[len(result)] = row

        return result