#!/usr/bin/python3
# # -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : MapBiomas Alert
Description          : Class for work with MapBiomas Alert
Date                 : April, 2019
copyright            : (C) 2019 by Luiz Motta
email                : motta.luiz@gmail.com

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import json, os

from qgis.PyQt.QtCore import (
    Qt, QEvent,
    QObject, pyqtSlot, pyqtSignal,
    QUrl,
    QDate,
    QRegularExpression,
    QSortFilterProxyModel
)
from qgis.PyQt.QtWidgets import (
    QWidget, QPushButton,
    QLabel, QDateEdit, QSpinBox, QSpacerItem, QSizePolicy,
    QVBoxLayout, QHBoxLayout,
    QApplication, # widgets = QApplication.instance().allWidgets()
    QStyle, QLineEdit, QComboBox, QStackedWidget,
    QCompleter
)
from qgis.PyQt.QtGui import (
    QColor, QPixmap, QIcon,
    QDesktopServices, # QDesktopServices.openUrl( QUrl( url ) )
    QRegularExpressionValidator
)
from qgis.PyQt.QtNetwork import QNetworkRequest
 
from qgis.core import (
    Qgis, QgsApplication, QgsProject,
    QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsVectorLayer, QgsFeature,
    QgsBlockingNetworkRequest,QgsTask, QgsSymbol,
    QgsFillSymbol, QgsSingleSymbolRenderer
)
from qgis.gui import QgsGui, QgsMessageBar, QgsLayerTreeEmbeddedWidgetProvider
from qgis import utils as QgsUtils
from qgis.core import QgsSettings

from .mapbiomasalert_layer_api import DbAlerts, API_MapbiomasAlert, TerritoryBbox
from .form import setForm as FORM_setForm
from .dialog_email_password import DialogEmailPassword, exec_dialog

# PyQt5/PyQt6 compatibility
try:
    # PyQt6 style
    from qgis.PyQt.QtCore import Qt
    ISO_DATE = Qt.DateFormat.ISODate
    STANDARD_PIXMAP_APPLY = QStyle.StandardPixmap.SP_DialogApplyButton
    STANDARD_PIXMAP_CANCEL = QStyle.StandardPixmap.SP_DialogCancelButton
    STRONG_FOCUS = Qt.FocusPolicy.StrongFocus
    CASE_INSENSITIVE = Qt.CaseSensitivity.CaseInsensitive
    UNFILTERED_POPUP_COMPLETION = QCompleter.CompletionMode.UnfilteredPopupCompletion
    KEY_PRESS = QEvent.Type.KeyPress
except AttributeError:
    # PyQt5 style (fallback)
    ISO_DATE = Qt.ISODate
    STANDARD_PIXMAP_APPLY = QStyle.SP_DialogApplyButton
    STANDARD_PIXMAP_CANCEL = QStyle.SP_DialogCancelButton
    STRONG_FOCUS = Qt.StrongFocus
    CASE_INSENSITIVE = Qt.CaseInsensitive
    UNFILTERED_POPUP_COMPLETION = QCompleter.UnfilteredPopupCompletion
    KEY_PRESS = QEvent.KeyPress


class ExtendedComboBox(QComboBox):
    """
    A QComboBox with text filtering capability using a completer.
    """
    def __init__(self, parent=None):
        super(ExtendedComboBox, self).__init__(parent)

        self.setFocusPolicy(STRONG_FOCUS)
        self.setEditable(True)

        # add a filter model to filter matching items
        self.pFilterModel = QSortFilterProxyModel(self)
        self.pFilterModel.setFilterCaseSensitivity(CASE_INSENSITIVE)
        self.pFilterModel.setSourceModel(self.model())

        # add a completer, which uses the filter model
        self.completer = QCompleter(self.pFilterModel, self)
        # always show all (filtered) completions
        self.completer.setCompletionMode(UNFILTERED_POPUP_COMPLETION)
        self.setCompleter(self.completer)

        # connect signals
        self.lineEdit().textEdited[str].connect(self.pFilterModel.setFilterFixedString)
        self.completer.activated.connect(self.on_completer_activated)
        
        # Connect text editing to control popup behavior
        self.lineEdit().textEdited.connect(self.on_text_edited)
        
        # Ensure line edit gets focus when combo box is clicked
        self.lineEdit().setFocus()

    def showPopup(self):
        """Override to ensure line edit is focused when dropdown opens"""
        super().showPopup()
        # Select all text so typing will replace it
        self.lineEdit().setFocus()
        self.lineEdit().selectAll()
        # Install event filter on the popup view to capture key events
        self.view().installEventFilter(self)

    def eventFilter(self, obj, event):
        """Filter events to redirect typing to line edit when popup is open"""
        if obj == self.view() and event.type() == KEY_PRESS:
            # If it's a printable character, send it to the line edit
            if event.text() and event.text().isprintable():
                self.lineEdit().setFocus()
                # Send the key event to the line edit
                self.lineEdit().event(event)
                return True  # Event handled
        return super().eventFilter(obj, event)

    def on_text_edited(self, text):
        """Handle text editing to manage popup visibility"""
        if text:
            # When user is typing, hide the combo box popup and show completer
            self.hidePopup()
            if self.completer.completionCount() > 0:
                self.completer.complete()
        else:
            # When text is empty, we can show the normal popup
            if not self.view().isVisible():
                self.showPopup()

    # on selection of an item from the completer, select the corresponding item from combobox
    def on_completer_activated(self, text):
        if text:
            index = self.findText(text)
            self.setCurrentIndex(index)
            # Use currentTextChanged signal instead for better compatibility
            self.currentTextChanged.emit(self.itemText(index))

    # on model change, update the models of the filter and completer as well
    def setModel(self, model):
        super(ExtendedComboBox, self).setModel(model)
        self.pFilterModel.setSourceModel(model)
        self.completer.setModel(self.pFilterModel)

    # on model column change, update the model column of the filter and completer as well
    def setModelColumn(self, column):
        self.completer.setCompletionColumn(column)
        self.pFilterModel.setFilterKeyColumn(column)
        super(ExtendedComboBox, self).setModelColumn(column)


class MapBiomasAlertWidget(QWidget):

    def __init__(self, layer, layerTerritory, api, localSetting):
        super().__init__()  # Make sure parent initialization happens first
        
        # Store these as instance variables before setting up UI
        self.layer = layer
        self.canvas = QgsUtils.iface.mapCanvas()
        self.project = QgsProject.instance()
        self.crsCatalog = QgsCoordinateReferenceSystem('EPSG:4674')
        self.msgBar = QgsUtils.iface.messageBar()
        self.api = api
        self.alert = DbAlerts(layer, self.api)
        self.layerTerritory = layerTerritory
        self.localSetting = localSetting
        self.clearAfterSearch = False
        
        # Get dates from layer properties if they exist
        from_date = layer.customProperty('mapbiomasalert/from_date')
        to_date = layer.customProperty('mapbiomasalert/to_date')
        
        if from_date and to_date:
            self.last_from_date = QDate.fromString(from_date, ISO_DATE)
            self.last_to_date = QDate.fromString(to_date, ISO_DATE)
        else:
            self.last_from_date = None
            self.last_to_date = None
        
        # Create main layout first
        self.mainLayout = QVBoxLayout()
        self.setLayout(self.mainLayout)
        
        # Setup UI components
        self.setupUI()
        
        # Restore saved filters if they exist
        saved_territory_type = layer.customProperty('mapbiomasalert/territory_type', "Bioma")
        saved_biome = layer.customProperty('mapbiomasalert/biome')
        saved_state = layer.customProperty('mapbiomasalert/state')
        saved_city = layer.customProperty('mapbiomasalert/city')
        
        # Restore territory type first
        index = self.territoryTypeCombo.findText(saved_territory_type)
        if index >= 0:
            self.territoryTypeCombo.setCurrentIndex(index)
        
        # Restore other filters
        if saved_biome:
            index = self.biomeCombo.findText(saved_biome)
            if index >= 0:
                self.biomeCombo.setCurrentIndex(index)
        
        if saved_state:
            index = self.stateCombo.findText(saved_state)
            if index >= 0:
                self.stateCombo.setCurrentIndex(index)
                # State change will trigger city update via _onStateChanged
        
        if saved_city:
            index = self.cityCombo.findText(saved_city)
            if index >= 0:
                self.cityCombo.setCurrentIndex(index)
        
        # Connect signals after UI is set up
        def messageHandler(msg, level):
            self.msgBar.pushMessage(msg, level)
            if level == Qgis.Info:
                self.processingLabel.setText(msg)
            elif level == Qgis.Success:
                self.processingLabel.setText("")
            elif level == Qgis.Warning:
                self.processingLabel.setText(f"Warning: {msg}")
            elif level == Qgis.Critical:
                self.processingLabel.setText(f"Error: {msg}")

        def statusHandler(msg):
            self.processingLabel.setText(msg)
            
        self.api.message.connect(messageHandler)
        self.api.connectAlerts(self.alert)
        self.api.finishedAlert.connect(self._onFinishedAlert)
        self.api.status.connect(statusHandler)
        
        # Initialize dates
        self.initializeDates()
        
        # Initialize filter summary
        self.updateFilterSummary()

    def setupUI(self):
        """Setup all UI components"""
        self.icons = self.getIcons()
        self.textSearch = {'apply': 'Buscar', 'cancel': 'Cancelar'}
        
        # Get territory options from API
        territory_options = self.api.getTerritoriesOptions()
        if not territory_options:
            # Fallback to hardcoded lists if API fails
            territory_options = {
                'biomes': ["Todos os Biomas", "Amazônia", "Caatinga", "Cerrado", "Mata Atlânticaa", "Pampa", "Pantanal"],
                'states': ["Todos os Estados", "ACRE", "ALAGOAS", "AMAPÁ", "AMAZONAS", "BAHIA", 
                          "CEARÁ", "DISTRITO FEDERAL", "ESPÍRITO SANTO", "GOIÁS", "MARANHÃO",
                          "MATO GROSSO", "MATO GROSSO DO SUL", "MINAS GERAIS", "PARÁ", "PARAÍBA",
                          "PARANÁ", "PERNAMBUCO", "PIAUÍ", "RIO DE JANEIRO", "RIO GRANDE DO NORTE",
                          "RIO GRANDE DO SUL", "RONDÔNIA", "RORAIMA", "SANTA CATARINA", "SÃO PAULO",
                          "SERGIPE", "TOCANTINS"],
                'cities': ["Todas as Cidades"]  # Empty fallback for cities
            }
        
        # Create search layout
        searchLayout = self.createSearchLayout(territory_options)
        self.mainLayout.addLayout(searchLayout)
        
        # Add simple status text
        self.status = QLabel(self)
        self.mainLayout.addWidget(self.status)

    def createSearchLayout(self, territory_options):
        """Create the search part of the UI"""
        layout = QVBoxLayout()
        layout.setSpacing(8)
        
        # Store territory options for later use
        self.territory_options = territory_options
        
        # Main header with filter status
        headerLayout = QVBoxLayout()
        filterLabel = QLabel("Filtrar alertas por:", self)
        filterLabel.setStyleSheet("font-weight: bold; font-size: 11pt;")
        headerLayout.addWidget(filterLabel)

        # Active filters summary
        self.filterSummary = QLabel("Nenhum filtro ativo", self)
        self.filterSummary.setStyleSheet("color: #666; font-size: 9pt; font-style: italic;")
        headerLayout.addWidget(self.filterSummary)
        layout.addLayout(headerLayout)
        
        # Quick search section
        quickSearchGroup = self.createGroupBox("Busca Rápida")
        quickSearchLayout = QHBoxLayout()
        quickSearchLayout.setSpacing(10)
        
        self.alertCodeInput = QLineEdit(self)
        self.alertCodeInput.setPlaceholderText("Digite o código do alerta (ex. 12345)")
        self.alertCodeInput.setValidator(QRegularExpressionValidator(QRegularExpression("^[0-9]*$")))
        self.alertCodeInput.setMaximumWidth(200)
        self.alertCodeInput.textChanged.connect(self.updateFilterSummary)
        
        quickSearchLayout.addWidget(QLabel("Código do Alerta:", self))
        quickSearchLayout.addWidget(self.alertCodeInput)
        quickSearchLayout.addStretch()
        quickSearchGroup.setLayout(quickSearchLayout)
        layout.addWidget(quickSearchGroup)
        
        # Geographic filters section
        geoGroup = self.createGroupBox("Filtros Geográficos")
        geoLayout = QVBoxLayout()
        
        # Territory type and value on same line
        territoryLayout = QHBoxLayout()
        territoryLayout.setSpacing(10)
        
        territoryLayout.addWidget(QLabel("Filtrar por:", self))
        self.territoryTypeCombo = ExtendedComboBox(self)
        self.territoryTypeCombo.addItems(["Bioma", "Estado", "Cidade"])
        self.territoryTypeCombo.currentTextChanged.connect(self._onTerritoryTypeChanged)
        self.territoryTypeCombo.setMaximumWidth(120)
        territoryLayout.addWidget(self.territoryTypeCombo)
        
        # Create territory filter widgets in a compact stacked layout
        self.territoryStack = QStackedWidget(self)
        self.territoryStack.setMaximumWidth(250)
        
        # Biome widget
        self.biomeCombo = ExtendedComboBox(self)
        self.biomeCombo.addItems(territory_options['biomes'])
        self.biomeCombo.currentTextChanged.connect(self.updateFilterSummary)
        self.territoryStack.addWidget(self.biomeCombo)
        
        # State widget
        self.stateCombo = ExtendedComboBox(self)
        self.stateCombo.addItems(territory_options['states'])
        self.stateCombo.currentTextChanged.connect(self._onStateChanged)
        self.stateCombo.currentTextChanged.connect(self.updateFilterSummary)
        self.territoryStack.addWidget(self.stateCombo)
        
        # City widget
        self.cityCombo = ExtendedComboBox(self)
        self.cityCombo.addItems(territory_options.get('cities', ["Todas as Cidades"]))
        self.cityCombo.currentTextChanged.connect(self.updateFilterSummary)
        self.territoryStack.addWidget(self.cityCombo)
        
        territoryLayout.addWidget(self.territoryStack)
        territoryLayout.addStretch()
        
        geoLayout.addLayout(territoryLayout)
        geoGroup.setLayout(geoLayout)
        layout.addWidget(geoGroup)
        
        # Store the mapping between territory types and stack indices
        self.territory_indices = {
            "Bioma": 0,
            "Estado": 1,
            "Cidade": 2
        }
        
        # Date filters section
        dateGroup = self.createGroupBox("Filtro de Datas")
        dateLayout = QVBoxLayout()
        
        # Date type selector
        dateTypeLayout = QHBoxLayout()
        dateTypeLayout.addWidget(QLabel("Tipo de data:", self))
        self.dateTypeCombo = ExtendedComboBox(self)
        self.dateTypeCombo.addItems(["Data de Detecção", "Data de Publicação"])
        self.dateTypeCombo.setMaximumWidth(150)
        self.dateTypeCombo.currentTextChanged.connect(self.updateFilterSummary)
        dateTypeLayout.addWidget(self.dateTypeCombo)
        dateTypeLayout.addStretch()
        dateLayout.addLayout(dateTypeLayout)
        
        # Date range - more compact
        dateRangeLayout = QHBoxLayout()
        dateRangeLayout.setSpacing(8)
        
        dateRangeLayout.addWidget(QLabel("De:", self))
        self.fromDate = QDateEdit(self)
        self.fromDate.setCalendarPopup(True)
        self.fromDate.setDisplayFormat('yyyy-MM-dd')
        self.fromDate.setMaximumWidth(120)
        self.fromDate.dateChanged.connect(self.updateFilterSummary)
        dateRangeLayout.addWidget(self.fromDate)
        
        dateRangeLayout.addWidget(QLabel("até", self))
        self.toDate = QDateEdit(self)
        self.toDate.setCalendarPopup(True)
        self.toDate.setDisplayFormat('yyyy-MM-dd')
        self.toDate.setMaximumWidth(120)
        self.toDate.dateChanged.connect(self.updateFilterSummary)
        dateRangeLayout.addWidget(self.toDate)
        
        dateRangeLayout.addStretch()
        dateLayout.addLayout(dateRangeLayout)
        dateGroup.setLayout(dateLayout)
        layout.addWidget(dateGroup)
        
        # Search action section
        actionLayout = QHBoxLayout()
        actionLayout.setSpacing(10)
        
        self.search = QPushButton(self.textSearch['apply'], self)
        self.search.setIcon(self.icons['apply'])
        self.search.clicked.connect(self._onSearch)
        self.search.setMinimumHeight(32)
        actionLayout.addWidget(self.search)
        
        # Clear filters button
        self.clearFilters = QPushButton("Limpar Filtros", self)
        self.clearFilters.clicked.connect(self.clearAllFilters)
        self.clearFilters.setMaximumWidth(100)
        actionLayout.addWidget(self.clearFilters)
        
        actionLayout.addStretch()
        layout.addLayout(actionLayout)
        
        return layout

    def createGroupBox(self, title):
        """Create a styled group box for visual organization"""
        from qgis.PyQt.QtWidgets import QGroupBox
        groupBox = QGroupBox(title, self)
        groupBox.setStyleSheet("""
            QGroupBox {
                font-weight: bold;
                border: 1px solid #ccc;
                border-radius: 4px;
                margin-top: 8px;
                padding-top: 8px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                left: 8px;
                padding: 0 4px 0 4px;
            }
        """)
        return groupBox

    def updateFilterSummary(self):
        """Update the filter summary display"""
        filters = []
        
        # Alert code
        if self.alertCodeInput.text().strip():
            filters.append(f"Alerta: {self.alertCodeInput.text()}")
        
        # Territory filter
        territory_type = self.territoryTypeCombo.currentText()
        if territory_type == "Bioma" and self.biomeCombo.currentText() != "Todos os Biomas":
            filters.append(f"Bioma: {self.biomeCombo.currentText()}")
        elif territory_type == "Estado" and self.stateCombo.currentText() != "Todos os Estados":
            filters.append(f"Estado: {self.stateCombo.currentText()}")
        elif territory_type == "Cidade" and self.cityCombo.currentText() != "Todas as Cidades":
            filters.append(f"Cidade: {self.cityCombo.currentText()}")
        
        # Date filter
        date_type = self.dateTypeCombo.currentText()
        from_date = self.fromDate.date().toString('MMM dd')
        to_date = self.toDate.date().toString('MMM dd, yyyy')
        filters.append(f"{date_type}: {from_date} - {to_date}")
        
        if filters:
            self.filterSummary.setText(" • ".join(filters))
        else:
            self.filterSummary.setText("Nenhum filtro ativo")

    def clearAllFilters(self):
        """Clear all filter selections"""
        self.alertCodeInput.clear()
        self.territoryTypeCombo.setCurrentText("Bioma")
        self.biomeCombo.setCurrentText("Todos os Biomas")
        self.stateCombo.setCurrentText("Todos os Estados")
        self.cityCombo.setCurrentText("Todas as Cidades")
        self.dateTypeCombo.setCurrentText("Data de Detecção")
        self.updateFilterSummary()

    def getIcons(self):
        fIcon = self.style().standardIcon
        return {
            'apply': fIcon(STANDARD_PIXMAP_APPLY),
            'cancel': fIcon(STANDARD_PIXMAP_CANCEL)
        }

    def createDateEdit(self, name, layout, displayFormat, hasCalendar):
        layout.addWidget( QLabel( name ) )
        w = QDateEdit( self )
        w.setCalendarPopup( True )
        w.setDisplayFormat( displayFormat )
        w.setCalendarPopup( hasCalendar )
        layout.addWidget( w )
        return w

    def initializeDates(self):
        """Initialize the date range"""
        def changedFromDate(date):
            self.toDate.setMinimumDate(date.addDays(+1))

        def changedToDate(date):
            self.fromDate.setMaximumDate(date.addDays(-1))

        # Use stored dates if they exist, otherwise use defaults
        if self.last_from_date and self.last_to_date:
            self.fromDate.setDate(self.last_from_date)
            self.toDate.setDate(self.last_to_date)
        else:
            d2 = QDate.currentDate()
            d1 = d2.addMonths(-1)
            self.fromDate.setDate(d1)
            self.fromDate.setMaximumDate(d2.addDays(-1))
            self.toDate.setDate(d2)
            self.toDate.setMinimumDate(d1.addDays(+1))
        
        # Connect signals
        self.fromDate.dateChanged.connect(changedFromDate)
        self.toDate.dateChanged.connect(changedToDate)

    @pyqtSlot(bool)
    def _onSearch(self, checked):
        # Get the selected territory type and its value
        territory_type = self.territoryTypeCombo.currentText()
        selected_biome = None
        selected_state = None
        selected_city = None
        
        if territory_type == "Bioma":
            selected_biome = self.biomeCombo.currentText()
        elif territory_type == "Estado":
            selected_state = self.stateCombo.currentText()
        elif territory_type == "Cidade":
            selected_city = self.cityCombo.currentText()
        
        # Store dates and filters in layer properties before search
        self.layer.setCustomProperty('mapbiomasalert/from_date', 
                                   self.fromDate.date().toString(ISO_DATE))
        self.layer.setCustomProperty('mapbiomasalert/to_date', 
                                   self.toDate.date().toString(ISO_DATE))
        self.layer.setCustomProperty('mapbiomasalert/territory_type',
                                   territory_type)
        if selected_biome:
            self.layer.setCustomProperty('mapbiomasalert/biome', selected_biome)
        if selected_state:
            self.layer.setCustomProperty('mapbiomasalert/state', selected_state)
        if selected_city:
            self.layer.setCustomProperty('mapbiomasalert/city', selected_city)
        
        if not self.api.tokenOk:
            dlg = DialogEmailPassword("Login MapBiomas Alert", False)
            if not exec_dialog(dlg):
                return
                
            params = dlg.getParams()
            if not dlg.isValidEmail():
                self.msgBar.pushMessage("Formato de email inválido", Qgis.Critical)
                self.status.setText("Erro: Formato de email inválido")
                return
                
            self.api.setToken(params['email'], params['password'])
            if not self.api.tokenOk:
                return
                
            if not dlg.isCheckedSave():
                self.clearAfterSearch = True

        # Continue with search...
        if self.api.taskAlerts:
            self.api.cancelAlerts()
            self.search.setIcon(self.icons['apply'])
            self.status.setText("Busca cancelada")
            return

        # Show processing message
        processing_msg = "Processando sua solicitação. Isso pode demorar um pouco..."
        self.status.setText(processing_msg)
        self.msgBar.pushMessage(
            "MapBiomas Alert", 
            processing_msg,
            Qgis.Info,
            duration=0  # Message stays until explicitly cleared
        )
        QApplication.processEvents()
        
        self.search.setIcon(self.icons['cancel'])
        
        # Get alert code if provided
        alert_code = self.alertCodeInput.text().strip()
        
        # Get date type
        use_published_date = self.dateTypeCombo.currentText() == "Data de Publicação"
        
        if alert_code:
            status_msg = f'Obtendo alerta {alert_code}...'
        else:
            filters = []
            if selected_biome:
                filters.append(f"bioma '{selected_biome}'")
            if selected_state:
                filters.append(f"estado '{selected_state}'")
            if selected_city:
                filters.append(f"cidade '{selected_city}'")
            date_type = "publicação" if use_published_date else "detecção"
            filters.append(f"data de {date_type}")
            status_msg = f'Obtendo alertas por {" e ".join(filters)}...'
            
        self.status.setText(status_msg)
        
        # Set up the layer
        self.alert.setLayer(self.fromDate.date().toString(ISO_DATE), self.toDate.date().toString(ISO_DATE))
        ids = self.layerTerritory.getIdsCanvas()
        
        # Get the URL with optional alert code and date type
        url = self.api.getUrlAlerts(
            self.fromDate.date().toString(ISO_DATE), 
            self.toDate.date().toString(ISO_DATE),
            alert_code if alert_code else None,
            use_published_date,
            selected_biome,
            selected_state,
            selected_city
        )
        print('url', url)
        
        # Fetch the alerts
        result = self.api.getAlertsWFSnonThread(url, self.alert, self.fromDate.date().toString(ISO_DATE), self.toDate.date().toString(ISO_DATE), ids)
        
        # Clear the processing message and show appropriate message
        self.msgBar.clearWidgets()
        
        if result['total'] == 0:
            if alert_code:
                error_msg = f"Código de alerta {alert_code} não encontrado"
                self.status.setText(error_msg)
                self.msgBar.pushMessage("MapBiomas Alert", error_msg, Qgis.Warning)
            else:
                date_type = "publicação" if use_published_date else "detecção"
                warning_msg = f"Nenhum alerta encontrado para o período de {date_type} especificado"
                self.status.setText(warning_msg)
                self.msgBar.pushMessage("MapBiomas Alert", warning_msg, Qgis.Warning)
        else:
            success_msg = f"Encontrados {result['total']} alerta{'s' if result['total'] > 1 else ''}"
            self.status.setText(success_msg)
            self.msgBar.pushMessage("MapBiomas Alert", success_msg, Qgis.Success)

    @pyqtSlot()
    def _onFinishedAlert(self):
        self.search.setIcon(self.icons['apply'])
        self.status.setText("")  # Clear status message when finished
        
        if hasattr(self, 'clearAfterSearch') and self.clearAfterSearch:
            # Only restore dates if we're clearing temporary credentials
            if self.last_from_date and self.last_to_date:
                self.fromDate.setDate(self.last_from_date)
                self.toDate.setDate(self.last_to_date)
            self.api.clearToken()
            delattr(self, 'clearAfterSearch')
            
        # Check if layer has features and zoom to them
        if self.layer and self.layer.featureCount() > 0:
            canvas = QgsUtils.iface.mapCanvas()
            extent = self.layer.extent()
            extent.scale(1.1)
            canvas.setExtent(extent)
            canvas.refresh()

    def __del__(self):
        """Clean up when widget is destroyed"""
        if hasattr(self, 'layer') and self.layer:
            self.api.disconnectAlerts(self.layer.id())

    def _onStateChanged(self, state):
        """Update city combo box when state changes"""
        self.cityCombo.clear()
        if state == "Todos os Estados":
            # Show all cities when no state is selected
            self.cityCombo.addItems(self.territory_options.get('cities', ["Todas as Cidades"]))
        else:
            # Filter cities for the selected state
            cities = ["Todas as Cidades"]
            for city in self.territory_options.get('cities', [])[1:]:  # Skip "Todas as Cidades"
                if city.startswith(state):  # Simple filtering, adjust based on your data format
                    cities.append(city)
            self.cityCombo.addItems(cities)

    def _onTerritoryTypeChanged(self, territory_type):
        """Switch to the selected territory filter"""
        # Check if territory_type is valid before accessing dictionary
        if territory_type and territory_type in self.territory_indices:
            self.territoryStack.setCurrentIndex(self.territory_indices[territory_type])
            # Store the selected territory type
            self.layer.setCustomProperty('mapbiomasalert/territory_type', territory_type)


class LayerMapBiomasAlertWidgetProvider(QgsLayerTreeEmbeddedWidgetProvider):
    def __init__(self, api, localSetting):
        super().__init__()
        self.layerTerritory = TerritoryBbox()
        self.layerTerritory.setLayer()
        self.api = api
        self.localSetting = localSetting

    def id(self):
        return self.__class__.__name__

    def name(self):
        return "Layer MapBiomas Alert"

    def createWidget(self, layer, widgetIndex):
        return MapBiomasAlertWidget(layer, self.layerTerritory, self.api, self.localSetting)

    def supportsLayer(self, layer):
        return bool( layer.customProperty( MapBiomasAlert.MODULE, 0) )


class MapBiomasAlert(QObject):
    MODULE = 'MapBiomasAlert'
    def __init__(self, iface, api):
        super().__init__()        
        # Enable Python macros for the session
        QgsSettings().setEnumValue("qgis/enableMacros", Qgis.PythonMacroMode.SessionOnly)
        # Support for versions 3.40.7 and above
        QgsSettings().setEnumValue("qgis/enablePythonEmbedded", Qgis.PythonMacroMode.SessionOnly)
        
        self.project = QgsProject.instance()
        self.msgBar = iface.messageBar()
        self.widgetProvider = None
        self.layer = None
        self.canvas = iface.mapCanvas()
        self.styleFile = os.path.join(os.path.dirname(__file__), 'mapbiomas_alert.qml')
        self.api = api
        self.localSetting = 'mapbiomas_alert_plugin/{}'
        self._connected_alerts = False

    def register(self):
        self.widgetProvider = LayerMapBiomasAlertWidgetProvider(self.api, self.localSetting)
        registry = QgsGui.layerTreeEmbeddedWidgetRegistry()
        if bool( registry.provider( self.widgetProvider.id() ) ):
            registry.removeProvider( self.widgetProvider.id() )
        registry.addProvider( self.widgetProvider )

    def addLayerRegisterProperty(self, layer):
        totalEW = int( layer.customProperty('embeddedWidgets/count', 0) )
        layer.setCustomProperty('embeddedWidgets/count', totalEW + 1 )
        layer.setCustomProperty(f"embeddedWidgets/{totalEW}/id", self.widgetProvider.id() )
        layer.setCustomProperty( self.MODULE, 1)

        # Comment out or remove QML loading if this programmatic style is preferred
        # layer.loadNamedStyle( self.styleFile )

        # Apply programmatic styling
        # Create a fill symbol for red outline, no fill
        symbol = QgsFillSymbol.createSimple({
            'color': 'transparent',
            'style': 'no',
            'outline_style': 'solid',
            'outline_width': '0.5',
            'outline_width_unit': 'MM',
            'outline_color': 'red'
        })

        # Create a single symbol renderer with this symbol
        renderer = QgsSingleSymbolRenderer(symbol)

        # Apply the renderer to the layer
        layer.setRenderer(renderer)
        
        FORM_setForm( layer )
        self.project.addMapLayer( layer )

    def run(self):
        print("\n[Main] Starting MapBiomas Alert plugin...")
        if not self.api.tokenOk:
            params = DialogEmailPassword.getConfig(self.localSetting)
            
            if params['email'] is not None:
                print("[Main] Found saved credentials, attempting authentication")
                self.api.setToken(params['email'], params['password'])
            
            if not self.api.tokenOk:
                print("[Main] No valid token, showing login dialog")
                dlg = DialogEmailPassword("Login MapBiomas Alert", False)
                if not exec_dialog(dlg):
                    return
                
                params = dlg.getParams()
                if not dlg.isValidEmail():
                    self.msgBar.pushMessage("Formato de email inválido", Qgis.Critical)
                    return
                
                self.api.setToken(params['email'], params['password'])
                if not self.api.tokenOk:
                    return
                
                if not dlg.isCheckedSave():
                    self.clearTokenAfterSetup = True
        
        print("[Main] Adding alerts to layer")
        layer = DbAlerts.createLayer()
        self.addLayerRegisterProperty(layer)
        
        if hasattr(self, 'clearTokenAfterSetup') and self.clearTokenAfterSetup:
            print("[Main] Clearing temporary token after setup")
            self.api.clearToken()
            delattr(self, 'clearTokenAfterSetup')

    def actionsForm(self, nameAction, feature_id=None):
        """
        Run action defined in layer, provide by style file
        :param nameAction: Name of action
        :params feature_id: Feature ID
        """
        # Actions functions
        def flash(feature_id):
            geom = self.alert.getFeature( feature_id ).geometry()
            self.mapCanvasGeom.flash( [ geom ], self.alert )
            return { 'isOk': True }

        def zoom(feature_id):
            geom = self.alert.getFeature( feature_id ).geometry()
            self.mapCanvasGeom.zoom( [ geom ], self.alert )
            return { 'isOk': True }

        def report(feature_id):
            feat = self.alert.getFeature(feature_id)
            alerta_id = feat['alertCode']
            car_codes = feat['carCode']
            if len(car_codes) == 0:
                url = "{}/{}".format(API_MapbiomasAlert.urlReport, alerta_id)
                QDesktopServices.openUrl(QUrl(url))
            else:
                for car_code in car_codes.split(','):
                    url = "{}/{}/car/{}".format(API_MapbiomasAlert.urlReport, alerta_id, car_code)
                    QDesktopServices.openUrl(QUrl(url))
            return { 'isOk': True }

        actionsFunc = {
            'flash':  flash,
            'zoom':   zoom,
            'report': report
        }
        if not nameAction in actionsFunc.keys():
            return { 'isOk': False, 'message': "Missing action '{}'".format( nameAction ) }
        return actionsFunc[ nameAction ]( feature_id )

    def connectAlerts(self, dbAlerts):
        if not self._connected_alerts:
            print("[API] Connecting alerts signals")
            self.alerts.connect(dbAlerts.addFeatures)
            self._connected_alerts = True
