# -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : MapBiomas Alert Oficial
Description          : MapBiomas Alert Oficial
Date                 : January, 2023
copyright            : (C) 2019 by Luiz Motta (Deprecated), (C) 2022 by Luiz Cortinhas
email                : motta.luiz@gmail.com, luiz.cortinhas@solved.eco.br

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Luiz Motta, Luiz Cortinhas'
__date__ = '2023-01-20'
__copyright__ = '(C) 2019, Luiz Motta, (C) 2023, Luiz Cortinhas'
__revision__ = '$Format:%H$'


import os

from qgis.PyQt.QtCore import QObject, Qt, pyqtSlot
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QToolButton, QMenu

from qgis.core import Qgis, QgsApplication, QgsSettings

from .mapbiomasalert import MapBiomasAlert
from .dialog_email_password import DialogEmailPassword, runDialogEmailPassword, InfoDialog, exec_dialog
from .mapbiomasalert_layer_api import API_MapbiomasAlert

# PyQt5/PyQt6 compatibility
try:
    # PyQt6 style
    MENU_BUTTON_POPUP = QToolButton.ToolButtonPopupMode.MenuButtonPopup
except AttributeError:
    # PyQt5 style (fallback)
    MENU_BUTTON_POPUP = QToolButton.MenuButtonPopup

def classFactory(iface):
  return MBAlertPlugin( iface )

class MBAlertPlugin(QObject):
    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.name = 'MapBiomas Alert'
        self.api = API_MapbiomasAlert()
        self.api.message.connect(self.iface.messageBar().pushMessage)
        self.mbalert = MapBiomasAlert(self.iface, self.api)
        self.localSetting = 'mapbiomas_alert_plugin/{}'
        
        self.toolButton = QToolButton()
        self.toolButton.setMenu( QMenu() )
        self.toolButton.setPopupMode(MENU_BUTTON_POPUP)
        self.toolBtnAction = iface.addToolBarWidget( self.toolButton )
        self.action, self.actionAlert, self.actionConfig, self.actionClear, self.actionInfo = None, None, None, None, None
        self.nameActionAlert = 'MapBiomas Alert'

    def initGui(self):
        # Alert
        about = 'Obter alertas do MapBiomas'
        icon = QIcon( os.path.join( os.path.dirname(__file__), 'mapbiomas_alert.png' ) )
        self.actionAlert = QAction( icon, self.nameActionAlert, self.iface.mainWindow() )
        self.actionAlert.setObjectName( self.nameActionAlert.replace(' ', '') )
        self.actionAlert.setWhatsThis( about )
        self.actionAlert.setStatusTip( about )
        self.actionAlert.triggered.connect( self.runAlert )
        self.iface.addPluginToMenu( f"&{self.nameActionAlert}" , self.actionAlert )
        # Config
        title = 'Configurar email/senha'
        icon = QgsApplication.getThemeIcon('mActionOptions.svg')
        self.actionConfig = QAction( icon, title, self.iface.mainWindow() )
        self.actionConfig.setToolTip( title )
        self.actionConfig.triggered.connect( self.runConfig )
        self.iface.addPluginToMenu( f"&{self.nameActionAlert}" , self.actionConfig )
        # Clear Credentials
        title = 'Limpar credenciais'
        icon = QgsApplication.getThemeIcon('mActionDeleteSelected.svg')
        self.actionClear = QAction(icon, title, self.iface.mainWindow())
        self.actionClear.setToolTip(title)
        self.actionClear.triggered.connect(self.clearCredentials)
        self.iface.addPluginToMenu(f"&{self.nameActionAlert}", self.actionClear)
        # Info
        title = 'Informações do Plugin'
        icon = QgsApplication.getThemeIcon('mActionHelpContents.svg')
        self.actionInfo = QAction(icon, title, self.iface.mainWindow())
        self.actionInfo.setToolTip(title)
        self.actionInfo.triggered.connect(self.showInfo)
        self.iface.addPluginToMenu(f"&{self.nameActionAlert}", self.actionInfo)
        # 
        m = self.toolButton.menu()
        m.addAction( self.actionAlert )
        m.addAction( self.actionConfig )
        m.addAction( self.actionClear )
        m.addAction( self.actionInfo )
        self.toolButton.setDefaultAction( self.actionAlert )

        self.mbalert.register()

    def unload(self):
        self.iface.removeToolBarIcon( self.actionAlert )
        self.iface.removePluginRasterMenu( self.name, self.actionAlert )
        self.iface.removePluginRasterMenu( self.name, self.actionConfig )
        self.iface.removePluginRasterMenu( self.name, self.actionClear )
        self.iface.removePluginRasterMenu( self.name, self.actionInfo )
        del self.actionAlert
        del self.actionConfig
        del self.actionClear
        del self.actionInfo
        del self.mbalert

    @pyqtSlot(bool)
    def runAlert(self, checked):
        # Show info dialog every time
        info_dialog = InfoDialog(self.iface.mainWindow())
        if not exec_dialog(info_dialog):
            # User cancelled the info dialog, don't proceed
            return
        
        if not self.api.tokenOk:
            params = DialogEmailPassword.getConfig(self.localSetting)
            hasRegister = not params['email'] is None
            if hasRegister:
                self.api.setToken(**params)
        self.mbalert.run()

    @pyqtSlot(bool)
    def runConfig(self, checked):
        title = 'MapBiomas Alert Email/Senha'
        runDialogEmailPassword( title, self.api, self.localSetting )

    @pyqtSlot(bool)
    def clearCredentials(self, checked):
        DialogEmailPassword.clearConfig(self.localSetting)
        self.api.tokenOk = False
        self.iface.messageBar().pushMessage(
            "MapBiomas Alert",
            "Credenciais removidas com sucesso",
            Qgis.Success
        )

    @pyqtSlot(bool)
    def showInfo(self, checked):
        """Show the plugin information dialog"""
        info_dialog = InfoDialog(self.iface.mainWindow())
        exec_dialog(info_dialog)
