#!/usr/bin/python3
# # -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : DialogEmailPassword
Description          : Class for Dialog with email and password
Date                 : April, 2021
copyright            : (C) 2021 by Luiz Motta
email                : motta.luiz@gmail.com

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import (
    QSettings,
    QRegularExpression,
)
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QLabel, QLineEdit,
    QCheckBox, QRadioButton,
    QDialogButtonBox,
    QHBoxLayout, QVBoxLayout, QPushButton,
    QScrollArea
)
from qgis.PyQt.QtGui import QRegularExpressionValidator

from qgis.gui import QgsPasswordLineEdit
from qgis.core import Qgis
from qgis import utils as QgsUtils

# PyQt5/PyQt6 compatibility
try:
    # PyQt6 style
    REGEX_CASE_INSENSITIVE = QRegularExpression.PatternOption.CaseInsensitiveOption
    ECHO_MODE_PASSWORD = QLineEdit.EchoMode.Password
    BUTTON_OK = QDialogButtonBox.StandardButton.Ok
    BUTTON_CANCEL = QDialogButtonBox.StandardButton.Cancel
except AttributeError:
    # PyQt5 style (fallback)
    REGEX_CASE_INSENSITIVE = QRegularExpression.CaseInsensitiveOption
    ECHO_MODE_PASSWORD = QLineEdit.Password
    BUTTON_OK = QDialogButtonBox.Ok
    BUTTON_CANCEL = QDialogButtonBox.Cancel

# PyQt5/PyQt6 compatibility for exec method
def exec_dialog(dialog):
    """Execute dialog in a PyQt5/PyQt6 compatible way"""
    try:
        # PyQt6 style
        return dialog.exec()
    except AttributeError:
        # PyQt5 style (fallback)
        return dialog.exec_()

# PyQt5/PyQt6 compatibility for QRegularExpression.exactMatch
def regex_exact_match(regex, text):
    """Check regex exact match in a PyQt5/PyQt6 compatible way"""
    try:
        # PyQt6 style
        return regex.match(text).hasMatch()
    except AttributeError:
        # PyQt5 style (fallback)
        return regex.exactMatch(text)


class InfoDialog(QDialog):
    """Information dialog shown when plugin is loaded"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("MapBiomas Alert Plugin - Nota Informativa")
        self.setModal(True)
        self.setMinimumWidth(700)
        self.setMinimumHeight(600)
        self.resize(750, 650)
        self.setupUI()
    
    def setupUI(self):
        layout = QVBoxLayout()
        
        # Create scroll area for the content
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setStyleSheet("QScrollArea { border: none; }")
        
        # Content area with informative note
        content_widget = QLabel()
        content_widget.setWordWrap(True)
        content_widget.setOpenExternalLinks(True)
        content_widget.setStyleSheet("""
            QLabel {
                margin: 20px;
                font-size: 10pt;
                line-height: 1.4;
            }
        """)
        
        content_text = """
        <p>O MapBiomas Alerta é um sistema de validação e refinamento de alertas de desmatamento de vegetação nativa em todos os biomas brasileiros com imagens de alta resolução.</p>
        
        <p>Este sistema está em constante desenvolvimento pela rede colaborativa de co-criadores do MapBiomas contando com sugestões dos órgãos governamentais usuários (ex. MMA, IBAMA, SFB, ICMBio, MPF e TCU) e os provedores de alertas (ex. INPE, IMAZON, Universidade de Maryland, ISA, e outros).</p>
        
        <p><strong>IMPORTANTE!</strong></p>
        
        <p><strong>O MapBiomas Alerta publica toda e qualquer perda de vegetação nativa detectada pelos sistemas provedores de alertas e validada em imagens de satélite de alta resolução. O MapBiomas informa que não realiza avaliação sobre legalidade, responsabilidade e/ou restrição acerca dos alertas de desmatamento validados e publicados nesta plataforma. Essa avaliação é de exclusiva competência dos órgãos públicos e/ou das instituições privadas e financeiras que podem acessar livremente os dados disponibilizados pelo MapBiomas Alerta. O MapBiomas não se responsabiliza pelas decisões tomadas por esses órgãos e instituições com base nos alertas publicados, pois estes contêm dados objetivos e concretos sobre a existência de desmatamento (perda de vegetação nativa).</strong></p>
        
        <p>A seguir alguns esclarecimentos sobre o escopo e as limitações dos dados dos alertas:</p>
        
        <p><strong>1.</strong> <strong>Período dos alertas:</strong> a fase operacional busca avaliar todos os alertas de desmatamento detectados no país a partir de janeiro de 2019. Os alertas anteriores a esse período (outubro a dezembro de 2018) representam a fase pré-operacional com uma amostra dos alertas do período, e não aparecem nas estatísticas e no mapa (apenas continuam no banco de dados da plataforma para consulta por meio dos seus respectivos códigos do alerta). A data de detecção é aquela em que o alerta foi gerado pelos provedores (ex. DETER, SAD, GLAD) e não necessariamente representa o momento em que o desmatamento ocorreu. Desmatamentos detectados em 2019, por exemplo, podem ter se iniciado ou ter acontecido em 2018. Os dados do ano corrente são sempre parciais, sujeitos a alteração, devido ao tempo de processamento e publicação dos alertas. Quando os dados para um ano são consolidados, sistematizamos e publicamos o Relatório Anual do Desmatamento (RAD), o qual fica disponível no site do MapBiomas Alerta (http://alerta.mapbiomas.org/relatorio).</p>
        
        <p><strong>2.</strong> <strong>Área do alerta:</strong> Durante a etapa de refinamento do polígono é aplicado um processo de generalização que pode ocasionar uma variação de até 5% (para mais ou para menos) na área final. A área mínima do desmatamento para ser publicado na Plataforma do MapBiomas Alerta é 0,3 hectares. Para ser considerado sobreposição com CAR a área mínima também é de 0,3 ha</p>
        
        <p><strong>3.</strong> <strong>Vegetação não lenhosa:</strong> o MapBiomas Alerta utiliza os indícios de desmatamento dos sistemas DETER/INPE, SAD/IMAZON, GLAD/Univ. Maryland, SIRAD-X/ISA, SAD Caatinga/UEFS-Geodatin, SAD Mata Atlântica/SOS Mata Atlântica-Arcplan, SAD Pantanal/SOS Pantanal-ArcPlan, SAD Pampa/Geokarten-UFRGS e SAD Cerrado/IPAM. Destes sistemas apenas o DETER-Cerrado gera alertas em vegetação não lenhosa. Portanto, apenas no bioma Cerrado é monitorado o desmatamento em vegetação campestre. Nos demais biomas o desmatamento em vegetação não lenhosa só é identificado no MapBiomas Alerta quando associado a indícios de desmatamento em formações florestais e savânicas (veja mais sobre as fontes de dados e o método no site http://alerta.mapbiomas.org/). Nota Explicativa do Desmatamento no Pampa.</p>
        
        <p><strong>4.</strong> <strong>Embargos:</strong> as áreas de embargo correspondem aquelas disponibilizadas no geo-serviço do IBAMA e do ICMBio (http://siscom.ibama.gov.br/geoserver/) e podem estar incompletas em relação aos embargos estaduais e municipais.</p>
        
        <p><strong>5.</strong> <strong>Cadastro Ambiental Rural:</strong> os dados do Cadastro Ambiental Rural são aqueles consultados por geo-serviço do SICAR gerido pelo Serviço Florestal Brasileiro. Imóveis que estejam cadastrados nos estados e não tenham sido sincronizados com o SICAR não são contemplados.</p>
        
        <p><strong>6.</strong> <strong>Autorizações de supressão de vegetação nativa:</strong> as autorizações de uso alternativo e supressão da vegetação, bem como planos de manejo florestal, têm como fonte o SINAFLOR/IBAMA que integra os dados de todos os estados brasileiros, com limitações de atualizações. Para os estados do Pará e Mato Grosso, as informações foram obtidas dos geo-serviço das SEMAs destes estados (https://monitoramento.semas.pa.gov.br/monitoramento/#/sig e http://monitoramento.sema.mt.gov.br/simlam/). As autorizações feitas no âmbito municipal podem estar incompletas, assim como as autorizações emitidas pelos estados anteriores à implementação do SINAFLOR em 2018.</p>
        
        <p><strong>7.</strong> <strong>Legalidade, responsabilidade e/ou restrição acerca dos alertas de desmatamento validados e publicados na plataforma:</strong> este tipo de avaliação não é feita pelo MapBiomas, cabendo aos usuários (órgãos ambientais, instituições financeiras etc) buscar este tipo de informação junto às autoridades competentes ou aos seus clientes.</p>
        
        <p><strong>8.</strong> <strong>Alertas pendentes:</strong> o propósito é revisar todos os alertas gerados pelos principais sistemas, no entanto, parte dos alertas pode ter seu processo de validação pendente por motivos diversos (ex. falta de imagem por cobertura de nuvem, excesso de alertas em meses de maior quantidade de desmatamento etc.). A quantidade de alertas pendentes pode variar ao longo do tempo, já que são continuamente produzidos e publicados.</p>
        
        <p><strong>9.</strong> <strong>Omissões:</strong> o MapBiomas não gera alertas próprios e depende da qualidade dos sistemas que operam em território brasileiro para evitar omissões. O desenvolvimento de novos sistemas de detecção de alertas e a inclusão deles como fonte no MapBiomas Alerta tem reduzido omissões e permitindo a correta identificação dos desmatamentos em todos os biomas.</p>
        
        <p><strong>10.</strong> Alguns alertas foram revisados após a publicação na plataforma e CANCELADOS por motivo de limpeza de pasto ou bordas, silvicultura ou outros. Estes alertas não aparecem nas estatísticas e nem no mapa. Mas, eles continuam no banco de dados da plataforma para consulta por meio dos seus respectivos códigos do alerta.</p>
        
        <p><strong>11.</strong> <strong>A iniciativa MapBiomas Alerta</strong> está em constante melhoria e atualização para que informações que qualifiquem os alertas (propriedades do SIGEF, imóveis do CAR, limites das Unidades de Conservação, autorização de supressão da vegetação ou uso alternativo do solo, embargos e desembargos, ações de fiscalização, etc.) estejam o mais atualizadas e completas possível. Sempre disponibilizamos a fonte, mês e ano que foram consultadas tais bases para que os(as) usuários(as), quando necessário, possam utilizar as informações oficiais mais atualizadas em suas análises e decisões.</p>
        
        <p>Caso tenha sugestões, críticas e ideias para aprimorar o trabalho entre em contato pelo e-mail: <a href="mailto:suporte.alerta@mapbiomas.org">suporte.alerta@mapbiomas.org</a></p>
        """
        
        content_widget.setText(content_text)
        scroll_area.setWidget(content_widget)
        layout.addWidget(scroll_area)
        
        # Buttons
        button_layout = QHBoxLayout()
        button_layout.addStretch()
        
        ok_button = QPushButton("Concordo")
        ok_button.setStyleSheet("font-size: 10pt; padding: 6px 12px;")
        ok_button.clicked.connect(self.accept)
        button_layout.addWidget(ok_button)
        
        layout.addLayout(button_layout)
        self.setLayout(layout)


class DialogEmailPassword(QDialog):
    def __init__(self, title, hasRegister):
        def setupUI():
            def createLayoutPassword():
                layout = QHBoxLayout()

                # Add help text at the top
                helpText = QLabel(
                    "Se você precisa criar uma conta, visite nossa página "
                    "<a href='https://plataforma.alerta.mapbiomas.org/sign-up'>https://plataforma.alerta.mapbiomas.org/sign-up</a> "
                    "para criar uma ou use <a href='https://plataforma.alerta.mapbiomas.org/api/docs/index.html#mutation-signIn'>"
                    "https://plataforma.alerta.mapbiomas.org/api/docs/index.html#mutation-signIn</a> "
                    "com sua conta existente para obter seu Token Bearer.",
                    self
                )
                helpText.setWordWrap(True)
                helpText.setOpenExternalLinks(True)
                layout.addWidget(helpText)

                # Email/Password layout
                credentialsLayout = QHBoxLayout()
                # Email
                w = QLabel('Email/Senha:', self)
                credentialsLayout.addWidget(w)
                w = QLineEdit(self)
                self.__dict__['email'] = w
                w.setPlaceholderText('alguem@exemplo.com')
                w.setToolTip('Email registrado na plataforma MapBiomas Alerta')
                rx = QRegularExpression(
                    self.emailExpEdit, REGEX_CASE_INSENSITIVE)
                w.setValidator(QRegularExpressionValidator(rx))
                credentialsLayout.addWidget(w)
                # Password
                w = QgsPasswordLineEdit(self)
                self.__dict__['password'] = w
                w.setToolTip('Senha registrada na plataforma MapBiomas Alerta')
                w.setEchoMode(ECHO_MODE_PASSWORD)
                credentialsLayout.addWidget(w)
                # Save
                w = QCheckBox('Salvar credenciais', self)
                self.__dict__['register_email_password'] = w
                w.setToolTip('Salvar credenciais na configuração do QGIS')
                w.setChecked(True)
                credentialsLayout.addWidget(w)

                # Add credentials layout below help text
                mainLayout = QVBoxLayout()
                mainLayout.addWidget(helpText)
                mainLayout.addLayout(credentialsLayout)

                return mainLayout

            def createLayoutRegister():
                layout = QHBoxLayout()
                # Register
                w = QRadioButton('Copiar credenciais', self)
                self.__dict__['clipboard'] = w
                w.setToolTip('Copiar credenciais salvas para a área de transferência')
                layout.addWidget(w)
                w = QRadioButton('Limpar credenciais', self)
                self.__dict__['clear'] = w
                w.setToolTip('Remover credenciais salvas')
                layout.addWidget(w)
                return layout

            lyt = QVBoxLayout()
            f = createLayoutPassword
            subLayout = f()
            lyt.addLayout(subLayout)
            # Ok and Cancel buttons
            w = QDialogButtonBox(BUTTON_OK | BUTTON_CANCEL, self)
            w.button(BUTTON_OK).setText("Entrar")
            w.button(BUTTON_CANCEL).setText("Cancelar")
            w.accepted.connect(self.accept)
            w.rejected.connect(self.reject)
            lyt.addWidget(w)
            #
            self.setLayout(lyt)

        super().__init__(QgsUtils.iface.mainWindow())
        self.setWindowTitle(title)
        self.emailExpEdit = '\\b[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,4}\\b'
        self.emailExpMath = r"(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)"
        setupUI()

    def getParams(self):
        params = {}
        for k in ('email', 'password'):
            params[k] = self.__dict__[k].text()
        return params

    def isValidEmail(self):
        rx = QRegularExpression(self.emailExpMath)
        return regex_exact_match(rx, self.email.text())

    def isCheckedClipboard(self):
        return self.clipboard.isChecked()

    def isCheckedClear(self):
        return self.clear.isChecked()

    def isCheckedSave(self):
        checked = self.register_email_password.isChecked()
        return checked

    @staticmethod
    def getConfig(localSetting):
        s = QSettings()
        params = {}
        for k in ('email', 'password'):
            value = s.value(localSetting.format(k), None)
            params[k] = value
        return params

    @staticmethod
    def setConfig(localSetting, email, password):
        s = QSettings()
        s.setValue(localSetting.format('email'), email)
        s.setValue(localSetting.format('password'), password)

    @staticmethod
    def clearConfig(localSetting):
        s = QSettings()
        s.remove(localSetting.format('email'))
        s.remove(localSetting.format('password'))

    @staticmethod
    def copy2Clipboard(email, password):
        msg = f"{email}/{password}"
        clipboard = QApplication.clipboard()
        clipboard.setText(msg)


def runDialogEmailPassword(title, api, localSetting):
    print("\n[Login] Starting email/password dialog")
    msgBar = QgsUtils.iface.messageBar()

    def messageHandler(msg, level):
        msgBar.pushMessage(msg, level)
    api.message.connect(messageHandler)

    def checkHasRegister():
        hasRegister = True
        if api.tokenOk:
            return hasRegister
        params = DialogEmailPassword.getConfig(localSetting)
        hasRegister = not params['email'] is None
        if hasRegister:
            print("[Login] Found saved credentials, attempting authentication")
            api.setToken(**params)
            if not api.tokenOk:
                DialogEmailPassword.clearConfig(localSetting)
                hasRegister = False
        return hasRegister

    hasRegister = checkHasRegister()
    dlg = DialogEmailPassword(title, hasRegister)
    result = exec_dialog(dlg)
    if not result:
        return

    # Save clipboard or Clear
    if hasRegister:
        if dlg.isCheckedClipboard():
            params = DialogEmailPassword.getConfig(localSetting)
            dlg.copy2Clipboard(**params)
        if dlg.isCheckedClear():
            dlg.clearConfig(localSetting)
        return

    # Get params and test token
    params = dlg.getParams()

    # Add input validation
    if not params['email'] or not params['password']:
        msgBar.pushMessage("Email e senha não podem estar vazios", Qgis.Critical)
        return

    if not dlg.isValidEmail():
        msgBar.pushMessage(
            f"Formato de email inválido: {params['email']}", Qgis.Critical)
        return

    api.setToken(params['email'], params['password'])

    # Only save credentials if checkbox is checked
    if dlg.isCheckedSave():
        if 'sendMessage' in params:
            del params['sendMessage']
        DialogEmailPassword.setConfig(
            localSetting, params['email'], params['password'])
