#!/usr/bin/python3
# # -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : MapBiomas Alert
Description          : Class for work with MapBiomas Alert
Date                 : April, 2019
copyright            : (C) 2019 by Luiz Motta
email                : motta.luiz@gmail.com

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from ast import Raise
import urllib.parse

import json
import binascii
import os
import csv
from xml.etree import ElementTree as ET
from .accesssite import AccessSite
from qgis.PyQt.QtCore import (
    QObject, QUrl,
    pyqtSlot, pyqtSignal
)
from qgis.PyQt.QtNetwork import QNetworkRequest
from qgis.PyQt.QtGui import QPixmap

from qgis.core import (
    Qgis, QgsProject, QgsApplication,
    QgsVectorLayer, QgsFeature, QgsGeometry,
    QgsFeatureRequest,
    QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsBlockingNetworkRequest,
    QgsTask, QgsVectorLayer, QgsDataSourceUri
)
from qgis import utils as QgsUtils


class Geometry_WKB():
    # Adaptation from https://github.com/elpaso/quickwkt/blob/master/QuickWKT.py
    SRID_FLAG = 0x20000000

    @staticmethod
    def decodeBinary(geom_hex):
        value = binascii.a2b_hex(geom_hex)
        value = value[::-1]
        value = binascii.b2a_hex(value)
        return value.decode("UTF-8")

    @staticmethod
    def encodeBinary(geom_hex):
        wkb = binascii.a2b_hex("{:08x}".format(geom_hex))
        wkb = wkb[::-1]
        wkb = binascii.b2a_hex(wkb)
        return wkb.decode("UTF-8")

    @staticmethod
    def getQgsGeometry_SRID(geom_hex):
        """ geomHex: ST_AsHexEWKB(geom) or ST_AsHexWKB(geom) """
        srid = None
        geomType = int("0x" + Geometry_WKB.decodeBinary(geom_hex[2:10]), 0)
        if geomType & Geometry_WKB.SRID_FLAG:
            srid = int("0x" + Geometry_WKB.decodeBinary(geom_hex[10:18]), 0)
            # String the srid from the wkb string
            geom_hex = geom_hex[:2] + Geometry_WKB.encodeBinary(
                geomType ^ Geometry_WKB.SRID_FLAG) + geom_hex[18:]

        geom = QgsGeometry()
        geom.fromWkb(binascii.a2b_hex(geom_hex))

        return (geom, srid)


class API_MapbiomasAlert(QObject):

    # 'https://geoserver.ecostage.com.br/geoserver/ows'
    urlGeoserver = 'https://geoserver.alerta.mapbiomas.org/geoserver/wfs'
    urlReport = 'http://plataforma.alerta.mapbiomas.org/alerta'
    access = AccessSite()
    fields = {
        'alert_code': {'definition': 'int(-1)'},
        'detected_at':  {'definition': 'string(10)'},
        'source': {'definition': 'string(100)'},
        # 'dt_antes': {'definition': 'string(10)'},
        # 'img_antes': {'definition': 'string(150)'},
        # 'dt_depois': {'definition': 'string(10)'},
        # 'img_depois': {'definition': 'string(150)'},
        'cars': {'definition': 'string(150)'},
        # 'cars_qtd': {'definition': 'int(-1)'},
        'area_ha': {'definition': 'double'}
    }

    Q_TOKEN = """
    {
        "query": "mutation($email: String!, $password: String!)
            { 
                    signIn(email: $email, password: $password) { token }
            }",
        "variables": {
            "email": "_email_",
            "password": "_password_"
        }
    }
    """
    Q_ALLPUBLISHEDALERTS = """
    {
        "query": "
        query
        (
        $startDetectedAt: String
        $endDetectedAt: String
        $startPublishedAt: String
        $endPublishedAt: String
        $territoryIds: [Int!]
        $limit: Int
        $offset: Int
        )
        {
        publishedAlerts 
        (
            startDetectedAt: $startDetectedAt
            endDetectedAt: $endDetectedAt
            startPublishedAt: $startPublishedAt
            endPublishedAt: $endPublishedAt
            territoryIds: $territoryIds
            limit: $limit
            offset: $offset
        )
            { _fields_  }
        }",
        "variables": {
            "limit": _limit_, "offset": _offset_ ,
            "startDetectedAt": "_startDetectedAt_", 
            "endDetectedAt": "_endDetectedAt_",
            "startPublishedAt": "_startDetectedAt_",
            "endPublishedAt": "_endDetectedAt_",
            "territoryIds": [ _territoryIds_ ]
        }
    }
    """
    Q_IMAGES = """
    {
        "query": "
        query ($alertCode: Int!) {
            alert(alertCode: $alertCode) {
                publishedImages{
                    reference
                    satellite
                    constellation
                    acquiredAt
                    url
                }
            }
        }",
        "variables": { "alertCode": _alertId_ }
    }
    """
    LIMIT = 50
    OFFSETID = 0
    NETWORKREQUEST = QNetworkRequest(
        QUrl('https://plataforma.alerta.mapbiomas.org/api/v2/graphql'))
    message = pyqtSignal(str, Qgis.MessageLevel)
    status = pyqtSignal(str)
    alerts = pyqtSignal(list)
    finishedAlert = pyqtSignal()
    images = pyqtSignal(dict)

    def __init__(self):
        super().__init__()
        self.taskManager = QgsApplication.taskManager()
        self.taskAlerts, self.taskImage = None, None
        self.request = QgsBlockingNetworkRequest()
        API_MapbiomasAlert.NETWORKREQUEST.setHeader(
            QNetworkRequest.ContentTypeHeader, 'application/json')
        self.tokenOk = False
        self._connected_alerts = {}  # Track connections by layer ID
        self._alert_connections = {} # Store actual connection references

    def connectAlerts(self, dbAlerts):
        """Manage signal connections for a specific layer"""
        layer_id = dbAlerts.layer.id()
        
        # If already connected, disconnect first
        if layer_id in self._alert_connections:
            try:
                self.alerts.disconnect(self._alert_connections[layer_id])
            except:
                pass
            del self._alert_connections[layer_id]
            
        # Create new connection
        self._alert_connections[layer_id] = dbAlerts.addFeatures
        self.alerts.connect(self._alert_connections[layer_id])
        self._connected_alerts[layer_id] = dbAlerts

    def disconnectAlerts(self, layer_id):
        """Disconnect signals when layer is removed"""
        if layer_id in self._alert_connections:
            try:
                self.alerts.disconnect(self._alert_connections[layer_id])
            except:
                pass
            del self._alert_connections[layer_id]
            if layer_id in self._connected_alerts:
                del self._connected_alerts[layer_id]

    def _request(self, data):
        try:
            data = data.replace('\n', '').encode('utf-8')
            err = self.request.post(API_MapbiomasAlert.NETWORKREQUEST, data)

            if err != QgsBlockingNetworkRequest.NoError:
                return None

            reply = self.request.reply()
            if not reply:
                return None

            content = bytes(reply.content()).decode()
            return json.loads(content)

        except Exception as e:
            self.message.emit(f"Request failed: {str(e)}", Qgis.Critical)
            return None

    def _replaceQuery(self, values, query):
        q = query
        for v in values.keys():
            q = q.replace(f"_{v}_", str(values[v]))
        return q

    def clearToken(self):
        self.tokenOk = False
        API_MapbiomasAlert.NETWORKREQUEST.setRawHeader(b'Authorization', b'')

    def setToken(self, email, password):
        try:
            print("\n[Authentication] Attempting to authenticate user...")
            values = {'email': email, 'password': password}
            data = self._replaceQuery(values, self.Q_TOKEN)
            response = self._request(data)

            if not response:
                print("[Authentication] Failed to connect to authentication server")
                self.message.emit(
                    "Failed to connect to authentication server", Qgis.Critical)
                self.clearToken()
                return

            if 'errors' in response:
                print(
                    f"[Authentication] Error: {response['errors'][0]['message']}")
                error_msg = "\n".join([e['message']
                                      for e in response['errors']])
                self.message.emit(error_msg, Qgis.Critical)
                self.clearToken()
                return

            if not response.get('data') or not response['data'].get('signIn'):
                print("[Authentication] Invalid response structure from server")
                self.message.emit(
                    "Unexpected server response format", Qgis.Critical)
                self.clearToken()
                return

            print("[Authentication] Successfully authenticated")
            token = response['data']['signIn']['token']
            value = f"Bearer {token}"
            API_MapbiomasAlert.NETWORKREQUEST.setRawHeader(
                b'Authorization', value.encode('utf-8'))
            self.tokenOk = True

            self.message.emit('Successfully authenticated', Qgis.Success)

        except Exception as e:
            self.message.emit(f"Authentication error: {str(e)}", Qgis.Critical)
            self.clearToken()

    def checkTokenValidity(self):
        # Add method to check if token is still valid
        pass

    def refreshToken(self):
        # Add method to refresh token if expired
        pass

    def getAlertsWFS(self, url, dbAlerts, fromDate, toDate, ids):
        def run(task):
            print('WFS mode')
            print('url= '+url)
            # uri = QgsDataSourceUri()
            # uri.setConnection("34.86.182.142", "5432", "alerta", "stg_geoserver_usr", "hJ2zZn0uW4af")
            # uri.setDataSource ("public", "temp_gis_published_alerts", 'geom')
            # layer= QgsVectorLayer (uri.uri(False), "temp_gis_published_alerts", "postgres")
            layer = QgsVectorLayer(url, 'qgis_published_alerts', 'WFS')
            # QgsProject.instance().addMapLayer(layer)
            # print('Attributos:')
            for field in layer.fields():
                print(field.name(), field.typeName())
            # print('Getting Features')
            values = layer.getFeatures()
            # print('Next')
            # print(layer.getFeatures().next())
            # print('Value')
            values = [dbAlerts.transformItemWFS(v, self) for v in values]
            self.alerts.emit(values)
            layer.reload()
            return {'total': 1}

        def finished(exception, dataResult=None):
            self.finishedAlert.emit()
            self.taskAlerts = None
            msg = f"Finished {dataResult['total']} alerts" if dataResult else ''
            self.status.emit(msg)

        task = QgsTask.fromFunction('Alert Task', run, on_finished=finished)
        task.setDependentLayers([dbAlerts.layer])
        self.taskAlerts = task
        self.taskManager.addTask(task)

    def getAlertsWFSnonThread(self, url, dbAlerts, fromDate, toDate, ids):
        print("[API] Creating WFS layer from URL")
        layer = QgsVectorLayer(url, 'qgis_published_alerts', 'WFS')
        print(f"[API] WFS Layer valid: {layer.isValid()}")
        
        print("[API] Transforming features")
        values = layer.getFeatures()
        transformed_values = [dbAlerts.transformItemWFS(v, self) for v in values]
        
        print("[API] Emitting alerts signal")
        self.alerts.emit(transformed_values)
        self.finishedAlert.emit()
        return {'total': len(transformed_values)}

        # print(maxValue)
        # p = {
        #    'url': QUrl( url ),
        # }
        # self.access.requestUrl( p, self._addFeaturesLinkResponse, setFinished )

    def _addFeaturesLinkResponse(self, response):
        def getFeaturesResponse(data):
            def getGeometry(geometry):
                def getPolygonPoints(coordinates):
                    polylines = []
                    for line in coordinates:
                        polyline = [QgsPointXY(p[0], p[1]) for p in line]
                        polylines.append(polyline)
                    return polylines

                if geometry['type'] == 'Polygon':
                    polygon = getPolygonPoints(geometry['coordinates'])
                    return QgsGeometry.fromMultiPolygonXY([polygon])
                elif geometry['type'] == 'MultiPolygon':
                    polygons = []
                    for polygon in geometry['coordinates']:
                        polygons.append(getPolygonPoints(polygon))
                    return QgsGeometry.fromMultiPolygonXY(polygons)

                else:
                    None

            features = []
            for feat in data['features']:
                if self.access.isKill:
                    return features
                geom = getGeometry(feat['geometry'])
                del feat['geometry']
                properties = feat['properties']
                item = {}
                for k in self.fields.keys():
                    item[k] = properties[k]
                if not item['cars'] is None:
                    item['cars'] = item['cars'].replace(';', '\n')
                else:
                    item['cars'] = ''
                item['geometry'] = geom
                features.append(item)
            return features

    @staticmethod
    def getUrlAlertsBySource(wktGeom, sourcename):
        params = {
            'service': 'WFS',
            'version': '1.0.0',
            'request': 'GetFeature',
            'typeName': 'mapbiomas-alerta:qgis_published_alerts',
            # 'count':''+str(step),
            # 'startIndex':''+str(page)
            # 'outputFormat': 'application/json',
            # 'MAXFEATURES':'5000'
            'cql_filter': "source ilike '%"+sourcename+"%'"
        }
        params = '&'.join(["{k}={v}".format(k=k, v=params[k])
                          for k in params.keys()])
        return "{url}?{params}".format(url=API_MapbiomasAlert.urlGeoserver, params=params)

    @staticmethod
    def getUrlAlertsbyCQL(wktGeom, cql):
        params = {
            'service': 'WFS',
            'version': '1.0.0',
            'request': 'GetFeature',
            'typeName': 'mapbiomas-alerta:qgis_published_alerts',
            # 'count':''+str(step),
            # 'startIndex':''+str(page)
            # 'outputFormat': 'application/json',
            # 'MAXFEATURES':'5000'
            'cql_filter': cql
        }
        params = '&'.join(["{k}={v}".format(k=k, v=params[k])
                          for k in params.keys()])
        return "{url}?{params}".format(url=API_MapbiomasAlert.urlGeoserver, params=params)

    @staticmethod
    def getUrlAlertsZero():
        params = {
            'service': 'WFS',
            'version': '1.0.0',
            'request': 'GetFeature',
            'typename': 'mapbiomas-alerta:qgis_published_alerts',
            # 'count':''+str(step),
            # 'startIndex':''+str(page)
            # 'outputFormat': 'application/json',
            # 'MAXFEATURES':'5000'
            'cql_filter': "id <= 20000"
        }
        params = '&'.join(["{k}={v}".format(k=k, v=params[k])
                          for k in params.keys()])
        return "{url}?{params}".format(url=API_MapbiomasAlert.urlGeoserver, params=params)

    @staticmethod
    def getFeatureCount(after, before):
        params = {
            'service': 'WFS',
            'version': '1.1.0',
            'request': 'GetFeature',
            'typeName': 'mapbiomas-alertas:mv_qgis_published_alerts_snap_to_grid',
            'resultType': 'hits',
            'cql_filter': f"detected_at between '{after}' and '{before}'"
        }
        url = QUrl(API_MapbiomasAlert.urlGeoserver)
        url.setQuery(urllib.parse.urlencode(params))

        request = QgsBlockingNetworkRequest()
        response = request.get(QNetworkRequest(url))

        if response == QgsBlockingNetworkRequest.NoError:
            reply = request.reply()
            if reply and reply.content():
                xml_str = bytes(reply.content()).decode('utf-8')
                root = ET.fromstring(xml_str)
                return int(root.get('numberOfFeatures', 0))
        return 0

    @staticmethod
    def getUrlAlertsPaginated(step, offset, fromDate, toDate):
        params = {
            'service': 'WFS',
            'version': '1.1.0',
            'request': 'GetFeature',
            'typeName': 'mapbiomas-alertas:mv_qgis_published_alerts_snap_to_grid',
            'maxFeatures': str(step),
            'startIndex': str(offset),
            'sortBy': 'alert_code',
            'srsName': 'EPSG:4674',
            'cql_filter': f"detected_at between '{fromDate}' and '{toDate}'"
        }
        params = '&'.join(["{k}={v}".format(k=k, v=params[k])
                          for k in params.keys()])
        return "{url}?{params}".format(url=API_MapbiomasAlert.urlGeoserver, params=params)

    @staticmethod
    def getUrlAlerts(before, after):
        params = {
            'service': 'WFS',
            'version': '1.0.0',
            'request': 'GetFeature',
            'typeName': 'mapbiomas-alertas:mv_qgis_published_alerts_snap_to_grid',
            'sortBy': 'alert_code',
            'srs': 'EPSG:4674',
            'srsName': 'EPSG:4674',
            'cql_filter': f"detected_at between '{before}' and '{after}'"
        }
        params = '&'.join(["{k}={v}".format(k=k, v=params[k])
                          for k in params.keys()])
        return "{url}?{params}".format(url=API_MapbiomasAlert.urlGeoserver, params=params)

    def getAlerts(self, dbAlerts, startDetectedAt, endDetectedAt, territoryIds):
        def run(task):
            def request(values):
                maxValue = 0
                data = self._replaceQuery(values, self.Q_ALLPUBLISHEDALERTS)
                print('Request DATA')
                print(data)
                response = self._request(data)
                if not response:
                    msg = f"MapBiomas Alert: {self.request.errorMessage()}"
                    self.message.emit(msg, Qgis.Critical)
                    return -1
                if 'errors' in response:
                    l_messages = [v['message'] for v in response['errors']]
                    msg = f"MapBiomas Alert: {','.join(l_messages)}"
                    self.message.emit(msg,  Qgis.Critical)
                    return -1
                values = response['data']['publishedAlerts']
                values = [dbAlerts.transformItem(v) for v in values]
                for v in values:
                    if maxValue < int(v['alertCode']):
                        maxValue = int(v['alertCode'])
                # print(maxValue)
                self.alerts.emit(values)

                return maxValue  # len(values)

            def getFieldsName():
                fields = list(dbAlerts.FIELDSDEF.keys())
                toField = fields.index('detectedAt') + 1
                fields = fields[:toField]
                return " ".join(fields) + " cars { id carCode } geometry { geom }"

            fieldsName = getFieldsName()
            s_territoryIds = ','.join([str(v) for v in territoryIds])
            offset = 0
            previousLast = 0
            while True:
                values = {
                    'startDetectedAt': startDetectedAt, 'endDetectedAt': endDetectedAt,
                    'territoryIds': s_territoryIds,
                    'limit': self.LIMIT, 'offset': offset,
                    'fields': fieldsName
                }
                # print('Requesting OFFSET=')
                # print(offset)

                # print(values)
                # self.message.emit(values, Qgis.Warning)
                last = request(values)
                # print('Last ID:')
                # print(last)
                if task.isCanceled():
                    self.message.emit('Canceled by user', Qgis.Warning)
                    return
                if last == -1:
                    return
                if previousLast == last:
                    break
                offset = offset + self.LIMIT  # total
                previousLast = last
                self.status.emit(f"Receiving {offset}...")

            return {'total': offset + total}

        def finished(exception, dataResult=None):
            self.finishedAlert.emit()
            self.taskAlerts = None
            msg = f"Finished {dataResult['total']} alerts" if dataResult else ''
            self.status.emit(msg)

        task = QgsTask.fromFunction('Alert Task', run, on_finished=finished)
        task.setDependentLayers([dbAlerts.layer])
        self.taskAlerts = task
        self.taskManager.addTask(task)
        # Debug
        # r = run( task )
        # finished(None, r)

    def cancelAlerts(self):
        if self.taskAlerts:
            self.taskAlerts.cancel()

    def getImages(self, alertId):
        def run(task):
            def getThumbnail(url):
                err = self.request.get(QNetworkRequest(QUrl(url)))
                if not err == QgsBlockingNetworkRequest.NoError:
                    self.message.emit(
                        self.request.errorMessage(), Qgis.Critical)
                    return None
                data = self.request.reply().content().data()
                pixmap = QPixmap()
                pixmap.loadFromData(data)
                return pixmap

            values = {'alertId': alertId}
            data = self._replaceQuery(values, self.Q_IMAGES)
            response = self._request(data)
            print('REQUEST [IMAGE data]:')
            print(data)
            if not response:
                self.message.emit(self.request.errorMessage(), Qgis.Critical)
                return None
            data = response['data']['alert']['publishedImages']
            data_return = {}
            for k in ('before', 'after'):
                # get k from data reference
                reference = [v for v in data if v['reference'] == k]
                reference = reference[0]
                data_return[k] = {}
                data_return[k]['acquiredAt'] = reference['acquiredAt']
                data_return[k]['satellite'] = reference['satellite']
                data_return[k]['constellation'] = reference['constellation']
                data_return[k]['url'] = reference['url']
                data_return[k]['thumbnail'] = getThumbnail(reference['url'])
            return data_return

        def finished(exception, dataResult=None):
            self.taskImage = None
            if dataResult:
                self.images.emit(dataResult)

        task = QgsTask.fromFunction(
            'Alert/Get Images Task', run, on_finished=finished)
        self.taskImage = task
        self.taskManager.addTask(task)
        # Debug
        # r = run( task )
        # finished(None, r)


class TerritoryBbox():
    FIELDSDEF = {
        'id': 'string(25)'
    }
    CRS = QgsCoordinateReferenceSystem('EPSG:4674')
    CSV = 'territory_bbox.csv'

    def __init__(self):
        self.mapCanvas = QgsUtils.iface.mapCanvas()
        self.taskManager = QgsApplication.taskManager()
        self.threadMain = QgsApplication.instance().thread()
        self.csv = os.path.join(os.path.dirname(__file__),  self.CSV)
        self.project = QgsProject.instance()
        self.layer = None

    def __del__(self):
        if self.layer:
            self.layer = None

    def _createLayer(self):
        name = self.CSV
        l_fields = [f"field={k}:{v}" for k, v in self.FIELDSDEF.items()]
        l_fields.insert(0, f"polygon?crs={self.CRS.authid().lower()}")
        l_fields.append("index=yes")
        uri = '&'.join(l_fields)
        return QgsVectorLayer(uri, name, 'memory')

    def setLayer(self):
        def run(task):
            layer = self._createLayer()
            provider = layer.dataProvider()
            with open(self.csv) as csv_file:
                csv_reader = csv.reader(csv_file, delimiter=';')
                for row in csv_reader:
                    id, wkt = row[0], row[1]
                    feat = QgsFeature()
                    feat.setAttributes([id])
                    geom = QgsGeometry.fromWkt(wkt)
                    feat.setGeometry(geom)
                    provider.addFeature(feat)
            layer.moveToThread(self.threadMain)
            return {'layer': layer}

        def finished(exception, dataResult=None):
            if dataResult:
                self.layer = dataResult['layer']

        task = QgsTask.fromFunction('Alert Task', run, on_finished=finished)
        self.taskManager.addTask(task)
        # Debug
        # Comment layer.moveToThread(self.threadMain )
        # r = run( task, layer )
        # finished(None, r)

    def getIdsCanvas(self):
        crs = self.project.crs()
        extent = self.mapCanvas.extent()
        if not crs == self.CRS:
            ct = QgsCoordinateTransform(crs, self.CRS, self.project)
            extent = ct.transform(extent)
        fr = QgsFeatureRequest().setFilterRect(extent)
        features = self.layer.getFeatures(fr)
        return [feat['id'] for feat in features]


class DbAlerts(QObject):
    FIELDSDEF = {
        'alertCode': 'string(25)',
        'source': 'string(-1)',
        'areaHa': 'double',
        'detectedAt': 'string(10)',
        'carCode': 'string(-1)',
    }
    CRS = QgsCoordinateReferenceSystem('EPSG:4674')

    def __init__(self, layer, api):
        super().__init__()
        self.layer = layer
        self.project = QgsProject.instance()
        self.project.layerWillBeRemoved.connect(self.removeLayer)
        self.api = api

    @staticmethod
    def transformItemWFS(item, context):
        # Source
        new_item = {}
        new_item['alertCode'] = item['alert_code']
        new_item['source'] = item['source']
        new_item['areaHa'] = item['area_ha']
        
        # Date handling - check if it's QDate or string
        detected_at = item['detected_at']
        if isinstance(detected_at, str):
            # Original string handling
            detectedAt = detected_at.replace('Z', '').split('-')
            detectedAt.reverse()
            new_item['detectedAt'] = '-'.join(detectedAt)
        else:
            # QDate handling
            new_item['detectedAt'] = detected_at.toString('dd-MM-yyyy')
        
        # Cars handling
        if item['cars'] == None or item['cars'] == 'NULL':
            item['cars'] = '{}'
        cars = json.loads(item['cars'])
        new_item['carCode'] = [v['car_code'] for v in cars]
        new_item['carCode'] = ','.join(new_item['carCode'])
        
        # Geometry
        new_item['geom'] = item.geometry()
        new_item['srid'] = '4674'
        
        return new_item

    @staticmethod
    def transformItem(item):
        # Source
        item['source'] = ','.join(item['source'])
        # Date
        detectedAt = item['detectedAt'].split('/')
        detectedAt.reverse()
        item['detectedAt'] = '-'.join(detectedAt)
        # carCode
        item['carCode'] = [v['carCode'] for v in item['cars']]
        item['carCode'] = ','.join(item['carCode'])
        del item['cars']
        # Geometry
        geom, srid = Geometry_WKB.getQgsGeometry_SRID(item['geometry']['geom'])
        del item['geometry']
        item['geom'] = geom
        item['srid'] = srid

        return item

    @staticmethod
    def createLayer():
        name = 'Alert ..'
        l_fields = [f"field={k}:{v}" for k, v in DbAlerts.FIELDSDEF.items()]
        l_fields.insert(0, f"Multipolygon?crs={DbAlerts.CRS.authid().lower()}")
        l_fields.append("index=yes")
        uri = '&'.join(l_fields)
        return QgsVectorLayer(uri, name, 'memory')

    def setLayer(self, startDetectedAt, endDetectedAt):
        name = f"Alert {startDetectedAt} .. {endDetectedAt}"
        self.layer.dataProvider().truncate()
        self.layer.setName(name)
        self.layer.updateExtents()
        self.layer.triggerRepaint()

    def addFeatures(self, data):
        if not self.layer or not self.layer.isValid():
            print("[DbAlerts] Layer invalid or None")
            return

        provider = self.layer.dataProvider()
        features_added = 0
        
        for item in data:
            # Geometry
            crs = QgsCoordinateReferenceSystem(f"EPSG:{item['srid']}")
            if not crs == self.CRS:
                ct = QgsCoordinateTransform(crs, self.CRS, self.project)
                item['geom'].transform(ct)

            # Add feature
            atts = [item[k] for k in self.FIELDSDEF]
            feat = QgsFeature()
            feat.setAttributes(atts)
            feat.setGeometry(item['geom'])
            provider.addFeature(feat)
            features_added += 1

        print(f"[DbAlerts] Added {features_added} features")
        self.layer.updateExtents()
        self.layer.triggerRepaint()

        # Zoom to layer if features were added
        if features_added:
            canvas = QgsUtils.iface.mapCanvas()
            extent = self.layer.extent()
            # Add a small buffer around the extent (10%)
            extent.scale(1.1)
            canvas.setExtent(extent)
            canvas.refresh()

    @pyqtSlot(str)
    def removeLayer(self, layerId):
        if self.layer and layerId == self.layer.id():
            self.api.disconnectAlerts(layerId)  # Disconnect signals
            self.layer.dataProvider().truncate()
            self.layer = None
