#!/usr/bin/python3
# # -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : MapBiomas Alert
Description          : Class for work with MapBiomas Alert
Date                 : April, 2019
copyright            : (C) 2019 by Luiz Motta
email                : motta.luiz@gmail.com

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import json, os

from qgis.PyQt.QtCore import (
    Qt,
    QObject, pyqtSlot, pyqtSignal,
    QUrl,
    QDate
)
from qgis.PyQt.QtWidgets import (
    QWidget, QPushButton,
    QLabel, QDateEdit, QSpinBox, QSpacerItem, QSizePolicy,
    QVBoxLayout, QHBoxLayout,
    QApplication, # widgets = QApplication.instance().allWidgets()
    QStyle, QLineEdit
)
from qgis.PyQt.QtGui import (
    QColor, QPixmap, QIcon,
    QDesktopServices # QDesktopServices.openUrl( QUrl( url ) )
)
from qgis.PyQt.QtNetwork import QNetworkRequest
 
from qgis.core import (
    Qgis, QgsApplication, QgsProject,
    QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsVectorLayer, QgsFeature,
    QgsBlockingNetworkRequest,QgsTask, QgsSymbol,
    QgsFillSymbol, QgsSingleSymbolRenderer
)
from qgis.gui import QgsGui, QgsMessageBar, QgsLayerTreeEmbeddedWidgetProvider
from qgis import utils as QgsUtils
from qgis.core import QgsSettings

from .mapbiomasalert_layer_api import DbAlerts, API_MapbiomasAlert, TerritoryBbox
from .form import setForm as FORM_setForm
from .dialog_email_password import DialogEmailPassword


class MapBiomasAlertWidget(QWidget):

    def __init__(self, layer, layerTerritory, api, localSetting):
        super().__init__()  # Make sure parent initialization happens first
        
        # Store these as instance variables before setting up UI
        self.layer = layer
        self.canvas = QgsUtils.iface.mapCanvas()
        self.project = QgsProject.instance()
        self.crsCatalog = QgsCoordinateReferenceSystem('EPSG:4674')
        self.msgBar = QgsUtils.iface.messageBar()
        self.api = api
        self.alert = DbAlerts(layer, self.api)
        self.layerTerritory = layerTerritory
        self.localSetting = localSetting
        self.clearAfterSearch = False
        
        # Get dates from layer properties if they exist
        from_date = layer.customProperty('mapbiomasalert/from_date')
        to_date = layer.customProperty('mapbiomasalert/to_date')
        
        if from_date and to_date:
            self.last_from_date = QDate.fromString(from_date, Qt.ISODate)
            self.last_to_date = QDate.fromString(to_date, Qt.ISODate)
        else:
            self.last_from_date = None
            self.last_to_date = None
        
        # Create main layout first
        self.mainLayout = QVBoxLayout()
        self.setLayout(self.mainLayout)
        
        # Setup UI components
        self.setupUI()
        
        # Connect signals after UI is set up
        def messageHandler(msg, level):
            self.msgBar.pushMessage(msg, level)
            if level == Qgis.Info:
                self.processingLabel.setText(msg)
            elif level == Qgis.Success:
                self.processingLabel.setText("")
            elif level == Qgis.Warning:
                self.processingLabel.setText(f"Warning: {msg}")
            elif level == Qgis.Critical:
                self.processingLabel.setText(f"Error: {msg}")

        def statusHandler(msg):
            self.processingLabel.setText(msg)
            
        self.api.message.connect(messageHandler)
        self.api.connectAlerts(self.alert)
        self.api.finishedAlert.connect(self._onFinishedAlert)
        self.api.status.connect(statusHandler)
        
        # Initialize dates
        self.initializeDates()

    def setupUI(self):
        """Setup all UI components"""
        self.icons = self.getIcons()
        self.textSearch = {'apply': 'Search', 'cancel': 'Cancel'}
        
        # Create search layout
        searchLayout = self.createSearchLayout()
        self.mainLayout.addLayout(searchLayout)
        
        # Add simple status text
        self.status = QLabel(self)
        self.mainLayout.addWidget(self.status)

    def createSearchLayout(self):
        """Create the search part of the UI"""
        layout = QVBoxLayout()  # Changed to QVBoxLayout to stack elements vertically
        
        # Add explanatory label
        dateLabel = QLabel("Search alerts by detection date:", self)
        dateLabel.setStyleSheet("font-weight: bold;")
        layout.addWidget(dateLabel)
        
        # Create horizontal layout for date controls
        controlsLayout = QHBoxLayout()
        
        # Dates layout
        datesLayout = QHBoxLayout()
        self.fromDate = self.createDateEdit('From', datesLayout, 'yyyy-MM-dd', True)
        self.toDate = self.createDateEdit('To', datesLayout, 'yyyy-MM-dd', True)
        controlsLayout.addLayout(datesLayout)
        
        # Days spinbox
        self.numDays = QSpinBox(self)
        self.numDays.setSingleStep(1)
        self.numDays.setSuffix(' Days')
        self.numDays.setRange(1, 360000)
        controlsLayout.addWidget(self.numDays)
        
        # Search button
        self.search = QPushButton(self.textSearch['apply'], self)
        self.search.setIcon(self.icons['apply'])
        self.search.clicked.connect(self._onSearch)
        controlsLayout.addWidget(self.search)

        # Add the controls layout to the main layout
        layout.addLayout(controlsLayout)
                
        return layout

    def getIcons(self):
        fIcon = self.style().standardIcon
        return {
            'apply': fIcon( QStyle.SP_DialogApplyButton ),
            'cancel': fIcon( QStyle.SP_DialogCancelButton )
        }

    def createDateEdit(self, name, layout, displayFormat, hasCalendar):
        layout.addWidget( QLabel( name ) )
        w = QDateEdit( self )
        w.setCalendarPopup( True )
        w.setDisplayFormat( displayFormat )
        w.setCalendarPopup( hasCalendar )
        layout.addWidget( w )
        return w

    def initializeDates(self):
        def setSpin(date1, date2):
            self.numDays.valueChanged.disconnect(changedNumDay)
            days = date1.daysTo(date2)
            self.numDays.setValue(days)
            self.numDays.valueChanged.connect(changedNumDay)

        @pyqtSlot(QDate)
        def changedFromDate(date):
            self.toDate.setMinimumDate(date.addDays(+1))
            setSpin(date, self.toDate.date())

        @pyqtSlot(QDate)
        def changedToDate(date):
            self.fromDate.setMaximumDate(date.addDays(-1))
            setSpin(self.fromDate.date(), date)

        @pyqtSlot(int)
        def changedNumDay(days):
            newDate = self.toDate.date().addDays(-1 * days)
            self.fromDate.dateChanged.disconnect(changedFromDate)
            self.fromDate.setDate(newDate)
            self.toDate.setMinimumDate(newDate.addDays(+1))
            self.fromDate.dateChanged.connect(changedFromDate)

        # Use stored dates if they exist, otherwise use defaults
        if self.last_from_date and self.last_to_date:
            self.fromDate.setDate(self.last_from_date)
            self.toDate.setDate(self.last_to_date)
            self.numDays.setValue(self.last_from_date.daysTo(self.last_to_date))
        else:
            d2 = QDate.currentDate()
            d1 = d2.addMonths(-1)
            self.fromDate.setDate(d1)
            self.fromDate.setMaximumDate(d2.addDays(-1))
            self.toDate.setDate(d2)
            self.toDate.setMinimumDate(d1.addDays(+1))
            self.numDays.setValue(d1.daysTo(d2))
        
        # Connect signals
        self.fromDate.dateChanged.connect(changedFromDate)
        self.toDate.dateChanged.connect(changedToDate)
        self.numDays.valueChanged.connect(changedNumDay)

    @pyqtSlot(bool)
    def _onSearch(self, checked):
        # Store dates in layer properties before search
        self.layer.setCustomProperty('mapbiomasalert/from_date', 
                                   self.fromDate.date().toString(Qt.ISODate))
        self.layer.setCustomProperty('mapbiomasalert/to_date', 
                                   self.toDate.date().toString(Qt.ISODate))
        
        if not self.api.tokenOk:
            dlg = DialogEmailPassword("MapBiomas Alert Login", False)
            if not dlg.exec_():
                return
                
            params = dlg.getParams()
            if not dlg.isValidEmail():
                self.msgBar.pushMessage("Invalid email format", Qgis.Critical)
                self.status.setText("Error: Invalid email format")
                return
                
            self.api.setToken(params['email'], params['password'])
            if not self.api.tokenOk:
                return
                
            if not dlg.isCheckedSave():
                self.clearAfterSearch = True

        # Continue with search...
        if self.api.taskAlerts:
            self.api.cancelAlerts()
            self.search.setIcon(self.icons['apply'])
            self.status.setText("Search cancelled")
            return

        # Show processing message
        processing_msg = "Processing your request. This may take a while for large date ranges..."
        self.status.setText(processing_msg)
        self.msgBar.pushMessage(
            "MapBiomas Alert", 
            processing_msg,
            Qgis.Info,
            duration=0  # Message stays until explicitly cleared
        )
        QApplication.processEvents()
        
        self.search.setIcon(self.icons['cancel'])
        self.alert.setLayer(self.fromDate.date().toString(Qt.ISODate), self.toDate.date().toString(Qt.ISODate))
        ids = self.layerTerritory.getIdsCanvas()
        status_msg = 'Fetching alerts from map extent and dates...'
        self.status.setText(status_msg)
        url = self.api.getUrlAlerts(self.fromDate.date().toString(Qt.ISODate), self.toDate.date().toString(Qt.ISODate))
        print('url', url )
        self.api.getAlertsWFSnonThread(url, self.alert, self.fromDate.date().toString(Qt.ISODate), self.toDate.date().toString(Qt.ISODate), ids)
        
        # Clear the processing message and show success
        self.msgBar.clearWidgets()
        success_msg = "Alerts successfully added to layer"
        self.status.setText(success_msg)
        self.msgBar.pushMessage("MapBiomas Alert", success_msg, Qgis.Success)

    @pyqtSlot()
    def _onFinishedAlert(self):
        self.search.setIcon(self.icons['apply'])
        self.status.setText("")  # Clear status message when finished
        
        # Restore the dates if they were saved
        if self.last_from_date and self.last_to_date:
            self.fromDate.setDate(self.last_from_date)
            self.toDate.setDate(self.last_to_date)
            # Update the days spinbox
            self.numDays.setValue(self.last_from_date.daysTo(self.last_to_date))
        
        if hasattr(self, 'clearAfterSearch') and self.clearAfterSearch:
            self.api.clearToken()
            delattr(self, 'clearAfterSearch')
            
        # Check if layer has features and zoom to them
        if self.layer and self.layer.featureCount() > 0:
            canvas = QgsUtils.iface.mapCanvas()
            extent = self.layer.extent()
            extent.scale(1.1)
            canvas.setExtent(extent)
            canvas.refresh()

    def __del__(self):
        """Clean up when widget is destroyed"""
        if hasattr(self, 'layer') and self.layer:
            self.api.disconnectAlerts(self.layer.id())


class LayerMapBiomasAlertWidgetProvider(QgsLayerTreeEmbeddedWidgetProvider):
    def __init__(self, api, localSetting):
        super().__init__()
        self.layerTerritory = TerritoryBbox()
        self.layerTerritory.setLayer()
        self.api = api
        self.localSetting = localSetting

    def id(self):
        return self.__class__.__name__

    def name(self):
        return "Layer MapBiomas Alert"

    def createWidget(self, layer, widgetIndex):
        return MapBiomasAlertWidget(layer, self.layerTerritory, self.api, self.localSetting)

    def supportsLayer(self, layer):
        return bool( layer.customProperty( MapBiomasAlert.MODULE, 0) )


class MapBiomasAlert(QObject):
    MODULE = 'MapBiomasAlert'
    def __init__(self, iface, api):
        super().__init__()        
        # Enable Python macros for the session
        QgsSettings().setEnumValue("qgis/enableMacros", Qgis.PythonMacroMode.SessionOnly)
        
        self.project = QgsProject.instance()
        self.msgBar = iface.messageBar()
        self.widgetProvider = None
        self.layer = None
        self.canvas = iface.mapCanvas()
        self.styleFile = os.path.join(os.path.dirname(__file__), 'mapbiomas_alert.qml')
        self.api = api
        self.localSetting = 'mapbiomas_alert_plugin/{}'
        self._connected_alerts = False

    def register(self):
        self.widgetProvider = LayerMapBiomasAlertWidgetProvider(self.api, self.localSetting)
        registry = QgsGui.layerTreeEmbeddedWidgetRegistry()
        if bool( registry.provider( self.widgetProvider.id() ) ):
            registry.removeProvider( self.widgetProvider.id() )
        registry.addProvider( self.widgetProvider )

    def addLayerRegisterProperty(self, layer):
        totalEW = int( layer.customProperty('embeddedWidgets/count', 0) )
        layer.setCustomProperty('embeddedWidgets/count', totalEW + 1 )
        layer.setCustomProperty(f"embeddedWidgets/{totalEW}/id", self.widgetProvider.id() )
        layer.setCustomProperty( self.MODULE, 1)

        # Comment out or remove QML loading if this programmatic style is preferred
        # layer.loadNamedStyle( self.styleFile )

        # Apply programmatic styling
        # Create a fill symbol for red outline, no fill
        symbol = QgsFillSymbol.createSimple({
            'color': 'transparent',
            'style': 'no',
            'outline_style': 'solid',
            'outline_width': '0.5',
            'outline_width_unit': 'MM',
            'outline_color': 'red'
        })

        # Create a single symbol renderer with this symbol
        renderer = QgsSingleSymbolRenderer(symbol)

        # Apply the renderer to the layer
        layer.setRenderer(renderer)
        
        FORM_setForm( layer )
        self.project.addMapLayer( layer )

    def run(self):
        print("\n[Main] Starting MapBiomas Alert plugin...")
        if not self.api.tokenOk:
            params = DialogEmailPassword.getConfig(self.localSetting)
            
            if params['email'] is not None:
                print("[Main] Found saved credentials, attempting authentication")
                self.api.setToken(params['email'], params['password'])
            
            if not self.api.tokenOk:
                print("[Main] No valid token, showing login dialog")
                dlg = DialogEmailPassword("MapBiomas Alert Login", False)
                if not dlg.exec_():
                    return
                
                params = dlg.getParams()
                if not dlg.isValidEmail():
                    self.msgBar.pushMessage("Invalid email format", Qgis.Critical)
                    return
                
                self.api.setToken(params['email'], params['password'])
                if not self.api.tokenOk:
                    return
                
                if not dlg.isCheckedSave():
                    self.clearTokenAfterSetup = True
        
        print("[Main] Adding alerts to layer")
        layer = DbAlerts.createLayer()
        self.addLayerRegisterProperty(layer)
        
        if hasattr(self, 'clearTokenAfterSetup') and self.clearTokenAfterSetup:
            print("[Main] Clearing temporary token after setup")
            self.api.clearToken()
            delattr(self, 'clearTokenAfterSetup')

    def actionsForm(self, nameAction, feature_id=None):
        """
        Run action defined in layer, provide by style file
        :param nameAction: Name of action
        :params feature_id: Feature ID
        """
        # Actions functions
        def flash(feature_id):
            geom = self.alert.getFeature( feature_id ).geometry()
            self.mapCanvasGeom.flash( [ geom ], self.alert )
            return { 'isOk': True }

        def zoom(feature_id):
            geom = self.alert.getFeature( feature_id ).geometry()
            self.mapCanvasGeom.zoom( [ geom ], self.alert )
            return { 'isOk': True }

        def report(feature_id):
            feat = self.alert.getFeature(feature_id)
            alerta_id = feat['alertCode']
            car_codes = feat['carCode']
            if len(car_codes) == 0:
                url = "{}/{}".format(API_MapbiomasAlert.urlReport, alerta_id)
                QDesktopServices.openUrl(QUrl(url))
            else:
                for car_code in car_codes.split(','):
                    url = "{}/{}/car/{}".format(API_MapbiomasAlert.urlReport, alerta_id, car_code)
                    QDesktopServices.openUrl(QUrl(url))
            return { 'isOk': True }

        actionsFunc = {
            'flash':  flash,
            'zoom':   zoom,
            'report': report
        }
        if not nameAction in actionsFunc.keys():
            return { 'isOk': False, 'message': "Missing action '{}'".format( nameAction ) }
        return actionsFunc[ nameAction ]( feature_id )

    def connectAlerts(self, dbAlerts):
        if not self._connected_alerts:
            print("[API] Connecting alerts signals")
            self.alerts.connect(dbAlerts.addFeatures)
            self._connected_alerts = True
