#!/usr/bin/python3
# # -*- coding: utf-8 -*-
"""
/***************************************************************************
Name                 : DialogEmailPassword
Description          : Class for Dialog with email and password
Date                 : April, 2021
copyright            : (C) 2021 by Luiz Motta
email                : motta.luiz@gmail.com

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import (
    QSettings,
    QRegExp, QRegularExpression,
)
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QLabel, QLineEdit,
    QCheckBox, QRadioButton,
    QDialogButtonBox,
    QHBoxLayout, QVBoxLayout
)
from qgis.PyQt.QtGui import QRegularExpressionValidator

from qgis.gui import QgsPasswordLineEdit
from qgis.core import Qgis
from qgis import utils as QgsUtils


class DialogEmailPassword(QDialog):
    def __init__(self, title, hasRegister):
        def setupUI():
            def createLayoutPassword():
                layout = QHBoxLayout()

                # Add help text at the top
                helpText = QLabel(
                    "If you need to create an account, visit our page "
                    "<a href='https://plataforma.alerta.mapbiomas.org/sign-up'>https://plataforma.alerta.mapbiomas.org/sign-up</a> "
                    "to create one or use <a href='https://plataforma.alerta.mapbiomas.org/api/docs/index.html#mutation-signIn'>"
                    "https://plataforma.alerta.mapbiomas.org/api/docs/index.html#mutation-signIn</a> "
                    "with your current account to get your Bearer Token.",
                    self
                )
                helpText.setWordWrap(True)
                helpText.setOpenExternalLinks(True)
                layout.addWidget(helpText)

                # Email/Password layout
                credentialsLayout = QHBoxLayout()
                # Email
                w = QLabel('Email/Password:', self)
                credentialsLayout.addWidget(w)
                w = QLineEdit(self)
                self.__dict__['email'] = w
                w.setPlaceholderText('someone@example.com')
                w.setToolTip('Email registered at MapBiomas Alert platform')
                rx = QRegularExpression(
                    self.emailExpEdit, QRegularExpression.CaseInsensitiveOption)
                w.setValidator(QRegularExpressionValidator(rx))
                credentialsLayout.addWidget(w)
                # Password
                w = QgsPasswordLineEdit(self)
                self.__dict__['password'] = w
                w.setToolTip('Password registered at MapBiomas Alert platform')
                w.setEchoMode(QLineEdit.Password)
                credentialsLayout.addWidget(w)
                # Save
                w = QCheckBox('Save credentials', self)
                self.__dict__['register_email_password'] = w
                w.setToolTip('Save credentials in QGIS configuration')
                w.setChecked(True)
                credentialsLayout.addWidget(w)

                # Add credentials layout below help text
                mainLayout = QVBoxLayout()
                mainLayout.addWidget(helpText)
                mainLayout.addLayout(credentialsLayout)

                return mainLayout

            def createLayoutRegister():
                layout = QHBoxLayout()
                # Register
                w = QRadioButton('Copy credentials', self)
                self.__dict__['clipboard'] = w
                w.setToolTip('Copy saved credentials to clipboard')
                layout.addWidget(w)
                w = QRadioButton('Clear credentials', self)
                self.__dict__['clear'] = w
                w.setToolTip('Remove saved credentials')
                layout.addWidget(w)
                return layout

            lyt = QVBoxLayout()
            f = createLayoutPassword
            subLayout = f()
            lyt.addLayout(subLayout)
            # Ok and Cancel buttons
            w = QDialogButtonBox(QDialogButtonBox.Ok |
                                 QDialogButtonBox.Cancel, self)
            w.button(QDialogButtonBox.Ok).setText("OK")
            w.button(QDialogButtonBox.Cancel).setText("Cancel")
            w.accepted.connect(self.accept)
            w.rejected.connect(self.reject)
            lyt.addWidget(w)
            #
            self.setLayout(lyt)

        super().__init__(QgsUtils.iface.mainWindow())
        self.setWindowTitle(title)
        self.emailExpEdit = '\\b[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,4}\\b'
        self.emailExpMath = r"(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)"
        setupUI()

    def getParams(self):
        params = {}
        for k in ('email', 'password'):
            params[k] = self.__dict__[k].text()
        return params

    def isValidEmail(self):
        rx = QRegExp(self.emailExpMath)
        return rx.exactMatch(self.email.text())

    def isCheckedClipboard(self):
        return self.clipboard.isChecked()

    def isCheckedClear(self):
        return self.clear.isChecked()

    def isCheckedSave(self):
        checked = self.register_email_password.isChecked()
        return checked

    @staticmethod
    def getConfig(localSetting):
        s = QSettings()
        params = {}
        for k in ('email', 'password'):
            value = s.value(localSetting.format(k), None)
            params[k] = value
        return params

    @staticmethod
    def setConfig(localSetting, email, password):
        s = QSettings()
        s.setValue(localSetting.format('email'), email)
        s.setValue(localSetting.format('password'), password)

    @staticmethod
    def clearConfig(localSetting):
        s = QSettings()
        s.remove(localSetting.format('email'))
        s.remove(localSetting.format('password'))

    @staticmethod
    def copy2Clipboard(email, password):
        msg = f"{email}/{password}"
        clipboard = QApplication.clipboard()
        clipboard.setText(msg)


def runDialogEmailPassword(title, api, localSetting):
    print("\n[Login] Starting email/password dialog")
    msgBar = QgsUtils.iface.messageBar()

    def messageHandler(msg, level):
        msgBar.pushMessage(msg, level)
    api.message.connect(messageHandler)

    def checkHasRegister():
        hasRegister = True
        if api.tokenOk:
            return hasRegister
        params = DialogEmailPassword.getConfig(localSetting)
        hasRegister = not params['email'] is None
        if hasRegister:
            print("[Login] Found saved credentials, attempting authentication")
            api.setToken(**params)
            if not api.tokenOk:
                DialogEmailPassword.clearConfig(localSetting)
                hasRegister = False
        return hasRegister

    hasRegister = checkHasRegister()
    dlg = DialogEmailPassword(title, hasRegister)
    result = dlg.exec_()
    if not result:
        return

    # Save clipboard or Clear
    if hasRegister:
        if dlg.isCheckedClipboard():
            params = DialogEmailPassword.getConfig(localSetting)
            dlg.copy2Clipboard(**params)
        if dlg.isCheckedClear():
            dlg.clearConfig(localSetting)
        return

    # Get params and test token
    params = dlg.getParams()

    # Add input validation
    if not params['email'] or not params['password']:
        msgBar.pushMessage("Email and password cannot be empty", Qgis.Critical)
        return

    if not dlg.isValidEmail():
        msgBar.pushMessage(
            f"Invalid email format: {params['email']}", Qgis.Critical)
        return

    api.setToken(params['email'], params['password'])

    # Only save credentials if checkbox is checked
    if dlg.isCheckedSave():
        if 'sendMessage' in params:
            del params['sendMessage']
        DialogEmailPassword.setConfig(
            localSetting, params['email'], params['password'])
