# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OverlapClipper
                                 A QGIS plugin
 This modules clips features
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-02
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Benjamin Dadson
        email                : benjamindadson32@gmail.com
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QButtonGroup, QTreeWidgetItem, QCheckBox
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .overlap_clipper_dockwidget import OverlapClipperDockWidget, DialogWindow
from .algorithms import clean_overlap, get_feature_area, clean_geometry_artifacts
import os.path
from qgis.core import QgsFeatureRequest, QgsProject, QgsFeature, QgsGeometry, Qgis, QgsWkbTypes
from itertools import combinations

from qgis.core import (
    QgsProcessingFeatureSourceDefinition,
    QgsProcessing,
    QgsProject,
    QgsProcessingAlgorithm
)
from qgis import processing


class OverlapClipper:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """

        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'OverlapClipper_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Overlap Clipper')
        self.toolbar = self.iface.addToolBar(u'OverlapClipper')
        self.toolbar.setObjectName(u'OverlapClipper')

        self.pluginIsActive = False
        self.dockwidget = None
        self.layer = None  # Initialize layer attribute

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('OverlapClipper', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/overlap_clipper/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Overlap Clipper'),
            callback=self.run,
            parent=self.iface.mainWindow())

    # --------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # disconnects
        if self.dockwidget:
            self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Overlap Clipper'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                self.layer = self.iface.activeLayer()
                layer = self.layer
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = OverlapClipperDockWidget()

                # Make necessary Events & Signals connections
                self.dockwidget.pushButton.clicked.connect(self.do_clip)
                # self.dockwidget.pushButton_2.clicked.connect(self.open_dialog)

                self.button_group = QButtonGroup(self.dockwidget)
                self.button_group.addButton(self.dockwidget.firstSelected)
                self.dockwidget.firstSelected.clicked.connect(
                    lambda: self.change_overlap_option(self.dockwidget.firstSelected))
                self.button_group.addButton(self.dockwidget.largest)
                self.dockwidget.largest.clicked.connect(
                    lambda: self.change_overlap_option(self.dockwidget.largest))
                self.button_group.addButton(self.dockwidget.smallest)
                self.dockwidget.smallest.clicked.connect(
                    lambda: self.change_overlap_option(self.dockwidget.smallest))

                # Populate Tree widget
                if self.layer:
                    self.dockwidget.treeWidget.setHeaderLabel(layer.name())
                self.populate_treewidget()

                self.selection_order = []
                self.overlap_option = 'First'

                self.iface.layerTreeView().currentLayerChanged.connect(self.on_active_layer_changed)

                if self.layer:
                    self.layer.selectionChanged.connect(self.track_selection_order)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def log_message(self, message, level=0):
        """
        Helper function to show logs on the screen (QGIS message bar).
        level: 0=Info, 1=Warning, 2=Critical
        """
        if level == 0:
            self.iface.messageBar().pushMessage("OverlapClipper Info", message, level=Qgis.MessageLevel.Info, duration=5)
        elif level == 1:
            self.iface.messageBar().pushMessage("OverlapClipper Warning", message, level=Qgis.MessageLevel.Warning,
                                                duration=5)
        elif level == 2:
            self.iface.messageBar().pushMessage("OverlapClipper Error", message, level=Qgis.MessageLevel.Critical,
                                                duration=10)

        print(f"OverlapClipper Log: {message}")  # Also print to console for debugging

    def pair_features(self):
        """
        Generates pairs of features that overlap and need to be processed.
        Implements the new logic based on the 'Edit all intersecting features' checkbox.

        Returns:
            list: A list of (fid1, fid2) tuples representing the pairs to process.
        """
        if not self.layer:
            # self.log_message("No active layer found.", 1)
            return []

        selected_features = self.layer.selectedFeatures()
        if not selected_features:
            # self.log_message("No features selected!", 1)
            return []

        # Determine the set of features to process based on the 'Edit all intersecting' checkbox.
        edit_all_intersecting = self.dockwidget.check_all_intersecting.isChecked()

        if edit_all_intersecting:
            # New logic: Find all features in the active layer that intersect with any selected feature.
            # self.log_message("Checking for all intersecting features in the active layer.", 0)

            # Features to be considered for pairing (start with selected features)
            features_to_process = {f.id(): f for f in selected_features}

            # Iterate over selected features to find all intersecting neighbors
            for selected_feat in selected_features:
                # Use a spatial filter (bounding box) for efficient initial query
                request = QgsFeatureRequest().setFilterRect(selected_feat.geometry().boundingBox())

                # Iterate over features that intersect the bounding box
                for feature in self.layer.getFeatures(request):
                    # Skip if it's already in our list
                    if feature.id() in features_to_process:
                        continue

                    # Perform a precise geometric intersection check
                    try:
                        if feature.geometry().intersects(selected_feat.geometry()):
                            features_to_process[feature.id()] = feature
                    except Exception as e:
                        self.log_message(f"Error checking intersection for feature {feature.id()}: {e}", 1)

            # Get FIDs of all features to process
            fids_to_process = [f.id() for f in features_to_process.values()]

            if len(fids_to_process) < 2:
                self.log_message("Less than two features found for processing (selected + intersecting).", 1)
                return []

        else:
            # Original logic: Only selected features are considered.
            # self.log_message("Processing only selected features.", 0)
            if len(selected_features) < 2:
                self.log_message("Please select at least two features to process.", 1)
                return []

            fids_to_process = [f.id() for f in selected_features]

        # Apply sorting logic based on the selected overlap option
        if self.overlap_option == "First":
            # Sort by selection order (maintains the order in which features were selected)
            fids_sorted = self.sort_selection_order(fids_to_process)
        else:
            # Sort by area (ascending for smallest, descending for largest)
            fids_sorted = self.sort_fids_by_area(fids_to_process)
            if self.overlap_option == "Largest":
                fids_sorted.reverse()

        # Create combinations of two features from the sorted list
        # The order in the tuple (f_id1, f_id2) is determined by priority (f_id1 is the clipper).
        fid_combinations = list(combinations(fids_sorted, 2))

        # Filter for actual overlaps to avoid unnecessary processing
        overlapping_combinations = []
        for f_id1, f_id2 in fid_combinations:
            try:
                feature1 = next(self.layer.getFeatures(QgsFeatureRequest(f_id1)), None)
                feature2 = next(self.layer.getFeatures(QgsFeatureRequest(f_id2)), None)

                if feature1 and feature2 and feature1.geometry() and feature2.geometry():
                    if feature1.geometry().intersects(feature2.geometry()):
                        overlapping_combinations.append((f_id1, f_id2))
            except Exception as e:
                self.log_message(f"Error checking overlap between {f_id1} and {f_id2}: {e}", 2)

        if not overlapping_combinations:
            self.log_message("No overlapping feature pairs found.", 0)
            return []

        return overlapping_combinations

    def do_clip(self):
        """
        Executes the main overlap cleaning process.
        """
        if not self.layer:
            # self.log_message("No active layer selected.", 2)
            return

        # Check if the layer is a polygon layer before proceeding
        if self.layer.geometryType() != QgsWkbTypes.PolygonGeometry:
            # self.log_message("The active layer is not a polygon layer. Operation aborted.", 2)
            return

        if not self.layer.isEditable():
            # self.log_message("Layer is not in editing mode. Starting edit session.", 0)
            self.layer.startEditing()

        try:
            fid_combinations = self.pair_features()
        except Exception as e:
            self.log_message(f"An error occurred during feature pairing: {e}", 2)
            return

        if not fid_combinations:
            # self.log_message("Clip operation aborted: No feature pairs to process.", 1)
            return

        # self.log_message(f"Starting clip operation on {len(fid_combinations)} feature pairs.", 0)

        processed_count = 0
        for f_id1, f_id2 in fid_combinations:
            # f_id1 is the dominant feature (clipper), f_id2 is the clipped feature
            if self.modify_features(f_id1, f_id2):
                processed_count += 1

        # Commit changes and clean up
        if processed_count > 0:
            try:
                self.layer.commitChanges()
                # self.log_message(f"Successfully clipped {processed_count} feature pairs. Changes committed.", 0)
            except Exception as e:
                self.log_message(f"Failed to commit changes: {e}. Rolling back.", 2)
                self.layer.rollBack()
        else:
            self.layer.rollBack()
            self.log_message("No features were modified. Edit session rolled back.", 0)

        self.layer.removeSelection()
        self.populate_treewidget()  # Refresh the tree widget

    def modify_features(self, f_id1, f_id2):
        """
        Applies the overlap cleaning logic for a single pair of features.
        f_id1 is the feature that maintains its geometry (clipper).
        f_id2 is the feature that gets clipped (clipped).

        Returns:
            bool: True if feature2 was modified and updated, False otherwise.
        """
        layer = self.layer

        try:
            # Get the two features by their FID (Feature ID)
            feature1 = next(layer.getFeatures(QgsFeatureRequest(f_id1)), None)
            feature2 = next(layer.getFeatures(QgsFeatureRequest(f_id2)), None)

            if not feature1 or not feature2:
                self.log_message(f"Could not retrieve features with IDs {f_id1} or {f_id2}. Skipping pair.", 1)
                return False

            # Get the geometries of the two features
            geom1 = feature1.geometry()
            geom2 = feature2.geometry()

            # Handle null geometries
            if geom1.isNull() or geom2.isNull():
                self.log_message(f"One or both features ({f_id1}, {f_id2}) have null geometry. Skipping.", 1)
                return False

            # Pre-clean geometries to ensure robustness
            geom1 = clean_geometry_artifacts(geom1)
            geom2 = clean_geometry_artifacts(geom2)

            # Step 1: Check if they overlap
            if not geom1.intersects(geom2):
                # self.log_message(f"Features {f_id1} and {f_id2} do not overlap. Skipping.", 0)
                return False

            # Step 2: Apply the core overlap cleaning logic
            # f_id2 is the one that gets clipped by f_id1
            modified_geom2 = clean_overlap(geom2, geom1)

            # Check if the geometry actually changed (e.g., if it was completely inside geom1)
            if modified_geom2.equals(geom2):
                self.log_message(f"Feature {f_id2} was not modified by clipping with {f_id1}.", 0)
                return False

            # Step 3: Update the geometry of feature2 in the layer
            feature2.setGeometry(modified_geom2)

            # Update the feature in the layer's edit buffer
            if not layer.updateFeature(feature2):
                self.log_message(f"Failed to update feature {f_id2} in the layer's edit buffer.", 2)
                return False

            # self.log_message(f"Feature {f_id2} clipped by Feature {f_id1}.", 0)
            return True

        except Exception as e:
            self.log_message(f"An unexpected error occurred while modifying features {f_id1} and {f_id2}: {e}", 2)
            return False

    # --- Utility methods (mostly unchanged, but simplified/cleaned) ---

    def track_selection_order(self, added, removed, changed):
        """
        This method is triggered when selection changes.
        """
        # Populate treeWidget
        self.populate_treewidget()

        # Add newly selected features to the order list
        for fid in added:
            if fid not in self.selection_order:
                self.selection_order.append(fid)

        # Remove deselected features from the order list
        for fid in removed:
            if fid in self.selection_order:
                self.selection_order.remove(fid)

    def sort_selection_order(self, current_selection_fids):
        """
        Sorts a list of FIDs based on the original selection order,
        placing unselected but intersecting features at the end.
        """
        selection_order_set = set(self.selection_order)
        current_selection_set = set(current_selection_fids)

        # 1. FIDs that were originally selected (and thus have an order)
        in_selection_order = [fid for fid in current_selection_set if fid in selection_order_set]

        # 2. FIDs that were not originally selected (newly included intersecting features)
        not_in_selection_order = [fid for fid in current_selection_set if fid not in selection_order_set]

        # Sort the originally selected FIDs based on the selection_order list
        in_selection_order_sorted = sorted(in_selection_order, key=lambda x: self.selection_order.index(x))

        # Combine: originally selected (in order) followed by newly included (no specific order)
        result = in_selection_order_sorted + not_in_selection_order

        return result

    def sort_fids_by_area(self, fid_list):
        """
        Sorts a list of FIDs based on the area of their corresponding geometries in ascending order.
        """
        layer = self.layer
        if not layer or not fid_list:
            return []

        fid_area_list = []
        for fid in fid_list:
            feature = next(layer.getFeatures(QgsFeatureRequest(fid)), None)
            if feature:
                area = get_feature_area(feature)
                fid_area_list.append((fid, area))

        # Sort the list by area in ascending order
        sorted_fid_area_list = sorted(fid_area_list, key=lambda x: x[1])

        # Extract only the sorted FIDs
        sorted_fid_list = [fid for fid, area in sorted_fid_area_list]

        return sorted_fid_list

    def change_overlap_option(self, radioButton):
        if radioButton.isChecked():
            self.overlap_option = radioButton.text().split()[0]

    def populate_treewidget(self):
        """
        Populates the QTreeWidget with the selected features and handles UI state.
        """
        treewidget = self.dockwidget.treeWidget
        treewidget.clear()

        layer = self.layer
        if not layer:
            # Clear header if no layer is active
            treewidget.setHeaderLabel("No Active Layer")
            return

        # Check if the layer is a polygon layer
        if layer.geometryType() != QgsWkbTypes.PolygonGeometry:
            treewidget.setHeaderLabel(f"Layer: {layer.name()} (Not Polygon)")
            return

        # If it is a polygon layer, set the header and proceed with population
        treewidget.setHeaderLabel(f"Layer: {layer.name()}")

        selected_features = layer.selectedFeatures()
        if not selected_features:
            return

        for feature in selected_features:
            fid_str = f"Feature ID: {feature.id()}"
            feature_item = QTreeWidgetItem(treewidget)
            feature_item.setText(0, fid_str)

            attributes = feature.attributes()
            fields = layer.fields()

            for idx, field in enumerate(fields):
                field_name = field.name()
                field_value = attributes[idx]
                attribute_item = QTreeWidgetItem(feature_item)
                attribute_item.setText(0, f"{field_name}: {field_value}")

    def on_active_layer_changed(self, new_layer):
        """
        Updates the plugin state when the active layer changes.
        This method handles connecting signals and updating the UI based on the layer type.
        """
        # 1. Disconnect old layer signal
        if self.layer and hasattr(self.layer, 'selectionChanged'):
            try:
                self.layer.selectionChanged.disconnect(self.track_selection_order)
            except (TypeError, RuntimeError):
                pass # Signal was not connected or already disconnected

        self.layer = new_layer
        self.clear_selection_order()

        # 2. Check layer validity and type
        is_polygon_layer = False
        if new_layer and new_layer.isValid():
            # Check if it's a vector layer (QgsWkbTypes.WkbType.WKBPoint is an arbitrary WkbType, better to check layer type)
            # A more robust check is to see if it's a QgsVectorLayer and check its geometry type
            try:
                if new_layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                    is_polygon_layer = True
                    # self.log_message(f"Active layer changed to {new_layer.name()} (Polygon).", 0)
                else:
                    pass
                    # self.log_message(f"Active layer changed to {new_layer.name()} (Non-Polygon). Clip button disabled.", 1)
            except AttributeError:
                pass
                # Handle case where layer might not have geometryType (e.g., non-vector layer)
                # self.log_message(f"Active layer changed to {new_layer.name()} (Unsupported Type). Clip button disabled.", 1)
        else:
            pass
            # self.log_message("No active layer selected. Clip button disabled.", 1)

        # 3. Update UI state
        self.dockwidget.pushButton.setEnabled(is_polygon_layer)
        self.dockwidget.firstSelected.setEnabled(is_polygon_layer)
        self.dockwidget.largest.setEnabled(is_polygon_layer)
        self.dockwidget.smallest.setEnabled(is_polygon_layer)
        self.dockwidget.check_all_intersecting.setEnabled(is_polygon_layer)

        # 4. Update tree widget and connect new signal if applicable
        self.populate_treewidget()

        if is_polygon_layer:
            if hasattr(self.layer, 'selectionChanged'):
                self.layer.selectionChanged.connect(self.track_selection_order)
        else:
            # Clear tree widget header if not a polygon layer
            self.dockwidget.treeWidget.setHeaderLabel("Select a Polygon Layer")

    def clear_selection_order(self):
        # Clears the current selection order
        self.selection_order = []

    def open_dialog(self):
        # Create an instance of the dialog
        dialog = DialogWindow(self.dockwidget)
        # Show the dialog
        dialog.exec_()
