# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OsmAnd bridge
                                 A QGIS plugin
 Import tracks, favourites, itinerary and AV notes from OsmAnd
 Generated by Plugin Builder: https://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-10
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Sylvain Théry - UMR 5281 ART-Dev
        email                : sylvain.thery@cnrs.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path
import socket
from datetime import datetime

from qgis import processing
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, Qt

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QProgressBar, QApplication
from qgis.core import QgsWkbTypes, QgsField, QgsMessageLog, Qgis, QgsProject, QgsFields, QgsRasterLayer, \
    QgsRectangle, QgsCoordinateReferenceSystem, QgsSettings

# Initialize Qt resources from file resources.py
# from .resources import *
# Import the code for the dialog
from .OsmAnd_bridge_import_dialog import OsmAndBridgeImportDialog
# Import the code for the process
from .OsmAnd_bridge_import_process import import_gpx_track_file, import_avnotes, move_to_group
from .OsmAnd_bridge_geopackage_management import create_empty_gpkg_layer

# Pycharm debug server
# To use it, you need to use a 'python remote debug' configuration into pycharm *pro*
# Then 'pip install pydevd-pycharm~=221.5591.52' # at the time of writing (20022-05-27)
debug = False
if debug:
    try:
        import pydevd_pycharm

        pydevd_pycharm.settrace('localhost', port=53100, stdoutToServer=True, stderrToServer=True)
        print("Debugging into pycharm")
    except Exception as e:
        print("No remote debug configuration")


class OsmAndBridge:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """

        # Save reference to the QGIS interface
        self.iface = iface
        # Save reference to the QGIS project instance
        self.project = QgsProject.instance()
        # Save reference to the plugin name
        self.plugin_name = 'OsmAnd bridge'

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'OsmAnd_bridge_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&OsmAnd bridge')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Add a toolbar
        self.toolbar = self.iface.addToolBar(u'OsmAnd bridge')
        self.toolbar.setObjectName(u'OsmAnd bridge')

        # delai for maptip display
        QgsSettings().setValue('qgis/mapTipsDelay', 500)

    def tr(self, message):
        """
        Get the translation for a string using Qt translation API.
        We implement this ourselves since we do not inherit QObject.
        :param message: String for translation.
        :type message: str, QString
        :returns: Translated version of message.
        :rtype: QString
        """
        return QCoreApplication.translate('OsmAndBridge', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.
        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str
        :param text: Text that should be shown in menu items for this action.
        :type text: str
        :param callback: Function to be called when the action is triggered.
        :type callback: function
        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool
        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool
        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool
        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str
        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget
        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.
        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Add plugin icon to its own toolbar
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = f"{self.plugin_dir}/OsmAnd_logo.png"
        self.add_action(
            icon_path,
            text=self.tr(u"Import tracks, favourites, itinerary and AV notes"),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&OsmAnd bridge'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg_import = OsmAndBridgeImportDialog()

        # show the dialog
        self.dlg_import.show()
        # Run the dialog event loop
        result = self.dlg_import.exec_()
        # See if OK was pressed
        if result:
            ## define empty extent that will be updated when adding layers
            self.extent = QgsRectangle()
            self.extent.setMinimal()

            # sets the project CRS
            self.crs = QgsCoordinateReferenceSystem.fromEpsgId(4326)
            self.project.setCrs(self.crs)

            # get time to generate gpkg name, and project name if not already set
            now = datetime.now().strftime("%Y%m%d-%Hh%Mm%Ss")

            # affect or get filename and save project before continuing
            if self.project.fileName() == '':
                qgis_project_filename = f'{self.dlg_import.QgsFW_dest_path.filePath()}/{now}_OsmAnd_bridge.qgz'
            else:
                qgis_project_filename = self.project.fileName()
            self.project.write(qgis_project_filename)

            self.osmand_root_path = self.dlg_import.QgsFW_osmand_root_path.filePath()
            self.dest_path = self.dlg_import.QgsFW_dest_path.filePath()
            self.dest_gpkg = f'{self.dlg_import.QgsFW_dest_path.filePath()}/{now}_OsmAnd_bridge.gpkg'

            # Create dest directory if not exists
            os.makedirs(os.path.dirname(f'{self.dlg_import.QgsFW_dest_path.filePath()}'), exist_ok=True)

            # Work around to create GPKG file (with an empty table that will be removed)
            # see https://gis.stackexchange.com/a/417950
            schema = QgsFields()
            schema.append(QgsField("bool_field", QVariant.Bool))
            create_empty_gpkg_layer(self.dest_gpkg, "temp_table", QgsWkbTypes.NoGeometry, '', schema)
            if not os.path.exists(os.path.dirname(self.dest_gpkg)):
                message = self.tr(f'Issue when trying to create destination geopackage file ({self.dest_gpkg})')
                QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Critical)
                self.iface.messageBar().pushMessage(message, level=Qgis.Critical)
                return

            ## Now dealing with AV notes
            if self.dlg_import.cB_AVnotes.isChecked():
                # create destination folder if not exists
                os.makedirs(f'{self.dlg_import.QgsFW_dest_path.filePath()}/avnotes', exist_ok=True)

                import_avnotes(self, f"{self.osmand_root_path}/avnotes/")

            ## Now dealing with favourites gpx file
            if self.dlg_import.cB_favourites.isChecked():
                # user and log info
                file = 'favourites.gpx'
                message = self.tr(f"Importing favourites ({file})")
                self.iface.messageBar().pushMessage(message, level=Qgis.Info)
                QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Info)
                # Process and test result
                result = import_gpx_track_file(self, f'{self.osmand_root_path}/{file}')
                if not result:
                    self.iface.messageBar().clearWidgets()
                    message = self.tr(f'Something went wrong while importing favourites ({file})')
                    QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Critical)
                    self.iface.messageBar().pushMessage(message, level=Qgis.Critical)
                    return
            self.iface.messageBar().clearWidgets()

            ## Now dealing with itinerary gpx file
            if self.dlg_import.cB_itinerary.isChecked():
                # user and log info
                file = 'itinerary.gpx'
                message = self.tr(f"Importing itinerary ({file})")
                self.iface.messageBar().pushMessage(message, level=Qgis.Info)
                QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Info)
                # Process and test result
                result = import_gpx_track_file(self, f'{self.osmand_root_path}/{file}')
                if not result:
                    self.iface.messageBar().clearWidgets()
                    message = self.tr(f'Something went wrong while importing itinerary ({file})')
                    QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Critical)
                    self.iface.messageBar().pushMessage(message, level=Qgis.Critical)
                    return
            self.iface.messageBar().clearWidgets()

            ## Now dealing with selected gpx tracks files
            # We iterate thru selected row(s) of the gpx file table first to count files to import and prepare
            # a message bar
            i = 0
            for currentQTableWidgetItem in self.dlg_import.tW_tracks.selectedItems():
                if currentQTableWidgetItem.column() == 0:
                    i += 1
            progressMessageBar = self.iface.messageBar().createMessage(self.tr("Importing track files..."))
            progress = QProgressBar()
            progress.setMaximum(10)
            progress.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            progressMessageBar.layout().addWidget(progress)
            self.iface.messageBar().pushWidget(progressMessageBar, Qgis.Info)

            # We now iterate thru selected row(s) of the gpx file table to import data
            j = 0
            for currentQTableWidgetItem in self.dlg_import.tW_tracks.selectedItems():
                # We just need to get first column value (gpx filename)
                if currentQTableWidgetItem.column() == 0:
                    result = import_gpx_track_file(self,
                                                   f'{self.osmand_root_path}/tracks/rec/{currentQTableWidgetItem.text()}')
                    j += 1
                    progress.setValue(j)
                    if not result:
                        self.iface.messageBar().clearWidgets()
                        message = self.tr(f'Something went wrong while importing {currentQTableWidgetItem.text()}')
                        QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Critical)
                        self.iface.messageBar().pushMessage(message, level=Qgis.Critical)
                        return
            self.iface.messageBar().clearWidgets()

            ## Now deasling with map background
            # cheking internet connection
            try :
                socket.create_connection(("tile.openstreetmap.org", 443))
                tms = 'type=xyz&url=https://tile.openstreetmap.org/{z}/{x}/{y}.png&zmax=19&zmin=0'
                layer = QgsRasterLayer(tms, 'OpenStreetMap', 'wms')
                QgsProject.instance().addMapLayer(layer)
                move_to_group(layer, self.tr('Map background'))
            except:
                message = self.tr(f'No internet connection. Unable to load OSM tile background')
                QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Warning)
                self.iface.messageBar().pushMessage(message, level=Qgis.Warning)

            ## set map canvas extent
            QApplication.instance().processEvents() # needed but not safe (see https://issues.qgis.org/issues/19311)
            self.iface.mapCanvas().zoomToFeatureExtent(self.extent)
            self.iface.mapCanvas().refresh()

            # if present, remove the temp_layer previously created to genarete destination gpkg
            try:
                processing.run("native:spatialiteexecutesql",
                               {'DATABASE': f'{self.dest_gpkg}|layername=temp_table', 'SQL': 'drop table temp_table'})
            except Exception as e:
                pass

            # Show success message when finished
            self.iface.messageBar().clearWidgets()
            message = self.tr("♪♪ This is the End, my only friend, the End ♪♪")
            QgsMessageLog.logMessage(message, self.plugin_name, level=Qgis.Success)
            self.iface.messageBar().pushMessage(message, level=Qgis.Success, duration=0)

            # close dialog and save project
            self.dlg_import.close()
            self.project.write(qgis_project_filename)

