import os
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QDialog, QLineEdit, QWidget, QToolBar
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsVectorLayer, QgsProject, Qgis
from qgis.gui import QgisInterface
from . import resources_rc  # fichier généré par pyrcc5

# Chargement du fichier .ui
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), "ui", "openstardt_chargeGML.ui"
))

class LoadDialog(QDialog, FORM_CLASS):
    def __init__(self, onBrowse, onLoad, parent=None):
        super().__init__(parent)
        self.setupUi(self)
        # Repérer dynamiquement le QLineEdit pour le chemin
        edit_widgets = self.findChildren(QLineEdit)
        if edit_widgets:
            self.le_path = edit_widgets[0]
        else:
            self.le_path = None
        # Connexion des boutons
        for w in self.findChildren(QWidget):
            name = w.objectName().lower()
            if name.startswith('pushbrowse'):
                w.clicked.connect(onBrowse)
            if name.startswith('pushload') or name.startswith('pushok'):
                w.clicked.connect(onLoad)

class OpenStaRDTLoader:
    def __init__(self, iface: QgisInterface):
        self.iface = iface
        self.dialog = LoadDialog(self.chooseGML, self.loadGML)
        # Préparer l'action avec icône
        icon_path = os.path.join(os.path.dirname(__file__), 'icon_openstardt_load.svg')
        self.action = QAction(QIcon(icon_path), "Charger StaR-DT", iface.mainWindow())
        self.action.triggered.connect(self.showDialog)

    def initGui(self):
        """Appelé par QGIS au chargement du plugin : crée un menu OpenStaR et un sous-menu OpenStaR-DT"""
        # Menu OpenStaR
        # Accéder à la barre de menu principale
        menubar = self.iface.mainWindow().menuBar()
        # Tenter de récupérer le menu OpenStaR, sinon le créer
        existing = menubar.findChild(type(menubar), "&OpenStaR")
        if not existing:
            self.openstar_menu = menubar.addMenu("&OpenStaR")
        else:
            self.openstar_menu = existing
        
        # Menu OpenStaR-DT avec icône
        # Chemin de l'icône pour le sous-menu
        icon_osdt_path = os.path.join(os.path.dirname(__file__), 'icon_openstardt.svg')
        # Créer un sous-menu OpenStaR-DT
        self.submenu = self.openstar_menu.addMenu(QIcon(icon_osdt_path), "OpenStaR-DT")
        # Ajouter l'action dans ce sous-menu
        self.submenu.addAction(self.action)

        # Barre d'outils dédiée à OpenStaR-DT
        self.toolbar = QToolBar("OpenStaR-DT")
        self.toolbar.setObjectName("OpenStaR-DT")
        self.toolbar.addAction(self.action)
        self.iface.mainWindow().addToolBar(self.toolbar)

    def unload(self):
        """Appelé par QGIS à la désactivation du plugin : supprime le sous-menu et l'action"""
        # Supprimer l'action de la barre d'outils
        if self.toolbar:
            self.iface.mainWindow().removeToolBar(self.toolbar)
        # Retirer l'action du sous-menu
        if hasattr(self, 'submenu'):
            self.submenu.removeAction(self.action)
            # Si le sous-menu est vide, on le retire du menu OpenStaR
            if not self.submenu.actions():
                self.openstar_menu.removeAction(self.submenu.menuAction())
        # Si le menu OpenStaR est vide également, on peut le retirer
        if hasattr(self, 'openstar_menu') and not self.openstar_menu.actions():
            menubar = self.iface.mainWindow().menuBar()
            menubar.removeAction(self.openstar_menu.menuAction())

    def showDialog(self):
        self.dialog.show()

    def chooseGML(self):
        path, _ = QFileDialog.getOpenFileName(
            self.dialog, "Sélectionner un GML", "", "GML files (*.gml);;All files (*)"
        )
        if path and self.dialog.le_path:
            self.dialog.le_path.setText(path)
        elif path:
            self.iface.messageBar().pushMessage(
                "OpenStaR-DT", "Impossible d'afficher le chemin dans l'interface",
                level=Qgis.Warning, duration=10
            )

    def loadGML(self):
        # Vérifie que le chemin est disponible
        if not self.dialog.le_path:
            self.iface.messageBar().pushMessage(
                "OpenStaR-DT", "Champ de chemin introuvable.", level=Qgis.Critical, duration=10
            )
            return
        gml_path = self.dialog.le_path.text()
        if not os.path.exists(gml_path):
            self.iface.messageBar().pushMessage(
                "OpenStaR-DT", "Fichier invalide ou introuvable.",
                level=Qgis.Critical, duration=10
            )
            return

        # Lister sous-couches
        temp = QgsVectorLayer(gml_path, "temp", "ogr")
        subs = [s.split('!!::!!')[1] for s in temp.dataProvider().subLayers()]
        # Récupérer responsable
        lvl = QgsVectorLayer(f"{gml_path}|layername=ReseauUtilite", "ReseauUtilite", "ogr")
        feat = next(lvl.getFeatures(), None)
        responsable = feat["responsable"] if feat else "inconnu"
        theme = feat["theme"] if feat else "inconnu"
        from qgis.PyQt.QtCore import QDateTime
        dt = QDateTime.currentDateTime().toString("yyyy-MM-dd hh:mm:ss")
        group = self.iface.layerTreeView().layerTreeModel().rootGroup().addGroup(
            f"{responsable} {dt}"
        )

        # Charger couches
        for name in subs:
            uri = f"{gml_path}|layername={name}"
            lyr = QgsVectorLayer(uri, name, "ogr")
            if not lyr.isValid():
                self.iface.messageBar().pushMessage(
                    "OpenStaR-DT", f"Échec du chargement de {name}.",
                    level=Qgis.Critical, duration=10
                )
                continue
            # Appliquer style si disponible
            qml = os.path.join(os.path.dirname(__file__), "styles", theme, f"{name}.qml")
            
            if os.path.exists(qml):
                ret = lyr.loadNamedStyle(qml)
                if isinstance(ret, tuple) and len(ret) == 2 and isinstance(ret[1], bool):
                    _, ok = ret
                elif isinstance(ret, list):
                    ok = len(ret) == 0
                else:
                    ok = True
                if ok:
                    lyr.triggerRepaint()
                    self.iface.messageBar().pushMessage(
                        "OpenStaR-DT", f"Style appliqué à {name}.",
                        level=Qgis.Success, duration=10
                    )
                else:
                    self.iface.messageBar().pushMessage(
                        "OpenStaR-DT", f"Style ko pour {name}.",
                        level=Qgis.Warning, duration=10
                    )
            # Ajouter au projet
            QgsProject.instance().addMapLayer(lyr, False)
            group.addLayer(lyr)
        # Notification finale
        self.iface.messageBar().pushMessage(
            "OpenStaR-DT", "Chargement terminé.",
            level=Qgis.Success, duration=10
        )
