           # -*- coding: utf-8 -*-
"""
/***************************************************************************
 OnlineRoutingMapper
                                 A QGIS plugin
 Generate routes by using online services (Openrouteservice, GraphHopper , TomTom, Here, Yandex)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-25
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Mehmet Selim BILGIN, Yunus Emre ONDER
        email                : mselimbilgin@yahoo.com, eonder0@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QInputDialog, QLineEdit
from qgis.core import *
from qgis.gui import *

# Import the code for the dialog
from .onlineroutingmapper_dialog import OnlineRoutingMapperDialog
from .providers import get_provider_by_title, get_provider_titles


# Initialize Qt resources from file resources.py

class OnlineRoutingMapper:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.no = 0
        self.dialog = None
        self.click_tool = None
        self.start_rubber_band = None
        self.stop_rubber_band = None
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'OnlineRoutingMapper_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Online Routing Mapper')
        self.toolbar = self.iface.addToolBar(u'OnlineRoutingMapper')
        self.toolbar.setObjectName(u'OnlineRoutingMapper')
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('OnlineRoutingMapper', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # icon_path = ':/plugins/onlineroutingmapper/icon.png'
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Online Routing Mapper'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Online Routing Mapper'),
                action)
            self.iface.removeToolBarIcon(action)

    def click_handler(self, point):
        if self.no == 0:
            self.start_rubber_band.removeLastPoint()
            self.start_rubber_band.removeLastPoint()
            self.start_rubber_band.addPoint(point)
            self.dialog.startTxt.setText(str(point.x()) + ',' + str(point.y()))
        else:
            self.stop_rubber_band.removeLastPoint()
            self.stop_rubber_band.removeLastPoint()
            self.stop_rubber_band.addPoint(point)
            self.dialog.stopTxt.setText(str(point.x()) + ',' + str(point.y()))
        self.dialog.showNormal()

        # free them
        self.iface.mapCanvas().unsetMapTool(self.click_tool)
        self.click_tool.canvasClicked.disconnect(self.click_handler)

    def tool_activator(self, no):
        self.no = no
        self.dialog.showMinimized()
        self.click_tool.canvasClicked.connect(self.click_handler)
        self.iface.mapCanvas().setMapTool(self.click_tool)  # clickTool is activated

    def crs_transform(self, point):
        source_crs = self.iface.mapCanvas().mapSettings().destinationCrs()  # getting the project CRS
        destination_crs = QgsCoordinateReferenceSystem(4326)  # google uses this CRS
        transformer = QgsCoordinateTransform(source_crs, destination_crs,
                                             QgsProject.instance())  # defining a CRS transformer

        output_point = transformer.transform(point, QgsCoordinateTransform.ForwardTransform)
        return '{0},{1}'.format(output_point.y(), output_point.x())

    def show_apikey_dialog(self):
        provider = get_provider_by_title(self.dialog.serviceCombo.currentText())
        apikey = ''
        try:
            apikey = provider.get_apikey()
        except:
            pass

        value, ok = QInputDialog.getText(None, "Set API Key", provider.title + ":", QLineEdit.Normal, apikey)
        if ok:
            provider.set_apikey(value)

    def draw_route(self, wkt_linestring):
        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromWkt(wkt_linestring))
        vector_layer = QgsVectorLayer('LineString?crs=epsg:4326', 'Routing Result', 'memory')
        layer_provider = vector_layer.dataProvider()
        vector_layer.startEditing()
        layer_provider.addFeatures([feature])
        vector_layer.commitChanges()
        vector_layer.updateExtents()
        vector_layer.loadNamedStyle(self.plugin_dir + os.sep + 'OnlineRoutingMapper.qml')
        QgsProject.instance().addMapLayer(vector_layer)
        destination_crs = self.iface.mapCanvas().mapSettings().destinationCrs()  # getting the project CRS
        source_crs = QgsCoordinateReferenceSystem(4326)
        transformer = QgsCoordinateTransform(source_crs, destination_crs, QgsProject.instance())
        zoom_extent = transformer.transform(vector_layer.extent(), QgsCoordinateTransform.ForwardTransform)
        self.iface.mapCanvas().setExtent(zoom_extent)
        self.iface.mapCanvas().zoomScale(self.iface.mapCanvas().scale() * 1.03)  # zoom out a little bit.

    def run_analysis(self):
        if len(self.dialog.startTxt.text()) > 0 and len(self.dialog.stopTxt.text()) > 0:
            arr_start_point = self.dialog.startTxt.text().replace(" ", "").split(",")
            arr_stop_point = self.dialog.stopTxt.text().replace(" ", "").split(",")

            start_point = QgsPointXY(float(arr_start_point[0]), float(arr_start_point[1]))
            stop_point = QgsPointXY(float(arr_stop_point[0]), float(arr_stop_point[1]))

            start_point_wgs84 = self.crs_transform(start_point)
            stop_point_wgs84 = self.crs_transform(stop_point)

            try:
                provider = get_provider_by_title(self.dialog.serviceCombo.currentText())
                wkt = provider.solve(start_point_wgs84, stop_point_wgs84)
                self.draw_route(wkt)

                # clear rubberbands
                self.start_rubber_band.removeLastPoint()
                self.stop_rubber_band.removeLastPoint()
                self.start_rubber_band.removeLastPoint()
                self.stop_rubber_band.removeLastPoint()
            except Exception as err:
                QMessageBox.warning(self.dialog, "Error", str(err))
        else:
            QMessageBox.information(self.dialog, 'Warning', 'Please choose Start Location and Stop Location.')

    def generate_rubber_band(self):
        rubber_band = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PointGeometry)
        rubber_band.setColor(QColor("#000000"))
        rubber_band.setIconSize(7)
        rubber_band.setIcon(QgsRubberBand.ICON_FULL_BOX)
        return rubber_band

    def remove_rubber_bands(self, event):
        self.iface.mapCanvas().scene().removeItem(self.start_rubber_band)
        self.iface.mapCanvas().scene().removeItem(self.stop_rubber_band)

    def run(self):
        self.no = 0

        if self.first_start:
            self.first_start = False
            self.dialog = OnlineRoutingMapperDialog()

        self.start_rubber_band = self.generate_rubber_band()
        self.stop_rubber_band = self.generate_rubber_band()

        self.dialog = OnlineRoutingMapperDialog()
        self.dialog.setFixedSize(self.dialog.size())
        self.dialog.serviceCombo.addItems(get_provider_titles())
        self.click_tool = QgsMapToolEmitPoint(self.iface.mapCanvas())

        self.dialog.setApiKeyBtn.clicked.connect(self.show_apikey_dialog)
        self.dialog.startBtn.clicked.connect(lambda: self.tool_activator(0))
        self.dialog.stopBtn.clicked.connect(lambda: self.tool_activator(1))
        self.dialog.runBtn.clicked.connect(self.run_analysis)

        self.dialog.show()
        self.dialog.closeEvent = self.remove_rubber_bands
