# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OMRAT
                                 A QGIS plugin
 This is an open source implementation of Pedersens equations from 1995
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-12-22
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Axel Hörteborn
        email                : axel.horteborn@ri.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QVariant, QMetaType
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QMenu, QMenuBar
from qgis.core import (QgsVectorLayer, QgsFeature, QgsGeometry, QgsLineString, QgsPoint, QgsProject, 
                       QgsField, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsFields, QgsPalLayerSettings, QgsVectorLayerSimpleLabeling)
from qgis.core import QgsSingleSymbolRenderer, QgsLineSymbol, QgsPointXY, QgsMessageLog, Qgis

from qgis.gui import QgsMapToolPan
import gc
import sys
import os
from operator import xor
import os.path
sys.path.append('.')
# Initialize Qt resources from file resources.py
from resources import *

# Import the code for the DockWidget
from compute.run_calculations import Calculation
from geometries.handle_qgis_iface import HandleQGISIface
from omrat_utils.causation_factors import CausationFactors
from omrat_utils.handle_ais import AIS
from omrat_utils.handle_settings import DriftSettings
from omrat_utils.handle_traffic import Traffic
from omrat_utils.storage import Storage
from omrat_utils.handle_object import OObject
from omrat_utils.handle_ship_cat import ShipCategories
from omrat_utils.gather_data import GatherData
from omrat_widget import OMRATDockWidget


class OMRAT:
    """QGIS Plugin Implementation."""

    def __init__(self, iface, testing=False):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.testing = testing
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        if not self.testing:
            # initialize locale
            locale = QSettings().value('locale/userLocale')[0:2]
            locale_path = os.path.join(
                self.plugin_dir,
                'i18n',
                f'OMRAT_{locale}.qm')

            if os.path.exists(locale_path):
                self.translator = QTranslator()
                self.translator.load(locale_path)
                QCoreApplication.installTranslator(self.translator)

            # Declare instance attributes
            self.actions = []
            self.menu = self.tr(u'&Open Maritime Risk Analysis Tool')
            # TODO: We are going to let the user set this up in a future iteration
            self.toolbar = self.iface.addToolBar(u'OMRAT')
            self.toolbar.setObjectName(u'OMRAT')

            #print "** INITIALIZING OMRAT"
        self.qgis_geoms = HandleQGISIface(self)
        self.drift_settings = DriftSettings(self)
        self.causation_f = CausationFactors(self)
        self.calc = Calculation(self)
        self.object = OObject(self)
        self.pluginIsActive = False
        self.dockwidget = None
        self.segment_id = 0
        self.traffic_data = {}
        self.segment_data = {}
        self.traffic = None
        self.cur_route_id = 1
        self.drift_values = self.drift_settings.drift_values
        self.causation_values = self.causation_f.data

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('OMRAT', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = self.plugin_dir + '/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Omrat'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # Disconnect signals
        self.dockwidget.pbAddRoute.clicked.disconnect()
        self.dockwidget.pbStopRoute.clicked.disconnect()
        self.dockwidget.pbUpdateAIS.clicked.disconnect()
        self.dockwidget.pbEditTrafficData.clicked.disconnect()
        self.dockwidget.pbAddSimpleDepth.clicked.disconnect()
        self.dockwidget.pbAddSimpleObject.clicked.disconnect()
        self.dockwidget.pbRunModel.clicked.disconnect()

        # Set plugin state to inactive
        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        # Call the unload method of Traffic
        if hasattr(self, 'traffic') and self.traffic is not None:
            self.traffic.unload()
            self.traffic = None

        # Call the unload method of AIS
        if hasattr(self, 'ais') and self.ais is not None:
            self.ais.unload()
            self.ais = None

        # Remove actions from menu and toolbar
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Open Maritime Risk Analysis Tool'),
                action)
            self.iface.removeToolBarIcon(action)

        # Remove the toolbar
        if hasattr(self, 'toolbar'):
            del self.toolbar
            
        # Unload geometries
        print(dir(self.object))
        self.object.unload()
        self.qgis_geoms.unload()
        self.qgis_geoms = None
        del self.qgis_geoms

        # Disconnect signals and clean up dock widget
        if self.dockwidget is not None:
            try:
                self.onClosePlugin()
            except Exception as e:
                print(f"Error during onClosePlugin: {e}")
            # Disconnect cbTrafficSelectSeg signal
            try:
                self.dockwidget.cbTrafficSelectSeg.currentIndexChanged.disconnect()
                print("Disconnected cbTrafficSelectSeg.currentIndexChanged")
            except TypeError:
                print("No connection for cbTrafficSelectSeg.currentIndexChanged")
            self.dockwidget = None
        gc.collect()
        for folder in ['omrat_utils', 'compute', 'geometries.get_drifting_overlap', 'geometries.route', 
                       'geometries.handle_qgis_iface']:
            to_remove = [m for m in sys.modules if m.startswith(folder)]
            for m in to_remove:
                print(m)
                del sys.modules[m]
        QgsMessageLog.logMessage("Plugin unloaded", "OMRAT", Qgis.Info)
        
    def point4326_from_wkt(self, coord_str:str, crs:str) -> QgsGeometry:
        coords = coord_str.split(' ')
        coords = [float(coord) for coord in coords]
        q_point = QgsPointXY(coords[0], coords[1])
        return q_point
        
    def load_lines(self, data):
        self.segment_id = 0
        crs = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
        for key, seg_data in data["segment_data"].items():
            name = f"Segment {seg_data['Route Id']} - {seg_data['Segment Id']}"
            vl = QgsVectorLayer(f"LineString?crs=EPSG:4326", name, "memory")
            QgsProject.instance().addMapLayer(vl)
            
            self.segment_id += 1
            if not vl.isEditable():
                vl.startEditing()
            fields = QgsFields()
            fields.append(QgsField("segmentId", QMetaType.Int))
            fields.append(QgsField("routeId", QMetaType.Int))
            fields.append(QgsField("startPoint", QMetaType.QString))
            fields.append(QgsField("endPoint", QMetaType.QString))
            fields.append(QgsField("label", QMetaType.QString)) 

            vl.dataProvider().addAttributes(fields.toList())
            vl.updateFields()

            # Create feature
            fet = QgsFeature(fields)
            start = self.point4326_from_wkt(seg_data["Start Point"], crs)
            end = self.point4326_from_wkt(seg_data["End Point"], crs)
            fet.setGeometry(QgsLineString([QgsPoint(start), QgsPoint(end)]))
            fet.setAttributes([seg_data["Segment Id"], seg_data["Route Id"], seg_data["Start Point"], 
                               seg_data["End Point"], f"LEG_{seg_data['Segment Id']}_{seg_data['Route Id']}"])
            self.qgis_geoms.style_layer(vl)
            fet.setId(1)

            vl.dataProvider().addFeatures([fet])
            self.qgis_geoms.label_layer(vl)
            vl.updateExtents()
            
            # Validate geometry
            if not fet.geometry().isGeosValid():
                print(f"Invalid geometry for segment {seg_data['Segment Id']}")
                continue

            # Commit changes and refresh
            vl.commitChanges()
            vl.triggerRepaint()

        # Refresh map canvas
        self.iface.mapCanvas().refresh()
        
    def get_length_and_dir_from_line(self, p1, p2):
        degrees = p1.azimuth(p2)

    def format_wkt(self, point):
        return f'{point.x():.6f} {point.y():.6f}'
    

    def reset_route_table(self):
        self.dockwidget.twRouteList.setColumnCount(5)
        self.dockwidget.twRouteList.setHorizontalHeaderLabels(['Segment Id', 'Route Id', 
                                                               'Start Point', 'End Point', 'Width'])
        self.dockwidget.twRouteList.setColumnWidth(1, 75)
        self.dockwidget.twRouteList.setColumnWidth(2, 125)
        self.dockwidget.twRouteList.setColumnWidth(3, 125)
        self.dockwidget.twRouteList.setColumnWidth(4, 75)
        self.dockwidget.twRouteList.setRowCount(0)
    
    def show_traffic_widget(self):
        self.traffic.traffic_data = self.traffic_data
        self.traffic.fill_cbTrafficSelectSeg()
        self.traffic.update_direction_select()
        self.traffic.run()
        
    def save_work(self):
        store = Storage(self)
        store.store_all()

    def load_work(self):
        store = Storage(self)
        store.load_all()
               
    def run_calculation(self):
        gd = GatherData(self)
        data = gd.get_all_for_save()
        self.calc.run_drift_visualization(data)
        #self.calc.run_model()
        
    def open_drift(self):
        self.drift_settings.run()
    
    def stop_route(self):
        """Stop the current route, commit changes, and set the active tool to 'Pan Map'."""
        # Disable the "Stop Route" button
        self.dockwidget.pbStopRoute.setEnabled(False)

        # Commit changes and stop editing for all layers in self.vector_layers
        for layer in self.qgis_geoms.vector_layers:
            if layer.isEditable():
                layer.commitChanges()  # Save changes
            layer.triggerRepaint()  # Ensure the layer is refreshed

        # Clear the current start point and increment the route ID
        self.cur_route_id += 1

        # Set the active tool to "Pan Map"
        canvas = self.iface.mapCanvas()
        map_tool_pan = QgsMapToolPan(canvas)
        canvas.setMapTool(map_tool_pan)
        self.iface.actionPan().trigger()
        
    def ais_settings(self):
        self.ais.run()
        
    def update_ais(self):
        self.ais.update_legs()
        self.dockwidget.cbTrafficSelectSeg.setEnabled(True)
    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = OMRATDockWidget()
            self.ship_cat = ShipCategories(self)
            self.ais = AIS(self)
            self.traffic = Traffic(self, self.dockwidget)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            #if not self.testing:
            #    self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.pbAddRoute.clicked.connect(self.qgis_geoms.add_new_route)
            self.dockwidget.pbStopRoute.clicked.connect(self.stop_route)
            self.dockwidget.pbUpdateAIS.clicked.connect(self.update_ais)
            self.dockwidget.pbEditTrafficData.clicked.connect(self.show_traffic_widget)
            
            menubar = QMenuBar(self.dockwidget)
            menubar.setMinimumSize(320,20)

            fileMenu = menubar.addMenu('File')
            SettingMenu = menubar.addMenu('Settings')
            analyse_sen_Menu = menubar.addMenu('Analyse sensitivity')
            HelpMenu = menubar.addMenu('Help')
            
            # mnuSub1 = viewMenu.addMenu('Sub-menu')
            fileMenu.addAction("Save", self.save_work)
            fileMenu.addAction("Load", self.load_work)
            SettingMenu.addAction("Drift settings", self.open_drift)
            SettingMenu.addAction("AIS connection settings", self.ais_settings)
            #self.dockwidget.actionSave_project.clicked.connect(self.save_work)
            #self.dockwidget.actionOpen_project.clicked.connect(self.load_work)
            self.dockwidget.cbTrafficSelectSeg.currentIndexChanged.connect(self.traffic.change_dist_segment)
            self.dockwidget.pbAddSimpleDepth.clicked.connect(self.object.add_simple_depth)
            self.dockwidget.pbAddSimpleObject.clicked.connect(self.object.add_simple_object)
            self.dockwidget.pbRunModel.clicked.connect(self.run_calculation)
            
            self.reset_route_table()
            self.dockwidget.show()


