# -*- coding: utf-8 -*-

# ***************************************************************************
#   This program is free software; you can redistribute it and/or modify    *
#   it under the terms of the GNU General Public License as published by    *
#   the Free Software Foundation; either version 2 of the License, or       *
#   (at your option) any later version.                                     *
# ***************************************************************************
#     begin                : 2019-10-28                                     *
#     updated              : 2025-08-07                                     *
#     copyright            : (C) 2025 by Adrian Bocianowski                 *
#     email                : adrian at bocianowski.com.pl                   *
# ***************************************************************************

from PyQt5 import QtWidgets, uic
from PyQt5.QtCore import QMetaType, Qt, pyqtSignal, QThread, QVariant
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QApplication, QMessageBox, QTableWidgetItem

from qgis.core import (
    QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsFeature, Qgis,
    QgsField, QgsGeometry, QgsLayerTreeLayer, QgsMapLayerType, QgsPoint,
    QgsProject, QgsVector3D, QgsVectorLayer
)
from qgis.gui import QgsMapToolEmitPoint

from typing import List, Tuple
from .gugik_service import GugikService
from ..gui.resources import *

import os

LEFT_PANEL, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__),'../','ui','leftPanel.ui'))
PROFILE_DIALOG, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__),'../','ui','calculate_decrease.ui'))

class CalculateHeight:
    def __init__(self, iface) -> None:
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.plugin_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))

        self.actions = []
        self.menu = u'&Oblicz wysokość (GUGiK NMT)'
        self.icon_path = ':/plugins/ObliczWysokosc/icons/'

        self.qgsProject = QgsProject.instance()

        # service used for fetching height values
        self.service = GugikService()

        self.toolsToolbar = self.iface.addToolBar(u'Oblicz wysokość (GUGiK NMT)')
        self.toolsToolbar.setObjectName(u'Oblicz wysokość (GUGiK NMT)')

        self.panel = LeftPanel()
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.panel)

        self.panel.closingPanel.connect(lambda: self.captureButton.setChecked(False))
        self.panel.clearButton.setIcon(QIcon(os.path.join(self.icon_path,'mActionDeleteTable.svg')))
        self.panel.clearButton.clicked.connect(self.clearTable)
        self.panel.copyButton.setIcon(QIcon(os.path.join(self.icon_path,'mActionEditCopy.svg')))
        self.panel.copyButton.clicked.connect(self.copyToClipboard)
        self.panel.clearLayer.setIcon(QIcon(os.path.join(self.icon_path,'mActionDeleteSelected.png')))
        self.panel.clearLayer.clicked.connect(self.clearLayer)
        
        self.tool = CanvasTool(self.iface,self.canvas)
        self.tool.clicked.connect(self.capturePoint)
        self.tool.deact.connect(self.panel.hide)

        self.panel.closingPanel.connect(lambda: self.canvas.unsetMapTool(self.tool))

        self.panel.hide()

        self.profileDialog = ProfileDialog(parent=self.iface.mainWindow())
        self.profileDialog.refreshButton.setIcon(QIcon(os.path.join(self.icon_path,'mActionRefresh.svg')))
        self.profileDialog.refreshButton.clicked.connect(lambda: self.refreshComboBox(self.profileDialog.comboBox, 1))
        self.profileDialog.cancel.clicked.connect(self.taskCanceled)
        self.profileDialog.close.clicked.connect(self.closeDialog)
        self.profileDialog.run.clicked.connect(self.generateProfile)
        self.profileDialog.cancel.setEnabled(False)

        self.first_start = None

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
        checkable=False,
        checked=False,
        shortcut=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip is not None:
            action.setStatusTip(status_tip)
        if whats_this is not None:
            action.setWhatsThis(whats_this)
        if add_to_toolbar:
            self.toolsToolbar.addAction(action)
        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action
                )
        if checkable:
            action.setCheckable(True)
        if checked:
            action.setChecked(1)
        if shortcut:
            action.setShortcut(shortcut)
        self.actions.append(action)

        return action

    def addMemoryLayer(self, source_layer: QgsVectorLayer, sect_length: int) -> QgsVectorLayer:
        project = QgsProject.instance()
        layer_fields = source_layer.fields()
        output_layer_name = f'Spadek terenu - {sect_length} [m] - GUGiK NMT'
        output_layer = QgsVectorLayer('LineStringZ?crs=epsg:2180', output_layer_name, 'memory')
        project.addMapLayer(output_layer, False)
        layerTree = project.layerTreeRoot()
        layerTree.insertChildNode(0, QgsLayerTreeLayer(output_layer))
        treeRoot = project.layerTreeRoot()  
        if treeRoot.hasCustomLayerOrder():
            order = treeRoot.customLayerOrder()
            order.insert(0, order.pop(order.index(project.mapLayersByName(output_layer_name)[0])))
            treeRoot.setCustomLayerOrder(order)

        pr = output_layer.dataProvider()
        att = [i for i in layer_fields]

        fields = [
            ("id", "int"),
            ("roznica_z", "double"),
            ("dlugosc_3d", "double"),
            ("spadek", "double")
        ]

        for f in fields:
            field = self.makeField(f[0], f[1])
            att.append(field)

        pr.addAttributes(att)

        output_layer.updateFields()

        return output_layer

    def addPointToLayer(self, x: float, y: float, z: float) -> None:
        x = float(x)
        y = float(y)
        z = float(z)

        name = 'Obliczone wysokości - GUGiK NMT'
        layers = QgsProject.instance().mapLayersByName(name)
        if not layers:
            layer = QgsVectorLayer('PointZ?crs=epsg:2180&field=x:double&field=y:double&field=z:double', name, 'memory')
            QgsProject.instance().addMapLayer(layer, False)
            layerTree = QgsProject.instance().layerTreeRoot()
            layerTree.insertChildNode(0, QgsLayerTreeLayer(layer))

            treeRoot = QgsProject.instance().layerTreeRoot()  
            if treeRoot.hasCustomLayerOrder():
                order = treeRoot.customLayerOrder()
                order.insert(0, order.pop(order.index(QgsProject.instance().mapLayersByName(name)[0])))
                treeRoot.setCustomLayerOrder(order) 
            
            layer.loadNamedStyle(os.path.join(self.plugin_dir,'styles','layer_style.qml'), True)
        else:
            layer = layers[0]

        feature = QgsFeature()
        point = QgsGeometry(QgsPoint(x, y, z))
        feature.setGeometry(point)
        feature.setAttributes([x,y,z])
        layer.startEditing()
        layer.addFeature(feature)
        layer.commitChanges()
        layer.reload()

    def capturePoint(self, point: List[float]) -> None:
        """Handle map click and display retrieved height."""
        res = self.service.get_height(point)
        if res[0] != False:
            rows = self.panel.tableWidget.rowCount()
            self.panel.tableWidget.setRowCount(rows + 1)
            self.panel.tableWidget.setItem(rows,0, QTableWidgetItem(str(round(point[0],2))))
            self.panel.tableWidget.setItem(rows,1, QTableWidgetItem(str(round(point[1],2))))
            self.panel.tableWidget.setItem(rows,2, QTableWidgetItem(str(res[1])))
            self.panel.tableWidget.selectRow(rows)

            if self.panel.checkBox.isChecked():
                self.addPointToLayer(point[0],point[1],res[1])
        else:
            QMessageBox.warning(None,res[1][0], res[1][1])
    
    def clearLayer(self) -> None:
        name = 'Obliczone wysokości - GUGiK NMT'
        layers = QgsProject.instance().mapLayersByName(name)

        if layers:
            layer = layers[0]
            layer.commitChanges()
            layer.dataProvider().truncate()
            self.iface.mapCanvas().refreshAllLayers()

    def clearTable(self) -> None:
        self.panel.tableWidget.setRowCount(0)

    def clickProfileButton(self) -> None:
        self.profileDialog.show()
        self.refreshComboBox(self.profileDialog.comboBox, 1)

    def clickGetHeightButton(self) -> None:
        if self.captureButton.isChecked():
            self.canvas.setMapTool(self.tool)
            self.panel.show()
        else:
            self.canvas.unsetMapTool(self.tool)
            self.panel.hide()

    def closeDialog(self) -> None:
        try:
            if self.pTask.stopTask == False:
                self.pTask.stopTask = True
                self.pTask.terminate()
                QMessageBox.warning(None,'Zatrzymanie procesu', 'Proces generowania spadku terenu został zatrzymany.')
        except:
            pass

        self.profileDialog.run.setEnabled(True)
        self.profileDialog.cancel.setEnabled(False)
        self.profileDialog.hide()
        self.profileDialog.progressBar.setValue(0)

    def copyToClipboard(self) -> None:
        tmp = ''
        for i in range(0,self.panel.tableWidget.rowCount()):
            x = self.panel.tableWidget.item(i,0).text()
            y = self.panel.tableWidget.item(i,1).text()
            z = self.panel.tableWidget.item(i,2).text()
            tmp += f"{x}\t{y}\t{z}\n"
        
        clip = QApplication.clipboard()
        clip.setText(tmp)

    def getLayers(self, geometry_type: int) -> List[QgsVectorLayer]:
        # 1 = Line layers
        # 2 = Polygon layers
        layers = []
        for l in self.qgsProject.mapLayers():
            layer = self.qgsProject.mapLayer(l)
            if layer.type() == QgsMapLayerType.VectorLayer:
                if layer.geometryType() == geometry_type:
                    layers.append(layer)
        return layers

    def generateProfile(self) -> None:
        l_idx = self.profileDialog.comboBox.currentIndex()

        if l_idx == 0:
            QMessageBox.warning(None,'Brak warstwy', 'Wybierz warstwę źródłową')
            return
        
        layer = self.lineLayers[l_idx - 1]

        if self.profileDialog.onlySelected.isChecked() and len(layer.selectedFeatures()) == 0:
            QMessageBox.warning(None,'Brak obiektów', 'Brak odcinków w selekcji')
            return

        try:
            self.dest_profile_layer = self.addMemoryLayer(layer, int(self.profileDialog.spinBox.value()))
        except:
            QMessageBox.warning(None,'Brakująca warstwa wejściowa', 'Wskazana warstwa wejściowa nie istnieje (prawdopodobnie została usunięta)')
            self.refreshComboBox(self.profileDialog.comboBox, 1)
            return

        self.dest_profile_layer.loadNamedStyle(os.path.join(self.plugin_dir,'styles','layer_style2.qml'), True)
        self.pTask = ProfileTool(
            layer,
            self.profileDialog.onlySelected.isChecked(),
            self.profileDialog.spinBox.value(),
            self.service,
        )
               
        self.pTask.progress.connect(self.profileDialog.progressBar.setValue)
        self.pTask.end.connect(self.taskFinished)
        self.pTask.error.connect(self.taskError)
        self.pTask.add_feature.connect(self.taskAddFeature)

        self.pTask.start()
        
        self.profileDialog.run.setEnabled(False)
        self.profileDialog.cancel.setEnabled(True)
        self.profileDialog.comboBox.setEnabled(False)
        self.profileDialog.spinBox.setEnabled(False)
        self.profileDialog.onlySelected.setEnabled(False)
        self.profileDialog.refreshButton.setEnabled(False)

    def initGui(self) -> None:
        self.first_start = True

        # <div>Icons made by <a href="https://www.flaticon.com/authors/wissawa-khamsriwath" title="Wissawa Khamsriwath">Wissawa Khamsriwath</a> from <a href="https://www.flaticon.com/"             title="Flaticon">www.flaticon.com</a></div>
        self.captureButton = self.add_action(
            os.path.join(self.icon_path,'cardinal-points.svg'),
            'Oblicz wysokość',
            self.clickGetHeightButton,
            checkable=True,
            parent=self.iface.mainWindow(),
            )
        self.tool.action = self.captureButton

        # <div>Icons made by <a href="https://www.flaticon.com/authors/wissawa-khamsriwath" title="Wissawa Khamsriwath">Wissawa Khamsriwath</a> from <a href="https://www.flaticon.com/"             title="Flaticon">www.flaticon.com</a></div>
        self.profileButton = self.add_action(
            os.path.join(self.icon_path,'line-chart.svg'),
            'Oblicz spadek terenu',
            self.clickProfileButton,
            checkable=False,
            parent=self.iface.mainWindow(),
            )

        if self.first_start == True:
            self.first_start = False

    def makeField(self, name: str, type_name: str) -> QgsField:
        """
        Create a QgsField depending on the QGIS version.
        Supports QMetaType from QGIS 3.38+.
        """
        if Qgis.QGIS_VERSION_INT >= 33800:
            # QGIS 3.38+ uses QMetaType
            type_map = {
                "int": QMetaType.Int,
                "double": QMetaType.Double,
                "string": QMetaType.QString,
            }
            return QgsField(name, type_map[type_name], type_name)
        else:
            # Older QGIS versions use QVariant.Type
            type_map = {
                "int": QVariant.Int,
                "double": QVariant.Double,
                "string": QVariant.String,
            }
            return QgsField(name, type_map[type_name], type_name)

    def refreshComboBox(self, combo: QtWidgets.QComboBox, geometry_type: int) -> None:
        combo.clear()
        combo.addItem(None)

        if geometry_type == 1:
            self.lineLayers = self.getLayers(1)
        
            for i in self.lineLayers:
                combo.addItem(i.name())  
    
    def taskAddFeature(self, feature: QgsFeature) -> None:
        self.dest_profile_layer.startEditing()
        self.dest_profile_layer.addFeature(feature)
        self.dest_profile_layer.commitChanges()

    def taskCanceled(self) -> None:
        self.pTask.stopTask = True
        self.pTask.terminate()
        QMessageBox.warning(None,'Zatrzymanie procesu', 'Proces generowania spadku terenu został zatrzymany.')
        self.profileDialog.run.setEnabled(True)
        self.profileDialog.cancel.setEnabled(False)
        self.profileDialog.progressBar.setValue(0)
        self.profileDialog.comboBox.setEnabled(True)
        self.profileDialog.spinBox.setEnabled(True)
        self.profileDialog.onlySelected.setEnabled(True)
        self.profileDialog.refreshButton.setEnabled(True)

    def taskError(self, e: List[str]) -> None:
        self.pTask.stopTask = True
        self.pTask.terminate()
        QMessageBox.warning(None,e[0], e[1])
        self.profileDialog.run.setEnabled(True)
        self.profileDialog.cancel.setEnabled(False)
        self.profileDialog.progressBar.setValue(0)
        self.profileDialog.comboBox.setEnabled(True)
        self.profileDialog.spinBox.setEnabled(True)
        self.profileDialog.onlySelected.setEnabled(True)
        self.profileDialog.refreshButton.setEnabled(True)

    def taskFinished(self) -> None:
        self.profileDialog.run.setEnabled(True)
        self.profileDialog.cancel.setEnabled(False)
        QMessageBox.information(self.iface.mainWindow(),'Spadek terenu', 'Proces generowania został zakończony')
        self.profileDialog.progressBar.setValue(0)
        self.profileDialog.comboBox.setEnabled(True)
        self.profileDialog.spinBox.setEnabled(True)
        self.profileDialog.onlySelected.setEnabled(True)
        self.profileDialog.refreshButton.setEnabled(True)
        
        self.pTask.quit()
        self.pTask.wait()

    def unload(self) -> None:
        # Unset the map tool if it's currently active
        if self.canvas.mapTool() == self.tool:
            self.canvas.unsetMapTool(self.tool)

        # Remove and delete the dock panel
        if self.panel:
            self.iface.removeDockWidget(self.panel)
            self.panel.deleteLater()
            self.panel = None

        # Remove all actions from the menu and toolbar
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)
        
        self.actions = []

        # Remove and delete the toolbar
        if self.toolsToolbar:
            self.iface.mainWindow().removeToolBar(self.toolsToolbar)
            self.toolsToolbar.deleteLater()
            self.toolsToolbar = None

        # Stop the background task thread if it's still running
        try:
            if self.pTask.isRunning():
                self.pTask.stopTask = True
                self.pTask.quit()
                self.pTask.wait()
                self.pTask = None
        except:
            pass

        # Close and delete the profile dialog if it's open
        if self.profileDialog.isVisible():
            self.profileDialog.close()
            self.profileDialog.deleteLater()
            self.profileDialog = None

class CanvasTool(QgsMapToolEmitPoint):
    clicked = pyqtSignal(list)
    deact = pyqtSignal()

    def __init__(self, iface, canvas) -> None:
        QgsMapToolEmitPoint.__init__(self, canvas)
        self.canvas = canvas
        self.iface = iface
        self.action = None

    def activate(self) -> None:
        self.action.setChecked(True)
        self.setCursor(Qt.CrossCursor)

    def canvasPressEvent(self, e) -> None:
        point = self.toMapCoordinates(self.canvas.mouseLastXY())
        point = QgsGeometry.fromPointXY(point)

        canvas_crs = self.canvas.mapSettings().destinationCrs().authid()
        target_crs = QgsCoordinateReferenceSystem(2180)

        point = self.geometryCrs2Crs(point, canvas_crs, target_crs)
        point = point.asPoint()
        x = point.x()
        y = point.y()

        self.clicked.emit([x,y])

    def deactivate(self) -> None:
        self.action.setChecked(False)
        self.deact.emit()

    def geometryCrs2Crs(
        self, geometry: QgsGeometry, source_crs: str, destination_crs: str
    ) -> QgsGeometry:
        geometry = QgsGeometry(geometry)
        src_crs = QgsCoordinateReferenceSystem(source_crs)
        dest_crs = QgsCoordinateReferenceSystem(destination_crs)
        crs2crs = QgsCoordinateTransform(src_crs, dest_crs, QgsProject.instance())
        geometry.transform(crs2crs)
        return geometry

class LeftPanel(QtWidgets.QDockWidget, LEFT_PANEL):
    closingPanel = pyqtSignal()

    def __init__(self, parent=None) -> None:
        super(LeftPanel, self).__init__(parent)
        self.setupUi(self)

    def closeEvent(self, event) -> None:
        self.closingPanel.emit()
        event.accept()

class ProfileDialog(QtWidgets.QDialog, PROFILE_DIALOG):
    def __init__(self, parent=None) -> None:
        super(ProfileDialog, self).__init__(parent)
        self.setupUi(self)

class ProfileTool(QThread):
    progress = pyqtSignal(int)
    end = pyqtSignal()
    error = pyqtSignal(list)
    add_feature = pyqtSignal(object)

    def __init__(self, source_layer: QgsVectorLayer, only_selected: bool, distance: float, service: GugikService):
        QThread.__init__(self)
        self.source_layer = source_layer
        self.only_selected = only_selected
        self.distance = distance
        self.stopTask = False

        # service instance for height requests
        self.service = service
    
    def run(self) -> None:
        f_count = 0
        
        if self.only_selected:
            f_count += len(self.source_layer.selectedFeatures())
            features = [i for i in self.source_layer.selectedFeatures()]
        else:
            f_count += self.source_layer.featureCount()
            features = [i for i in self.source_layer.getFeatures()]
        i = 0
        id = 0
        prg = 0

        for f in features:
            f_geom = f.geometry()

            for part in f_geom.parts():
                part_geom = QgsGeometry.fromWkt(part.asWkt())
                part_geom_sections = self.generateSections(part_geom, self.distance)
                part_geom_sections_len = len(part_geom_sections)
                i2 = 0

                for pgs in part_geom_sections:
                    if not self.stopTask:
                        att = f.attributes()
                        geom_z = self.addZvalue(pgs)
                        if geom_z[0] == False:
                            self.error.emit(geom_z[1])
                            return
                        feature = QgsFeature()
                        v_lst = [i for i in geom_z[1].vertices()]

                        # id
                        att.append(id)

                        # height difference
                        z_diff= round(abs(v_lst[0].z() - v_lst[-1].z()),2)
                        att.append(z_diff)

                        # 3d length
                        dl_3d = [ QgsVector3D (i.x(), i.y(), i.z()) for i in v_lst]
                        dl_3d = [dl_3d[i].distance(dl_3d[i+1])  for i,e in enumerate(dl_3d[:-1])]
                        dl_3d = round(sum(dl_3d),2)
                        att.append(dl_3d)

                        # slope
                        if dl_3d == 0:
                            continue
                        slope = round((z_diff/dl_3d) * 100,2)
                        att.append(slope)
                        
                        feature.setAttributes(att)
                        feature.setGeometry(geom_z[1])
                        self.add_feature.emit(feature)

                        i2 += 1
                        prg2 = ((i2 / part_geom_sections_len / f_count) * 100) + prg
                        if (prg2 - prg) > 1:
                            self.progress.emit(prg2)

                        id += 1
                    else:
                        return
            i += 1
            prg = (i/f_count) * 100
            self.progress.emit(prg)

        self.end.emit()

    def addZvalue(self, geometry: QgsGeometry) -> Tuple[bool, QgsGeometry]:
        """Add Z values to vertices of a geometry using online service."""
        geom_list = []
        for v in geometry.vertices():
            x = float(v.x())
            y = float(v.y())
            z = self.service.get_height([x, y])
            if z[0] == False:
                return False,z[1]
            point = QgsPoint(x, y, float(z[1]))
            geom_list.append(point)

        geom = QgsGeometry.fromPolyline(geom_list)
        return True, geom

    def generateSections(self, geometry: QgsGeometry, distance: float) -> List[QgsGeometry]:
        """Split geometry into sections of given distance."""
        geom_length = geometry.length()
        vertices = [geometry.lineLocatePoint(QgsGeometry.fromWkt(i.asWkt())) for i in geometry.vertices()]
        geom_list = []
        i = 0

        while i < geom_length:
            end = i + distance
            if end > geom_length:
                end = geom_length

            tmp_geom = []
            tmp_geom.append(i)
            for v in vertices:
                if v > i and v < end:
                    tmp_geom.append(v)
            tmp_geom.append(end)
            tmp_geom = QgsGeometry.fromPolylineXY([geometry.interpolate(i).asPoint() for i in tmp_geom])
            geom_list.append(tmp_geom)
            i += distance
        return geom_list