# -*- coding: utf-8 -*-

"""
/***************************************************************************
 OSWeGe-Tools - a QGIS plugin
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-04-09
        copyright            : (C) 2024 by Jannik Schilling
        email                : jannik.schilling@uni-rostock.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jannik Schilling'
__date__ = '2024-04-09'
__copyright__ = '(C) 2024 by Jannik Schilling'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import pandas as pd
import os

from qgis.PyQt.QtCore import (
    QCoreApplication
)
from qgis.core import (
    NULL,
    Qgis,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingOutputFile,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterVectorLayer
)

from .attributpruefung import (
    handle_test_missing_fields,
    handle_tests_attributes
)

from .check_gew_report import (
    create_layers_from_report_dict,
    save_layer_to_file,
    layerReport
)

from .config_tools import (
    get_config_from_json,
    config_layer_if_in_project
)

from .defaults import (
    file_config_user,
    output_layer_prefixes
)

from .geometriepruefungen import (
    handle_tests_single_geometries,
    handle_tests_geoms_comparisons
)

from .hilfsfunktionen import (
    simpleTimeStepLogger,
    handle_rl_and_dl
)

class checkGewaesserDaten(QgsProcessingAlgorithm):
    """
    Prueft Gewaesserdaten
    """
    LAYER_GEWAESSER = 'LAYER_GEWAESSER'
    LAYER_ROHRLEITUNGEN = 'LAYER_ROHRLEITUNGEN'
    LAYER_DURCHLAESSE = 'LAYER_DURCHLAESSE'
    LAYER_WEHRE = 'LAYER_WEHRE'
    LAYER_SCHAECHTE = 'LAYER_SCHAECHTE'
    REPORT = 'REPORT'
    REPORT_OUT = 'REPORT_OUT'
    
    if (int(Qgis.version().split('.')[0]) == 3 and int(Qgis.version().split('.')[1]) >= 36) or (int(Qgis.version().split('.')[0]) > 3):
        newer_qgis_version = True
    else:
        newer_qgis_version = False
 
    def initAlgorithm(self, config):
        """
        Definition von Input und Output des Werkzeugs
        """
        # User config laden
        dict_layer_defaults = config_layer_if_in_project(file_config_user)
 
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.LAYER_GEWAESSER,
                self.tr('Gewässer-Layer'),
                [QgsProcessing.SourceType.TypeVectorLine],
                defaultValue=dict_layer_defaults['gewaesser']
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.LAYER_ROHRLEITUNGEN,
                self.tr('Rohrleitungs-Layer'),
                [QgsProcessing.SourceType.TypeVectorLine],
                optional=True,
                defaultValue=dict_layer_defaults['rohrleitungen']
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.LAYER_DURCHLAESSE,
                self.tr('Durchlässe-Layer'),
                [QgsProcessing.SourceType.TypeVectorLine],
                optional=True,
                defaultValue=dict_layer_defaults['durchlaesse']
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.LAYER_WEHRE,
                self.tr('Wehre-Layer'),
                [QgsProcessing.SourceType.TypeVectorPoint],
                optional=True,
                defaultValue=dict_layer_defaults['wehre']
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.LAYER_SCHAECHTE,
                self.tr('Schächte-Layer'),
                [QgsProcessing.SourceType.TypeVectorPoint],
                optional=True,
                defaultValue=dict_layer_defaults['schaechte']
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.REPORT,
                self.tr('Reportdatei'),
                'Geopackage (*.gpkg)'                #'Textdatei (*.txt)',
            )
        )
        if not self.newer_qgis_version:
            self.addOutput(
                QgsProcessingOutputFile(
                    self.REPORT_OUT,
                    self.tr('Reportdatei: Geopackage File(*.gpkg)')
                )
            ) 

    def processAlgorithm(self, parameters, context, feedback):
        """
        Hier findet die Verarbeitung statt
        """
        # Festlegung für Tests 
        timeLogger = simpleTimeStepLogger()
        timeLogger.start_logging()
        test_output_all = False  # Ueberspringt das bereinigen des report_dict, wenn True
        is_test_version = False  # Hinweis zum Output, Ausgabe der Zeiten, wenn True

        # Layerdefinitionen
        layer_gew = self.parameterAsVectorLayer(parameters, self.LAYER_GEWAESSER, context)
        layer_rohrleitungen = self.parameterAsVectorLayer(parameters, self.LAYER_ROHRLEITUNGEN, context)
        layer_durchlaesse = self.parameterAsVectorLayer(parameters, self.LAYER_DURCHLAESSE, context)
        layer_wehre = self.parameterAsVectorLayer(parameters, self.LAYER_WEHRE, context)
        layer_schaechte = self.parameterAsVectorLayer(parameters, self.LAYER_SCHAECHTE, context)
        reportdatei = self.parameterAsString(parameters, self.REPORT, context)


        # Zusammenfassendes dictionary fuer Prozessparameter, die an Funktionen uebergeben werden
        feedback.setProgressText('Vorbereitung der Tests')
        layer_dict = {}
        list_crs = []
        list_layer_types = [
            'gewaesser',
            'rohrleitungen',
            'durchlaesse',
            'wehre',
            'schaechte'
        ]  # wichtig für die Reihenfolge der Tests; in dictionaries kann diese variieren (!)
        for layer_key, layer in zip(
            list_layer_types ,
            [
                layer_gew,
                layer_rohrleitungen,
                layer_durchlaesse,
                layer_wehre,
                layer_schaechte,
            ]
        ):
            if layer:
                list_crs.append(layer.crs().authid())
                ft_count = layer.featureCount() if layer.featureCount() else 0
                layer_steps = 100.0/ft_count if ft_count != 0 else 0
                layer_dict[layer_key] = {
                    'layer': layer,
                    'count': ft_count,
                    'steps': layer_steps
                }
        if len(set(list_crs)) == 1:
            crs_out = list_crs[0]  # fuer die Ergebnisausgabe
        else:
            raise QgsProcessingException(
                'Alle Layer müssen im gleichen Koordinatenbezugssystem gespeichert sein!'
            )

        # Dictionary fuer immer wiederkehrende Parameter
        # User config laden
        user_config_dict = get_config_from_json(file_config_user)
        params_processing = {
            'layer_dict': layer_dict,  # zu pruefende Layer
            'feedback': feedback,  # QgsProcessingFeedback fuer Statusinfos waehrend des Durchlaufs
            'ereign_gew_id_field': user_config_dict['check_layer_defaults']['primaerschluessel_gew'],  # Name des Felds mit dem Primaerschluessel: "gu_cd" oder "ba_cd"
            'gew_primary_key_missing': False,
            'field_merged_id': 'merged_id',  # Feldname fuer neue ID, wenn rl und dl vorhanden
            'emptystrdef': [NULL, ''],  # moegliche "Leer"-Definitionen für Zeichenketten
            'n_layer': len(layer_dict) # Anzahl der zu bearbeitenden Layer
        }

        # dictionary fuer Feedback / Fehlermeldungen
        report_object = layerReport(layer_dict)

        # rl und dl zusammenfassen fuer gemeinsame Auswertung, wenn beide vorhanden      
        handle_rl_and_dl(
            layer_rohrleitungen,
            layer_durchlaesse,
            params_processing,
            report_object
        )
        feedback.setProgressText('Abgeschlossen \n ')
        timeLogger.log_time('Vorbereitung')


        # Hauptfunktion
        def main_check(
            layer_key,
            report_object,
            params_processing,
            i_run
        ):
            """
            Diese Hauptfunktion wird durchlaufen, um die Vektorobjekte aller Layer zu pruefen (Attribute + Geometrien)
            :param str key
            :param layerReport report_object
            :param dict params_processing
            :param int i_run: Zaehler fuers feedback
            """
            feedback = params_processing['feedback']
            key = layer_key
            layer = params_processing['layer_dict'][layer_key]['layer']
            layer_steps = params_processing['layer_dict'][layer_key]['steps']

            feedback.setProgressText(
                output_layer_prefixes[key] + '-Layer \"'
                + layer.name() + '\" (' + str(i_run+1) + '/'
                + str(params_processing['n_layer'])
                + '):')

            # Sind die pflichtfelder vorhanden?
            feedback.setProgressText('> Prüfe benötigte Attributfelder...')
            handle_test_missing_fields(
                layer_key,
                layer,
                report_object,
                user_config_dict['check_layer_defaults']['pflichtfelder'],
                params_processing
            )
            timeLogger.log_time(layer_key+'_Fields')


            # Pruefroutinen fuer Attribute
            feedback.setProgressText('> Prüfe alle Einzelobjekte...')
            handle_tests_attributes(
                layer_key,
                layer,
                report_object,
                params_processing
            )
            timeLogger.log_time(layer_key+'_Attr')


            # Pruefroutinen fuer Geometrien
            feedback.setProgressText('-- Geometrien')
            # fuer alle Layer: Einzelgeometrien pruefen (leer, Multigeometrien und Selbstueberschneidungen)
            handle_tests_single_geometries(
                layer,
                layer_key,
                layer_steps,
                report_object,
                feedback
            )
            timeLogger.log_time((layer_key+'_handle_tests_single_geometries'))
            # Geometrien pruefen durch Vergleich mit anderen Geometrien
            handle_tests_geoms_comparisons(
                layer_key,
                report_object,
                params_processing
            )
            timeLogger.log_time((key+'_geom_comp_others'))
            feedback.setProgressText('Abgeschlossen \n ')


        # run test
        feedback.setProgressText('Tests fuer einzelne Layer')
        feedback.setProgressText('-------------------------')
        i = 0
        for key in list_layer_types:
            if key in params_processing['layer_dict'].keys():
                main_check(
                    key,
                    report_object,
                    params_processing,
                    i
                )
                i = i+1


        # Ausgabe:
        # 1 report_dict bereinigen
        if not test_output_all:
            feedback.setProgressText('Bereinige Fehlerliste...')
            report_dict_prepared = report_object.prepare_report_dict(feedback)
            feedback.setProgressText('Abgeschlossen \n ')
            

        # 2 Ausgabe schreiben
        feedback.setProgressText('Generiere Layer / Ausgabe...')
        vector_layer_list, list_messages = create_layers_from_report_dict(
            report_dict_prepared,
            crs_out, feedback
        )
        feedback.setProgressText('Abgeschlossen \n ')

        if len(vector_layer_list) > 0:
            feedback.setProgressText('Speichere Layer in Datei...')
            save_layer_to_file(vector_layer_list, reportdatei)   
            feedback.setProgressText('Abgeschlossen \n ')     
            timeLogger.log_time('WriteLayer')

        if is_test_version:
            feedback.setProgressText(' \nDauer der Schritte:')
            timing_txt = timeLogger.report_time_logs()
            feedback.setProgressText('\n'.join(timing_txt))
        
        # Dinge loeschen (zur Sicherheit)
        del timeLogger
        del report_object

        # 3 Feedback
        if self.newer_qgis_version:
            feedback.pushFormattedMessage(
                html=(
                    '<p><b>Ergebnis der Datenprüfung</b>'
                    + '<br>-----------------------</p>'
                ),
                text='Ergebnis der Datenprüfung'
            )
            # Fehlende Spalten
            if len(list_messages) > 0:
                feedback_txt = 'weiteren '
                for msg in list_messages:
                    feedback.pushWarning(msg)
            else:
                feedback_txt = ''
            if len(vector_layer_list) == 0:
                feedback.setProgressText(f'Keine {feedback_txt}Fehler im Datensatz; es wird keine Reportdatei erzeugt.')
            else:
                res_file_name = os.path.split(reportdatei)[1]
                feedback.pushFormattedMessage(
                html=(
                    f'<a href=\"file:///{reportdatei}\">Link zur Datei mit Geometrie-/Datenfehlern ({res_file_name})</a>'
                ),
                text=f'Ergebnis in {reportdatei}'
            )
            feedback.setProgressText('-----------------------')
            return {}
        else:  # for older QGIS Versions
            feedback.setProgressText('\nErgebnis der Datenprüfung\n-----------------------')
            # Fehlende Spalten
            if len(list_messages) > 0:
                feedback_txt = 'weiteren '
                for msg in list_messages:
                    feedback.pushWarning(msg)
            else:
                feedback_txt = ''
            if len(vector_layer_list) == 0:
                feedback.setProgressText(f'Keine {feedback_txt}Fehler im Datensatz; es wird keine Reportdatei erzeugt\n---------------------')
            else:
                return {self.REPORT_OUT: reportdatei} 

    def name(self):
        return '1_Pruefroutine_Gewaesserdaten'

    def displayName(self):
        return '1_Pruefroutine_Gewaesserdaten'

    def group(self):
        return self.tr(self.groupId())

    def shortHelpString(self) -> str:
        return (
            'Mit diesem Werkzeug werden Geodaten für den Export ins '
            + 'FIS-Gewässer geprüft. \n'
            + 'Die Ergebnisse der Prüfroutine werden als '
            + 'einzelne Layer in einer Geopackage-Datei (\"Reportdatei\") gespeichert'
        )

    def groupId(self):
        return 'Datenexport'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return checkGewaesserDaten()
