# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OSMDataSyncDockWidget
                                 A QGIS plugin
 This Plugin allows the user to traverse and sync features from OSM.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2020-06-25
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Christopher Hilfing
        email                : chilfing@hsr.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import locale
import operator
import os
import time
import webbrowser
import traceback
from enum import Enum

from qgis.core import NULL, QgsProject, QgsTask, QgsApplication, Qgis
from qgis.PyQt import QtCore, uic, QtGui
from qgis.PyQt.QtWidgets import QMessageBox, QCheckBox, QDockWidget

from .configs import Configuration
from .fileloader import found_config_file, get_config_file_path, update_config_file, create_config_file, create_backup_file
from .helper import (
    amount_of_features,
    date_time,
    edit_attribute,
    field_names,
    get_field_id,
    get_layer_from_id,
    passed_sec,
    select_feature,
    first_occurrence_index,
)
from .synchronizer import Synchronizer

locale.setlocale(locale.LC_ALL, "")

FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "osm_data_sync_dockwidget_base.ui")
)


# TODO Refactoring: Rename some method and attribute names for better understanding?

class ButtonType(Enum):
    NEXT = 1
    PREVIOUS = 2
    FIRST = 3
    LAST = 4
    RESET = 5


class OSMDataSyncDockWidget(QDockWidget, FORM_CLASS):
    closingPlugin = QtCore.pyqtSignal()
    plugin_name = "OSM Data Sync"

    def __init__(self, iface, parent=None):
        """Constructor."""
        super(OSMDataSyncDockWidget, self).__init__(parent)
        self.setupUi(self)

        self.iface = iface

        self.feats_list = []
        self.layer_id = None
        self.layer = None
        self.configuration = None
        self.feat_index = None
        self.selected_feat_index = None
        self.selected_features = None

        self.controls_toggled_on = False
        self.start_sync_time = None
        self.released_button = None
        self.user_changed_state = False

        self.states = {
            self.rad_state_changed.text(): self.rad_state_changed,
            self.rad_state_unchanged.text(): self.rad_state_unchanged,
            self.rad_state_new.text(): self.rad_state_new,
            self.rad_state_not_found.text(): self.rad_state_not_found,
            self.rad_state_local.text(): self.rad_state_local,
            self.rad_state_ignore.text(): self.rad_state_ignore,
            self.rad_state_undefined.text(): self.rad_state_undefined,
        }

        self.state_change_blacklist = {
            self.rad_state_changed.text(): [self.rad_state_local.text()],
            self.rad_state_unchanged.text(): [self.rad_state_local.text()],
            self.rad_state_new.text(): [self.rad_state_local.text()],
            self.rad_state_not_found.text(): [self.rad_state_ignore.text()],
            self.rad_state_local.text(): [self.rad_state_ignore.text()],
            self.rad_state_ignore.text(): [self.rad_state_local.text()],
            NULL: [],
        }

        self.state_undefined_text = self.rad_state_undefined.text()

        self.user_changed_states = {
            self.rad_state_local,
            self.rad_state_ignore,
            self.rad_state_undefined,
        }

        self.changed_state_with_show = False
        self.unable_to_set_state = False

        self.confirmed = False
        self.del_not_found = True
        self.is_synchronizing = False
        self.create_config = False

        self.setup()

    def setup(self):
        self.setWindowTitle(OSMDataSyncDockWidget.plugin_name)

        QgsProject.instance().layerWasAdded.connect(self.update_layers_combos)
        QgsProject.instance().layerRemoved.connect(self.update_layers_combos)
        QgsProject.instance().layerRemoved.connect(self.remove_layer_event)

        self.cbo_layer.activated.connect(self.cbo_layer_activated)
        self.cbo_order.activated.connect(self.cbo_filter_activated)

        self.chk_use_selected.toggled.connect(self.chk_use_selected_clicked)

        self.btn_prev.pressed.connect(lambda: self.next_feat(ButtonType.PREVIOUS))
        self.btn_first_feature.pressed.connect(lambda: self.next_feat(ButtonType.FIRST))
        self.btn_last_feature.pressed.connect(lambda: self.next_feat(ButtonType.LAST))
        self.btn_next.pressed.connect(lambda: self.next_feat(ButtonType.NEXT))

        self.btn_show_osm.pressed.connect(self.btn_show_osm_pressed)
        self.btn_sync.pressed.connect(self.btn_sync_pressed)
        self.btn_reload.pressed.connect(self.btn_reload_pressed)

        self.update_layers_combos()

        self.rad_state_changed.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_changed)
        )
        self.rad_state_unchanged.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_unchanged)
        )
        self.rad_state_new.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_new)
        )
        self.rad_state_not_found.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_not_found)
        )
        self.rad_state_local.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_local)
        )
        self.rad_state_ignore.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_ignore)
        )
        self.rad_state_undefined.toggled.connect(
            lambda: self.rad_btn_state(self.rad_state_undefined)
        )

        self.btn_show_osm.setToolTip(
            "Show a feature on OSM by its @id value (OSM-id)"
        )

        # Shortcuts
        self.btn_next.setShortcut("F8")

    def remove_layer_event(self):
        self.layer_id = self.cbo_layer.itemData(self.cbo_layer.currentIndex())
        if not self.layer_id:
            self.cbo_layer_activated()

    def layer_modified_event(self):
        self.set_state_occurrence()
        if not self.user_changed_state:
            try:
                current_osm_id = self.current_feature().attribute(
                    self.configuration.osm_id_attribute
                )
            except:
                current_osm_id = None
            self.cbo_attrib_activated()
            new_position = None

            if (
                    current_osm_id
                    and len(self.feats_list) > 0
                    and current_osm_id
                    != self.current_feature().attribute(self.configuration.osm_id_attribute)
            ):
                for nr, feature in enumerate(self.feats_list):
                    if (
                            feature.attribute(self.configuration.osm_id_attribute)
                            == current_osm_id
                    ):
                        new_position = nr
                        break

            if new_position:
                self.feat_index = new_position
                self.change_feature_count()

            if len(self.feats_list) <= 1:
                self.cbo_layer_activated()
            self.show_id()
            self.show_state()

        else:
            self.user_changed_state = False

    def close_event(self, event):
        self.closingPlugin.emit()
        event.accept()

    def setup_order_by(self):
        self.cbo_order.clear()
        for field in sorted(self.layer.fields(), key=lambda field: field.name().upper()):
            self.cbo_order.addItem(field.name(), None)

    def update_layers_combos(self):
        prev_layer_id = None
        if self.cbo_layer.currentIndex() > -1:
            prev_layer_id = self.cbo_layer.itemData(self.cbo_layer.currentIndex())

        # Find layers IDs and names
        layer_ids = QgsProject.instance().mapLayers()
        names_ids_dict = {}
        for layer_id in layer_ids:
            layer = get_layer_from_id(layer_id)
            if layer is not None and layer.type() == 0:
                names_ids_dict[layer_id] = layer.name()

        # Fill combo
        self.cbo_layer.clear()
        self.cbo_layer.addItem("", None)

        sorted_ids_names = sorted(list(names_ids_dict.items()), key=operator.itemgetter(1))

        for name_id in sorted_ids_names:
            self.cbo_layer.addItem(name_id[1], name_id[0])

        if prev_layer_id is not None:
            self.layer_id = self.set_layercombo_index(self.cbo_layer, prev_layer_id)
        else:
            self.toggle_navigation_btn(self.btn_prev, self.layer_id is not None)
            self.toggle_navigation_btn(self.btn_next, self.layer_id is not None)
            self.cbo_layer_activated()
            self.change_feature_count()

    def cbo_filter_activated(self):
        self.feat_index = None
        self.btn_reload.setEnabled(True)

        if amount_of_features(self.layer) > 0:
            self.load_settings()

        if len(self.feats_list) == 0:
            self.toggle_controls(False)
        self.show_id()

    def cbo_layer_activated(self, should_reset_order_by=True):
        self.layer_id = self.cbo_layer.itemData(self.cbo_layer.currentIndex())
        self.feat_index = None

        if self.layer_id is not None:
            self.btn_reload.setEnabled(True)
            self.layer = get_layer_from_id(self.layer_id)
            self.layer.layerModified.connect(self.layer_modified_event)
            self.layer.nameChanged.connect(self.update_layers_combos)
            self.configuration = Configuration(self.layer)
            if should_reset_order_by:
                self.setup_order_by()
            self.load_settings()
            self.layer.selectionChanged.connect(self.layer_selection_changed)
            self.selected_features = self.sort_list(list(self.layer.getSelectedFeatures()))
        else:
            self.layer = None
            self.cbo_order.clear()
            self.feats_list.clear()
            self.reset_sync_btn_text()
            self.btn_sync.setEnabled(False)
            self.btn_reload.setEnabled(False)
            self.toggle_controls(False)
            self.change_feature_count()
            self.show_state()

        self.show_id()

    def layer_selection_changed(self):
        self.selected_features = self.sort_list(list(self.layer.getSelectedFeatures()))
        self.chk_use_selected.setChecked(False)
        self.selected_feat_index = None

    def toggle_user_changed_states(self, should_activate):
        for rad_btn in self.user_changed_states:
            rad_btn.setEnabled(should_activate)

    def reset_sync_btn_text(self):
        for state, rad_btn in self.states.items():
            rad_btn.setText(state)

    def load_settings(self):
        if self.update_config_file_state():
            if not self.is_synchronizing:
                self.btn_sync.setEnabled(True)
            self.cbo_attrib_activated()
            if len(self.feats_list) > 0:
                self.toggle_controls(True)

    def cbo_attrib_activated(self):
        self.set_state_occurrence()
        self.feats_list = self.sort_list(self.layer.getFeatures())
        self.change_feature_count()

    def sort_list(self, qgis_feature_list):
        return sorted(qgis_feature_list, key=lambda x: x.attribute(self.cbo_order.currentText()))

    def set_state_occurrence(self):
        state_occurrences = dict.fromkeys(self.states.keys(), 0)
        for feature in self.layer.getFeatures():
            attribute_value = str(
                feature.attribute(self.configuration.state_field_name)
            )
            if attribute_value in state_occurrences.keys():
                state_occurrences[attribute_value] += 1
            else:
                state_occurrences[self.state_undefined_text] += 1

        for key, value in state_occurrences.items():
            self.states[key].setText(f"{key} ({value})")

    def toggle_controls(self, should_activate):
        self.controls_toggled_on = should_activate

        self.chk_select.setEnabled(should_activate)
        self.chk_start_sel.setEnabled(should_activate)
        self.chk_use_selected.setEnabled(should_activate)
        self.rad_action_pan.setEnabled(should_activate)
        self.rad_action_zoom.setEnabled(should_activate)
        self.cbo_order.setEnabled(should_activate)

        self.toggle_prev_buttons(False)
        self.btn_show_osm.setEnabled(False)
        self.toggle_user_changed_states(False)

        if len(self.feats_list) == 0:
            should_activate = False

        self.set_state(self.state_undefined_text)
        self.toggle_next_buttons(should_activate)

    def update_last_sync(self, last_sync_string=None, duration_sting=None):
        if not duration_sting:
            duration = "-"
        else:
            duration = duration_sting

        if not last_sync_string:
            last_sync = "-"
        else:
            last_sync = last_sync_string

        self.lbl_last_sync.setText(f"Last sync.: {last_sync} ({duration} sec.)")

    def update_config_file_state(self):
        config_file_state, config_dict = self.configuration.config_file_state()
        self.set_config_state(config_file_state)

        if config_file_state == "Ok":
            if not self.is_synchronizing:
                self.btn_sync.setEnabled(True)
            self.update_last_sync(
                config_dict[1]["last_sync"], config_dict[1]["last_sync_duration"]
            )
            self.create_config = False
            return True
        elif config_file_state == "invalid":
            self.set_config_invalid(config_dict)
            self.create_config = False
        elif config_file_state == "Not found":
            self.set_config_not_found()
            self.create_config = True
        self.btn_sync.setEnabled(False)
        self.toggle_controls(False)

        return False

    def chk_start_selected_clicked(self):
        if self.chk_start_selected.isChecked():
            if len(self.selected_features) == 1:
                self.feat_index = None
                self.reset_navigation_buttons()
            else:
                self.warn("Please select one feature.")

    def chk_use_selected_clicked(self):
        if self.chk_use_selected.isChecked() and len(self.selected_features) < 1:
            self.chk_use_selected.setChecked(False)
            self.warn("Please select at least one feature.")
            return

        self.feat_index = None
        self.selected_feat_index = None
        self.chk_start_sel.setEnabled(not self.chk_use_selected.isChecked())
        self.chk_select.setEnabled(not self.chk_use_selected.isChecked())

        self.reset_navigation_buttons()

    def reset_navigation_buttons(self):
        self.toggle_navigation_btn(self.btn_prev, False)
        self.toggle_navigation_btn(self.btn_next, True)
        self.toggle_navigation_btn(self.btn_first_feature, False)
        self.toggle_navigation_btn(self.btn_last_feature, True)
        self.btn_show_osm.setEnabled(False)

    def set_config_invalid(self, config_dict):
        if config_dict[0]:
            self.warn(
                f"{config_dict[1]['sync_state_attribute']} or {config_dict[1]['osm_id_attribute']} does not exist"
            )
        else:
            self.warn(config_dict[1])
        self.btn_sync.setEnabled(False)

    def set_config_not_found(self):
        self.update_last_sync()
        self.btn_sync.setEnabled(False)

    def set_config_state(self, state):
        self.lbl_configuration_state.setText(f"Configuration: {state}")

    def osm_id_feature_dict(self):
        osm_id_fields_attributes = {}
        for nr, feature in enumerate(self.layer.getFeatures()):
            fields = [field.name() for field in feature.fields()]
            attributes = feature.attributes()
            osm_key = f"{attributes[fields.index(self.configuration.osm_id_attribute)]}"
            if osm_key == "NULL":
                osm_key = feature[self.configuration.primary_key_name]
            osm_id_fields_attributes[osm_key] = feature
        return osm_id_fields_attributes

    def rad_btn_state(self, button):
        if button.isChecked():
            if self.changed_state_with_show:
                self.changed_state_with_show = False
            else:
                if self.layer and self.layer.isEditable():
                    if self.button_text(button) not in self.state_change_blacklist[
                        self.current_feature()[self.configuration.state_field_name]
                    ]:
                        self.change_current_feature_state(self.button_text(button))
                    else:
                        self.warn(
                            f"Cannot change state '{self.current_feature()[self.configuration.state_field_name]}'"
                            f" to '{self.button_text(button)}'"
                        )
                        self.set_state(self.button_text(self.released_button))
                else:
                    self.warn(
                        "Please select a layer and activate edit mode first."
                    )
                    self.set_state(self.button_text(self.released_button))
        else:
            self.released_button = button

    def change_current_feature_state(self, button_text):
        self.user_changed_state = True
        if button_text == self.state_undefined_text:
            self.edit_feature_state(
                self.current_feature(), NULL,
            )
        else:
            self.edit_feature_state(
                self.current_feature(), button_text,
            )

    def edit_feature_state(self, feature, new_state):
        edit_attribute(
            self.layer, self.configuration.state_field_name, feature, new_state,
        )

    @staticmethod
    def button_text(button):
        return button.text().split(" (")[0]

    def btn_reload_pressed(self):
        if self.create_config:
            if not create_config_file(self.layer.dataProvider().dataSourceUri().split('|')[0]):
                self.warn("Could not create config file: Permission denied.")
        if not self.controls_toggled_on:
            self.load_settings()
        else:
            self.update_config_file_state()

    def btn_sync_pressed(self):
        if self.layer and not self.layer.isEditable():
            self.warn(
                "Please select a layer and activate edit mode first."
            )
        else:
            self.show_confirmation_dialog()
            if self.confirmed:
                self.is_synchronizing = True
                self.cbo_layer.setEnabled(False)
                self.btn_sync.setEnabled(False)
                # save any changes the user might have done before syncing
                self.layer.commitChanges()
                self.createBackup()
                self.layer.startEditing()
                self.synchronize_async()
                self.confirmed = False

    def btn_show_osm_pressed(self):
        if self.configuration.osm_id_attribute in field_names(self.current_feature()):
            osm_id = self.current_feature().attribute(
                self.configuration.osm_id_attribute
            )
            webbrowser.open(f"https://www.openstreetmap.org/{osm_id}")
        else:
            self.warn(
                f'no "{self.configuration.osm_id_attribute}" field found'
            )

    def toggle_next_buttons(self, should_activate):
        self.toggle_navigation_btn(self.btn_next, should_activate)
        self.toggle_navigation_btn(self.btn_last_feature, should_activate)

    def toggle_prev_buttons(self, should_activate):
        self.toggle_navigation_btn(self.btn_prev, should_activate)
        self.toggle_navigation_btn(self.btn_first_feature, should_activate)

    def toggle_navigation_btn(self, btn, should_activate):
        font = QtGui.QFont()
        font.setBold(should_activate)
        btn.setFont(font)

        btn.setEnabled(should_activate)

    def show_state(self):
        if self.layer and self.feat_index is not None:
            state_attribute = str(
                self.current_feature().attributes()[
                    get_field_id(self.layer, self.configuration.state_field_name)
                ]
            )
        else:
            state_attribute = ""

        if state_attribute not in self.states.keys():
            self.set_state(self.state_undefined_text)
        else:
            self.set_state(state_attribute)

    def set_state(self, state_attribute):
        if not self.states[state_attribute].isChecked():
            self.changed_state_with_show = True
            self.states[state_attribute].setChecked(True)

    def show_id(self):
        if self.layer and self.configuration.osm_id_attribute in field_names(self.layer):
            if self.feat_index is not None:
                osm_id = self.current_feature().attribute(self.configuration.osm_id_attribute)
                self.txt_primarykey.setText(str(osm_id))
            else:
                self.txt_primarykey.setText("-")
            self.txt_primarykey.setCursorPosition(0)
            self.lbl_primarykey.setText(f"{self.configuration.osm_id_attribute}:")
        else:
            self.txt_primarykey.setText("-")
            self.lbl_primarykey.setText("None:")

    def change_feature_count(self):
        if not self.layer_id or self.feat_index is None:
            self.write_feature_count("-")
        else:
            self.write_feature_count(self.feat_index + 1)

    def write_feature_count(self, feature_nr):
        self.lbl_feature_value.setText(
            f"Feature no. {feature_nr} of {len(self.feats_list)}"
        )

    def next_feat(self, button_type):
        self.toggle_user_changed_states(True)

        if self.chk_start_sel.isChecked():
            try:
                self.start_from_selected()
            except ValueError:
                return
        elif self.chk_use_selected.isChecked():
            self.go_to_next_selected_feature(button_type)

        else:
            self.feat_index = self.change_index(button_type, self.feat_index, self.feats_list)

        self.focus_on_feature()

        if self.chk_select.isChecked():
            select_feature(self.layer, self.current_feature())

        self.change_feature_count()
        self.show_state()
        self.show_id()
        self.btn_show_osm.setEnabled(True)

    def focus_on_feature(self):
        geom = self.current_feature().geometry()
        if geom is None or geom.isNull():
            self.warn("The geometry of the feature is null: can neither zoom nor pan to it.")
        else:
            self.move_to_geometry(geom)
            self.iface.mapCanvas().refresh()

    def go_to_next_selected_feature(self, button_type):
        self.selected_feat_index = self.change_index(button_type, self.selected_feat_index, self.selected_features)
        reference_id = self.selected_features[self.selected_feat_index].id()
        self.feat_index = first_occurrence_index(reference_id, (feat.id() for feat in self.feats_list))

    def change_index(self, button_type, index, generic_list):
        if index is None:
            index = 0
        elif button_type == ButtonType.NEXT:
            index += 1
        elif button_type == ButtonType.PREVIOUS:
            index += -1
        elif button_type == ButtonType.FIRST:
            index = 0
        elif button_type == ButtonType.LAST:
            index = len(generic_list) - 1
        else:
            self.warn("selected_feat_index is invalid")

        self.enable_disable_navigation(index, generic_list)
        return index

    def enable_disable_navigation(self, index, generic_list):
        has_next = index < len(generic_list) - 1
        self.toggle_navigation_btn(self.btn_next, has_next)
        self.toggle_navigation_btn(self.btn_last_feature, has_next)

        has_previous = index > 0
        self.toggle_navigation_btn(self.btn_prev, has_previous)
        self.toggle_navigation_btn(self.btn_first_feature, has_previous)

    def move_to_geometry(self, geom):
        renderer = self.iface.mapCanvas().mapSettings()

        if self.rad_action_pan.isChecked():
            self.iface.mapCanvas().setCenter(
                renderer.layerToMapCoordinates(self.layer, geom.centroid().asPoint())
            )
        elif self.rad_action_zoom.isChecked():
            max_length = max(geom.boundingBox().width(), geom.boundingBox().height())

            self.iface.mapCanvas().setExtent(
                renderer.layerToMapCoordinates(self.layer, geom.boundingBox().buffered(max_length * 0.4))
            )

    def start_from_selected(self):
        if len(self.selected_features) != 1:
            self.warn("Please select exactly one feature.")
            raise ValueError()

        reference_id = self.selected_features[0].id()
        self.feat_index = first_occurrence_index(reference_id, (feat.id() for feat in self.feats_list))
        self.chk_start_sel.setChecked(False)
        self.enable_disable_navigation(self.feat_index, self.feats_list)

    def set_layercombo_index(self, combo, layer_id):
        index = combo.findData(layer_id)
        if index >= 0:
            combo.setCurrentIndex(index)
            return get_layer_from_id(self.get_combo_current_data(combo))
        else:
            if combo.count() > 0:
                combo.setCurrentIndex(0)
                return combo.itemData(0)
            else:
                return None

    def synchronize_async(self):
        self.cbo_attrib_activated()
        self.start_sync_time = time.time()
        globals()["synchronizing"] = QgsTask.fromFunction(
            "Synchronizing",
            self.start_synchronizing,
            on_finished=self.done_synchronizing,
            layer=self.layer,
            state_field_name=self.configuration.state_field_name,
            osm_id_attribute=self.configuration.osm_id_attribute,
            primary_key_name=self.configuration.primary_key_name,
            config_dict=self.configuration.config_dict[1],
            osm_id_feature_dict=self.osm_id_feature_dict(),
            overpass_api=self.configuration.config_dict[1]["overpass_api"],
        )
        QgsApplication.taskManager().addTask(globals()["synchronizing"])

    def start_synchronizing(
            self,
            feedback,
            layer,
            state_field_name,
            osm_id_attribute,
            primary_key_name,
            config_dict,
            osm_id_feature_dict,
            overpass_api,
    ):
        try:
            result = Synchronizer(
                feedback, layer, state_field_name, osm_id_attribute, primary_key_name, overpass_api
            ).synchronize(config_dict, osm_id_feature_dict, self.del_not_found)
            print(result[1])
        except Exception as e:
            traceback.print_exc()
            result = (False, e)
        return result

    def done_synchronizing(self, _, result):
        self.is_synchronizing = False
        self.cbo_layer.setEnabled(True)
        self.btn_sync.setEnabled(True)
        self.cbo_layer_activated(False)
        if result[0]:
            passed_time = passed_sec(self.start_sync_time)
            self.update_last_sync(date_time(), duration_sting=passed_time)

            if found_config_file(self.layer):
                update_config_file(
                    get_config_file_path(self.layer), date_time(), passed_time
                )
            else:
                self.warn(
                    'The config file was not found -> could not update "Last Sync"/"Last Sync Duration'
                )
        else:
            self.warn(
                f"The syncing process was unsuccessful: {result[1]}."
            )

    def show_confirmation_dialog(self):
        dialog = QMessageBox()
        dialog.setWindowTitle("Synchronize..")
        dialog.setText("Save Layer and continue synchronizing?")
        dialog.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)
        dialog.buttonClicked.connect(self.dialog_btn)

        check_box = self.create_check_box()
        dialog.layout().addWidget(check_box, 3, 1)

        dialog.exec_()

        self.del_not_found = check_box.isChecked()

    def createBackup(self):
        create_backup_file(self.layer.dataProvider().dataSourceUri().split('|')[0])

    @staticmethod
    def create_check_box():
        check_box = QCheckBox()
        check_box.setText('Delete features with state set to "not found" ?')
        check_box.setChecked(True)
        return check_box

    def dialog_btn(self, btn_text):
        if btn_text.text() == "OK":
            self.confirmed = True

    @staticmethod
    def get_combo_current_data(combo):
        index = combo.currentIndex()
        return combo.itemData(index)

    def current_feature(self):
        if self.feat_index >= len(self.feats_list):
            self.feat_index = len(self.feats_list) - 1
            geom = self.current_feature().geometry()
            self.move_to_geometry(geom)
        return self.feats_list[self.feat_index]

    def warn(self, message):
        self.iface.messageBar().pushMessage(
            OSMDataSyncDockWidget.plugin_name, message, Qgis.Warning,
        )
