# main.py - New NewQuickAzimuthPlus plugin
# Combined and fixed plugin (commit behavior adapted from main.py and new_main features)
from qgis.PyQt.QtWidgets import (
    QDockWidget, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit,
    QTableWidget, QTableWidgetItem, QPushButton, QComboBox,
    QHeaderView, QAbstractItemView, QDialog, QFormLayout, QDialogButtonBox,
    QCheckBox, QDateEdit, QDateTimeEdit
)
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsProject, QgsPointXY, QgsGeometry, QgsVectorLayer,
    QgsFeature, QgsWkbTypes, QgsCoordinateTransform, QgsCoordinateReferenceSystem,
    QgsField
)
from qgis.gui import QgsRubberBand
import math
from qgis.PyQt.QtCore import QVariant

class NewQuickAzimuthPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.dock = None

        self.rubber_band = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        self.rubber_band.setColor(QColor(255, 0, 0, 220))
        self.rubber_band.setWidth(2)
        self.rubber_band.setFillColor(QColor(0, 0, 0, 0))

        self.points_layer_crs = []
        self.selected_layer = None
        self._xfm_input_to_layer = None
        self._xfm_layer_to_input = None
        self._xfm_layer_to_canvas = None

        # Watch QGIS layer changes if available
        try:
            self.iface.layerTreeView().currentLayerChanged.connect(self._on_active_layer_changed)
        except Exception:
            pass
        QgsProject.instance().layerWillBeRemoved.connect(self._on_layer_removed)
        QgsProject.instance().layersAdded.connect(self._refresh_polygon_layer_list)

    def initGui(self):
        self.action_button = QPushButton("🧭 NewQuickAzimuth+")
        self.action_button.clicked.connect(self.open_dock)
        self.iface.addToolBarWidget(self.action_button)

    def unload(self):
        if self.dock:
            self.iface.removeDockWidget(self.dock)
            self.dock = None
        if self.action_button:
            self.action_button.deleteLater()
            self.action_button = None
        try:
            self.iface.layerTreeView().currentLayerChanged.disconnect(self._on_active_layer_changed)
        except Exception:
            pass
        try:
            QgsProject.instance().layerWillBeRemoved.disconnect(self._on_layer_removed)
        except Exception:
            pass
        try:
            QgsProject.instance().layersAdded.disconnect(self._refresh_polygon_layer_list)
        except Exception:
            pass
        if self.selected_layer:
            try:
                self.selected_layer.selectionChanged.disconnect(self.on_selection_changed)
            except Exception:
                pass

    def open_dock(self):
        if self.dock and self.dock.isVisible():
            self.dock.raise_()
            return
        elif self.dock:
            self.dock.show()
            return

        self.dock = QDockWidget("NewQuickAzimuth+", self.iface.mainWindow())
        self.widget = QWidget()
        layout = QVBoxLayout()

        # Layer dropdown
        layer_layout = QHBoxLayout()
        self.layer_combo = QComboBox()
        self.layer_combo.currentIndexChanged.connect(self._on_layer_dropdown_changed)
        layer_layout.addWidget(QLabel("Polygon Layer:"))
        layer_layout.addWidget(self.layer_combo)
        layout.addLayout(layer_layout)

        # Input CRS
        crs_layout = QHBoxLayout()
        self.input_crs_combo = QComboBox()
        self.input_crs_combo.setEditable(True)
        project_crs = QgsProject.instance().crs().authid() or "EPSG:4326"
        self.input_crs_combo.addItems([project_crs, "EPSG:26331", "EPSG:26332", "EPSG:4326"])
        self.input_crs_combo.setCurrentText(project_crs)
        self.input_crs_combo.currentTextChanged.connect(self.rebuild_transforms)
        crs_layout.addWidget(QLabel("Input CRS:"))
        crs_layout.addWidget(self.input_crs_combo)
        layout.addLayout(crs_layout)

        # Start coordinates
        coord_layout = QHBoxLayout()
        self.x_input = QLineEdit()
        self.x_input.setPlaceholderText("X / Easting")
        self.y_input = QLineEdit()
        self.y_input.setPlaceholderText("Y / Northing")
        coord_layout.addWidget(QLabel("Start X:"))
        coord_layout.addWidget(self.x_input)
        coord_layout.addWidget(QLabel("Y:"))
        coord_layout.addWidget(self.y_input)
        layout.addLayout(coord_layout)

        # Info labels
        self.layer_info = QLabel("Active Polygon Layer: (none)")
        self.layer_crs_info = QLabel("Layer CRS: (unknown)")
        layout.addWidget(self.layer_info)
        layout.addWidget(self.layer_crs_info)

        # Table
        self.table = QTableWidget(3, 3)
        self.table.setHorizontalHeaderLabels(["Azimuth °", "Azimuth '", "Distance"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(self.table)

        # Row controls
        row_btns = QHBoxLayout()
        add_btn = QPushButton("➕ Add Row")
        del_btn = QPushButton("➖ Delete Row")
        add_btn.clicked.connect(self.add_row)
        del_btn.clicked.connect(self.delete_row)
        row_btns.addWidget(add_btn)
        row_btns.addWidget(del_btn)

        up_btn = QPushButton("⬆️ Move Up")
        down_btn = QPushButton("⬇️ Move Down")
        up_btn.clicked.connect(self.move_row_up)
        down_btn.clicked.connect(self.move_row_down)
        row_btns.addWidget(up_btn)
        row_btns.addWidget(down_btn)
    
        layout.addLayout(row_btns)

        # Action buttons
        btns = QHBoxLayout()
        self.preview_btn = QPushButton("👁 Preview")
        self.commit_btn = QPushButton("✅ Commit")
        reset_btn = QPushButton("🔄 New")
        self.preview_btn.clicked.connect(self.preview)
        self.commit_btn.clicked.connect(self.commit)
        reset_btn.clicked.connect(self.reset)
        btns.addWidget(self.preview_btn)
        btns.addWidget(self.commit_btn)
        btns.addWidget(reset_btn)
        layout.addLayout(btns)

        self.widget.setLayout(layout)
        self.dock.setWidget(self.widget)
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dock)
        self.dock.closeEvent = self.on_dock_closed
        self.dock.show()

        self._refresh_polygon_layer_list()
        self._select_active_layer()

    def _refresh_polygon_layer_list(self):
        self.layer_combo.blockSignals(True)
        self.layer_combo.clear()
        layers = QgsProject.instance().mapLayers().values()
        self.polygon_layers = [lyr for lyr in layers if isinstance(lyr, QgsVectorLayer) and lyr.geometryType() == QgsWkbTypes.PolygonGeometry]
        for lyr in self.polygon_layers:
            self.layer_combo.addItem(f"{lyr.name()} ({lyr.crs().authid()})", lyr.id())
        self.layer_combo.blockSignals(False)
        self._select_active_layer()

    def _select_active_layer(self):
        active = self.iface.activeLayer()
        if isinstance(active, QgsVectorLayer) and active.geometryType() == QgsWkbTypes.PolygonGeometry:
            idx = self.layer_combo.findData(active.id())
            if idx >= 0:
                self.layer_combo.setCurrentIndex(idx)
                self._on_layer_dropdown_changed(idx)
        else:
            self._set_selected_layer(None)

    def _on_layer_dropdown_changed(self, index):
        if index < 0 or index >= len(self.polygon_layers):
            self._set_selected_layer(None)
            return
        layer = self.polygon_layers[index]
        self._set_selected_layer(layer)

    def _set_selected_layer(self, layer):
        if self.selected_layer:
            try:
                self.selected_layer.selectionChanged.disconnect(self.on_selection_changed)
            except Exception:
                pass

        if not isinstance(layer, QgsVectorLayer) or layer.geometryType() != QgsWkbTypes.PolygonGeometry:
            self.selected_layer = None
            self.layer_info.setText("Active Polygon Layer: (none)")
            self.layer_crs_info.setText("Layer CRS: (unknown)")
            self.preview_btn.setEnabled(False)
            self.commit_btn.setEnabled(False)
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            return

        self.selected_layer = layer
        self.layer_info.setText(f"Active Polygon Layer: {layer.name()}")
        self.layer_crs_info.setText(f"Layer CRS: {layer.crs().authid()}")
        self.preview_btn.setEnabled(True)
        self.commit_btn.setEnabled(True)
        self.rebuild_transforms()

        self.selected_layer.selectionChanged.connect(self.on_selection_changed)

    def _on_active_layer_changed(self, layer):
        self._refresh_polygon_layer_list()

    def _on_layer_removed(self, layer_id):
        if self.selected_layer and self.selected_layer.id() == layer_id:
            self.selected_layer = None
        self._refresh_polygon_layer_list()

    def on_selection_changed(self, selected_ids, _, __):
        if not self.selected_layer or not selected_ids:
            return

        feature = self.selected_layer.getFeature(selected_ids[0])
        geom = feature.geometry()
        if not geom or geom.isEmpty():
            return

        polygon = geom.asPolygon()
        if not polygon or len(polygon[0]) < 4:
            return

        vertices = polygon[0][:-1]

        tc = QgsProject.instance().transformContext()
        layer_crs = self.selected_layer.crs()
        input_crs = self._crs_from_user_input(self.input_crs_combo.currentText())
        transformer = QgsCoordinateTransform(layer_crs, input_crs, tc)
        try:
            transformed_vertices = [transformer.transform(pt) for pt in vertices]
        except Exception:
            transformed_vertices = vertices

        self.x_input.setText(f"{transformed_vertices[0].x():.3f}")
        self.y_input.setText(f"{transformed_vertices[0].y():.3f}")

        self.table.setRowCount(len(transformed_vertices) - 1)
        for i in range(len(transformed_vertices) - 1):
            p1 = transformed_vertices[i]
            p2 = transformed_vertices[i + 1]

            dx = p2.x() - p1.x()
            dy = p2.y() - p1.y()
            dist = math.sqrt(dx**2 + dy**2)
            angle_rad = math.atan2(dx, dy)
            angle_deg = math.degrees(angle_rad)
            if angle_deg < 0:
                angle_deg += 360

            deg = int(angle_deg)
            mins = (angle_deg - deg) * 60

            self.table.setItem(i, 0, QTableWidgetItem(f"{deg:.0f}"))
            self.table.setItem(i, 1, QTableWidgetItem(f"{mins:.2f}"))
            self.table.setItem(i, 2, QTableWidgetItem(f"{dist:.3f}"))

    def _crs_from_user_input(self, text):
        text = (text or "").strip()
        crs = QgsCoordinateReferenceSystem(text)
        if not crs.isValid():
            crs = QgsProject.instance().crs()
        return crs

    def rebuild_transforms(self):
        layer_crs = self.selected_layer.crs() if self.selected_layer else None
        input_crs = self._crs_from_user_input(self.input_crs_combo.currentText())
        canvas_crs = self.canvas.mapSettings().destinationCrs()
        tc = QgsProject.instance().transformContext()

        self._xfm_input_to_layer = None
        self._xfm_layer_to_input = None
        self._xfm_layer_to_canvas = None

        if layer_crs and input_crs.isValid():
            try:
                self._xfm_input_to_layer = QgsCoordinateTransform(input_crs, layer_crs, tc)
                self._xfm_layer_to_input = QgsCoordinateTransform(layer_crs, input_crs, tc)
            except Exception:
                self._xfm_input_to_layer = None
                self._xfm_layer_to_input = None

        if layer_crs and canvas_crs:
            try:
                self._xfm_layer_to_canvas = QgsCoordinateTransform(layer_crs, canvas_crs, tc)
            except Exception:
                self._xfm_layer_to_canvas = None

    def _start_point_layer_crs_from_inputs(self):
        try:
            x = float(self.x_input.text())
            y = float(self.y_input.text())
        except ValueError:
            return None

        self.rebuild_transforms()
        if not self._xfm_input_to_layer:
            return QgsPointXY(x, y)
        try:
            return self._xfm_input_to_layer.transform(QgsPointXY(x, y))
        except Exception:
            return QgsPointXY(x, y)

    def _build_points_in_layer_crs(self):
        start = self._start_point_layer_crs_from_inputs()
        if start is None:
            return []

        points = [start]
        current = QgsPointXY(start.x(), start.y())
        for i in range(self.table.rowCount()):
            item_deg = self.table.item(i, 0)
            item_min = self.table.item(i, 1)
            item_dst = self.table.item(i, 2)
            if not item_deg or not item_min or not item_dst:
                continue
            try:
                deg = float(item_deg.text())
                mins = float(item_min.text())
                dist = float(item_dst.text())
            except Exception:
                continue

            angle = math.radians(deg + mins / 60.0)
            dx = dist * math.sin(angle)
            dy = dist * math.cos(angle)
            current = QgsPointXY(current.x() + dx, current.y() + dy)
            points.append(current)

        if points and (points[0].x() != points[-1].x() or points[0].y() != points[-1].y()):
            points.append(points[0])
        return points

    def preview(self):
        if not self.selected_layer:
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            return

        pts_layer = self._build_points_in_layer_crs()
        if not pts_layer or len(pts_layer) < 4:
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            return

        self.points_layer_crs = pts_layer[:]
        self.rebuild_transforms()

        pts_canvas = pts_layer
        if self._xfm_layer_to_canvas:
            try:
                pts_canvas = [self._xfm_layer_to_canvas.transform(p) for p in pts_layer]
            except Exception:
                pts_canvas = pts_layer

        geom_canvas = QgsGeometry.fromPolygonXY([pts_canvas])
        self.rubber_band.setToGeometry(geom_canvas, None)

        extent = geom_canvas.boundingBox()
        if extent and not extent.isEmpty():
            try:
                self.canvas.setExtent(extent)
            except Exception:
                pass
            self.canvas.refresh()

    def commit(self):
        # Build polygon points if not already available
        if not self.points_layer_crs or len(self.points_layer_crs) < 4:
            self.points_layer_crs = self._build_points_in_layer_crs()
            if not self.points_layer_crs or len(self.points_layer_crs) < 4:
                try:
                    self.iface.messageBar().pushWarning("NewQuickAzimuth+", "No valid polygon points to commit.")
                except Exception:
                    pass
                return

        # Determine layer from combo (supports itemData id or name fallback)
        layer = None
        try:
            idx = self.layer_combo.currentIndex()
            data = self.layer_combo.itemData(idx)
            if data:
                lyr = QgsProject.instance().mapLayer(data)
                if isinstance(lyr, QgsVectorLayer):
                    layer = lyr
        except Exception:
            layer = None

        if not layer:
            simple_name = self.layer_combo.currentText().split(" (")[0]
            layer = next((l for l in QgsProject.instance().mapLayers().values() if l.name() == simple_name), None)

        if not layer:
            try:
                self.iface.messageBar().pushWarning("NewQuickAzimuth+", "No valid target layer selected.")
            except Exception:
                pass
            return

        # Ensure required fields exist
        required_fields = [
            "objectid", "plot_no_tdp", "file_no_new", "file_no_old", "owner",
            "tdp_size", "lga", "location", "street_name", "problem",
            "plot_match", "remark", "document", "origincontrol", "surveyed_by",
            "surveyed_date", "ds_created", "ds_created_by", "ds_modified",
            "ds_modified_by", "gdb_geomattr_data", "migrationid"
        ]

        provider = layer.dataProvider()
        existing_fields = [f.name().lower() for f in layer.fields()]
        new_fields = []
        for field in required_fields:
            if field.lower() not in existing_fields:
                new_fields.append(QgsField(field, QVariant.String))

        if new_fields:
            try:
                provider.addAttributes(new_fields)
                layer.updateFields()
            except Exception:
                try:
                    if not layer.isEditable():
                        layer.startEditing()
                    layer.dataProvider().addAttributes(new_fields)
                    layer.updateFields()
                except Exception:
                    pass

        # Start editing if not already
        if not layer.isEditable():
            layer.startEditing()

        # Create feature and set geometry
        f = QgsFeature(layer.fields())
        geom = QgsGeometry.fromPolygonXY([self.points_layer_crs])
        f.setGeometry(geom)

        try:
            ok = layer.addFeature(f)
        except Exception:
            try:
                f = QgsFeature()
                f.setGeometry(geom)
                layer.addFeature(f)
            except Exception:
                return

        # Open attribute form (same behavior as main.py)
        try:
            self.iface.openFeatureForm(layer, f)
        except Exception:
            pass

        try:
            layer.triggerRepaint()
        except Exception:
            pass

    def add_row(self):
        self.table.insertRow(self.table.rowCount())

    def delete_row(self):
        selected = self.table.currentRow()
        if selected >= 0:
            self.table.removeRow(selected)

    def reset(self):
        self.x_input.clear()
        self.y_input.clear()
        self.table.clearContents()
        self.table.setRowCount(3)
        self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
        self.points_layer_crs = []
        
        # ✅ Force reset Input CRS to UTM Zone 32 (EPSG:26332)
        try:
            self.input_crs_combo.setCurrentText("EPSG:26332")
            self.iface.messageBar().pushMessage(
                "NewQuickAzimuth+",
                "Input CRS has been reset to UTM Zone 32 (EPSG:26332).",
                level=0, duration=3
            )
        except Exception:
            pass

    def on_dock_closed(self, event):
        event.ignore()
        self.dock.hide()


    def move_row_up(self):
        current_row = self.table.currentRow()
        if current_row > 0:
            self._swap_rows(current_row, current_row - 1)
            self.table.selectRow(current_row - 1)

    def move_row_down(self):
        current_row = self.table.currentRow()
        if current_row < self.table.rowCount() - 1:
            self._swap_rows(current_row, current_row + 1)
            self.table.selectRow(current_row + 1)

    def _swap_rows(self, row1, row2):
        for col in range(self.table.columnCount()):
            item1 = self.table.takeItem(row1, col)
            item2 = self.table.takeItem(row2, col)
            self.table.setItem(row1, col, item2)
            self.table.setItem(row2, col, item1)
    