# -*- coding: utf-8 -*-
"""
/***************************************************************************
 NADMaps
                                 A QGIS plugin
 Centrale plek om handige kaarten voor waterketen en rioolbeheer te vinden en snel in te laden.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-01-09
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Netwerk Waterketen Delfland
        email                : dataplatform@waterketendelfland.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

# https://hinot.alwaysdata.net/astuces/84-pyqgis?showall=1
# General packages
import getpass
import os.path

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsProject,
    QgsProcessingFeedback,
    QgsApplication,
)
from qgis.PyQt.QtCore import QCoreApplication, QSettings, Qt, QTimer, QThread, QUrl
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import (
    QAction,
    QFileDialog,
    QSizePolicy,
    QMessageBox,
    QDockWidget,
)

from .lib.constants import (
    PLUGIN_NAME,
    ADMIN_USERNAMES,
    PAPER_OPTIONS,
    FORMAT_OPTIONS,
    PLACEMENT_OPTIONS,
    PRINT_QUALITY_OPTIONS,
    WIKI_URL,
)

from .gui.nad_maps_dockwidget import NADMapsDockWidget

# from .lib.load_layers import LoadLayers ### LayerManager
from .lib.layer import LayerManager
from .lib.thema import ThemaManager
from .lib.style import StyleManager
from .lib.log import LoggingManager
from .lib.export import ExportManager
from .lib.search_location import SearchLocationManager
from .lib.ingest import IngestLayersManager

#########################################################################################
####################  Run main script to initiate when NAD button is pressed ############
#########################################################################################


class NADMaps:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.dlg = NADMapsDockWidget(parent=self.iface.mainWindow())
        self.dlg.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Maximum)

        if getpass.getuser() in ADMIN_USERNAMES:
            self.creator = "Plugin"
        else:
            self.creator = getpass.getuser()
            self.dlg.groupBoxGetLayers.setVisible(False)

        self.log_manager = LoggingManager(dlg=self.dlg)
        self.log = self.log_manager.log

        self.log_manager = LoggingManager(dlg=self.dlg)
        self.log = self.log_manager.log

        # initialize the working directory from settings
        # QSettings().setValue("NADmaps/working_dir", "") # for testing purposes, always re-comment after usage!
        self.working_dir = QSettings().value("NADmaps/working_dir")
        if self.working_dir in ["", None]:
            set_directory = QMessageBox.question(
                self.dlg,
                "Werkmap selecteren",
                "Deze plug-in gebruikt een lokale werkmap om goed te functioneren. Wilt u de werkmap nu selecteren? Anders kunt u later nog de werkmap veranderen of instellen in het tabblad 'Instellingen'",
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
            )
            if set_directory == QMessageBox.StandardButton.Yes:
                self.working_dir = QFileDialog.getExistingDirectory(
                    self.dlg, "Selecteer een werkmap", ""
                )

        # if user did not select a working directory, then skip the creation of folder and path creation
        if self.working_dir in ["", None]:  
            self.log(
                "Geen werkmap opgegeven. De plugin kan niet goed functioneren zonder werkmap.",
                1,
            )  # set warning message
        else:
            os.makedirs(self.working_dir, exist_ok=True)
            os.makedirs(os.path.join(self.working_dir, "styling"), exist_ok=True)
            os.makedirs(
                os.path.join(self.working_dir, "styling", "qml_files"), exist_ok=True
            )

            # save the working directory to the settings, such that it is available next time the plugin is started
            QSettings().setValue("NADmaps/working_dir", self.working_dir)
            self.dlg.lineEditFilePath.setText(self.working_dir)

            self.user_styling_path = os.path.join(
                self.working_dir, "styling", "styling.json"
            )
            self.user_styling_files_path = os.path.join(
                self.working_dir, "styling", "qml_files"
            )

        # define plugin paths
        self.plugin_styling_path = os.path.join(
            self.plugin_dir, "resources", "styling", "styling.json"
        )
        self.plugin_styling_files_path = os.path.join(
            self.plugin_dir, "resources", "styling", "qml_files"
        )

        self.style_manager = StyleManager(
            dlg=self.dlg,
            iface=self.iface,
            plugin_dir=self.plugin_dir,
            working_dir=self.working_dir,
            creator=self.creator,
            log=self.log,
        )
        self.search_manager = SearchLocationManager(
            dlg=self.dlg, iface=self.iface, log=self.log
        )
        self.layer_manager = LayerManager(
            dlg=self.dlg,
            iface=self.iface,
            plugin_dir=self.plugin_dir,
            style_manager=self.style_manager,
            log=self.log,
        )
        self.thema_manager = ThemaManager(
            dlg=self.dlg,
            plugin_dir=self.plugin_dir,
            working_dir=self.working_dir,
            creator=self.creator,
            log=self.log,
        )
        self.ingest_manager = IngestLayersManager(
            dlg=self.dlg,
            iface=self.iface,
            plugin_dir=self.plugin_dir,
            log=self.log,
        )

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&NAD Waterketen Kaarten")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        # will be set True in run()
        self.setup_completed = False
        self.current_layer = None
        self.selected_active_layers = None
        self.selected_layer = None
        self.zoom_completed = False

        # Check if the autostart option is set to true in the settings
        self.autostart_triggered = False

        if QSettings().value("NADmaps/autostart", "false") == "true":
            self.log("Autostart is enabled...")
            task_mgr = QgsApplication.taskManager()
            task_mgr.allTasksFinished.connect(self.safe_autostart)
            # ^ In case a task is already running, this will trigger the autostart

            if not self.autostart_triggered:
                # Fallback timer in case no tasks are running
                QTimer.singleShot(1500, self.safe_autostart)

        if QSettings().value("NADmaps/maxNumFeaturesCheck", False) == True:
            self.dlg.checkBox_MaxNumFeatures.setCheckState(Qt.CheckState(2))
        else:
            self.dlg.checkBox_MaxNumFeatures.setCheckState(Qt.CheckState(0))

        # parallel rendering
        QSettings().setValue("/qgis/parallel_rendering", True)
        threadcount = QThread.idealThreadCount()
        QgsApplication.setMaxThreads(threadcount)
        QSettings().setValue("/core/OpenClEnabled", True)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        self.run_icon = QIcon(os.path.join(self.plugin_dir, "resources", "nad.png"))

        self.add_action(
            icon_path=self.run_icon,
            text=PLUGIN_NAME,
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

    #########################################################################################
    ####################  Run main script to initiate when NAD button is pressed ############
    #########################################################################################

    def safe_autostart(self):
        try:
            self.log("Running autostart...")
            self.run(hiddenDialog=True)  # Delay the UI part
            QTimer.singleShot(1500, self.show_dialog)
            self.autostart_triggered = True
            self.log("Autostart completed successfully.")
        except Exception as e:
            self.log(f"Autostart failed. Error message: {e}")

    def show_dialog(self):
        self.log("Showing NADMaps dialog after short delay.", 0)
        self.dlg.show()
        area = self.iface.mainWindow().dockWidgetArea(self.dlg)
        if self.dlg.isFloating() or area != Qt.RightDockWidgetArea:
            self.iface.mainWindow().removeDockWidget(self.dlg)
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dlg)
            self.dlg.setFloating(False)
        self.dlg.raise_()
        self.dlg.activateWindow()

    def run(self, hiddenDialog=False):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.setup_completed == False:
            # setup the (proxy)models
            self.setup_interactions()

            # create an initial list of active layers
            self.layer_manager.update_active_layers_list()

            # create a list of existing themas
            self.thema_manager.update_thema_list()

            # Create a list of all layers available via the plugin
            self.layer_list = self.layer_manager.load_layer_list()
            self.style_manager.set_layer_list(self.layer_list)

            projectCrs = QgsCoordinateReferenceSystem.fromEpsgId(28992)
            QgsProject.instance().setCrs(projectCrs)
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dlg)

            self.setup_completed = True

        # deactivate the styling box until a layer is selected 
        self.dlg.stylingGroupBox.setEnabled(False)
        self.dlg.stylingGroupBox.setToolTip(
            "Selecteer één laag om de styling aan te passen"
        )

        # init the values for the export settings
        self.init_export_comboboxes()
        self.load_export_settings()
        self.check_map_name()  # To enable or disable pushbutton

        # init autostart checkbox
        self.dlg.checkBox_AutoStart.setChecked(
            QSettings().value("NADmaps/autostart", "false") == "true"
        )

        # init standard area
        self.dlg.lineEdit_StandardArea.setText(
            QSettings().value("NADmaps/standard_area")
        )

        # init autoload standard area checkbox
        checked = QSettings().value("NADmaps/autoload_standardarea", False, type=bool)
        self.dlg.checkBox_StandardArea.setChecked(checked)
        if not checked:  # if unchecked, zoom is not required during this session
            self.log("zoom_completed is set to True", 0)
            self.zoom_completed = True

        # init max number of features value en checkbox
        self.dlg.spinBox_MaxNumFeatures.setValue(
            int(QSettings().value("NADmaps/maxNumFeatures", 5000))
        )

        self.dlg.checkBox_MaxNumFeatures.setCheckState(
            bool(QSettings().value("NADmaps/maxNumFeaturesCheck", False))
        )
        self.dlg.checkBox_MaxNumFeatures.setTristate(False)
        self.set_maxnumfeatures_checkbox()

        # show the dialog
        if not hiddenDialog:
            self.show_dialog()

        # Zoom to standard area after render when no layer was active at startup
        self.iface.mapCanvas().renderComplete.connect(self.check_and_execute_zoom)

    #########################################################################################
    #################################  Setup functions ######################################
    #########################################################################################

    # Check if zoom should be executed
    def check_zoom_required(self):
        # Load settingsinfo
        zoom_checkbox = QSettings().value("NADmaps/autoload_standardarea", False)
        standard_area = QSettings().value("NADmaps/standard_area", "")
        # Load layer info
        # TODO: variabele 'active_layers' opnemen in __init__ en updaten na layer change/ render complete
        root = QgsProject.instance().layerTreeRoot()
        layers = root.layerOrder()
        if self.zoom_completed == True:
            return False, None
        elif zoom_checkbox == True and standard_area != "" and len(layers) >= 1:
            return True, standard_area

        return False, None

    # Zoom to standard area
    def check_and_execute_zoom(self):
        zoom_required, standard_area = self.check_zoom_required()
        if zoom_required == True:
            self.search_manager.get_lookup_id_and_zoom(standard_area)
            self.zoom_completed = True

    def open_wiki(self):
        url = QUrl(WIKI_URL)
        QDesktopServices.openUrl(url)

    def setup_interactions(self):
        """
        This does a setup of all the button interactions.
        """
        # Click functions
        self.dlg.helpButton.clicked.connect(
            lambda: self.open_wiki()
        )
        self.dlg.loadStyleButton.clicked.connect(
            lambda: self.style_manager.load_styling()
        )
        self.dlg.loadStyleButton.clicked.connect(
            lambda: self.layer_manager.update_active_layers_list()
        )
        self.dlg.removeStyleButton.clicked.connect(
            lambda: self.style_manager.delete_styling()
        )
        self.dlg.removeStyleButton.clicked.connect(
            lambda: self.layer_manager.update_active_layers_list()
        )
        self.dlg.saveStyleButton.clicked.connect(
            lambda: self.style_manager.save_styling(self.selected_layer)
        )
        self.dlg.saveStyleButton.clicked.connect(
            lambda: self.layer_manager.update_active_layers_list()
        )

        self.dlg.saveThemaButton.setEnabled(False)
        self.dlg.saveThemaButton.setToolTip("Geen lagen geselecteerd")
        self.dlg.saveThemaButton.clicked.connect(
            lambda: self.thema_manager.save_thema(False, self.selected_active_layers)
        )
        self.dlg.saveAllThemaButton.clicked.connect(
            lambda: self.thema_manager.save_thema(True, self.selected_active_layers)
        )

        self.dlg.pluginThemaCheckBox.clicked.connect(
            lambda: self.thema_manager.filter_thema_list()
        )
        self.dlg.userThemaCheckBox.clicked.connect(
            lambda: self.thema_manager.filter_thema_list()
        )
        self.dlg.favoriteThemaCheckBox.clicked.connect(
            lambda: self.thema_manager.filter_thema_list()
        )

        self.dlg.deleteThemaButton.clicked.connect(
            lambda: self.thema_manager.delete_thema()
        )

        # Setting interactions
        self.dlg.set_working_dir.clicked.connect(
            lambda: self.set_working_directory(
                QFileDialog.getExistingDirectory(
                    self.dlg, "Selecteer een werkmap", self.working_dir
                )
            )
        )
        self.dlg.checkBox_AutoStart.stateChanged.connect(
            lambda: QSettings().setValue(
                "NADmaps/autostart",
                "true" if self.dlg.checkBox_AutoStart.isChecked() else "false",
            )
        )

        # Save button for standard work area
        self.dlg.pushButton_SaveStandardArea.clicked.connect(
            lambda: self.set_standard_area()
        )

        # Save checkbox for autoload standard work area
        self.dlg.checkBox_StandardArea.stateChanged.connect(
            lambda: self.set_autoload_checkbox()
        )
        # maxNumFeatures spinbox
        self.dlg.checkBox_MaxNumFeatures.clicked.connect(
            lambda: self.set_maxnumfeatures_checkbox()
        )
        self.dlg.spinBox_MaxNumFeatures.valueChanged.connect(
            lambda: self.set_maxnumfeatures()
        )

        # Export_tab interactions
        self.dlg.lineEdit_FileName.textChanged.connect(self.check_map_name)
        self.dlg.comboBox_PapierFormaat.currentIndexChanged.connect(
            self.on_paper_format_changed
        )
        self.dlg.comboBox_BestandsFormaat.currentIndexChanged.connect(
            self.on_file_format_changed
        )
        self.dlg.comboBox_PrintQuality.currentIndexChanged.connect(
            self.on_print_quality_changed
        )
        self.dlg.checkBox_Noordpijl.stateChanged.connect(self.on_north_checkbox_changed)
        self.dlg.checkBox_Legenda.stateChanged.connect(self.on_legend_checkbox_changed)
        self.dlg.checkBox_Schaalbalk.stateChanged.connect(
            self.on_scale_checkbox_changed
        )
        self.dlg.checkBox_Titel.stateChanged.connect(self.on_titel_checkbox_changed)
        self.dlg.pushButton_ExporteerMap.clicked.connect(self.export_map_button_pressed)

        self.dlg.stylingGroupBox.setEnabled(False)

        # update the information on the current selection of active layers
        self.dlg.activeMapListView.selectionModel().selectionChanged.connect(
            self.get_selected_active_layers
        )

        # Activate to log performance by tracking load times of the canvas
        # self.iface.mapCanvas().renderStarting.connect(self.log_manager.start_time)
        # self.iface.mapCanvas().renderComplete.connect(self.log_manager.stop_time)
        self.dlg.pushButtonGetLayers.clicked.connect(self.ingest_manager.get_layer_list)

        # self.dlg.stylingGroupBox.setToolTip("Selecteer maar één laag om de styling aan te passen")

    #########################################################################################
    ################################  General utility functions ########################
    #########################################################################################

    def set_working_directory(self, path):
        """Set the working directory for the plugin"""

        # some checks if the path is not empty or a directory
        if path == "" or path == None:
            return
        if not os.path.isdir(path):
            return
        # plugin path & user path
        os.makedirs(path, exist_ok=True)
        os.makedirs(os.path.join(path, "styling"), exist_ok=True)
        os.makedirs(os.path.join(path, "styling", "qml_files"), exist_ok=True)

        self.thema_manager.set_working_directory(path)

        self.user_styling_path = os.path.join(path, "styling", "styling.json")
        self.user_styling_files_path = os.path.join(path, "styling", "qml_files")

        # save the working directory to the settings, such that it is available next time the plugin is started
        QSettings().setValue("NADmaps/working_dir", path)

        self.working_dir = path
        self.dlg.lineEditFilePath.setText(path)

    def set_standard_area(self):
        self.dlg.lineEdit_StandardArea.setText(
            self.dlg.zoomLineEdit.text()
        )  # Insert search area text into protected field
        QSettings().setValue(
            "NADmaps/standard_area", self.dlg.zoomLineEdit.text()
        )  # Save the standard area to the settings

    def set_autoload_checkbox(self):
        checked = True if self.dlg.checkBox_StandardArea.isChecked() else False
        QSettings().setValue(
            "NADmaps/autoload_standardarea",
            checked,
        )

    def get_selected_active_layers(self):
        """
        Get the selected layers from the active layers-tab
        """
        selectedIndexes = self.dlg.activeMapListView.selectedIndexes()
        nr_of_selected_rows = len(set(index.row() for index in selectedIndexes))
        # self.log(f"nr selected rows = {nr_of_selected_rows}")

        # enable or disable the styling-functions
        if nr_of_selected_rows == 1:
            self.style_manager.update_styling_list()
            self.dlg.stylingGroupBox.setEnabled(True)
            self.dlg.stylingGroupBox.setToolTip("")
            self.dlg.saveThemaButton.setEnabled(True)
            self.dlg.saveThemaButton.setToolTip("")
        elif nr_of_selected_rows > 1:
            self.dlg.stylingGroupBox.setEnabled(False)
            self.dlg.stylingGroupBox.setToolTip(
                "Selecteer één laag om de styling aan te passen"
            )
            self.dlg.saveThemaButton.setEnabled(True)
            self.dlg.saveThemaButton.setToolTip("")
        elif nr_of_selected_rows == 0:
            self.dlg.stylingGroupBox.setEnabled(False)
            self.dlg.stylingGroupBox.setToolTip(
                "Selecteer één laag om de styling aan te passen"
            )
            self.dlg.saveThemaButton.setEnabled(False)
            self.dlg.saveThemaButton.setToolTip("Geen lagen geselecteerd")
            self.selected_active_layers = None
            return

        self.selected_active_layers = []
        first_index_list = set(index.siblingAtColumn(0) for index in selectedIndexes)
        for index in first_index_list:
            active_layer = index.data(Qt.ItemDataRole.UserRole)
            self.selected_active_layers.append(active_layer)
            if nr_of_selected_rows == 1:
                self.selected_layer = active_layer
            # self.log(f"selected active layers = {self.selected_active_layers}")

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&NAD Waterketen Kaarten"), action)
            self.iface.removeToolBarIcon(action)

    # General add_action function to add action-buttons to the QGIS toolbar
    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    # General translation function (can probably be deleted)
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("NADMaps", message)

    def init_export_comboboxes(self):
        paper_items = [
            str(self.dlg.comboBox_PapierFormaat.itemText(i))
            for i in range(self.dlg.comboBox_PapierFormaat.count())
        ]
        file_items = [
            str(self.dlg.comboBox_BestandsFormaat.itemText(i))
            for i in range(self.dlg.comboBox_BestandsFormaat.count())
        ]
        print_quality_items = [
            str(self.dlg.comboBox_PrintQuality.itemText(i))
            for i in range(self.dlg.comboBox_PrintQuality.count())
        ]

        legend_placement_items = [
            str(self.dlg.comboBox_LegendaPlacement.itemText(i))
            for i in range(self.dlg.comboBox_LegendaPlacement.count())
        ]
        schaalbalk_placement_items = [
            str(self.dlg.comboBox_SchaalbalkPlacement.itemText(i))
            for i in range(self.dlg.comboBox_SchaalbalkPlacement.count())
        ]
        noordpijl_placement_items = [
            str(self.dlg.comboBox_NoordpijlPlacement.itemText(i))
            for i in range(self.dlg.comboBox_NoordpijlPlacement.count())
        ]

        if paper_items != PAPER_OPTIONS:
            self.dlg.comboBox_PapierFormaat.clear()
            for item in PAPER_OPTIONS:
                self.dlg.comboBox_PapierFormaat.addItem(item)

        if file_items != FORMAT_OPTIONS:
            self.dlg.comboBox_BestandsFormaat.clear()
            for item in FORMAT_OPTIONS:
                self.dlg.comboBox_BestandsFormaat.addItem(item)

        print_quality_options = list(PRINT_QUALITY_OPTIONS.keys())
        if print_quality_items != print_quality_options:
            self.dlg.comboBox_PrintQuality.clear()
            for item in print_quality_options:
                self.dlg.comboBox_PrintQuality.addItem(item)

        if (
            legend_placement_items != PLACEMENT_OPTIONS
            or schaalbalk_placement_items != PLACEMENT_OPTIONS
            or noordpijl_placement_items != PLACEMENT_OPTIONS
        ):
            self.dlg.comboBox_LegendaPlacement.clear()
            self.dlg.comboBox_SchaalbalkPlacement.clear()
            self.dlg.comboBox_NoordpijlPlacement.clear()
            for item in PLACEMENT_OPTIONS:
                self.dlg.comboBox_LegendaPlacement.addItem(item)
                self.dlg.comboBox_SchaalbalkPlacement.addItem(item)
                self.dlg.comboBox_NoordpijlPlacement.addItem(item)

    def save_export_settings(self):
        QSettings().setValue(
            "NADmaps/export/paper_format", self.dlg.comboBox_PapierFormaat.currentText()
        )
        QSettings().setValue(
            "NADmaps/export/file_format",
            self.dlg.comboBox_BestandsFormaat.currentText(),
        )
        QSettings().setValue(
            "NADmaps/export/print_quality", self.dlg.comboBox_PrintQuality.currentText()
        )
        QSettings().setValue(
            "NADmaps/export/include_north",
            "true" if self.dlg.checkBox_Noordpijl.isChecked() else "false",
        )
        QSettings().setValue(
            "NADmaps/export/noordpijl_placement",
            self.dlg.comboBox_NoordpijlPlacement.currentText(),
        )
        QSettings().setValue(
            "NADmaps/export/include_legend",
            "true" if self.dlg.checkBox_Legenda.isChecked() else "false",
        )
        QSettings().setValue(
            "NADmaps/export/legend_placement",
            self.dlg.comboBox_LegendaPlacement.currentText(),
        )
        QSettings().setValue(
            "NADmaps/export/include_scale",
            "true" if self.dlg.checkBox_Schaalbalk.isChecked() else "false",
        )
        QSettings().setValue(
            "NADmaps/export/scale_placement",
            self.dlg.comboBox_SchaalbalkPlacement.currentText(),
        )
        QSettings().setValue(
            "NADmaps/export/include_title",
            "true" if self.dlg.checkBox_Titel.isChecked() else "false",
        )
        QSettings().setValue("NADmaps/export/title", self.dlg.lineEdit_Titel.text())
        QSettings().setValue(
            "NADmaps/export/title_font_size",
            str(self.dlg.spinBox_TitelFontSize.value()),
        )

    def load_export_settings(self):
        saved_paper = str(QSettings().value("NADmaps/export/paper_format", "A4 staand"))
        if saved_paper in [
            str(self.dlg.comboBox_PapierFormaat.itemText(i))
            for i in range(self.dlg.comboBox_PapierFormaat.count())
        ]:
            self.dlg.comboBox_PapierFormaat.setCurrentText(saved_paper)
        else:
            self.dlg.comboBox_PapierFormaat.setCurrentIndex(0)

        saved_format = str(QSettings().value("NADmaps/export/file_format", "PNG"))
        format_items = [
            str(self.dlg.comboBox_BestandsFormaat.itemText(i))
            for i in range(self.dlg.comboBox_BestandsFormaat.count())
        ]
        if saved_format in format_items:
            self.dlg.comboBox_BestandsFormaat.setCurrentText(saved_format)
        else:
            self.dlg.comboBox_BestandsFormaat.setCurrentIndex(0)

        saved_quality = str(
            QSettings().value("NADmaps/export/print_quality", "Normale kwaliteit")
        )
        quality_items = [
            str(self.dlg.comboBox_PrintQuality.itemText(i))
            for i in range(self.dlg.comboBox_PrintQuality.count())
        ]
        if saved_quality in quality_items:
            # self.dlg.comboBox_PrintQuality.setCurrentText(saved_quality)
            self.dlg.comboBox_PrintQuality.setCurrentText("Iets geks")
        else:
            # self.dlg.comboBox_PrintQuality.setCurrentIndex(0)
            self.dlg.comboBox_PrintQuality.setCurrentText(
                f"Saved quality: {saved_quality} zit niet in de lijst van opties: {quality_items}"
            )

        self.dlg.checkBox_Noordpijl.setChecked(
            QSettings().value("NADmaps/export/include_north", "false") == "true"
        )
        self.set_noordpijl_placement_combobox()
        self.dlg.checkBox_Legenda.setChecked(
            QSettings().value("NADmaps/export/include_legend", "false") == "true"
        )
        self.set_legenda_placement_combobox()
        self.dlg.checkBox_Schaalbalk.setChecked(
            QSettings().value("NADmaps/export/include_scale", "false") == "true"
        )
        self.set_schaalbalk_placement_combobox()
        self.dlg.checkBox_Titel.setChecked(
            QSettings().value("NADmaps/export/include_title", "false") == "true"
        )
        self.set_titel_line_edit()
        self.dlg.lineEdit_Titel.setText(QSettings().value("NADmaps/export/title", ""))
        self.dlg.spinBox_TitelFontSize.setValue(
            int(QSettings().value("NADmaps/export/title_font_size", 20))
        )

    def on_paper_format_changed(self):
        self.save_export_settings()

    def on_file_format_changed(self):
        self.save_export_settings()

    def on_print_quality_changed(self):
        self.save_export_settings()

    def on_north_checkbox_changed(self):
        self.set_noordpijl_placement_combobox()
        self.save_export_settings()

    def on_legend_checkbox_changed(self):
        self.set_legenda_placement_combobox()
        self.save_export_settings()

    def on_scale_checkbox_changed(self):
        self.set_schaalbalk_placement_combobox()
        self.save_export_settings()

    def on_titel_checkbox_changed(self):
        self.save_export_settings()
        self.set_titel_line_edit()

    def set_noordpijl_placement_combobox(self):
        if self.dlg.checkBox_Noordpijl.isChecked():
            self.dlg.comboBox_NoordpijlPlacement.setVisible(True)
            self.dlg.comboBox_NoordpijlPlacement.setEnabled(True)
            self.dlg.comboBox_NoordpijlPlacement.setFocus()
        else:
            self.dlg.comboBox_NoordpijlPlacement.setVisible(False)

    def set_legenda_placement_combobox(self):
        if self.dlg.checkBox_Legenda.isChecked():
            self.dlg.comboBox_LegendaPlacement.setVisible(True)
            self.dlg.comboBox_LegendaPlacement.setEnabled(True)
            self.dlg.comboBox_LegendaPlacement.setFocus()
        else:
            self.dlg.comboBox_LegendaPlacement.setVisible(False)

    def set_schaalbalk_placement_combobox(self):
        if self.dlg.checkBox_Schaalbalk.isChecked():
            self.dlg.comboBox_SchaalbalkPlacement.setVisible(True)
            self.dlg.comboBox_SchaalbalkPlacement.setEnabled(True)
            self.dlg.comboBox_SchaalbalkPlacement.setFocus()
        else:
            self.dlg.comboBox_SchaalbalkPlacement.setVisible(False)

    def set_titel_line_edit(self):
        if self.dlg.checkBox_Titel.isChecked():
            self.dlg.lineEdit_Titel.setText(self.dlg.lineEdit_FileName.text())
            self.dlg.lineEdit_Titel.setVisible(True)
            self.dlg.spinBox_TitelFontSize.setVisible(True)
            self.dlg.lineEdit_Titel.setEnabled(True)
            self.dlg.lineEdit_Titel.setFocus()
        else:
            self.dlg.lineEdit_Titel.setVisible(False)
            self.dlg.spinBox_TitelFontSize.setVisible(False)

    def check_map_name(self):
        map_name = self.dlg.lineEdit_FileName.text()
        self.dlg.pushButton_ExporteerMap.setEnabled(bool(map_name))

    def set_maxnumfeatures(self):
        maxnumfeatures = self.dlg.spinBox_MaxNumFeatures.value()
        QSettings().setValue("NADmaps/maxNumFeatures", maxnumfeatures)

    def set_maxnumfeatures_checkbox(self):
        use_maxnumfeatures = self.dlg.checkBox_MaxNumFeatures.isChecked()
        QSettings().setValue("NADmaps/maxNumFeaturesCheck", use_maxnumfeatures)

        if use_maxnumfeatures:
            self.dlg.label_MaxNumFeatures.setEnabled(True)
            self.dlg.spinBox_MaxNumFeatures.setEnabled(True)
        else:
            self.dlg.label_MaxNumFeatures.setEnabled(False)
            self.dlg.spinBox_MaxNumFeatures.setEnabled(False)

    def export_map_button_pressed(self):
        if not os.path.exists(self.working_dir):
            self.log("Geen opslaglocatie gevonden. Selecteer eerst de juiste werkmap in de Instellingen.")
            return

        file_path = self.generate_export_path()
        if not file_path:
            self.log("Geen bestandsnaam opgegeven", 1)
            return
        if os.path.exists(file_path):
            overwrite = QMessageBox.question(
                self.dlg,
                "Bestand bestaat al",
                f"Het bestand {file_path} bestaat al. Wilt u het overschrijven?",
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
            )
            if overwrite == QMessageBox.StandardButton.No:
                return

        if not os.path.exists(os.path.dirname(file_path)):
            os.makedirs(os.path.dirname(file_path))

        export_manager = ExportManager(dlg=self.dlg, log=self.log, project=None)

        print_quality = self.dlg.comboBox_PrintQuality.currentText()
        dpi = PRINT_QUALITY_OPTIONS.get(print_quality)

        settings_dict = {
            "paper_format": self.dlg.comboBox_PapierFormaat.currentText(),
            "file_format": self.dlg.comboBox_BestandsFormaat.currentText().lower(),
            "dpi": dpi,
            "include_north": self.dlg.checkBox_Noordpijl.isChecked(),
            "north_placement": self.dlg.comboBox_NoordpijlPlacement.currentText(),
            "include_legend": self.dlg.checkBox_Legenda.isChecked(),
            "legend_placement": self.dlg.comboBox_LegendaPlacement.currentText(),
            "include_scale": self.dlg.checkBox_Schaalbalk.isChecked(),
            "scale_placement": self.dlg.comboBox_SchaalbalkPlacement.currentText(),
            "include_title": self.dlg.checkBox_Titel.isChecked(),
            "title": self.dlg.lineEdit_Titel.text(),
            "title_font_size": self.dlg.spinBox_TitelFontSize.value(),
            "canvas": self.iface.mapCanvas(),
        }
        layout = export_manager.build_layout(settings_dict)

        success = export_manager.export(layout, file_path)
        if success:
            self.log(f"Kaart succesvol geëxporteerd naar {file_path}", 3)
            QMessageBox.information(
                self.dlg,
                "Export succesvol",
                f"De kaart is succesvol geëxporteerd naar {file_path}.",
            )
        else:
            self.log(f"Fout bij het exporteren van de kaart naar {file_path}", 2)
            QMessageBox.critical(
                self.dlg,
                "Export mislukt",
                f"Het exporteren van de kaart naar {file_path} is mislukt.",
            )

    def generate_export_path(self):
        map_name = self.dlg.lineEdit_FileName.text()
        if not map_name:
            return

        file_format = self.dlg.comboBox_BestandsFormaat.currentText()
        return os.path.join(
            self.working_dir, "export", f"{map_name}.{file_format.lower()}"
        )
