# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MiMiGIS (Middlebury Minimal GIS)
                                 A QGIS plugin
 This plugin contains tools for teaching introductory QGIS.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-01-29
        copyright            : (C) 2021 by Maja Cannavo and Joseph Holler
        email                : mcannavo@middlebury.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Maja Cannavo and Joseph Holler'
__date__ = '2021-01-29'
__copyright__ = '(C) 2021 by Maja Cannavo and Joseph Holler'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import textwrap
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis import processing
from qgis.core import (QgsProcessing,
                        QgsFeatureSink,
                        QgsProcessingException,
                        QgsProcessingParameters,
                        QgsProcessingAlgorithm,
                        QgsProcessingMultiStepFeedback,
                        QgsProcessingParameterVectorLayer,
                        QgsProcessingParameterField,
                        QgsProcessingParameterBoolean,
                        QgsProcessingParameterFeatureSink,
                        QgsExpression,
                        QgsProcessingParameterFeatureSource,
                        QgsProcessingParameterVectorDestination,
                        QgsProcessingParameterVectorLayer)

# the path of the MiMiGIS plugin folder
mimigis_path = os.path.dirname(__file__)


class GroupByAlgorithm(QgsProcessingAlgorithm):
    # This is the class that contains all the information for our algorithm.
    # It extends the QgsProcessingAlgorithm class.

    # allow easier access to input and output
    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'groupby'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr('Group By')

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        # leave as empty string because we don't need groups
        return ''

    def createInstance(self):
        return GroupByAlgorithm()

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def shortHelpString(self):
        """
        Returns a localized short help string for the algorithm.
        """
        return self.tr('Group features with common values in the group field(s).\
            Optionally, dissolve geometries and calculate summary statistics for numeric fields.\
            All outputs will have a "featCount" field with the number of features in each group.\n\
            Input parameters:\nInput layer: Input layer with features to be dissolved.\n\
            Group Fields: In which field(s) do you want to search for values with which to form the new groups?\
            A new group will be formed for each combination of values in the group field(s).\
            If you do not select any group fields, the output will be a single feature.\n\
            Summary Fields: Which numerical field(s) do you want to calculate summary statistics for?\n\
            Dissolve Geometry: Do you want to dissolve the geometries (geographic data) associated with the features?\
            Disjoint geometries are still dissolved into multi-part features.\
            If this option is unchecked, the output will be a table with no geographic data.\n\
            Average: Calculate the average or mean of your summary field(s)?\n\
            Count Values: Count non-null values in your summary field(s)?\n\
            Sum: Calculate the sum of your summary field(s)?\n\
            Maximum: Calculate the maximum of your summary field(s)?\n\
            Minimum: Calculate the minimum of your summary field(s)?\n\
            Output:\nGrouped Output: New layer with grouped features and summary statistics.\
            If you are not dissolving geometries,\
            then save the output as a .csv, .xlsx, or database table.\n\
            Algorithm authors: Maja Cannavo and Joseph Holler')

    def helpUrl(self):
        return 'https://github.com/GIS4DEV/MiMiGIS'

    def icon(self):
        return QIcon(os.path.join(mimigis_path, 'icons', 'sigma.svg'))

    def initAlgorithm(self, config=None):
        """
        Defines the inputs and outputs of the algorithm.
        """

        # input layer
        self.addParameter(
            # if we use QgsProcessingParameterFeatureSource as input,
            # we will get the checkbox for "Selected Features Only" in the dialog
            QgsProcessingParameterVectorLayer(
                self.INPUT,
                self.tr('Input layer'),
                # type is vector
                [QgsProcessing.TypeVectorAnyGeometry])
            )

        self.addParameter(
            QgsProcessingParameterField(
                'GROUPFIELDS',
                'Group Fields',
                optional=True,
                type=QgsProcessingParameterField.Any,
                parentLayerParameterName='INPUT',
                allowMultiple=True,
                defaultValue=None)
            )

        self.addParameter(
            QgsProcessingParameterField(
                'SUMMARYFIELDS',
                'Summary Fields',
                optional=True,
                type=QgsProcessingParameterField.Numeric,
                parentLayerParameterName='INPUT',
                allowMultiple=True,
                defaultValue=None)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'DISSOLVEGEOMETRY',
                'Dissolve Geometry',
                defaultValue=False)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'AVERAGE',
                'Average',
                defaultValue=False)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'COUNTVALUES',
                'Count Values',
                defaultValue=False)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'SUM',
                'Sum',
                defaultValue=False)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'MAXIMUM',
                'Maximum',
                defaultValue=False)
            )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'MINIMUM',
                'Minimum',
                defaultValue=False)
            )

        # output
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Grouped Output')
                )
            )


    def processAlgorithm(self, parameters, context, feedback):

        # Retrieve the input layer
        inputLayer = self.parameterAsLayer(parameters, self.INPUT, context)

        # If source was not found, throw an exception to indicate that the algorithm
        # encountered a fatal error. The exception text can be any string, but in this
        # case we use the pre-built invalidSourceError method to return a standard
        # helper text for when a source cannot be evaluated
        if inputLayer is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))


        # construct SQL query as string

        # retrieve list of group fields
        group_fields = self.parameterAsFields(parameters, 'GROUPFIELDS', context)

        # retrieve list of summary fields
        summary_fields = self.parameterAsFields(parameters, 'SUMMARYFIELDS', context)

        # initialize string
        sql_query = 'Select count(*) as featCount'

        # check to see if group_fields is empty
        if group_fields != []:
            # if not empty, concatenate group field names to the query
            # we use backticks to protect against "bad" field names that could
            # cause an error
            for field in group_fields:
                sql_query += ', `' + field + '`'

        # see if we need to count values
        if parameters['COUNTVALUES'] and summary_fields != []:
            for field in summary_fields:
                sql_query += ', count(`' + field + '`)' + ' as `cnt' + field + '`'

        # see if we need to calculate sum
        if parameters['SUM'] and summary_fields != []:
            for field in summary_fields:
                sql_query += ', sum(`' + field + '`)' + ' as `sum' + field + '`'

        # see if we need to calculate avg
        if parameters['AVERAGE'] and summary_fields != []:
            for field in summary_fields:
                sql_query += ', avg(`' + field + '`)' + ' as `avg' + field + '`'

        # see if we need to calculate min
        if parameters['MINIMUM'] and summary_fields != []:
            for field in summary_fields:
                sql_query += ', min(`' + field + '`)' + ' as `min' + field + '`'

        # see if we need to calculate max
        if parameters['MAXIMUM'] and summary_fields != []:
            for field in summary_fields:
                sql_query += ', max(`' + field + '`)' + ' as `max' + field + '`'

        # check whether to dissolve geometry
        if parameters['DISSOLVEGEOMETRY']:
            sql_query += ', CastToMulti(ST_Union(geometry)) as geometry'

        # specify input--input1 is the layer we pass to the 'INPUT_DATASOURCES' parameter
        sql_query += ' from input1'

        # see if we need to do any grouping
        if group_fields != []:
            sql_query += ' group by `' + group_fields[0] + '`'
            for field in group_fields[1:]:
                sql_query += ', `' + field + '`'

        # report the SQL query to the user before running the Execute SQL tool
        feedback.setProgressText('SQL Query:\n')

        # we need to wrap our query so it doesn't make the dialog box become too wide
        # textwrap.wrap returns a list of lines
        wrapped_query = textwrap.wrap(sql_query, break_long_words=False)

        # iterate through the list, printing each line
        for line in wrapped_query:
            feedback.setProgressText(line)
        feedback.setProgressText('\n')

        # determine the geometry type we want
        # if dissolving geometries, we want the Execute SQL tool to autodetect
        # geometry (option 0); otherwise, we want no geometry (option 1)
        geom_type = 1
        if parameters['DISSOLVEGEOMETRY']:
            geom_type = 0


        # Execute SQL
        executeSQL_result = processing.run('qgis:executesql',
            {'INPUT_DATASOURCES': inputLayer,
            'INPUT_GEOMETRY_CRS': None,
            'INPUT_GEOMETRY_FIELD': '',
            'INPUT_GEOMETRY_TYPE': geom_type,
            'INPUT_QUERY': sql_query,
            'INPUT_UID_FIELD': '',
            'OUTPUT': parameters['OUTPUT']},
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        # return final output as a dictionary
        return {'OUTPUT': executeSQL_result['OUTPUT']}
