# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MetaGAM QMD XML
                                 A QGIS plugin
 Plugin qui permet de gérer les métadonnées uniques à la métropole de Grenoble.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2022-12-20
        git sha              : $Format:%H$
        copyright            : (C) 2022-2025 by Service SIT - Amr HAMADEH, Steven PION-ROUX
        email                : demande_sit@grenoblealpesmetropole.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import shutil
import zipfile
from datetime import datetime
from xml.dom import minidom
import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element, SubElement
import lxml.etree as lxml

from PyQt5.QtCore import QSize
from PyQt5.QtGui import QImage, QPainter, QColor
from qgis.core import (
    QgsProject,
    QgsMapSettings,
    QgsRectangle,
    QgsMapRendererCustomPainterJob,
)

GMD_URL = "http://www.isotc211.org/2005/gmd"
GCO_URL = "http://www.isotc211.org/2005/gco"

# Variables locales pour les chemins des fichiers
current_file_path = os.path.abspath(__file__)
ISO_file_path = os.path.join(os.path.dirname(current_file_path), "resources")
temp_file = os.path.join(os.path.dirname(current_file_path), "temp")


def save_temp_qmd(layer_name):
    """_summary_

    Fonction qui sauvegarde le fichier .qmd depuis qgis dans le répertoire temporaire.

    Args:
        layer_name (string): nom de la couche
    Returns:
        fichier qmd : fichier qmd généré par Qgis contenant les metadata
    """
    layer = QgsProject.instance().mapLayersByName(layer_name)
    qmd_file = temp_file + "/" + layer_name + ".qmd"
    layer[0].saveNamedMetadata(qmd_file)
    return qmd_file


def transform_qmd_to_xml(layer_name):
    """_summary_

    Fonction qui va traduire le fichier .qmd en texte xml (car le qmd n'est pas lisible par geonetwork).

    Args:
        layer_name (str): nom de la couche
    Returns:
        str : génére le code xml qui correspond au fichier qmd
    """
    input_qmd_file = save_temp_qmd(layer_name)
    xslt_file = ISO_file_path + "/qgis-to-iso19139.xsl"
    parser = lxml.XMLParser(remove_blank_text=True)
    in_dom = lxml.parse(input_qmd_file, parser=parser)
    xslt = lxml.parse(xslt_file, parser=parser)
    transform = lxml.XSLT(xslt)
    out_dom = transform(in_dom)
    return lxml.tostring(out_dom)


def add_thumbnail(uuid, layer_name):
    """_summary_

    Fonction qui  créer une image de taille 800x800 pixels de l'étendu
    spatiale des données en question avec un arrière-plan blanc.

    Args:
        uuid (str): code unique identifiant une fiche de metadata
        layer_name (str): nom de la couche
    """
    temp_folder = temp_file
    # créer l'image
    img = QImage(QSize(800, 800), QImage.Format_ARGB32_Premultiplied)
    # définir la couleur de l'arriére plan (Blanc)
    color = QColor(255, 255, 255, 255)
    img.fill(color.rgba())
    # créer le painter
    p = QPainter()
    p.begin(img)
    p.setRenderHint(QPainter.Antialiasing)
    # créer les paramétres de la carte
    ms = QgsMapSettings()
    ms.setBackgroundColor(color)
    # set layers to render
    layer = QgsProject.instance().mapLayersByName(layer_name)
    ms.setLayers([layer[0]])
    # définir l'étendu spatiale
    rect = QgsRectangle(ms.fullExtent())
    rect.scale(1.1)
    ms.setExtent(rect)
    # définir la taille de l'image
    ms.setOutputSize(img.size())
    # setup le rendu de la carte
    render = QgsMapRendererCustomPainterJob(ms, p)
    render.start()
    render.waitForFinished()
    p.end()
    # sauvegarder l'image
    img.save(temp_folder + "/" + uuid + ".png")


def add_sub_element(parent, tag, value=None, attrib=None):
    """_summary_

    Fonction qui définit les parents/enfants dans un fichier xml.

    Args:
        parent (str): l'élément parent auquel l'élément enfant sera ajouté.
        tag (str): la balise XML à utiliser pour créer l'élément enfant.
        value (str, optional):  la valeur de texte à assigner à l'élément enfant. Defaults to None.
        attrib (dict, optional): les attributs à assigner à l'élément enfant. Defaults to None.

    Returns:
        Element: l'élément enfant créé.
    """
    sub = SubElement(parent, tag, attrib=attrib or {})
    if value is not None:
        sub.text = value
    return sub


def create_info_xml(uuid, thumb_filename):
    """_summary_

    Cette fonction créer le fichier info.xml qui contient des infromations nécessaires au fichier zip de metadata.

    Args:
        uuid (str): code unique identifiant une fiche de metadata.
        thumb_filename (str): une chaîne de caractères qui représente le nom du fichier image miniature.

    Returns:
        str: code xml.
    """
    root = Element("info", {"version": "1.1"})
    general = add_sub_element(root, "general")
    d = datetime.now().isoformat()
    add_sub_element(general, "changeDate", d)
    add_sub_element(general, "createDate", d)
    add_sub_element(general, "schema", "iso19139")
    add_sub_element(general, "isTemplate", "n")
    add_sub_element(general, "format", "full")
    add_sub_element(general, "localId")
    add_sub_element(general, "uuid", uuid)
    add_sub_element(general, "siteId", "site_test")
    add_sub_element(general, "siteName", "site_name_test")
    add_sub_element(root, "categories", attrib={"name": "photo"})
    privs = add_sub_element(root, "privileges")
    grp = add_sub_element(privs, "group", attrib={"name": "all"})
    add_sub_element(grp, "operation", attrib={"name": "dynamic"})
    add_sub_element(grp, "operation", attrib={"name": "featured"})
    add_sub_element(grp, "operation", attrib={"name": "view"})
    add_sub_element(grp, "operation", attrib={"name": "download"})
    public = add_sub_element(root, "public")
    add_sub_element(
        public,
        "file",
        attrib={"name": os.path.basename(thumb_filename), "changeDate": d},
    )
    add_sub_element(root, "private")
    xmlstring = ET.tostring(root, encoding="UTF-8", method="xml").decode()
    dom = minidom.parseString(xmlstring)
    return dom.toprettyxml(indent="  ")


def create_zip(
    layer_name, uuid, inspire_keywords, layer_type, layer_denominateur, date_publication
):
    """_summary_

    Cette fonction permet de créer le fichier zip contenant les fichiers xml et
    images sous un format lisible par geonetwork.

    Args:
        layer_name (str): nom de la couche.
        uuid (str): code unique identifiant une fiche de metadata.
        inspire_keywords (str): une liste de chaînes de caractères représentant
                                les mots-clés INSPIRE associés à la couche.
    """
    thumb_filename = temp_file + "/" + uuid + ".png"
    zip_filename = temp_file + "/" + uuid + ".zip"
    add_thumbnail(uuid, layer_name)
    with zipfile.ZipFile(zip_filename, "w") as z:
        meta_xml = transform_qmd_to_xml(layer_name)
        if inspire_keywords:
            meta_xml = insert_inspire_xml(meta_xml, inspire_keywords)
        meta_xml = insert_layer_type(meta_xml, layer_type)
        meta_xml = insert_layer_denominateur(meta_xml, layer_denominateur)
        meta_xml = update_file_identifier(meta_xml, uuid)
        if date_publication is not None:
            meta_xml = insert_date_publication(meta_xml, date_publication)

        # pretty print XML
        dom = minidom.parseString(b'<?xml version="1.0" encoding="UTF-8"?>' + meta_xml)
        meta_xml = dom.toprettyxml(indent="  ")

        z.writestr(
            os.path.join(uuid, os.path.join("metadata", "metadata.xml")), meta_xml
        )
        z.write(
            thumb_filename,
            os.path.join(uuid, "public", os.path.basename(thumb_filename)),
        )
        z.writestr(os.path.join(uuid, os.path.join("private", "")), "")
        info = create_info_xml(uuid, thumb_filename)
        z.writestr(os.path.join(uuid, "info.xml"), info)


def clean_temp():
    """_summary_

    Cette fonction permet de supprimer tous les fichiers et dossiers du répertoire
    temporaire (variable temp_file) sauf les fichiers zip et le fichier cookie.txt.
    """
    for filename in os.listdir(temp_file):
        if filename.endswith(".zip") or filename == "cookie.txt":
            continue
        file_path = os.path.join(temp_file, filename)
        if os.path.isfile(file_path) or os.path.islink(file_path):
            os.unlink(file_path)
        elif os.path.isdir(file_path):
            shutil.rmtree(file_path)
            # except Exception as e:
            #     print('Pas possible de supprimer %s. Raison: %s' % (file_path, e))


def remove_all_zip_files():
    """_summary_

    Cette fonction permet de supprimer tous les fichiers zip du fichier temporaire.
    """
    for filename in os.listdir(temp_file):
        if filename.endswith(".zip"):
            file_path = os.path.join(temp_file, filename)
            os.unlink(file_path)


def insert_inspire_xml(xml_string, keywords):
    """_summary_

    Cette fonction renvoie une chaîne de caractères représentant le document XML
    mis à jour avec les mots-clés INSPIRE ajoutés, formatée avec une indentation
    pour une meilleure lisibilité.

    Args:
        xml_string (str): une chaîne de caractères représentant un document XML contenant des métadonnées.
        keywords (str): une liste de chaînes de caractères représentant les mots-clés
                        INSPIRE à ajouter aux métadonnées.

    Returns:
        str : code xml.
    """
    tree = ET.fromstring(xml_string)
    root = tree

    namespaces = {
        "gmd": GMD_URL,
        "gco": GCO_URL,
    }

    ET.register_namespace("gmd", GMD_URL)
    ET.register_namespace("gco", GCO_URL)
    descriptive_keywords = [
        el
        for el in root.findall(".//gmd:descriptiveKeywords", namespaces)
        if el.find(
            "gmd:MD_Keywords/gmd:thesaurusName/gmd:CI_Citation/"
            "gmd:title[gco:CharacterString='GEMET - INSPIRE themes, version 1.0']",
            namespaces,
        )
        is not None
    ]

    if descriptive_keywords:
        descriptive_keyword = descriptive_keywords[0]
        md_keywords = descriptive_keyword.find("./gmd:MD_Keywords", namespaces)

        for keyword in keywords:
            gmd_keyword = ET.SubElement(md_keywords, "gmd:keyword")
            gco_string = ET.SubElement(gmd_keyword, "gco:CharacterString")
            gco_string.text = keyword

        return ET.tostring(root, encoding="utf-8", method="xml")
    return xml_string


def insert_layer_type(xml_string, layer_type):
    """_summary_

    Cette fonction renvoie une chaîne de caractères représentant le document
    XML mis à jour avec le type de la couche, formatée avec une indentation
    pour une meilleure lisibilité.

    Args:
        xml_string (str): une chaîne de caractères représentant un document XML contenant des métadonnées.
        layer_type (str): chaîne de caractères représentant le type de la couche.

    Returns:
        str : code xml.
    """
    tree = ET.fromstring(xml_string)
    root = tree

    namespaces = {
        "gmd": GMD_URL,
        "gco": GCO_URL,
    }

    ET.register_namespace("gmd", GMD_URL)
    ET.register_namespace("gco", GCO_URL)
    key = "{http://www.isotc211.org/2005/gmd}spatialRepresentationType"
    spatial_representation_types_code = [
        el
        for el in root.findall(".//" + key, namespaces)
        if el.find("./gmd:MD_SpatialRepresentationTypeCode", namespaces).attrib.get(
            "codeList"
        )
        == "http://standards.iso.org/iso/19139/resources/gmxCodelists.xml#MD_SpatialRepresentationTypeCode"
    ]

    if spatial_representation_types_code:
        md_spatial_representation_type_code = spatial_representation_types_code[0].find(
            "./gmd:MD_SpatialRepresentationTypeCode", namespaces
        )
        md_spatial_representation_type_code.attrib["codeListValue"] = layer_type

        return ET.tostring(root, encoding="utf-8", method="xml")
    return xml_string


def insert_layer_denominateur(xml_string, layer_denominateur):
    """_summary_

    Cette fonction renvoie une chaîne de caractères représentant le document XML
    mis à jour avec le layer_denominateur inséré entre les balises appropriées,
    formaté avec une indentation pour une meilleure lisibilité.

    Args:
        xml_string (str): une chaîne de caractères représentant un document XML contenant des métadonnées.
        layer_denominateur (str): chaîne de caractères représentant la valeur à
                                  insérer dans les balises <gco:Integer></gco:Integer>.

    Returns:
        str : code xml.
    """
    tree = ET.fromstring(xml_string)
    root = tree

    namespaces = {
        "gmd": GMD_URL,
        "gco": GCO_URL,
    }

    ET.register_namespace("gmd", GMD_URL)
    ET.register_namespace("gco", GCO_URL)

    key = "{http://www.isotc211.org/2005/gmd}denominator"
    denominators = root.findall(".//" + key, namespaces)

    for denominator in denominators:
        integer_element = denominator.find("./gco:Integer", namespaces)
        if integer_element is not None:
            integer_element.text = str(layer_denominateur)

    return ET.tostring(root, encoding="utf-8", method="xml")


def insert_date_publication(xml_string, date_publication):
    """Insère la valeur date_publication entre les balises <gco:Date></gco:Date> du document XML.

    Args:
        xml_string (str): Une chaîne de caractères représentant un document XML contenant des métadonnées.
        date_publication (str): La valeur à insérer dans les balises <gco:Date></gco:Date>.

    Returns:
        str: Le document XML mis à jour avec date_publication, formaté avec une indentation.
    """
    tree = ET.fromstring(xml_string)
    root = tree

    namespaces = {
        "gmd": GMD_URL,
        "gco": GCO_URL,
    }

    ET.register_namespace("gmd", GMD_URL)
    ET.register_namespace("gco", GCO_URL)

    # key = "{http://www.isotc211.org/2005/gco}Date"
    date_elements = root.findall(
        ".//gmd:date/gmd:CI_Date/gmd:date/gco:Date", namespaces
    )

    for date_element in date_elements:
        date_element.text = str(date_publication)

    return ET.tostring(root, encoding="utf-8", method="xml")


def update_file_identifier(xml_string, meta_id):
    """Cette fonction renvoie une chaîne de caractères représentant le document XML
    mis à jour l'identifiant de la fiche au lieu de celui de la table.

    Args:
        xml_string (str): une chaîne de caractères représentant un document XML contenant des métadonnées.
        meta_id (str): chaîne de caractères représentant l'identifiant de la fiche.

    Returns:
        str : code xml.
    """
    tree = ET.fromstring(xml_string)
    namespaces = {
        "gmd": GMD_URL,
        "gco": GCO_URL,
    }

    file_identifier_element = tree.find(
        ".//gmd:fileIdentifier/gco:CharacterString", namespaces
    )

    if file_identifier_element is not None:
        file_identifier_element.text = meta_id

    file_name_element = tree.find(".//gmd:fileName/gco:CharacterString", namespaces)

    if file_name_element is not None:
        file_name_element.text = meta_id + ".png"

    return ET.tostring(tree, encoding="utf-8", method="xml")
