from qgis.PyQt.QtWidgets import QAction, QInputDialog, QMessageBox
from qgis.core import QgsPointXY, QgsVectorLayer, QgsFeature, QgsGeometry, QgsField, QgsFields, QgsWkbTypes, QgsProject
from qgis.PyQt.QtCore import QVariant
from math import cos, sin, radians

class PluginPuntosEnPoligono:
    def __init__(self, iface):
        self.iface = iface
        self.action = None

    def initGui(self):
        self.action = QAction("Marcos de plantación", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addPluginToMenu("Marcos de plantación", self.action)

    def unload(self):
        self.iface.removePluginMenu("Puntos en polígono", self.action)

    def run(self):
        distancia, ok1 = QInputDialog.getDouble(None, "Distancia entre puntos", "Introduce la distancia (m):", 3, 0.1)
        if not ok1:
            return

        tipo, ok2 = QInputDialog.getItem(None, "Tipo de distribución", "Selecciona el tipo:", ["real", "tresbolillo", "rectangular"], 1, False)
        if not ok2:
            return

        angulo, ok3 = QInputDialog.getDouble(None, "Ángulo de orientación", "Introduce el ángulo (°):", 0, -360, 360)
        if not ok3:
            return

        capa_poligonos = self.iface.activeLayer()
        if not capa_poligonos or capa_poligonos.geometryType() != QgsWkbTypes.PolygonGeometry:
            QMessageBox.critical(None, "Error", "Selecciona una capa de polígonos válida")
            return

        seleccionados = capa_poligonos.selectedFeatures()
        if not seleccionados:
            QMessageBox.critical(None, "Error", "No hay polígonos seleccionados")
            return

        capa_puntos = QgsVectorLayer(f"Point?crs={capa_poligonos.crs().authid()}", f"Puntos_{tipo}", "memory")
        provider = capa_puntos.dataProvider()
        provider.addAttributes([QgsField("ID", QVariant.Int)])
        capa_puntos.updateFields()

        id_punto = 1

        def rotar_punto(x, y, cx, cy, angulo):
            ang_rad = radians(angulo)
            x_rot = cos(ang_rad) * (x - cx) - sin(ang_rad) * (y - cy) + cx
            y_rot = sin(ang_rad) * (x - cx) + cos(ang_rad) * (y - cy) + cy
            return x_rot, y_rot

        def agregar_puntos(x_min, y_min, x_max, y_max, geom, tipo, angulo):
            nonlocal id_punto
            centro = geom.centroid().asPoint()
            y = y_min
            fila_par = True
            while y < y_max:
                x = x_min
                if tipo == "tresbolillo" and not fila_par:
                    x += distancia / 2
                while x < x_max:
                    x_rot, y_rot = rotar_punto(x, y, centro.x(), centro.y(), angulo)
                    punto = QgsPointXY(x_rot, y_rot)
                    if geom.contains(QgsGeometry.fromPointXY(punto)):
                        f = QgsFeature()
                        f.setGeometry(QgsGeometry.fromPointXY(punto))
                        f.setAttributes([id_punto])
                        provider.addFeature(f)
                        id_punto += 1
                    x += distancia
                y += distancia * (3 ** 0.5) / 2 if tipo == "tresbolillo" else distancia
                fila_par = not fila_par

        for pol in seleccionados:
            geom = pol.geometry()
            x_min, y_min, x_max, y_max = geom.boundingBox().toRectF().getCoords()
            agregar_puntos(x_min, y_min, x_max, y_max, geom, tipo, angulo)

        QgsProject.instance().addMapLayer(capa_puntos)
        QMessageBox.information(None, "Listo", f"Puntos generados con patrón {tipo} y orientación {angulo}°")