
# -*- coding: utf-8 -*-
"""
MapLogis QGIS Plugin v1.8.7

Mudanças em relação à 1.8.4 (sem alterar processamento da rota):

1) Campo Token mascarado (asteriscos) na interface.
2) Popup ao final do processamento com resumo da rota:
   - Distância em km
   - Tempo estimado em minutos
3) Novo campo 'ordem_vis' na camada de pontos, com sequência 1,2,3...
   - Usado apenas para rótulo.
   - Campo 'ordem' continua com a ordem original da API.
"""

import csv
import json
import time
import re
import os
import sys
import subprocess
import requests

from qgis.PyQt.QtCore import QVariant, QSettings, Qt, QSize
from qgis.PyQt.QtWidgets import (
    QAction, QFileDialog, QMessageBox,
    QDialog, QVBoxLayout, QHBoxLayout,
    QLabel, QLineEdit, QPushButton,
    QCheckBox, QPlainTextEdit, QFrame,
    QTableWidget, QTableWidgetItem, QDialogButtonBox,
    QApplication, QProgressDialog, QGroupBox
)
from qgis.PyQt.QtGui import QColor, QPixmap, QFont, QPainter, QIcon

def make_icon_with_margin(path, extra=6):
    """Cria um QIcon com margem transparente à direita para aumentar o espaço entre o ícone e o texto."""
    pm = QPixmap(path)
    try:
        from qgis.PyQt.QtCore import Qt
    except Exception:
        from PyQt5.QtCore import Qt  # fallback, embora no QGIS o primeiro deva funcionar
    if pm.isNull():
        return QIcon(path)
    w = pm.width() + int(extra)
    h = pm.height()
    new_pm = QPixmap(w, h)
    new_pm.fill(Qt.transparent)
    painter = QPainter(new_pm)
    painter.drawPixmap(0, 0, pm)
    painter.end()
    return QIcon(new_pm)

from qgis.core import (
QgsVectorLayer, QgsProject, QgsVectorFileWriter, QgsCoordinateTransformContext, QgsFields, QgsField,
    QgsFeature, QgsPointXY, QgsGeometry,
    QgsSymbol, QgsPalLayerSettings, QgsTextFormat,
    QgsVectorLayerSimpleLabeling,
    QgsCategorizedSymbolRenderer, QgsRendererCategory

)
CSV_DELIMITER = ";"
CSV_ENCODING = "latin-1"

GEOCODE_URL        = "https://api.maplogis.com/api/v1/geocoding"
ORDER_POINTS_URL   = "https://api.maplogis.com/api/v1/order-points"
GENERATE_ROUTE_URL = "https://api.maplogis.com/api/v1/generate-route"

IFACE = None


def clean_header(s):
    if not isinstance(s, str):
        return s
    s = s.replace('\ufeff', '')
    return s.strip()


def clean_value(v):
    if not isinstance(v, str):
        return v
    return v.strip()


def clean_string(s):
    if not isinstance(s, str):
        return s
    s = s.replace('\ufeff', '')
    s = re.sub(r'[^\x00-\x7F]+', '', s)
    return s.strip()


def to_float(val):
    if val is None or val == "":
        return None
    return float(str(val).replace(",", "."))


def geocode_row(row, token, debug_index=None, total=None, logger=None):
    secondary_search = int(row.get("secondary_search", 3) or 3)
    result_count = int(row.get("result_count", 1) or 1)
    confidence_precision_raw = row.get("confidence_precision", 0.7) or 0.7

    if isinstance(confidence_precision_raw, str):
        confidence_precision_raw = confidence_precision_raw.replace(",", ".")
    confidence_precision = float(confidence_precision_raw)

    payload = {
        "state":        row.get("state", ""),
        "city":         row.get("city", ""),
        "district":     row.get("district", ""),
        "street":       row.get("street", ""),
        "number":       str(row.get("number", "")),
        "postal_code":  row.get("postal_code", ""),
        "country":      row.get("country", "Brasil"),
        "secondary_search": secondary_search,
        "result_count": result_count,
        "confidence_precision": confidence_precision
    }

    headers = {
        "Accept": "application/json",
        "Content-Type": "application/json",
        "Authorization": f"Token {token}",
    }

    def log(msg):
        print(msg)
        if logger:
            logger(msg)

    try:
        resp = requests.post(GEOCODE_URL, headers=headers, json=payload, timeout=15)
    except Exception as e:
        log(f"⚠ ERRO DE CONEXÃO NA LINHA {debug_index}/{total}: {e}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        return None, f"ERRO_CONEXAO: {e}"

    if resp.status_code != 200:
        log(f"⚠ HTTP {resp.status_code} NA LINHA {debug_index}/{total}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        log("➜ Resposta bruta: " + resp.text[:1000])
        return None, f"HTTP_{resp.status_code}: {resp.text}"

    try:
        data = resp.json()
    except Exception:
        log(f"⚠ ERRO_JSON NA LINHA {debug_index}/{total}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        log("➜ Resposta bruta: " + resp.text[:1000])
        return None, "ERRO_JSON"

    if not data:
        log(f"⚠ SEM_RESULTADO NA LINHA {debug_index}/{total}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        log("➜ Resposta bruta: " + resp.text[:1000])
        return None, "SEM_RESULTADO"

    if isinstance(data, dict):
        data = [data]

    r0 = data[0]
    if "location" not in r0:
        log(f"⚠ SEM_CAMPO_LOCATION NA LINHA {debug_index}/{total}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        log("➜ Resposta bruta: " + json.dumps(data, ensure_ascii=False))
        return None, "SEM_COORD"

    loc = r0["location"]
    lat = loc.get("lat")
    lng = loc.get("lng")
    if lat is None or lng is None:
        log(f"⚠ LOCATION SEM LAT/LNG NA LINHA {debug_index}/{total}")
        log("➜ Payload usado: " + json.dumps(payload, ensure_ascii=False))
        log("➜ Resposta bruta: " + json.dumps(data, ensure_ascii=False))
        return None, "SEM_COORD"

    out = {
        "lat": lat,
        "lng": lng,
        "formatted_address": r0.get("formatted_address", ""),
        "provider": r0.get("confidence", {}).get("provider", ""),
        "confidence_score": r0.get("confidence", {}).get("score", ""),
        "confidence_level": r0.get("confidence", {}).get("level", "")
    }
    return out, "OK"


def generate_route_from_csv(csv_path, token, blocked_areas, logger=None):
    """
    Gera a camada de rota a partir do CSV de pontos ordenados.

    Retorna:
        rota_layer, dist_km, dur_min
    """
    def log(msg):
        print(msg)
        if logger:
            logger(msg)

    headers = {
        "Accept": "application/json",
        "Content-Type": "application/json",
        "Authorization": f"Token {token}",
    }

    log("\n🔍 Lendo CSV de locations para gerar rota...")

    with open(csv_path, encoding=CSV_ENCODING, newline="") as f:
        raw = f.read()

    raw = clean_string(raw)
    rows = list(csv.DictReader(raw.splitlines(), delimiter=CSV_DELIMITER))

    if not rows:
        raise Exception("CSV de rota está vazio.")

    log("➡ Cabeçalhos lidos (rota): " + ", ".join(list(rows[0].keys())))

    cleaned_rows = []
    for row in rows:
        nr = {}
        for k, v in row.items():
            nr[clean_string(k).lower()] = clean_string(v)
        cleaned_rows.append(nr)

    log("➡ Cabeçalhos normalizados (rota): " + ", ".join(cleaned_rows[0].keys()))

    required = ["id", "latitude", "longitude"]
    for col in required:
        if col not in cleaned_rows[0]:
            raise Exception(f"Coluna ausente no CSV de rota: {col}")

    locations = []
    for r in cleaned_rows:
        lat = to_float(r["latitude"])
        lng = to_float(r["longitude"])
        locations.append({
            "id": r["id"],
            "latitude": lat,
            "longitude": lng
        })

    log(f"✔ {len(locations)} locations carregados para generate-route.")

    payload = {
        "locations": locations,
        "costing": "auto",
        "verbose": False,
        "units": "kilometers",
        "language": "pt-BR",
        "narrative": False,
        "shape_format": "geojson",
        "costing_options": {
            "avoid_tolls": 1,
            "auto": [
                "shortest"
            ]
        }
    }
    if blocked_areas:
        payload["blocked_areas"] = blocked_areas

    log("\n📤 Payload enviado para generate-route:")
    log(json.dumps(payload, ensure_ascii=False, indent=2))

    log("\n📡 Enviando requisição para generate-route...")
    resp = requests.post(GENERATE_ROUTE_URL, headers=headers, json=payload)

    log(f"➡ Status generate-route: {resp.status_code}")
    log("➡ Resposta (primeiros 1000 chars):")
    log(resp.text[:1000])

    if resp.status_code != 200:
        msg = f"API generate-route retornou erro HTTP {resp.status_code}"
        try:
            j = resp.json()
            if "detail" in j:
                msg += f": {j['detail']}"
        except Exception:
            pass
        raise Exception(msg)

    data = resp.json()

    dist_km = None
    dur_min = None
    try:
        routes = data.get("routes", [])
        if routes:
            dist = routes[0].get("distance")
            dur = routes[0].get("duration")
            if dist is not None:
                dist_km = float(dist) / 1000.0
            if dur is not None:
                dur_min = float(dur) / 60.0
            if dist_km is not None and dur_min is not None:
                log(f"ℹ Distância total ≈ {dist_km:.2f} km | Duração estimada ≈ {dur_min:.1f} min")
    except Exception as e:
        log(f"⚠ Não foi possível interpretar distância/duração: {e}")

    coords_all = []

    try:
        routes = data.get("routes", [])
        if not routes:
            raise Exception("Resposta de generate-route não contém 'routes'.")

        # ✅ Novo formato (geojson consolidado): routes[0].geometry
        route_geom = routes[0].get("geometry")
        if isinstance(route_geom, dict):
            route_coords = route_geom.get("coordinates")
            # aceita LineString (com ou sem "type") desde que tenha >= 2 pontos
            if isinstance(route_coords, list) and len(route_coords) >= 2:
                coords_all = route_coords

        # ✅ Formato antigo (fallback): legs/steps
        if len(coords_all) < 2:
            for leg in (routes[0].get("legs", []) or []):
                for step in (leg.get("steps", []) or []):
                    geom = step.get("geometry")
                    if not isinstance(geom, dict):
                        continue
                    step_coords = geom.get("coordinates", [])
                    if not step_coords:
                        continue
                    if coords_all and coords_all[-1] == step_coords[0]:
                        coords_all.extend(step_coords[1:])
                    else:
                        coords_all.extend(step_coords)

        if len(coords_all) < 2:
            raise Exception("Não foi possível montar a linha da rota (menos de 2 vértices).")

    except Exception as e:
        raise Exception(f"Erro ao interpretar geometria da rota: {e}")

    log(f"\n✔ Rota montada com {len(coords_all)} vértices.")

    rota_layer = QgsVectorLayer("LineString?crs=EPSG:4326", "maplogis_rota", "memory")
    rota_provider = rota_layer.dataProvider()

    rota_fields = QgsFields()
    rota_fields.append(QgsField("id", QVariant.String))
    rota_fields.append(QgsField("dist_km", QVariant.Double))
    rota_fields.append(QgsField("dur_min", QVariant.Double))
    rota_provider.addAttributes(rota_fields)
    rota_layer.updateFields()

    feat = QgsFeature()
    feat.setFields(rota_fields)
    feat["id"] = "route"
    if dist_km is not None:
        feat["dist_km"] = dist_km
    if dur_min is not None:
        feat["dur_min"] = dur_min
    pts = [QgsPointXY(float(x), float(y)) for x, y in coords_all]
    geom = QgsGeometry.fromPolylineXY(pts)
    feat.setGeometry(geom)
    rota_provider.addFeature(feat)

    rota_layer.updateExtents()

    symbol = QgsSymbol.defaultSymbol(rota_layer.geometryType())
    symbol.setWidth(1.8)
    symbol.setColor(QColor(164, 90, 255))
    rota_layer.renderer().setSymbol(symbol)

    QgsProject.instance().addMapLayer(rota_layer)

    log("🟢 Camada de rota 'maplogis_rota' adicionada ao QGIS.")
    return rota_layer, dist_km, dur_min



def run_maplogis_pipeline(csv_input, csv_output, only_geocode, token, blocked_areas,
                          use_existing_coords, start_id, end_id, logger=None):
    """
    Executa todo o fluxo MapLogis.

    Retorna:
        dict com pelo menos a chave 'mode':
          - {"mode": "geocode"} no modo apenas geocodificação
          - {"mode": "rota", "dist_km": ..., "dur_min": ...} no modo completo
    """
    def log(msg):
        print(msg)
        if logger:
            logger(msg)

    if not token:
        raise Exception("Token da API não informado.")

    headers = {
        "Accept": "application/json",
        "Content-Type": "application/json",
        "Authorization": f"Token {token}",
    }

    # 1) Ler CSV de endereços
    log("🔍 Lendo CSV de endereços...")

    with open(csv_input, encoding=CSV_ENCODING, newline="") as f:
        raw = f.read()

    rows = list(csv.DictReader(raw.splitlines(), delimiter=CSV_DELIMITER))
    if not rows:
        raise Exception("CSV de entrada está vazio.")

    log("➡ Cabeçalhos lidos: " + ", ".join(list(rows[0].keys())))

    cleaned_rows = []
    for row in rows:
        nr = {}
        for k, v in row.items():
            nr[clean_header(k).lower()] = clean_value(v)
        cleaned_rows.append(nr)

    log("➡ Cabeçalhos normalizados: " + ", ".join(cleaned_rows[0].keys()))

    # ----- detectar colunas de coordenadas flexíveis -----
    def detect_coord_columns(sample):
        cols = [c.lower() for c in sample.keys()]
        lat_col = None
        lng_col = None
        for cand in ("lat", "latitude"):
            if cand in cols:
                lat_col = cand
                break
        for cand in ("lng", "long", "longitude"):
            if cand in cols:
                lng_col = cand
                break
        return lat_col, lng_col

    lat_col, lng_col = detect_coord_columns(cleaned_rows[0])
    has_lat_any = lat_col is not None
    has_lng_any = lng_col is not None

    if not use_existing_coords:
        required_geo = ["state", "city", "street", "number"]
        for col in required_geo:
            if col not in cleaned_rows[0]:
                raise Exception(f"Coluna obrigatória para geocodificação ausente no CSV: {col}")
    else:
        if not (has_lat_any and has_lng_any):
            raise Exception("Opção 'usar lat/lng do CSV' marcada, mas colunas 'lat/latitude' e 'lng/long/longitude' não foram encontradas.")

    # garantir id / priority
    tem_id = "id" in cleaned_rows[0]
    tem_priority = "priority" in cleaned_rows[0]
    for idx, r in enumerate(cleaned_rows, start=1):
        if not tem_id:
            r["id"] = f"P{idx}"
        if not tem_priority:
            r["priority"] = "20"

    log("🌐 Iniciando preparação de coordenadas...")

    geocoded = []
    geo_by_id = {}
    total_rows = len(cleaned_rows)

    if use_existing_coords:
        log("ℹ Usando colunas de coordenadas do CSV (sem chamar geocodificação).")
        for i, r in enumerate(cleaned_rows, start=1):
            try:
                lat = float(r.get(lat_col))
                lng = float(r.get(lng_col))
            except Exception:
                log(f"⚠ Linha {i}: coordenadas inválidas em '{lat_col}'/'{lng_col}', ignorando.")
                continue
            r["lat"] = lat
            r["lng"] = lng
            r.setdefault("formatted_address", "")
            r.setdefault("confidence_score", "")
            r.setdefault("confidence_level", "")
            r.setdefault("provider", "")
            r["status_geocod"] = "FROM_CSV"
            geocoded.append(r)
            geo_by_id[r["id"]] = r
            log(f"[{i}/{total_rows}] Coordenada CSV ID={r.get('id')} → OK: {lat}, {lng}")
    else:
        for i, r in enumerate(cleaned_rows, start=1):
            log(f"[{i}/{total_rows}] Geocodificando ID={r.get('id')}...")
            result, status = geocode_row(r, token, debug_index=i, total=total_rows, logger=logger)
            r["status_geocod"] = status

            if result:
                r["lat"] = result["lat"]
                r["lng"] = result["lng"]
                r["formatted_address"] = result["formatted_address"]
                r["confidence_score"] = result["confidence_score"]
                r["confidence_level"] = result["confidence_level"]
                r["provider"] = result["provider"]

                geocoded.append(r)
                geo_by_id[r["id"]] = r
                log(f"  → OK: {result['lat']}, {result['lng']}")
            else:
                r["lat"] = ""
                r["lng"] = ""
                r["formatted_address"] = ""
                r["confidence_score"] = ""
                r["confidence_level"] = ""
                r["provider"] = ""
                log(f"  → FALHA: {status}")

            time.sleep(0.15)

    log(f"✔ Coordenadas prontas: {len(geocoded)} de {len(cleaned_rows)} endereços com coordenadas.")

    if not geocoded:
        raise Exception("Nenhum endereço com coordenadas válidas. Abortando.")

    # ---------- MODO SÓ GEOCODIFICAÇÃO ----------
    if only_geocode:
        log("🟡 Modo: apenas georreferenciar endereços (sem rota).")

        fieldnames = [
            "id","state","city","district","street","number","postal_code",
            "country","lat","long","formatted_address","provider",
            "confidence_score","confidence_level","status_geocod"
        ]

        log(f"💾 Gerando CSV de geocodificação: {csv_output}")
        with open(csv_output, "w", encoding=CSV_ENCODING, newline="") as f_out:
            writer = csv.DictWriter(f_out, fieldnames=fieldnames, delimiter=CSV_DELIMITER)
            writer.writeheader()
            for r in cleaned_rows:
                writer.writerow({
                    "id": r.get("id", ""),
                    "state": r.get("state", ""),
                    "city": r.get("city", ""),
                    "district": r.get("district", ""),
                    "street": r.get("street", ""),
                    "number": r.get("number", ""),
                    "postal_code": r.get("postal_code", ""),
                    "country": r.get("country", ""),
                    "lat": r.get("lat", ""),
                    "long": r.get("lng", ""),
                    "formatted_address": r.get("formatted_address", ""),
                    "provider": r.get("provider", ""),
                    "confidence_score": r.get("confidence_score", ""),
                    "confidence_level": r.get("confidence_level", ""),
                    "status_geocod": r.get("status_geocod", "")
                })

        log("✔ CSV de geocodificação gerado.")

        pontos_layer = QgsVectorLayer("Point?crs=EPSG:4326", "maplogis_pontos", "memory")
        pontos_provider = pontos_layer.dataProvider()

        pontos_fields = QgsFields()
        pontos_fields.append(QgsField("id", QVariant.String))
        pontos_fields.append(QgsField("formatted_address", QVariant.String))
        pontos_fields.append(QgsField("lat", QVariant.Double))
        pontos_fields.append(QgsField("long", QVariant.Double))
        pontos_fields.append(QgsField("confidence", QVariant.Double))
        pontos_provider.addAttributes(pontos_fields)
        pontos_layer.updateFields()

        for r in geocoded:
            lat = float(r.get("lat", 0))
            lng = float(r.get("lng", 0))
            feat = QgsFeature()
            feat.setFields(pontos_fields)
            feat["id"] = r.get("id", "")
            feat["formatted_address"] = r.get("formatted_address", "")
            try:
                feat["lat"] = float(lat)
                feat["long"] = float(lng)
                feat["confidence"] = float(r.get("confidence_score") or 0)
            except Exception:
                feat["confidence"] = 0
            feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(lng, lat)))
            pontos_provider.addFeature(feat)

        pontos_layer.updateExtents()

        sym = QgsSymbol.defaultSymbol(pontos_layer.geometryType())
        sym.setSize(2.0)
        sym.setColor(QColor(0, 120, 255))
        pontos_layer.renderer().setSymbol(sym)

        QgsProject.instance().addMapLayer(pontos_layer)
        log(f"🟢 Camada de pontos 'maplogis_pontos' adicionada com {pontos_layer.featureCount()} feições.")
        log("🎉 Processo de geocodificação concluído.")
        return {"mode": "geocode"}

    # ---------- MODO COMPLETO (ROTA) ----------
    log("🟢 Modo: geocodificação + ordenação + rota.")

    has_route_start = "route_start" in geocoded[0]
    has_route_end = "route_end" in geocoded[0]

    def is_marked(v):
        if v is None:
            return False
        v = str(v).strip().lower()
        return v in ("1", "true", "sim", "s", "start", "inicio", "end", "fim")

    start_row = None
    end_row = None

    if start_id:
        for r in geocoded:
            if str(r.get("id")) == start_id:
                start_row = r
                log(f"ℹ Início da rota definido pelo ID informado (ID={start_id}).")
                break
        if not start_row:
            raise Exception(f"ID de início informado ('{start_id}') não encontrado no CSV.")

    if end_id:
        for r in geocoded:
            if str(r.get("id")) == end_id:
                end_row = r
                log(f"ℹ Fim da rota definido pelo ID informado (ID={end_id}).")
                break
        if not end_row:
            raise Exception(f"ID de fim informado ('{end_id}') não encontrado no CSV.")

    if not start_row:
        if has_route_start:
            for r in geocoded:
                if is_marked(r.get("route_start")):
                    start_row = r
                    log(f"ℹ Início da rota definido pela coluna route_start (ID={start_row.get('id')}).")
                    break
        if not start_row:
            start_row = geocoded[0]
            log("ℹ route_start não definido ou não marcado: usando o primeiro endereço como início da rota.")

    if not end_row:
        if has_route_end:
            for r in geocoded:
                if is_marked(r.get("route_end")):
                    end_row = r
                    log(f"ℹ Fim da rota definido pela coluna route_end (ID={end_row.get('id')}).")
                    break
        if not end_row:
            end_row = geocoded[-1]
            log("ℹ route_end não definido ou não marcado: usando o último endereço como fim da rota.")

    start_lat = float(start_row["lat"])
    start_lng = float(start_row["lng"])
    end_lat = float(end_row["lat"])
    end_lng = float(end_row["lng"])

    log(f"➡ Coordenadas início:  lat={start_lat}, lng={start_lng}")
    log(f"➡ Coordenadas fim   :  lat={end_lat}, lng={end_lng}")

    log("📍 Preparando dados para ordenação (order-points)...")

    locations_for_order = []
    for g in geocoded:
        locations_for_order.append({
            "id": g["id"],
            "latitude": float(g["lat"]),
            "longitude": float(g["lng"]),
            "priority": str(g["priority"])
        })

    order_payload = {
        "locations": locations_for_order,
        "start_point": {
            "id": "start",
            "latitude": start_lat,
            "longitude": start_lng
        },
        "end_point": {
            "id": "end",
            "latitude": end_lat,
            "longitude": end_lng
        }
    }

    log("📤 Payload para /order-points:")
    log(json.dumps(order_payload, ensure_ascii=False))

    log("📡 Chamando /order-points...")
    resp_order = requests.post(ORDER_POINTS_URL, headers=headers, json=order_payload)
    log(f"➡ Status: {resp_order.status_code}")
    log("➡ Resposta: " + resp_order.text[:1000])

    if resp_order.status_code != 200:
        msg = f"Erro em /order-points (HTTP {resp_order.status_code})"
        try:
            j = resp_order.json()
            if "detail" in j:
                msg += f": {j['detail']}"
        except Exception:
            pass
        raise Exception(msg)

    data_order = resp_order.json()
    ordered_points = data_order.get("ordered_points")
    if not ordered_points:
        log("⚠ Resposta de /order-points não contém 'ordered_points'. Usando ordem original do CSV para a rota.")
        ordered_points = []
        # ponto inicial
        ordered_points.append({
            "id": "start",
            "latitude": start_lat,
            "longitude": start_lng,
            "priority": 0
        })
        # pontos na ordem original geocodificada
        for g in geocoded:
            ordered_points.append({
                "id": g["id"],
                "latitude": float(g["lat"]),
                "longitude": float(g["lng"]),
                "priority": str(g["priority"])
            })
        # ponto final
        ordered_points.append({
            "id": "end",
            "latitude": end_lat,
            "longitude": end_lng,
            "priority": 9999
        })

    if ordered_points and ordered_points[0].get("id") == "start" and ordered_points[-1].get("id") == "start":
        ordered_points = ordered_points[:-1]
        log("ℹ Removido ponto 'start' final para evitar rota circular.")

    log(f"✔ {len(ordered_points)} pontos (após ajuste) na ordem retornada.")

    log(f"💾 Gerando CSV de rota (que será usado no generate-route): {csv_output}")
    fieldnames = [
        "ordem",
        "id",
        "formatted_address",
        "latitude",
        "longitude",
        "priority",
        "state",
        "city",
        "district",
        "street",
        "number",
        "postal_code",
        "country"
    ]

    with open(csv_output, "w", encoding=CSV_ENCODING, newline="") as f_out:
        writer = csv.DictWriter(f_out, fieldnames=fieldnames, delimiter=CSV_DELIMITER)
        writer.writeheader()

        for idx, p in enumerate(ordered_points, start=1):
            pid = str(p.get("id", ""))
            lat = p.get("latitude", "")
            lng = p.get("longitude", "")
            prio = p.get("priority", "")

            ginfo = geo_by_id.get(pid, {})

            if pid == "start":
                formatted = "Ponto inicial"
            elif pid == "end":
                formatted = "Ponto final"
            else:
                formatted = ginfo.get("formatted_address", "")

            row_out = {
                "ordem": idx,
                "id": pid,
                "formatted_address": formatted,
                "latitude": lat,
                "longitude": lng,
                "priority": prio,
                "state": ginfo.get("state", ""),
                "city": ginfo.get("city", ""),
                "district": ginfo.get("district", ""),
                "street": ginfo.get("street", ""),
                "number": ginfo.get("number", ""),
                "postal_code": ginfo.get("postal_code", ""),
                "country": ginfo.get("country", "")
            }
            writer.writerow(row_out)

    log("✔ CSV de rota gerado (rota_pontos).")

    # camada de pontos ordenados
    pontos_layer = QgsVectorLayer("Point?crs=EPSG:4326", "maplogis_rota_pontos (ordem)", "memory")
    pontos_provider = pontos_layer.dataProvider()

    pontos_fields = QgsFields()
    pontos_fields.append(QgsField("ordem", QVariant.Int))
    pontos_fields.append(QgsField("ordem_vis", QVariant.Int))
    pontos_fields.append(QgsField("id", QVariant.String))
    pontos_fields.append(QgsField("formatted_address", QVariant.String))
    pontos_fields.append(QgsField("priority", QVariant.Double))
    pontos_fields.append(QgsField("lat", QVariant.Double))
    pontos_fields.append(QgsField("lng", QVariant.Double))
    pontos_fields.append(QgsField("tipo", QVariant.String))
    pontos_provider.addAttributes(pontos_fields)
    pontos_layer.updateFields()

    def same_coord(a_lat, a_lng, b_lat, b_lng, tol=1e-6):
        return (abs(a_lat - b_lat) <= tol) and (abs(a_lng - b_lng) <= tol)

    ordem_vis_counter = 0

    for idx, p in enumerate(ordered_points, start=1):
        pid = str(p.get("id", ""))
        lat = float(p.get("latitude", 0))
        lng = float(p.get("longitude", 0))
        prio = float(p.get("priority", 0))

        if pid not in ("start", "end"):
            if same_coord(lat, lng, start_lat, start_lng) or same_coord(lat, lng, end_lat, end_lng):
                log(f"ℹ Ponto '{pid}' tem mesmas coordenadas de início/fim e foi omitido na camada de paradas.")
                continue

        ginfo = geo_by_id.get(pid, {})
        if pid == "start":
            formatted = "Ponto inicial"
            tipo = "start"
        elif pid == "end":
            formatted = "Ponto final"
            tipo = "end"
        else:
            formatted = ginfo.get("formatted_address", "")
            tipo = "stop"

        ordem_vis_counter += 1

        feat = QgsFeature()
        feat.setFields(pontos_fields)
        feat["ordem"] = idx
        feat["ordem_vis"] = ordem_vis_counter
        feat["id"] = pid
        feat["formatted_address"] = formatted
        feat["priority"] = prio
        feat["lat"] = lat
        feat["lng"] = lng
        feat["tipo"] = tipo
        feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(lng, lat)))
        pontos_provider.addFeature(feat)

    pontos_layer.updateExtents()

    categories = []

    start_symbol = QgsSymbol.defaultSymbol(pontos_layer.geometryType())
    start_symbol.setColor(QColor(0, 180, 0))
    start_symbol.setSize(2.0)
    categories.append(QgsRendererCategory("start", start_symbol, "Início"))

    end_symbol = QgsSymbol.defaultSymbol(pontos_layer.geometryType())
    end_symbol.setColor(QColor(220, 0, 0))
    end_symbol.setSize(2.0)
    categories.append(QgsRendererCategory("end", end_symbol, "Fim"))

    stop_symbol = QgsSymbol.defaultSymbol(pontos_layer.geometryType())
    stop_symbol.setColor(QColor(0, 120, 255))
    stop_symbol.setSize(2.0)
    categories.append(QgsRendererCategory("stop", stop_symbol, "Paradas"))

    renderer = QgsCategorizedSymbolRenderer("tipo", categories)
    pontos_layer.setRenderer(renderer)

    pal = QgsPalLayerSettings()
    pal.fieldName = "ordem_vis"
    pal.enabled = True
    text_format = QgsTextFormat()
    text_format.setSize(11)
    text_format.setColor(QColor(40, 40, 40))
    font = QFont()
    font.setBold(True)
    text_format.setFont(font)
    pal.setFormat(text_format)
    labeling = QgsVectorLayerSimpleLabeling(pal)
    pontos_layer.setLabelsEnabled(True)
    pontos_layer.setLabeling(labeling)

    QgsProject.instance().addMapLayer(pontos_layer)
    log(f"🟢 Camada de pontos 'maplogis_rota_pontos (ordem)' adicionada com {pontos_layer.featureCount()} feições.")

    rota_layer, dist_km, dur_min = generate_route_from_csv(csv_output, token, blocked_areas, logger=logger)

    if IFACE and rota_layer:
        canvas = IFACE.mapCanvas()
        canvas.setExtent(rota_layer.extent())
        canvas.refresh()
        log("🔎 Zoom aplicado para a camada de rota.")

    log("🎉 Pipeline completo concluído: geocodificação + ordenação + rota + CSV de rota.")
    return {"mode": "rota", "dist_km": dist_km, "dur_min": dur_min}


class MapLogisDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("MapLogis – Geocodificação e Rotas")
        self.setMinimumWidth(580)

        self.log_buffer = []

        self._plugin_dir = os.path.dirname(__file__)

        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(8, 8, 8, 8)
        self.setLayout(main_layout)

        header = QFrame()
        header_layout = QHBoxLayout()
        header_layout.setContentsMargins(12, 8, 12, 8)
        header.setLayout(header_layout)
        header.setStyleSheet("""
            QFrame {
                background-color: #f2ecff;
                border-radius: 6px;
            }
        """)
        logo_label = QLabel()
        try:
            icon_path = os.path.join(os.path.dirname(__file__), "icone.png")
            if os.path.exists(icon_path):
                pix = QPixmap(icon_path)
                if not pix.isNull():
                    logo_label.setPixmap(pix.scaled(40, 40, Qt.KeepAspectRatio, Qt.SmoothTransformation))
        except Exception:
            pass
        title_label = QLabel("<b>MapLogis</b><br/>Geocodificação, ordenação de pontos e rotas")
        title_label.setStyleSheet("color: #5b2d91; font-size: 13px;")
        header_layout.addWidget(logo_label)
        header_layout.addSpacing(8)
        header_layout.addWidget(title_label)
        header_layout.addStretch()
        main_layout.addWidget(header)

        body = QVBoxLayout()
        body.setContentsMargins(4, 8, 4, 4)
        main_layout.addLayout(body)


        # Grupo: Autenticação
        auth_group = QGroupBox("Autenticação")
        auth_layout = QVBoxLayout()
        auth_group.setLayout(auth_layout)

        token_layout = QHBoxLayout()
        token_label = QLabel("Token:")
        self.token_edit = QLineEdit()
        self.token_edit.setEchoMode(QLineEdit.Password)  # mascarar token
        token_test_btn = QPushButton("Testar token")
        token_test_btn.setToolTip("Envia uma requisição simples para validar o token na API MapLogis.")
        token_test_btn.clicked.connect(self.on_test_token_clicked)
        token_layout.addWidget(token_label)
        token_layout.addWidget(self.token_edit)
        token_layout.addWidget(token_test_btn)
        auth_layout.addLayout(token_layout)

        blocked_layout = QHBoxLayout()
        blocked_label = QLabel("Código de bloqueio (blocked_areas):")
        self.blocked_edit = QLineEdit()
        blocked_layout.addWidget(blocked_label)
        blocked_layout.addWidget(self.blocked_edit)
        auth_layout.addLayout(blocked_layout)

        body.addWidget(auth_group)

        # Grupo: Configuração da rota
        route_group = QGroupBox("Configuração da rota")
        route_layout = QVBoxLayout()
        route_group.setLayout(route_layout)

        se_layout = QHBoxLayout()
        start_label = QLabel("ID início (opcional):")
        self.start_id_edit = QLineEdit()
        end_label = QLabel("ID fim (opcional):")
        self.end_id_edit = QLineEdit()
        se_layout.addWidget(start_label)
        se_layout.addWidget(self.start_id_edit)
        se_layout.addWidget(end_label)
        se_layout.addWidget(self.end_id_edit)
        route_layout.addLayout(se_layout)

        self.only_geocode_cb = QCheckBox("Apenas georreferenciar endereços (não gerar rota)")
        route_layout.addWidget(self.only_geocode_cb)

        self.use_existing_coords_cb = QCheckBox("Usar colunas lat/lng do CSV (pular geocodificação)")
        self.use_existing_coords_cb.setToolTip("Marque se o CSV já possui colunas 'lat' e 'lng' com coordenadas em WGS84.")
        route_layout.addWidget(self.use_existing_coords_cb)

        body.addWidget(route_group)

        # Grupo: Arquivos
        files_group = QGroupBox("Arquivos")
        files_layout = QVBoxLayout()
        files_group.setLayout(files_layout)

        in_layout = QHBoxLayout()
        in_label = QLabel("CSV de entrada:")
        self.in_edit = QLineEdit()
        in_btn = QPushButton("...")
        in_btn.clicked.connect(self.browse_in)
        in_layout.addWidget(in_label)
        in_layout.addWidget(self.in_edit)
        in_layout.addWidget(in_btn)
        files_layout.addLayout(in_layout)

        out_layout = QHBoxLayout()
        out_label = QLabel("CSV de saída:")
        self.out_edit = QLineEdit()
        out_btn = QPushButton("...")
        out_btn.clicked.connect(self.browse_out)
        out_layout.addWidget(out_label)
        out_layout.addWidget(self.out_edit)
        out_layout.addWidget(out_btn)
        files_layout.addLayout(out_layout)

        template_layout = QHBoxLayout()
        self.template_btn = QPushButton("Baixar modelo de CSV")
        self.template_btn.clicked.connect(self.save_template_csv)
        template_layout.addStretch()
        template_layout.addWidget(self.template_btn)
        files_layout.addLayout(template_layout)

        body.addWidget(files_group)

        log_label = QLabel("Log:")
        self.log_text = QPlainTextEdit()
        self.log_text.setReadOnly(True)
        body.addWidget(log_label)
        body.addWidget(self.log_text)

        footer = QHBoxLayout()
        contact_label = QLabel("Site: <a href=\"https://www.maplogis.com\">www.maplogis.com</a>  |  Suporte: <a href=\"mailto:support@maplogis.com\">support@maplogis.com</a>")
        contact_label.setOpenExternalLinks(True)
        contact_label.setStyleSheet("color: #5b2d91; font-size: 10px;")
        footer.addStretch()
        footer.addWidget(contact_label)
        body.addLayout(footer)

        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        self.run_btn = QPushButton("Processar")
        self.run_btn.setIcon(make_icon_with_margin(os.path.join(self._plugin_dir, "automation_branco.png")))
        self.run_btn.setStyleSheet("QPushButton { background-color: #5b2d91; color: white; padding:4px 8px; border-radius:6px; } QPushButton:hover { background-color: #7542b5; }")
        self.qr_btn = QPushButton("QR Code (mobile)")
        self.qr_btn.setIcon(make_icon_with_margin(os.path.join(self._plugin_dir, "qr-code.png")))
        self.qr_btn.setToolTip("Gera um QR Code com link da rota para abrir no celular.")
        self.qr_btn.setEnabled(False)
        self.whatsapp_btn = QPushButton("WhatsApp")
        self.whatsapp_btn.setIcon(make_icon_with_margin(os.path.join(self._plugin_dir, "whatsapp.png")))
        self.whatsapp_btn.setToolTip("Compartilha a rota via WhatsApp usando o link do Google Maps.")
        self.whatsapp_btn.setEnabled(False)

        self.kml_btn = QPushButton("KML")
        self.kml_btn.setToolTip("Salva a rota final como arquivo KML no seu computador.")
        self.kml_btn.setEnabled(False)
        close_btn = QPushButton("Fechar")
        close_btn.setIcon(make_icon_with_margin(os.path.join(self._plugin_dir, "close.png")))
        btn_layout.addWidget(self.run_btn)
        btn_layout.addWidget(self.qr_btn)
        btn_layout.addWidget(self.whatsapp_btn)
        btn_layout.addWidget(self.kml_btn)
        btn_layout.addWidget(close_btn)
        close_btn.clicked.connect(self.reject)
        body.addLayout(btn_layout)

        self.setStyleSheet("""
            QDialog { background-color: #ffffff; }
            QLineEdit, QPlainTextEdit { background-color: #fbfbff; }
        """)

        self.token_edit.setToolTip("Token de autenticação da API MapLogis.")
        self.blocked_edit.setToolTip("Código de polígonos de áreas bloqueadas (pode ficar em branco).")
        self.start_id_edit.setToolTip("ID do ponto inicial da rota (igual ao campo 'id' do CSV). Opcional.")
        self.end_id_edit.setToolTip("ID do ponto final da rota (igual ao campo 'id' do CSV). Opcional.")
        self.only_geocode_cb.setToolTip("Se marcado, apenas geocodifica e gera pontos + CSV, sem ordenar nem calcular rota.")

        self.load_settings()
        self.last_route_csv = None
        self.qr_btn.clicked.connect(self.on_qr_clicked)
        self.whatsapp_btn.clicked.connect(self.on_whatsapp_clicked)
        self.kml_btn.clicked.connect(self.on_kml_clicked)
        self.run_btn.clicked.connect(self.on_run_clicked)


    def on_test_token_clicked(self):
        """Teste rápido de token chamando /geocoding com um endereço fictício simples."""
        token = self.token_edit.text().strip()
        if not token:
            QMessageBox.warning(self, "MapLogis", "Informe um token antes de testar.")
            return

        payload = {
            "state": "SP",
            "city": "Campinas",
            "district": "",
            "street": "Rua Teste",
            "number": "1",
            "postal_code": "",
            "country": "Brasil",
            "secondary_search": 0,
            "result_count": 1,
            "confidence_precision": 0.1,
        }
        headers = {
            "Accept": "application/json",
            "Content-Type": "application/json",
            "Authorization": f"Token {token}",
        }

        try:
            self.log("🔎 Testando token na API MapLogis...")
            resp = requests.post(GEOCODE_URL, headers=headers, json=payload, timeout=10)
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao testar o token:\n{e}")
            self.log(f"ERRO ao testar token: {e}")
            return

        status = resp.status_code

        # 200/201/204/422  => consideramos token VALIDADO
        if status in (200, 201, 204, 422):
            QMessageBox.information(
                self,
                "MapLogis",
                "✅ Token VALIDADO com sucesso."
            )
            self.log(f"✔ Token validado (HTTP {status}).")
            return

        # 401/403 => claramente NÃO VALIDADO
        if status in (401, 403):
            QMessageBox.warning(
                self,
                "MapLogis",
                "❌ Token NÃO VALIDADO. Verifique o valor informado."
            )
            self.log(f"⚠ Token não validado (HTTP {status}).")
            return

        # Qualquer outro erro
        QMessageBox.warning(
            self,
            "MapLogis",
            f"❌ Token NÃO VALIDADO. Código HTTP {status}."
        )
        self.log(f"⚠ Validação de token retornou HTTP {status}: {resp.text[:200]}")


    def on_qr_clicked(self):
        """Gera e exibe um QR Code com link da rota para abrir no celular."""
        if not self.last_route_csv:
            QMessageBox.warning(self, "MapLogis", "Nenhuma rota foi gerada ainda.")
            return

        import csv as _csv
        import urllib.parse, urllib.request

        csv_path = self.last_route_csv
        if not os.path.exists(csv_path):
            QMessageBox.warning(self, "MapLogis", "Arquivo de rota não encontrado.")
            return

        try:
            with open(csv_path, encoding=CSV_ENCODING, newline="") as f:
                rows = list(_csv.DictReader(f, delimiter=CSV_DELIMITER))
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao ler CSV de rota:\n{e}")
            return

        if not rows:
            QMessageBox.warning(self, "MapLogis", "CSV de rota está vazio.")
            return

        # ordenar por 'ordem'
        try:
            rows.sort(key=lambda r: int(r.get("ordem", 0)))
        except Exception:
            pass

        coords = []
        for r in rows:
            try:
                lat = float(str(r.get("latitude", "")).replace(",", "."))
                lng = float(str(r.get("longitude", "")).replace(",", "."))
            except Exception:
                continue
            pid = str(r.get("id", "")).strip().lower()
            coords.append((pid, lat, lng))

        if len(coords) < 2:
            QMessageBox.warning(self, "MapLogis", "Não há pontos suficientes para gerar rota móvel.")
            return

        # origem = primeiro ponto real
        origin = None
        destination = None
        waypoints = []

        for idx, (pid, lat, lng) in enumerate(coords):
            if idx == 0:
                origin = (lat, lng)
            elif idx == len(coords) - 1:
                destination = (lat, lng)
            else:
                if pid not in ("start", "end"):
                    waypoints.append((lat, lng))

        if not destination:
            destination = (coords[-1][1], coords[-1][2])

        # Monta URL do Google Maps Directions
        origin_str = f"{origin[0]:.6f},{origin[1]:.6f}"
        dest_str = f"{destination[0]:.6f},{destination[1]:.6f}"
        wps = "|".join([f"{lat:.6f},{lng:.6f}" for (lat, lng) in waypoints])
        base_url = "https://www.google.com/maps/dir/?api=1"
        params = {
            "origin": origin_str,
            "destination": dest_str,
            "travelmode": "driving",
        }
        if wps:
            params["waypoints"] = wps

        query = urllib.parse.urlencode(params, safe="|,")
        route_url = base_url + "&" + query

        # Gera URL do QR code usando api.qrserver.com (mais simples e estável)
        data_param = urllib.parse.quote(route_url, safe="")
        qr_url = (
            "https://api.qrserver.com/v1/create-qr-code/"
            f"?size=300x300&data={data_param}"
        )

        # Baixa imagem do QR
        try:
            resp = urllib.request.urlopen(qr_url, timeout=15)
            img_data = resp.read()
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao gerar QR Code:\n{e}")
            return



        qr_pix = QPixmap()
        if not qr_pix.loadFromData(img_data):
            QMessageBox.critical(self, "MapLogis", "Não foi possível carregar a imagem do QR Code.")
            return

        # Carrega logo e desenha no centro
        logo_path = os.path.join(os.path.dirname(__file__), "maplogis_logo.png")
        logo_pix = QPixmap()
        if os.path.exists(logo_path):
            logo_pix.load(logo_path)

        if not logo_pix.isNull():
            # redimensiona logo para ~30% do tamanho do QR
            qr_w = qr_pix.width()
            qr_h = qr_pix.height()
            logo_size = int(min(qr_w, qr_h) * 0.30)
            logo_pix = logo_pix.scaled(logo_size, logo_size, Qt.KeepAspectRatio, Qt.SmoothTransformation)

            composed = QPixmap(qr_pix.size())
            composed.fill(Qt.transparent)
            painter = QPainter(composed)
            painter.drawPixmap(0, 0, qr_pix)
            x = (qr_w - logo_pix.width()) // 2
            y = (qr_h - logo_pix.height()) // 2
            painter.drawPixmap(x, y, logo_pix)
            painter.end()
            qr_pix = composed

        # Dialog para exibir o QR
        dlg = QDialog(self)
        dlg.setWindowTitle("QR Code da rota (MapLogis)")
        v = QVBoxLayout(dlg)

        lbl = QLabel()
        lbl.setPixmap(qr_pix)
        lbl.setAlignment(Qt.AlignCenter)
        v.addWidget(lbl)

        url_label = QLabel("Link da rota (para copiar/compartilhar):")
        v.addWidget(url_label)

        url_edit = QLineEdit()
        url_edit.setText(route_url)
        url_edit.setReadOnly(True)
        v.addWidget(url_edit)

        btns = QDialogButtonBox(QDialogButtonBox.Close)
        # Botão extra para salvar PNG
        save_btn = QPushButton("Salvar QR como PNG")
        btns.addButton(save_btn, QDialogButtonBox.ActionRole)
        btns.rejected.connect(dlg.reject)
        v.addWidget(btns)

        def save_png():
            path, _ = QFileDialog.getSaveFileName(
                self,
                "Salvar QR Code",
                "",
                "Imagem PNG (*.png)"
            )
            if not path:
                return
            if not path.lower().endswith(".png"):
                path += ".png"
            if qr_pix.save(path, "PNG"):
                QMessageBox.information(self, "MapLogis", f"QR Code salvo em:\n{path}")
            else:
                QMessageBox.warning(self, "MapLogis", "Não foi possível salvar o arquivo PNG.")

        save_btn.clicked.connect(save_png)

        dlg.exec_()

    def on_whatsapp_clicked(self):
        """Compartilha a rota via WhatsApp usando o link da rota do Google Maps."""
        if not self.last_route_csv:
            QMessageBox.warning(self, "MapLogis", "Nenhuma rota foi gerada ainda.")
            return

        import csv as _csv
        import urllib.parse, urllib.request
        import webbrowser

        csv_path = self.last_route_csv
        if not os.path.exists(csv_path):
            QMessageBox.warning(self, "MapLogis", "Arquivo de rota não encontrado.")
            return

        try:
            with open(csv_path, encoding=CSV_ENCODING, newline="") as f:
                rows = list(_csv.DictReader(f, delimiter=CSV_DELIMITER))
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao ler CSV de rota:\n{e}")
            return

        if not rows:
            QMessageBox.warning(self, "MapLogis", "CSV de rota está vazio.")
            return

        # ordenar por 'ordem'
        try:
            rows.sort(key=lambda r: int(r.get("ordem", 0)))
        except Exception:
            pass

        coords = []
        for r in rows:
            try:
                lat = float(str(r.get("latitude", "")).replace(",", "."))
                lng = float(str(r.get("longitude", "")).replace(",", "."))
            except Exception:
                continue
            pid = str(r.get("id", "")).strip().lower()
            coords.append((pid, lat, lng))

        if len(coords) < 2:
            QMessageBox.warning(self, "MapLogis", "Não há pontos suficientes para gerar rota móvel.")
            return

        # origem = primeiro ponto real
        origin = None
        destination = None
        waypoints = []

        for idx, (pid, lat, lng) in enumerate(coords):
            if idx == 0:
                origin = (lat, lng)
            elif idx == len(coords) - 1:
                destination = (lat, lng)
            else:
                if pid not in ("start", "end"):
                    waypoints.append((lat, lng))

        if not destination:
            destination = (coords[-1][1], coords[-1][2])

        # Monta URL do Google Maps Directions
        origin_str = f"{origin[0]:.6f},{origin[1]:.6f}"
        dest_str = f"{destination[0]:.6f},{destination[1]:.6f}"
        wps = "|".join([f"{lat:.6f},{lng:.6f}" for (lat, lng) in waypoints])
        base_url = "https://www.google.com/maps/dir/?api=1"
        params = {
            "origin": origin_str,
            "destination": dest_str,
            "travelmode": "driving",
        }
        if wps:
            params["waypoints"] = wps

        query = urllib.parse.urlencode(params, safe="|,")
        route_url = base_url + "&" + query

        # Abre WhatsApp (Web ou app) com a mensagem contendo o link da rota
        msg = f"Rota MapLogis: {route_url}"
        wa_url = "https://api.whatsapp.com/send?text=" + urllib.parse.quote(msg, safe="")
        try:
            webbrowser.open(wa_url)
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao abrir WhatsApp:\n{e}")


    def on_kml_clicked(self):
        """Salva a rota final (camada 'maplogis_rota') como arquivo KML."""
        layers = QgsProject.instance().mapLayersByName("maplogis_rota")
        if not layers:
            QMessageBox.warning(self, "MapLogis", "Camada 'maplogis_rota' não encontrada. Gere uma rota primeiro.")
            return

        rota_layer = layers[0]

        path, _ = QFileDialog.getSaveFileName(
            self,
            "Salvar rota como KML",
            "",
            "KML (*.kml)"
        )
        if not path:
            return
        if not path.lower().endswith(".kml"):
            path += ".kml"

        try:
            opts = QgsVectorFileWriter.SaveVectorOptions()
            opts.driverName = "KML"
            opts.fileEncoding = "UTF-8"
            # sobrescreve se existir
            if hasattr(QgsVectorFileWriter, "CreateOrOverwriteFile"):
                opts.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile

            res, err = QgsVectorFileWriter.writeAsVectorFormatV2(
                rota_layer,
                path,
                QgsCoordinateTransformContext(),
                opts
            )
            if res != QgsVectorFileWriter.NoError:
                raise Exception(err or f"Erro ao exportar (código {res}).")

            QMessageBox.information(self, "MapLogis", f"KML salvo com sucesso:\n{path}")
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Falha ao salvar KML: {e}")

    def preview_csv_dialog(self, csv_path, only_geocode, use_existing_coords):
        """Mostra um popup com pré-visualização do CSV e validação básica de colunas."""
        import csv as _csv
        if not os.path.exists(csv_path):
            QMessageBox.warning(self, "MapLogis", "CSV de entrada não encontrado.")
            return False
        try:
            with open(csv_path, encoding="latin-1", newline="") as f:
                rows = list(_csv.reader(f, delimiter=';'))
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao ler CSV:\n{e}")
            return False

        if not rows:
            QMessageBox.warning(self, "MapLogis", "CSV de entrada está vazio.")
            return False

        header = [h.strip() for h in rows[0]]
        data_rows = rows[1:13]  # até 12 linhas para visualização
        header_l = [h.lower() for h in header]

        missing = []

        if use_existing_coords:
            has_lat = any(h in header_l for h in ("lat", "latitude"))
            has_lng = any(h in header_l for h in ("lng", "long", "longitude"))
            if not has_lat or not has_lng:
                missing.append("Colunas de coordenadas (lat/latitude, lng/long/longitude)")
        else:
            for col in ("state", "city", "street", "number"):
                if col not in header_l:
                    missing.append(f"Coluna '{col}' para geocodificação")

        dlg = QDialog(self)
        dlg.setWindowTitle("Pré-visualização do CSV - MapLogis")
        dlg.resize(700, 400)
        layout = QVBoxLayout()
        dlg.setLayout(layout)

        info_label = QLabel(f"Arquivo: <b>{os.path.basename(csv_path)}</b>")
        layout.addWidget(info_label)

        table = QTableWidget(len(data_rows), len(header))
        table.setHorizontalHeaderLabels(header)
        for i, row in enumerate(data_rows):
            for j, val in enumerate(row):
                item = QTableWidgetItem(val)
                table.setItem(i, j, item)
        table.resizeColumnsToContents()
        layout.addWidget(table)

        if missing:
            msg = "Foram detectados possíveis problemas nas colunas:<br/>- " + "<br/>- ".join(missing)
            warn_label = QLabel(msg)
            warn_label.setStyleSheet("color: #b00020;")
            layout.addWidget(warn_label)
        else:
            ok_label = QLabel("Estrutura básica do CSV parece adequada.")
            ok_label.setStyleSheet("color: #006600;")
            layout.addWidget(ok_label)

        btn_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        btn_box.button(QDialogButtonBox.Ok).setText("Continuar")
        btn_box.button(QDialogButtonBox.Cancel).setText("Cancelar")
        btn_box.accepted.connect(dlg.accept)
        btn_box.rejected.connect(dlg.reject)
        layout.addWidget(btn_box)

        result = dlg.exec_()
        if result == QDialog.Accepted:
            if missing:
                self.log("⚠ CSV com avisos de estrutura, prosseguindo por decisão do usuário.")
            else:
                self.log("✔ CSV validado e aprovado pelo usuário.")
            return True
        else:
            self.log("⏹ Processamento cancelado pelo usuário na pré-visualização do CSV.")
            return False
    def log(self, msg):
        text = str(msg)
        self.log_text.appendPlainText(text)
        self.log_buffer.append(text)
        QApplication.processEvents()

    def browse_in(self):
        path, _ = QFileDialog.getOpenFileName(
            self,
            "Selecione o CSV de endereços",
            "",
            "Arquivos CSV (*.csv);;Todos (*.*)"
        )
        if path:
            self.in_edit.setText(path)

    def browse_out(self):
        path, _ = QFileDialog.getSaveFileName(
            self,
            "Selecione o CSV de saída",
            "",
            "Arquivos CSV (*.csv);;Todos (*.*)"
        )
        if path:
            if not path.lower().endswith(".csv"):
                path += ".csv"
            self.out_edit.setText(path)

    def save_template_csv(self):
        path, _ = QFileDialog.getSaveFileName(
            self,
            "Salvar modelo de CSV de endereços",
            "modelo_enderecos.csv",
            "Arquivos CSV (*.csv);;Todos (*.*)"
        )
        if not path:
            return
        if not path.lower().endswith(".csv"):
            path += ".csv"

        headers = [
            "id",
            "state",
            "city",
            "district",
            "street",
            "number",
            "postal_code",
            "country",
            "secondary_search",
            "result_count",
            "confidence_precision",
            "route_start",
            "route_end"
        ]
        example = [
            "P1",
            "SP",
            "Campinas",
            "Centro",
            "Rua Exemplo",
            "123",
            "13000-000",
            "Brasil",
            "3",
            "1",
            "0,7",
            "1",
            ""
        ]

        try:
            with open(path, "w", encoding=CSV_ENCODING, newline="") as f:
                writer = csv.writer(f, delimiter=CSV_DELIMITER)
                writer.writerow(headers)
                writer.writerow(example)
            QMessageBox.information(self, "MapLogis", f"Modelo de CSV salvo em:\n{path}")
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao salvar modelo de CSV:\n{e}")

    def load_settings(self):
        s = QSettings()
        token = s.value("MapLogis/token", "", type=str)
        if token:
            self.token_edit.setText(token)
        blocked = s.value("MapLogis/blockedAreas", "", type=str)
        if blocked:
            self.blocked_edit.setText(blocked)
        last_in = s.value("MapLogis/lastInput", "", type=str)
        if last_in:
            self.in_edit.setText(last_in)
        last_out = s.value("MapLogis/lastOutput", "", type=str)
        if last_out:
            self.out_edit.setText(last_out)

    def save_settings(self):
        s = QSettings()
        s.setValue("MapLogis/token", self.token_edit.text())
        s.setValue("MapLogis/blockedAreas", self.blocked_edit.text())
        s.setValue("MapLogis/lastInput", self.in_edit.text())
        s.setValue("MapLogis/lastOutput", self.out_edit.text())


    def open_output_csv(self, path):
        """Abre o CSV de saída no aplicativo padrão do sistema."""
        if not path:
            QMessageBox.warning(self, "MapLogis", "Caminho do CSV de saída não informado.")
            return
        if not os.path.exists(path):
            QMessageBox.warning(self, "MapLogis", f"Arquivo de saída não encontrado:\n{path}")
            return
        try:
            if sys.platform.startswith("win"):
                os.startfile(path)
            elif sys.platform.startswith("darwin"):
                subprocess.call(["open", path])
            else:
                subprocess.call(["xdg-open", path])
        except Exception as e:
            QMessageBox.critical(self, "MapLogis", f"Erro ao abrir CSV de saída:\n{e}")


    def on_run_clicked(self):
        token = self.token_edit.text().strip()
        blocked_areas = self.blocked_edit.text().strip()
        csv_in = self.in_edit.text().strip()
        csv_out = self.out_edit.text().strip()
        start_id = self.start_id_edit.text().strip()
        end_id = self.end_id_edit.text().strip()
        only_geocode = self.only_geocode_cb.isChecked()
        use_existing_coords = self.use_existing_coords_cb.isChecked()

        if not csv_in:
            QMessageBox.warning(self, "MapLogis", "Informe o CSV de entrada.")
            return
        if not csv_out:
            QMessageBox.warning(self, "MapLogis", "Informe o CSV de saída.")
            return
        if not token:
            QMessageBox.warning(self, "MapLogis", "Informe o Token da API.")
            return

        # Pré-visualização do CSV
        if not self.preview_csv_dialog(csv_in, only_geocode, use_existing_coords):
            return

        # Salva configurações como na versão estável original
        self.save_settings()

        # Limpa log e inicia mensagem
        self.log_buffer.clear()
        self.log_text.clear()
        self.log("Iniciando processamento...")

        from qgis.PyQt.QtCore import Qt
        QApplication.setOverrideCursor(Qt.WaitCursor)
        self.run_btn.setEnabled(True if False else False)  # garante bool e evita erro de nome

        # Popup de progresso simples
        progress = QProgressDialog("Processando com a API MapLogis...", None, 0, 0, self)
        progress.setWindowTitle("MapLogis")
        progress.setWindowModality(Qt.ApplicationModal)
        progress.setAutoClose(False)
        progress.setAutoReset(False)
        progress.show()
        QApplication.processEvents()

        try:
            result = run_maplogis_pipeline(
                csv_input=csv_in,
                csv_output=csv_out,
                only_geocode=only_geocode,
                token=token,
                blocked_areas=blocked_areas,
                use_existing_coords=use_existing_coords,
                start_id=start_id,
                end_id=end_id,
                logger=self.log
            )

            try:
                base, _ = os.path.splitext(csv_out)
                log_path = base + "_log.txt"
                with open(log_path, "w", encoding="utf-8") as f:
                    f.write("\n".join(self.log_buffer))
                self.log(f"ℹ Log salvo em: {log_path}")
            except Exception as e:
                self.log(f"⚠ Não foi possível salvar o log em arquivo: {e}")

            if result and isinstance(result, dict) and result.get("mode") == "rota":
                self.last_route_csv = csv_out
                self.qr_btn.setEnabled(True)
                self.whatsapp_btn.setEnabled(True)
                self.kml_btn.setEnabled(True)
                dk = result.get("dist_km")
                dm = result.get("dur_min")
                if dk is not None and dm is not None:
                    dist_str = f"{dk:.2f}".replace(".", ",")
                    dur_str = f"{dm:.1f}".replace(".", ",")
                    msg = (
                        "Rota concluída.\n"
                        f"Distância: {dist_str} km\n"
                        f"Tempo estimado: {dur_str} min"
                    )
                else:
                    msg = "Rota concluída.\n(sem informações de distância/tempo)"
            else:
                msg = "Processamento concluído.\n(geocodificação sem rota)."

            progress.close()
            # Dialogo final com acoes rapidas
            box = QMessageBox(self)
            box.setWindowTitle("MapLogis")
            box.setText(msg)
            btn_open_csv = box.addButton("Abrir CSV de saída", QMessageBox.ActionRole)
            btn_whatsapp = box.addButton("Enviar via WhatsApp", QMessageBox.ActionRole)
            btn_ok = box.addButton(QMessageBox.Ok)
            box.exec_()
            clicked = box.clickedButton()
            if clicked == btn_open_csv:
                self.open_output_csv(csv_out)
            elif clicked == btn_whatsapp:
                # Usa mesma logica do botao WhatsApp
                try:
                    self.on_whatsapp_clicked()
                except Exception as _e:
                    QMessageBox.critical(self, "MapLogis", f"Erro ao abrir WhatsApp:\n{_e}")

        except Exception as e:
            QMessageBox.critical(
                self,
                "MapLogis - Erro",
                f"Ocorreu um erro:\n{e}"
            )
            self.log(f"ERRO: {e}")
        finally:
            # Garante que o cursor volte ao normal (limpa toda a pilha de overrides)
            while QApplication.overrideCursor() is not None:
                QApplication.restoreOverrideCursor()
            # reabilita o botão (se existir)
            # Fecha o diálogo de progresso caso tenha sido criado
            try:
                progress.close()
            except Exception:
                pass
            try:
                self.run_btn.setEnabled(True)
            except Exception:
                pass
class MapLogisPlugin:
    def __init__(self, iface):
        global IFACE
        self.iface = iface
        IFACE = iface
        self.action = None
        self.dialog = None

    def initGui(self):
        self.action = QAction("MapLogis", self.iface.mainWindow())
        self.action.setStatusTip("Geocodificação e rotas via MapLogis")
        # Ícone no menu/toolbar do QGIS
        self.action.setIcon(QIcon(os.path.join(os.path.dirname(__file__), "icone.png")))
        self.action.triggered.connect(self.run)
        self.iface.addPluginToMenu("&MapLogis", self.action)
        # Toolbar própria para permitir ícone maior/largo
        self.toolbar = self.iface.addToolBar('MapLogis')
        self.toolbar.setObjectName('MapLogisToolbar')
        self.toolbar.setIconSize(QSize(40, 32))
        self.toolbar.addAction(self.action)

    def unload(self):
        if self.action:
            self.iface.removePluginMenu("&MapLogis", self.action)
            self.iface.removeToolBarIcon(self.action)
            try:
                if hasattr(self, 'toolbar') and self.toolbar is not None:
                    self.iface.mainWindow().removeToolBar(self.toolbar)
                    self.toolbar = None
            except Exception:
                pass

    def run(self):
        if self.dialog is None:
            self.dialog = MapLogisDialog(self.iface.mainWindow())
        self.dialog.show()
        self.dialog.raise_()
        self.dialog.activateWindow()