"""
/***************************************************************************
 MIHADialog
                                 A QGIS plugin
 This plugin calculates environmental and social indexes
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-01-14
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Unicampania
        email                : savino.giacobbe@unicampania.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets

"""" Import Modifiche """
from qgis.core import QgsFields, QgsField, QgsVectorLayer, QgsFeature, QgsProject
from PyQt5.QtCore import QVariant, QSettings, Qt
from PyQt5.QtWidgets import QProgressDialog
from qgis.PyQt.QtCore import QVariant # Forse deprecato

from qgis.core import QgsGraduatedSymbolRenderer, QgsRendererRange, QgsSymbol
from PyQt5.QtGui import QColor

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'MIHA_dialog_base.ui'))


class MIHADialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(MIHADialog, self).__init__(parent)
        self.setupUi(self)

        """ Modifiche Plugin """
        self.CalcolaIndicatoriPermeabilita_pushButton.clicked.connect(self.calculate_permeability_indicators)
        self.CalcolaIndicatoriVerdeUrbano_pushButton.clicked.connect(self.calculate_green_indicators)
        self.SalvaStato1_pushButton.clicked.connect(self.save_state_1)
        self.SalvaStato2_pushButton.clicked.connect(self.save_state_2)
        self.CalcolaIndicatoriSociali_pushButton.clicked.connect(self.calculate_social_indicators)

        # Call the function to populate combo boxes when the dialog is initialized
        self.populate_layer_comboboxes()

        # Carica lo stato salvato
        self.load_state()

        # Resetta le combobox selezionate (reset button)
        self.ResetSuperfici_pushButton.clicked.connect(lambda: self.reset_state("ResetSuperfici_pushButton"))
        self.ResetVegetazione_pushButton.clicked.connect(lambda: self.reset_state("ResetVegetazione_pushButton"))
        self.ResetAlberi_pushButton.clicked.connect(lambda: self.reset_state("ResetAlberi_pushButton"))
        self.ResetEdifici_pushButton.clicked.connect(lambda: self.reset_state("ResetEdifici_pushButton"))
        self.ResetParcheggi_pushButton.clicked.connect(lambda: self.reset_state("ResetParcheggi_pushButton"))
        self.ResetSTP_pushButton.clicked.connect(lambda: self.reset_state("ResetSTP_pushButton"))
        self.ResetPisteCiclabili_pushButton.clicked.connect(lambda: self.reset_state("ResetPisteCiclabili_pushButton"))
        self.ResetFermateTrasportoPubblico_pushButton.clicked.connect(lambda: self.reset_state("ResetFermateTrasportoPubblico_pushButton"))

        # Connect signals for layer selection changes
        self.LayerSuperfici_comboBox.currentIndexChanged.connect(self.update_superfici_attributes)
        self.LayerVegetazione_comboBox.currentIndexChanged.connect(self.update_vegetazione_attributes)
        self.LayerAlberi_comboBox.currentIndexChanged.connect(self.update_alberi_attributes)
        self.LayerEdifici_comboBox.currentIndexChanged.connect(self.update_edifici_attributes)
        self.LayerParcheggi_comboBox.currentIndexChanged.connect(self.update_parcheggi_attributes)
        self.LayerSistemaTrasportoPubblico_comboBox.currentIndexChanged.connect(self.update_STP_attributes)
        self.LayerPisteCiclabili_comboBox.currentIndexChanged.connect(self.update_pisteCiclabili_attributes)
        self.LayerFermateTrasportoPubblico_comboBox.currentIndexChanged.connect(self.update_FTP_attributes)


        # Disable input scelta bacino
        self.SceltaBacino_checkBox.stateChanged.connect(self.toggle_scelta_bacino_lineedit)

    def populate_layer_comboboxes(self):
        """Populate combo boxes with the names of layers in the current QGIS project."""
        # Get the list of layers from the current QGIS project
        layers = QgsProject.instance().mapLayers().values()

        # Clear existing items in combo boxes
        self.LayerSuperfici_comboBox.clear()
        self.LayerVegetazione_comboBox.clear()
        self.LayerAlberi_comboBox.clear()
        self.LayerEdifici_comboBox.clear()
        self.LayerParcheggi_comboBox.clear()
        self.LayerSistemaTrasportoPubblico_comboBox.clear()
        self.LayerPisteCiclabili_comboBox.clear()
        self.LayerFermateTrasportoPubblico_comboBox.clear()

        # Add the default "Seleziona il layer" item
        default_text = "Seleziona il layer"
        self.LayerSuperfici_comboBox.addItem(default_text)
        self.LayerVegetazione_comboBox.addItem(default_text)
        self.LayerAlberi_comboBox.addItem(default_text)
        self.LayerEdifici_comboBox.addItem(default_text)
        self.LayerParcheggi_comboBox.addItem(default_text)
        self.LayerSistemaTrasportoPubblico_comboBox.addItem(default_text)
        self.LayerPisteCiclabili_comboBox.addItem(default_text)
        self.LayerFermateTrasportoPubblico_comboBox.addItem(default_text)

        # Populate combo boxes with layer names
        for layer in layers:
            layer_name = layer.name()
            self.LayerSuperfici_comboBox.addItem(layer_name)
            self.LayerVegetazione_comboBox.addItem(layer_name)
            self.LayerAlberi_comboBox.addItem(layer_name)
            self.LayerEdifici_comboBox.addItem(layer_name)
            self.LayerParcheggi_comboBox.addItem(layer_name)
            self.LayerSistemaTrasportoPubblico_comboBox.addItem(layer_name)
            self.LayerPisteCiclabili_comboBox.addItem(layer_name)
            self.LayerFermateTrasportoPubblico_comboBox.addItem(layer_name)


        # Set the default item as the selected item
        self.LayerSuperfici_comboBox.setCurrentIndex(0)
        self.LayerVegetazione_comboBox.setCurrentIndex(0)
        self.LayerAlberi_comboBox.setCurrentIndex(0)
        self.LayerEdifici_comboBox.setCurrentIndex(0)
        self.LayerParcheggi_comboBox.setCurrentIndex(0)
        self.LayerSistemaTrasportoPubblico_comboBox.setCurrentIndex(0)
        self.LayerPisteCiclabili_comboBox.setCurrentIndex(0)
        self.LayerFermateTrasportoPubblico_comboBox.setCurrentIndex(0)

    def update_superfici_attributes(self):
        """Update Superfici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerSuperfici_comboBox, [
            self.MaterialiSuperfici_comboBox,
            self.AreaSuperfici_comboBox,
            self.ProprietaSuperfici_comboBox,
            self.IDBacinoSuperfici_comboBox,
            self.IDLottoSuperfici_comboBox
        ])

    def update_vegetazione_attributes(self):
        """Update Vegetazione attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerVegetazione_comboBox, [
            self.MaterialiVegetazione_comboBox,
            self.AreaVegetazione_comboBox,
            self.ProprietaVegetazione_comboBox,
            self.IDBacinoVegetazione_comboBox,
            self.IDLottoVegetazione_comboBox,
            self.AreaVegetazione_comboBox
        ])

    def update_alberi_attributes(self):
        """Update Alberi attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerAlberi_comboBox, [
            self.AltezzaAlberi_comboBox,
            self.IDLottoAlberi_comboBox,
            self.ProprietaAlberi_comboBox,
            self.IDBacinoAlberi_comboBox
        ])

    def update_edifici_attributes(self):
        """Update Edifici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerEdifici_comboBox, [
            self.AreaEdifici_comboBox,
            self.IDBacinoEdifici_comboBox,
            self.ProprietaEdifici_comboBox,
            self.PopolazioneResidenteEdifici_comboBox,
            self.IDBacinoEdifici_comboBox,
            self.MaterialiEdifici_comboBox,
            self.PendenzaCopertureEdifici_comboBox,
            self.IDLottoEdifici_comboBox,
            self.Famiglie_comboBox
        ])

    def update_parcheggi_attributes(self):
        """Update Superfici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerParcheggi_comboBox, [
            self.IDBacinoParcheggi_comboBox,
            self.AreaParcheggi_comboBox
        ])

    def update_STP_attributes(self):
        """Update Superfici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerSistemaTrasportoPubblico_comboBox, [
            self.IDBacinoSistemaTrasportoPubblico_comboBox,
            self.LunghezzaSistemaTrasportoPubblico_comboBox
        ])

    def update_pisteCiclabili_attributes(self):
        """Update Superfici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerPisteCiclabili_comboBox, [
            self.IDBacinoPisteCiclabili_comboBox,
            self.LunghezzaPisteCiclabili_comboBox
        ])

    def update_FTP_attributes(self):
        """Update Superfici attribute combo boxes based on the selected layer."""
        self.populate_attribute_combobox(self.LayerFermateTrasportoPubblico_comboBox, [
            self.IDBacinoFermateTrasportoPubblico_comboBox
        ])

    def populate_attribute_combobox(self, layer_combobox, attribute_comboboxes):
        """Populate attribute combo boxes based on the selected layer."""
        # Get selected layer
        layer_name = layer_combobox.currentText()
        if layer_name == "Seleziona il layer":
            for combo in attribute_comboboxes:
                combo.clear()
                combo.addItem("Seleziona un attributo")
            return

        # Find the layer in the current project
        layers = QgsProject.instance().mapLayersByName(layer_name)
        if not layers or not isinstance(layers[0], QgsVectorLayer):
            for combo in attribute_comboboxes:
                combo.clear()
                combo.addItem("Nessun attributo disponibile")
            return

        layer = layers[0]

        # Populate each attribute combo box
        for combo in attribute_comboboxes:
            combo.clear()
            combo.addItem("Seleziona un attributo")
            for field in layer.fields():
                combo.addItem(field.name())

    def toggle_scelta_bacino_lineedit(self):
        """Disattiva la SceltaBacino_lineEdit se la checkbox è selezionata, altrimenti la riattiva."""
        if self.SceltaBacino_checkBox.isChecked():
            self.SceltaBacino_lineEdit.setDisabled(True)  # Disattiva la LineEdit
        else:
            self.SceltaBacino_lineEdit.setDisabled(False)  # Riattiva la LineEdit

    def reset_state(self, button_name):
        """Resetta le combobox associate a un determinato pulsante"""

        reset_map = {
            "ResetSuperfici_pushButton": [
                self.LayerSuperfici_comboBox,
                self.MaterialiSuperfici_comboBox,
                self.AreaSuperfici_comboBox,
                self.ProprietaSuperfici_comboBox,
                self.IDBacinoSuperfici_comboBox,
                self.IDLottoSuperfici_comboBox
            ],
            "ResetVegetazione_pushButton": [
                self.LayerVegetazione_comboBox,
                self.MaterialiVegetazione_comboBox,
                self.AreaVegetazione_comboBox,
                self.ProprietaVegetazione_comboBox,
                self.IDBacinoVegetazione_comboBox,
                self.IDLottoVegetazione_comboBox
            ],
            "ResetAlberi_pushButton": [
                self.LayerAlberi_comboBox,
                self.AltezzaAlberi_comboBox,
                self.IDLottoAlberi_comboBox,
                self.ProprietaAlberi_comboBox,
                self.IDBacinoAlberi_comboBox
            ],
            "ResetEdifici_pushButton": [
                self.LayerEdifici_comboBox,
                self.AreaEdifici_comboBox,
                self.PopolazioneResidenteEdifici_comboBox,
                self.ProprietaEdifici_comboBox,
                self.IDBacinoEdifici_comboBox,
                self.MaterialiEdifici_comboBox,
                self.PendenzaCopertureEdifici_comboBox,
                self.IDLottoEdifici_comboBox,
                self.Famiglie_comboBox
            ],
            "ResetParcheggi_pushButton": [
                self.LayerParcheggi_comboBox,
                self.IDBacinoParcheggi_comboBox,
                self.AreaParcheggi_comboBox
            ],
            "ResetSTP_pushButton": [
                self.LayerSistemaTrasportoPubblico_comboBox,
                self.IDBacinoSistemaTrasportoPubblico_comboBox,
                self.LunghezzaSistemaTrasportoPubblico_comboBox
            ],
            "ResetPisteCiclabili_pushButton": [
                self.LayerPisteCiclabili_comboBox,
                self.IDBacinoPisteCiclabili_comboBox,
                self.LunghezzaPisteCiclabili_comboBox
            ],
            "ResetFermateTrasportoPubblico_pushButton": [
                self.LayerFermateTrasportoPubblico_comboBox,
                self.IDBacinoFermateTrasportoPubblico_comboBox
            ]
        }

        if button_name in reset_map:
            for combobox in reset_map[button_name]:
                combobox.setCurrentIndex(0)  # Resetta alla prima opzione disponibile

    def save_state_1(self):
        """Salva lo stato delle combobox."""
        settings = QSettings()

        comboboxes = {
            "LayerSuperfici": self.LayerSuperfici_comboBox,
            "LayerVegetazione": self.LayerVegetazione_comboBox,
            "LayerAlberi": self.LayerAlberi_comboBox,
            "LayerEdifici": self.LayerEdifici_comboBox,
            "MaterialiSuperfici": self.MaterialiSuperfici_comboBox,
            "AreaSuperfici": self.AreaSuperfici_comboBox,
            "ProprietaSuperfici": self.ProprietaSuperfici_comboBox,
            "IDLottoSuperfici": self.IDLottoSuperfici_comboBox,
            "IDBacinoSuperfici": self.IDBacinoSuperfici_comboBox,
            "MaterialiVegetazione": self.MaterialiVegetazione_comboBox,
            "AreaVegetazione": self.AreaVegetazione_comboBox,
            "IDLottoVegetazione": self.IDLottoVegetazione_comboBox,
            "IDBacinoVegetazione": self.IDBacinoVegetazione_comboBox,
            "ProprietaVegetazione": self.ProprietaVegetazione_comboBox,
            "AltezzaAlberi": self.AltezzaAlberi_comboBox,
            "IDLottoAlberi": self.IDLottoAlberi_comboBox,
            "IDBacinoAlberi": self.IDBacinoAlberi_comboBox,
            "ProprietaAlberi": self.ProprietaAlberi_comboBox,
            "AreaEdifici": self.AreaEdifici_comboBox,
            "IDLottoEdifici": self.IDLottoEdifici_comboBox,
            "Famiglie": self.Famiglie_comboBox,
            "IDBacinoEdifici": self.IDBacinoEdifici_comboBox,
            "PopolazioneResidenteEdifici": self.PopolazioneResidenteEdifici_comboBox,
            "MaterialiEdifici": self.MaterialiEdifici_comboBox,
            "PendenzaCopertureEdifici": self.PendenzaCopertureEdifici_comboBox,
            "ProprietaEdifici": self.ProprietaEdifici_comboBox,
        }

        for key, combobox in comboboxes.items():
            settings.setValue(f"MIHA/{key}", combobox.currentText())

        settings.sync()  # Forza il salvataggio

        QtWidgets.QMessageBox.information(self, "Stato Salvato", "Lo stato è stato salvato con successo!")

    def save_state_2(self):
        """Salva lo stato delle combobox relative ai trasporti e parcheggi."""
        settings = QSettings()

        # Salva i valori delle combobox per parcheggi
        settings.setValue("MIHA/LayerParcheggi", self.LayerParcheggi_comboBox.currentText())
        settings.setValue("MIHA/IDBacinoParcheggi", self.IDBacinoParcheggi_comboBox.currentText())
        settings.setValue("MIHA/AreaParcheggi", self.AreaParcheggi_comboBox.currentText())

        # Salva i valori delle combobox per il sistema di trasporto pubblico
        settings.setValue("MIHA/LayerSistemaTrasportoPubblico",
                          self.LayerSistemaTrasportoPubblico_comboBox.currentText())
        settings.setValue("MIHA/IDBacinoSistemaTrasportoPubblico",
                          self.IDBacinoSistemaTrasportoPubblico_comboBox.currentText())
        settings.setValue("MIHA/LunghezzaSistemaTrasportoPubblico",
                          self.LunghezzaSistemaTrasportoPubblico_comboBox.currentText())

        # Salva i valori delle combobox per le piste ciclabili
        settings.setValue("MIHA/LayerPisteCiclabili", self.LayerPisteCiclabili_comboBox.currentText())
        settings.setValue("MIHA/IDBacinoPisteCiclabili", self.IDBacinoPisteCiclabili_comboBox.currentText())
        settings.setValue("MIHA/LunghezzaPisteCiclabili",
                          self.LunghezzaPisteCiclabili_comboBox.currentText())

        # Salva i valori delle combobox per le fermate del trasporto pubblico
        settings.setValue("MIHA/LayerFermateTrasportoPubblico",
                          self.LayerFermateTrasportoPubblico_comboBox.currentText())
        settings.setValue("MIHA/IDBacinoFermateTrasportoPubblico",
                          self.IDBacinoFermateTrasportoPubblico_comboBox.currentText())

        QtWidgets.QMessageBox.information(self, "Stato Salvato",
                                          "Lo stato è stato salvato con successo!")

    def load_state(self):
        """Carica lo stato delle combobox salvate, mantenendo gli attributi dopo un aggiornamento."""
        settings = QSettings()

        # Carica i layer salvati
        layer_comboboxes = {
            "LayerSuperfici": self.LayerSuperfici_comboBox,
            "LayerVegetazione": self.LayerVegetazione_comboBox,
            "LayerAlberi": self.LayerAlberi_comboBox,
            "LayerEdifici": self.LayerEdifici_comboBox,
            "LayerParcheggi": self.LayerParcheggi_comboBox,
            "LayerSistemaTrasportoPubblico": self.LayerSistemaTrasportoPubblico_comboBox,
            "LayerPisteCiclabili": self.LayerPisteCiclabili_comboBox,
            "LayerFermateTrasportoPubblico": self.LayerFermateTrasportoPubblico_comboBox,
        }

        for key, combobox in layer_comboboxes.items():
            saved_value = settings.value(f"MIHA/{key}", "Seleziona il layer")
            combobox.setCurrentText(saved_value)

        # **Forza il ricaricamento degli attributi DOPO che i layer sono stati caricati**
        self.update_superfici_attributes()
        self.update_vegetazione_attributes()
        self.update_alberi_attributes()
        self.update_edifici_attributes()
        self.update_parcheggi_attributes()
        self.update_STP_attributes()
        self.update_pisteCiclabili_attributes()
        self.update_FTP_attributes()

        # Carica gli attributi salvati dopo il caricamento dei layer
        attribute_comboboxes = {
            "MaterialiSuperfici": self.MaterialiSuperfici_comboBox,
            "AreaSuperfici": self.AreaSuperfici_comboBox,
            "ProprietaSuperfici": self.ProprietaSuperfici_comboBox,
            "IDLottoSuperfici": self.IDLottoSuperfici_comboBox,
            "IDBacinoSuperfici": self.IDBacinoSuperfici_comboBox,
            "MaterialiVegetazione": self.MaterialiVegetazione_comboBox,
            "AreaVegetazione": self.AreaVegetazione_comboBox,
            "IDLottoVegetazione": self.IDLottoVegetazione_comboBox,
            "IDBacinoVegetazione": self.IDBacinoVegetazione_comboBox,
            "ProprietaVegetazione": self.ProprietaVegetazione_comboBox,
            "AltezzaAlberi": self.AltezzaAlberi_comboBox,
            "IDLottoAlberi": self.IDLottoAlberi_comboBox,
            "IDBacinoAlberi": self.IDBacinoAlberi_comboBox,
            "ProprietaAlberi": self.ProprietaAlberi_comboBox,
            "AreaEdifici": self.AreaEdifici_comboBox,
            "IDLottoEdifici": self.IDLottoEdifici_comboBox,
            "Famiglie": self.Famiglie_comboBox,
            "IDBacinoEdifici": self.IDBacinoEdifici_comboBox,
            "PopolazioneResidenteEdifici": self.PopolazioneResidenteEdifici_comboBox,
            "MaterialiEdifici": self.MaterialiEdifici_comboBox,
            "PendenzaCopertureEdifici": self.PendenzaCopertureEdifici_comboBox,
            "ProprietaEdifici": self.ProprietaEdifici_comboBox,
            "IDBacinoParcheggi": self.IDBacinoParcheggi_comboBox,
            "AreaParcheggi": self.AreaParcheggi_comboBox,
            "IDBacinoSistemaTrasportoPubblico": self.IDBacinoSistemaTrasportoPubblico_comboBox,
            "LunghezzaSistemaTrasportoPubblico": self.LunghezzaSistemaTrasportoPubblico_comboBox,
            "IDBacinoPisteCiclabili": self.IDBacinoPisteCiclabili_comboBox,
            "LunghezzaPisteCiclabili": self.LunghezzaPisteCiclabili_comboBox,
            "IDBacinoFermateTrasportoPubblico": self.IDBacinoFermateTrasportoPubblico_comboBox,
        }

        for key, combobox in attribute_comboboxes.items():
            saved_value = settings.value(f"MIHA/{key}", "Seleziona un attributo")
            combobox.setCurrentText(saved_value)

    def get_selected_basin_ids(self):
        """Restituisce l'insieme degli ID bacino selezionati per il filtraggio.
        Se la checkbox è selezionata, restituisce None per considerare tutti i bacini.
        Se la checkbox non è selezionata e il campo è vuoto, mostra un avviso e impedisce il calcolo.
        """
        if self.SceltaBacino_checkBox.isChecked():
            return set()  # Nessun filtro, considera tutti i bacini

        bacini_text = self.SceltaBacino_lineEdit.text().strip()
        if not bacini_text:
            QtWidgets.QMessageBox.warning(self, "Errore",
                                          "Seleziona almeno un ID bacino o attiva l'opzione per considerare tutti i bacini.")
            return None  # Nessun bacino valido, impedisce il calcolo

        # Convertire la stringa in un set di numeri interi
        try:
            selected_basin_ids = {int(x.strip()) for x in bacini_text.split(",") if x.strip().isdigit()}
            if not selected_basin_ids:
                QtWidgets.QMessageBox.warning(self, "Errore",
                                              "Inserisci almeno un ID bacino valido o attiva la checkbox per considerarli tutti.")
                return None  # Nessun bacino valido, impedisce il calcolo
            return selected_basin_ids
        except ValueError:
            QtWidgets.QMessageBox.warning(self, "Errore", "Inserisci solo numeri separati da virgole nel campo Bacino.")
            return None  # Impedisce il calcolo

    def is_valid_feature(self, feature, bacino_attr, selected_basin_ids):
        """Verifica se un elemento deve essere considerato nel calcolo in base all'ID Bacino."""
        if selected_basin_ids is None:
            return True  # Nessun filtro attivo, considera tutti gli elementi

        bacino_value = feature[bacino_attr]
        if bacino_value is None or bacino_value == "":
            return False  # Escludi se il valore è vuoto

        try:
            return int(str(bacino_value).strip()) in selected_basin_ids
        except ValueError:
            return False  # Se il valore non è un numero valido, escludilo

    def should_include_feature(self, feature, property_field, consider_pubblica, consider_privata):
        """Verifica se l'elemento deve essere incluso in base alla proprietà selezionata dall'utente."""
        if not property_field:
            return True  # Se non è stato selezionato un campo proprietà, considera tutto

        property_value = (feature[property_field] or "").strip().lower()
        pubblica_values = {"pubblica", "public", "pubblico"}
        privata_values = {"privata", "private", "privato"}

        if consider_pubblica and property_value in pubblica_values:
            return True
        if consider_privata and property_value in privata_values:
            return True
        if consider_pubblica and consider_privata:
            return True  # Se entrambe sono selezionate, considera tutto

        return False  # Escludi gli altri valori

    def calculate_permeability_indicators(self):
        """Calcola RIE, BAF e Indice di Permeabilità e crea layer separati."""
        calculate_iperm = self.IndicePermeabilita_checkBox.isChecked()
        calculate_rie = self.RIE_checkBox.isChecked()
        calculate_baf = self.BAF_checkBox.isChecked()

        if not (calculate_iperm or calculate_rie or calculate_baf):
            QtWidgets.QMessageBox.warning(self, "Errore", "Seleziona almeno un indicatore da calcolare.")
            return

        # Recupera gli ID Bacino da considerare
        selected_basin_ids = self.get_selected_basin_ids()

        # Se nessun ID bacino è valido, interrompe il calcolo
        if selected_basin_ids is None:
            return

        # Mostra il popup di caricamento
        progress = QProgressDialog("Calcolo in corso...", None, 0, 0, self)
        progress.setWindowTitle("Calcolo Ind. Perm.")
        progress.setWindowModality(Qt.ApplicationModal)
        progress.show()

        id_bacino_superfici_attr = self.IDBacinoSuperfici_comboBox.currentText()
        id_bacino_edifici_attr = self.IDBacinoEdifici_comboBox.currentText()
        id_bacino_vegetazione_attr = self.IDBacinoVegetazione_comboBox.currentText()
        id_bacino_alberi_attr = self.IDBacinoAlberi_comboBox.currentText()

        # Recupera i campi delle proprietà
        prop_superfici_attr = self.ProprietaSuperfici_comboBox.currentText()
        prop_vegetazione_attr = self.ProprietaVegetazione_comboBox.currentText()
        prop_alberi_attr = self.ProprietaAlberi_comboBox.currentText()
        prop_edifici_attr = self.ProprietaEdifici_comboBox.currentText()

        # Recupera la selezione dell'utente (Pubblica, Privata o entrambe)
        consider_pubblica = self.Pubblica_checkBox.isChecked()
        consider_privata = self.Privata_checkBox.isChecked()

        # Recupera i layer selezionati
        layer_superfici = QgsProject.instance().mapLayersByName(self.LayerSuperfici_comboBox.currentText())[0]
        layer_edifici = QgsProject.instance().mapLayersByName(self.LayerEdifici_comboBox.currentText())[0]
        layer_vegetazione = QgsProject.instance().mapLayersByName(self.LayerVegetazione_comboBox.currentText())[0]
        layer_alberi = QgsProject.instance().mapLayersByName(self.LayerAlberi_comboBox.currentText())[0]

        results = {}

        # Processa le superfici
        for feature in layer_superfici.getFeatures():
            if not self.is_valid_feature(feature, id_bacino_superfici_attr, selected_basin_ids):
                continue

            if not self.should_include_feature(feature, prop_superfici_attr, consider_pubblica, consider_privata):
                continue

            id_lotto = feature[self.IDLottoSuperfici_comboBox.currentText()]
            area_superficie = float(feature[self.AreaSuperfici_comboBox.currentText()] or 0)
            materiali_superficie = (feature[self.MaterialiSuperfici_comboBox.currentText()] or "").strip().lower()
            pi0_BAF, pi0_RIE = self.get_material_coefficients(materiali_superficie, "superfici")

            if id_lotto not in results:
                results[id_lotto] = self.initialize_lotto_entry()

            results[id_lotto]["area_superfici"] += area_superficie
            results[id_lotto]["weighted_baf"] += area_superficie * pi0_BAF
            results[id_lotto]["denominator_rie"] += area_superficie * pi0_RIE

        # Processa la vegetazione
        for feature in layer_vegetazione.getFeatures():
            if not self.is_valid_feature(feature, id_bacino_vegetazione_attr, selected_basin_ids):
                continue

            if not self.should_include_feature(feature, prop_vegetazione_attr, consider_pubblica, consider_privata):
                continue

            id_lotto = feature[self.IDLottoVegetazione_comboBox.currentText()]
            area_vegetazione = float(feature[self.AreaVegetazione_comboBox.currentText()] or 0)
            materiali_vegetazione = (feature[self.MaterialiVegetazione_comboBox.currentText()] or "").strip().lower()
            pi1_BAF, pi1_RIE = self.get_material_coefficients(materiali_vegetazione, "vegetazione")

            if id_lotto not in results:
                results[id_lotto] = self.initialize_lotto_entry()

            results[id_lotto]["area_vegetazione"] += area_vegetazione
            results[id_lotto]["weighted_baf"] += area_vegetazione * pi1_BAF
            if pi1_RIE > 0:
                results[id_lotto]["weighted_rie"] += area_vegetazione / pi1_RIE
            results[id_lotto]["denominator_rie"] += area_vegetazione

        # Processa gli edifici
        for feature in layer_edifici.getFeatures():
            if not self.is_valid_feature(feature, id_bacino_edifici_attr, selected_basin_ids):
                continue

            if not self.should_include_feature(feature, prop_edifici_attr, consider_pubblica, consider_privata):
                continue

            id_lotto = feature[self.IDLottoEdifici_comboBox.currentText()]
            area_edificio = float(feature[self.AreaEdifici_comboBox.currentText()] or 0)
            materiali_edificio = (feature[self.MaterialiEdifici_comboBox.currentText()] or "").strip().lower()
            pendenza_edificio = float(feature[self.PendenzaCopertureEdifici_comboBox.currentText()] or 0)
            pi2_BAF, pi2_RIE = self.get_material_coefficients(materiali_edificio, "edifici", pendenza_edificio)

            if id_lotto not in results:
                results[id_lotto] = self.initialize_lotto_entry()

            results[id_lotto]["area_edifici"] += area_edificio
            results[id_lotto]["weighted_baf"] += area_edificio * pi2_BAF
            results[id_lotto]["denominator_rie"] += area_edificio * pi2_RIE

        # Processa gli alberi
        for feature in layer_alberi.getFeatures():
            if not self.is_valid_feature(feature, id_bacino_alberi_attr, selected_basin_ids):
                continue

            if not self.should_include_feature(feature, prop_alberi_attr, consider_pubblica, consider_privata):
                continue

            id_lotto = feature[self.IDLottoSuperfici_comboBox.currentText()]
            altezza_albero = float(feature[self.AltezzaAlberi_comboBox.currentText()] or 0)
            tree_area = self.calculate_tree_area(altezza_albero)

            if id_lotto not in results:
                results[id_lotto] = self.initialize_lotto_entry()

            results[id_lotto]["tree_area"] += tree_area

        # Genera le geometrie dei lotti
        lot_geometries = self.generate_lot_geometries()

        # Calcola i valori finali
        rie_results = {}
        baf_results = {}
        iperm_results = {}

        for id_lotto, values in results.items():
            total_area = values["area_superfici"] + values["area_vegetazione"] + values["area_edifici"]

            if total_area > 0 and values["denominator_rie"] > 0:
                if calculate_baf:
                    baf_results[id_lotto] = round(values["weighted_baf"] / total_area, 2)
                if calculate_rie:
                    rie_results[id_lotto] = round(
                        (values["weighted_rie"] + values["tree_area"]) / values["denominator_rie"], 2)
                if calculate_iperm:
                    iperm_results[id_lotto] = round(values["area_vegetazione"] / total_area, 2)

        # Crea layer separati
        if calculate_rie:
            self.create_sub_layer("Indicatori_RIE", "RIE", lot_geometries, rie_results, [
                (0, 1.5, "#b2e2a8"),
                (1.5, 2.5, "#66c2a5"),
                (2.5, 4.0, "#2ca25f"),
                (4.0, 6.0, "#006d2c"),
                (6.0, 10.0, "#00441b")
            ])

        if calculate_baf:
            self.create_sub_layer("Indicatori_BAF", "BAF", lot_geometries, baf_results, [
                (0, 0.3, "#e5f5f9"),
                (0.3, 0.5, "#99d8c9"),
                (0.5, 0.7, "#2ca25f"),
                (0.7, 1.0, "#006d2c")
            ], pattern=True)

        if calculate_iperm:
            self.create_sub_layer("Indicatori_IPerm", "Indice di Permeabilità", lot_geometries, iperm_results, [
                (0, 0.25, "#feedde"),
                (0.25, 0.5, "#fdbe85"),
                (0.5, 0.75, "#fd8d3c"),
                (0.75, 1.0, "#d94701")
            ], pattern=True)

        # Chiude il popup e mostra un messaggio
        progress.close()
        QtWidgets.QMessageBox.information(self, "Calcolo completato",
                                          "Il calcolo degli indicatori è stato completato con successo!")

    def create_sub_layer(self, name, field_name, geometries, results, color_ranges, pattern=False):
        """Crea un sottolayer con la simbologia specifica per un indicatore."""
        new_layer = QgsVectorLayer("Polygon?crs=EPSG:4326", name, "memory")
        provider = new_layer.dataProvider()

        # Definisci i campi
        fields = QgsFields()
        fields.append(QgsField("ID_Lotto", QVariant.String))
        fields.append(QgsField(field_name, QVariant.Double))
        provider.addAttributes(fields)
        new_layer.updateFields()

        # Aggiungi le feature al layer
        for id_lotto, value in results.items():
            # Usa direttamente il valore, non il sotto-attributo
            if value is None:
                continue

            feature = QgsFeature()
            feature.setGeometry(geometries.get(id_lotto))  # Assegna la geometria
            feature.setAttributes([id_lotto, round(value, 2)])  # Arrotonda il valore
            provider.addFeature(feature)

        QgsProject.instance().addMapLayer(new_layer)

        # Applica la simbologia graduata
        self.apply_graduated_symbology(new_layer, field_name, color_ranges, pattern)

    def calculate_green_indicators(self):
        """Calcola Verde Urbano Fruibile e Numero di Alberi per 100 Abitanti in un unico layer."""
        calculate_vuf = self.VerdeUrbanoFruibile_checkBox.isChecked()
        calculate_num_alberi = self.NumAlberi_checkBox.isChecked()

        if not (calculate_vuf or calculate_num_alberi):
            QtWidgets.QMessageBox.warning(self, "Errore", "Seleziona almeno un indicatore da calcolare.")
            return

        # Recupera gli ID Bacino da considerare
        selected_basin_ids = self.get_selected_basin_ids()

        # Se nessun ID bacino è valido, interrompe il calcolo
        if selected_basin_ids is None:
            return

        # Mostra il popup di caricamento
        progress = QProgressDialog("Calcolo in corso...", None, 0, 0, self)
        progress.setWindowTitle("Calcolo Ind. Verde Urbano")
        progress.setWindowModality(Qt.ApplicationModal)
        progress.show()

        id_bacino_vegetazione_attr = self.IDBacinoVegetazione_comboBox.currentText()
        id_bacino_edifici_attr = self.IDBacinoEdifici_comboBox.currentText()
        id_bacino_alberi_attr = self.IDBacinoAlberi_comboBox.currentText()
        pop_residente_attr = self.PopolazioneResidenteEdifici_comboBox.currentText()
        area_vegetazione_attr = self.AreaVegetazione_comboBox.currentText()
        proprieta_vegetazione_attr = self.ProprietaVegetazione_comboBox.currentText()
        proprieta_alberi_attr = self.ProprietaAlberi_comboBox.currentText()

        layer_vegetazione = QgsProject.instance().mapLayersByName(self.LayerVegetazione_comboBox.currentText())[0]
        layer_edifici = QgsProject.instance().mapLayersByName(self.LayerEdifici_comboBox.currentText())[0]
        layer_alberi = QgsProject.instance().mapLayersByName(self.LayerAlberi_comboBox.currentText())[0]

        # Dizionari per memorizzare i valori aggregati
        vegetazione_pubblica_per_bacino = {}
        num_alberi_per_bacino = {}
        popolazione_per_bacino = {}

        # Somma le aree di vegetazione pubblica per ID Bacino
        for feature in layer_vegetazione.getFeatures():
            id_bacino = feature[id_bacino_vegetazione_attr]
            if not id_bacino:
                continue
            proprieta_val = feature[proprieta_vegetazione_attr].strip().lower()
            if proprieta_val in ["pubblica", "public"]:
                area_vegetazione = float(feature[area_vegetazione_attr] or 0)
                vegetazione_pubblica_per_bacino[id_bacino] = vegetazione_pubblica_per_bacino.get(id_bacino,
                                                                                                 0) + area_vegetazione

        # Conta le occorrenze degli alberi per ID Bacino solo se la proprietà è "Pubblica" o "Public"
        for feature in layer_alberi.getFeatures():
            id_bacino = feature[id_bacino_alberi_attr]
            if not id_bacino:
                continue
            proprieta_val = feature[proprieta_alberi_attr].strip().lower()
            if proprieta_val in ["pubblica", "public"]:
                num_alberi_per_bacino[id_bacino] = num_alberi_per_bacino.get(id_bacino, 0) + 1

        # Somma la popolazione residente per ID Bacino
        for feature in layer_edifici.getFeatures():
            id_bacino = feature[id_bacino_edifici_attr]
            if not id_bacino:
                continue
            pop_residente = float(feature[pop_residente_attr] or 0)
            popolazione_per_bacino[id_bacino] = popolazione_per_bacino.get(id_bacino, 0) + pop_residente

        # Crea il layer di output
        layer_name = "Indicatori del Verde Urbano"
        new_layer = QgsVectorLayer("Polygon?crs=EPSG:4326", layer_name, "memory")
        provider = new_layer.dataProvider()

        # Definisci i campi del layer
        fields = QgsFields()
        fields.append(QgsField("ID_Bacino", QVariant.String))
        fields.append(QgsField("VUF", QVariant.Double))
        fields.append(QgsField("Num_Alberi_Pop", QVariant.Double))
        provider.addAttributes(fields)
        new_layer.updateFields()

        # Calcola i valori finali per ogni ID Bacino e crea una feature per ciascuno
        for id_bacino in set(vegetazione_pubblica_per_bacino.keys()).union(num_alberi_per_bacino.keys()):
            vuf = 0.0
            num_alberi_popolazione = 0.0

            if id_bacino in vegetazione_pubblica_per_bacino and id_bacino in popolazione_per_bacino and \
                    popolazione_per_bacino[id_bacino] > 0:
                vuf = vegetazione_pubblica_per_bacino[id_bacino] / popolazione_per_bacino[id_bacino]
                vuf = round(vuf, 2)  # Arrotonda alla seconda cifra decimale

            if id_bacino in num_alberi_per_bacino and id_bacino in popolazione_per_bacino and popolazione_per_bacino[
                id_bacino] > 0:
                num_alberi_popolazione = (num_alberi_per_bacino[id_bacino] * 100) / popolazione_per_bacino[id_bacino]
                num_alberi_popolazione = round(num_alberi_popolazione, 2)  # Arrotonda alla seconda cifra decimale

            # Crea una feature per il layer
            feature = QgsFeature()
            feature.setAttributes([id_bacino, vuf, num_alberi_popolazione])
            provider.addFeature(feature)

        # Aggiungi il layer al progetto
        QgsProject.instance().addMapLayer(new_layer)

        # Chiude il popup e mostra un messaggio con il pulsante OK
        progress.close()
        QtWidgets.QMessageBox.information(self, "Calcolo completato",
                                          "Il calcolo degli indicatori del verde urbano è stato completato con successo!")

    def calculate_social_indicators(self):
        """Calcola solo gli indicatori sociali selezionati nelle checkbox, mantenendo le formule corrette."""

        # Controlla quali checkbox sono selezionate
        calculate_parcheggi = self.ParcheggiAbitanti_checkBox.isChecked()
        calculate_nuclei_familiari = self.NumeroNucleiFamiliari_checkBox.isChecked()
        calculate_persone_unita = self.PersonePerUnitaAbitativa_checkBox.isChecked()
        calculate_spazio_pubblico_coperto = self.SpazioPubblicoCopertoAbitanti_checkBox.isChecked()
        calculate_spazio_pubblico_aperto = self.SpazioPubblicoApertoAbitanti_checkBox.isChecked()

        if not (calculate_parcheggi or calculate_nuclei_familiari or calculate_persone_unita or
                calculate_spazio_pubblico_coperto or calculate_spazio_pubblico_aperto):
            QtWidgets.QMessageBox.warning(self, "Errore", "Seleziona almeno un indicatore da calcolare.")
            return

        # Recupera gli ID Bacino da considerare
        selected_basin_ids = self.get_selected_basin_ids()
        if selected_basin_ids is None:
            return

        # Mostra il popup di caricamento
        progress = QProgressDialog("Calcolo in corso...", None, 0, 0, self)
        progress.setWindowTitle("Calcolo Indicatori Sociali")
        progress.setWindowModality(Qt.ApplicationModal)
        progress.show()

        # Recupera gli attributi
        id_bacino_parcheggi_attr = self.IDBacinoParcheggi_comboBox.currentText()
        id_bacino_edifici_attr = self.IDBacinoEdifici_comboBox.currentText()
        id_bacino_vegetazione_attr = self.IDBacinoVegetazione_comboBox.currentText()
        id_bacino_superfici_attr = self.IDBacinoSuperfici_comboBox.currentText()

        area_parcheggi_attr = self.AreaParcheggi_comboBox.currentText()
        pop_residente_attr = self.PopolazioneResidenteEdifici_comboBox.currentText()
        famiglie_attr = self.Famiglie_comboBox.currentText()
        area_edifici_attr = self.AreaEdifici_comboBox.currentText()
        proprieta_edifici_attr = self.ProprietaEdifici_comboBox.currentText()
        area_vegetazione_attr = self.AreaVegetazione_comboBox.currentText()
        area_superfici_attr = self.AreaSuperfici_comboBox.currentText()

        # Recupera i layer
        layer_parcheggi = QgsProject.instance().mapLayersByName(self.LayerParcheggi_comboBox.currentText())[0]
        layer_edifici = QgsProject.instance().mapLayersByName(self.LayerEdifici_comboBox.currentText())[0]
        layer_vegetazione = QgsProject.instance().mapLayersByName(self.LayerVegetazione_comboBox.currentText())[0]
        layer_superfici = QgsProject.instance().mapLayersByName(self.LayerSuperfici_comboBox.currentText())[0]

        # Dizionari per aggregare i dati per ID Bacino
        parcheggi_per_bacino = {}
        popolazione_per_bacino = {}
        nuclei_familiari_per_bacino = {}
        edifici_pubblici_per_bacino = {}
        area_pubblica_aperta_per_bacino = {}

        # Aggrega i dati per ID Bacino
        for feature in layer_parcheggi.getFeatures():
            id_bacino = feature[id_bacino_parcheggi_attr]
            if self.is_valid_feature(feature, id_bacino_parcheggi_attr, selected_basin_ids):
                area_parcheggi = float(feature[area_parcheggi_attr] or 0)
                parcheggi_per_bacino[id_bacino] = parcheggi_per_bacino.get(id_bacino, 0) + area_parcheggi

        for feature in layer_edifici.getFeatures():
            id_bacino = feature[id_bacino_edifici_attr]
            if not self.is_valid_feature(feature, id_bacino_edifici_attr, selected_basin_ids):
                continue

            popolazione = float(feature[pop_residente_attr] or 0)
            nuclei_familiari = max(1, int(feature[famiglie_attr] or 0))  # Evita divisione per zero
            proprieta = feature[proprieta_edifici_attr].strip().lower()

            popolazione_per_bacino[id_bacino] = popolazione_per_bacino.get(id_bacino, 0) + popolazione
            nuclei_familiari_per_bacino[id_bacino] = nuclei_familiari_per_bacino.get(id_bacino, 0) + nuclei_familiari

            if proprieta in ["pubblica", "public"]:
                area_edificio = float(feature[area_edifici_attr] or 0)
                edifici_pubblici_per_bacino[id_bacino] = edifici_pubblici_per_bacino.get(id_bacino, 0) + area_edificio

        for feature in layer_vegetazione.getFeatures():
            id_bacino = feature[id_bacino_vegetazione_attr]
            if self.is_valid_feature(feature, id_bacino_vegetazione_attr, selected_basin_ids):
                area_vegetazione = float(feature[area_vegetazione_attr] or 0)
                area_pubblica_aperta_per_bacino[id_bacino] = area_pubblica_aperta_per_bacino.get(id_bacino,
                                                                                                 0) + area_vegetazione

        for feature in layer_superfici.getFeatures():
            id_bacino = feature[id_bacino_superfici_attr]
            if self.is_valid_feature(feature, id_bacino_superfici_attr, selected_basin_ids):
                area_superfici = float(feature[area_superfici_attr] or 0)
                area_pubblica_aperta_per_bacino[id_bacino] = area_pubblica_aperta_per_bacino.get(id_bacino,
                                                                                                 0) + area_superfici

        # Creazione del layer di output
        new_layer = QgsVectorLayer("Polygon?crs=EPSG:4326", "Indicatori Sociali", "memory")
        provider = new_layer.dataProvider()
        fields = QgsFields()
        fields.append(QgsField("ID_Bacino", QVariant.String))

        if calculate_parcheggi:
            fields.append(QgsField("Parcheggi", QVariant.Double))
        if calculate_nuclei_familiari or calculate_persone_unita:
            fields.append(QgsField("Nuclei_Familiari", QVariant.Int))
        if calculate_persone_unita:
            fields.append(QgsField("Persone_Unita_Abitativa", QVariant.Double))
        if calculate_spazio_pubblico_coperto:
            fields.append(QgsField("Spazio_Pub_Coperto", QVariant.Double))
        if calculate_spazio_pubblico_aperto:
            fields.append(QgsField("Spazio_Pub_Aperto", QVariant.Double))

        provider.addAttributes(fields)
        new_layer.updateFields()

        # Calcola i valori finali per ogni ID Bacino e crea una feature per ciascuno
        for id_bacino in nuclei_familiari_per_bacino.keys():
            pop = max(1, popolazione_per_bacino.get(id_bacino, 1))  # Evita divisione per zero
            attributes = [id_bacino]

            if calculate_parcheggi:
                attributes.append(round(parcheggi_per_bacino.get(id_bacino, 0) / pop, 2))
            if calculate_nuclei_familiari or calculate_persone_unita:
                attributes.append(nuclei_familiari_per_bacino.get(id_bacino, 0))
            if calculate_persone_unita:
                attributes.append(round(pop / max(1, nuclei_familiari_per_bacino.get(id_bacino, 1)), 2))
            if calculate_spazio_pubblico_coperto:
                attributes.append(round(edifici_pubblici_per_bacino.get(id_bacino, 0) / pop, 2))
            if calculate_spazio_pubblico_aperto:
                attributes.append(round(area_pubblica_aperta_per_bacino.get(id_bacino, 0) / pop, 2))

            feature = QgsFeature()
            feature.setAttributes(attributes)
            provider.addFeature(feature)

        QgsProject.instance().addMapLayer(new_layer)
        progress.close()
        QtWidgets.QMessageBox.information(self, "Calcolo completato",
                                          "Il calcolo degli indicatori sociali è stato completato con successo!")

    def apply_graduated_symbology(self, layer, field_name, color_ranges=None, pattern=False):
        """Applica una simbologia graduata al layer basata su un campo specifico."""
        ranges = []

        # Se non vengono forniti color_ranges, usa i default basati sul campo
        if color_ranges is None:
            if field_name == "Indice di Permeabilità":
                color_ranges = [
                    (0, 0.25, "#b2e2a8"),  # Verde chiaro
                    (0.25, 0.5, "#66c2a5"),  # Verde medio
                    (0.5, 0.75, "#2ca25f"),  # Verde più scuro
                    (0.75, 1.0, "#006d2c")  # Verde molto scuro
                ]
            elif field_name == "BAF":
                color_ranges = [
                    (0, 0.3, "#deebf7"),  # Blu chiaro
                    (0.3, 0.5, "#9ecae1"),  # Blu medio
                    (0.5, 0.7, "#3182bd"),  # Blu più scuro
                    (0.7, 1.0, "#08519c")  # Blu molto scuro
                ]
            elif field_name == "RIE":
                color_ranges = [
                    (0, 1.5, "#feedde"),  # Arancione chiaro
                    (1.5, 2.5, "#fdbe85"),  # Arancione medio
                    (2.5, 4.0, "#fd8d3c"),  # Arancione più scuro
                    (4.0, 6.0, "#e6550d"),  # Arancione scuro
                    (6.0, 10.0, "#a63603")  # Arancione molto scuro
                ]
            else:
                return  # Esci se il campo non è supportato

        # Creazione dei range con conversione in QColor
        for lower, upper, color in color_ranges:
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            symbol.setColor(QColor(color))  # ✅ Converti stringa colore in QColor
            label = f"{lower} - {upper}"
            range_item = QgsRendererRange(lower, upper, symbol, label)
            ranges.append(range_item)

        # Crea il renderer graduato
        renderer = QgsGraduatedSymbolRenderer(field_name, ranges)
        renderer.setMode(QgsGraduatedSymbolRenderer.EqualInterval)  # Modalità intervalli uguali
        layer.setRenderer(renderer)

        # Aggiorna il layer
        layer.triggerRepaint()

    def initialize_lotto_entry(self):
        """Crea una nuova entry con valori inizializzati a zero."""
        return {
            "IPerm": None, "RIE": None, "BAF": None,
            "area_vegetazione": 0, "area_superfici": 0, "area_edifici": 0,
            "weighted_baf": 0, "weighted_rie": 0, "tree_area": 0, "denominator_rie": 0
        }

    def get_material_coefficients(self, material, category, slope=None):
        """Restituisce i coefficienti Pi per BAF e RIE in base alla categoria di materiale."""
        material = material.lower()

        if category == "superfici":
            mapping = {
                "pavimentazione in cubetti o pietre a fuga non sigillata su sabbia": (0.1, 0.7),
                "pavimentazione in cubetti, pietre o lastre a fuga sigillata": (0.0, 0.8),
                "asfalto": (0.0, 0.9),
                "cemento": (0.0, 0.9)
            }
        elif category == "vegetazione":
            mapping = {
                "area di impianto sportivo con sistemi drenanti e superficie a prato": (0.4, 0.6),
                "incolto": (1.0, 0.2),
                "giardini, aree verdi, prati, orti, superfici boscate ed agricole": (1.0, 0.1)
            }
        elif category == "edifici":
            mapping = {
                "alluminium": (0.0, 0.95 if slope >= 3 else 0.9),
                "glass": (0.0, 0.9),
                "plaster": (0.0, 0.9 if slope >= 3 else 0.85),
                "tile": (0.0, 0.9)
            }
        else:
            mapping = {}

        return mapping.get(material, (0.0, 0.0))

    def calculate_tree_area(self, height):
        """Restituisce l'area equivalente dell'albero in base all'altezza."""
        return 20 if 4 <= height <= 12 else 65 if 12 < height <= 18 else 115 if height > 18 else 0

    def create_new_layer(self, layer_name, attributes, formula_results):
            """
            Crea un nuovo layer vettoriale con attributi specifici e valori calcolati.
            """
            layer = QgsVectorLayer("Point?crs=EPSG:4326", layer_name, "memory")
            provider = layer.dataProvider()

            fields = QgsFields()
            for attr in attributes:
                field_type = QVariant.Double if "Indice" in attr or attr in ["RIE", "BAF"] else QVariant.String
                fields.append(QgsField(attr, field_type))

            provider.addAttributes(fields)
            layer.updateFields()

            for row in formula_results:
                feature = QgsFeature()
                feature.setGeometry(None)
                feature.setAttributes(row)
                provider.addFeature(feature)

            QgsProject.instance().addMapLayer(layer)

    def generate_lot_geometries(self):
        """Crea geometrie aggregate per ciascun lotto dai layer selezionati."""
        # Recupera i layer selezionati e gli attributi ID lotto
        layer_superfici = QgsProject.instance().mapLayersByName(self.LayerSuperfici_comboBox.currentText())[0]
        layer_vegetazione = QgsProject.instance().mapLayersByName(self.LayerVegetazione_comboBox.currentText())[0]
        layer_edifici = QgsProject.instance().mapLayersByName(self.LayerEdifici_comboBox.currentText())[0]

        id_lotto_superfici_attr = self.IDLottoSuperfici_comboBox.currentText()
        id_lotto_vegetazione_attr = self.IDLottoVegetazione_comboBox.currentText()
        id_lotto_edifici_attr = self.IDLottoEdifici_comboBox.currentText()

        # Dizionario per memorizzare le geometrie aggregate per lotto
        lot_geometries = {}

        # Aggiungi geometrie dal layer delle superfici
        for feature in layer_superfici.getFeatures():
            id_lotto = feature[id_lotto_superfici_attr]
            geometry = feature.geometry()
            if id_lotto not in lot_geometries:
                lot_geometries[id_lotto] = geometry
            else:
                lot_geometries[id_lotto] = lot_geometries[id_lotto].combine(geometry)

        # Aggiungi geometrie dal layer della vegetazione
        for feature in layer_vegetazione.getFeatures():
            id_lotto = feature[id_lotto_vegetazione_attr]
            geometry = feature.geometry()
            if id_lotto not in lot_geometries:
                lot_geometries[id_lotto] = geometry
            else:
                lot_geometries[id_lotto] = lot_geometries[id_lotto].combine(geometry)

        # Aggiungi geometrie dal layer degli edifici
        for feature in layer_edifici.getFeatures():
            id_lotto = feature[id_lotto_edifici_attr]
            geometry = feature.geometry()
            if id_lotto not in lot_geometries:
                lot_geometries[id_lotto] = geometry
            else:
                lot_geometries[id_lotto] = lot_geometries[id_lotto].combine(geometry)

        return lot_geometries