# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LocalSoftware
                                 A QGIS plugin
 Connecting CAD and GIS software
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-10-12
        git sha              : $Format:%H$
        copyright            : (C) 2021 by LocalSoftware
        email                : csandova@mit.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from pathlib import Path
import json
import glob
from collections import OrderedDict
import numbers

from qgis.core import QgsVectorLayer, QgsProject, QgsJsonUtils, QgsFeature, QgsField, QgsCoordinateTransform, QgsFeatureRequest, QgsCoordinateReferenceSystem, QgsGeometry, QgsMessageLog
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.utils import iface
import processing
# from processing.core.Processing import Processing
# Processing.initialize()


# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .local_software_dialog import LocalSoftwareDialog
import os.path



class LocalSoftware:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'LocalSoftware_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&LocalSoftware')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('LocalSoftware', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/local_software/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Local Software'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&LocalSoftware'),
                action)
            self.iface.removeToolBarIcon(action)

    def reproject_layer(self, layer, epsg):
        parameter = {'INPUT': layer, 'TARGET_CRS': epsg.authid(),
                        'OUTPUT': 'memory:Reprojected'}

        new_layer = processing.run('native:reprojectlayer', parameter)['OUTPUT']
        new_layer.setName(layer.name())
        return new_layer
        
    def create_buffer(self, site, crs, radius):
        transformContext = QgsProject.instance().transformContext()
        epsg = QgsCoordinateReferenceSystem(3857)
        xform = QgsCoordinateTransform(crs, epsg, transformContext)
        geom2 = QgsGeometry(site.geometry())
        geom2.transform(xform)
        old_buffer = geom2.buffer(radius, 5)
        
        xform = QgsCoordinateTransform(epsg, crs, transformContext)
        buffer = QgsGeometry(old_buffer)
        buffer.transform(xform)

        return buffer    
        
    def spatial_query(self, site_geom, other_layer):
        # parameters = { 'INPUT' : layers[0], 
        #            'INTERSECT' : layers[1], 
        #            'METHOD' : 0, 
        #            'PREDICATE' : [0] }

        # processing.run('qgis:selectbylocation', parameters )

        queried_features = []
        #performance boost: get point features by poly bounding box first
        sel_features = other_layer.getFeatures(QgsFeatureRequest().setFilterRect(site_geom.boundingBox()))
        for feat in sel_features:
            #iterate preselected point features and perform exact check with current polygon
            if feat.geometry().intersects(site_geom):
                queried_features.append(feat)
        return queried_features

    def features_to_json(self, layer, site_features, fields, epsg): 
        features = []
        for i in site_features:
            curr_props = []
            for ix, val in enumerate(i.attributes()):
                if ix < len(fields):
                    if not isinstance(val, numbers.Number): curr_props.append((fields[ix], str(val)))
                    else: curr_props.append((fields[ix], val))
            prop_dict = OrderedDict(curr_props)
            features.append(
                {
                    "type": "Feature",
                    "properties": prop_dict,
                    "geometry": json.loads(i.geometry().asJson())
                }
            )

        return features

    def export_layer(self, layer_name, site_features, site_epsg, site_name='site'):
        if isinstance(layer_name, str):
            layer = QgsProject.instance().mapLayersByName(layer_name)[0]
            if layer_name == site_name: site = True
            else: site = False
        else:
            layer = layer_name
            if layer_name.name() == site_name: site = True
            else: site = False

        # Get the provider
        prov = layer.dataProvider()
        # Get fields names with the order
        fields = [field.name() for field in prov.fields()]
        feats = self.features_to_json(layer, site_features, fields, site_epsg)

        if len(feats) > 0:
            # loop to generate GeoJSON looping layer features properties and getting geometry as GeoJSON geometry format
            geojson = {
                "type": "FeatureCollection",
                "features": feats,
                "crs": {
                    "properties": {
                        "name": site_epsg.authid()
                    },
                    "type": "name"
                }
            }
        else: geojson = None 

        return site, geojson
        
    def site_json(self, layer, is_site, geojson):
        return {
                "name": layer,
                "site": is_site,
                "otherSite": False,
                "contents": geojson,
                "type": "Layer"
            }

    def export_site(self, layers, data_folder, site_id):
        export_geojson = {
            "layers": layers,
            "type": "LayerCollection"
        }
        file_name = str(site_id) + '.json'
        # If you want to write to an external file
        with open(data_folder+'/'+file_name, 'w') as outfile:
            # print(export_geojson)
            json.dump(export_geojson, outfile)

    def make_sites(self, site_name, data_folder, radius):
        site_layer = QgsProject.instance().mapLayersByName(site_name)[0]

        site_epsg = site_layer.crs()
        other_layers = [layer for layer_name, layer in QgsProject.instance().mapLayers().items() if layer.name() != site_name]

        reprojected_layers = []
        for other_layer in other_layers:
            if site_epsg.authid() != other_layer.crs().authid():
                reprojected_layers.append(self.reproject_layer(other_layer, site_epsg))
            else: reprojected_layers.append(other_layer)

        for site in site_layer.getFeatures():
            new_layers = []
            site_id = site.id()
            
            is_site, geojson = self.export_layer(site_layer, [site], site_epsg, site_name)
            new_layers.append(self.site_json('site', is_site, geojson))

            site_buffer = self.create_buffer(site, site_epsg, radius)

            # Now query new layers, and add them to the site geoJSON
            for other_layer in reprojected_layers:
                layer_features = self.spatial_query(site_buffer, other_layer)
                is_site, geojson = self.export_layer(other_layer, layer_features, site_epsg, site_name)
                if geojson != None:
                    new_layers.append(self.site_json(other_layer.name(), is_site, geojson))

            # export the site as a geoJSON 
            self.export_site(new_layers, data_folder, site_id)
        iface.messageBar().pushMessage("Success", "Your sites packages have been created!", level=3)

    def batch_export(self, data_folder, radius=10):
        # Load the layers that were originally imported
        rel_layer = QgsProject.instance().mapLayersByName('relationship_layer')[0]
        idx = rel_layer.fields().indexOf('layerName')
        layer_names = rel_layer.uniqueValues(idx)

        layers = QgsProject.instance().mapLayers()
        site_layers = [QgsProject.instance().mapLayersByName(layer_name)[0] for layer_name in layer_names]
        site_epsg = [layer.crs() for layer in layers.values() if layer.name() == 'site'][0]
        site_epsg = QgsCoordinateReferenceSystem("EPSG:3857")

        other_layers = [layer for layer in list(set(layers.values())-set(site_layers)) if layer.name() != 'relationship_layer']
        reprojected_layers = []
        for other_layer in other_layers:
            if site_epsg.authid() != other_layer.crs().authid():
                reprojected_layers.append(self.reproject_layer(other_layer, site_epsg))
            else: reprojected_layers.append(other_layer)

        site_layer = QgsProject.instance().mapLayersByName('site')[0]
        for site in site_layer.getFeatures():
            new_layers = []
            site_id = site.id()
            is_site, geojson = self.export_layer(site_layer, [site], site_epsg)
            new_layers.append(self.site_json('site', is_site, geojson))

            for layer in layer_names: 
                if layer != 'site':
                    expression = '"siteID" = ' + str(site_id) + ' and "layerName" = \'' + layer + '\''
                    site_features = rel_layer.getFeatures(expression)
                    selected_ids = [i.attributes()[2] for i in site_features]

                    layer_features = QgsProject.instance().mapLayersByName(layer)[0].getFeatures(selected_ids)
                    is_site, geojson = self.export_layer(layer, layer_features, site_epsg)
                    new_layers.append(self.site_json(layer, is_site, geojson))

            if len(reprojected_layers) > 0:
                # Create a buffer for the site query 
                site_buffer = self.create_buffer(site, site_epsg, radius)

                # Now query new layers, and add them to the site geoJSON
                for other_layer in reprojected_layers:
                    layer_features = self.spatial_query(site_buffer, other_layer)
                    is_site, geojson = self.export_layer(other_layer, layer_features, site_epsg)
                    if geojson != None:
                        new_layers.append(self.site_json(other_layer.name(), is_site, geojson))

            # export the site as a geoJSON 
            self.export_site(new_layers, data_folder, site_id)
        iface.messageBar().pushMessage("Success", "Your layers have been exported to as new site packages!", level=3)

    def batch_import(self, file_list):
        # Before I start looping through all the files, I create an empty layer for the relationships to add
        relationship_layer = QgsVectorLayer('Polygon', 'relationship_layer', "memory")
        relationship_layer_pr = relationship_layer.dataProvider()

        # add fields
        relationship_layer_pr.addAttributes([QgsField("siteId", QVariant.Int), QgsField("layerName",  QVariant.String), QgsField("featureId", QVariant.Int)])
        relationship_layer.updateFields() # tell the vector layer to fetch changes from the provider


        # Opening JSON file
        for site_number, file in enumerate(file_list):
            f = open(file)
            
            # returns JSON object as
            # a dictionary
            geojson = json.load(f)
            for geojson_layer in geojson['layers']:
                if len(geojson_layer['contents']['features']) > 0:
                    if geojson_layer['site'] == True: layer_name = 'site'
                    else: layer_name = geojson_layer['name']
                    geometry_type = geojson_layer['contents']['features'][0]['geometry']['type']

                    # if len(QgsProject.instance().mapLayersByName(layer_name)) != 0:
                    #     QgsProject.instance().removeMapLayers( [QgsProject.instance().mapLayersByName(layer_name)[0].id()])
                    #     iface.messageBar().pushMessage("Warning", "Some of your layers already exist in the project and were overwritten!", level=1)

                    curr_layers = QgsProject.instance().mapLayersByName(layer_name)
                    if len(curr_layers) != 0:
                        layer = curr_layers[0]
                    else:
                        try: layer = QgsVectorLayer(geometry_type, layer_name, "memory")
                        except: iface.messageBar().pushMessage("Error", "Your layer's geometry is invalid. Make sure your layer's geoemetries are vectors", level=2)
                    pr = layer.dataProvider()

                    crs = layer.crs()
                    crs.createFromId(int(geojson_layer['contents']['crs']['properties']['name'].split(':')[1]))
                    try: layer.setCrs(crs)
                    except: iface.messageBar().pushMessage("Error", "Your layer's EPSG code is invalid, we cannot import it!", level=2)


                    # add fields
                    try:
                        fields = QgsJsonUtils.stringToFields(json.dumps(geojson_layer['contents']['features'][0]))
                        pr.addAttributes(fields)
                        layer.updateFields() # tell the vector layer to fetch changes from the provider
                    except: pass

                    # try: 
                    #     curr_field = fields.field(fields.indexOf('created_da'))
                    #     print(curr_field.type(), curr_field.subType())
                    #     print(curr_field.typeName(), curr_field.constraints().constraintDescription())
                    # except: pass
                    pr.addAttributes(fields)
                    layer.updateFields() # tell the vector layer to fetch changes from the provider


                    # Enter editing mode
                    layer.startEditing()
                    for feature in geojson_layer['contents']['features']:
                        try:
                            # Create QGIS feature from a geoJSON object
                            qgis_feature = QgsJsonUtils.stringToFeatureList(json.dumps(feature))[0]

                            # Set the fields of the feature to the layer's fields
                            qgis_feature.setFields(fields)
                            # Set the properties of each feature by looping through all the keys of the JSON properties
                            try:
                                for key in feature['properties']:
                                    prop = feature['properties'][key]
                                    if type(prop) is dict: prop = json.dumps(prop)
                                    qgis_feature.setAttribute(key, prop)
                                #    print(key, feature_dict['properties'][key])
                            except: pass

                            # # Set the properties of each feature by looping through all the keys of the JSON properties
                            # for key in feature['properties']:
                            #     prop = feature['properties'][key]
                            #     if type(prop) is dict: prop = json.dumps(prop)
                            #     qgis_feature.setAttribute(key, prop)
                            
                            # Add the features to the layer
                            (result, newFeatures) = pr.addFeatures([qgis_feature])
                            # then we will add a new row with site_number (we need to add +1 to site number), layer_name, and featureID to the empty layer
                            fet = QgsFeature()
                            fet.setAttributes([site_number+1, layer_name, newFeatures[0].id()])
                            relationship_layer_pr.addFeatures([fet])
                            # print(site_number, newFeatures[0].id(), layer_name)

                            layer.commitChanges()
                        except:
                            iface.messageBar().pushMessage("Error", "Your layer contains some invalid geometries", level=2)
                    layer.updateExtents()
                    relationship_layer.updateExtents()


                    # Show in project if the layer is not already in the project
                    if len(curr_layers) == 0:
                        QgsProject.instance().addMapLayer(layer)
                    iface.messageBar().pushMessage("Success", "Your layer has been added to the model!", level=3)

        QgsProject.instance().addMapLayer(relationship_layer)
        iface.messageBar().pushMessage("Success", "Your Site Packages have been imported to the model!", level=3)

    def select_folder_path(self):

        filename = QFileDialog.getExistingDirectory(self.dlg, "Select you Firebase Credential JSON file")

        if self.dlg.tabWidget.currentIndex() == 0:
            self.dlg.folder_path_2.setText(filename)
        if self.dlg.tabWidget.currentIndex() == 1:
            self.dlg.folder_path_3.setText(filename)
        if self.dlg.tabWidget.currentIndex() == 2:
            self.dlg.folder_path_4.setText(filename)

    def push_warning(self, warning_text):
        iface.messageBar().pushMessage("Error", warning_text, level=2)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = LocalSoftwareDialog()
            self.dlg.push_button_path_2.clicked.connect(self.select_folder_path)
            self.dlg.push_button_path_3.clicked.connect(self.select_folder_path)
            self.dlg.push_button_path_4.clicked.connect(self.select_folder_path)
            # print(self.dlg.layer_name_label)

        # Fetch the currently loaded layers
        layers = [layer for layer in QgsProject.instance().layerTreeRoot().children() if layer.name() != 'relationship_layer']
        # Clear the contents of the comboBox from previous runs
        self.dlg.comboBox_2.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.comboBox_2.addItems([layer.name() for layer in layers])

        # Clear the contents of the comboBox from previous runs
        self.dlg.units_2.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.units_2.addItems(['mts', 'ft'])

        self.dlg.units_3.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.units_3.addItems(['mts', 'ft'])

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()

        # Package Sites
        if self.dlg.tabWidget.currentIndex() == 0:
            data_folder = self.dlg.folder_path_2.text()
            if (result and data_folder != '') and len(layers) > 0:
                package_radius = int(self.dlg.radius_2.text())

                package_selected_unit_index = self.dlg.units_2.currentIndex()
                package_unit_name = ['mt','ft'][package_selected_unit_index]
                if package_unit_name == 'ft': package_radius *= 0.3048

                selectedLayerIndex = self.dlg.comboBox_2.currentIndex()
                site_name = layers[selectedLayerIndex].layer().name()
                self.make_sites(site_name, data_folder, package_radius)
            elif data_folder == '' and len(layers) == 0:
                self.push_warning('Please select an output folder and add layers to your model')
            elif data_folder == '':
                self.push_warning('Please select an output folder')
            elif len(layers) == 0:
                self.push_warning("Your model doesn't have any layers, please add at least one layer")

        # Import Sites
        elif self.dlg.tabWidget.currentIndex() == 1:
            data_folder = self.dlg.folder_path_3.text()
            if result and data_folder != '':
                file_list = glob.glob(data_folder+"/*.json") #2
                self.batch_import(file_list)   
            elif data_folder == '':
                self.push_warning('Please select an output folder')

        # Export Sites
        elif self.dlg.tabWidget.currentIndex() == 2:
            data_folder = self.dlg.folder_path_4.text()
            if result and data_folder != '':
                export_radius = int(self.dlg.radius_3.text())

                export_selected_unit_index = self.dlg.units_3.currentIndex()
                export_unit_name = ['mt','ft'][export_selected_unit_index]
                if export_unit_name == 'ft': export_radius *= 0.3048

                if export_radius != 0: self.batch_export(data_folder, export_radius)
                else: self.batch_export(data_folder)
            elif data_folder == '':
                self.push_warning('Please select an output folder')

        self.dlg.folder_path_2.clear()
        self.dlg.folder_path_3.clear()
        self.dlg.folder_path_4.clear()
        self.dlg.radius_2.clear()
        self.dlg.radius_3.clear()
