"""
Linkscape Corridor Analysis - Vector Workflow (v23.3)
----------------------------------------------------
Runs the single-strategy corridor optimization workflow for polygon
patch datasets.

Updates in v23.3:
- Fixed Redundancy: Allows parallel connections between components if budget permits.
- Fixed Traversal: Efficient spatial indexing for detecting intermediate patch crossings.
- Fixed Logic: Corridors crossing intermediate patches (A->C->B) are now prioritized
  if C is not yet connected, even if A and B are.
"""

from __future__ import annotations

import heapq
import math
import os
import time
from contextlib import contextmanager, nullcontext
from collections import defaultdict, deque
from dataclasses import dataclass, field
from typing import Callable, Dict, List, Optional, Set, Tuple

import numpy as np
# NumPy 2.x removed np.int; add shim for any legacy references.
if not hasattr(np, "int"):  # pragma: no cover
    np.int = int  # type: ignore[attr-defined]

import networkx as nx  # Required for HDFM
from .linkscape_engine import NetworkOptimizer, UnionFind
from .utils import emit_progress, log_error
# Import the new HDFM math library
try:
    from . import hdfm_math
except ImportError:
    hdfm_math = None
from PyQt5.QtCore import QVariant, QUrl
from PyQt5.QtGui import QDesktopServices
import random
from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsFeature,
    QgsField,
    QgsFields,
    QgsGeometry,
    QgsPointXY,
    QgsProject,
    QgsRectangle,
    QgsSpatialIndex,
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsWkbTypes,
)

BUFFER_SEGMENTS = 16


def _log_message(message: str, level: int = Qgis.Info) -> None:
    """Log to the QGIS Log Messages Panel with a Linkscape tag."""
    try:
        QgsApplication.messageLog().logMessage(message, "Linkscape", level)
    except Exception:
        # Fallback for environments where the message log is unavailable
        print(f"Linkscape Log: {message}")


def _open_path_in_default_app(path: str) -> None:
    """Attempt to open a local file in the OS-default handler."""
    try:
        url = QUrl.fromLocalFile(path)
        if not url.isValid():
            return
        opened = QDesktopServices.openUrl(url)
        if not opened:
            print(f"  ⚠ Could not auto-open file: {path}")
    except Exception as exc:  # noqa: BLE001
        print(f"  ⚠ Could not auto-open file: {exc}")


def clone_geometry(geom: QgsGeometry) -> QgsGeometry:
    """
    Lightweight copy helper; QgsGeometry uses implicit sharing so this is cheap
    and avoids unnecessary deep clones unless a write occurs.
    """
    return QgsGeometry(geom)


class VectorAnalysisError(RuntimeError):
    """Raised when the vector analysis cannot be completed."""


class _TimingBlock:
    """Context manager that records elapsed time for a named step."""

    def __init__(self, label: str, sink: List[Dict[str, float]]):
        self.label = label
        self.sink = sink
        self._start = 0.0

    def __enter__(self) -> None:
        self._start = time.perf_counter()
        return None

    def __exit__(self, exc_type, exc, tb) -> bool:
        duration = time.perf_counter() - self._start
        self.sink.append({"label": self.label, "duration_s": duration})
        return False


class TimingRecorder:
    """Lightweight helper to track fine-grained step timings."""

    def __init__(self) -> None:
        self.records: List[Dict[str, float]] = []

    def time_block(self, label: str) -> _TimingBlock:
        return _TimingBlock(label, self.records)

    def add(self, label: str, duration: float) -> None:
        self.records.append({"label": label, "duration_s": duration})

    def write_report(self, path: str, total_elapsed: Optional[float] = None) -> None:
        try:
            with open(path, "w", encoding="utf-8") as fh:
                fh.write("Linkscape Vector Timing\n")
                fh.write("=" * 30 + "\n")
                for entry in self.records:
                    fh.write(f"{entry['label']}: {entry['duration_s']:.3f}s\n")
                if total_elapsed is not None:
                    fh.write("\n")
                    fh.write(f"Total wall time: {total_elapsed:.3f}s\n")
            print(f"  ✓ Timing report saved: {path}")
        except Exception as exc:  # noqa: BLE001
            print(f"  ⚠ Could not write timing report: {exc}")


@dataclass
class VectorRunParams:
    min_corridor_width: float  # metres
    max_corridor_area: Optional[float]  # hectares
    min_patch_size: float  # hectares
    budget_area: float  # hectares
    max_search_distance: float  # metres
    unit_system: str
    output_name: str
    grid_resolution: float  # metres
    stepping_enabled: bool = False
    merge_distance: float = 0.0  # metres (virtual merge / functional connectivity)
    hop_distance: float = 0.0  # metres (max impassable hop / hole-to-hole hop)
    obstacle_layer_ids: List[str] = field(default_factory=list)
    obstacle_enabled: bool = False
    debug_enabled: bool = False
    debug_lines: List[str] = field(default_factory=list, repr=False)


def _to_dataclass(params: Dict) -> VectorRunParams:
    output_name = params.get("output_name") or "linkscape_corridors.gpkg"
    if not output_name.lower().endswith(".gpkg"):
        output_name = f"{output_name}.gpkg"
    obstacle_ids_raw = params.get("obstacle_layer_ids") or []
    if not obstacle_ids_raw and params.get("obstacle_layer_id"):
        obstacle_ids_raw = [params.get("obstacle_layer_id")]
    obstacle_ids = [str(val) for val in obstacle_ids_raw if val]
    stepping_enabled = bool(params.get("stepping_enabled", False))
    hop_distance_raw = float(params.get("hop_distance", 0.0) or 0.0)
    merge_distance_raw = float(params.get("merge_distance", hop_distance_raw) or hop_distance_raw or 0.0)
    obstacle_flag = bool(params.get("obstacle_enabled", False) and obstacle_ids)
    debug_enabled = bool(params.get("debug_enabled", False))
    if not debug_enabled:
        env_flag = os.getenv("LINKSCAPE_DEBUG") or os.getenv("LINKSCAPE_DEBUG_VECTOR")
        debug_enabled = str(env_flag).strip().lower() in ("1", "true", "yes", "on")
    debug_lines_init: List[str] = []

    max_search_distance_value = params.get("max_search_distance")
    if max_search_distance_value is None or str(max_search_distance_value).strip() == "":
        max_search_distance = 0.0
    else:
        try:
            max_search_distance = float(max_search_distance_value)
        except (TypeError, ValueError):
            max_search_distance = 0.0

    return VectorRunParams(
        min_corridor_width=float(params.get("min_corridor_width", 200.0)),
        max_corridor_area=(
            float(params["max_corridor_area"])
            if params.get("max_corridor_area") not in (None, 0, 0.0)
            else None
        ),
        min_patch_size=float(params.get("min_patch_size", 10.0)),
        budget_area=float(params.get("budget_area", 50.0)),
        max_search_distance=max_search_distance,
        unit_system=str(params.get("unit_system", "metric")),
        output_name=output_name,
        grid_resolution=max(float(params.get("grid_resolution", 50.0)), 1.0),
        stepping_enabled=bool(params.get("stepping_enabled", False)),
        merge_distance=float(params.get("merge_distance", merge_distance_raw) or merge_distance_raw or 0.0),
        hop_distance=float(params.get("hop_distance", 0.0) or 0.0),
        obstacle_layer_ids=obstacle_ids,
        obstacle_enabled=bool(params.get("obstacle_enabled", False) and obstacle_ids),
        debug_enabled=debug_enabled,
        debug_lines=debug_lines_init,
    )


def _debug(params: VectorRunParams, message: str) -> None:
    if not getattr(params, "debug_enabled", False):
        return
    try:
        params.debug_lines.append(message)
    except Exception:
        pass


def _count_interior_rings(geom: QgsGeometry) -> int:
    try:
        if geom.isMultipart():
            polys = geom.asMultiPolygon()
        else:
            polys = [geom.asPolygon()]
        rings = 0
        for poly in polys:
            if not poly:
                continue
            # poly[0] is exterior, poly[1:] are interior rings
            rings += max(len(poly) - 1, 0)
        return rings
    except Exception:
        return 0


def _extract_hole_geometries(geom: QgsGeometry) -> List[QgsGeometry]:
    """Return interior rings (holes) as polygon geometries."""
    holes: List[QgsGeometry] = []
    try:
        polys = geom.asMultiPolygon() if geom.isMultipart() else [geom.asPolygon()]
        for poly in polys:
            if not poly or len(poly) < 2:
                continue
            for ring in poly[1:]:
                if not ring:
                    continue
                hole_geom = QgsGeometry.fromPolygonXY([ring])
                if hole_geom is None or hole_geom.isEmpty():
                    continue
                hole_geom = hole_geom.makeValid()
                if hole_geom is None or hole_geom.isEmpty():
                    continue
                holes.append(clone_geometry(hole_geom))
    except Exception:
        return []
    return holes


def get_utm_crs_from_extent(layer: QgsVectorLayer) -> QgsCoordinateReferenceSystem:
    extent = layer.extent()
    center = extent.center()
    source_crs = layer.crs()

    if not source_crs.isGeographic():
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(source_crs, wgs84, QgsProject.instance())
        center = transform.transform(center)

    utm_zone = int((center.x() + 180) / 6) + 1
    epsg_code = 32600 + utm_zone if center.y() >= 0 else 32700 + utm_zone
    return QgsCoordinateReferenceSystem(f"EPSG:{epsg_code}")


def load_and_prepare_patches(
    layer: QgsVectorLayer,
    target_crs: QgsCoordinateReferenceSystem,
    params: VectorRunParams,
) -> Tuple[Dict[int, Dict], QgsSpatialIndex]:
    print("  Loading patches and building spatial index...")
    source_crs = layer.crs()
    transform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())

    patches: Dict[int, Dict] = {}
    patch_id = 1
    filtered_count = 0
    indexed_features: List[QgsFeature] = []

    for feature in layer.getFeatures():
        geom = QgsGeometry(feature.geometry())
        if geom.isEmpty():
            continue
        geom.transform(transform)
        try:
            geom = geom.makeValid()
        except Exception:
            pass
        area_ha = geom.area() / 10000.0

        if area_ha < params.min_patch_size:
            filtered_count += 1
            continue

        feat = QgsFeature()
        feat.setGeometry(geom)
        feat.setId(patch_id)
        indexed_features.append(feat)

        patches[patch_id] = {
            "geom": clone_geometry(geom),
            "area_ha": area_ha,
        }
        patch_id += 1

    spatial_index = QgsSpatialIndex(flags=QgsSpatialIndex.FlagStoreFeatureGeometries)
    if indexed_features:
        spatial_index.addFeatures(indexed_features)

    print(f"  ✓ Loaded {len(patches)} patches (filtered {filtered_count} too small)")
    return patches, spatial_index


def _compute_vector_hop_adjacency(
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    hop_distance: float,
) -> Dict[int, Set[int]]:
    if hop_distance <= 0:
        return {}
    adjacency: Dict[int, Set[int]] = defaultdict(set)
    for pid, pdata in patches.items():
        geom1 = pdata["geom"]
        rect = geom1.boundingBox()
        rect.grow(hop_distance)
        for other_id in spatial_index.intersects(rect):
            if other_id <= pid:
                continue
            pdata2 = patches.get(other_id)
            if not pdata2:
                continue
            geom2 = pdata2["geom"]
            try:
                distance = geom1.distance(geom2)
            except Exception:
                continue
            if distance <= hop_distance + 1e-9:
                adjacency[pid].add(other_id)
                adjacency[other_id].add(pid)
    return adjacency


def _build_virtual_components(
    patches: Dict[int, Dict],
    hop_adjacency: Dict[int, Set[int]],
) -> Tuple[Dict[int, int], Dict[int, float], Dict[int, int]]:
    uf = UnionFind()
    for pid, pdata in patches.items():
        uf.find(pid)
        uf.size[pid] = pdata["area_ha"]
        uf.count[pid] = 1

    for pid, neighbors in hop_adjacency.items():
        for nbr in neighbors:
            uf.union(pid, nbr)

    component_map: Dict[int, int] = {}
    component_sizes: Dict[int, float] = {}
    component_counts: Dict[int, int] = {}
    for pid, pdata in patches.items():
        comp = uf.find(pid)
        component_map[pid] = comp
        component_sizes[comp] = component_sizes.get(comp, 0.0) + pdata["area_ha"]
        component_counts[comp] = component_counts.get(comp, 0) + 1

    return component_map, component_sizes, component_counts


def _extract_obstacle_holes(
    obstacle_layers: List[QgsVectorLayer],
    target_crs: QgsCoordinateReferenceSystem,
    start_id: int = -1,
) -> Dict[int, Dict]:
    if not obstacle_layers:
        return {}
    print("  Extracting holes from impassable land classes to use as stepping stones...")
    hole_patches: Dict[int, Dict] = {}
    hole_id = start_id
    transform_map: Dict[str, QgsCoordinateTransform] = {}
    project = QgsProject.instance()

    for layer in obstacle_layers:
        if not isinstance(layer, QgsVectorLayer) or not layer.isValid():
            continue
        authid = layer.crs().authid()
        if authid not in transform_map:
            transform_map[authid] = QgsCoordinateTransform(layer.crs(), target_crs, project)
        transform = transform_map[authid]

        for feature in layer.getFeatures():
            geom = QgsGeometry(feature.geometry())
            if geom.isEmpty():
                continue
            geom = clone_geometry(geom)
            geom.transform(transform)
            geom = geom.makeValid()
            if geom.isEmpty():
                continue

            polygons = geom.asMultiPolygon() if geom.isMultipart() else [geom.asPolygon()]
            for poly in polygons:
                if len(poly) < 2:
                    continue
                for ring in poly[1:]:
                    if not ring:
                        continue
                    hole_geom = QgsGeometry.fromPolygonXY([ring])
                    if hole_geom.isEmpty() or hole_geom.area() <= 1.0:
                        continue
                    hole_geom = hole_geom.makeValid()
                    if hole_geom.isEmpty():
                        continue
                    hole_patches[hole_id] = {
                        "geom": clone_geometry(hole_geom),
                        "area_ha": hole_geom.area() / 10000.0,
                        "is_hole": True,
                    }
                    hole_id -= 1
    print(f"  ✓ Found {len(hole_patches)} usable holes in impassable land classes.")
    return hole_patches


def _generate_obstacle_hopping_candidates(
    real_patches: Dict[int, Dict],
    hole_patches: Dict[int, Dict],
    hop_distance: float,
    params: VectorRunParams,
    obstacle_geoms: Optional[List[QgsGeometry]] = None,
) -> List[Dict]:
    if hop_distance <= 0 or not hole_patches:
        return []
    print("  Building stepping-stone graph through impassable holes...")

    all_nodes: Dict[int, Dict] = {**real_patches, **hole_patches}
    node_index = QgsSpatialIndex()
    indexed_feats: List[QgsFeature] = []
    for nid, data in all_nodes.items():
        feat = QgsFeature()
        feat.setId(nid)
        geom = data.get("geom")
        if not geom or geom.isEmpty():
            continue
        feat.setGeometry(clone_geometry(geom))
        indexed_feats.append(feat)
    if indexed_feats:
        node_index.addFeatures(indexed_feats)

    G = nx.Graph()
    for nid in all_nodes:
        G.add_node(nid, is_real=(nid > 0))

    for nid1, data1 in all_nodes.items():
        geom1 = data1.get("geom")
        if not geom1 or geom1.isEmpty():
            continue
        rect = geom1.boundingBox()
        rect.grow(hop_distance * 1.05)
        for nid2 in node_index.intersects(rect):
            if nid2 <= nid1:
                continue
            data2 = all_nodes.get(nid2)
            if not data2:
                continue
            geom2 = data2.get("geom")
            if not geom2 or geom2.isEmpty():
                continue
            dist = geom1.distance(geom2)
            if dist <= hop_distance + 1e-6:
                G.add_edge(nid1, nid2, weight=dist)
                if params.debug_enabled:
                    _debug(
                        params,
                        f"Hop graph edge: {nid1} <-> {nid2} dist={dist:.2f}",
                    )

    real_pids = [pid for pid in real_patches.keys()]
    hopping_candidates: List[Dict] = []
    processed_pairs: Set[frozenset] = set()
    print(f"  Searching for paths through {G.number_of_nodes()} nodes and {G.number_of_edges()} hop edges...")

    for i, pid1 in enumerate(real_pids):
        for pid2 in real_pids[i + 1 :]:
            pair = frozenset({pid1, pid2})
            if pair in processed_pairs:
                continue
            try:
                path_nodes = nx.shortest_path(G, source=pid1, target=pid2, weight="weight")
            except (nx.NetworkXNoPath, nx.NodeNotFound):
                continue
            if len(path_nodes) <= 2:
                continue
            segment_geoms: List[QgsGeometry] = []
            total_area = 0.0
            valid_segments = True
            for nid_a, nid_b in zip(path_nodes, path_nodes[1:]):
                geom_a = all_nodes.get(nid_a, {}).get("geom")
                geom_b = all_nodes.get(nid_b, {}).get("geom")
                if not geom_a or not geom_b or geom_a.isEmpty() or geom_b.isEmpty():
                    valid_segments = False
                    break
                p_a = geom_a.nearestPoint(geom_b).asPoint()
                p_b = geom_b.nearestPoint(geom_a).asPoint()
                if p_a.isEmpty() or p_b.isEmpty():
                    valid_segments = False
                    break
                seg_dist = p_a.distance(p_b)
                if seg_dist > hop_distance + 1e-6:
                    valid_segments = False
                    break
                segment = _create_corridor_geometry(
                    [QgsPointXY(p_a), QgsPointXY(p_b)],
                    geom_a,
                    geom_b,
                    params,
                    obstacle_geoms=obstacle_geoms,
                    hop_distance=params.hop_distance,
                )
                if not segment or segment.isEmpty():
                    valid_segments = False
                    break
                segment_geoms.append(segment)
                total_area += segment.area() / 10000.0
                if params.debug_enabled:
                    _debug(
                        params,
                        f"Segment hop: {nid_a}->{nid_b} length={seg_dist:.2f} area={segment.area():.2f}",
                    )
            if not valid_segments or not segment_geoms:
                continue
            try:
                merged_geom = QgsGeometry.unaryUnion(segment_geoms)
            except Exception:
                merged_geom = segment_geoms[0]
                for extra in segment_geoms[1:]:
                    try:
                        merged_geom = merged_geom.combine(extra)
                    except Exception:
                        pass
            if merged_geom is None or merged_geom.isEmpty():
                continue
            cand = {
                "patch1": pid1,
                "patch2": pid2,
                "patch_ids": set(path_nodes),
                "hop_path_ids": set(path_nodes),
                "geom": clone_geometry(merged_geom),
                "area_ha": total_area,
                "is_stepping_chain": True,
                "is_hole_hop": True,
            }
            hopping_candidates.append(cand)
            processed_pairs.add(pair)
            if params.debug_enabled:
                _debug(
                    params,
                    f"Found hole-hop path: P{pid1} -> ... -> P{pid2} (nodes={path_nodes})",
                )

    print(f"  ✓ Found {len(hopping_candidates)} chains hopping through impassable holes.")
    return hopping_candidates


def _annotate_candidates_with_components(
    candidates: List[Dict],
    component_map: Dict[int, int],
) -> None:
    for cand in candidates:
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        cand["patch_ids"] = set(cand.get("patch_ids", set()))
        if not cand["patch_ids"]:
            cand["patch_ids"] = {p1, p2}
        comp1 = component_map.get(p1, p1)
        comp2 = component_map.get(p2, p2)
        cand["comp1"] = comp1
        cand["comp2"] = comp2
        cand["component_ids"] = {component_map.get(pid, pid) for pid in cand["patch_ids"]}
        cand.setdefault("is_stepping_chain", False)


def _build_component_pair_index(candidates: List[Dict]) -> Set[frozenset]:
    pairs: Set[frozenset] = set()
    for cand in candidates:
        comp1 = cand.get("comp1")
        comp2 = cand.get("comp2")
        if comp1 is None or comp2 is None or comp1 == comp2:
            continue
        pairs.add(frozenset({comp1, comp2}))
    return pairs


def _build_component_adjacency(candidates: List[Dict]) -> Dict[int, Set[int]]:
    adjacency: Dict[int, Set[int]] = defaultdict(set)
    for cand in candidates:
        comp1 = cand.get("comp1")
        comp2 = cand.get("comp2")
        if comp1 is None or comp2 is None or comp1 == comp2:
            continue
        adjacency[comp1].add(comp2)
        adjacency[comp2].add(comp1)
    return adjacency


def _detect_corridor_intersections(
    corridor_geom: QgsGeometry,
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    connected_patches: Set[int],
) -> Set[int]:
    """
    Efficiently detect which patches are traversed by a corridor geometry
    using the spatial index.
    """
    intersected: Set[int] = set()
    bbox = corridor_geom.boundingBox()
    
    # Use spatial index to find candidate interactions (fast)
    candidate_ids = spatial_index.intersects(bbox)
    
    for pid in candidate_ids:
        if pid in connected_patches:
            continue
        pdata = patches.get(pid)
        if not pdata:
            continue
            
        try:
            # Check for actual intersection
            if corridor_geom.intersects(pdata["geom"]):
                intersection = corridor_geom.intersection(pdata["geom"])
                if intersection and (not intersection.isEmpty()) and intersection.area() > 0:
                    intersected.add(pid)
        except Exception:
            continue
            
    return intersected


def _finalize_corridor_geometry(
    pid1: int,
    pid2: int,
    corridor_geom: QgsGeometry,
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    patch_union: Optional[QgsGeometry] = None,
) -> Tuple[Optional[QgsGeometry], Set[int]]:
    """
    Detect traversed patches, clip corridor geometry so it doesn't overlap them,
    and update the set of connected patches.
    """
    if corridor_geom is None or corridor_geom.isEmpty():
        return None, set()

    patch_ids: Set[int] = {pid1, pid2}
    
    # 1. Detect intermediate patches (A -> C -> B)
    # Using spatial index here is crucial for performance
    intersected = _detect_corridor_intersections(corridor_geom, patches, spatial_index, patch_ids)
    if intersected:
        patch_ids.update(intersected)

    # 2. Clip the corridor geometry against ALL involved patches
    # This makes the corridor "free" where it crosses existing habitat
    # and prevents drawing on top of patches.
    final_geom = clone_geometry(corridor_geom)
    
    if patch_union and not patch_union.isEmpty():
        try:
            final_geom = final_geom.difference(patch_union)
        except Exception:
            pass
    else:
        for pid in patch_ids:
            pdata = patches.get(pid)
            if not pdata: 
                continue
            patch_geom = pdata.get("geom")
            if not patch_geom or patch_geom.isEmpty():
                continue
                
            try:
                if final_geom.intersects(patch_geom):
                    final_geom = final_geom.difference(patch_geom)
                    if final_geom.isEmpty():
                        break
            except Exception:
                pass

    if final_geom is None or final_geom.isEmpty():
        # If the corridor is entirely consumed by patches, it means the patches 
        # touch or overlap. In vector analysis, this is valid connectivity (0 area cost).
        # We return an empty geom but valid patch_ids. However, to display it,
        # we might want to return None if strictly "corridor building".
        # But usually we return None to avoid 0-area features being written.
        return None, set()

    final_geom = final_geom.makeValid()
    if final_geom.isEmpty():
        return None, set()

    return final_geom, patch_ids


def _build_hop_edge_lookup(
    candidates: List[Dict],
    hop_adjacency: Dict[int, Set[int]],
) -> Dict[frozenset, Dict]:
    if not hop_adjacency:
        for cand in candidates:
            cand["is_hop_edge"] = False
        return {}

    lookup: Dict[frozenset, Dict] = {}
    for cand in candidates:
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        is_hop = p2 in hop_adjacency.get(p1, set())
        cand["is_hop_edge"] = is_hop
        if is_hop:
            lookup[frozenset({p1, p2})] = cand
    return lookup


def _build_stepping_chains(
    hop_lookup: Dict[frozenset, Dict],
    component_map: Dict[int, int],
    component_sizes: Optional[Dict[int, float]] = None,
) -> List[Dict]:
    """Build daisy-chain corridors by walking hop edges and summing real segment costs."""
    if not hop_lookup:
        return []

    comp_sizes = component_sizes or {}
    graph: Dict[int, Set[int]] = defaultdict(set)
    for pair in hop_lookup.keys():
        if len(pair) != 2:
            continue
        a, b = tuple(pair)
        graph[a].add(b)
        graph[b].add(a)

    chains: List[Dict] = []
    seen_paths: Set[Tuple[int, ...]] = set()

    start_nodes = sorted(graph.keys(), key=lambda pid: -comp_sizes.get(component_map.get(pid, pid), 0.0))

    for start in start_nodes:
        stack: List[Tuple[int, List[int], float, float, Set[int], List[QgsGeometry]]] = []
        stack.append((start, [start], 0.0, 0.0, {start}, []))
        while stack:
            node, path, cost_so_far, dist_so_far, patch_ids, geoms = stack.pop()
            path_key = tuple(path)
            if path_key in seen_paths:
                continue
            seen_paths.add(path_key)

            for nbr in graph.get(node, set()):
                if nbr in path:
                    continue
                edge = hop_lookup.get(frozenset({node, nbr}))
                if edge is None or edge.get("geom") is None:
                    continue
                edge_cost = float(edge.get("area_ha", 0.0))
                edge_dist = float(edge.get("distance_m", 0.0))
                new_cost = cost_so_far + edge_cost
                new_dist = dist_so_far + edge_dist
                new_path = path + [nbr]
                new_pids = set(patch_ids)
                new_pids.update(edge.get("patch_ids", set()))

                new_geoms = list(geoms)
                new_geoms.append(clone_geometry(edge.get("geom")))

                comp_start = component_map.get(path[0], path[0])
                comp_end = component_map.get(nbr, nbr)
                if comp_start != comp_end and len(new_path) >= 2:
                    merged_geom = new_geoms[0]
                    if len(new_geoms) > 1:
                        try:
                            merged_geom = QgsGeometry.unaryUnion(new_geoms)
                        except Exception:
                            merged_geom = new_geoms[0]
                    chains.append(
                        {
                            "patch1": path[0],
                            "patch2": nbr,
                            "patch_ids": new_pids,
                            "component_ids": {component_map.get(pid, pid) for pid in new_pids},
                            "geom": merged_geom,
                            "area_ha": new_cost,
                            "distance_m": new_dist,
                            "comp1": comp_start,
                            "comp2": comp_end,
                            "is_stepping_chain": True,
                        }
                    )
                stack.append((nbr, new_path, new_cost, new_dist, new_pids, new_geoms))
    _log_message(f"  **_build_stepping_chains:** Successfully generated {len(chains)} stepping-stone chains.", Qgis.Info)
    return chains


def _buffer_line_segment(line_geom: QgsGeometry, width: float) -> QgsGeometry:
    return line_geom.buffer(width / 2.0, BUFFER_SEGMENTS)


def _corridor_passes_width(corridor_geom: QgsGeometry, min_width: float) -> bool:
    if corridor_geom.isEmpty():
        return False
    # If the corridor is multipolygon (due to crossing patches), check each part?
    # Actually, if it's multipolygon, it means we clipped out patches.
    # The 'neck' check is complex on multipolygons. We check the buffer on the original,
    # but since we already clipped, we skip aggressive width validation on the final result
    # to avoid rejecting valid patch-traversals.
    return True


def _format_no_corridor_reason(
    stage: str,
    patch_count: int,
    candidate_count: int,
    params: VectorRunParams,
) -> str:
    return (
        f"{stage}: no feasible corridors could be generated.\n"
        f"- Patches meeting criteria: {patch_count}\n"
        f"- Candidate corridors generated: {candidate_count}\n"
        f"- Max search distance: {params.max_search_distance:.2f}\n"
        f"- Min corridor width: {params.min_corridor_width:.2f}\n"
        f"- Max corridor area: {params.max_corridor_area or 'None (no limit)'}\n"
        "Try increasing the search distance, lowering the minimum corridor width/area, "
        "or simplifying the patch layer."
    )


def _create_corridor_geometry(
    waypoints: List[QgsPointXY],
    source_geom: QgsGeometry,
    target_geom: QgsGeometry,
    params: VectorRunParams,
    obstacle_geoms: Optional[List[QgsGeometry]] = None,
    smooth_iterations: int = 0,
    hop_distance: float = 0.0,
) -> Optional[QgsGeometry]:
    if not waypoints or len(waypoints) < 2:
        return None

    corridor_line = QgsGeometry.fromPolylineXY([QgsPointXY(pt) for pt in waypoints])
    if smooth_iterations > 0:
        try:
            smoothed = corridor_line.smooth(smooth_iterations)
            if smoothed and not smoothed.isEmpty():
                corridor_line = smoothed
        except Exception:
            pass
            
    # Buffer to full width
    corridor_geom = _buffer_line_segment(corridor_line, params.min_corridor_width)
    
    # Clip start/end immediately to get the "bridge" geometry
    corridor_geom = corridor_geom.difference(source_geom)
    corridor_geom = corridor_geom.difference(target_geom)
    
    if obstacle_geoms:
        def _extract_line_parts(geom: QgsGeometry) -> List[QgsGeometry]:
            if geom is None or geom.isEmpty():
                return []
            if QgsWkbTypes.geometryType(geom.wkbType()) == QgsWkbTypes.LineGeometry:
                try:
                    if geom.isMultipart():
                        parts = []
                        for line in geom.asMultiPolyline():
                            if line:
                                parts.append(QgsGeometry.fromPolylineXY(line))
                        return parts
                    line = geom.asPolyline()
                    if line:
                        return [QgsGeometry.fromPolylineXY(line)]
                    return [geom]
                except Exception:
                    return [geom]
            try:
                parts: List[QgsGeometry] = []
                for sub in geom.asGeometryCollection():
                    parts.extend(_extract_line_parts(sub))
                return parts
            except Exception:
                return []

        for obstacle in obstacle_geoms:
            try:
                if not corridor_geom.intersects(obstacle):
                    continue
            except Exception:
                continue

            path_intersection = corridor_line.intersection(obstacle)
            if not path_intersection or path_intersection.isEmpty():
                _debug(params, "Corridor buffer overlaps impassable area; clipping.")
                corridor_geom = corridor_geom.difference(obstacle)
                if corridor_geom.isEmpty():
                    _debug(params, "Corridor emptied after impassable clip.")
                    return None
                continue
            line_parts = _extract_line_parts(path_intersection)
            part_lengths = [p.length() for p in line_parts if p and not p.isEmpty() and p.length() > 0]
            if not part_lengths:
                continue

            max_part = max(part_lengths)
            total_len = sum(part_lengths)
            _debug(
                params,
                f"Impassable crossed by line: parts={len(part_lengths)} total_len={total_len:.2f} "
                f"max_part={max_part:.2f} hop_dist={hop_distance:.2f}",
            )

            if hop_distance <= 0:
                _debug(params, "Impassable crossing not allowed (hop disabled); rejecting corridor.")
                return None
            if max_part > hop_distance + 1e-6:
                _debug(params, "Impassable hop exceeds hop distance; rejecting corridor.")
                return None

            _debug(params, "Hop permitted; clipping impassable area from corridor geometry (gap will remain).")
            corridor_geom = corridor_geom.difference(obstacle)
            if corridor_geom.isEmpty():
                _debug(params, "Corridor emptied after impassable clip.")
                return None
                
    corridor_geom = corridor_geom.makeValid()
    if corridor_geom.isEmpty():
        return None

    return corridor_geom


class RasterNavigator:
    def __init__(
        self,
        patches: Dict[int, Dict],
        obstacle_layers: List[QgsVectorLayer],
        target_crs: QgsCoordinateReferenceSystem,
        params: VectorRunParams,
    ):
        if not obstacle_layers:
            raise VectorAnalysisError("Select at least one polygon impassable layer for impassable land classes.")

        self._params = params
        self.resolution = max(params.grid_resolution, 1.0)
        self.obstacle_geoms: List[QgsGeometry] = []

        extent: Optional[QgsRectangle] = None
        for patch in patches.values():
            bbox = patch["geom"].boundingBox()
            if extent is None:
                extent = QgsRectangle(bbox)
            else:
                extent.combineExtentWith(bbox)

        for obstacle_layer in obstacle_layers:
            if obstacle_layer is None or QgsWkbTypes.geometryType(obstacle_layer.wkbType()) != QgsWkbTypes.PolygonGeometry:
                raise VectorAnalysisError("Select a polygon impassable layer for impassable land classes.")

            transform = QgsCoordinateTransform(obstacle_layer.crs(), target_crs, QgsProject.instance())
            for feature in obstacle_layer.getFeatures():
                geom = QgsGeometry(feature.geometry())
                if geom.isEmpty():
                    continue
                geom.transform(transform)
                geom = geom.makeValid()
                if geom.isEmpty():
                    continue
                if params.debug_enabled:
                    rings = _count_interior_rings(geom)
                    _debug(
                        params,
                        f"Impassable feature loaded: area={geom.area():.2f} rings={rings} bbox={geom.boundingBox().toString()}",
                    )
                self.obstacle_geoms.append(clone_geometry(geom))
                bbox = geom.boundingBox()
                if extent is None:
                    extent = QgsRectangle(bbox)
                else:
                    extent.combineExtentWith(bbox)

        if extent is None:
            raise VectorAnalysisError("Unable to determine extent for impassable land class routing.")

        pad = max(params.max_search_distance, params.min_corridor_width)
        extent = QgsRectangle(
            extent.xMinimum() - pad,
            extent.yMinimum() - pad,
            extent.xMaximum() + pad,
            extent.yMaximum() + pad,
        )
        width = max(extent.width(), self.resolution)
        height = max(extent.height(), self.resolution)

        self.origin_x = extent.xMinimum()
        self.origin_y = extent.yMaximum()
        self.cols = max(1, int(math.ceil(width / self.resolution)))
        self.rows = max(1, int(math.ceil(height / self.resolution)))
        self.passable = np.ones((self.rows, self.cols), dtype=bool)

        safety_buffer = params.min_corridor_width / 2.0
        _debug(
            params,
            f"RasterNavigator grid={self.cols}x{self.rows} res={self.resolution:.2f} safety_buffer={safety_buffer:.2f}",
        )

        hole_geoms: List[QgsGeometry] = []
        for geom in self.obstacle_geoms:
            try:
                mask_geom = geom.buffer(safety_buffer, 4) if safety_buffer > 0 else clone_geometry(geom)
            except Exception:
                mask_geom = clone_geometry(geom)

            if params.stepping_enabled and params.hop_distance > 0:
                holes = _extract_hole_geometries(geom)
                if holes:
                    hole_geoms.extend(holes)
                    try:
                        holes_union = QgsGeometry.unaryUnion(holes)
                    except Exception:
                        holes_union = holes[0]
                        for extra in holes[1:]:
                            try:
                                holes_union = holes_union.combine(extra)
                            except Exception:
                                pass
                    try:
                        mask_geom = mask_geom.difference(holes_union)
                    except Exception:
                        pass

            try:
                mask_geom = mask_geom.makeValid()
            except Exception:
                pass
            if mask_geom is None or mask_geom.isEmpty():
                continue
            self._burn_geometry(mask_geom)

        if params.debug_enabled:
            blocked = int((~self.passable).sum())
            total_cells = int(self.passable.size)
            _debug(params, f"Impassable mask burned: blocked_cells={blocked}/{total_cells} ({blocked/total_cells*100:.1f}%)")

        self._hop_offsets_cache: Dict[float, List[Tuple[int, int, float]]] = {}
        self._boundary_mask = self._compute_boundary_mask()
        self._outside_reachable = self._compute_outside_reachable() if (params.debug_enabled or params.stepping_enabled) else np.ones_like(self.passable, dtype=bool)

        if params.debug_enabled and (params.stepping_enabled and params.hop_distance > 0):
            enclosed = int((self.passable & ~self._outside_reachable).sum())
            boundary_cells = int(self._boundary_mask.sum())
            _debug(
                params,
                f"Raster stats: boundary_cells={boundary_cells} enclosed_passable_cells={enclosed} "
                f"(holes={len(hole_geoms)})",
            )
            if hole_geoms:
                max_list = 12
                for idx, hole in enumerate(hole_geoms[:max_list], start=1):
                    p_cells, b_cells, t_cells = self._cell_stats_in_geom(hole)
                    _debug(
                        params,
                        f"Hole {idx}: area={hole.area():.2f} m² cell_centers_inside={t_cells} "
                        f"passable_inside={p_cells} blocked_inside={b_cells}",
                    )
                if len(hole_geoms) > max_list:
                    _debug(params, f"... {len(hole_geoms) - max_list} more holes omitted")
                if enclosed <= 0:
                    _debug(
                        params,
                        "No enclosed passable cells detected; holes may be too small for the grid or filled by buffers. "
                        "Try smaller grid_resolution and/or smaller min_corridor_width.",
                    )

    def _world_to_rc(self, point: QgsPointXY) -> Optional[Tuple[int, int]]:
        col = int(math.floor((point.x() - self.origin_x) / self.resolution))
        row = int(math.floor((self.origin_y - point.y()) / self.resolution))
        if 0 <= row < self.rows and 0 <= col < self.cols:
            return row, col
        return None

    def _rc_to_world(self, row: int, col: int) -> QgsPointXY:
        x = self.origin_x + (col + 0.5) * self.resolution
        y = self.origin_y - (row + 0.5) * self.resolution
        return QgsPointXY(x, y)

    def _burn_geometry(self, geom: QgsGeometry) -> None:
        bbox = geom.boundingBox()
        min_col = max(0, int(math.floor((bbox.xMinimum() - self.origin_x) / self.resolution)))
        max_col = min(self.cols - 1, int(math.ceil((bbox.xMaximum() - self.origin_x) / self.resolution)))
        min_row = max(0, int(math.floor((self.origin_y - bbox.yMaximum()) / self.resolution)))
        max_row = min(self.rows - 1, int(math.ceil((self.origin_y - bbox.yMinimum()) / self.resolution)))

        if min_col > max_col or min_row > max_row:
            return

        for row in range(min_row, max_row + 1):
            y = self.origin_y - (row + 0.5) * self.resolution
            for col in range(min_col, max_col + 1):
                x = self.origin_x + (col + 0.5) * self.resolution
                try:
                    if geom.contains(QgsPointXY(x, y)):
                        self.passable[row, col] = False
                except Exception:
                    pass

    def _cell_stats_in_geom(self, geom: QgsGeometry) -> Tuple[int, int, int]:
        """Return (passable_cells, blocked_cells, total_cells) for cell centers inside `geom`."""
        bbox = geom.boundingBox()
        min_col = max(0, int(math.floor((bbox.xMinimum() - self.origin_x) / self.resolution)))
        max_col = min(self.cols - 1, int(math.ceil((bbox.xMaximum() - self.origin_x) / self.resolution)))
        min_row = max(0, int(math.floor((self.origin_y - bbox.yMaximum()) / self.resolution)))
        max_row = min(self.rows - 1, int(math.ceil((self.origin_y - bbox.yMinimum()) / self.resolution)))

        if min_col > max_col or min_row > max_row:
            return 0, 0, 0

        passable = 0
        blocked = 0
        total = 0
        for row in range(min_row, max_row + 1):
            y = self.origin_y - (row + 0.5) * self.resolution
            for col in range(min_col, max_col + 1):
                x = self.origin_x + (col + 0.5) * self.resolution
                try:
                    if not geom.contains(QgsPointXY(x, y)):
                        continue
                except Exception:
                    continue
                total += 1
                if self.passable[row, col]:
                    passable += 1
                else:
                    blocked += 1
        return passable, blocked, total

    def _compute_boundary_mask(self) -> np.ndarray:
        """Passable cells adjacent to at least one blocked cell (8-neighborhood)."""
        try:
            rows, cols = self.passable.shape
            blocked = ~self.passable
            padded = np.pad(blocked, 1, constant_values=True)
            neighbors_blocked = (
                padded[0:rows, 0:cols]
                | padded[0:rows, 1 : cols + 1]
                | padded[0:rows, 2 : cols + 2]
                | padded[1 : rows + 1, 0:cols]
                | padded[1 : rows + 1, 2 : cols + 2]
                | padded[2 : rows + 2, 0:cols]
                | padded[2 : rows + 2, 1 : cols + 1]
                | padded[2 : rows + 2, 2 : cols + 2]
            )
            return self.passable & neighbors_blocked
        except Exception:
            return np.zeros_like(self.passable, dtype=bool)

    def _compute_outside_reachable(self) -> np.ndarray:
        """Flood-fill passable cells connected to the grid edge (i.e. not enclosed holes)."""
        rows, cols = self.passable.shape
        reachable = np.zeros((rows, cols), dtype=bool)
        q: deque = deque()
        # Seed from boundary passable cells
        for c in range(cols):
            if self.passable[0, c] and not reachable[0, c]:
                reachable[0, c] = True
                q.append((0, c))
            if self.passable[rows - 1, c] and not reachable[rows - 1, c]:
                reachable[rows - 1, c] = True
                q.append((rows - 1, c))
        for r in range(rows):
            if self.passable[r, 0] and not reachable[r, 0]:
                reachable[r, 0] = True
                q.append((r, 0))
            if self.passable[r, cols - 1] and not reachable[r, cols - 1]:
                reachable[r, cols - 1] = True
                q.append((r, cols - 1))

        moves = [(-1, 0), (1, 0), (0, -1), (0, 1), (-1, -1), (-1, 1), (1, -1), (1, 1)]
        while q:
            r, c = q.popleft()
            for dr, dc in moves:
                nr, nc = r + dr, c + dc
                if not (0 <= nr < rows and 0 <= nc < cols):
                    continue
                if reachable[nr, nc] or not self.passable[nr, nc]:
                    continue
                reachable[nr, nc] = True
                q.append((nr, nc))
        return reachable

    def _hop_offsets(self, hop_distance: float) -> List[Tuple[int, int, float]]:
        key = round(float(hop_distance), 6)
        cached = self._hop_offsets_cache.get(key)
        if cached is not None:
            return cached
        hop = float(hop_distance)
        if hop <= 0:
            self._hop_offsets_cache[key] = []
            return []
        max_cells = int(math.ceil(hop / self.resolution))
        offsets: List[Tuple[int, int, float]] = []
        for dr in range(-max_cells, max_cells + 1):
            for dc in range(-max_cells, max_cells + 1):
                if dr == 0 and dc == 0:
                    continue
                dist = math.hypot(dr * self.resolution, dc * self.resolution)
                if dist <= hop + 1e-6:
                    offsets.append((dr, dc, dist))
        offsets.sort(key=lambda x: x[2])
        self._hop_offsets_cache[key] = offsets
        return offsets

    def _line_passable_blocked_counts(self, start: Tuple[int, int], end: Tuple[int, int]) -> Tuple[int, int]:
        """Count passable vs blocked intermediate cells along the straight line."""
        r0, c0 = start
        r1, c1 = end
        dr = abs(r1 - r0)
        dc = abs(c1 - c0)
        sr = 1 if r1 > r0 else -1
        sc = 1 if c1 > c0 else -1
        err = dr - dc
        r, c = r0, c0
        passable = 0
        blocked = 0
        while (r, c) != (r1, c1):
            e2 = 2 * err
            if e2 > -dc:
                err -= dc
                r += sr
            if e2 < dr:
                err += dr
                c += sc
            if (r, c) == (r1, c1):
                break
            if self.passable[r, c]:
                passable += 1
            else:
                blocked += 1
        return passable, blocked

    def _shortest_path_with_hops(
        self, start: Tuple[int, int], goal: Tuple[int, int], hop_distance: float
    ) -> Optional[List[Tuple[int, int]]]:
        rows, cols = self.passable.shape
        moves = [(-1, 0), (1, 0), (0, -1), (0, 1), (-1, -1), (-1, 1), (1, -1), (1, 1)]
        hop_offsets = self._hop_offsets(hop_distance)

        best = np.full((rows, cols), float("inf"), dtype=float)
        parent = np.full((rows, cols), -1, dtype=np.int64)
        parent_is_hop = np.zeros((rows, cols), dtype=bool)
        best[start[0], start[1]] = 0.0

        heap: List[Tuple[float, int, int]] = [(0.0, start[0], start[1])]
        while heap:
            cost, r, c = heapq.heappop(heap)
            if cost != best[r, c]:
                continue
            if (r, c) == goal:
                break

            for dr, dc in moves:
                nr, nc = r + dr, c + dc
                if not (0 <= nr < rows and 0 <= nc < cols):
                    continue
                if not self.passable[nr, nc]:
                    continue
                step = math.sqrt(2) if dr != 0 and dc != 0 else 1.0
                new_cost = cost + (step * self.resolution)
                if new_cost < best[nr, nc]:
                    best[nr, nc] = new_cost
                    parent[nr, nc] = r * cols + c
                    parent_is_hop[nr, nc] = False
                    heapq.heappush(heap, (new_cost, nr, nc))

            if hop_offsets and self._boundary_mask[r, c]:
                for dr, dc, _dist in hop_offsets:
                    nr, nc = r + dr, c + dc
                    if not (0 <= nr < rows and 0 <= nc < cols):
                        continue
                    if not self.passable[nr, nc]:
                        continue
                    if not self._boundary_mask[nr, nc]:
                        continue
                    passable_steps, blocked_steps = self._line_passable_blocked_counts((r, c), (nr, nc))
                    if blocked_steps <= 0:
                        continue
                    # Approximate corridor cost for this hop: only charge for the portion of the segment
                    # that lies in passable space (blocked portion is "free" to hop over).
                    hop_extra = passable_steps * self.resolution
                    new_cost = cost + hop_extra
                    if new_cost < best[nr, nc]:
                        best[nr, nc] = new_cost
                        parent[nr, nc] = r * cols + c
                        parent_is_hop[nr, nc] = True
                        heapq.heappush(heap, (new_cost, nr, nc))

        if best[goal[0], goal[1]] == float("inf"):
            return None

        path: List[Tuple[int, int]] = [goal]
        hops_used = 0
        hop_dists: List[float] = []
        hop_costs: List[float] = []
        cur = goal
        while cur != start:
            r, c = cur
            prev = int(parent[r, c])
            if prev < 0:
                return None
            pr, pc = divmod(prev, cols)
            if parent_is_hop[r, c]:
                hops_used += 1
                hop_dists.append(math.hypot((r - pr) * self.resolution, (c - pc) * self.resolution))
                passable_steps, blocked_steps = self._line_passable_blocked_counts((pr, pc), (r, c))
                hop_costs.append(passable_steps * self.resolution)
            cur = (pr, pc)
            path.append(cur)
        path.reverse()
        if getattr(self._params, "debug_enabled", False):
            hole_nodes = sum(
                1
                for (rr, cc) in path
                if self.passable[rr, cc] and not bool(self._outside_reachable[rr, cc])
            )
            hop_max = max(hop_dists) if hop_dists else 0.0
            hop_cost_sum = sum(hop_costs)
            _debug(
                self._params,
                f"Navigator path: {len(path)} nodes, hops_used={hops_used}, hole_nodes={hole_nodes}, "
                f"hop_max={hop_max:.1f}, hop_passable_cost_sum={hop_cost_sum:.1f}",
            )
        return path

    def find_path(
        self, start_point: QgsPointXY, end_point: QgsPointXY, hop_distance: float = 0.0
    ) -> Optional[List[QgsPointXY]]:
        start = self._world_to_rc(start_point)
        end = self._world_to_rc(end_point)
        if start is None or end is None:
            return None
        start_node = _nearest_passable_node(self.passable, start)
        end_node = _nearest_passable_node(self.passable, end)
        if start_node is None or end_node is None:
            return None

        if hop_distance and hop_distance > 0:
            path = self._shortest_path_with_hops(start_node, end_node, hop_distance)
        else:
            path = _shortest_path_on_mask(self.passable, start_node, end_node)
        if not path:
            return None
        return [self._rc_to_world(r, c) for r, c in path]


def _nearest_passable_node(mask: np.ndarray, node: Tuple[int, int], search_radius: int = 6) -> Optional[Tuple[int, int]]:
    r0, c0 = node
    rows, cols = mask.shape
    if 0 <= r0 < rows and 0 <= c0 < cols and mask[r0, c0]:
        return node
    for radius in range(1, search_radius + 1):
        for dr in range(-radius, radius + 1):
            for dc in range(-radius, radius + 1):
                nr = r0 + dr
                nc = c0 + dc
                if 0 <= nr < rows and 0 <= nc < cols and mask[nr, nc]:
                    return nr, nc
    return None


def _shortest_path_on_mask(
    mask: np.ndarray, start: Tuple[int, int], goal: Tuple[int, int]
) -> Optional[List[Tuple[int, int]]]:
    rows, cols = mask.shape
    moves = [(-1, 0), (1, 0), (0, -1), (0, 1), (-1, -1), (-1, 1), (1, -1), (1, 1)]

    heap: List[Tuple[float, int, int]] = []
    heapq.heappush(heap, (0.0, start[0], start[1]))
    best_cost: Dict[Tuple[int, int], float] = {start: 0.0}
    parents: Dict[Tuple[int, int], Tuple[int, int]] = {}

    while heap:
        cost, r, c = heapq.heappop(heap)
        if (r, c) == goal:
            break
        if cost > best_cost.get((r, c), float("inf")):
            continue
        for dr, dc in moves:
            nr, nc = r + dr, c + dc
            if not (0 <= nr < rows and 0 <= nc < cols):
                continue
            if not mask[nr, nc]:
                continue
            step = math.sqrt(2) if dr != 0 and dc != 0 else 1.0
            new_cost = cost + step
            if new_cost >= best_cost.get((nr, nc), float("inf")):
                continue
            best_cost[(nr, nc)] = new_cost
            parents[(nr, nc)] = (r, c)
            heapq.heappush(heap, (new_cost, nr, nc))

    if goal not in parents and goal != start:
        return None

    path: List[Tuple[int, int]] = [goal]
    current = goal
    while current != start:
        current = parents.get(current)
        if current is None:
            return None
        path.append(current)
    path.reverse()
    return path


def find_all_possible_corridors(
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    params: VectorRunParams,
    hop_adjacency: Optional[Dict[int, Set[int]]] = None,
    patch_union: Optional[QgsGeometry] = None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
    progress_start: int = 30,
    progress_end: int = 55,
    navigator: Optional[RasterNavigator] = None,
    timings: Optional[TimingRecorder] = None,
) -> List[Dict]:
    print("  Finding all possible corridors...")
    all_corridors: List[Dict] = []
    processed_pairs: Set[frozenset] = set()
    total = len(patches) or 1
    accum_durations: Dict[str, float] = defaultdict(float)
    accum_counts: Dict[str, int] = defaultdict(int)
    max_keep_per_pair = 2
    min_distinct_overlap_ratio = 0.75  # higher = allow more similar corridors

    @contextmanager
    def _measure(label: str):
        start = time.perf_counter()
        accum_counts[label] += 1
        try:
            yield
        finally:
            accum_durations[label] += time.perf_counter() - start

    def _sample_boundary_points(
        patch_geom: QgsGeometry, max_points: int = 32, allow_densify: bool = True
    ) -> List[QgsPointXY]:
        pts: List[QgsPointXY] = []
        try:
            ring = patch_geom.constGet().exteriorRing()
            if ring:
                pts = [QgsPointXY(v) for v in ring.vertices()]
        except Exception:
            try:
                pts = [QgsPointXY(v) for v in patch_geom.vertices()]
            except Exception:
                pts = []
        if len(pts) > max_points and max_points > 0:
            step = max(1, len(pts) // max_points)
            pts = pts[::step][:max_points]
        if allow_densify and len(pts) < max_points:
            try:
                perim = max(patch_geom.length(), 1.0)
                target_spacing = perim / max_points
                densified = patch_geom.densifyByDistance(target_spacing)
                extra = [QgsPointXY(v) for v in densified.vertices()]
                if extra:
                    pts.extend(extra)
            except Exception:
                pass
            if len(pts) > max_points:
                step = max(1, len(pts) // max_points)
                pts = pts[::step][:max_points]
        return pts[:max_points]

    def _pick_extremes_along_axis(
        points: List[QgsPointXY], ax_dx: float, ax_dy: float, cx: float, cy: float
    ) -> Tuple[Optional[QgsPointXY], Optional[QgsPointXY]]:
        if abs(ax_dx) < 1e-12 and abs(ax_dy) < 1e-12:
            return None, None
        if not points:
            return None, None

        def _proj(pt: QgsPointXY) -> float:
            return (pt.x() - cx) * ax_dx + (pt.y() - cy) * ax_dy

        p_max = max(points, key=_proj)
        p_min = min(points, key=_proj)
        if p_max.distance(p_min) < 1e-6:
            return p_max, None
        return p_max, p_min

    def _anchor_variants_for_pair(
        g1: QgsGeometry, g2: QgsGeometry, nearest1: QgsPointXY, nearest2: QgsPointXY
    ) -> List[Tuple[str, QgsPointXY, QgsPointXY]]:
        variants: List[Tuple[str, QgsPointXY, QgsPointXY]] = [("nearest", nearest1, nearest2)]
        # Build two additional variants that are on the *facing* edges of each patch.
        # This prevents "teleport" corridors that rely on travel through patch interior
        # to exit on the opposite side.
        try:
            c1 = g1.centroid().asPoint()
            c2 = g2.centroid().asPoint()
            c1x, c1y = c1.x(), c1.y()
            c2x, c2y = c2.x(), c2.y()
            vx, vy = (c2x - c1x), (c2y - c1y)
        except Exception:
            c1x, c1y, c2x, c2y = 0.0, 0.0, 1.0, 0.0
            vx, vy = 1.0, 0.0

        pts1 = _sample_boundary_points(g1, max_points=96, allow_densify=True)
        pts2 = _sample_boundary_points(g2, max_points=96, allow_densify=True)
        if not pts1 or not pts2:
            return variants

        # "Facing" = high projection along the axis towards the other patch.
        def _proj_to_other_from_1(pt: QgsPointXY) -> float:
            return (pt.x() - c1x) * vx + (pt.y() - c1y) * vy

        def _proj_to_other_from_2(pt: QgsPointXY) -> float:
            # towards patch1, so invert axis
            return (pt.x() - c2x) * (-vx) + (pt.y() - c2y) * (-vy)

        p1_max = max((_proj_to_other_from_1(p) for p in pts1), default=0.0)
        p2_max = max((_proj_to_other_from_2(p) for p in pts2), default=0.0)
        # keep points in the top 40% of "facing" scores
        p1_cut = p1_max * 0.60
        p2_cut = p2_max * 0.60
        facing1 = [p for p in pts1 if _proj_to_other_from_1(p) >= p1_cut]
        facing2 = [p for p in pts2 if _proj_to_other_from_2(p) >= p2_cut]
        if not facing1:
            facing1 = pts1
        if not facing2:
            facing2 = pts2

        # Now spread along perpendicular direction, but only within facing sets.
        px, py = (-vy, vx)
        a1_pos, a1_neg = _pick_extremes_along_axis(facing1, px, py, c1x, c1y)
        a2_pos, a2_neg = _pick_extremes_along_axis(facing2, px, py, c2x, c2y)
        if a1_pos is not None and a2_pos is not None:
            variants.append(("side_pos", a1_pos, a2_pos))
        if a1_neg is not None and a2_neg is not None:
            variants.append(("side_neg", a1_neg, a2_neg))

        seen = set()
        uniq: List[Tuple[str, QgsPointXY, QgsPointXY]] = []
        for tag, pta, ptb in variants:
            key = (round(pta.x(), 3), round(pta.y(), 3), round(ptb.x(), 3), round(ptb.y(), 3))
            if key in seen:
                continue
            seen.add(key)
            uniq.append((tag, pta, ptb))
        return uniq

    def _overlap_ratio(g1: QgsGeometry, g2: QgsGeometry) -> float:
        try:
            a1 = g1.area()
            a2 = g2.area()
            denom = min(a1, a2)
            if denom <= 0:
                return 1.0
            inter = g1.intersection(g2)
            if inter is None or inter.isEmpty():
                return 0.0
            return inter.area() / denom
        except Exception:
            return 1.0

    for idx, (pid1, pdata1) in enumerate(patches.items(), start=1):
        if idx % 10 == 0:
            print(f"    Analyzing patch {idx}/{len(patches)}...", end="\r")
        if progress_cb is not None:
            span = max(progress_end - progress_start, 1)
            progress_value = progress_start + ((idx - 1) / total) * span
            emit_progress(progress_cb, progress_value, "Building corridor candidates…")

        geom1 = pdata1["geom"]
        rect = geom1.boundingBox()
        rect.grow(params.max_search_distance)
        candidate_ids = spatial_index.intersects(rect)

        with _measure("Patch iteration"):
            for pid2 in candidate_ids:
                if pid1 >= pid2:
                    continue

                pair = frozenset({pid1, pid2})
                if pair in processed_pairs:
                    continue

                pdata2 = patches.get(pid2)
                if not pdata2:
                    continue
                geom2 = pdata2["geom"]

                distance = geom1.distance(geom2)
                if distance > params.max_search_distance:
                    continue

                if params.debug_enabled and len(all_corridors) < 5:
                    _debug(
                        params,
                        f"Pair {pid1}-{pid2} dist={distance:.2f} max_search={params.max_search_distance:.2f} "
                        f"navigator={'yes' if navigator else 'no'}",
                    )

                p1 = geom1.nearestPoint(geom2).asPoint()
                p2 = geom2.nearestPoint(geom1).asPoint()
                if p1.isEmpty() or p2.isEmpty():
                    continue

                hop_dist = params.hop_distance if params.stepping_enabled else 0.0
                p1_xy = QgsPointXY(p1)
                p2_xy = QgsPointXY(p2)

                def _path_cost_length(points: List[QgsPointXY]) -> float:
                    return sum(points[i].distance(points[i + 1]) for i in range(len(points) - 1))

                def _best_path_to_boundary(
                    start_pt: QgsPointXY, patch_geom: QgsGeometry
                ) -> Tuple[Optional[List[QgsPointXY]], Optional[QgsPointXY], float]:
                    if not navigator:
                        return None, None, float("inf")
                    candidates = _sample_boundary_points(patch_geom, max_points=16, allow_densify=False)
                    best_path: Optional[List[QgsPointXY]] = None
                    best_cost = float("inf")
                    best_pt: Optional[QgsPointXY] = None
                    for target in candidates:
                        pts = navigator.find_path(start_pt, target, hop_distance=hop_dist)
                        if not pts:
                            continue
                        cost = _path_cost_length(pts)
                        if cost < best_cost:
                            best_cost = cost
                            best_path = pts
                            best_pt = target
                    if not best_path:
                        dense_candidates = _sample_boundary_points(patch_geom, max_points=40, allow_densify=True)
                        for target in dense_candidates:
                            pts = navigator.find_path(start_pt, target, hop_distance=hop_dist)
                            if not pts:
                                continue
                            cost = _path_cost_length(pts)
                            if cost < best_cost:
                                best_cost = cost
                                best_path = pts
                                best_pt = target
                    return best_path, best_pt, best_cost

                pair_candidates: List[Dict] = []
                variants = _anchor_variants_for_pair(geom1, geom2, p1_xy, p2_xy)
                for variant_tag, start_xy, end_xy in variants:
                    raw_geom_candidates: List[Tuple[str, QgsGeometry]] = []
                    path_points: Optional[List[QgsPointXY]] = None

                    if navigator:
                        with _measure("Navigator routing"):
                            path_points = navigator.find_path(start_xy, end_xy, hop_distance=hop_dist)

                    if navigator and path_points:
                        with _measure("Corridor geometry (navigator)"):
                            nav_geom = _create_corridor_geometry(
                                path_points,
                                geom1,
                                geom2,
                                params,
                                obstacle_geoms=navigator.obstacle_geoms,
                                smooth_iterations=0 if (params.stepping_enabled and params.hop_distance > 0) else 3,
                                hop_distance=hop_dist,
                            )
                        if nav_geom:
                            raw_geom_candidates.append((f"navigator:{variant_tag}", nav_geom))

                    with _measure("Corridor geometry (direct)"):
                        direct_geom = _create_corridor_geometry(
                            [start_xy, end_xy],
                            geom1,
                            geom2,
                            params,
                            obstacle_geoms=navigator.obstacle_geoms if navigator else None,
                            hop_distance=hop_dist,
                        )
                    if direct_geom:
                        raw_geom_candidates.append((f"direct:{variant_tag}", direct_geom))

                    if not raw_geom_candidates:
                        continue

                    best_source, raw_geom = min(
                        raw_geom_candidates,
                        key=lambda item: item[1].area() if item[1] and not item[1].isEmpty() else float("inf"),
                    )
                    if raw_geom is None or raw_geom.isEmpty():
                        continue

                    # Reject candidates that "reach through" an endpoint patch to exit on the far side.
                    # We do this by measuring how much of the *raw centerline* lies within each endpoint patch.
                    try:
                        if navigator and path_points:
                            raw_line = QgsGeometry.fromPolylineXY(path_points)
                        else:
                            raw_line = QgsGeometry.fromPolylineXY([start_xy, end_xy])
                        max_inside_len = max(params.min_corridor_width * 3.0, 5.0)
                        for _pid, _pgeom in ((pid1, geom1), (pid2, geom2)):
                            try:
                                inter = raw_line.intersection(_pgeom)
                                inside_len = inter.length() if inter and not inter.isEmpty() else 0.0
                                if inside_len > max_inside_len:
                                    raw_geom = None
                                    break
                            except Exception:
                                continue
                        if raw_geom is None:
                            continue
                    except Exception:
                        pass

                    if navigator:
                        with _measure("Handle intermediate patch"):
                            intersected_temp = _detect_corridor_intersections(
                                raw_geom, patches, spatial_index, {pid1, pid2}
                            )
                            if intersected_temp:
                                intermediate_id = min(
                                    intersected_temp, key=lambda pid: geom1.distance(patches[pid]["geom"])
                                )
                                mid_geom = patches[intermediate_id]["geom"]

                                path_in, entry_pt, cost_in = _best_path_to_boundary(start_xy, mid_geom)
                                path_out, exit_pt, cost_out = _best_path_to_boundary(end_xy, mid_geom)

                                if path_in and path_out and entry_pt and exit_pt:
                                    entry_geom = _create_corridor_geometry(
                                        path_in,
                                        geom1,
                                        mid_geom,
                                        params,
                                        obstacle_geoms=navigator.obstacle_geoms,
                                        smooth_iterations=0 if (params.stepping_enabled and params.hop_distance > 0) else 3,
                                        hop_distance=hop_dist,
                                    )
                                    exit_geom = _create_corridor_geometry(
                                        path_out,
                                        mid_geom,
                                        geom2,
                                        params,
                                        obstacle_geoms=navigator.obstacle_geoms,
                                        smooth_iterations=0 if (params.stepping_enabled and params.hop_distance > 0) else 3,
                                        hop_distance=hop_dist,
                                    )
                                    internal_geom = QgsGeometry.fromPolylineXY([entry_pt, exit_pt]).buffer(
                                        params.min_corridor_width / 2.0, BUFFER_SEGMENTS
                                    )
                                    geoms_to_merge = [
                                        g for g in (entry_geom, exit_geom, internal_geom) if g and not g.isEmpty()
                                    ]
                                    if geoms_to_merge:
                                        try:
                                            split_geom = QgsGeometry.unaryUnion(geoms_to_merge)
                                        except Exception:
                                            split_geom = geoms_to_merge[0]
                                            for g in geoms_to_merge[1:]:
                                                try:
                                                    split_geom = split_geom.combine(g)
                                                except Exception:
                                                    pass
                                        if split_geom is not None and not split_geom.isEmpty():
                                            raw_geom = split_geom

                    with _measure("Finalize corridor geometry"):
                        corridor_geom, patch_ids = _finalize_corridor_geometry(
                            pid1, pid2, raw_geom, patches, spatial_index, patch_union=patch_union
                        )
                    if corridor_geom is None:
                        continue

                    corridor_area_ha = corridor_geom.area() / 10000.0
                    original_area_ha = corridor_area_ha
                    is_hop_edge = bool(hop_adjacency) and (
                        pid2 in hop_adjacency.get(pid1, set()) or pid1 in hop_adjacency.get(pid2, set())
                    )
                    if corridor_area_ha < 0:
                        continue
                    if (
                        params.max_corridor_area is not None
                        and corridor_area_ha > params.max_corridor_area
                        and not is_hop_edge
                    ):
                        continue
                    if (
                        params.stepping_enabled
                        and params.hop_distance > 0
                        and is_hop_edge
                        and navigator is None
                    ):
                        corridor_area_ha = 0.0

                    pair_candidates.append(
                        {
                            "patch1": pid1,
                            "patch2": pid2,
                            "patch_ids": patch_ids,
                            "geom": clone_geometry(corridor_geom),
                            "area_ha": corridor_area_ha,
                            "original_area_ha": original_area_ha,
                            "distance_m": distance,
                            "is_hop_edge": is_hop_edge,
                            "variant": variant_tag,
                            "source": best_source,
                        }
                    )

                if not pair_candidates:
                    processed_pairs.add(pair)
                    continue

                pair_candidates.sort(key=lambda c: (c.get("area_ha", 0.0), c.get("distance_m", 0.0)))
                kept: List[Dict] = []
                for cand in pair_candidates:
                    if len(kept) >= max_keep_per_pair:
                        break
                    g = cand.get("geom")
                    if g is None or g.isEmpty():
                        continue
                    too_similar = False
                    for prev in kept:
                        if _overlap_ratio(g, prev["geom"]) >= min_distinct_overlap_ratio:
                            too_similar = True
                            break
                    if too_similar:
                        continue
                    kept.append(cand)

                if params.debug_enabled:
                    _debug(
                        params,
                        f"Pair {pid1}-{pid2} variants tried={len(pair_candidates)} kept={len(kept)} "
                        f"({', '.join(str(c.get('source')) for c in kept)})",
                    )

                all_corridors.extend(kept)
                processed_pairs.add(pair)

    emit_progress(progress_cb, progress_end, "Candidate corridors ready.")
    print(f"\n  ✓ Found {len(all_corridors)} possible corridors")

    if timings:
        for label, duration in accum_durations.items():
            count = accum_counts.get(label, 0)
            timings.add(f"Find corridors | {label} (count={count})", duration)

    return all_corridors


def optimize_hdfm_strategy(
    patches: Dict[int, Dict],
    candidates: List[Dict],
    params: VectorRunParams,
    allow_loops: bool = True,
    mode: str = "resilient",
) -> Tuple[Dict[int, Dict], Dict]:
    """
    HDFM Strategy: Entropy Minimization

    Modes:
      - resilient: MST + optional strategic loops (multi-network allowed).
      - largest_network: Grow the largest single network (seed = largest patch), then optional loops.
    """
    if hdfm_math is None:
        raise VectorAnalysisError("NetworkX is required for HDFM optimization but could not be imported.")

    mode = (mode or "resilient").lower()
    selected: Dict[int, Dict] = {}
    budget_used = 0.0
    remaining = params.budget_area

    # ------------------------------------------------------------------
    # Phase 1: Backbone construction
    # ------------------------------------------------------------------
    def _add_corridor(cand: Dict, corr_type: str) -> None:
        nonlocal budget_used, remaining
        cid = len(selected) + 1
        selected[cid] = {
            "geom": clone_geometry(cand["geom"]),
            "patch_ids": cand.get("patch_ids", {cand.get("patch1"), cand.get("patch2")}),
            "area_ha": cand.get("area_ha", 0.0),
            "p1": cand.get("patch1"),
            "p2": cand.get("patch2"),
            "distance": cand.get("distance_m", 1.0),
            "type": corr_type,
            "variant": cand.get("variant"),
            "source": cand.get("source"),
        }
        budget_used += cand.get("area_ha", 0.0)
        remaining -= cand.get("area_ha", 0.0)

    backbone_edges: List[Tuple[int, int, float, float]] = []

    if mode == "largest_network":
        print("  Strategy: HDFM (Largest Network)")
        # Seed with largest patch by area and grow outward (Prim-style) to avoid hub-and-spoke
        seed_patch = max(patches.keys(), key=lambda pid: patches[pid]["area_ha"])
        visited: Set[int] = {seed_patch}

        adjacency: Dict[int, List[Dict]] = defaultdict(list)
        for cand in candidates:
            p1, p2 = cand.get("patch1"), cand.get("patch2")
            adjacency[p1].append(cand)
            adjacency[p2].append(cand)

        heap: List[Tuple[float, float, int, Dict]] = []
        counter = 0
        for cand in adjacency.get(seed_patch, []):
            counter += 1
            heapq.heappush(
                heap,
                (cand.get("area_ha", 0.0), cand.get("distance_m", 0.0), counter, cand),
            )

        while heap and remaining > 0:
            cost, _dist, _idx, cand = heapq.heappop(heap)
            if cost > remaining:
                continue
            p1, p2 = cand.get("patch1"), cand.get("patch2")
            in1, in2 = p1 in visited, p2 in visited
            if in1 and in2:
                continue  # already connected; skip to avoid redundant spokes
            if not in1 and not in2:
                continue  # does not touch current network

            new_node = p2 if in1 else p1
            _add_corridor(cand, "backbone")
            backbone_edges.append((p1, p2, cost, cand.get("distance_m", 1.0)))
            visited.add(new_node)

            for nxt in adjacency.get(new_node, []):
                n_p1, n_p2 = nxt.get("patch1"), nxt.get("patch2")
                if (n_p1 in visited and n_p2 in visited) or (n_p1 not in visited and n_p2 not in visited):
                    continue
                counter += 1
                heapq.heappush(
                    heap,
                    (nxt.get("area_ha", 0.0), nxt.get("distance_m", 0.0), counter, nxt),
                )

    else:
        print(f"  Strategy: HDFM (Resilient, loops allowed={allow_loops})")
        _log_message("--- HDFM Phase 1: Building Dendritic Backbone ---")
        mst_candidates = sorted(candidates, key=lambda x: x.get("area_ha", 0.0))
        uf = UnionFind()
        for pid in patches:
            uf.find(pid)

        for cand in mst_candidates:
            cost = cand.get("area_ha", 0.0)
            p1 = cand.get("patch1")
            p2 = cand.get("patch2")

            if cost > remaining:
                continue

            if uf.union(p1, p2):
                _add_corridor(cand, "backbone")
                backbone_edges.append((p1, p2, cost, cand.get("distance_m", 1.0)))

    _log_message(f"  Backbone complete. Budget used: {budget_used:.2f}/{params.budget_area:.2f}")

    # ------------------------------------------------------------------
    # Phase 2a: Parallel Reinforcement (Optional)
    # ------------------------------------------------------------------
    def _pair_key(p1: int, p2: int) -> Tuple[int, int]:
        return (p1, p2) if p1 <= p2 else (p2, p1)

    def _overlap_ratio(g1: QgsGeometry, g2: QgsGeometry) -> float:
        try:
            a1 = g1.area()
            a2 = g2.area()
            denom = min(a1, a2)
            if denom <= 0:
                return 1.0
            inter = g1.intersection(g2)
            if inter is None or inter.isEmpty():
                return 0.0
            return inter.area() / denom
        except Exception:
            return 1.0

    if remaining > 0 and selected and backbone_edges:
        _log_message("--- HDFM Phase 2a: Reinforcing Existing Links ---")
        # Prefer spending remaining budget on redundant (spatially distinct) links
        # between already-backbone-connected patch pairs, rather than creating many
        # long-distance loops across the landscape.
        max_total_links_per_pair = 2  # 1 backbone + 1 redundant
        overlap_reject_ratio = 0.85

        backbone_pairs: Set[Tuple[int, int]] = {_pair_key(p1, p2) for p1, p2, _c, _d in backbone_edges}

        selected_geoms_by_pair: Dict[Tuple[int, int], List[QgsGeometry]] = defaultdict(list)
        for data in selected.values():
            p1 = data.get("p1")
            p2 = data.get("p2")
            if p1 is None or p2 is None:
                continue
            g = data.get("geom")
            if g is None or g.isEmpty():
                continue
            selected_geoms_by_pair[_pair_key(int(p1), int(p2))].append(g)

        reinforce_ranked: List[Tuple[float, float, Dict]] = []
        for cand in candidates:
            p1, p2 = cand.get("patch1"), cand.get("patch2")
            if p1 is None or p2 is None:
                continue
            pk = _pair_key(int(p1), int(p2))
            if pk not in backbone_pairs:
                continue
            if len(selected_geoms_by_pair.get(pk, [])) >= max_total_links_per_pair:
                continue
            cost = float(cand.get("area_ha", 0.0))
            if cost <= 0 or cost > remaining:
                continue
            g = cand.get("geom")
            if g is None or g.isEmpty():
                continue
            prior_geoms = selected_geoms_by_pair.get(pk, [])
            overlap = max((_overlap_ratio(g, pg) for pg in prior_geoms), default=0.0)
            if overlap >= overlap_reject_ratio:
                continue
            dist = float(cand.get("distance_m", 1.0))
            # Reward: low cost, low overlap, short distance.
            score = (1.0 - overlap) / (cost + 1e-9) * (1.0 / (dist + 1e-6))
            reinforce_ranked.append((score, overlap, cand))

        reinforce_ranked.sort(key=lambda t: t[0], reverse=True)

        reinforced = 0
        for _score, overlap, cand in reinforce_ranked:
            cost = float(cand.get("area_ha", 0.0))
            if cost > remaining:
                continue
            p1 = int(cand.get("patch1"))
            p2 = int(cand.get("patch2"))
            pk = _pair_key(p1, p2)
            if len(selected_geoms_by_pair.get(pk, [])) >= max_total_links_per_pair:
                continue
            _add_corridor(cand, "reinforcement")
            geom = cand.get("geom")
            if geom is not None:
                selected_geoms_by_pair.setdefault(pk, []).append(clone_geometry(geom))
            reinforced += 1
            if params.debug_enabled:
                _debug(
                    params,
                    f"Reinforced {p1}-{p2}: cost={cost:.4f}ha overlap={overlap:.2f} "
                    f"variant={cand.get('variant')} source={cand.get('source')}",
                )

        _log_message(f"  Added {reinforced} redundant link(s) across backbone pairs.")

    # ------------------------------------------------------------------
    # Phase 2: Strategic Loops (Optional)
    # ------------------------------------------------------------------
    if allow_loops and remaining > 0 and selected:
        _log_message("--- HDFM Phase 2: Adding Strategic Loops for Resilience ---")

        G_backbone = nx.Graph()
        for pid in patches:
            G_backbone.add_node(pid)
        for p1, p2, _, dist in backbone_edges:
            G_backbone.add_edge(p1, p2, weight=dist)

        loop_candidates = []
        for cand in candidates:
            p1, p2 = cand.get("patch1"), cand.get("patch2")
            if nx.has_path(G_backbone, p1, p2) and not G_backbone.has_edge(p1, p2):
                cost = cand.get("area_ha", 0.0)
                if cost <= remaining and cost > 0:
                    score = hdfm_math.score_edge_for_loops(G_backbone, p1, p2, cand.get("distance_m", 1.0))
                    efficiency = score / cost if cost else 0.0
                    loop_candidates.append((efficiency, cand))

        loop_candidates.sort(key=lambda x: x[0], reverse=True)

        loops_added = 0
        for _eff, cand in loop_candidates:
            cost = cand.get("area_ha", 0.0)
            if cost <= remaining:
                _add_corridor(cand, "strategic_loop")
                p1, p2 = cand.get("patch1"), cand.get("patch2")
                backbone_edges.append((p1, p2, cost, cand.get("distance_m", 1.0)))
                loops_added += 1

        _log_message(f"  Added {loops_added} strategic loops.")

    # ------------------------------------------------------------------
    # Phase 3: Metrics
    # ------------------------------------------------------------------
    final_corridors_list = []
    for cid, data in selected.items():
        final_corridors_list.append(
            {"id": cid, "patch1": data.get("p1"), "patch2": data.get("p2"), "distance_m": data.get("distance", 1.0)}
        )

    G_final = hdfm_math.build_graph_from_corridors(patches, final_corridors_list)
    entropy_stats = hdfm_math.calculate_total_entropy(G_final)
    rho2 = hdfm_math.calculate_two_edge_connectivity(G_final)

    stats = {
        "strategy": "hdfm_optimization",
        "corridors_used": len(selected),
        "budget_used_ha": budget_used,
        "patches_connected": G_final.number_of_nodes(),
        "hdfm_entropy_total": entropy_stats["H_total"],
        "hdfm_robustness_rho2": rho2,
        "hdfm_mode": "Largest Network" if mode == "largest_network" else "Most Connectivity",
        "total_connected_area_ha": sum(p["area_ha"] for p in patches.values()),
        "largest_group_area_ha": 0.0,
    }

    comps = list(nx.connected_components(G_final))
    if comps:
        max_area = max(sum(patches[pid]["area_ha"] for pid in comp) for comp in comps)
        stats["largest_group_area_ha"] = max_area

    for data in selected.values():
        connected_area = stats.get("largest_group_area_ha", 0.0)
        data["connected_area_ha"] = connected_area
        area = data.get("area_ha", 0.0)
        data["efficiency"] = (connected_area / area) if area else 0.0

    return selected, stats


def _thicken_corridors(
    corridors: Dict[int, Dict],
    remaining_budget: float,
    params: VectorRunParams,
    max_width_factor: float = 5.0,
) -> float:
    """
    Use remaining budget to widen existing corridors up to a max factor of the base width.
    Returns additional budget used.
    """
    if remaining_budget <= 0 or not corridors:
        return 0.0

    print(f"\n  Thickening corridors with remaining budget: {remaining_budget:.4f} ha")
    budget_used = 0.0
    max_buffer = (params.min_corridor_width * (max_width_factor - 1)) / 2.0
    buffer_step = params.min_corridor_width * 0.1  # grow by 10% of base width per pass
    current_buffer = 0.0

    while budget_used < remaining_budget and current_buffer < max_buffer:
        current_buffer = min(current_buffer + buffer_step, max_buffer)
        step_cost = 0.0
        temp_updates: Dict[int, Tuple[QgsGeometry, float]] = {}

        for cid, cdata in corridors.items():
            base_geom = cdata.get("_thicken_base_geom")
            if base_geom is None:
                base_geom = clone_geometry(cdata.get("geom"))
                cdata["_thicken_base_geom"] = base_geom
            if base_geom is None or base_geom.isEmpty():
                continue
            try:
                thickened_geom = base_geom.buffer(current_buffer, BUFFER_SEGMENTS)
            except Exception:
                continue
            if thickened_geom is None or thickened_geom.isEmpty():
                continue
            new_area = thickened_geom.area() / 10000.0
            old_area = float(cdata.get("area_ha", new_area))
            added_area = new_area - old_area
            if added_area <= 0:
                continue
            step_cost += added_area
            temp_updates[cid] = (thickened_geom, new_area)

        if budget_used + step_cost > remaining_budget or not temp_updates:
            print("  Budget exhausted or no further thickening possible; stopping.")
            break

        for cid, (geom, new_area) in temp_updates.items():
            corridors[cid]["geom"] = geom
            corridors[cid]["area_ha"] = new_area

        budget_used += step_cost
        print(f"    - Thickened by {current_buffer * 2:.1f} m (Cost: {step_cost:.4f} ha)")
        if current_buffer >= max_buffer:
            print(f"  Maximum corridor width reached ({max_width_factor}x min width).")
            break

    print(f"  ✓ Finalized thickening. Total extra budget used: {budget_used:.4f} ha")
    return budget_used


def write_corridors_layer_to_gpkg(
    corridors: Dict[int, Dict],
    output_path: str,
    layer_name: str,
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    unit_system: str,
    overwrite_file: bool = False,
) -> bool:
    print(f"\nWriting layer '{layer_name}' to {os.path.basename(output_path)} ...")
    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())

    is_imperial = unit_system == "imperial"
    area_field = "area_ac" if is_imperial else "area_ha"
    conn_field = "conn_area_ac" if is_imperial else "conn_area_ha"
    area_factor = 2.471053814 if is_imperial else 1.0

    fields = QgsFields()
    fields.append(QgsField("corridor_id", QVariant.Int))
    fields.append(QgsField("patch_ids", QVariant.String))
    fields.append(QgsField(area_field, QVariant.Double))
    fields.append(QgsField(conn_field, QVariant.Double))
    fields.append(QgsField("efficiency", QVariant.Double))
    fields.append(QgsField("multipart", QVariant.Bool))
    fields.append(QgsField("segment_count", QVariant.Int))

    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GPKG"
    save_options.fileEncoding = "UTF-8"
    save_options.layerName = layer_name
    save_options.actionOnExistingFile = (
        QgsVectorFileWriter.CreateOrOverwriteFile if overwrite_file else QgsVectorFileWriter.CreateOrOverwriteLayer
    )

    writer = QgsVectorFileWriter.create(
        output_path,
        fields,
        QgsWkbTypes.Polygon,
        original_crs,
        QgsProject.instance().transformContext(),
        save_options,
    )

    if writer.hasError() != QgsVectorFileWriter.NoError:
        print(f"  ✗ Error: {writer.errorMessage()}")
        return False

    for cid, cdata in corridors.items():
        feat = QgsFeature(fields)
        geom = clone_geometry(cdata["geom"])
        geom.transform(transform)
        feat.setGeometry(geom)
        multipart = geom.isMultipart()
        segment_count = geom.constGet().numGeometries() if multipart else 1
        feat.setAttributes(
            [
                cid,
                ",".join(map(str, sorted(cdata["patch_ids"]))),
                round(cdata["area_ha"] * area_factor, 4),
                round(cdata["connected_area_ha"] * area_factor, 2),
                round(cdata["efficiency"], 6),
                multipart,
                segment_count,
            ]
        )
        writer.addFeature(feat)

    del writer
    print(f"  ✓ Wrote {len(corridors)} corridors to layer '{layer_name}'")
    return True


def add_layer_to_qgis_from_gpkg(gpkg_path: str, layer_name: str) -> None:
    uri = f"{gpkg_path}|layername={layer_name}"
    layer = QgsVectorLayer(uri, layer_name, "ogr")
    if layer.isValid():
        QgsProject.instance().addMapLayer(layer)
        print(f"  ✓ Added '{layer_name}' to QGIS project")
    else:
        print(f"  ✗ Could not add '{layer_name}' from {gpkg_path}")


def create_step_stone_vector_layer(
    patches: Dict[int, Dict],
    component_map: Dict[int, int],
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    output_path: str,
    layer_name: str = "Step stone connectivity",
) -> bool:
    components: Dict[int, List[QgsGeometry]] = defaultdict(list)
    for pid, pdata in patches.items():
        comp_id = component_map.get(pid, pid)
        geom = pdata.get("geom")
        if geom and not geom.isEmpty():
            components[comp_id].append(clone_geometry(geom))

    layer = QgsVectorLayer(f"Polygon?crs={original_crs.authid()}", layer_name, "memory")
    provider = layer.dataProvider()
    provider.addAttributes(
        [
            QgsField("component_id", QVariant.Int),
            QgsField("patch_count", QVariant.Int),
            QgsField("area_ha", QVariant.Double),
        ]
    )
    layer.updateFields()

    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())
    features: List[QgsFeature] = []
    for comp_id, geom_list in components.items():
        merged_geom = geom_list[0] if len(geom_list) == 1 else QgsGeometry.unaryUnion(geom_list)
        if merged_geom is None or merged_geom.isEmpty():
            continue
        merged_geom.transform(transform)
        feat = QgsFeature(layer.fields())
        feat.setGeometry(merged_geom)
        feat.setAttributes([comp_id, len(geom_list), merged_geom.area() / 10000.0])
        features.append(feat)

    provider.addFeatures(features)
    layer.updateExtents()

    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GPKG"
    save_options.fileEncoding = "UTF-8"
    save_options.layerName = layer_name
    save_options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer

    writer = QgsVectorFileWriter.create(
        output_path,
        layer.fields(),
        QgsWkbTypes.Polygon,
        original_crs,
        QgsProject.instance().transformContext(),
        save_options,
    )
    if writer.hasError() != QgsVectorFileWriter.NoError:
        return False
    for feat in features:
        writer.addFeature(feat)
    del writer
    return True


def create_memory_layer_from_corridors(
    corridors: Dict[int, Dict],
    layer_name: str,
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    unit_system: str,
) -> Optional[QgsVectorLayer]:
    is_imperial = unit_system == "imperial"
    area_field = "area_ac" if is_imperial else "area_ha"
    conn_field = "conn_area_ac" if is_imperial else "conn_area_ha"
    area_factor = 2.471053814 if is_imperial else 1.0

    layer = QgsVectorLayer(f"Polygon?crs={original_crs.authid()}", layer_name, "memory")
    provider = layer.dataProvider()
    provider.addAttributes(
        [
            QgsField("corridor_id", QVariant.Int),
            QgsField("patch_ids", QVariant.String),
            QgsField(area_field, QVariant.Double),
            QgsField(conn_field, QVariant.Double),
            QgsField("efficiency", QVariant.Double),
            QgsField("multipart", QVariant.Bool),
            QgsField("segment_count", QVariant.Int),
        ]
    )
    layer.updateFields()

    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())

    features = []
    for cid, cdata in corridors.items():
        geom = clone_geometry(cdata["geom"])
        geom.transform(transform)
        multipart = geom.isMultipart()
        segment_count = geom.constGet().numGeometries() if multipart else 1
        feat = QgsFeature(layer.fields())
        feat.setGeometry(geom)
        feat.setAttributes(
            [
                cid,
                ",".join(map(str, sorted(cdata["patch_ids"]))),
                round(cdata["area_ha"] * area_factor, 4),
                round(cdata["connected_area_ha"] * area_factor, 2),
                round(cdata["efficiency"], 6),
                multipart,
                segment_count,
            ]
        )
        features.append(feat)

    provider.addFeatures(features)
    layer.updateExtents()
    QgsProject.instance().addMapLayer(layer)
    print(f"  ✓ Added temporary layer '{layer_name}' to QGIS project")
    return layer


def _convert_stats_for_units(stats: Dict, unit_system: str) -> Dict:
    factor = 2.471053814 if unit_system == "imperial" else 1.0
    label = "ac" if unit_system == "imperial" else "ha"
    converted = dict(stats)
    if "budget_used_ha" in stats:
        converted["budget_used_display"] = stats["budget_used_ha"] * factor
    if "total_connected_area_ha" in stats:
        converted["total_connected_area_display"] = stats["total_connected_area_ha"] * factor
    if "largest_group_area_ha" in stats:
        converted["largest_group_area_display"] = stats["largest_group_area_ha"] * factor
    if "seed_area_ha" in stats:
        converted["seed_area_display"] = stats["seed_area_ha"] * factor
    if "final_patch_area_ha" in stats:
        converted["final_patch_area_display"] = stats["final_patch_area_ha"] * factor
    if "total_patch_area_ha" in stats:
        converted["total_patch_area_display"] = stats["total_patch_area_ha"] * factor
    converted["area_units_label"] = label
    converted["conversion_factor"] = factor
    return converted


def run_vector_analysis(
    layer: QgsVectorLayer,
    output_dir: str,
    raw_params: Dict,
    strategy: str = "hdfm_most_connectivity",
    temporary: bool = False,
    iface=None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
) -> List[Dict]:
    """Execute the vector corridor analysis for the provided polygon layer."""
    if not isinstance(layer, QgsVectorLayer) or not layer.isValid():
        raise VectorAnalysisError("Selected layer is not a valid vector layer.")

    timings = TimingRecorder()

    params = _to_dataclass(raw_params)
    unit_system = params.unit_system
    area_factor = 2.471053814 if unit_system == "imperial" else 1.0
    area_label = "ac" if unit_system == "imperial" else "ha"

    if temporary:
        output_path = ""
    else:
        out_dir = output_dir or os.path.dirname(layer.source())
        os.makedirs(out_dir, exist_ok=True)
        output_path = os.path.join(out_dir, params.output_name)

    summary_dir = (
        os.path.dirname(output_path)
        if output_path
        else (output_dir or os.path.dirname(layer.source()) or os.getcwd())
    )
    if not summary_dir:
        summary_dir = os.getcwd()
    os.makedirs(summary_dir, exist_ok=True)

    overall_start = time.time()
    print("=" * 70)
    print("LINKSCAPE VECTOR ANALYSIS v23.3")
    print("=" * 70)
    print("\n1. Loading vector layer...")
    print(f"  ✓ Using layer: {layer.name()} ({layer.featureCount()} features)")
    emit_progress(progress_cb, 5, "Loading vector layer…")

    original_crs = layer.crs()
    print(f"  CRS: {original_crs.authid()}")

    print("\n2. Determining analysis CRS...")
    with timings.time_block("Determine analysis CRS"):
        target_crs = get_utm_crs_from_extent(layer)
        print(f"  ✓ Using {target_crs.authid()} for measurements")
        emit_progress(progress_cb, 15, "Preparing data…")

        # Auto-scale max search distance if user left it unset/zero
        try:
            extent_src = layer.extent()
            transform_extent = QgsCoordinateTransform(layer.crs(), target_crs, QgsProject.instance())
            extent = transform_extent.transformBoundingBox(extent_src)
            max_dimension = max(extent.width(), extent.height())
            DEFAULT_SCALING_FACTOR = 0.25
            if params.max_search_distance <= 0 and max_dimension > 0:
                params.max_search_distance = max_dimension * DEFAULT_SCALING_FACTOR
                _log_message(
                    f"Auto-setting max search distance to {DEFAULT_SCALING_FACTOR*100:.0f}% of map extent "
                    f"({params.max_search_distance:.1f} units)."
                )
        except Exception:
            pass

    print("\n3. Loading patches...")
    with timings.time_block("Load patches and spatial index"):
        patches, spatial_index = load_and_prepare_patches(layer, target_crs, params)
    if not patches:
        raise VectorAnalysisError("No patches found after filtering.")
    patch_union: Optional[QgsGeometry] = None
    with timings.time_block("Build patch union"):
        try:
            patch_union = QgsGeometry.unaryUnion([p["geom"] for p in patches.values()])
        except Exception:
            patch_union = None
    merge_adjacency: Dict[int, Set[int]] = {}
    hop_adjacency: Dict[int, Set[int]] = {}
    component_map: Dict[int, int] = {pid: pid for pid in patches}
    component_sizes: Dict[int, float] = {pid: pdata["area_ha"] for pid, pdata in patches.items()}
    component_counts: Dict[int, int] = {pid: 1 for pid in patches}
    merge_display = params.merge_distance * 3.280839895 if unit_system == "imperial" else params.merge_distance
    hop_display = params.hop_distance * 3.280839895 if unit_system == "imperial" else params.hop_distance
    dist_label = "ft" if unit_system == "imperial" else "m"
    stats_step: Dict[str, float] = {}

    if params.merge_distance > 0:
        with timings.time_block("Patch merge preprocessing"):
            emit_progress(progress_cb, 28, "Checking patch merge distance…")
            print("\n3b. Patch merge preprocessing...")
            merge_adjacency = _compute_vector_hop_adjacency(patches, spatial_index, params.merge_distance)
            if merge_adjacency:
                component_map, component_sizes, component_counts = _build_virtual_components(patches, merge_adjacency)
                print(
                    f"  ✓ Virtual merges: {len(component_sizes):,} starting components "
                    f"(merge distance ≤ {merge_display:.2f} {dist_label})"
                )
                original_patches = len(patches)
                virtual_components = len(component_sizes)
                patches_merged = original_patches - virtual_components
                stats_step = {
                    "original_patches": original_patches,
                    "virtual_components": virtual_components,
                    "patches_merged": patches_merged,
                    "merge_distance_m": params.merge_distance,
                }
                # Optional visual layer: only produce it when stepping is enabled (keeps outputs minimal).
                if params.stepping_enabled:
                    try:
                        if not temporary and output_path:
                            with timings.time_block("Create merge connectivity layer"):
                                step_success = create_step_stone_vector_layer(
                                    patches,
                                    component_map,
                                    target_crs,
                                    original_crs,
                                    output_path,
                                    "Merge connectivity (Within merge distance)",
                                )
                                if step_success:
                                    add_layer_to_qgis_from_gpkg(
                                        output_path, "Merge connectivity (Within merge distance)"
                                    )
                                    print("  ✓ Created merge connectivity layer")
                    except Exception as step_exc:  # noqa: BLE001
                        print(f"  ⚠ Could not create merge connectivity layer: {step_exc}")
            else:
                print("  ✓ No patches within merge distance; proceeding without virtual merges.")
    else:
        timings.add("Patch merge preprocessing (disabled)", 0.0)

    # Hop adjacency is only used for stepping-chain corridor logic; keep it disabled unless stepping is enabled.
    hop_adjacency = merge_adjacency if params.stepping_enabled and params.merge_distance > 0 else {}
    emit_progress(progress_cb, 35, "Searching for corridor candidates…")

    navigator: Optional[RasterNavigator] = None
    obstacle_layers: List[QgsVectorLayer] = []
    skipped_ids: List[str] = []
    if params.obstacle_enabled and params.obstacle_layer_ids:
        with timings.time_block("Impassable preparation"):
            for layer_id in params.obstacle_layer_ids:
                layer = QgsProject.instance().mapLayer(layer_id)
                if isinstance(layer, QgsVectorLayer) and layer.isValid() and QgsWkbTypes.geometryType(layer.wkbType()) == QgsWkbTypes.PolygonGeometry:
                    obstacle_layers.append(layer)
                else:
                    skipped_ids.append(str(layer_id))

            if skipped_ids:
                print(
                    f"  ⚠ Skipped {len(skipped_ids)} impassable layer(s) that are unavailable or not polygon geometry."
                )

            if obstacle_layers:
                try:
                    navigator = RasterNavigator(patches, obstacle_layers, target_crs, params)
                    print(
                        f"  ✓ Raster navigator grid: {navigator.cols} × {navigator.rows} cells "
                        f"@ {navigator.resolution:.1f} units "
                        f"using {len(obstacle_layers)} impassable layer(s)"
                    )
                except VectorAnalysisError as exc:
                    print(f"  ⚠ Impassable land classes disabled: {exc}")
                    navigator = None
            else:
                print("  ⚠ Selected impassable layers are unavailable; continuing without impassable land classes.")
    else:
        timings.add("Impassable preparation (disabled)", 0.0)

    hole_patches: Dict[int, Dict] = {}
    if (
        params.stepping_enabled
        and params.hop_distance > 0
        and params.obstacle_enabled
        and obstacle_layers
    ):
        with timings.time_block("Extract impassable holes"):
            hole_patches = _extract_obstacle_holes(obstacle_layers, target_crs)

    print("\n4. Precomputing candidate corridors...")
    all_possible = find_all_possible_corridors(
        patches,
        spatial_index,
        params,
        hop_adjacency=hop_adjacency,
        patch_union=patch_union,
        progress_cb=progress_cb,
        progress_start=35,
        progress_end=60,
        navigator=navigator,
        timings=timings,
    )

    if params.stepping_enabled and params.hop_distance > 0 and hole_patches:
        with timings.time_block("Generate hole-hopping candidates"):
            hole_candidates = _generate_obstacle_hopping_candidates(
                patches,
                hole_patches,
                params.hop_distance,
                params,
                obstacle_geoms=navigator.obstacle_geoms if navigator else [],
            )
            if hole_candidates:
                all_possible.extend(hole_candidates)

    with timings.time_block("Annotate candidate components"):
        _annotate_candidates_with_components(all_possible, component_map)
    with timings.time_block("Build component pair index"):
        direct_pairs = _build_component_pair_index(all_possible)
    with timings.time_block("Build hop edge lookup"):
        hop_lookup = _build_hop_edge_lookup(all_possible, hop_adjacency)
    with timings.time_block("Build component adjacency"):
        adjacency_map = _build_component_adjacency(all_possible)
    if params.stepping_enabled and params.hop_distance > 0:
        with timings.time_block("Build stepping-stone chains"):
            chains = _build_stepping_chains(hop_lookup, component_map, component_sizes)
        if chains:
            print(f"  ✓ Added {len(chains):,} stepping-stone corridor chains")
            all_possible.extend(chains)
    else:
        timings.add("Build stepping-stone chains (disabled)", 0.0)

    strategy = (strategy or "hdfm_most_connectivity").lower()
    if strategy not in ("hdfm_most_connectivity", "hdfm_largest_network"):
        raise VectorAnalysisError(
            f"Unsupported strategy '{strategy}'. Only HDFM strategies are supported in this version."
        )

    print("\nTOP 20 CANDIDATES BY COST (after hop bonus):")
    sorted_cands = sorted(all_possible, key=lambda x: x.get("area_ha", 999))
    for i, c in enumerate(sorted_cands[:20]):
        chain = " [CHAIN]" if c.get("is_stepping_chain") else ""
        hop_flag = " [HOP]" if c.get("is_hop_edge") else ""
        cost = c.get("area_ha", 0.0)
        orig = c.get("original_area_ha", cost)
        comps = f"{c.get('comp1')}-{c.get('comp2')}"
        patch_count = len(c.get("patch_ids", {c.get('patch1'), c.get('patch2')}))
        print(
            f"  {i+1:2d}. {cost:8.5f} ha{chain}{hop_flag}  comp {comps}  via {patch_count} patches  (orig {orig:.3f})"
        )

    print("\n5. Running optimization...")
    print("=" * 70)
    print(f"--- {strategy.replace('_', ' ').upper()} ---")

    if not all_possible:
        raise VectorAnalysisError(
            _format_no_corridor_reason(
                "Precomputation",
                len(patches),
                len(all_possible),
                params,
            )
    )

    # 5. Optimization
    emit_progress(progress_cb, 60, "Optimizing network...")
    strategy_key = strategy.lower().replace(" ", "_")

    # --- UPDATED DISPATCH LOGIC ---
    opt_label = f"Optimization ({strategy_key})"
    with timings.time_block(opt_label):
        if strategy_key == "hdfm_most_connectivity":
            corridors, stats = optimize_hdfm_strategy(patches, all_possible, params, allow_loops=True, mode="resilient")
            layer_name = "Corridors (HDFM Most Connectivity)"

        elif strategy_key == "hdfm_largest_network":
            corridors, stats = optimize_hdfm_strategy(patches, all_possible, params, allow_loops=True, mode="largest_network")
            layer_name = "Corridors (HDFM Largest Network)"
        else:
            raise VectorAnalysisError(f"Unsupported strategy '{strategy_key}'.")

    if not corridors:
        raise VectorAnalysisError(
            _format_no_corridor_reason(
                "Optimization",
                len(patches),
                len(all_possible),
                params,
            )
        )

    if params.stepping_enabled and (params.merge_distance > 0 or params.hop_distance > 0):
        stats["note"] = (
            f"Stepping enabled (merge {merge_display:.2f} {dist_label}, hop {hop_display:.2f} {dist_label})"
        )
        if params.merge_distance > 0:
            stats["patches_merged"] = len(patches) - len(component_sizes)
            if stats_step:
                stats["step_stone_stats"] = stats_step

    timings.add("Thicken corridors (removed)", 0.0)

    stats = _convert_stats_for_units(stats, unit_system)
    stats["budget_total_display"] = params.budget_area * area_factor

    print("  Preparing outputs...")
    emit_progress(progress_cb, 90, "Writing outputs…")
    with timings.time_block("Write corridor outputs"):
        if temporary:
            create_memory_layer_from_corridors(corridors, layer_name, target_crs, original_crs, unit_system)
        else:
            write_corridors_layer_to_gpkg(
                corridors,
                output_path,
                layer_name,
                target_crs,
                original_crs,
                unit_system,
                overwrite_file=True,
            )
            add_layer_to_qgis_from_gpkg(output_path, layer_name)

    elapsed = time.time() - overall_start
    emit_progress(progress_cb, 100, "Vector analysis complete.")

    print("\n" + "=" * 70)
    print("FINAL SUMMARY")
    print("=" * 70)
    strategy_key = (strategy or "hdfm_most_connectivity").lower()
    strategy_label = (
        "HDFM Most Connectivity"
        if strategy_key == "hdfm_most_connectivity"
        else "HDFM Largest Network"
    )
    print(f"Strategy:          {strategy_label}")
    print(f"Corridors created: {stats.get('corridors_used', 0)}")
    print(f"Connections:       {stats.get('connections_made', 0)}")
    print(f"Total connected:   {stats.get('total_connected_area_display', 0):.2f} {area_label}")
    print(f"Largest group:     {stats.get('largest_group_area_display', 0):.2f} {area_label}")
    if "redundant_links" in stats:
        print(f"Redundant links:   {stats.get('redundant_links', 0)}")
    if "avg_degree" in stats:
        print(f"Average degree:    {stats.get('avg_degree', 0):.2f}")
    if "step_stone_stats" in stats:
        ss = stats["step_stone_stats"]
        print(
            f"Step stone merge:  {ss.get('patches_merged', 0)} patches → {ss.get('virtual_components', 0)} components "
            f"(merge {ss.get('merge_distance_m', ss.get('merge_distance_px', 0))})"
        )
    print(
        f"Budget used:      {stats.get('budget_used_display', 0):.2f} / {params.budget_area * area_factor:.2f} {area_label}"
    )
    print(f"Processing time:   {elapsed:.1f}s")
    if temporary:
        print("Output:            Temporary layer (memory)")
    else:
        print(f"Output GPKG:       {output_path}")
    print("=" * 70)

    # Do not write/open extra logs or reports automatically; keep results in the project layers only.

    stats["layer_name"] = layer_name
    stats["output_path"] = output_path if not temporary else ""
    stats["unit_system"] = unit_system

    return [
        {
            "strategy": strategy,
            "stats": stats,
            "output_path": output_path if not temporary else "",
            "layer_name": layer_name,
        }
    ]
