"""
Linkscape Corridor Analysis - Raster Workflow (v23.0)
-----------------------------------------------------
Runs the selected raster optimization workflow for corridor analysis.

The logic is adapted from the standalone raster script and packaged so it can
be invoked from the QGIS plugin with user-supplied parameters.
"""

from __future__ import annotations

import heapq
import math
import os
import random
import tempfile
import time
from collections import defaultdict
from dataclasses import dataclass, field
from typing import Callable, Dict, List, Optional, Set, Tuple

import numpy as np
# NumPy 2.x removed np.int; add shim for legacy references.
if not hasattr(np, "int"):  # pragma: no cover
    np.int = int  # type: ignore[attr-defined]
try:
    from scipy import ndimage

    HAS_NDIMAGE = True
except Exception:  # pragma: no cover
    ndimage = None  # type: ignore
    HAS_NDIMAGE = False
try:
    from osgeo import gdal
except ImportError:  # pragma: no cover
    gdal = None  # type: ignore

try:
    from qgis.core import (
        Qgis,
        QgsApplication,
        QgsFeature,
        QgsField,
        QgsProject,
        QgsRasterLayer,
        QgsVectorLayer,
    )
except ImportError:  # pragma: no cover
    Qgis = None  # type: ignore
    QgsApplication = None  # type: ignore
    QgsFeature = None  # type: ignore
    QgsField = None  # type: ignore
    QgsProject = None  # type: ignore
    QgsRasterLayer = None  # type: ignore
    QgsVectorLayer = None  # type: ignore

from PyQt5.QtCore import QVariant

# Optional OpenCV import for faster connected component labeling
try:
    import cv2

    HAS_CV2 = True
except ImportError:
    HAS_CV2 = False

# Import HDFM math library for entropy/robustness calculations
try:
    from . import hdfm_math
    import networkx as nx

    HAS_HDFM = True
except ImportError:
    hdfm_math = None  # type: ignore
    nx = None  # type: ignore
    HAS_HDFM = False

GTIFF_OPTIONS = ["COMPRESS=LZW", "TILED=YES", "BIGTIFF=IF_SAFER"]

PIXEL_COUNT_WARNING_THRESHOLD = 40_000_000  # warn when raster exceeds ~40 million pixels
PIXEL_SIZE_WARNING_THRESHOLD = 10.0  # warn when pixel size < 10 map units
PIXEL_FINE_CRITICAL_COUNT = 2_000_000  # only block fine rasters when they are also large

def _log_message(message: str, level: int = Qgis.Info) -> None:
    """Log to the QGIS Log Messages Panel with a Linkscape tag."""
    try:
        QgsApplication.messageLog().logMessage(message, "Linkscape", level)
    except Exception:
        print(f"Linkscape Log: {message}")


def _debug_raster(params: RasterRunParams, message: str) -> None:
    if not getattr(params, "debug_enabled", False):
        return
    try:
        params.debug_lines.append(message)
    except Exception:
        pass


def _write_raster_debug_report(
    params: RasterRunParams,
    summary_dir: str,
    layer: QgsRasterLayer,
    strategy_key: str,
) -> None:
    if not getattr(params, "debug_enabled", False) or not params.debug_lines:
        return
    try:
        suffix = random.randint(1000, 9999)
        name = f"linkscape_raster_debug_{strategy_key}_{suffix}.txt"
        path = os.path.join(summary_dir, name)
        with open(path, "w", encoding="utf-8") as fh:
            fh.write("Linkscape Raster Debug Log\n")
            fh.write("=" * 30 + "\n")
            fh.write(f"Input layer: {layer.name()}\n")
            fh.write(f"Strategy: {strategy_key}\n")
            fh.write(f"Max search distance: {params.max_search_distance}\n")
            fh.write(f"Min corridor width: {params.min_corridor_width}\n")
            fh.write(f"Stepping enabled: {params.stepping_enabled}\n")
            fh.write(f"Hop distance: {params.hop_distance}\n")
            fh.write(f"Impassable enabled: {params.obstacle_enabled}\n\n")
            fh.write("Messages:\n")
            for line in params.debug_lines:
                fh.write(f"- {line}\n")
        print(f"  ✓ Debug report saved: {path}")
    except Exception as exc:  # noqa: BLE001
        print(f"  ⚠ Could not write raster debug report: {exc}")


def _emit_progress(
    progress_cb: Optional[Callable[[int, Optional[str]], None]],
    value: float,
    message: Optional[str] = None,
) -> None:
    if progress_cb is None:
        return
    try:
        progress_cb(int(max(0, min(100, value))), message)
    except Exception:
        pass


class RasterAnalysisError(RuntimeError):
    """Raised when the raster analysis cannot be completed."""


@dataclass
class RasterRunParams:
    patch_connectivity: int
    patch_mode: str
    patch_values: List[float]
    range_lower: Optional[float]
    range_upper: Optional[float]
    obstacle_enabled: bool
    obstacle_mode: str
    obstacle_values: List[float]
    obstacle_range_lower: Optional[float]
    obstacle_range_upper: Optional[float]
    value_tolerance: float
    nodata_fallback: float
    min_patch_size: int
    budget_pixels: int
    max_search_distance: int
    max_corridor_area: Optional[int]
    min_corridor_width: int
    allow_bottlenecks: bool
    stepping_enabled: bool = False
    hop_distance: int = 0
    debug_enabled: bool = False
    debug_lines: List[str] = field(default_factory=list, repr=False)


class UnionFind:
    """Union-Find data structure for tracking connected components."""

    def __init__(self):
        self.parent: Dict[int, int] = {}
        self.size: Dict[int, int] = {}
        self.count: Dict[int, int] = {}

    def find(self, x: int) -> int:
        if x not in self.parent:
            self.parent[x] = x
            self.size[x] = 0
            self.count[x] = 0
        if self.parent[x] != x:
            self.parent[x] = self.find(self.parent[x])
        return self.parent[x]

    def union(self, a: int, b: int) -> int:
        ra, rb = self.find(a), self.find(b)
        if ra == rb:
            return ra
        sa, sb = self.size.get(ra, 0), self.size.get(rb, 0)
        if sa < sb:
            ra, rb = rb, ra
            sa, sb = sb, sa
        self.parent[rb] = ra
        self.size[ra] = sa + sb
        self.count[ra] = self.count.get(ra, 0) + self.count.get(rb, 0)
        return ra

    def get_size(self, x: int) -> int:
        return self.size.get(self.find(x), 0)

    def get_count(self, x: int) -> int:
        return self.count.get(self.find(x), 0)


def label_components_numpy(binary_array: np.ndarray, connectivity: int = 8) -> Tuple[np.ndarray, int]:
    """Label connected components using numpy (no external dependencies)."""
    rows, cols = binary_array.shape
    uf = UnionFind()
    if connectivity == 4:
        neighbors = [(-1, 0), (0, -1)]
    else:
        neighbors = [(-1, -1), (-1, 0), (-1, 1), (0, -1)]

    for i in range(rows):
        for j in range(cols):
            if binary_array[i, j]:
                cur = (i, j)
                uf.find(cur)
                for di, dj in neighbors:
                    ni, nj = i + di, j + dj
                    if 0 <= ni < rows and 0 <= nj < cols and binary_array[ni, nj]:
                        uf.union(cur, (ni, nj))

    root_to_label: Dict[Tuple[int, int], int] = {}
    next_label = 1
    labeled = np.zeros_like(binary_array, dtype=np.int32)
    for i in range(rows):
        for j in range(cols):
            if binary_array[i, j]:
                root = uf.find((i, j))
                if root not in root_to_label:
                    root_to_label[root] = next_label
                    next_label += 1
                labeled[i, j] = root_to_label[root]

    return labeled, next_label - 1


def label_components_opencv(binary_array: np.ndarray, connectivity: int = 8) -> Tuple[np.ndarray, int]:
    """Label connected components using OpenCV when available."""
    cv_conn = 4 if connectivity == 4 else 8
    n_labels, labeled = cv2.connectedComponents(binary_array.astype(np.uint8), connectivity=cv_conn)
    return labeled.astype(np.int32), n_labels - 1


def label_patches(binary_array: np.ndarray, connectivity: int = 8) -> Tuple[np.ndarray, int]:
    """Label connected components using the fastest available approach."""
    if HAS_CV2:
        print("  Using OpenCV for labeling...")
        return label_components_opencv(binary_array, connectivity)
    print("  Using numpy for labeling...")
    return label_components_numpy(binary_array, connectivity)


def read_band(
    band: gdal.Band,
    rows: int,
    cols: int,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
    progress_start: int = 0,
    progress_end: int = 10,
) -> np.ndarray:
    """Read a raster band as a numpy array with incremental progress updates."""
    data = np.empty((rows, cols), dtype=np.float32)
    chunk_rows = max(1, min(1024, rows // 50 or 1))
    span = max(progress_end - progress_start, 1)

    for start_row in range(0, rows, chunk_rows):
        this_rows = min(chunk_rows, rows - start_row)
        buf = band.ReadRaster(0, start_row, cols, this_rows, cols, this_rows, gdal.GDT_Float32)
        if not buf:
            raise RasterAnalysisError("Failed to read raster data chunk.")
        arr = np.frombuffer(buf, dtype=np.float32, count=cols * this_rows).reshape(this_rows, cols)
        data[start_row : start_row + this_rows] = arr

        if progress_cb is not None:
            ratio = (start_row + this_rows) / max(rows, 1)
            progress_value = progress_start + ratio * span
            _emit_progress(progress_cb, progress_value, "Reading raster data…")

    return data


def write_raster(path: str, arr: np.ndarray, gt: Tuple[float, ...], proj: str, nodata: float = 0) -> None:
    """Write a numpy array out to GeoTIFF."""
    rows, cols = arr.shape
    drv = gdal.GetDriverByName("GTiff")
    ds = drv.Create(path, cols, rows, 1, gdal.GDT_Int32, options=GTIFF_OPTIONS)
    if ds is None:
        raise RasterAnalysisError(f"Unable to create output dataset: {path}")
    ds.SetGeoTransform(gt)
    ds.SetProjection(proj)
    band = ds.GetRasterBand(1)
    band.SetNoDataValue(int(nodata))

    if arr.dtype != np.int32:
        arr = arr.astype(np.int32)

    buf = np.ascontiguousarray(arr).tobytes()
    band.WriteRaster(0, 0, cols, rows, buf, cols, rows)
    band.FlushCache()
    ds = None


def define_habitat(data: np.ndarray, nodata_mask: np.ndarray, params: RasterRunParams) -> np.ndarray:
    """Identify patch pixels based on the selected configuration."""
    valid = ~nodata_mask
    patches = np.zeros(data.shape, dtype=np.uint8)
    mode = params.patch_mode.lower()
    tol = params.value_tolerance

    if mode == "value" and params.patch_values:
        for val in params.patch_values:
            patches |= (np.abs(data - val) < tol) & valid
        print(f"  Patch = values {params.patch_values}")
    elif mode == "range" and params.range_lower is not None and params.range_upper is not None:
        patches = ((data >= params.range_lower) & (data <= params.range_upper)) & valid
        print(
            "  Patch = range "
            f"{params.range_lower:.4f} - {params.range_upper:.4f}"
        )
    else:
        raise RasterAnalysisError("Patch configuration did not yield any valid pixels.")

    return patches


def define_obstacles(data: np.ndarray, nodata_mask: np.ndarray, patch_mask: np.ndarray, params: RasterRunParams) -> np.ndarray:
    """Create a boolean mask for impassable pixels corridors must avoid."""
    if not params.obstacle_enabled:
        return np.zeros(data.shape, dtype=bool)

    mask = np.zeros(data.shape, dtype=bool)
    tol = params.value_tolerance
    mode = params.obstacle_mode.lower()

    if mode == "range" and params.obstacle_range_lower is not None and params.obstacle_range_upper is not None:
        lower = min(params.obstacle_range_lower, params.obstacle_range_upper)
        upper = max(params.obstacle_range_lower, params.obstacle_range_upper)
        mask = (data >= lower) & (data <= upper)
    elif mode == "value" and params.obstacle_values:
        for val in params.obstacle_values:
            mask |= np.abs(data - val) < tol
    else:
        return np.zeros(data.shape, dtype=bool)

    mask &= ~nodata_mask
    return mask


def find_shortest_corridor(
    start_patches: Set[int],
    labels: np.ndarray,
    habitat: np.ndarray,
    max_width: int,
    connectivity: int,
    obstacle_mask: Optional[np.ndarray] = None,
    passable_mask: Optional[np.ndarray] = None,
    params: Optional[RasterRunParams] = None,
) -> List[Tuple[frozenset, int, float]]:
    """
    Dijkstra search to find shortest corridors connecting start_patches to other patches.
    Returns a list of (path_pixels, target_patch, length).
    """
    rows, cols = labels.shape
    if connectivity == 4:
        moves = [(-1, 0), (1, 0), (0, -1), (0, 1)]
    else:
        moves = [
            (-1, -1),
            (-1, 0),
            (-1, 1),
            (0, -1),
            (0, 1),
            (1, -1),
            (1, 0),
            (1, 1),
        ]

    hop_limit = 0
    hop_enabled = False
    if params is not None:
        hop_enabled = bool(getattr(params, "stepping_enabled", False) and getattr(params, "hop_distance", 0) > 0)
        hop_limit = int(getattr(params, "hop_distance", 0) or 0)

    # Track start cells as (r, c, initial_hop_used, include_in_path)
    start_positions: List[Tuple[int, int, int, bool]] = []
    for i in range(rows):
        for j in range(cols):
            if not habitat[i, j]:
                is_obstacle = bool(obstacle_mask is not None and obstacle_mask[i, j])
                if is_obstacle:
                    if not hop_enabled or hop_limit < 1:
                        continue
                else:
                    if passable_mask is not None and not passable_mask[i, j]:
                        continue
                for di, dj in moves:
                    ni, nj = i + di, j + dj
                    if 0 <= ni < rows and 0 <= nj < cols and labels[ni, nj] in start_patches:
                        start_positions.append((i, j, 1 if is_obstacle else 0, not is_obstacle))
                        break

    if params and getattr(params, "debug_enabled", False):
        patch_label = next(iter(start_patches)) if start_patches else -1
        obstacle_starts = sum(1 for _, _, hop_used, include in start_positions if hop_used > 0 or not include)
        _debug_raster(
            params,
            f"Patch {patch_label}: start positions={len(start_positions)} (impassable_starts={obstacle_starts})",
        )
    if not start_positions:
        if params and getattr(params, "debug_enabled", False):
            patch_label = next(iter(start_patches)) if start_patches else -1
            _debug_raster(params, f"Patch {patch_label}: no valid step cells (adjacent passable).")
        return []

    heap: List[Tuple[float, int, int, int, frozenset]] = []
    best_cost: Dict[Tuple[int, int, int], float] = {}
    results: List[Tuple[frozenset, int, float]] = []
    visited_targets: Set[int] = set()
    for r, c, init_hop, include_in_path in start_positions:
        is_obstacle = bool(obstacle_mask is not None and obstacle_mask[r, c])
        if is_obstacle and (not hop_enabled or init_hop > hop_limit):
            continue
        if (not is_obstacle) and passable_mask is not None and not passable_mask[r, c]:
            continue
        path = frozenset({(r, c)}) if include_in_path else frozenset()
        heapq.heappush(heap, (0.0, init_hop, r, c, path))
        best_cost[(r, c, init_hop)] = 0.0

    while heap:
        cost, hop_used, r, c, path = heapq.heappop(heap)
        if cost > max_width:
            continue

        for dr, dc in moves:
            nr, nc = r + dr, c + dc
            if 0 <= nr < rows and 0 <= nc < cols:
                lbl = labels[nr, nc]
                if lbl > 0 and lbl not in start_patches and lbl not in visited_targets:
                    visited_targets.add(lbl)
                    results.append((path, lbl, cost))
                    continue

                is_obstacle = bool(obstacle_mask is not None and obstacle_mask[nr, nc])
                if is_obstacle:
                    if not hop_enabled:
                        continue
                    next_hop = hop_used + 1
                    if next_hop > hop_limit:
                        continue
                else:
                    next_hop = 0

                if passable_mask is not None and not passable_mask[nr, nc] and not is_obstacle:
                    continue
                if habitat[nr, nc]:
                    if lbl == 0:
                        move_cost = 0.0
                        new_path = path  # do not draw corridors over stepping-stone habitat
                        next_hop = 0
                    else:
                        continue
                else:
                    move_cost = math.sqrt(2) if dr != 0 and dc != 0 else 1.0
                    if is_obstacle:
                        new_path = path  # do not draw corridors over impassables; treat as hop gap
                    else:
                        new_path = path | frozenset({(nr, nc)})
                new_cost = cost + move_cost
                if new_cost > max_width:
                    continue
                prev_best = best_cost.get((nr, nc, next_hop))
                if prev_best is not None and prev_best <= new_cost:
                    continue
                best_cost[(nr, nc, next_hop)] = new_cost
                heapq.heappush(heap, (new_cost, next_hop, nr, nc, new_path))

    if params and getattr(params, "debug_enabled", False):
        patch_label = next(iter(start_patches)) if start_patches else -1
        _debug_raster(params, f"Patch {patch_label}: corridor results={len(results)}")
    return results


def find_all_possible_corridors(
    labels: np.ndarray,
    habitat: np.ndarray,
    patch_sizes: Dict[int, int],
    max_width: int,
    max_area: Optional[int],
    connectivity: int,
    min_corridor_width: int,
    obstacle_mask: Optional[np.ndarray] = None,
    passable_mask: Optional[np.ndarray] = None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
    progress_start: int = 45,
    progress_end: int = 75,
    params: Optional[RasterRunParams] = None,
) -> List[Dict]:
    """Find all possible corridors between patch pairs."""
    print("  Finding all possible corridors...")
    rows, cols = labels.shape
    all_corridors: List[Dict] = []
    processed_pairs: Set[frozenset] = set()
    offsets = _corridor_offsets(min_corridor_width)

    unique_patches = [p for p in patch_sizes.keys() if p > 0]
    total = len(unique_patches) or 1
    span = max(progress_end - progress_start, 1)
    for idx, patch_id in enumerate(unique_patches):
        if (idx + 1) % 10 == 0:
            print(f"    Analyzing patch {idx + 1}/{len(unique_patches)}...", end="\r")
        if progress_cb is not None:
            pre_value = progress_start + ((idx + 0.25) / total) * span
            _emit_progress(
                progress_cb,
                pre_value,
                f"Analyzing patch {idx + 1}/{total}…",
            )

        results = find_shortest_corridor(
            {patch_id},
            labels,
            habitat,
            max_width,
            connectivity,
            obstacle_mask=obstacle_mask,
            passable_mask=passable_mask,
            params=params,
        )
        if not results:
            continue

        for path_pixels, target_id, path_len in results:
            pair = frozenset({patch_id, target_id})
            if pair in processed_pairs:
                continue
            processed_pairs.add(pair)

            if max_area is not None and path_len > max_area:
                continue

            buffered = _inflate_corridor_pixels(set(path_pixels), offsets, rows, cols, obstacle_mask=obstacle_mask)
            area_px = len(buffered)

            all_corridors.append(
                {
                    "patch1": patch_id,
                    "patch2": target_id,
                    "pixels": path_pixels,
                    "length": path_len,
                    "area": area_px,
                    "buffered_pixels": frozenset(buffered),
                }
            )
        if progress_cb is not None:
            post_value = progress_start + ((idx + 1) / total) * span
            _emit_progress(
                progress_cb,
                post_value,
                f"Finished patch {idx + 1}/{total}",
            )

    _emit_progress(progress_cb, progress_end, "Corridor candidates ready.")
    print(f"\n  ✓ Found {len(all_corridors)} possible corridors")
    return all_corridors


def _corridor_offsets(min_corridor_width: int) -> List[Tuple[int, int]]:
    """Precompute offsets used to inflate corridors to the minimum width."""
    width = max(1, int(min_corridor_width))
    if width <= 1:
        return [(0, 0)]

    radius = max(0.0, width / 2.0)
    max_offset = int(math.ceil(radius))
    radius_sq = radius * radius

    offsets: List[Tuple[int, int]] = []
    for dr in range(-max_offset, max_offset + 1):
        for dc in range(-max_offset, max_offset + 1):
            if dr * dr + dc * dc <= radius_sq + 1e-9:
                offsets.append((dr, dc))

    if not offsets:
        offsets.append((0, 0))
    return offsets


def _inflate_corridor_pixels(
    pixels: Set[Tuple[int, int]],
    offsets: List[Tuple[int, int]],
    rows: int,
    cols: int,
    obstacle_mask: Optional[np.ndarray] = None,
) -> Set[Tuple[int, int]]:
    """Apply offsets to centerline pixels, respecting impassables when provided."""
    inflated: Set[Tuple[int, int]] = set()
    use_mask = obstacle_mask is not None

    for r, c in pixels:
        for dr, dc in offsets:
            nr, nc = r + dr, c + dc
            if 0 <= nr < rows and 0 <= nc < cols:
                if use_mask and obstacle_mask[nr, nc]:
                    continue
                inflated.add((nr, nc))
    return inflated


def _shift_mask(mask: np.ndarray, dr: int, dc: int) -> np.ndarray:
    """Return a shifted copy of mask aligned so (r, c) reads (r+dr, c+dc) from original."""
    rows, cols = mask.shape
    shifted = np.zeros_like(mask, dtype=bool)

    if abs(dr) >= rows or abs(dc) >= cols:
        return shifted

    if dr >= 0:
        src_r = slice(dr, rows)
        dst_r = slice(0, rows - dr)
    else:
        src_r = slice(0, rows + dr)
        dst_r = slice(-dr, rows)

    if dc >= 0:
        src_c = slice(dc, cols)
        dst_c = slice(0, cols - dc)
    else:
        src_c = slice(0, cols + dc)
        dst_c = slice(-dc, cols)

    shifted[dst_r, dst_c] = mask[src_r, src_c]
    return shifted


def _erode_mask(mask: np.ndarray, offsets: List[Tuple[int, int]]) -> np.ndarray:
    """Morphologically erode a mask using the provided offsets."""
    if not offsets:
        return mask.copy()

    eroded = mask.copy()
    for dr, dc in offsets:
        if dr == 0 and dc == 0:
            continue
        shifted = _shift_mask(mask, dr, dc)
        eroded &= shifted
        if not eroded.any():
            break
    return eroded


def _build_passable_mask(
    habitat: np.ndarray,
    obstacle_mask: np.ndarray,
    min_corridor_width: int,
    allow_bottlenecks: bool,
    stepping_mask: Optional[np.ndarray] = None,
) -> np.ndarray:
    """Derive which pixels can host corridor centerlines under width constraints."""
    habitat_bool = habitat.astype(bool)
    obstacle_bool = obstacle_mask.astype(bool) if obstacle_mask is not None else np.zeros_like(habitat_bool)
    base_passable = (~habitat_bool) & (~obstacle_bool)
    stepping_bool = stepping_mask.astype(bool) if stepping_mask is not None else None

    if allow_bottlenecks or min_corridor_width <= 1:
        passable = base_passable
    else:
        offsets = _corridor_offsets(min_corridor_width)
        # Ignore habitat when evaluating clearance so corridors can still touch patches.
        true_obstacles = obstacle_bool & (~habitat_bool)
        clearance_space = ~true_obstacles
        clearance_ok = _erode_mask(clearance_space, offsets)
        passable = base_passable & clearance_ok

    if stepping_bool is not None and stepping_bool.any():
        passable = passable | stepping_bool

    return passable


def _has_adjacent_passable(mask: np.ndarray, base_passable: Optional[np.ndarray]) -> bool:
    """Determine if a patch has at least one passable pixel next to it."""
    if base_passable is None or not base_passable.any():
        return False
    rows, cols = mask.shape
    offsets = [(-1, 0), (1, 0), (0, -1), (0, 1), (-1, -1), (-1, 1), (1, -1), (1, 1)]
    for dr, dc in offsets:
        shifted = _shift_mask(mask, dr, dc)
        if (shifted & base_passable).any():
            return True
    return False


def _identify_stepstone_patches(
    labels: np.ndarray,
    obstacle_mask: np.ndarray,
    base_passable: Optional[np.ndarray] = None,
    min_overlap_ratio: float = 0.95,
) -> List[Tuple[int, int, float]]:
    """Detect patches that sit largely within impassables or have no adjacent passable pixels."""
    if obstacle_mask is None or not obstacle_mask.any():
        return []
    obstacle_bool = obstacle_mask.astype(bool)
    if base_passable is None:
        patch_mask = labels > 0
        base_passable = (~patch_mask) & (~obstacle_bool)
    stepping: List[Tuple[int, int, float]] = []
    unique_labels = np.unique(labels)
    for pid in unique_labels:
        if pid <= 0:
            continue
        mask = labels == pid
        total = int(mask.sum())
        if total == 0:
            continue
        overlap = int(np.sum(obstacle_bool[mask]))
        ratio = overlap / total
        isolated = not _has_adjacent_passable(mask, base_passable)
        if ratio >= min_overlap_ratio or isolated:
            stepping.append((pid, total, ratio))
    return stepping


def create_output_raster(
    labels: np.ndarray,
    corridors: Dict[int, Dict],
    min_corridor_width: int,
    obstacle_mask: Optional[np.ndarray] = None,
) -> np.ndarray:
    """Create an output raster with corridors marked by connected size."""
    output = np.zeros_like(labels, dtype=np.int32)
    offsets = _corridor_offsets(min_corridor_width)
    rows, cols = labels.shape
    use_mask = obstacle_mask is not None

    for corridor_data in corridors.values():
        score = corridor_data["connected_size"]
        for r, c in corridor_data["pixels"]:
            for dr, dc in offsets:
                nr, nc = r + dr, c + dc
                if 0 <= nr < rows and 0 <= nc < cols:
                    if use_mask and obstacle_mask[nr, nc]:
                        continue
                    if score > output[nr, nc]:
                        output[nr, nc] = score
    return output


def create_combined_raster(
    labels: np.ndarray,
    corridors: Dict[int, Dict],
    min_corridor_width: int,
    obstacle_mask: Optional[np.ndarray] = None,
) -> np.ndarray:
    """Create a raster where habitat patches and corridors are merged and labeled by connectivity."""
    rows, cols = labels.shape
    use_mask = obstacle_mask is not None
    combined = np.zeros_like(labels, dtype=np.int32)

    # Start with patches (existing labels)
    combined = labels.copy().astype(np.int32)

    # Burn corridors as a unique label (negative), then re-label components
    offsets = _corridor_offsets(min_corridor_width)
    next_label = int(labels.max()) + 1
    corridor_mask = np.zeros_like(labels, dtype=bool)
    for corridor_data in corridors.values():
        for r, c in corridor_data["pixels"]:
            for dr, dc in offsets:
                nr, nc = r + dr, c + dc
                if 0 <= nr < rows and 0 <= nc < cols:
                    if use_mask and obstacle_mask is not None and obstacle_mask[nr, nc]:
                        continue
                    corridor_mask[nr, nc] = True

    combined[corridor_mask] = -1  # temporary mark corridors

    if HAS_NDIMAGE and ndimage is not None:
        habitat_corr = combined != 0
        structure = np.ones((3, 3), dtype=np.uint8)
        labeled, _ = ndimage.label(habitat_corr, structure=structure)
        labeled[combined == 0] = 0
        return labeled.astype(np.int32)

    # Fallback: return patches with corridor mask burned as a unique label
    combined[corridor_mask] = labels.max() + 1 if labels.size > 0 else 1
    return combined.astype(np.int32)


def _compute_connectivity_metrics(corridors: Dict[int, Dict], patch_sizes: Dict[int, int]) -> Dict[str, float]:
    """Compute consistent graph metrics for summaries across strategies."""
    n_nodes = len(patch_sizes)
    uf = UnionFind()
    for pid in patch_sizes:
        uf.find(pid)

    edges_used = len(corridors)
    for data in corridors.values():
        p1 = data.get("p1") or data.get("patch1")
        p2 = data.get("p2") or data.get("patch2")
        if p1 is not None and p2 is not None:
            uf.union(int(p1), int(p2))
        else:
            pids = list(data.get("patch_ids", []))
            if len(pids) >= 2:
                uf.union(int(pids[0]), int(pids[1]))

    components = len({uf.find(pid) for pid in patch_sizes})
    redundant_links = max(0, edges_used - (n_nodes - components))
    avg_degree = (2 * edges_used / n_nodes) if n_nodes > 0 else 0.0

    return {
        "patches_total": n_nodes,
        "edges_used": edges_used,
        "components_remaining": components,
        "redundant_links": redundant_links,
        "avg_degree": avg_degree,
    }


def _run_deep_diagnostic(
    labels: np.ndarray,
    obstacle_mask: np.ndarray,
    passable_mask: np.ndarray,
    stepping_mask: Optional[np.ndarray],
    params: RasterRunParams,
    output_dir: str,
) -> None:
    """
    Generates a forensic report on why patches are not connecting.
    """
    print("\n" + "!" * 60)
    print("RUNNING DEEP DIAGNOSTIC (User Requested)")
    print("!" * 60)

    report_path = os.path.join(output_dir, "linkscape_diagnostic_dump.txt")

    passable_labels, n_components = label_patches(passable_mask.astype(np.uint8), 8)
    obstacle_only = obstacle_mask.astype(bool) & (labels == 0)

    lines = []
    lines.append("--- DEEP DIAGNOSTIC REPORT ---")
    lines.append(f"Map Size: {labels.shape}")
    lines.append(
        f"Passable Pixels: {np.sum(passable_mask):,} "
        f"({(np.sum(passable_mask) / labels.size) * 100:.1f}%)"
    )
    lines.append(f"Impassable Pixels: {np.sum(obstacle_only):,}")
    lines.append(f"Passable Components found: {n_components}")

    p1_mask = labels == 1
    if not p1_mask.any():
        lines.append("\nFATAL: Patch 1 does not exist in labels.")
    else:
        struct = np.ones((3, 3))
        if HAS_NDIMAGE:
            p1_dilated = ndimage.binary_dilation(p1_mask, structure=struct)
        else:
            p1_dilated = p1_mask.copy()

        boundary = p1_dilated & (~p1_mask)
        boundary_passable = boundary & passable_mask.astype(bool)
        touching_comps = np.unique(passable_labels[boundary_passable])
        touching_comps = touching_comps[touching_comps > 0]

        lines.append("\nPATCH 1 ANALYSIS:")
        if len(touching_comps) == 0:
            lines.append("  Result: ISOLATED. Patch 1 boundary touches NO passable pixels.")
            rows, cols = labels.shape
            ys, xs = np.where(p1_mask)
            cy, cx = int(np.mean(ys)), int(np.mean(xs))
            lines.append(f"  Centroid: ({cy}, {cx})")
            lines.append("  Neighborhood Dump (Values around centroid):")
            for r in range(max(0, cy - 2), min(rows, cy + 3)):
                row_vals = []
                for c in range(max(0, cx - 2), min(cols, cx + 3)):
                    status = "unk"
                    if obstacle_only[r, c]:
                        status = "OBS"
                    elif passable_mask[r, c]:
                        status = "PAS"
                    elif labels[r, c] > 0:
                        status = "PAT"
                    else:
                        status = "NOD/WLL"
                    row_vals.append(f"{status}")
                lines.append("    " + " ".join(row_vals))
        else:
            lines.append(f"  Patch 1 connects to Passable Component IDs: {touching_comps}")
            main_comp = touching_comps[0]
            comp_size = np.sum(passable_labels == main_comp)
            lines.append(f"  Main Component Size: {comp_size:,} pixels")

            if stepping_mask is not None and stepping_mask.any():
                lines.append("\nSTEPPING STONE ANALYSIS:")
                step_lbl, n_step = label_patches(stepping_mask.astype(np.uint8), 8)

                for i in range(1, min(n_step + 1, 6)):
                    stone_mask = step_lbl == i
                    stone_touching = np.unique(passable_labels[stone_mask])
                    stone_touching = stone_touching[stone_touching > 0]

                    connected = "NO"
                    if main_comp in stone_touching:
                        connected = "YES! (Reachable)"

                    lines.append(
                        f"  Stone {i}: Touches Passable IDs {stone_touching} -> "
                        f"Connected to Patch 1? {connected}"
                    )

                    if connected == "NO":
                        s_ys, s_xs = np.where(stone_mask)
                        sy, sx = s_ys[0], s_xs[0]
                        p_ys, p_xs = np.where(p1_mask)
                        py, px = p_ys[0], p_xs[0]

                        lines.append(f"    FAILURE DEBUG: Path from P1({py},{px}) to Stone({sy},{sx})")
                        dist = math.hypot(sy - py, sx - px)
                        lines.append(f"    Euclidean Distance: {dist:.1f} pixels")
                        if dist > params.max_search_distance:
                            lines.append(
                                f"    CAUSE: Too far! (Max Dist {params.max_search_distance} < {dist:.1f})"
                            )
                        else:
                            num_samples = int(dist)
                            blockers = []
                            for step in range(0, num_samples, max(1, num_samples // 10)):
                                t = step / num_samples
                                r = int(py + (sy - py) * t)
                                c = int(px + (sx - px) * t)
                                if 0 <= r < labels.shape[0] and 0 <= c < labels.shape[1]:
                                    if obstacle_only[r, c]:
                                        blockers.append(f"Impassable at {r},{c}")
                                    elif not passable_mask[r, c] and labels[r, c] == 0:
                                        blockers.append(f"NoData/Void at {r},{c}")

                            if blockers:
                                lines.append(f"    CAUSE: Line of sight blocked by: {blockers[:3]} ...")
                            else:
                                lines.append(f"    CAUSE: Unknown complex pathing (Labyrinth).")

    os.makedirs(output_dir, exist_ok=True)
    with open(report_path, "w", encoding="utf-8") as f:
        f.write("\n".join(lines))

    print(f"  ✓ Diagnostic report written: {report_path}")
    try:
        _open_path_in_default_app(report_path)
    except Exception:
        pass


def _build_corridor_stats(
    selected: Dict[int, Dict],
    patch_sizes: Dict[int, int],
) -> Tuple[Dict[int, int], Dict[int, int]]:
    """Compute component sizes and counts for selected corridors."""
    uf = UnionFind()
    for pid, size in patch_sizes.items():
        uf.find(pid)
        uf.size[pid] = size
        uf.count[pid] = 1

    for corr in selected.values():
        pids = list(corr.get("patch_ids", []))
        if len(pids) >= 2:
            uf.union(int(pids[0]), int(pids[1]))

    comp_sizes: Dict[int, int] = {}
    comp_counts: Dict[int, int] = {}
    for pid in patch_sizes.keys():
        root = uf.find(pid)
        comp_sizes[root] = comp_sizes.get(root, 0) + patch_sizes.get(pid, 0)
        comp_counts[root] = comp_counts.get(root, 0) + 1

    return comp_sizes, comp_counts


def optimize_hdfm_resilient(
    candidates: List[Dict], patch_sizes: Dict[int, int], budget_pixels: int
) -> Tuple[Dict[int, Dict], Dict]:
    """
    Lightweight resilient strategy: MST-style on corridor cost within budget.
    """
    uf = UnionFind()
    for pid, size in patch_sizes.items():
        uf.find(pid)
        uf.size[pid] = size
        uf.count[pid] = 1

    selected: Dict[int, Dict] = {}
    remaining = max(0, int(budget_pixels))
    budget_used = 0

    for cand in sorted(candidates, key=lambda c: c.get("area", 0)):
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        if p1 is None or p2 is None:
            continue
        cost = int(cand.get("area", 0))
        if cost <= 0 or cost > remaining:
            continue
        if uf.find(p1) == uf.find(p2):
            continue

        uf.union(p1, p2)
        remaining -= cost
        budget_used += cost
        comp_root = uf.find(p1)
        conn_size = uf.size.get(comp_root, cost)

        cid = len(selected) + 1
        selected[cid] = {
            "pixels": cand.get("pixels", set()),
            "patch_ids": {p1, p2},
            "area": cost,
            "connected_size": conn_size,
        }

    comp_sizes, comp_counts = _build_corridor_stats(selected, patch_sizes)
    largest_group_size = max(comp_sizes.values()) if comp_sizes else 0
    largest_group_patches = max(comp_counts.values()) if comp_counts else 0

    stats = {
        "strategy": "hdfm_most_connectivity",
        "corridors_used": len(selected),
        "budget_used": budget_used,
        "budget_total": budget_pixels,
        "patches_total": len(patch_sizes),
        "patches_connected": len(patch_sizes),
        "components_remaining": len(comp_sizes) if comp_sizes else len(patch_sizes),
        "largest_group_size": largest_group_size,
        "largest_group_patches": largest_group_patches,
        "total_connected_size": sum(comp_sizes.values()) if comp_sizes else 0,
        "redundant_links": 0,
        "avg_degree": (2 * len(selected) / max(len(patch_sizes), 1)) if patch_sizes else 0,
    }
    return selected, stats


def optimize_hdfm_largest_network(
    candidates: List[Dict], patch_sizes: Dict[int, int], budget_pixels: int
) -> Tuple[Dict[int, Dict], Dict]:
    """
    Grow corridors outward from the largest patch, preferring cheapest connections.
    """
    if not patch_sizes:
        return {}, {}
    seed = max(patch_sizes.items(), key=lambda kv: kv[1])[0]
    visited: Set[int] = {seed}
    remaining = max(0, int(budget_pixels))
    budget_used = 0
    selected: Dict[int, Dict] = {}

    adjacency: Dict[int, List[Dict]] = defaultdict(list)
    for cand in candidates:
        adjacency[cand.get("patch1")].append(cand)
        adjacency[cand.get("patch2")].append(cand)

    heap: List[Tuple[int, int, Dict]] = []
    counter = 0
    for cand in adjacency.get(seed, []):
        counter += 1
        heapq.heappush(heap, (int(cand.get("area", 0)), counter, cand))

    while heap and remaining > 0:
        cost, _, cand = heapq.heappop(heap)
        if cost <= 0 or cost > remaining:
            continue
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        in1, in2 = p1 in visited, p2 in visited
        if in1 and in2:
            continue
        if not in1 and not in2:
            continue

        new_node = p2 if in1 else p1
        visited.add(new_node)
        remaining -= cost
        budget_used += cost
        comp_size = sum(patch_sizes.get(pid, 0) for pid in visited)

        cid = len(selected) + 1
        selected[cid] = {
            "pixels": cand.get("pixels", set()),
            "patch_ids": {p1, p2},
            "area": cost,
            "connected_size": comp_size,
        }

        for nxt in adjacency.get(new_node, []):
            n1, n2 = nxt.get("patch1"), nxt.get("patch2")
            touches = (n1 in visited) ^ (n2 in visited)
            if touches:
                counter += 1
                heapq.heappush(heap, (int(nxt.get("area", 0)), counter, nxt))

    comp_sizes, comp_counts = _build_corridor_stats(selected, patch_sizes)
    largest_group_size = max(comp_sizes.values()) if comp_sizes else 0
    largest_group_patches = max(comp_counts.values()) if comp_counts else 0

    stats = {
        "strategy": "hdfm_largest_network",
        "corridors_used": len(selected),
        "budget_used": budget_used,
        "budget_total": budget_pixels,
        "patches_total": len(patch_sizes),
        "patches_connected": len(visited),
        "components_remaining": len(comp_sizes) if comp_sizes else len(patch_sizes),
        "largest_group_size": largest_group_size,
        "largest_group_patches": largest_group_patches,
        "total_connected_size": sum(comp_sizes.values()) if comp_sizes else 0,
        "redundant_links": 0,
        "avg_degree": (2 * len(selected) / max(len(patch_sizes), 1)) if patch_sizes else 0,
    }
    return selected, stats


def _to_dataclass(params: Dict) -> RasterRunParams:
    """Convert raw parameter dict into the expected dataclass."""
    debug_enabled = bool(params.get("debug_enabled", False))
    if not debug_enabled:
        env_flag = os.environ.get("LINKSCAPE_DEBUG") or os.environ.get("LINKSCAPE_DEBUG_RASTER")
        debug_enabled = str(env_flag or "").strip().lower() in ("1", "true", "yes", "on")
    debug_lines_init: List[str] = []

    return RasterRunParams(
        patch_connectivity=int(params.get("patch_connectivity", 4)),
        patch_mode=str(params.get("patch_mode", "value")).lower(),
        patch_values=list(params.get("patch_values", [])),
        range_lower=params.get("range_lower"),
        range_upper=params.get("range_upper"),
        obstacle_enabled=bool(params.get("obstacle_enabled", False)),
        obstacle_mode=str(params.get("obstacle_mode", "value")).lower(),
        obstacle_values=list(params.get("obstacle_values", [])),
        obstacle_range_lower=params.get("obstacle_range_lower"),
        obstacle_range_upper=params.get("obstacle_range_upper"),
        value_tolerance=float(params.get("value_tolerance", 1e-6)),
        nodata_fallback=float(params.get("nodata_fallback", -9999)),
        min_patch_size=int(params.get("min_patch_size", 0)),
        budget_pixels=int(params.get("budget_pixels", 0)),
        max_search_distance=int(params.get("max_search_distance", 50)),
        max_corridor_area=(
            int(params["max_corridor_area"]) if params.get("max_corridor_area") is not None else None
        ),
        min_corridor_width=int(params.get("min_corridor_width", 1)),
        stepping_enabled=bool(params.get("stepping_enabled", False)),
        hop_distance=max(0, int(params.get("hop_distance", 0) or 0)),
        allow_bottlenecks=bool(params.get("allow_bottlenecks", True)),
        debug_enabled=debug_enabled,
        debug_lines=debug_lines_init,
    )


def run_raster_analysis(
    layer: QgsRasterLayer,
    output_dir: str,
    raw_params: Dict,
    strategy: str = "hdfm_most_connectivity",
    temporary: bool = False,
    iface=None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
) -> List[Dict]:
    """Execute the raster corridor analysis for the provided layer."""
    if not isinstance(layer, QgsRasterLayer) or not layer.isValid():
        raise RasterAnalysisError("Selected layer is not a valid raster layer.")

    params = _to_dataclass(raw_params)
    overall_start = time.time()
    src_path = layer.source()
    summary_dir = output_dir or os.path.dirname(src_path) or os.getcwd()
    if not summary_dir:
        summary_dir = os.getcwd()
    os.makedirs(summary_dir, exist_ok=True)
    return _run_raster_analysis_core(
        layer,
        output_dir,
        raw_params,
        strategy,
        temporary,
        iface,
        progress_cb,
        params,
        overall_start,
    )


def _add_raster_run_summary_layer(
    *,
    input_layer_name: str,
    strategy: str,
    stats: Dict,
    out_path: str,
    combined_path: Optional[str] = None,
) -> None:
    """Add a small in-project summary table for the raster run."""
    if QgsVectorLayer is None or QgsProject is None or QgsFeature is None or QgsField is None:
        return
    try:
        title = f"Linkscape Raster Summary ({input_layer_name})"
        summary_layer = QgsVectorLayer("NoGeometry", title, "memory")
        if not summary_layer.isValid():
            return
        provider = summary_layer.dataProvider()
        provider.addAttributes([QgsField("item", QVariant.String), QgsField("value", QVariant.String)])
        summary_layer.updateFields()

        def _add(item: str, value: object) -> None:
            feat = QgsFeature(summary_layer.fields())
            feat.setAttributes([str(item), "" if value is None else str(value)])
            provider.addFeature(feat)

        _add("Input layer", input_layer_name)
        _add("Strategy", str(strategy))
        _add("Corridors created", stats.get("corridors_used", 0))
        _add("Patches connected", stats.get("patches_connected", 0))
        _add("Connected groups", stats.get("components_remaining", 0))
        _add("Budget used (px)", stats.get("budget_used", 0))
        _add("Budget total (px)", stats.get("budget_total", 0))
        _add("Output raster", out_path or "")
        if combined_path:
            _add("Combined raster", combined_path)

        QgsProject.instance().addMapLayer(summary_layer)
    except Exception:
        return


def _run_raster_analysis_core(
    layer: QgsRasterLayer,
    output_dir: str,
    raw_params: Dict,
    strategy: str = "hdfm_most_connectivity",
    temporary: bool = False,
    iface=None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
    params: Optional[RasterRunParams] = None,
    overall_start: Optional[float] = None,
) -> List[Dict]:
    """Core raster corridor logic (called by `run_raster_analysis`)."""
    if params is None:
        raise RasterAnalysisError("Raster parameters not supplied.")
    if overall_start is None:
        overall_start = time.time()

    ds = gdal.Open(layer.source())
    if ds is None:
        raise RasterAnalysisError(f"Cannot open raster source: {layer.source()}")

    rows, cols = ds.RasterYSize, ds.RasterXSize
    gt = ds.GetGeoTransform()
    proj = ds.GetProjection()
    print("=" * 70)
    print("LINKSCAPE RASTER ANALYSIS v23.0")
    print("=" * 70)
    print("\n1. Loading raster...")
    print(f"  ✓ Using layer: {layer.name()}")
    total_pixels = rows * cols
    print(f"  Size: {rows:,} x {cols:,} = {total_pixels:,} pixels")

    pixel_w = abs(gt[1])
    pixel_h = abs(gt[5]) if gt[5] != 0 else pixel_w
    pixel_size = max(pixel_w, pixel_h)

    _debug_raster(params, f"Raster size: {rows}x{cols} pixels, pixel size ≈ {pixel_size:.2f}")

    warnings: List[str] = []
    blockers: List[str] = []

    if total_pixels >= PIXEL_COUNT_WARNING_THRESHOLD:
        blockers.append(
            "Large raster detected (>{:,} pixels).".format(
                PIXEL_COUNT_WARNING_THRESHOLD
            )
        )
    if 0 < pixel_size < PIXEL_SIZE_WARNING_THRESHOLD:
        msg = (
            f"High-resolution data detected (≈{pixel_size:.2f} map units per pixel). "
            "Consider resampling to a coarser resolution for faster corridor modelling."
        )
        if total_pixels >= PIXEL_FINE_CRITICAL_COUNT:
            blockers.append(msg)
        else:
            warnings.append(msg + " Proceeding because raster size is small.")

    if warnings:
        warning_text = " ".join(warnings)
        if iface and hasattr(iface, "messageBar"):
            try:
                iface.messageBar().pushMessage("Linkscape", warning_text, level=Qgis.Warning)
            except Exception:
                print(f"WARNING: {warning_text}")
        else:
            print(f"WARNING: {warning_text}")

    if blockers:
        warning_text = " ".join(blockers)
        if iface and hasattr(iface, "messageBar"):
            try:
                iface.messageBar().pushWarning("Linkscape", warning_text)
            except Exception:
                print(f"WARNING: {warning_text}")
        else:
            print(f"WARNING: {warning_text}")
        raise RasterAnalysisError(
            f"Raster is too large/fine for Linkscape to process efficiently. {warning_text} "
            "Please resample to a coarser resolution or process in smaller chunks."
        )
    _emit_progress(progress_cb, 5, "Loading raster data…")

    band = ds.GetRasterBand(1)
    nodata = band.GetNoDataValue()
    if nodata is None:
        nodata = params.nodata_fallback

    print("  Reading data...")
    data = read_band(
        band,
        rows,
        cols,
        progress_cb=progress_cb,
        progress_start=5,
        progress_end=18,
    )
    nodata_mask = np.abs(data - nodata) < params.value_tolerance if nodata is not None else np.zeros_like(
        data, dtype=bool
    )
    _emit_progress(progress_cb, 20, "Defining habitat patches…")

    print("\n2. Identifying patch pixels...")
    patch_mask = define_habitat(data, nodata_mask, params)
    habitat_mask = patch_mask.astype(np.uint8)
    patch_pixels = int(np.sum(habitat_mask))
    _debug_raster(params, f"Patch pixels: {patch_pixels:,}")
    if patch_pixels == 0:
        _debug_raster(params, "No patch pixels found with the current configuration.")
        raise RasterAnalysisError("No patch pixels found with the current configuration.")
    print(f"  ✓ Patch pixels: {patch_pixels:,}")
    _emit_progress(progress_cb, 25, "Processing habitat patches…")

    obstacle_mask = define_obstacles(data, nodata_mask, habitat_mask, params)
    if params.obstacle_enabled:
        obstacle_pixels = int(np.sum(obstacle_mask))
        if obstacle_pixels:
            print(f"  ✓ Impassable pixels: {obstacle_pixels:,}")
        else:
            print("  ⚠ Impassable land class configuration matched no pixels; proceeding without impassables.")
    else:
        print("  ✓ Impassable land classes disabled.")
    obstacle_pixels = int(np.sum(obstacle_mask))
    _debug_raster(
        params,
        f"Impassable pixels: {obstacle_pixels} (enabled={params.obstacle_enabled})",
    )

    # NOTE: Background ("matrix") is already treated as passable by default: passable_mask
    # is derived from (~habitat) & (~impassable). Do not clear impassable pixels here.
    _emit_progress(progress_cb, 35, "Labeling patches…")

    print("\n3. Labeling patches...")
    t0 = time.time()
    labels, n_patches = label_patches(habitat_mask, params.patch_connectivity)
    print(f"  ✓ Patches: {n_patches:,} in {time.time() - t0:.2f}s")
    _debug_raster(params, f"Labelled patches: {n_patches}")

    if params.min_patch_size > 0:
        unique_labels, counts = np.unique(labels[labels > 0], return_counts=True)
        valid_labels = unique_labels[counts >= params.min_patch_size]
        new_labels = np.zeros_like(labels)
        for new_id, old_id in enumerate(valid_labels, 1):
            new_labels[labels == old_id] = new_id
        labels = new_labels
        print(f"  ✓ After filter: {len(valid_labels):,} patches")

    initial_patch_mask = labels > 0
    obstacle_bool = obstacle_mask.astype(bool)
    base_passable = (~initial_patch_mask) & (~obstacle_bool)
    stepping_patches: List[Tuple[int, int, float]] = []
    if params.stepping_enabled and params.hop_distance > 0:
        stepping_patches = _identify_stepstone_patches(
            labels,
            obstacle_mask,
            base_passable=base_passable,
            min_overlap_ratio=0.95,
        )
    stepping_mask = np.zeros_like(labels, dtype=bool)
    if stepping_patches:
        _debug_raster(
            params,
            "Step stone patches detected (mostly within impassable land classes): "
            + ", ".join(
                f"P{pid}({count}px, {ratio*100:.1f}% impassable)"
                for pid, count, ratio in stepping_patches
            ),
        )
        stepping_ids = [pid for pid, _, _ in stepping_patches]
        for pid in stepping_ids:
            mask = labels == pid
            stepping_mask |= mask
        obstacle_removed = int(np.sum(obstacle_mask[stepping_mask]))
        obstacle_mask[stepping_mask] = False
        labels[stepping_mask] = 0
        _debug_raster(params, f"Removed {obstacle_removed} impassable pixels for step stones.")
        _debug_raster(params, f"Step stone cells made passable: {int(np.sum(stepping_mask))}")

    patch_mask = (labels > 0).astype(np.uint8)
    final_patch_ids = np.unique(labels[labels > 0])
    _debug_raster(params, f"Valid patches remaining: {len(final_patch_ids)}")

    obstacle_mask = obstacle_mask.astype(bool) | patch_mask.astype(bool)

    passable_mask = _build_passable_mask(
        patch_mask,
        obstacle_mask,
        params.min_corridor_width,
        params.allow_bottlenecks,
        stepping_mask=stepping_mask,
    )
    passable_cells = int(np.sum(passable_mask))
    _debug_raster(params, f"Passable mask cells: {passable_cells}")

    debug_label_env = os.environ.get("LINKSCAPE_SAVE_PATCH_LABELS")
    if debug_label_env:
        try:
            label_out_dir = output_dir or os.path.dirname(src_path)
            os.makedirs(label_out_dir, exist_ok=True)
            label_path = os.path.join(label_out_dir, "linkscape_patch_labels.tif")
            write_raster(label_path, labels, gt, proj, nodata=0)
            print(f"  ✓ Saved patch ID raster: {label_path}")
        except Exception as label_exc:  # noqa: BLE001
            print(f"  ⚠ Could not save patch ID raster: {label_exc}")

    unique_labels, counts = np.unique(labels[labels > 0], return_counts=True)
    patch_sizes = dict(zip(unique_labels.tolist(), counts.tolist()))
    if not patch_sizes:
        _debug_raster(params, "No valid patches remain after filtering.")
        raise RasterAnalysisError("No valid patches remain after filtering.")
    _emit_progress(progress_cb, 45, "Searching for corridors…")

    print("\n4. Finding possible corridors...")
    candidates = find_all_possible_corridors(
        labels,
        habitat_mask,
        patch_sizes,
        params.max_search_distance,
        params.max_corridor_area,
        params.patch_connectivity,
        min_corridor_width=params.min_corridor_width,
        obstacle_mask=obstacle_mask,
        passable_mask=passable_mask,
        progress_cb=progress_cb,
        progress_start=45,
        progress_end=75,
        params=params,
    )
    candidate_count = len(candidates)
    _debug_raster(
        params,
        f"Corridor candidates found: {candidate_count} (max_search={params.max_search_distance})",
    )

    if not candidates:
        _debug_raster(params, "No feasible corridors found with the current configuration.")
        raise RasterAnalysisError("No feasible corridors found with the current configuration.")
    _emit_progress(progress_cb, 78, "Optimizing corridor selection…")

    strategy = (strategy or "hdfm_most_connectivity").lower()
    strategy_map = {
        "hdfm_most_connectivity": (
            optimize_hdfm_resilient,
            "linkscape_hdfm_most_connectivity.tif",
            "Corridors (HDFM Most Connectivity)",
        ),
        "hdfm_largest_network": (
            optimize_hdfm_largest_network,
            "linkscape_hdfm_largest_network.tif",
            "Corridors (HDFM Largest Network)",
        ),
    }

    if strategy not in strategy_map:
        raise RasterAnalysisError(f"Unsupported strategy '{strategy}'.")

    optimize_func, default_filename, layer_name = strategy_map[strategy]

    print("\n5. Running optimization...")
    print("=" * 70)
    print(f"--- {strategy.replace('_', ' ').upper()} ---")

    corridors, stats = optimize_func(candidates, patch_sizes, params.budget_pixels)
    if not corridors:
        _debug_raster(params, "Selected optimization did not produce any corridors.")
        raise RasterAnalysisError("Selected optimization did not produce any corridors.")
    _emit_progress(progress_cb, 85, "Rendering output raster…")

    print("  Creating output raster...")
    output = create_output_raster(
        labels, corridors, params.min_corridor_width, obstacle_mask=obstacle_mask
    )
    combined_output = create_combined_raster(
        labels, corridors, params.min_corridor_width, obstacle_mask=obstacle_mask
    )

    if temporary:
        temp_file = tempfile.NamedTemporaryFile(prefix="linkscape_", suffix=".tif", delete=False)
        out_path = temp_file.name
        temp_file.close()
    else:
        out_dir = output_dir or os.path.dirname(src_path)
        os.makedirs(out_dir, exist_ok=True)
        out_path = os.path.join(out_dir, default_filename)
        combined_path = os.path.join(out_dir, f"combined_{default_filename}")

    write_raster(out_path, output, gt, proj, nodata=0)
    print(f"  ✓ Saved: {out_path}")
    try:
        result_layer = QgsRasterLayer(out_path, layer_name)
        if result_layer.isValid():
            QgsProject.instance().addMapLayer(result_layer)
            print("  ✓ Added to project")
    except Exception as add_exc:  # noqa: BLE001
        print(f"  ⚠ Could not add layer to project: {add_exc}")

    # Write and add combined raster
    if combined_output is not None and not temporary:
        try:
            write_raster(combined_path, combined_output, gt, proj, nodata=0)
            print(f"  ✓ Saved combined patches+corridors: {combined_path}")
            combined_layer = QgsRasterLayer(combined_path, f"{layer_name} (Combined)")
            if combined_layer.isValid():
                QgsProject.instance().addMapLayer(combined_layer)
                print("  ✓ Added combined raster to project")
        except Exception as comb_exc:  # noqa: BLE001
            print(f"  ⚠ Could not save/add combined raster: {comb_exc}")

    _emit_progress(progress_cb, 95, "Finishing up…")

    elapsed = time.time() - overall_start
    stats = dict(stats)
    stats["output_path"] = out_path if not temporary else ""
    stats["layer_name"] = layer_name
    stats["budget_total"] = params.budget_pixels
    stats["patches_total"] = stats.get("patches_total", len(patch_sizes))
    stats["habitat_pixels_total"] = sum(patch_sizes.values())
    stats.update(_compute_connectivity_metrics(corridors, patch_sizes))

    print("\n" + "=" * 70)
    print("FINAL SUMMARY")
    print("=" * 70)
    stats_strategy = stats.get("strategy", strategy)
    print(f"Strategy:          {stats_strategy.replace('_', ' ').title()}")
    print(f"Corridors created: {stats.get('corridors_used', 0)}")
    print(f"Connected groups:  {stats.get('components_remaining', 0)}")
    print(f"Edges used:        {stats.get('edges_used', 0)}")
    print(f"Avg degree:        {stats.get('avg_degree', 0):.2f}")
    print(f"Redundant links:   {stats.get('redundant_links', 0)}")
    if "connections_made" in stats:
        print(f"Connections:       {stats.get('connections_made', 0)}")
    if "seed_id" in stats:
        print(f"Seed patch:        {stats.get('seed_id')}")
    print(f"Final size:        {stats.get('final_patch_size', 0):,} px")
    print(f"Corridor budget:   {params.budget_pixels} px")
    print(f"Budget used:       {stats.get('budget_used', 0)} px")
    if "hdfm_entropy_total" in stats:
        print("\nHDFM Metrics:")
        print(f"  Entropy (H_total): {stats.get('hdfm_entropy_total', 0):.4f}")
        print(f"  Movement entropy:  {stats.get('hdfm_entropy_movement', 0):.4f}")
        print(f"  Robustness (ρ₂):   {stats.get('hdfm_robustness_rho2', 0):.4f}")
        print(f"  Topology penalty:  {stats.get('hdfm_topology_penalty', 0):.1f}")
        print(f"  HDFM mode:         {stats.get('hdfm_mode', 'N/A')}")
    print(f"Processing time:   {elapsed:.1f}s")
    if temporary:
        print("Output:            Temporary raster layer")
    else:
        print(f"Output GeoTIFF:    {out_path}")
    print("=" * 70)

    # Add a lightweight in-project summary table instead of writing/opening text files.
    try:
        _add_raster_run_summary_layer(
            input_layer_name=layer.name(),
            strategy=strategy.replace("_", " ").title(),
            stats=stats,
            out_path=out_path if not temporary else "",
            combined_path=combined_path if (not temporary and "combined_path" in locals()) else None,
        )
    except Exception:
        pass

    ds = None
    _emit_progress(progress_cb, 100, "Raster analysis complete.")
    return [{"strategy": strategy, "stats": stats, "output_path": out_path if not temporary else ""}]
