"""
Linkscape Corridor Analysis - Vector Workflow (v23.3)
----------------------------------------------------
Runs the single-strategy corridor optimization workflow for polygon
patch datasets.

Updates in v23.3:
- Fixed Redundancy: Allows parallel connections between components if budget permits.
- Fixed Traversal: Efficient spatial indexing for detecting intermediate patch crossings.
- Fixed Logic: Corridors crossing intermediate patches (A->C->B) are now prioritized
  if C is not yet connected, even if A and B are.
"""

from __future__ import annotations

import heapq
import math
import os
import time
from collections import defaultdict, deque
from dataclasses import dataclass, field
from typing import Callable, Dict, List, Optional, Set, Tuple

import numpy as np
from PyQt5.QtCore import QVariant
from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsFeature,
    QgsField,
    QgsFields,
    QgsGeometry,
    QgsPointXY,
    QgsProject,
    QgsRectangle,
    QgsSpatialIndex,
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsWkbTypes,
)

BUFFER_SEGMENTS = 16
from .linkscape_engine import NetworkOptimizer, UnionFind
from .utils import emit_progress, log_error


def _log_message(message: str, level: int = Qgis.Info) -> None:
    """Log to the QGIS Log Messages Panel with a Linkscape tag."""
    try:
        QgsApplication.messageLog().logMessage(message, "Linkscape", level)
    except Exception:
        # Fallback for environments where the message log is unavailable
        print(f"Linkscape Log: {message}")


def clone_geometry(geom: QgsGeometry) -> QgsGeometry:
    """
    Lightweight copy helper; QgsGeometry uses implicit sharing so this is cheap
    and avoids unnecessary deep clones unless a write occurs.
    """
    return QgsGeometry(geom)


class VectorAnalysisError(RuntimeError):
    """Raised when the vector analysis cannot be completed."""


@dataclass
class VectorRunParams:
    min_corridor_width: float  # metres
    max_corridor_area: Optional[float]  # hectares
    min_patch_size: float  # hectares
    budget_area: float  # hectares
    max_search_distance: float  # metres
    unit_system: str
    output_name: str
    grid_resolution: float  # metres
    stepping_enabled: bool = False
    hop_distance: float = 0.0  # metres
    obstacle_layer_ids: List[str] = field(default_factory=list)
    obstacle_enabled: bool = False


def _to_dataclass(params: Dict) -> VectorRunParams:
    output_name = params.get("output_name") or "linkscape_corridors.gpkg"
    if not output_name.lower().endswith(".gpkg"):
        output_name = f"{output_name}.gpkg"
    obstacle_ids_raw = params.get("obstacle_layer_ids") or []
    if not obstacle_ids_raw and params.get("obstacle_layer_id"):
        obstacle_ids_raw = [params.get("obstacle_layer_id")]
    obstacle_ids = [str(val) for val in obstacle_ids_raw if val]
    return VectorRunParams(
        min_corridor_width=float(params.get("min_corridor_width", 200.0)),
        max_corridor_area=(
            float(params["max_corridor_area"])
            if params.get("max_corridor_area") not in (None, 0, 0.0)
            else None
        ),
        min_patch_size=float(params.get("min_patch_size", 10.0)),
        budget_area=float(params.get("budget_area", 50.0)),
        max_search_distance=float(params.get("max_search_distance", 5000.0)),
        unit_system=str(params.get("unit_system", "metric")),
        output_name=output_name,
        grid_resolution=max(float(params.get("grid_resolution", 50.0)), 1.0),
        stepping_enabled=bool(params.get("stepping_enabled", False)),
        hop_distance=float(params.get("hop_distance", 0.0) or 0.0),
        obstacle_layer_ids=obstacle_ids,
        obstacle_enabled=bool(params.get("obstacle_enabled", False) and obstacle_ids),
    )


def get_utm_crs_from_extent(layer: QgsVectorLayer) -> QgsCoordinateReferenceSystem:
    extent = layer.extent()
    center = extent.center()
    source_crs = layer.crs()

    if not source_crs.isGeographic():
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(source_crs, wgs84, QgsProject.instance())
        center = transform.transform(center)

    utm_zone = int((center.x() + 180) / 6) + 1
    epsg_code = 32600 + utm_zone if center.y() >= 0 else 32700 + utm_zone
    return QgsCoordinateReferenceSystem(f"EPSG:{epsg_code}")


def load_and_prepare_patches(
    layer: QgsVectorLayer,
    target_crs: QgsCoordinateReferenceSystem,
    params: VectorRunParams,
) -> Tuple[Dict[int, Dict], QgsSpatialIndex]:
    print("  Loading patches and building spatial index...")
    source_crs = layer.crs()
    transform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())

    patches: Dict[int, Dict] = {}
    patch_id = 1
    filtered_count = 0
    indexed_features: List[QgsFeature] = []

    for feature in layer.getFeatures():
        geom = QgsGeometry(feature.geometry())
        if geom.isEmpty():
            continue
        geom.transform(transform)
        try:
            geom = geom.makeValid()
        except Exception:
            pass
        area_ha = geom.area() / 10000.0

        if area_ha < params.min_patch_size:
            filtered_count += 1
            continue

        feat = QgsFeature()
        feat.setGeometry(geom)
        feat.setId(patch_id)
        indexed_features.append(feat)

        patches[patch_id] = {
            "geom": clone_geometry(geom),
            "area_ha": area_ha,
        }
        patch_id += 1

    spatial_index = QgsSpatialIndex(flags=QgsSpatialIndex.FlagStoreFeatureGeometries)
    if indexed_features:
        spatial_index.addFeatures(indexed_features)

    print(f"  ✓ Loaded {len(patches)} patches (filtered {filtered_count} too small)")
    return patches, spatial_index


def _compute_vector_hop_adjacency(
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    hop_distance: float,
) -> Dict[int, Set[int]]:
    if hop_distance <= 0:
        return {}
    adjacency: Dict[int, Set[int]] = defaultdict(set)
    for pid, pdata in patches.items():
        geom1 = pdata["geom"]
        rect = geom1.boundingBox()
        rect.grow(hop_distance)
        for other_id in spatial_index.intersects(rect):
            if other_id <= pid:
                continue
            pdata2 = patches.get(other_id)
            if not pdata2:
                continue
            geom2 = pdata2["geom"]
            try:
                distance = geom1.distance(geom2)
            except Exception:
                continue
            if distance <= hop_distance + 1e-9:
                adjacency[pid].add(other_id)
                adjacency[other_id].add(pid)
    return adjacency


def _build_virtual_components(
    patches: Dict[int, Dict],
    hop_adjacency: Dict[int, Set[int]],
) -> Tuple[Dict[int, int], Dict[int, float], Dict[int, int]]:
    uf = UnionFind()
    for pid, pdata in patches.items():
        uf.find(pid)
        uf.size[pid] = pdata["area_ha"]
        uf.count[pid] = 1

    for pid, neighbors in hop_adjacency.items():
        for nbr in neighbors:
            uf.union(pid, nbr)

    component_map: Dict[int, int] = {}
    component_sizes: Dict[int, float] = {}
    component_counts: Dict[int, int] = {}
    for pid, pdata in patches.items():
        comp = uf.find(pid)
        component_map[pid] = comp
        component_sizes[comp] = component_sizes.get(comp, 0.0) + pdata["area_ha"]
        component_counts[comp] = component_counts.get(comp, 0) + 1

    return component_map, component_sizes, component_counts


def _annotate_candidates_with_components(
    candidates: List[Dict],
    component_map: Dict[int, int],
) -> None:
    for cand in candidates:
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        cand["patch_ids"] = set(cand.get("patch_ids", set()))
        if not cand["patch_ids"]:
            cand["patch_ids"] = {p1, p2}
        comp1 = component_map.get(p1, p1)
        comp2 = component_map.get(p2, p2)
        cand["comp1"] = comp1
        cand["comp2"] = comp2
        cand["component_ids"] = {component_map.get(pid, pid) for pid in cand["patch_ids"]}
        cand.setdefault("is_stepping_chain", False)


def _build_component_pair_index(candidates: List[Dict]) -> Set[frozenset]:
    pairs: Set[frozenset] = set()
    for cand in candidates:
        comp1 = cand.get("comp1")
        comp2 = cand.get("comp2")
        if comp1 is None or comp2 is None or comp1 == comp2:
            continue
        pairs.add(frozenset({comp1, comp2}))
    return pairs


def _build_component_adjacency(candidates: List[Dict]) -> Dict[int, Set[int]]:
    adjacency: Dict[int, Set[int]] = defaultdict(set)
    for cand in candidates:
        comp1 = cand.get("comp1")
        comp2 = cand.get("comp2")
        if comp1 is None or comp2 is None or comp1 == comp2:
            continue
        adjacency[comp1].add(comp2)
        adjacency[comp2].add(comp1)
    return adjacency


def _detect_corridor_intersections(
    corridor_geom: QgsGeometry,
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    connected_patches: Set[int],
) -> Set[int]:
    """
    Efficiently detect which patches are traversed by a corridor geometry
    using the spatial index.
    """
    intersected: Set[int] = set()
    bbox = corridor_geom.boundingBox()
    
    # Use spatial index to find candidate interactions (fast)
    candidate_ids = spatial_index.intersects(bbox)
    
    for pid in candidate_ids:
        if pid in connected_patches:
            continue
        pdata = patches.get(pid)
        if not pdata:
            continue
            
        try:
            # Check for actual intersection
            if corridor_geom.intersects(pdata["geom"]):
                intersection = corridor_geom.intersection(pdata["geom"])
                if intersection and (not intersection.isEmpty()) and intersection.area() > 0:
                    intersected.add(pid)
        except Exception:
            continue
            
    return intersected


def _finalize_corridor_geometry(
    pid1: int,
    pid2: int,
    corridor_geom: QgsGeometry,
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    patch_union: Optional[QgsGeometry] = None,
) -> Tuple[Optional[QgsGeometry], Set[int]]:
    """
    Detect traversed patches, clip corridor geometry so it doesn't overlap them,
    and update the set of connected patches.
    """
    if corridor_geom is None or corridor_geom.isEmpty():
        return None, set()

    patch_ids: Set[int] = {pid1, pid2}
    
    # 1. Detect intermediate patches (A -> C -> B)
    # Using spatial index here is crucial for performance
    intersected = _detect_corridor_intersections(corridor_geom, patches, spatial_index, patch_ids)
    if intersected:
        patch_ids.update(intersected)

    # 2. Clip the corridor geometry against ALL involved patches
    # This makes the corridor "free" where it crosses existing habitat
    # and prevents drawing on top of patches.
    final_geom = clone_geometry(corridor_geom)
    
    if patch_union and not patch_union.isEmpty():
        try:
            final_geom = final_geom.difference(patch_union)
        except Exception:
            pass
    else:
        for pid in patch_ids:
            pdata = patches.get(pid)
            if not pdata: 
                continue
            patch_geom = pdata.get("geom")
            if not patch_geom or patch_geom.isEmpty():
                continue
                
            try:
                if final_geom.intersects(patch_geom):
                    final_geom = final_geom.difference(patch_geom)
                    if final_geom.isEmpty():
                        break
            except Exception:
                pass

    if final_geom is None or final_geom.isEmpty():
        # If the corridor is entirely consumed by patches, it means the patches 
        # touch or overlap. In vector analysis, this is valid connectivity (0 area cost).
        # We return an empty geom but valid patch_ids. However, to display it,
        # we might want to return None if strictly "corridor building".
        # But usually we return None to avoid 0-area features being written.
        return None, set()

    final_geom = final_geom.makeValid()
    if final_geom.isEmpty():
        return None, set()

    return final_geom, patch_ids


def _build_hop_edge_lookup(
    candidates: List[Dict],
    hop_adjacency: Dict[int, Set[int]],
) -> Dict[frozenset, Dict]:
    if not hop_adjacency:
        for cand in candidates:
            cand["is_hop_edge"] = False
        return {}

    lookup: Dict[frozenset, Dict] = {}
    for cand in candidates:
        p1, p2 = cand.get("patch1"), cand.get("patch2")
        is_hop = p2 in hop_adjacency.get(p1, set())
        cand["is_hop_edge"] = is_hop
        if is_hop:
            lookup[frozenset({p1, p2})] = cand
    return lookup


def _build_stepping_chains(
    hop_lookup: Dict[frozenset, Dict],
    component_map: Dict[int, int],
    component_sizes: Optional[Dict[int, float]] = None,
) -> List[Dict]:
    """Build daisy-chain corridors by walking hop edges and summing real segment costs."""
    if not hop_lookup:
        return []

    comp_sizes = component_sizes or {}
    graph: Dict[int, Set[int]] = defaultdict(set)
    for pair in hop_lookup.keys():
        if len(pair) != 2:
            continue
        a, b = tuple(pair)
        graph[a].add(b)
        graph[b].add(a)

    chains: List[Dict] = []
    seen_paths: Set[Tuple[int, ...]] = set()

    start_nodes = sorted(graph.keys(), key=lambda pid: -comp_sizes.get(component_map.get(pid, pid), 0.0))

    for start in start_nodes:
        stack: List[Tuple[int, List[int], float, float, Set[int], List[QgsGeometry]]] = []
        stack.append((start, [start], 0.0, 0.0, {start}, []))
        while stack:
            node, path, cost_so_far, dist_so_far, patch_ids, geoms = stack.pop()
            path_key = tuple(path)
            if path_key in seen_paths:
                continue
            seen_paths.add(path_key)

            for nbr in graph.get(node, set()):
                if nbr in path:
                    continue
                edge = hop_lookup.get(frozenset({node, nbr}))
                if edge is None or edge.get("geom") is None:
                    continue
                edge_cost = float(edge.get("area_ha", 0.0))
                edge_dist = float(edge.get("distance_m", 0.0))
                new_cost = cost_so_far + edge_cost
                new_dist = dist_so_far + edge_dist
                new_path = path + [nbr]
                new_pids = set(patch_ids)
                new_pids.update(edge.get("patch_ids", set()))

                new_geoms = list(geoms)
                new_geoms.append(clone_geometry(edge.get("geom")))

                comp_start = component_map.get(path[0], path[0])
                comp_end = component_map.get(nbr, nbr)
                if comp_start != comp_end and len(new_path) >= 2:
                    merged_geom = new_geoms[0]
                    if len(new_geoms) > 1:
                        try:
                            merged_geom = QgsGeometry.unaryUnion(new_geoms)
                        except Exception:
                            merged_geom = new_geoms[0]
                    chains.append(
                        {
                            "patch1": path[0],
                            "patch2": nbr,
                            "patch_ids": new_pids,
                            "component_ids": {component_map.get(pid, pid) for pid in new_pids},
                            "geom": merged_geom,
                            "area_ha": new_cost,
                            "distance_m": new_dist,
                            "comp1": comp_start,
                            "comp2": comp_end,
                            "is_stepping_chain": True,
                        }
                    )
                stack.append((nbr, new_path, new_cost, new_dist, new_pids, new_geoms))
    _log_message(f"  **_build_stepping_chains:** Successfully generated {len(chains)} stepping-stone chains.", Qgis.Info)
    return chains


def _buffer_line_segment(line_geom: QgsGeometry, width: float) -> QgsGeometry:
    return line_geom.buffer(width / 2.0, BUFFER_SEGMENTS)


def _corridor_passes_width(corridor_geom: QgsGeometry, min_width: float) -> bool:
    if corridor_geom.isEmpty():
        return False
    # If the corridor is multipolygon (due to crossing patches), check each part?
    # Actually, if it's multipolygon, it means we clipped out patches.
    # The 'neck' check is complex on multipolygons. We check the buffer on the original,
    # but since we already clipped, we skip aggressive width validation on the final result
    # to avoid rejecting valid patch-traversals.
    return True


def _format_no_corridor_reason(
    stage: str,
    patch_count: int,
    candidate_count: int,
    params: VectorRunParams,
) -> str:
    return (
        f"{stage}: no feasible corridors could be generated.\n"
        f"- Patches meeting criteria: {patch_count}\n"
        f"- Candidate corridors generated: {candidate_count}\n"
        f"- Max search distance: {params.max_search_distance:.2f}\n"
        f"- Min corridor width: {params.min_corridor_width:.2f}\n"
        f"- Max corridor area: {params.max_corridor_area or 'None (no limit)'}\n"
        "Try increasing the search distance, lowering the minimum corridor width/area, "
        "or simplifying the patch layer."
    )


def _create_corridor_geometry(
    waypoints: List[QgsPointXY],
    source_geom: QgsGeometry,
    target_geom: QgsGeometry,
    params: VectorRunParams,
    obstacle_geoms: Optional[List[QgsGeometry]] = None,
    smooth_iterations: int = 0,
    hop_distance: float = 0.0,
) -> Optional[QgsGeometry]:
    if not waypoints or len(waypoints) < 2:
        return None

    corridor_line = QgsGeometry.fromPolylineXY([QgsPointXY(pt) for pt in waypoints])
    if smooth_iterations > 0:
        try:
            smoothed = corridor_line.smooth(smooth_iterations)
            if smoothed and not smoothed.isEmpty():
                corridor_line = smoothed
        except Exception:
            pass
            
    # Buffer to full width
    corridor_geom = _buffer_line_segment(corridor_line, params.min_corridor_width)
    
    # Clip start/end immediately to get the "bridge" geometry
    corridor_geom = corridor_geom.difference(source_geom)
    corridor_geom = corridor_geom.difference(target_geom)
    
    if obstacle_geoms:
        if hop_distance > 0:
            for obstacle in obstacle_geoms:
                intersection = corridor_geom.intersection(obstacle)
                if intersection and not intersection.isEmpty():
                    try:
                        if intersection.length() <= hop_distance:
                            continue  # hop over this obstacle span
                    except Exception:
                        pass
                    corridor_geom = corridor_geom.difference(obstacle)
                    if corridor_geom.isEmpty():
                        return None
        else:
            for obstacle in obstacle_geoms:
                corridor_geom = corridor_geom.difference(obstacle)
                if corridor_geom.isEmpty():
                    return None
                
    corridor_geom = corridor_geom.makeValid()
    if corridor_geom.isEmpty():
        return None

    return corridor_geom


class RasterNavigator:
    def __init__(
        self,
        patches: Dict[int, Dict],
        obstacle_layers: List[QgsVectorLayer],
        target_crs: QgsCoordinateReferenceSystem,
        params: VectorRunParams,
    ):
        if not obstacle_layers:
            raise VectorAnalysisError("Select at least one polygon obstacle layer for vector obstacle avoidance.")

        self.resolution = max(params.grid_resolution, 1.0)
        self.obstacle_geoms: List[QgsGeometry] = []

        extent: Optional[QgsRectangle] = None
        for patch in patches.values():
            bbox = patch["geom"].boundingBox()
            if extent is None:
                extent = QgsRectangle(bbox)
            else:
                extent.combineExtentWith(bbox)

        for obstacle_layer in obstacle_layers:
            if obstacle_layer is None or QgsWkbTypes.geometryType(obstacle_layer.wkbType()) != QgsWkbTypes.PolygonGeometry:
                raise VectorAnalysisError("Select a polygon obstacle layer for vector obstacle avoidance.")

            transform = QgsCoordinateTransform(obstacle_layer.crs(), target_crs, QgsProject.instance())
            for feature in obstacle_layer.getFeatures():
                geom = QgsGeometry(feature.geometry())
                if geom.isEmpty():
                    continue
                geom.transform(transform)
                geom = geom.makeValid()
                if geom.isEmpty():
                    continue
                self.obstacle_geoms.append(clone_geometry(geom))
                bbox = geom.boundingBox()
                if extent is None:
                    extent = QgsRectangle(bbox)
                else:
                    extent.combineExtentWith(bbox)

        if extent is None:
            raise VectorAnalysisError("Unable to determine extent for obstacle avoidance routing.")

        pad = max(params.max_search_distance, params.min_corridor_width)
        extent = QgsRectangle(
            extent.xMinimum() - pad,
            extent.yMinimum() - pad,
            extent.xMaximum() + pad,
            extent.yMaximum() + pad,
        )
        width = max(extent.width(), self.resolution)
        height = max(extent.height(), self.resolution)

        self.origin_x = extent.xMinimum()
        self.origin_y = extent.yMaximum()
        self.cols = max(1, int(math.ceil(width / self.resolution)))
        self.rows = max(1, int(math.ceil(height / self.resolution)))
        self.passable = np.ones((self.rows, self.cols), dtype=bool)

        safety_buffer = params.min_corridor_width / 2.0

        for geom in self.obstacle_geoms:
            buffered_mask = geom.buffer(safety_buffer, 4)
            self._burn_geometry(buffered_mask)

    def _world_to_rc(self, point: QgsPointXY) -> Optional[Tuple[int, int]]:
        col = int(math.floor((point.x() - self.origin_x) / self.resolution))
        row = int(math.floor((self.origin_y - point.y()) / self.resolution))
        if 0 <= row < self.rows and 0 <= col < self.cols:
            return row, col
        return None

    def _rc_to_world(self, row: int, col: int) -> QgsPointXY:
        x = self.origin_x + (col + 0.5) * self.resolution
        y = self.origin_y - (row + 0.5) * self.resolution
        return QgsPointXY(x, y)

    def _burn_geometry(self, geom: QgsGeometry) -> None:
        bbox = geom.boundingBox()
        min_col = max(0, int(math.floor((bbox.xMinimum() - self.origin_x) / self.resolution)))
        max_col = min(self.cols - 1, int(math.ceil((bbox.xMaximum() - self.origin_x) / self.resolution)))
        min_row = max(0, int(math.floor((self.origin_y - bbox.yMaximum()) / self.resolution)))
        max_row = min(self.rows - 1, int(math.ceil((self.origin_y - bbox.yMinimum()) / self.resolution)))

        if min_col > max_col or min_row > max_row:
            return

        for row in range(min_row, max_row + 1):
            y = self.origin_y - (row + 0.5) * self.resolution
            for col in range(min_col, max_col + 1):
                x = self.origin_x + (col + 0.5) * self.resolution
                try:
                    if geom.contains(QgsPointXY(x, y)):
                        self.passable[row, col] = False
                except Exception:
                    pass

    def find_path(self, start_point: QgsPointXY, end_point: QgsPointXY) -> Optional[List[QgsPointXY]]:
        start = self._world_to_rc(start_point)
        end = self._world_to_rc(end_point)
        if start is None or end is None:
            return None
        start_node = _nearest_passable_node(self.passable, start)
        end_node = _nearest_passable_node(self.passable, end)
        if start_node is None or end_node is None:
            return None

        path = _shortest_path_on_mask(self.passable, start_node, end_node)
        if not path:
            return None
        return [self._rc_to_world(r, c) for r, c in path]


def _nearest_passable_node(mask: np.ndarray, node: Tuple[int, int], search_radius: int = 6) -> Optional[Tuple[int, int]]:
    r0, c0 = node
    rows, cols = mask.shape
    if 0 <= r0 < rows and 0 <= c0 < cols and mask[r0, c0]:
        return node
    for radius in range(1, search_radius + 1):
        for dr in range(-radius, radius + 1):
            for dc in range(-radius, radius + 1):
                nr = r0 + dr
                nc = c0 + dc
                if 0 <= nr < rows and 0 <= nc < cols and mask[nr, nc]:
                    return nr, nc
    return None


def _shortest_path_on_mask(
    mask: np.ndarray, start: Tuple[int, int], goal: Tuple[int, int]
) -> Optional[List[Tuple[int, int]]]:
    rows, cols = mask.shape
    moves = [(-1, 0), (1, 0), (0, -1), (0, 1), (-1, -1), (-1, 1), (1, -1), (1, 1)]

    heap: List[Tuple[float, int, int]] = []
    heapq.heappush(heap, (0.0, start[0], start[1]))
    best_cost: Dict[Tuple[int, int], float] = {start: 0.0}
    parents: Dict[Tuple[int, int], Tuple[int, int]] = {}

    while heap:
        cost, r, c = heapq.heappop(heap)
        if (r, c) == goal:
            break
        if cost > best_cost.get((r, c), float("inf")):
            continue
        for dr, dc in moves:
            nr, nc = r + dr, c + dc
            if not (0 <= nr < rows and 0 <= nc < cols):
                continue
            if not mask[nr, nc]:
                continue
            step = math.sqrt(2) if dr != 0 and dc != 0 else 1.0
            new_cost = cost + step
            if new_cost >= best_cost.get((nr, nc), float("inf")):
                continue
            best_cost[(nr, nc)] = new_cost
            parents[(nr, nc)] = (r, c)
            heapq.heappush(heap, (new_cost, nr, nc))

    if goal not in parents and goal != start:
        return None

    path: List[Tuple[int, int]] = [goal]
    current = goal
    while current != start:
        current = parents.get(current)
        if current is None:
            return None
        path.append(current)
    path.reverse()
    return path


def find_all_possible_corridors(
    patches: Dict[int, Dict],
    spatial_index: QgsSpatialIndex,
    params: VectorRunParams,
    hop_adjacency: Optional[Dict[int, Set[int]]] = None,
    patch_union: Optional[QgsGeometry] = None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
    progress_start: int = 30,
    progress_end: int = 55,
    navigator: Optional[RasterNavigator] = None,
) -> List[Dict]:
    print("  Finding all possible corridors...")
    all_corridors: List[Dict] = []
    processed_pairs: Set[frozenset] = set()
    total = len(patches) or 1

    for idx, (pid1, pdata1) in enumerate(patches.items(), start=1):
        if idx % 10 == 0:
            print(f"    Analyzing patch {idx}/{len(patches)}...", end="\r")
        if progress_cb is not None:
            span = max(progress_end - progress_start, 1)
            progress_value = progress_start + ((idx - 1) / total) * span
            emit_progress(progress_cb, progress_value, "Building corridor candidates…")

        geom1 = pdata1["geom"]
        rect = geom1.boundingBox()
        rect.grow(params.max_search_distance)
        candidate_ids = spatial_index.intersects(rect)

        for pid2 in candidate_ids:
            if pid1 >= pid2:
                continue

            pair = frozenset({pid1, pid2})
            if pair in processed_pairs:
                continue

            pdata2 = patches.get(pid2)
            if not pdata2:
                continue
            geom2 = pdata2["geom"]

            distance = geom1.distance(geom2)
            if distance > params.max_search_distance:
                continue

            p1 = geom1.nearestPoint(geom2).asPoint()
            p2 = geom2.nearestPoint(geom1).asPoint()
            if p1.isEmpty() or p2.isEmpty():
                continue

            if navigator:
                path_points = navigator.find_path(QgsPointXY(p1), QgsPointXY(p2))
                if not path_points:
                    continue

                def _path_cost_length(points: List[QgsPointXY]) -> float:
                    return sum(points[i].distance(points[i + 1]) for i in range(len(points) - 1))

                def _best_path_to_boundary(
                    start_pt: QgsPointXY, patch_geom: QgsGeometry
                ) -> Tuple[Optional[List[QgsPointXY]], Optional[QgsPointXY], float]:
                    candidates: List[QgsPointXY] = []
                    try:
                        densified = patch_geom.densifyByCount(24)
                        geom_iter = densified.vertices()
                    except Exception:
                        geom_iter = patch_geom.vertices()
                    for v in geom_iter:
                        candidates.append(QgsPointXY(v))
                    best_path: Optional[List[QgsPointXY]] = None
                    best_cost = float("inf")
                    best_pt: Optional[QgsPointXY] = None
                    for target in candidates:
                        pts = navigator.find_path(start_pt, target)
                        if not pts:
                            continue
                        cost = _path_cost_length(pts)
                        if cost < best_cost:
                            best_cost = cost
                            best_path = pts
                            best_pt = target
                    return best_path, best_pt, best_cost

                # Build initial geom from straight path
                raw_geom = _create_corridor_geometry(
                    path_points,
                    geom1,
                    geom2,
                    params,
                    obstacle_geoms=navigator.obstacle_geoms,
                    smooth_iterations=3,
                    hop_distance=params.hop_distance if params.stepping_enabled else 0.0,
                )
            else:
                raw_geom = _create_corridor_geometry(
                    [QgsPointXY(p1), QgsPointXY(p2)],
                    geom1,
                    geom2,
                    params,
                    hop_distance=params.hop_distance if params.stepping_enabled else 0.0,
                )

            if raw_geom is None:
                continue

            # If this corridor traverses an intermediate patch, split into entry/exit LCPs
            if navigator:
                intersected_temp = _detect_corridor_intersections(raw_geom, patches, spatial_index, {pid1, pid2})
                if intersected_temp:
                    intermediate_id = min(intersected_temp, key=lambda pid: geom1.distance(patches[pid]["geom"]))
                    mid_geom = patches[intermediate_id]["geom"]

                    path_in, entry_pt, cost_in = _best_path_to_boundary(QgsPointXY(p1), mid_geom)
                    path_out, exit_pt, cost_out = _best_path_to_boundary(QgsPointXY(p2), mid_geom)

                    if path_in and path_out and entry_pt and exit_pt:
                        # Build entry corridor (P1 -> entry)
                        entry_geom = _create_corridor_geometry(
                            path_in,
                            geom1,
                            mid_geom,
                            params,
                            obstacle_geoms=navigator.obstacle_geoms,
                            smooth_iterations=3,
                            hop_distance=params.hop_distance if params.stepping_enabled else 0.0,
                        )
                        # Build exit corridor (exit -> P2)
                        exit_geom = _create_corridor_geometry(
                            path_out,
                            mid_geom,
                            geom2,
                            params,
                            obstacle_geoms=navigator.obstacle_geoms,
                            smooth_iterations=3,
                            hop_distance=params.hop_distance if params.stepping_enabled else 0.0,
                        )
                        # Internal link across intermediate patch (cheap/straight)
                        internal_geom = QgsGeometry.fromPolylineXY([entry_pt, exit_pt]).buffer(
                            params.min_corridor_width / 2.0, BUFFER_SEGMENTS
                        )
                        geoms_to_merge = [g for g in (entry_geom, exit_geom, internal_geom) if g and not g.isEmpty()]
                        if geoms_to_merge:
                            try:
                                split_geom = QgsGeometry.unaryUnion(geoms_to_merge)
                            except Exception:
                                split_geom = geoms_to_merge[0]
                                for g in geoms_to_merge[1:]:
                                    try:
                                        split_geom = split_geom.combine(g)
                                    except Exception:
                                        pass
                            if split_geom is not None and not split_geom.isEmpty():
                                raw_geom = split_geom
                                # Override area/cost with summed external path costs (internal assumed negligible)
                                raw_cost = (cost_in + cost_out) * params.min_corridor_width / 10000.0
                                # A rough conversion: path length * width -> area; keep original as fallback
                                raw_area = split_geom.area() / 10000.0
                                raw_geom_cost = raw_cost if raw_cost > 0 else raw_area
                                corridor_area_ha = raw_geom_cost

            # Pass spatial_index to finalize to allow fast traversal detection
            corridor_geom, patch_ids = _finalize_corridor_geometry(
                pid1, pid2, raw_geom, patches, spatial_index, patch_union=patch_union
            )
            if corridor_geom is None:
                continue

            corridor_area_ha = corridor_geom.area() / 10000.0
            original_area_ha = corridor_area_ha
            is_hop_edge = bool(hop_adjacency) and (
                pid2 in hop_adjacency.get(pid1, set()) or pid1 in hop_adjacency.get(pid2, set())
            )
            # If area is 0 (all overlap) but exists, it's a valid link with 0 cost
            if corridor_area_ha < 0:
                continue
            if (
                params.max_corridor_area is not None
                and corridor_area_ha > params.max_corridor_area
                and not is_hop_edge
            ):
                continue

            if (
                params.stepping_enabled
                and params.hop_distance > 0
                and is_hop_edge
                and navigator is None
            ):
                corridor_area_ha = 0.0

            all_corridors.append(
                {
                    "patch1": pid1,
                    "patch2": pid2,
                    "patch_ids": patch_ids,
                    "geom": clone_geometry(corridor_geom),
                    "area_ha": corridor_area_ha,
                    "original_area_ha": original_area_ha,
                    "distance_m": distance,
                    "is_hop_edge": is_hop_edge,
                }
            )
            processed_pairs.add(pair)

    emit_progress(progress_cb, progress_end, "Candidate corridors ready.")
    print(f"\n  ✓ Found {len(all_corridors)} possible corridors")
    return all_corridors


def optimize_most_connectivity(
    patches: Dict[int, Dict],
    candidates: List[Dict],
    params: VectorRunParams,
    component_sizes: Optional[Dict[int, float]] = None,
    component_counts: Optional[Dict[int, int]] = None,
    adjacency_map: Optional[Dict[int, Set[int]]] = None,
) -> Tuple[Dict[int, Dict], Dict]:
    """
    Enhanced connectivity strategy with proper daisy-chaining:
    1. Backbone: Connect disjoint components, prioritizing traversals (A->C->B over A->B)
    2. Redundancy: Add parallel connections for network resilience
    """
    print("  Strategy: RESILIENT NETWORK (State-Aware + Redundancy)")

    node_map = component_sizes or {pid: pdata["area_ha"] for pid, pdata in patches.items()}
    base_counts = component_counts or {pid: 1 for pid in node_map}

    parent = {pid: pid for pid in node_map}
    size = dict(node_map)
    count = dict(base_counts)

    def find(x: int) -> int:
        if parent[x] != x:
            parent[x] = find(parent[x])
        return parent[x]

    def union(a: int, b: int) -> bool:
        ra, rb = find(a), find(b)
        if ra == rb:
            return False
        if size.get(ra, 0) < size.get(rb, 0):
            ra, rb = rb, ra
        parent[rb] = ra
        size[ra] = size.get(ra, 0) + size.get(rb, 0)
        count[ra] = count.get(ra, 0) + count.get(rb, 0)
        return True

    used_geoms: Set[int] = set()
    covered_pairs: Set[frozenset] = set()
    _log_message("--- Analyzing Candidate Corridors ---")

    traversal_count = 0
    stepping_count = 0
    for cand in candidates:
        comp1 = cand.get("comp1") or cand.get("patch1")
        comp2 = cand.get("comp2") or cand.get("patch2")
        patch_ids = cand.get("patch_ids", {cand.get("patch1"), cand.get("patch2")})
        component_ids = cand.get("component_ids") or {comp1, comp2}
        component_ids = {cid for cid in component_ids if cid is not None}

        cand["_is_traversal"] = len(patch_ids) > 2
        if cand["_is_traversal"]:
            traversal_count += 1
        if cand.get("is_stepping_chain"):
            stepping_count += 1

        # Work in component space to avoid missing parent entries for raw patch IDs.
        cand_roots = {find(cid) for cid in component_ids if cid in parent}
        if not cand_roots and component_ids:
            # Initialize unseen component IDs into the disjoint set, if any.
            for cid in component_ids:
                if cid not in parent:
                    parent[cid] = cid
                    size[cid] = size.get(cid, 0.0)
                    count[cid] = count.get(cid, 1)
                    cand_roots.add(cid)

        base_cost = cand.get("area_ha", 0.0)
        cand["cost_metric"] = 0.0001 if cand.get("is_stepping_chain") else base_cost

        cand["_unique_roots"] = len(cand_roots)
        cand["_priority"] = sum(size.get(find(cid), 0) for cid in cand_roots)

        if cand.get("_is_traversal") and not cand.get("is_stepping_chain"):
            penalty = 1.8
            cand["area_ha"] *= penalty
            cand["cost_metric"] *= penalty

    _log_message(
        f"Candidates: total={len(candidates)}, stepping_chains={stepping_count}, traversals={traversal_count}"
    )

    log_candidates = [c for c in candidates if c.get("cost_metric", 0.0) > 1e-6]
    top_ten = sorted(log_candidates, key=lambda c: -c["_priority"] / c["cost_metric"])[:10]

    _log_message("--- Top 10 Candidates by Efficiency ---")
    _log_message("Type | Traversal | P1 -> P2 | uniq_roots | priority_ha | area_ha | cost_metric | eff_score")
    _log_message("-------------------------------------------------------------------------")
    for c in top_ten:
        efficiency = c["_priority"] / c["cost_metric"] if c["cost_metric"] > 0 else 0
        log_msg = (
            f"{'Chain' if c.get('is_stepping_chain') else 'Direct'} | "
            f"{c.get('_is_traversal', False)} | "
            f"{c.get('patch1')} -> {c.get('patch2')} | "
            f"{c.get('_unique_roots', 0)} | "
            f"{c.get('_priority', 0):.2f} | "
            f"{c.get('area_ha', 0):.4f} | "
            f"{c.get('cost_metric', 0):.6f} | "
            f"{efficiency:.2f}"
        )
        _log_message(log_msg)

    _log_message("--- Starting Optimization ---")

    selected: Dict[int, Dict] = {}
    budget_used = 0.0
    remaining = params.budget_area

    candidates_sorted = sorted(
        candidates,
        key=lambda c: (
            -c.get("_is_traversal", False),
            -c.get("_unique_roots", 2),
            -c["_priority"],
            c["cost_metric"],
        ),
    )

    backbone_count = 0
    for cand in candidates_sorted:
        cost = cand.get("area_ha", 0.0)
        if cost > remaining:
            _log_message(
                f"Skip (budget): P1={cand.get('patch1')} P2={cand.get('patch2')} cost={cost:.4f} remaining={remaining:.4f}"
            )
            continue

        geom_hash = hash(cand["geom"].asWkb()) if cand.get("geom") else None
        if geom_hash in used_geoms:
            _log_message(f"Skip (duplicate geometry): P1={cand.get('patch1')} P2={cand.get('patch2')}")
            continue

        patch_ids = cand.get("patch_ids", {cand.get("patch1"), cand.get("patch2")})
        component_ids = cand.get("component_ids") or {cand.get("comp1"), cand.get("comp2")}
        component_ids = {cid for cid in component_ids if cid is not None}
        roots = {find(cid) for cid in component_ids if cid in parent}

        if len(roots) <= 1:
            _log_message(
                f"Skip (already connected): P1={cand.get('patch1')} P2={cand.get('patch2')} traversal={cand.get('_is_traversal', False)}"
            )
            continue

        selected[len(selected) + 1] = {
            "geom": clone_geometry(cand["geom"]),
            "patch_ids": patch_ids,
            "component_ids": component_ids,
            "area_ha": cost,
            "connected_area_ha": 0.0,
            "efficiency": 0.0,
        }

        _log_message(
            f"Accept backbone: {'Chain' if cand.get('is_stepping_chain') else 'Direct'} "
            f"{cand.get('patch1')}->{cand.get('patch2')} "
            f"traversal={cand.get('_is_traversal', False)} "
            f"uniq_roots={len(roots)} "
            f"priority={cand.get('_priority', 0):.2f} "
            f"cost={cost:.4f}"
        )

        roots_list = list(roots)
        for r in roots_list[1:]:
            union(roots_list[0], r)

        budget_used += cost
        remaining -= cost
        backbone_count += 1
        if geom_hash:
            used_geoms.add(geom_hash)

        # Track component pairs spanned by this corridor to prevent redundant sub-segments later
        c_list = list(component_ids)
        for i in range(len(c_list)):
            for j in range(i + 1, len(c_list)):
                covered_pairs.add(frozenset({c_list[i], c_list[j]}))

    redundancy_candidates = [c for c in candidates if c.get("area_ha", 0.0) <= remaining]
    redundancy_candidates.sort(key=lambda c: c.get("area_ha", 0.0))

    redundant_count = 0
    for cand in redundancy_candidates:
        cost = cand.get("area_ha", 0.0)
        if cost > remaining:
            continue

        geom_hash = hash(cand["geom"].asWkb()) if cand.get("geom") else None
        if geom_hash in used_geoms:
            continue

        component_ids = cand.get("component_ids") or {cand.get("comp1"), cand.get("comp2")}
        component_ids = {cid for cid in component_ids if cid is not None}
        pair_key = frozenset(component_ids) if len(component_ids) == 2 else frozenset()
        if pair_key in covered_pairs:
            continue

        selected[len(selected) + 1] = {
            "geom": clone_geometry(cand["geom"]),
            "patch_ids": cand.get("patch_ids", set()),
            "component_ids": component_ids,
            "area_ha": cost,
            "connected_area_ha": 0.0,
            "efficiency": 0.0,
        }

        budget_used += cost
        remaining -= cost
        redundant_count += 1
        if geom_hash:
            used_geoms.add(geom_hash)

    final_sizes: Dict[int, float] = {}
    final_counts: Dict[int, int] = {}
    for node in node_map:
        root = find(node)
        final_sizes[root] = size.get(root, 0.0)
        final_counts[root] = count.get(root, 1)

    for entry in selected.values():
        component_ids = entry.get("component_ids") or set()
        roots = [find(cid) for cid in component_ids] if component_ids else []
        if roots:
            entry["connected_area_ha"] = final_sizes.get(roots[0], entry["area_ha"])
            entry["efficiency"] = (
                entry["connected_area_ha"] / entry["area_ha"] if entry["area_ha"] > 0 else 0.0
            )

    n_nodes = len(node_map)
    components = len(final_sizes)
    edges_used = len(selected)
    redundancy = max(0, edges_used - (n_nodes - components))
    avg_degree = (2 * edges_used / n_nodes) if n_nodes > 0 else 0.0

    stats = {
        "strategy": "resilient_network",
        "corridors_used": len(selected),
        "connections_made": len(selected),
        "budget_used_ha": budget_used,
        "total_connected_area_ha": sum(final_sizes.values()),
        "groups_created": len(final_sizes),
        "largest_group_area_ha": max(final_sizes.values()) if final_sizes else 0,
        "patches_connected": sum(final_counts.values()),
        "patches_total": n_nodes,
        "components_remaining": components,
        "redundant_links": redundancy,
        "avg_degree": avg_degree,
        "backbone_corridors": backbone_count,
        "redundant_corridors": redundant_count,
    }
    print(f"  ✓ Selected {len(selected)} corridors (backbone {backbone_count}, redundant {redundant_count})")
    return selected, stats


def optimize_largest_patch(
    patches: Dict[int, Dict],
    candidates: List[Dict],
    params: VectorRunParams,
    component_sizes: Optional[Dict[int, float]] = None,
    component_counts: Optional[Dict[int, int]] = None,
) -> Tuple[Dict[int, Dict], Dict]:
    print("  Strategy: LARGEST PATCH (priority growth)")

    base_component_sizes = component_sizes or {pid: pdata["area_ha"] for pid, pdata in patches.items()}
    base_component_counts = component_counts or {pid: 1 for pid in base_component_sizes}

    adjacency: Dict[int, List[Dict]] = {}
    for cand in candidates:
        for comp_id in cand.get("component_ids", {cand.get("comp1"), cand.get("comp2")}) or []:
            adjacency.setdefault(comp_id, []).append(cand)

    sorted_components = sorted(base_component_sizes.keys(), key=lambda k: base_component_sizes[k], reverse=True)
    seed_candidates = sorted_components[:1]

    print("  Testing 1 seed patch (largest)...")
    best_result = {"corridors": {}, "final_area": 0.0, "seed_id": None, "patch_count": 0, "budget_used": 0.0}

    for i, seed_id in enumerate(seed_candidates):
        component_set: Set[int] = {seed_id}
        current_area = base_component_sizes.get(seed_id, 0.0)
        patch_total = base_component_counts.get(seed_id, 1)
        remaining_budget = params.budget_area
        sim_corridors: Dict[int, Dict] = {}
        counter = 0
        pq: List[Tuple[float, float, int, Dict]] = []

        def enqueue_neighbors(comp_id: int) -> None:
            nonlocal counter
            for cand in adjacency.get(comp_id, []):
                cand_components = cand.get("component_ids") or {cand.get("comp1"), cand.get("comp2")}
                new_components = set(cand_components or set()) - component_set
                if not new_components:
                    continue
                potential_area = current_area + sum(base_component_sizes.get(tc, 0.0) for tc in new_components) + cand["area_ha"]
                heapq.heappush(pq, (-potential_area, cand["area_ha"], counter, cand))
                counter += 1

        enqueue_neighbors(seed_id)

        while pq and remaining_budget > 0:
            neg_potential, cost, _, cand = heapq.heappop(pq)
            cand_components = cand.get("component_ids") or {cand.get("comp1"), cand.get("comp2")}
            new_components = set(cand_components or set()) - component_set
            if not new_components:
                continue
            if cost > remaining_budget:
                continue

            remaining_budget -= cost
            component_set.update(new_components)
            added_area = sum(base_component_sizes.get(tc, 0.0) for tc in new_components)
            current_area += added_area + cost
            patch_total += sum(base_component_counts.get(tc, 1) for tc in new_components)

            sim_corridors[len(sim_corridors) + 1] = {
                "geom": cand["geom"],
                "patch_ids": set(cand.get("patch_ids", set())),
                "area_ha": cost,
                "connected_area_ha": current_area,
                "efficiency": cost / current_area if current_area > 0 else 0,
            }

            for comp_id in new_components:
                enqueue_neighbors(comp_id)

        if current_area > best_result["final_area"]:
            best_result = {
                "corridors": sim_corridors,
                "final_area": current_area,
                "seed_id": seed_id,
                "budget_used": params.budget_area - remaining_budget,
                "patch_count": patch_total,
            }

    print(f"\n  ✓ Found optimal seed patch {best_result['seed_id']}")

    if not best_result["corridors"]:
        return {}, {"strategy": "largest_patch", "corridors_used": 0}

    stats = {
        "strategy": "largest_patch",
        "seed_id": best_result["seed_id"],
        "seed_area_ha": base_component_sizes.get(best_result["seed_id"], 0.0),
        "final_patch_area_ha": best_result["final_area"],
        "corridors_used": len(best_result["corridors"]),
        "budget_used_ha": best_result["budget_used"],
        "patches_merged": best_result.get("patch_count", len(best_result["corridors"]) + 1),
        "groups_created": 1,
    }
    return best_result["corridors"], stats


def _thicken_corridors(
    corridors: Dict[int, Dict],
    remaining_budget: float,
    params: VectorRunParams,
    max_width_factor: float = 5.0,
) -> float:
    """
    Use remaining budget to widen existing corridors up to a max factor of the base width.
    Returns additional budget used.
    """
    if remaining_budget <= 0 or not corridors:
        return 0.0

    print(f"\n  Thickening corridors with remaining budget: {remaining_budget:.4f} ha")
    budget_used = 0.0
    max_buffer = (params.min_corridor_width * (max_width_factor - 1)) / 2.0
    buffer_step = params.min_corridor_width * 0.1  # grow by 10% of base width per pass
    current_buffer = 0.0

    while budget_used < remaining_budget and current_buffer < max_buffer:
        current_buffer = min(current_buffer + buffer_step, max_buffer)
        step_cost = 0.0
        temp_updates: Dict[int, Tuple[QgsGeometry, float]] = {}

        for cid, cdata in corridors.items():
            base_geom = cdata.get("_thicken_base_geom")
            if base_geom is None:
                base_geom = clone_geometry(cdata.get("geom"))
                cdata["_thicken_base_geom"] = base_geom
            if base_geom is None or base_geom.isEmpty():
                continue
            try:
                thickened_geom = base_geom.buffer(current_buffer, BUFFER_SEGMENTS)
            except Exception:
                continue
            if thickened_geom is None or thickened_geom.isEmpty():
                continue
            new_area = thickened_geom.area() / 10000.0
            old_area = float(cdata.get("area_ha", new_area))
            added_area = new_area - old_area
            if added_area <= 0:
                continue
            step_cost += added_area
            temp_updates[cid] = (thickened_geom, new_area)

        if budget_used + step_cost > remaining_budget or not temp_updates:
            print("  Budget exhausted or no further thickening possible; stopping.")
            break

        for cid, (geom, new_area) in temp_updates.items():
            corridors[cid]["geom"] = geom
            corridors[cid]["area_ha"] = new_area

        budget_used += step_cost
        print(f"    - Thickened by {current_buffer * 2:.1f} m (Cost: {step_cost:.4f} ha)")
        if current_buffer >= max_buffer:
            print(f"  Maximum corridor width reached ({max_width_factor}x min width).")
            break

    print(f"  ✓ Finalized thickening. Total extra budget used: {budget_used:.4f} ha")
    return budget_used


def write_corridors_layer_to_gpkg(
    corridors: Dict[int, Dict],
    output_path: str,
    layer_name: str,
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    unit_system: str,
    overwrite_file: bool = False,
) -> bool:
    print(f"\nWriting layer '{layer_name}' to {os.path.basename(output_path)} ...")
    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())

    is_imperial = unit_system == "imperial"
    area_field = "area_ac" if is_imperial else "area_ha"
    conn_field = "conn_area_ac" if is_imperial else "conn_area_ha"
    area_factor = 2.471053814 if is_imperial else 1.0

    fields = QgsFields()
    fields.append(QgsField("corridor_id", QVariant.Int))
    fields.append(QgsField("patch_ids", QVariant.String))
    fields.append(QgsField(area_field, QVariant.Double))
    fields.append(QgsField(conn_field, QVariant.Double))
    fields.append(QgsField("efficiency", QVariant.Double))
    fields.append(QgsField("multipart", QVariant.Bool))
    fields.append(QgsField("segment_count", QVariant.Int))

    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GPKG"
    save_options.fileEncoding = "UTF-8"
    save_options.layerName = layer_name
    save_options.actionOnExistingFile = (
        QgsVectorFileWriter.CreateOrOverwriteFile if overwrite_file else QgsVectorFileWriter.CreateOrOverwriteLayer
    )

    writer = QgsVectorFileWriter.create(
        output_path,
        fields,
        QgsWkbTypes.Polygon,
        original_crs,
        QgsProject.instance().transformContext(),
        save_options,
    )

    if writer.hasError() != QgsVectorFileWriter.NoError:
        print(f"  ✗ Error: {writer.errorMessage()}")
        return False

    for cid, cdata in corridors.items():
        feat = QgsFeature(fields)
        geom = clone_geometry(cdata["geom"])
        geom.transform(transform)
        feat.setGeometry(geom)
        multipart = geom.isMultipart()
        segment_count = geom.constGet().numGeometries() if multipart else 1
        feat.setAttributes(
            [
                cid,
                ",".join(map(str, sorted(cdata["patch_ids"]))),
                round(cdata["area_ha"] * area_factor, 4),
                round(cdata["connected_area_ha"] * area_factor, 2),
                round(cdata["efficiency"], 6),
                multipart,
                segment_count,
            ]
        )
        writer.addFeature(feat)

    del writer
    print(f"  ✓ Wrote {len(corridors)} corridors to layer '{layer_name}'")
    return True


def add_layer_to_qgis_from_gpkg(gpkg_path: str, layer_name: str) -> None:
    uri = f"{gpkg_path}|layername={layer_name}"
    layer = QgsVectorLayer(uri, layer_name, "ogr")
    if layer.isValid():
        QgsProject.instance().addMapLayer(layer)
        print(f"  ✓ Added '{layer_name}' to QGIS project")
    else:
        print(f"  ✗ Could not add '{layer_name}' from {gpkg_path}")


def create_step_stone_vector_layer(
    patches: Dict[int, Dict],
    component_map: Dict[int, int],
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    output_path: str,
    layer_name: str = "Step stone connectivity",
) -> bool:
    components: Dict[int, List[QgsGeometry]] = defaultdict(list)
    for pid, pdata in patches.items():
        comp_id = component_map.get(pid, pid)
        geom = pdata.get("geom")
        if geom and not geom.isEmpty():
            components[comp_id].append(clone_geometry(geom))

    layer = QgsVectorLayer(f"Polygon?crs={original_crs.authid()}", layer_name, "memory")
    provider = layer.dataProvider()
    provider.addAttributes(
        [
            QgsField("component_id", QVariant.Int),
            QgsField("patch_count", QVariant.Int),
            QgsField("area_ha", QVariant.Double),
        ]
    )
    layer.updateFields()

    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())
    features: List[QgsFeature] = []
    for comp_id, geom_list in components.items():
        merged_geom = geom_list[0] if len(geom_list) == 1 else QgsGeometry.unaryUnion(geom_list)
        if merged_geom is None or merged_geom.isEmpty():
            continue
        merged_geom.transform(transform)
        feat = QgsFeature(layer.fields())
        feat.setGeometry(merged_geom)
        feat.setAttributes([comp_id, len(geom_list), merged_geom.area() / 10000.0])
        features.append(feat)

    provider.addFeatures(features)
    layer.updateExtents()

    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GPKG"
    save_options.fileEncoding = "UTF-8"
    save_options.layerName = layer_name
    save_options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer

    writer = QgsVectorFileWriter.create(
        output_path,
        layer.fields(),
        QgsWkbTypes.Polygon,
        original_crs,
        QgsProject.instance().transformContext(),
        save_options,
    )
    if writer.hasError() != QgsVectorFileWriter.NoError:
        return False
    for feat in features:
        writer.addFeature(feat)
    del writer
    return True


def create_memory_layer_from_corridors(
    corridors: Dict[int, Dict],
    layer_name: str,
    target_crs: QgsCoordinateReferenceSystem,
    original_crs: QgsCoordinateReferenceSystem,
    unit_system: str,
) -> Optional[QgsVectorLayer]:
    is_imperial = unit_system == "imperial"
    area_field = "area_ac" if is_imperial else "area_ha"
    conn_field = "conn_area_ac" if is_imperial else "conn_area_ha"
    area_factor = 2.471053814 if is_imperial else 1.0

    layer = QgsVectorLayer(f"Polygon?crs={original_crs.authid()}", layer_name, "memory")
    provider = layer.dataProvider()
    provider.addAttributes(
        [
            QgsField("corridor_id", QVariant.Int),
            QgsField("patch_ids", QVariant.String),
            QgsField(area_field, QVariant.Double),
            QgsField(conn_field, QVariant.Double),
            QgsField("efficiency", QVariant.Double),
            QgsField("multipart", QVariant.Bool),
            QgsField("segment_count", QVariant.Int),
        ]
    )
    layer.updateFields()

    transform = QgsCoordinateTransform(target_crs, original_crs, QgsProject.instance())

    features = []
    for cid, cdata in corridors.items():
        geom = clone_geometry(cdata["geom"])
        geom.transform(transform)
        multipart = geom.isMultipart()
        segment_count = geom.constGet().numGeometries() if multipart else 1
        feat = QgsFeature(layer.fields())
        feat.setGeometry(geom)
        feat.setAttributes(
            [
                cid,
                ",".join(map(str, sorted(cdata["patch_ids"]))),
                round(cdata["area_ha"] * area_factor, 4),
                round(cdata["connected_area_ha"] * area_factor, 2),
                round(cdata["efficiency"], 6),
                multipart,
                segment_count,
            ]
        )
        features.append(feat)

    provider.addFeatures(features)
    layer.updateExtents()
    QgsProject.instance().addMapLayer(layer)
    print(f"  ✓ Added temporary layer '{layer_name}' to QGIS project")
    return layer


def _convert_stats_for_units(stats: Dict, unit_system: str) -> Dict:
    factor = 2.471053814 if unit_system == "imperial" else 1.0
    label = "ac" if unit_system == "imperial" else "ha"
    converted = dict(stats)
    if "budget_used_ha" in stats:
        converted["budget_used_display"] = stats["budget_used_ha"] * factor
    if "total_connected_area_ha" in stats:
        converted["total_connected_area_display"] = stats["total_connected_area_ha"] * factor
    if "largest_group_area_ha" in stats:
        converted["largest_group_area_display"] = stats["largest_group_area_ha"] * factor
    if "seed_area_ha" in stats:
        converted["seed_area_display"] = stats["seed_area_ha"] * factor
    if "final_patch_area_ha" in stats:
        converted["final_patch_area_display"] = stats["final_patch_area_ha"] * factor
    if "total_patch_area_ha" in stats:
        converted["total_patch_area_display"] = stats["total_patch_area_ha"] * factor
    converted["area_units_label"] = label
    converted["conversion_factor"] = factor
    return converted


def run_vector_analysis(
    layer: QgsVectorLayer,
    output_dir: str,
    raw_params: Dict,
    strategy: str = "most_connectivity",
    temporary: bool = False,
    iface=None,
    progress_cb: Optional[Callable[[int, Optional[str]], None]] = None,
) -> List[Dict]:
    """Execute the vector corridor analysis for the provided polygon layer."""
    if not isinstance(layer, QgsVectorLayer) or not layer.isValid():
        raise VectorAnalysisError("Selected layer is not a valid vector layer.")

    params = _to_dataclass(raw_params)
    unit_system = params.unit_system
    area_factor = 2.471053814 if unit_system == "imperial" else 1.0
    area_label = "ac" if unit_system == "imperial" else "ha"

    if temporary:
        output_path = ""
    else:
        out_dir = output_dir or os.path.dirname(layer.source())
        os.makedirs(out_dir, exist_ok=True)
        output_path = os.path.join(out_dir, params.output_name)

    overall_start = time.time()
    print("=" * 70)
    print("LINKSCAPE VECTOR ANALYSIS v23.3")
    print("=" * 70)
    print("\n1. Loading vector layer...")
    print(f"  ✓ Using layer: {layer.name()} ({layer.featureCount()} features)")
    emit_progress(progress_cb, 5, "Loading vector layer…")

    original_crs = layer.crs()
    print(f"  CRS: {original_crs.authid()}")

    print("\n2. Determining analysis CRS...")
    target_crs = get_utm_crs_from_extent(layer)
    print(f"  ✓ Using {target_crs.authid()} for measurements")
    emit_progress(progress_cb, 15, "Preparing data…")

    # Auto-scale max search distance if user left it unset/zero
    try:
        extent_src = layer.extent()
        transform_extent = QgsCoordinateTransform(layer.crs(), target_crs, QgsProject.instance())
        extent = transform_extent.transformBoundingBox(extent_src)
        max_dimension = max(extent.width(), extent.height())
        DEFAULT_SCALING_FACTOR = 0.25
        if params.max_search_distance <= 0 and max_dimension > 0:
            params.max_search_distance = max_dimension * DEFAULT_SCALING_FACTOR
            _log_message(
                f"Auto-setting max search distance to {DEFAULT_SCALING_FACTOR*100:.0f}% of map extent "
                f"({params.max_search_distance:.1f} units)."
            )
    except Exception:
        pass

    print("\n3. Loading patches...")
    patches, spatial_index = load_and_prepare_patches(layer, target_crs, params)
    if not patches:
        raise VectorAnalysisError("No patches found after filtering.")
    patch_union: Optional[QgsGeometry] = None
    try:
        patch_union = QgsGeometry.unaryUnion([p["geom"] for p in patches.values()])
    except Exception:
        patch_union = None
    hop_adjacency: Dict[int, Set[int]] = {}
    component_map: Dict[int, int] = {pid: pid for pid in patches}
    component_sizes: Dict[int, float] = {pid: pdata["area_ha"] for pid, pdata in patches.items()}
    component_counts: Dict[int, int] = {pid: 1 for pid in patches}
    hop_display = params.hop_distance * 3.280839895 if unit_system == "imperial" else params.hop_distance
    hop_label = "ft" if unit_system == "imperial" else "m"
    stats_step: Dict[str, float] = {}

    if params.stepping_enabled and params.hop_distance > 0:
        emit_progress(progress_cb, 28, "Checking stepping-stone proximity…")
        print("\n3b. Stepping-stone preprocessing...")
        hop_adjacency = _compute_vector_hop_adjacency(patches, spatial_index, params.hop_distance)
        if hop_adjacency:
            component_map, component_sizes, component_counts = _build_virtual_components(patches, hop_adjacency)
            print(
                f"  ✓ Virtual merges: {len(component_sizes):,} starting components "
                f"(hop distance ≤ {hop_display:.2f} {hop_label})"
            )
            try:
                if not temporary and output_path:
                    step_success = create_step_stone_vector_layer(
                        patches,
                        component_map,
                        target_crs,
                        original_crs,
                        output_path,
                        "Step stone connectivity (Below hop distance)",
                    )
                    if step_success:
                        add_layer_to_qgis_from_gpkg(output_path, "Step stone connectivity (Below hop distance)")
                        print("  ✓ Created step stone connectivity layer")
                original_patches = len(patches)
                virtual_components = len(component_sizes)
                patches_merged = original_patches - virtual_components
                stats_step = {
                    "original_patches": original_patches,
                    "virtual_components": virtual_components,
                    "patches_merged": patches_merged,
                    "hop_distance_m": params.hop_distance,
                }
            except Exception as step_exc:  # noqa: BLE001
                print(f"  ⚠ Could not create step stone connectivity layer: {step_exc}")
        else:
            print("  ✓ No patches within hop distance; proceeding without virtual merges.")
    emit_progress(progress_cb, 35, "Searching for corridor candidates…")

    navigator: Optional[RasterNavigator] = None
    if params.obstacle_enabled and params.obstacle_layer_ids:
        obstacle_layers: List[QgsVectorLayer] = []
        skipped_ids: List[str] = []
        for layer_id in params.obstacle_layer_ids:
            layer = QgsProject.instance().mapLayer(layer_id)
            if isinstance(layer, QgsVectorLayer) and layer.isValid() and QgsWkbTypes.geometryType(layer.wkbType()) == QgsWkbTypes.PolygonGeometry:
                obstacle_layers.append(layer)
            else:
                skipped_ids.append(str(layer_id))

        if skipped_ids:
            print(
                f"  ⚠ Skipped {len(skipped_ids)} obstacle layer(s) that are unavailable or not polygon geometry."
            )

        if obstacle_layers:
            try:
                navigator = RasterNavigator(patches, obstacle_layers, target_crs, params)
                print(
                    f"  ✓ Raster navigator grid: {navigator.cols} × {navigator.rows} cells "
                    f"@ {navigator.resolution:.1f} units "
                    f"using {len(obstacle_layers)} obstacle layer(s)"
                )
            except VectorAnalysisError as exc:
                print(f"  ⚠ Obstacle avoidance disabled: {exc}")
                navigator = None
        else:
            print("  ⚠ Selected obstacle layers are unavailable; continuing without obstacle avoidance.")

    print("\n4. Precomputing candidate corridors...")
    all_possible = find_all_possible_corridors(
        patches,
        spatial_index,
        params,
        hop_adjacency=hop_adjacency,
        patch_union=patch_union,
        progress_cb=progress_cb,
        progress_start=35,
        progress_end=60,
        navigator=navigator,
    )
    _annotate_candidates_with_components(all_possible, component_map)
    direct_pairs = _build_component_pair_index(all_possible)
    hop_lookup = _build_hop_edge_lookup(all_possible, hop_adjacency)
    if params.stepping_enabled and params.hop_distance > 0:
        chains = _build_stepping_chains(hop_lookup, component_map, component_sizes)
        if chains:
            print(f"  ✓ Added {len(chains):,} stepping-stone corridor chains")
            all_possible.extend(chains)

    strategy = (strategy or "most_connectivity").lower()
    if strategy not in ("most_connectivity", "largest_patch"):
        raise VectorAnalysisError(f"Unsupported strategy '{strategy}'.")
    adjacency_map = _build_component_adjacency(all_possible)

    print("\nTOP 20 CANDIDATES BY COST (after hop bonus):")
    sorted_cands = sorted(all_possible, key=lambda x: x.get("area_ha", 999))
    for i, c in enumerate(sorted_cands[:20]):
        chain = " [CHAIN]" if c.get("is_stepping_chain") else ""
        hop_flag = " [HOP]" if c.get("is_hop_edge") else ""
        cost = c.get("area_ha", 0.0)
        orig = c.get("original_area_ha", cost)
        comps = f"{c.get('comp1')}-{c.get('comp2')}"
        patch_count = len(c.get("patch_ids", {c.get('patch1'), c.get('patch2')}))
        print(
            f"  {i+1:2d}. {cost:8.5f} ha{chain}{hop_flag}  comp {comps}  via {patch_count} patches  (orig {orig:.3f})"
        )

    layer_name = "Corridors (Resilient Network)" if strategy == "most_connectivity" else "Corridors (Largest Patch)"

    print("\n5. Running optimization...")
    print("=" * 70)
    print(f"--- {strategy.replace('_', ' ').upper()} ---")
    if strategy == "largest_patch":
        print("  Strategy: SINGLE DOMINANT NETWORK (grow one cohesive network from the largest patch)")
    emit_progress(progress_cb, 65, "Running optimization…")

    if not all_possible:
        raise VectorAnalysisError(
            _format_no_corridor_reason(
                "Precomputation",
                len(patches),
                len(all_possible),
                params,
            )
        )

    if strategy == "most_connectivity":
        corridors, stats = optimize_most_connectivity(
            patches,
            all_possible,
            params,
            component_sizes=component_sizes,
            component_counts=component_counts,
            adjacency_map=adjacency_map,
        )
    else:
        corridors, stats = optimize_largest_patch(
            patches,
            all_possible,
            params,
            component_sizes=component_sizes,
            component_counts=component_counts,
        )

    if not corridors:
        raise VectorAnalysisError(
            _format_no_corridor_reason(
                "Optimization",
                len(patches),
                len(all_possible),
                params,
            )
        )

    if params.stepping_enabled and params.hop_distance > 0:
        stats["note"] = f"Stepping-stone connectivity enabled (hop distance {hop_display:.2f} {hop_label})"
        stats["patches_merged"] = len(patches) - len(component_sizes)
        if stats_step:
            stats["step_stone_stats"] = stats_step

    # Allocate leftover budget to thicken existing corridors (most_connectivity only)
    if strategy == "most_connectivity":
        budget_used = stats.get("budget_used_ha", 0.0)
        remaining_budget = params.budget_area - budget_used
        if remaining_budget > (params.budget_area * 0.05):
            extra_used = _thicken_corridors(corridors, remaining_budget, params)
            if extra_used > 0:
                stats["budget_used_ha"] = budget_used + extra_used
                stats["thickened_corridors"] = True

    stats = _convert_stats_for_units(stats, unit_system)
    stats["budget_total_display"] = params.budget_area * area_factor

    print("  Preparing outputs...")
    emit_progress(progress_cb, 90, "Writing outputs…")
    if temporary:
        create_memory_layer_from_corridors(corridors, layer_name, target_crs, original_crs, unit_system)
    else:
        write_corridors_layer_to_gpkg(
            corridors,
            output_path,
            layer_name,
            target_crs,
            original_crs,
            unit_system,
            overwrite_file=True,
        )
        add_layer_to_qgis_from_gpkg(output_path, layer_name)

    elapsed = time.time() - overall_start
    emit_progress(progress_cb, 100, "Vector analysis complete.")

    print("\n" + "=" * 70)
    print("FINAL SUMMARY")
    print("=" * 70)
    stats_strategy = stats.get("strategy", strategy)
    strategy_label = (
        "Resilient Network"
        if stats_strategy in ("most_connectivity", "resilient_network")
        else "Single Dominant Network"
        if stats_strategy == "largest_patch"
        else stats_strategy.replace("_", " ").title()
    )
    print(f"Strategy:          {strategy_label}")
    print(f"Corridors created: {stats.get('corridors_used', 0)}")
    if strategy == "largest_patch":
        print(f"Seed patch:        {stats.get('seed_id')}")
        print(f"Seed area:         {stats.get('seed_area_display', 0):.2f} {area_label}")
        print(f"Final patch size:  {stats.get('final_patch_area_display', 0):.2f} {area_label}")
        print(f"Patches merged:    {stats.get('patches_merged', 0)}")
    else:
        print(f"Connections:       {stats.get('connections_made', 0)}")
        print(f"Total connected:   {stats.get('total_connected_area_display', 0):.2f} {area_label}")
        print(f"Largest group:     {stats.get('largest_group_area_display', 0):.2f} {area_label}")
        if "redundant_links" in stats:
            print(f"Redundant links:   {stats.get('redundant_links', 0)}")
        if "avg_degree" in stats:
            print(f"Average degree:    {stats.get('avg_degree', 0):.2f}")
    if "step_stone_stats" in stats:
        ss = stats["step_stone_stats"]
        print(
            f"Step stone merge:  {ss.get('patches_merged', 0)} patches → {ss.get('virtual_components', 0)} components "
            f"(hop {ss.get('hop_distance_m', ss.get('hop_distance_px', 0))})"
        )
    print(
        f"Budget used:      {stats.get('budget_used_display', 0):.2f} / {params.budget_area * area_factor:.2f} {area_label}"
    )
    print(f"Processing time:   {elapsed:.1f}s")
    if temporary:
        print("Output:            Temporary layer (memory)")
    else:
        print(f"Output GPKG:       {output_path}")
    print("=" * 70)

    stats["layer_name"] = layer_name
    stats["output_path"] = output_path if not temporary else ""
    stats["unit_system"] = unit_system

    return [
        {
            "strategy": strategy,
            "stats": stats,
            "output_path": output_path if not temporary else "",
            "layer_name": layer_name,
        }
    ]
