#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
********************************************************************

* Part of the QGis-Plugin LinearReferencing:
* plugin-wide used Exceptions

********************************************************************

* Date                 : 2025-01-25
* Copyright            : (C) 2023 by Ludwig Kniprath
* Email                : ludwig at kni minus online dot de

********************************************************************

this program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

********************************************************************
"""

from typing import Any

class MyException(Exception):
    def __init__(self, additional_hint:str = ''):
        """Base-Class for all derived Exceptions

        Args:
            additional_hint (str): additional hint from derived Exceptions, mostly optional, used for __str__
        """
        self.additional_hint = additional_hint

    def __str__(self, derived_str):
        my_str = f"{self.__class__.__name__}:\n{derived_str}"
        if self.additional_hint:
            my_str += f":\n{self.additional_hint}"
        return my_str

class LayerNotFoundException(MyException):
    def __init__(self, layer_name:str, additional_hint:str = ''):
        """raised if a layer with layer_name not found

        Args:
            layer_name (str)
            additional_hint (str, optional)
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name

    def __str__(self):
        return super().__str__(f"Layer '{self.layer_name}' not found")

class LayerCreateException(MyException):
    def __init__(self, layer_hint:str, additional_hint:str = ''):
        """raised if a layer (DataLyr, PoProRefLyr, ShowLyr not be created

        Args:
            layer_hint (str): Which type of Layer could not be created
            additional_hint (str, optional)
        """
        super().__init__(additional_hint)
        self.layer_hint = layer_hint

    def __str__(self):
        return super().__str__(f"Layer '{self.layer_hint}' could not be created")

class LayerExpressionInvalidException(MyException):
    def __init__(self, layer_name:str,additional_hint:str = ''):
        """raised if the user-defined Layer-Expression (DataLyr/RefLyr) from Dialog is not valid

        Args:
            layer_name (str)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name

    def __str__(self):
        return super().__str__(f"Expression for Layer '{self.layer_name}' invalid")


class LayerIsEditableException(MyException):
    def __init__(self, layer_name:str, additional_hint:str = ''):
        """if layer is editable and should not, f. e. if Layer-Filter is applied

        Args:
            layer_name (str)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name

    def __str__(self):
        return super().__str__(f"Expression for Layer '{self.layer_name}' is editable")

class LayerTypeException(MyException):
    def __init__(self, layer_name:str, required_type:str, additional_hint:str = ''):
        """Wrong Layer Type (geometry-type, layer-type...)

        Args:
            layer_name (str)
            required_type (str)
            additional_hint (str)
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.required_type = required_type

    def __str__(self):
        return super().__str__(f"Layer '{self.layer_name}' not type '{self.required_type}'")


class LayerNotEditableException(MyException):
    def __init__(self, layer_name:str, additional_hint:str = ''):
        """raised for insert/updates/deletes... if layer is not editable"""
        super().__init__(additional_hint)
        self.layer_name = layer_name

    def __str__(self):
        return super().__str__(f"Layer '{self.layer_name}' not editable")


class LayerNotRegisteredException(MyException):
    def __init__(self, additional_hint):
        """raised if a layer was not registered or if the used layer is not the registered layer

        Args:
            additional_hint (str): error-description, name or type of layer
        """
        super().__init__(additional_hint)

    def __str__(self):
        return super().__str__(f"Layer not registered")

class GeometryConstructException(MyException):
    def __init__(self, additional_hint):
        """any error while constructing the linear reference geometry

        Args:
            additional_hint (str)
        """
        super().__init__(additional_hint)

    def __str__(self):
        return super().__str__("Geometry could not be constructed")

class PropOutOfRangeException(MyException):
    def __init__(self, prop_name:str, prop_value: Any, min_value:Any, max_value:Any, additional_hint:str = ''):
        """raised if property is out of range, f.e. stationing < 0 or > reference-line-length

        Args:
            prop_name (str)
            prop_value (Any)
            min_value (Any)
            max_value (Any)
        """
        super().__init__(additional_hint)
        self.prop_name = prop_name
        self.prop_value = prop_value
        self.min_value = min_value
        self.max_value = max_value

    def __str__(self):
        return super().__str__(f"value '{self.prop_value}' for property '{self.prop_name}' out of range {self.min_value} ... {self.max_value}")


class PropMissingException(MyException):
    def __init__(self, prop_name:str, additional_hint:str = ''):
        """raised, if a property inside an object is not set

        Args:
            prop_name (str)
        """
        super().__init__(additional_hint)
        self.prop_name = prop_name

    def __str__(self):
        return super().__str__(f"Property '{self.prop_name}' missing")

class PropInvalidException(MyException):
    def __init__(self, attribute_name:str, argument: Any, additional_hint:str = ''):
        """raised, if a property inside an object is not valid

        Args:
            prop_name (str)
            argument (Any)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.attribute_name = attribute_name
        self.argument = argument

    def __str__(self):
        return super().__str__(f"Value '{self.argument}' for Attribute '{self.attribute_name}' invalid")


class ArgumentInvalidException(MyException):
    def __init__(self, parameter:str, argument: Any, additional_hint:str = ''):
        """Function called with invalid Argument

        Args:
            parameter (str): Parameter-Name in function-definition
            argument (Any): Parameter-Value in function-call
            additional_hint (str, optional):  additional hint like 'lr_mode'
        """
        super().__init__(additional_hint)
        self.parameter = parameter
        self.argument = argument

    def __str__(self):
        return super().__str__(f"Argument '{self.argument}' for parameter '{self.parameter}' invalid")

class ArgumentMissingException(MyException):
    def __init__(self, parameter:str, additional_hint:str=''):
        """Function called with missing Argument
        used for functions accepting multiple arguments to query a feature like f.e. tool_get_reference_feature via ref_feature/ref_fid/ref_id/data_fid

        Args:
            parameter (str): Parameter-Name in function-definition
            additional_hint (str)
        """
        super().__init__(additional_hint)
        self.parameter = parameter

    def __str__(self):
        return super().__str__(f"Missing argument '{self.parameter}'")


class FeatureAttributeInvalidException(MyException):
    def __init__(self, layer_name:str, feature_id:int, field_name:str, additional_hint:str = ''):
        """Feature-Attribute invalid (missing, Null, None, '', wrong type)

        Args:
            layer_name (str)
            feature_id (int)
            field_name (str)
            additional_hint (str, optional). Defaults to ''.
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id
        self.field_name = field_name

    def __str__(self):
        return super().__str__(f"Feature '{self.feature_id}' in layer '{self.layer_name}': Value for field '{self.field_name}' missing")

class RuntimeObjectMissingException(MyException):

    def __init__(self, additional_hint:str):
        """missing or invalid user-defined Runtime-Object like digitized feature, PostProcessing-Feature, selected

        Args:
            additional_hint (str)
        """
        super().__init__(additional_hint)

    def __str__(self):
        return super().__str__(f"Runtime-Object missing or empty")

class IterableEmptyException(MyException):

    def __init__(self, additional_hint:str):
        """empty iterable Object like lists/sets (self.session_data.selected_data_fids, self.session_data.po_pro_cache) or Layers (PostProcessing-Reference-Layer)

        Args:
            additional_hint (str)
        """
        super().__init__(additional_hint)

    def __str__(self):
        return super().__str__(f"Runtime-List empty")

class UserSelectionException(MyException):
    def __init__(self, additional_hint:str):
        """Missing selection/contents in ComboBox, TreeView, TableView, feature-selection, PoPro-Selection

        Args:
            user_hint (str)
        """
        super().__init__(additional_hint)

    def __str__(self):
        return super().__str__(f"User-Selection empty")

class FeatureNotFoundException(MyException):
    def __init__(self, layer_name:str, feature_id:int = None, additional_hint:str = ''):
        """Feature not found in layer (query-result id/fid/attribute empty)

        Args:
            layer_name (str)
            feature_id (int, optional) => default query by fid, but variants
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id

    def __str__(self):
        if self.feature_id:
            return super().__str__(f"Feature '{self.feature_id}' in layer '{self.layer_name}' not found")
        else:
            # features my be defined by data_fid, data_id, data_feature...
            return super().__str__(f"Feature in layer '{self.layer_name}' not found")

class QueryResultEmptyException(MyException):
    def __init__(self, layer_name:str, field_name: str, field_value:Any, additional_hint:str = ''):
        """Layer-Query by value without result

        Args:
            layer_name (str)
            field_name (str) => queried field
            field_value (Any) => query-value
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.field_name = field_name
        self.field_value = field_value

    def __str__(self):
        return super().__str__(f"No features in '{self.layer_name}' with '[{self.field_name}]' == '{self.field_value}'")

class FeatureUpdateException(MyException):
    def __init__(self, layer_name:str, feature_id:int, additional_hint: str = ''):
        """MyException when updating Feature

        Args:
            layer_name (str)
            feature_id (int)
            additional_hint (str, optional) f.e. "Missing stationing"
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id

    def __str__(self):
        return super().__str__(f"Error updating feature '{self.feature_id}' in layer '{self.layer_name}'")




class FeatureInvalidException(MyException):
    def __init__(self, layer_name:str, feature_id:int = None, additional_hint:str = ''):
        """raised if not feature.isValid()

        Args:
            layer_name (str)
            feature_id (int, optional)
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id

    def __str__(self):
        return super().__str__(f"Feature '{self.feature_id}' in layer '{self.layer_name}' invalid or incomplete")


class FeatureWithoutGeometryException(MyException):
    def __init__(self, layer_name:str, feature_id:int, additional_hint:str = ''):
        """raised if not feature.hasGeometry()

        Args:
            layer_name (str)
            feature_id (int)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id


    def __str__(self):
        return super().__str__(f"Feature '{self.feature_id}' in layer '{self.layer_name}' without geometry")



class ReferencedFeatureNotCommittedException(MyException):
    def __init__(self, layer_name:str, feature_id:int = None, additional_hint:str = ''):
        """Ref-fid < 0 => feauture not committed to provider

        Args:
            layer_name (str)
            feature_id (int, optional)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.feature_id = feature_id

    def __str__(self):
        return super().__str__(f"Referenced-Feature '{self.feature_id}' in layer '{self.layer_name}' not yet committed")

class FieldNotFoundException(MyException):
    def __init__(self, layer_name:str, field_name:int, additional_hint:str = ''):
        """Field in layer not found

        Args:
            layer_name (str)
            field_name (int)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name
        self.field_name = field_name

    def __str__(self):
        return super().__str__(f"Field '{self.field_name}' in layer '{self.layer_name}' not found")



class GeometryNotMValidException(MyException):
    def __init__(self, invalid_reason:str, layer_name:str='', feature_id:int=None, additional_hint:str = ''):
        """Geometry not M-enabled or M-valid

        Args:
            invalid_reason (str)
            layer_name (str): optional (geometry without layer/feature)
            feature_id (int): optional (geometry without layer/feature)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.invalid_reason = invalid_reason
        self.layer_name = layer_name
        self.feature_id = feature_id

    def __str__(self):
        if self.layer_name and self.feature_id:
            return super().__str__(f"Geometry in layer '{self.layer_name}' #{self.feature_id} not M-enabled or valid: '{self.invalid_reason}'")
        else:
            return super().__str__(f"Geometry not M-enabled or valid: '{self.invalid_reason}'")

class GeometryInvalidException(MyException):
    def __init__(self, invalid_reason:str, additional_hint:str = ''):
        """Geometry not valid

        Args:
            invalid_reason (str)
            additional_hint (str, optional): error-description, f.e. layer-name and feature-id
        """
        super().__init__(additional_hint)
        self.invalid_reason = invalid_reason

    def __str__(self):
        return super().__str__(f"Geometry invalid: '{self.invalid_reason}'")



class LayerNotMEnabledException(MyException):
    def __init__(self, layer_name:str,additional_hint:str = ''):
        """raised, if Layer not M-enabled

        Args:
            layer_name (str)
        """
        super().__init__(additional_hint)
        self.layer_name = layer_name

    def __str__(self):
        return super().__str__(f"Layer '{self.layer_name}' not M-Enabled")


class NotInSelectionException(MyException):
    def __init__(self, selection_name:str, key:Any, additional_hint:str = ''):
        """Item not in selection, f.e. fid not in feature_selection

        Args:
            selection_name (str)
            key (Any)
            additional_hint (str, optional): error-description
        """
        super().__init__(additional_hint)
        self.selection_name = selection_name
        self.key = key

    def __str__(self):
        return super().__str__(f"Item with key '{self.key}' not in '{self.selection_name}'")