#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
********************************************************************

* Part of the QGis-Plugin LinearReferencing:
* plugin-wide used variables

********************************************************************

* Date                 : 2025-01-25
* Copyright            : (C) 2023 by Ludwig Kniprath
* Email                : ludwig at kni minus online dot de

********************************************************************

this program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

********************************************************************
"""
from enum import IntEnum

from qgis.core import Qgis

# check if Qt6 is enabled in this QGis-Version
from qgis.PyQt.QtCore import QVariant, QMetaType

try:
    from PyQt6.QtCore import QMetaType
    is_qt6 = True
except ImportError:
    is_qt6 = False

# Qt_Roles
# used for storing data in QStandardItemModel, f.e. MyQComboBox, QTreeView...
# QStandardItem.setData(value,role) rsp. value = QStandardItem.data(role)
# (Note QListWidgetItem uses setData(role,value))
# the model-items (QtGui.QStandardItem) store data in roles
# Qt uses standard-roles in range 0...255 (DisplayRole => 0, DecorationRole => 1, EditRole => 2, TextAlignmentRole, FontRole, ToolTipRole, ...UserRole => 256)
# QtCore.Qt.UserRole = 256
# additional roles should use integers > UserRole (256),
# each added value inside below IntEnum gets an incremented integer starting with 257
# usage:
# import to python-file via from LinearReferencing.settings.constants import Qt_Roles
# inside these files Qt_Roles is a global variable, access to integer-values via name: Qt_Roles.CUSTOM_SORT, Qt_Roles.REF_FID etc.

# CUSTOM_SORT: used inside MyStandardItem for sort by (numeric) value instead default by Text == DisplayRole, usefull for numerical values, displayed formatted, which should be sorted in numerical instead of alphabetical range
# ORIGINAL_VALUE: used inside MyStandardItem to store f.e. queried values without formatting, usefull for editable widgets to get the original value
# CURRENT_VALUE: used inside MyStandardItem for editable widgets to get the current value
# DATA_FID: -> role for the Feature-ID of data-layer-feature
# REF_FID: -> role for the Feature-ID of reference-layer-feature
# SHOW_FID: role for the Feature-ID of show-layer-feature
# OPTION_TEXT: MyQComboBox -> role for Wildcard-replacements {0} {1}... in option_text_template, used for the selected option (QLineEdit)
# RETURN_VALUE: universal role, f.e. QgsField inside MyQComboBox in Plugin-layer-field-configuration
Qt_Roles = IntEnum(
    "Qt_Roles",
    [
        "CUSTOM_SORT",
        "REF_FID",
        "DATA_FID",
        "SHOW_FID",
        "ROW_IDX",
        "COL_IDX",
        "RETURN_VALUE",
        "ORIGINAL_VALUE",
        "CURRENT_VALUE",
        "OPTION_TEXT",
    ],
    start=257,
)


# some Qt-internal used constants, dependend on qt5/qt6 and QGis-Version
# some contents "borrowed" from Tuflow-Plugin:
# ~/.local/share/QGIS/QGIS3/profiles/default/python/plugins/tuflow/compatibility_routines.py

# data types, used for QgsField-creation
# since 3.38 new signature for QgsField-constructor, see https://api.qgis.org/api/classQgsField.html
# The use of QVariant for the field-type is deprecated, QMetaType is used instead
# https://doc.qt.io/qt-5/qmetatype.html#Type-enum
# old -> new:
# QVariant.Int -> QMetaType.Int
# QVariant.Double -> QMetaType.Double
# QVariant.String -> QMetaType.QString (!)
# since qt6 (QGis 4.x) QMetaType, but other Qt-internal signature
if is_qt6:
    QT_STRING = QMetaType.Type.QString
    QT_INT = QMetaType.Type.Int
    QT_LONG = QMetaType.Type.Long
    QT_LONG_LONG = QMetaType.Type.LongLong
    QT_FLOAT = QMetaType.Type.Float
    QT_DOUBLE = QMetaType.Type.Double
else:

    if Qgis.QGIS_VERSION_INT < 33800:
        QT_STRING = QVariant.String
        QT_INT = QVariant.Int
        QT_LONG = QVariant.LongLong
        QT_LONG_LONG = QVariant.LongLong
        QT_FLOAT = QVariant.Double
        QT_DOUBLE = QVariant.Double
    else:
        QT_STRING = QMetaType.QString
        QT_INT = QMetaType.Int
        QT_LONG = QMetaType.Long
        QT_LONG_LONG = QMetaType.LongLong
        QT_FLOAT = QMetaType.Float
        QT_DOUBLE = QMetaType.Double

