# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LecoS
                                 A QGIS plugin
 Contains analytical functions for landscape analysis
                             -------------------
        begin                : 2012-09-06
        copyright            : (C) 2013 by Martin Jung
        email                : martinjung@zoho.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import PyQT bindings
from PyQt4.QtCore import *
from PyQt4.QtGui import *

# Import QGIS analysis tools
from qgis.core import *
from qgis.gui import *
from qgis.analysis import *

# Import base libraries
import os,sys,csv,string,math,operator,subprocess,tempfile,inspect
from os import path

# Import numpy and scipy
import numpy
try:
    import scipy
except ImportError:
    QMessageBox.warning(QDialog(),"LecoS: Warning","Please install scipy (http://scipy.org/) in your QGIS python path.")
    sys.exit(0)
from scipy import ndimage # import ndimage module seperately for easy access

# Import GDAL and ogr
try:
    from osgeo import gdal
except ImportError:
    import gdal
try:
    from osgeo import ogr
except ImportError:
    import ogr

# Register gdal and ogr drivers
if hasattr(gdal,"AllRegister"): # Can register drivers
    gdal.AllRegister() # register all gdal drivers
if hasattr(ogr,"RegisterAll"):
    ogr.RegisterAll() # register all ogr drivers

# import Ui
from ui.dlg_landscapestatistics import Ui_Lecos
from ui.dlg_PolygonPerLandCover import Ui_BatchDialog
from ui.dlg_DiversityDialog import Ui_DivDialog
from ui.dlg_LandscapeModifier import Ui_LandMod

# Import functions and metrics
import lecos_functions as func
import landscape_statistics as lcs
import landscape_diversity as ldiv
import landscape_polygonoverlay as pov 
import landscape_modifier as lmod

tmpdir = tempfile.gettempdir()

## CODE START ##
# create the dialog controls and set up the user interface from Designer.
class LecosDialog(QDialog, Ui_Lecos):
    def __init__(self, iface):
        # Initialize the Dialog
        QDialog.__init__( self )
        self.setupUi(self)
        self.iface = iface
        
        # Get and set current pixelsize
        if self.cbRaster.currentIndex() == -1:
            self.sp_cellsize.setEnabled( False )
            self.NoDataVal.setEnabled( False )
        else:
            self.sp_cellsize.setEnabled( True )
            self.NoDataVal.setEnabled( True )
            self.NoDataVal.setToolTip(QString(str("Please note that the nodata value must be an integer")))
            self.cellSizer( self.cbRaster.currentText() )
            self.setNoData( self.cbRaster.currentText() )
        
        #Change on Rasterfile
        QObject.connect( self.cbRaster, SIGNAL( "currentIndexChanged( QString )" ), self.cellSizer)
        QObject.connect( self.cbRaster, SIGNAL( "currentIndexChanged( QString )" ), self.setNoData)
                        
        # Save File
        QObject.connect( self.where2Save, SIGNAL( "clicked()" ), self.selectSaveFile )
        QObject.connect( self.rbDirect, SIGNAL( "clicked()" ), self.savetoggle )
        QObject.connect( self.rbSAVE, SIGNAL( "clicked()" ), self.savetoggle )


        # Change on Metric-Tabs
        QObject.connect( self.SingleMetric, SIGNAL( "currentIndexChanged( QString )" ), self.showFunctionHelp)
        QObject.connect( self.btn_right, SIGNAL( "clicked()" ), self.switchR )
        QObject.connect( self.btn_left, SIGNAL( "clicked()" ), self.switchL )
        QObject.connect( self.SelectAll, SIGNAL( "stateChanged( int )" ), self.SelectAllInListL )
        QObject.connect( self.SelectAll_2, SIGNAL( "stateChanged( int )" ), self.SelectAllInListR )

        # Button box 
        QObject.connect( self.btn_About, SIGNAL( "clicked()" ), self.showAbout )
        self.AcceptButton = self.bt_Accept.button( QDialogButtonBox.Ok )
        self.closeButton = self.bt_Accept.button( QDialogButtonBox.Cancel )
        
        # Manage current Raster-Layers
        self.Startup()
    
    # Shows the help for a single function
    def showFunctionHelp( self, name ):
        text = lcs.returnHelp(name,self.SingleMetricHelp)

    # Sets the nodata value inside the field
    def setNoData( self, rasterName ):
        if rasterName != -1:
            raster = func.getRasterLayerByName( self.cbRaster.currentText() )
            rasterPath = raster.source()
            raster = gdal.Open(str(rasterPath))
            band = raster.GetRasterBand(1)
            nodata = band.GetNoDataValue()
            self.NoDataVal.setEnabled( True )
            try:
                self.NoDataVal.setValidator(QDoubleValidator(-999999,999999,5))
            except TypeError: # Trying to catch up some strange Errors with this QT-Validator
                self.NoDataVal.setValidator(QDoubleValidator(-999999,999999,5,self))
                if isinstance(nodata,int)==False:
                    pass
                    #QMessageBox.warning( self, self.tr( "LecoS: Warning" ),self.tr( "Please format your rasters no-data value to integer (-99999 <-> 99999)" ) )
                
            self.NoDataVal.setText( QString(str(nodata)) ) 
            
    # Update Cellsize if a valid raster-file is selected
    def cellSizer( self, rasterName ):
        if rasterName != -1:
            pixelSize = func.getRasterLayerByName( rasterName ).rasterUnitsPerPixel()
            self.sp_cellsize.setEnabled( True )
            self.sp_cellsize.setValue( pixelSize )        
    
    # Save radio button
    def savetoggle( self ):
        if self.rbDirect.isChecked():
            self.SaveCsv.setEnabled( False )
            self.where2Save.setEnabled( False )
        elif self.rbSAVE.isChecked():
            self.SaveCsv.setEnabled( True )
            self.where2Save.setEnabled( True )
        
    # Where to save the csv
    def selectSaveFile( self ):   
        lastUsedDir = func.lastUsedDir()
        fileName = QFileDialog.getSaveFileName( self, self.tr( "Save data as" ),\
        lastUsedDir, "CSV files (*.csv *.CSV)" )
        if fileName.isEmpty():
            return
        func.setLastUsedDir( fileName )
        # ensure the user never ommited the extension from the file name
        if not fileName.toLower().endsWith( ".csv" ):
            fileName += ".csv"
        self.SaveCsv.setText( fileName )
        self.SaveCsv.setEnabled( True ) 
        
    # Manage Layout and startup parameters
    def Startup( self ):
        # Unable unused fields
        self.SaveCsv.setEnabled( False )
        self.where2Save.setEnabled( False )
        self.SingleMetricHelp.setEnabled( False )
        
        # Load in raster files
        self.cbRaster.addItems( func.getRasterLayersNames() )
        
        # Display functions
        self.SingleMetric.addItems( lcs.listStatistics() )
        self.showFunctionHelp( self.SingleMetric.currentText() )
        self.list_left.addItems( lcs.listStatistics() )
        self.rlistCounter.setText( str(self.list_left.count()) )
        self.rlistCounter_2.setText( str(self.list_right.count()) )
        
        self.MetricTab.setCurrentIndex( 0 ) # Startup on single metric tab
        self.MetricTab.setTabEnabled(2,False) # Disable Custom Tab
        
    # Switch Items on the several Metric Tab to the Right
    def switchR( self ):
        items = self.list_left.selectedItems()
        for item in items:
            n = self.list_left.row(item)    # get the index/row of the item
            i = self.list_left.takeItem(n)  # pop
            self.list_right.addItem(i)      # add to right QListWidget
            self.list_left.removeItemWidget( item ) # remove from left QListWidget
            #update counter
        self.rlistCounter.setText( str(self.list_left.count()) )
        self.rlistCounter_2.setText( str(self.list_right.count()) )
                
    # Switch Items on the several Metric Tab to the Left
    def switchL( self ):
        items = self.list_right.selectedItems()
        for item in items:
            n = self.list_right.row(item)    # get the index/row of the item
            i = self.list_right.takeItem(n)  # pop
            self.list_left.addItem(i)      # add to left QListWidget
            self.list_right.removeItemWidget( item ) # remove from right QListWidget
            #update counter
        self.rlistCounter.setText( str(self.list_left.count()) )
        self.rlistCounter_2.setText( str(self.list_right.count()) )

    # Select all Items in left List
    def SelectAllInListL ( self, state ):
        if state == Qt.Checked:
            allitems = self.list_left.findItems("*", Qt.MatchWrap | Qt.MatchWildcard)
            for item in allitems:
                item.setSelected( True )
        elif state == Qt.Unchecked:
            self.list_left.clearSelection()
                    
    # Select all Items in right List
    def SelectAllInListR ( self, state ):
        if state == Qt.Checked:
            allitems = self.list_right.findItems("*", Qt.MatchWrap | Qt.MatchWildcard)
            for item in allitems:
                item.setSelected( True )        
        elif state == Qt.Unchecked:
            self.list_right.clearSelection()
    
    # Show About Dialog
    def showAbout( self ):
        func.AboutDlg()
    
    # Accept current selection and metric
    def accept( self ):
        # check minimal input parameters
        if self.cbRaster.currentIndex() == -1:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please load a classified landcover layer into QGis first" ) )
            return
        if self.rbSAVE.isChecked():
            if self.SaveCsv.text().isEmpty():
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "Please select where to save the results" ) )
                return
            if path.exists(self.SaveCsv.text()):
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "File already exisits. Please select another path or delete file" ) )
                return
        if self.sp_cellsize.value() == 0:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please entry a correct cellsize (greater zero)" ) )
            return
        
        # Values and Preset
        self.progressBar.setRange( 0, 4)
        self.progressBar.setValue( 0 ) # pb start
        
        raster = func.getRasterLayerByName( self.cbRaster.currentText() )
        rasterPath = raster.source()
        dataPath = self.SaveCsv.text()
        cellsize = self.sp_cellsize.value()
        nodata = float(self.NoDataVal.text())
        
        ## Calculate Single Metric
        what = self.MetricTab.currentIndex()
        # Single
        if(what == 0):
            if len(self.SingleMetric.currentText()) >= 5:
                classes, array = lcs.f_landcover(rasterPath,nodata)
                self.progressBar.setValue( self.progressBar.value() + 1 )
                # Looping through all classes
                res = []
                res_tit = ["Class"]
                cl_analys = lcs.LandCoverAnalysis(array,cellsize,classes)
                for cl in classes:
                    cl_array = numpy.copy(array) # new working array
                    cl_array[cl_array!=cl] = 0
                    cl_analys.f_ccl(cl_array) # CC-labeling
                    name, result = cl_analys.execSingleMetric(self.SingleMetric.currentText(),cl) # Returns values for all 
                    
                    self.progressBar.setValue( self.progressBar.value() + 1 )
                    # Append Values to result class and table name array
                    r = [cl, result]
                    if len(res) == 0:
                        res.append(r)
                    else:
                        no_class_in_array = True
                        for id, item in enumerate(res):
                            if item[0] == cl:
                                res[id].append(result)
                                no_class_in_array = False
                        if no_class_in_array:
                            res.append(r)
                            
                    if res_tit.count(name) == 0:
                        res_tit.append(name)
            else:
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please select a valid single metric" ) )

        ## Several Metrics
        elif(what == 1):
            metrics = []
            allitems = self.list_right.findItems("*", Qt.MatchWrap | Qt.MatchWildcard)
            if len(allitems)==0:
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ), self.tr( "Please select at least one item from the left list" ) )
                return
            for item in allitems:
                metrics.append(unicode(item.text()))
            classes, array = lcs.f_landcover(rasterPath,nodata)
            self.progressBar.setValue( self.progressBar.value() + 1 )
            # Looping through all classes
            res = []
            res_tit = ["Class"]
            cl_analys = lcs.LandCoverAnalysis(array,cellsize,classes)
            for m in metrics:
                for cl in classes:
                    cl_array = numpy.copy(array) # new working array
                    cl_array[cl_array!=cl] = 0
                    cl_analys.f_ccl(cl_array) # CC-Labeling
                    name, result = cl_analys.execSingleMetric(m,cl)
                        
                    self.progressBar.setValue( self.progressBar.value() + 1 )
                    # Append Values to result class and table name array
                    r = [cl, result]
                    if len(res) == 0:
                        res.append(r)
                    else:
                        no_class_in_array = True
                        for id, item in enumerate(res):
                            if item[0] == cl:
                                res[id].append(result)
                                no_class_in_array = False
                        if no_class_in_array:
                            res.append(r)
                            
                    if res_tit.count(name) == 0:
                        res_tit.append(name)
        self.progressBar.setValue( self.progressBar.value() + 1 )
        # Write results
        if self.rbSAVE.isChecked():
            func.saveToCSV(res,res_tit,dataPath)
        else:
            func.ShowResultTableDialog(res_tit, res)
        self.progressBar.setValue( self.progressBar.value() + 1 )
        self.close()

# Gui for displaying certain diversity indices
class DivDialog(QDialog, Ui_DivDialog):
    def __init__(self, iface):
        # Initialize the Dialog
        QDialog.__init__( self )
        self.setupUi(self)
        self.iface = iface
        
        # Add Diversity Indices
        self.div = ["Shannon Index","Simpson Index", "Eveness"]
        self.cbDivselect.addItems( self.div )
        
        # Configure Connectors
        self.AcceptButton = self.btn_ok.button( QDialogButtonBox.Ok )
        self.closeButton = self.btn_ok.button( QDialogButtonBox.Cancel )
        QObject.connect( self.AcceptButton, SIGNAL( "clicked()" ), self.go )
        
        self.startup()
    
    # Configure GUI
    def startup(self):
        # Load in raster files
        self.cbRaster.addItems( func.getRasterLayersNames() )
    
    # Calculates Diversity indices    
    def go(self):
        if self.cbRaster.currentIndex() == -1:
            QMessageBox.warning( self, self.tr( "DivDialog: Warning" ),
                           self.tr( "Please load and select a classified raster first" ) )
            return
        raster = func.getRasterLayerByName( self.cbRaster.currentText() )
        rasterPath = raster.source()
        seldiv = unicode( self.cbDivselect.currentText() )
        
        div_cl = ldiv.LandscapeDiversity(rasterPath)
        cl = div_cl.classes # Get all the Landcover classes
        if len(cl) == 1:
            QMessageBox.warning(QDialog(),"LecoS: Warning","This tool needs at least two landcover classes to calculate landscape diversity!")
            return
        if seldiv == "Shannon Index":
            r = div_cl.f_returnDiversity("shannon")
        elif seldiv == "Simpson Index":
            r = div_cl.f_returnDiversity("simpson")
        elif seldiv == "Eveness":
            r = div_cl.f_returnDiversity("eveness")
        
        self.output(r,seldiv)
    
    # Creates a small output dialog
    def output(self, result, index):
        dlg = QDialog()
        dlg.setWindowTitle( QApplication.translate( "Diversity Results", "Results", "Window title" ) )
        lines = QVBoxLayout( dlg )
        lab = QLabel( QApplication.translate( "Diversity Results", "<b>Results for "+index+":</b>" ) )
        lines.addWidget( lab )
        res = QLineEdit()
        res.setText(str(result))
        lines.addWidget( res )

        btnClose = QPushButton( QApplication.translate( "Diversity Results", "Close" ) )
        lines.addWidget( btnClose )
        QObject.connect( btnClose, SIGNAL( "clicked()" ), dlg, SLOT( "close()" ) )

        dlg.exec_()

# Gui for batch computing Landcover for vector features
class BatchDialog(QDialog, Ui_BatchDialog):
    def __init__(self, iface):

        # Initialize the Dialog
        QDialog.__init__( self )
        self.setupUi(self)
        self.iface = iface
        
        # Connects
        QObject.connect( self.ch_selectAll, SIGNAL( "stateChanged( int )" ), self.SelectAllInList )
        QObject.connect( self.locateDir, SIGNAL( "clicked()" ), self.selectSaveFile )
        QObject.connect( self.cb_Raster, SIGNAL( "stateChanged()" ), self.EnableStuff )
        QObject.connect( self.cb_Raster, SIGNAL( "stateChanged()" ), self.loadIDFields )

        # Enable Stuff
        QObject.connect( self.cb_Raster, SIGNAL( "currentIndexChanged( QString )" ), self.EnableStuff) # Landscape
        QObject.connect( self.cb_Vector, SIGNAL( "currentIndexChanged( QString )" ), self.EnableStuff) # Poly overlay
        QObject.connect( self.rb_Landscape, SIGNAL( "clicked()" ), self.EnableStuff) # Change available stuff
        QObject.connect( self.rb_Class, SIGNAL( "clicked()" ), self.EnableStuff) # Change available stuff
        QObject.connect( self.ch_saveResult, SIGNAL( "stateChanged( int )" ), self.EnableStuff) # Change output

        # Button box 
        self.AcceptButton = self.startButtons.button( QDialogButtonBox.Ok )
        QObject.connect( self.AcceptButton, SIGNAL( "clicked()" ), self.go )
        self.closeButton = self.startButtons.button( QDialogButtonBox.Cancel )
        QObject.connect( self.btn_About, SIGNAL( "clicked()" ), self.showAbout )

        #Startup
        self.startup()
            
    # Startup function
    def startup(self):
         # Load in landscape and vector files
        vec = func.getVectorLayersNames()
        ras = func.getRasterLayersNames()
        #FIXME -> ALLOW VECTOR LANDSCAPE LAYERS
        self.ch_addToTable.setEnabled( False )
        self.cb_Raster.addItems( ras )
        self.cb_Vector.addItems( [""] + vec )        
        
        self.ListMetrics()
        
        # Unable most features
        self.EnableStuff("default")
                
    # Load Class metrics
    def ListMetrics(self):
        stats = lcs.listStatistics()
        self.Cl_Metrics.addItems(stats)

    # Enable all necessary functions
    def EnableStuff(self,what=""):
        if what != "default":
            ras = func.getLayerByName( what )
            vec = func.getVectorLayerByName( what )
            land = self.cb_Raster.currentText() == what
            if self.rb_Class.isChecked(): # Check if Landscape or Class structures should be computed
                self.cb_LClass.setEnabled( True )
                # Disable General methods
                self.ch_LC1.setEnabled( False ),self.ch_LC2.setEnabled( False ),self.ch_LC3.setEnabled( False ),self.ch_LC4.setEnabled( False ),self.ch_LC5.setEnabled( False ),self.ch_LC6.setEnabled( False ),self.ch_LC7.setEnabled( False ),self.ch_LC8.setEnabled( False )
                # Disable Diversity methods
                self.ch_div1.setEnabled( False ),self.ch_div2.setEnabled( False ),self.ch_div3.setEnabled( False )            
                # Set unchecked
                self.ch_LC1.setChecked( False ),self.ch_LC2.setChecked( False ),self.ch_LC3.setChecked( False ),self.ch_LC4.setChecked( False ),self.ch_LC5.setChecked( False ),self.ch_LC6.setChecked( False ),self.ch_LC7.setChecked( False ),self.ch_LC8.setChecked( False )
                # Disable Diversity methods
                self.ch_div1.setChecked( False ),self.ch_div2.setChecked( False ),self.ch_div3.setChecked( False )
                # Enable Class metrics
                self.Cl_Metrics.setEnabled( True )
                self.ch_selectAll.setEnabled( True )
            elif self.rb_Landscape.isChecked():
                # Enable General methods
                self.ch_LC1.setEnabled( True ),self.ch_LC2.setEnabled( True ),self.ch_LC3.setEnabled( True ),self.ch_LC4.setEnabled( True ),self.ch_LC5.setEnabled( True ),self.ch_LC6.setEnabled( True ),self.ch_LC7.setEnabled( True ),self.ch_LC8.setEnabled( True )
                # Disable Diversity methods
                self.ch_div1.setEnabled( True ),self.ch_div2.setEnabled( True ),self.ch_div3.setEnabled( True )
                self.cb_LClass.setEnabled( False )
                # Enable Class metrics
                self.Cl_Metrics.setEnabled( False )
                self.ch_selectAll.setEnabled( False )
                # Clear classes
                self.cb_LClass.clear()
            
            # TODO: Vector grid overlay
            # Disable vector grid overlay field
            if type(ras) == QgsVectorLayer and land:
                self.cb_Vector.setEnabled( False )
                self.cb_SelD.setEnabled( True )
                self.loadIDFields()
            elif (type(ras) == QgsRasterLayer) and land:
                self.cb_Vector.setEnabled( True )
                self.cb_SelD.setEnabled( False )        
                
            # Enable and load classes
            curLay = func.getLayerByName( self.cb_Raster.currentText() )
            if type(ras) == QgsRasterLayer or type(curLay) == QgsRasterLayer:
                if self.rb_Class.isChecked():
                    self.cb_LClass.setEnabled( True )
                    self.loadClasses("ras") # load classes
            elif type(ras) == QgsVectorLayer or type(curLay) == QgsVectorLayer:
                if self.rb_Class.isChecked():
                    self.cb_LClass.setEnabled( True )
                    self.loadClasses("vec") # load classes
                    self.loadIDFields() # load ID fields
            # Check Status of output    
            if self.ch_saveResult.isChecked():
                self.where2Save.setEnabled( True )
                self.locateDir.setEnabled( True )
            else:
                self.where2Save.setEnabled( False )
                self.locateDir.setEnabled( False )        
        
        else:
            self.Cl_Metrics.setEnabled( False )
            self.ch_selectAll.setEnabled( False )
            self.cb_LClass.setEnabled( False )
            self.cb_SelD.setEnabled( False )
                
        
    # Load ID fields
    def loadIDFields(self):
        vectorName = func.getLayerByName( self.cb_Raster.currentText() )
        test = func.getVectorLayerByName( self.cb_Vector.currentText() )
        if test != "" and type(vectorName) == QgsVectorLayer:
            self.cb_SelD.setEnabled( True )
            self.cb_SelD.clear()
            fields = func.getFieldList( vectorName )
            if QGis.QGIS_VERSION_INT >= 10900:
                for field in fields:
                    if field.type() in [ QVariant.Int, QVariant.String ]:
                        self.cb_SelD.addItem( field.name() )
            else:
                prov = vectorName.dataProvider()
                for k,  field in prov.items():
                    if field.type() in [ QVariant.Int, QVariant.String ]:
                        self.cb_SelD.addItem( field.name() )
        else:
            self.cb_SelD.clear()
            self.cb_SelD.setEnabled( False )

    
    # Get landscape classes
    def loadClasses(self,typ):
        if typ == "ras":
            rasterName = func.getRasterLayerByName( self.cb_Raster.currentText() )
            if rasterName != "":
                rasterPath = rasterName.source()
                raster = gdal.Open(str(rasterPath))
                band = raster.GetRasterBand(1)
                nodata = band.GetNoDataValue()
                array = band.ReadAsArray()
                self.classes = sorted(numpy.unique(array)) # get array of classes
                try:
                    self.classes.remove(nodata) # Remove nodata-values from classes array
                except ValueError:
                    try:
                        self.classes.remove(0)
                    except ValueError:
                        pass
                classes = [str(numeric_cl) for numeric_cl in self.classes]
                self.cb_LClass.clear()
                self.cb_LClass.addItems( classes )
            else:
                self.cb_LClass.clear()
                self.cb_LClass.addItems( [""] )
        elif typ == "vec":
            vectorName = func.getVectorLayerByName( self.cb_Raster.currentText() )
            if vectorName != "":
                self.cb_LClass.clear()
                fields = func.getFieldList( vectorName )
                if QGis.QGIS_VERSION_INT >= 10900:
                    for field in fields:
                        if field.type() in [ QVariant.Int, QVariant.Double ]:
                            self.cb_LClass.addItem( field.name() )
                else:
                    prov = vectorName.dataProvider()
                    for k,  field in prov.items():
                        if field.type() in [ QVariant.Int, QVariant.Double ]:
                            self.cb_LClass.addItem( field.name() )
            else:
                self.cb_LClass.clear()
            
    # Select all Class metrics
    def SelectAllInList(self, state):
        if state == Qt.Checked:
            allitems = self.Cl_Metrics.findItems("*", Qt.MatchWrap | Qt.MatchWildcard)
            for item in allitems:
                item.setSelected( True )
            self.ch_selectAll.setText(QString("Unselect all"))
        elif state == Qt.Unchecked:
            self.Cl_Metrics.clearSelection()
            self.ch_selectAll.setText(QString("Select all"))

    # Where to save the csv
    def selectSaveFile( self ):   
        lastUsedDir = func.lastUsedDir()
        fileName = QFileDialog.getSaveFileName( self, self.tr( "Save data as" ),\
        lastUsedDir, "CSV files (*.csv *.CSV)" )
        if fileName.isEmpty():
            return
        func.setLastUsedDir( fileName )
        # ensure the user never ommited the extension from the file name
        if not fileName.toLower().endsWith( ".csv" ):
            fileName += ".csv"
        self.where2Save.setText( fileName )
        self.where2Save.setEnabled( True ) 
    
#     # Show feature count
#     def featureCount(self, vectorName):
#         if vectorName != "":
#             # Load vector data
#             self.vector = func.getVectorLayerByName( self.cb_Vector.currentText() )
#             self.vectorPath = self.vector.source()
#             nf = self.vector.featureCount()
#             self.lab_feat.setText("Features ("+str(nf)+")")
#         else:
#             self.lab_feat.setText("Features (0)")

    # Show About Dialog
    def showAbout( self ):
        func.AboutDlg()
    
    # Returns the selected metrics
    def getSelMetric(self):
        res = []
        m = self.Cl_Metrics.selectedItems()
        for item in m:
            res.append(str( item.text() ))
        # Other Metrics ?
        if self.ch_LC1.isChecked():
            res.append("LC_Mean")
        if self.ch_LC2.isChecked():
            res.append("LC_Sum")
        if self.ch_LC3.isChecked():
            res.append("LC_Min")
        if self.ch_LC4.isChecked():
            res.append("LC_Max")
        if self.ch_LC5.isChecked():
            res.append("LC_SD")
        if self.ch_LC6.isChecked():
            res.append("LC_LQua")
        if self.ch_LC7.isChecked():
            res.append("LC_Med")
        if self.ch_LC8.isChecked():
            res.append("LC_UQua")
        if self.ch_div1.isChecked():
            res.append("DIV_SH")
        if self.ch_div2.isChecked():
            res.append("DIV_EV")
        if self.ch_div3.isChecked():
            res.append("DIV_SI")
        return res
        
    # Runs the routine
    def go(self):
        # Error catching and variable returning
        if self.cb_Raster.currentIndex() == -1:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please load and select a classified landscape layer first" ) )
            return
        self.landscape = func.getLayerByName( self.cb_Raster.currentText() )
        self.landPath = self.landscape.source()
        
        if type(self.landscape) == QgsVectorLayer: # Check if landscape vector layer is a polygon
            if self.landscape.geometryType() != QGis.Polygon:
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                    self.tr( "The landscape layer have to be of geometry type: Polygons" ) )
                return
        # Default values
        self.vector = None
        self.vectorPath = None 
        # Load in overlaying Grid and raise error if not a polygon or no landscape raster
        if self.cb_Vector.currentText() != "":
            self.vector = func.getVectorLayerByName( self.cb_Vector.currentText() )
            self.vectorPath = self.vector.source()
            if type(self.landscape) == QgsVectorLayer:
                if self.landscape.geometryType() != QGis.Polygon:
                    QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                        self.tr( "This tool need the vector grid to have the geometry type: Polygon" ) )
                    return
            # Check if both layers have the same projection
            cr1 = self.vector.crs()
            cr2 = self.landscape.crs()
            if cr1!=cr2:
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "Please make sure that both layers have the same spatial projection" ) )
                return
            # Check if a layer has been chosen twice
            if self.landPath == self.vectorPath:
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "It is not possible to overlay layers from the same source. Specify a Grouping ID" ) )
                return
        elif type(self.landscape) == QgsRasterLayer:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please load and select an overlaying vector grid" ) )
            return
        elif type(self.landscape) == QgsVectorLayer:
            if self.cb_SelD.currentText() == "":
                QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "You didn't choose an overlaying vector grid and therefore need to specify a grouping ID" ) )
                return
        
        if self.rb_Landscape.isChecked(): # Landscape or Class metrics
            self.cl_metric = False
            self.classIND = None
        else:
            self.cl_metric = True
            self.classIND = self.cb_LClass.currentText()
        if (type(self.landscape) == QgsVectorLayer) and (self.vector == None): # Use a ID?
            self.LandID = self.cb_SelD.currentText()
        else:
            self.LandID = None
    
        self.metrics = self.getSelMetric() # Get list of all Metrics
        if len(self.metrics) == 0:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please select at least one metric to compute!" ) )
            return
        if self.ch_saveResult.isChecked():
            self.FileSave = True
            self.FileSavePath = self.where2Save.text()
            # If no output has been defined -> create a temporary file
            if self.FileSavePath == "Select a destination or leave blank to create a temporary file":
                self.FileSavePath = tmpdir+os.path.sep+"temp_"+str(self.cb_Raster.currentText())+"_results.csv"              
        else:
            self.FileSave = False
        if self.ch_addToTable.isChecked(): # Add to Attribute Table
            self.Add2Table = True
        else:
            self.Add2Table = False
        
        ## Processing Start 
        ### Vector part
        if type(self.landscape) == QgsVectorLayer:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "Landscape vectors not yet supported" ) )
            return
            if self.cb_Vector.currentText() == "":
                # Use inherent vector grouping id
                #self.LandID
                pass
            else:
                # Use overlaying vector grid  -> need to cut landscape vectors first
                pass
        ### Raster part
        if type(self.landscape) == QgsRasterLayer:
            bat = pov.BatchConverter(self.landPath,self.vectorPath)
            # Landscape or classified
            if self.cl_metric:
                if(self.classIND == ""):
                    QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                            self.tr( "Please select a valid landcover class!" ) )
                    return
                else:
                    cl = int(self.classes[self.cb_LClass.currentIndex()]) # Get selected class
                cellsize = self.landscape.rasterUnitsPerPixel()
                #Calculate selected statistics for classified raster
                results = []
                for cmd in self.metrics:
                    results.append(bat.go(cmd,cl,cellsize))
                self.Output(results)   
            else:
                cellsize = self.landscape.rasterUnitsPerPixel()
                #Calculate statistics for unclassified raster
                results = []
                for el in ('LC','DIV'):
                    met = filter(lambda x:el in x,self.metrics)              
                    for cmd in met:
                        results.append(bat.go(cmd,None,cellsize))
                self.Output(results)

    # Function for Output generation
    def Output(self,results):
        if self.Add2Table:
            # Add to Attribute table
            if (type(self.landscape) == QgsVectorLayer) and self.cb_Vector.currentText() == "":
                pass
            else:
                #TODO: Doesn't work yet. QGIS crashes
                # Get number of polygon features
                add = func.addAttributesToLayer2(self.vector,results)

#                 feat = range(0,len(results[0]))
#                 for m in range(0,len(self.metrics)):
#                     r = []
#                     cmd = str(results[m][0][1])
#                     for feature in feat:
#                         values = [results[m][feature][0],results[m][feature][2]]
#                         r.append(values)
#                     add = func.addAttributesToLayer(self.vector,cmd,r,type="qgis")
#                     if add == False:
#                         QMessageBox.information( self, self.tr( "LecoS: Info" ),self.tr( "Calculated values couldn't be saved to the vector layers attribute-table" ) )
#                         return
        
        if self.FileSave:
            # Write to file
            title = ["PolygonFeatureID"]
            for x in results:
                title.append( str(x[0][1]) ) 
            f = open(self.FileSavePath, "wb" )
            writer = csv.writer(f,delimiter=';',quotechar="'",quoting=csv.QUOTE_ALL)
            writer.writerow(title)
            # Get number of polygon features
            feat = range(0,len(results[0]))
            for feature in feat: # Write feature to new line
                r = [feature]
                for item in results:
                    r.append(item[feature][2])
                writer.writerow(r)
            f.close()
                        
        if (self.Add2Table == False) and (self.FileSave == False):
            # Direct Output?
            title = ["PolygonFeatureID"]
            for x in results:
                title.append( str(x[0][1]) )
            func.ShowResultTableDialog2(title,results)
        
        # Add result to QGIS
        if self.ch_AddQGIS.isChecked() and self.ch_saveResult.isChecked():
            func.tableInQgis( self.FileSavePath )
        if self.ch_addToTable.isChecked():
            pass # Update ToC
        
# Gui for generating a LandMod out of a rasterized map
class LandMod(QDialog, Ui_LandMod):
    def __init__(self, iface):
        # Initialize the Dialog
        QDialog.__init__( self )
        self.setupUi(self)
        self.iface = iface
        
        # Configure Connectors
        self.AcceptButton = self.buttonBox.button( QDialogButtonBox.Ok )
        self.closeButton = self.buttonBox.button( QDialogButtonBox.Cancel )
        QObject.connect( self.AcceptButton, SIGNAL( "clicked()" ), self.go )
        QObject.connect( self.btn_Save, SIGNAL( "clicked()" ), self.selectSaveFile )# Save File
        QObject.connect( self.cb_Raster, SIGNAL( "currentIndexChanged( QString )" ), self.cellSizer)
        QObject.connect( self.cb_Raster, SIGNAL( "currentIndexChanged( QString )" ), self.loadClasses)
        
        self.startup()
    
    # Configure GUI
    def startup(self):
        # Load in raster files
        self.cb_Raster.addItems( func.getRasterLayersNames() )
        if(self.cb_Raster.count()!=0):
            self.loadClasses()
            self.cellSizer(self.cb_Raster.currentText())
    
    # Set the cellsizer value 
    def cellSizer(self,rasterName):
        ras = func.getRasterLayerByName( rasterName )
        pixelSize = ras.rasterUnitsPerPixel()
        self.CellsizeLine.setEnabled( True )
        self.CellsizeLine.setText( QString(str(pixelSize)) ) 
    
    # Where to save the raster output
    def selectSaveFile( self ):   
        lastUsedDir = func.lastUsedDir()
        fileName = QFileDialog.getSaveFileName( self, self.tr( "Save raster as" ),\
        lastUsedDir, "GeoTIFF files (*.tif *.TIF)" )
        if fileName.isEmpty():
            return
        func.setLastUsedDir( fileName )
        # ensure the user never ommited the extension from the file name
        if not fileName.toLower().endsWith( ".tif" ):
            fileName += ".tif"
        self.where2Save.setText( fileName )
        self.where2Save.setEnabled( True ) 
    
    # Get Raster classes
    def loadClasses(self,rasterName=None):
        rasterName = func.getRasterLayerByName( self.cb_Raster.currentText() )
        if rasterName != "":
            rasterPath = rasterName.source()
            raster = gdal.Open(str(rasterPath))
            band = raster.GetRasterBand(1)
            nodata = band.GetNoDataValue()
            array = band.ReadAsArray()
            self.classes = sorted(numpy.unique(array)) # get array of classes
            try:
                self.classes.remove(nodata) # Remove nodata-values from classes array
            except ValueError:
                try:
                    self.classes.remove(0)
                except ValueError:
                    pass
            classes = [str(numeric_cl) for numeric_cl in self.classes]
            self.cb_SelClass.clear()
            self.cb_SelClass.addItems( classes )
        else:
            self.cb_SelClass.clear()
            self.cb_SelClass.addItems( [""] )
    
    # Calculate new raster
    def go(self):
        if self.cb_Raster.currentIndex() == -1:
            QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Please load and select a classified raster first" ) )
            return
        savePath = str(self.where2Save.text())
        if savePath == "": # If no output has been defined -> create a temporary file
            savePath = tmpdir+os.path.sep+"temporary_raster.tif"    
        
        # Get basic input
        raster = func.getRasterLayerByName( self.cb_Raster.currentText() )
        rasterPath = raster.source()
        cl = int(self.classes[self.cb_SelClass.currentIndex()]) # Get selected class
        
        what = self.box_RasCalc.currentIndex()
        
        mod = lmod.LandscapeMod(rasterPath,cl)
        # Create class object
        if what == 0: # Patch Edges
            size = self.sp_EdgeMult.value()
            results = mod.extractEdges(size)
        elif what == 1: # Isolate smallest or greatest patch
            if self.rb_MaxMin1.isChecked():
                which = "min"
            else:
                which = "max"
            results = mod.getPatch(which)
        elif what == 2: # Increase/decrease landscape patches
            which = self.cb_IncDec.currentIndex()
            amount = self.sp_IncDecAm.value()
            results = mod.InDecPatch(which,amount)
        elif what == 3: # Fill holes inside landscape patches
            results = mod.closeHoles()
        elif what == 4: # Clean raster
            iter = self.sp_CleanIter.value()
            results = mod.cleanRaster(iter)
        
        # Save the results
        func.exportRaster(results,rasterPath,savePath)
        # Add to QGis if specified
        if self.addToToc.isChecked():
            func.rasterInQgis( savePath )

        