# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LecoS
                                 A QGIS plugin
 Contains analytical functions for landscape analysis
                             -------------------
        begin                : 2012-09-06
        copyright            : (C) 2012 by Martin Jung
        email                : martinjung@zoho.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt4.QtCore import *
from PyQt4.QtGui import *

from qgis.core import *
from qgis.gui import *
from qgis.analysis import *

import gdal, numpy, sys, scipy, string, math, ogr, os
import subprocess
from scipy import ndimage
gdal.AllRegister() # register all gdal drivers

gdal.UseExceptions()
ogr.UseExceptions()

helpdir = QFileInfo(QgsApplication.qgisUserDbFilePath()).path() + "/python/plugins/LecoS/metric_info/"


# List all available landscape functions
# All defined metrics must possess an info file in the metric_info folder
def listStatistics():
    functionList = []
    
    functionList.append(unicode("Land cover")) # Calculate Area
    functionList.append(unicode("Landscape Proportion")) # Landscape Proportion
    functionList.append(unicode("Number of Patches")) # Return Number of Patches
    functionList.append(unicode("Patch density")) # Return Patch density
    functionList.append(unicode("Greatest patch area")) # Return Greatest Patch area
    functionList.append(unicode("Smallest patch area")) # Return Smallest Patch area
    functionList.append(unicode("Mean patch area")) # Return Mean Patch area
    functionList.append(unicode("Overall Core area")) # Return Core area

    return functionList

# Returns definition and reference for given function
def returnHelp(name, textfield):
    s = string.replace(name," ","_")
    h = (helpdir+s+".html")
    #textfield.setHtml(open(h).read())
    f = QFile(h)
    f.open(QFile.ReadOnly|QFile.Text)
    istream = QTextStream(f)
    textfield.setHtml(istream.readAll())
    f.close()
    
# Executes the Metric functions
def execSingleMetric(name,unlabeled_array,array,numpatches,cellsize,cl):
    if(name == unicode("Land cover")):
        return unicode(name), f_returnArea(array,cellsize)
    if(name == unicode("Landscape Proportion")):
        return unicode(name), f_returnProportion(unlabeled_array,cl)
    elif(name == unicode("Number of Patches")):
        return unicode(name), numpatches
    elif(name == unicode("Patch density")):
        return unicode(name), f_patchDensity(array,cellsize,numpatches)
    elif(name == unicode("Greatest patch area")):
        return unicode(name), f_returnPatchArea(unlabeled_array,array,numpatches,cellsize,"max")
    elif(name == unicode("Smallest patch area")):
        return unicode(name), f_returnPatchArea(unlabeled_array,array,numpatches,cellsize,"min")
    elif(name == unicode("Mean patch area")):
        return unicode(name), f_returnPatchArea(unlabeled_array,array,numpatches,cellsize,"mean")
    elif(name == unicode("Overall Core area")):
        return unicode(name), f_getCoreArea(array,cellsize)
    else:
        QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Unfortunately the Metric has yet to be coded.") )
    

# Prepare raster for component labeling
def f_landcover(raster):
    raster = gdal.Open(str(raster))
    if(raster.RasterCount==1):
        band = raster.GetRasterBand(1)
        nodata = band.GetNoDataValue()
        if nodata == None:
            nodata = 0
        try:
            array =  band.ReadAsArray() 
        except ValueError:
            QMessageBox.warning(None,"LecoS: Warning","Raster file is to big for processing. Please crop the file and try again.")
            return
        classes = sorted(numpy.unique(array)) # get classes
        try: #omit nodata value
            classes.remove(nodata)
        except ValueError:
            pass
        return classes, array
    else:
        # TODO: Multiband Support?
        QMessageBox.warning( self, self.tr( "LecoS: Warning" ),
                           self.tr( "Multiband Rasters not implemented yet") )

# Returns the nodata value. Assumes an raster with one band
def f_returnNoDataValue(raster):
    raster = gdal.Open(str(raster))
    band = raster.GetRasterBand(1)
    nodata = band.GetNoDataValue()
    return nodata    

# Connected component labeling function
def f_ccl(cl_array):
    # Binary structure
    struct = scipy.ndimage.generate_binary_structure(2,2)
    labeled_array, numpatches = ndimage.label(cl_array,struct)
    return labeled_array, numpatches


## Landscape Metrics
# Return the total area for the given class
def f_returnArea(labeled_array,cellsize):
    #sizes = scipy.ndimage.sum(array, labeled_array, range(numpatches + 1)).astype(labeled_array.dtype)
    area = numpy.count_nonzero(labeled_array) *(cellsize*cellsize)
    return area

# Return Patchdensity
def f_patchDensity(array,cellsize,numpatches):
    area = f_returnArea(array,cellsize)
    return (float(numpatches) / float(area))

# Return array with a specific labeled patch
def f_returnPatch(array,labeled_array,patch):
    # Make an array of zeros the same shape as `a`.
    feature = numpy.zeros_like(array, dtype=int)
    feature[labeled_array == patch] = 1
    return feature

# Iter through all identified patches and count adjacent cells
def f_IterPatches(array,labeled_array,numpatches,s,overlap=False):
    res = []
    if overlap:
        for i in range(1,numpatches + 1):
            feature = f_returnPatch(array,labeled_array,i)
            ov = (feature * ndimage.convolve((feature == 0).astype(int), s)).sum()
            res.append(ov)
    else:
        for i in range(1, numpatches + 1):
            feature = f_returnPatch(array,labeled_array,i)
            dil = ndimage.binary_dilation(feature,s).astype(feature.dtype)
            n = dil - feature
            res.append(numpy.count_nonzero(n))
    return sum(res)

# Returns the overall Core-Area
def f_getCoreArea(labeled_array,cellsize):
    s = ndimage.generate_binary_structure(2,2)
    newlab = ndimage.binary_erosion(labeled_array,s).astype(labeled_array.dtype)
    return ndimage.sum(newlab)*(cellsize*cellsize)

# Return greatest, smallest or mean patch area
def f_returnPatchArea(array,labeled_array,numpatches,cellsize,what):
    sizes = scipy.ndimage.sum(array, labeled_array, range(numpatches + 1))
    sizes = sizes[sizes!=0] # remove zeros
    if what=="max":
        return numpy.max(sizes)*(cellsize*cellsize)
    elif what=="min":
        return numpy.min(sizes)*(cellsize*cellsize)
    elif what=="mean":
        return numpy.mean(sizes)*(cellsize*cellsize)
  
# Returns the proportion of the labeled class in the landscape
def f_returnProportion(array,cl):
    lt = float(numpy.count_nonzero(array))
    lc = float(numpy.count_nonzero(array[array==cl]))
    d = lc /lt 
    return d

# Returns the total number of cells in the array
def f_returnTotalCellNumber(array):
    return int(numpy.count_nonzero(array))
    
# Returns a tuple with the position of the largest patch
def f_returnPosLargestPatch(labeled_array):
    return numpy.unravel_index(labeled_array.argmax(),labeled_array.shape)

## Diversity Indices
# Returns different diversity indices for the whole landscape with exception of no_data_values
def f_returnDiversity(array,classes,no_data_value,index):
    if(index=="shannon"):
        sh = []
        cl_array = numpy.copy(array) # create working array
        cl_array[cl_array==int(no_data_value)] = 0
        for cl in classes:
            cl_prop = f_returnProportion(cl_array,cl)
            sh.append(cl_prop * math.log(cl_prop))
        return sum(sh)*-1
    elif(index=="simpson"):
        si = []
        cl_array = numpy.copy(array) # create working array
        cl_array[cl_array==int(no_data_value)] = 0
        for cl in classes:
            cl_prop = f_returnProportion(cl_array,cl)
            si.append(math.pow(cl_prop,2))
        return 1-sum(si)
    elif(index=="eveness"):
        return f_returnDiversity(array,classes,no_data_value,"shannon") / math.log(len(classes))
    else:
        print "Diversity Index not available (yet)"
    