# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon
import os

class LayoutZoomPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # Initialize plugin directory
        self.actions = []
        self.menu = 'Layout Zoom Plugin'
        
        # Check if plugin was started the first time in current QGIS session
        self.first_start = None

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        
        # Create action for creating new layout
        icon_path_new = os.path.join(self.plugin_dir, 'icons', 'create_layout.svg')
        self.new_layout_action = QAction(
            QIcon(icon_path_new),
            "Create New Layout from Template",
            self.iface.mainWindow()
        )
        self.new_layout_action.triggered.connect(self.run_new_layout)
        self.new_layout_action.setEnabled(True)
        
        # Create action for working with existing layout
        icon_path_existing = os.path.join(self.plugin_dir, 'icons', 'use_layout.svg')
        self.existing_layout_action = QAction(
            QIcon(icon_path_existing),
            "Edit Existing Layout",
            self.iface.mainWindow()
        )
        self.existing_layout_action.triggered.connect(self.run_existing_layout)
        self.existing_layout_action.setEnabled(True)
        
        # Add to menu
        self.iface.addPluginToMenu(self.menu, self.new_layout_action)
        self.iface.addPluginToMenu(self.menu, self.existing_layout_action)
        
        # Add to toolbar
        self.iface.addToolBarIcon(self.new_layout_action)
        self.iface.addToolBarIcon(self.existing_layout_action)
        
        self.actions.append(self.new_layout_action)
        self.actions.append(self.existing_layout_action)
        
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

    def run_new_layout(self):
        """Run method for creating new layout from template"""
        try:
            from .new_layout_dialog import LayoutMapViewer as NewLayoutDialog
            self.dialog = NewLayoutDialog()
            self.dialog.exec_()
        except Exception as e:
            QMessageBox.critical(self.iface.mainWindow(), "Error", f"Error in creating new layout:\n{str(e)}")

    def run_existing_layout(self):
        """Run method for editing existing layout"""
        try:
            from .existing_layout_dialog import LayoutMapViewer as ExistingLayoutDialog
            self.dialog = ExistingLayoutDialog()
            self.dialog.exec_()
        except Exception as e:
            QMessageBox.critical(self.iface.mainWindow(), "Error", f"Error in editing existing layout:\n{str(e)}")