# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QListWidget, QListWidgetItem,
    QPushButton, QMessageBox, QTextEdit, QGroupBox, QComboBox, QWidget,
    QSizePolicy
)
from qgis.PyQt.QtCore import Qt, QPointF
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProject, QgsLayoutItemMap, QgsPointXY, QgsGeometry, 
    QgsLayoutPoint, QgsLayoutSize, QgsUnitTypes, QgsRectangle,
    QgsProperty, QgsLayoutObject
)
from qgis.gui import QgsMapToolEmitPoint
from qgis.utils import iface


class PointSelectionTool(QgsMapToolEmitPoint):
    def __init__(self, canvas, callback):
        QgsMapToolEmitPoint.__init__(self, canvas)
        self.callback = callback
        self.canvas = canvas

    def canvasPressEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        self.callback(point)
        
    def activate(self):
        super().activate()
        self.canvas.setCursor(Qt.CrossCursor)
        
    def deactivate(self):
        super().deactivate()
        self.canvas.unsetCursor()


class LayoutMapViewer(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("View Map Items in Project Layouts")
        
        # Optimal size settings
        self.setMinimumSize(650, 550)
        self.resize(700, 600)
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        
        self.current_layout = None
        self.map_items = []
        self._layouts_by_key = {}
        self.selected_map_item = None
        self.selected_scale = None
        self.map_tool = None
        self.original_map_tool = None
        
        self.setup_ui()
        self.load_layouts()

    def setup_ui(self):
        main_layout = QVBoxLayout()
        main_layout.setSpacing(8)
        main_layout.setContentsMargins(10, 10, 10, 10)
        
        main_layout.addWidget(self._create_layout_selection_group())
        main_layout.addWidget(self._create_layout_info_group())
        main_layout.addWidget(self._create_maps_list_group())
        main_layout.addWidget(self._create_details_group())
        main_layout.addWidget(self._create_scale_selection_group())
        main_layout.addWidget(self._create_buttons())
        
        self.setLayout(main_layout)

    def _create_layout_selection_group(self):
        group = QGroupBox("Select Layout")
        group.setMaximumHeight(80)
        layout = QVBoxLayout()
        layout.setContentsMargins(8, 8, 8, 8)
        
        selection_layout = QHBoxLayout()
        selection_layout.addWidget(QLabel("Layout:"))
        
        self.layout_combo = QComboBox()
        self.layout_combo.setMaximumHeight(30)
        self.layout_combo.currentIndexChanged.connect(self.on_layout_selected)
        selection_layout.addWidget(self.layout_combo, 1)
        
        self.refresh_btn = QPushButton("Refresh")
        self.refresh_btn.setMaximumHeight(30)
        self.refresh_btn.setMaximumWidth(100)
        self.refresh_btn.clicked.connect(self.load_layouts)
        selection_layout.addWidget(self.refresh_btn)
        
        layout.addLayout(selection_layout)
        group.setLayout(layout)
        return group

    def _create_layout_info_group(self):
        group = QGroupBox("Layout Information")
        group.setMaximumHeight(100)
        layout = QVBoxLayout()
        layout.setContentsMargins(8, 8, 8, 8)
        
        self.layout_info = QTextEdit()
        self.layout_info.setMaximumHeight(70)
        self.layout_info.setReadOnly(True)
        self.layout_info.setPlaceholderText("Layout information will be displayed after selection...")
        self.layout_info.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        
        layout.addWidget(self.layout_info)
        group.setLayout(layout)
        return group

    def _create_maps_list_group(self):
        group = QGroupBox("Map Items Found in Layout")
        group.setMaximumHeight(150)
        layout = QVBoxLayout()
        layout.setContentsMargins(8, 8, 8, 8)
        
        self.maps_list = QListWidget()
        self.maps_list.setMaximumHeight(100)
        self.maps_list.itemSelectionChanged.connect(self.on_map_selected)
        self.maps_list.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        layout.addWidget(self.maps_list)
        
        group.setLayout(layout)
        return group

    def _create_details_group(self):
        group = QGroupBox("Selected Map Item Details")
        group.setMaximumHeight(150)
        layout = QVBoxLayout()
        layout.setContentsMargins(8, 8, 8, 8)
        
        self.details_text = QTextEdit()
        self.details_text.setMaximumHeight(100)
        self.details_text.setReadOnly(True)
        self.details_text.setPlaceholderText("Select a map item from the list...")
        self.details_text.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        
        layout.addWidget(self.details_text)
        group.setLayout(layout)
        return group

    def _create_scale_selection_group(self):
        group = QGroupBox("Select Scale")
        group.setMaximumHeight(80)
        layout = QVBoxLayout()
        layout.setContentsMargins(8, 8, 8, 8)
        
        scale_layout = QHBoxLayout()
        scale_layout.addWidget(QLabel("Map Scale:"))
        
        self.scale_combo = QComboBox()
        self.scale_combo.setEditable(True)
        self.scale_combo.setMaximumHeight(30)
        self.scale_combo.setToolTip("You can select from the list or enter a custom scale (example: 1000)")
        
        common_scales = [
            "100", "200", "500", "1000", "2000", "5000", 
            "10000", "25000", "50000", "100000", "250000", 
            "500000", "1000000", "2000000", "5000000"
        ]
        
        for scale in common_scales:
            self.scale_combo.addItem(f"1:{scale}")
        
        self.scale_combo.setCurrentText("1:1000")
        
        scale_layout.addWidget(self.scale_combo, 1)
        
        self.apply_scale_btn = QPushButton("Apply Scale")
        self.apply_scale_btn.setMaximumHeight(30)
        self.apply_scale_btn.setMaximumWidth(100)
        self.apply_scale_btn.clicked.connect(self.apply_scale_to_map)
        self.apply_scale_btn.setEnabled(False)
        scale_layout.addWidget(self.apply_scale_btn)
        
        layout.addLayout(scale_layout)
        group.setLayout(layout)
        return group

    def _create_buttons(self):
        button_layout = QHBoxLayout()
        button_layout.setContentsMargins(0, 5, 0, 5)
        
        self.continue_btn = QPushButton("Continue")
        self.continue_btn.setMaximumHeight(35)
        self.continue_btn.setMinimumWidth(80)
        self.continue_btn.clicked.connect(self.on_continue_clicked)
        self.continue_btn.setEnabled(False)
        
        self.close_btn = QPushButton("Close")
        self.close_btn.setMaximumHeight(35)
        self.close_btn.setMinimumWidth(80)
        self.close_btn.clicked.connect(self.close)
        
        button_layout.addWidget(self.continue_btn)
        button_layout.addStretch()
        button_layout.addWidget(self.close_btn)
        
        widget = QWidget()
        widget.setMaximumHeight(50)
        widget.setLayout(button_layout)
        return widget

    def load_layouts(self):
        self.layout_combo.clear()
        self.maps_list.clear()
        self.details_text.clear()
        self.layout_info.clear()
        self._layouts_by_key.clear()
        self.map_items = []
        self.current_layout = None
        
        self.continue_btn.setEnabled(False)
        self.apply_scale_btn.setEnabled(False)
        
        layout_manager = QgsProject.instance().layoutManager()
        layouts = layout_manager.layouts()
        
        if not layouts:
            self.layout_combo.addItem("No layouts in project", None)
            self.layout_combo.setEnabled(False)
            self.layout_info.setText("❌ No layouts found in current project.")
            return
        
        for layout in layouts:
            layout_name = layout.name()
            layout_id = layout.id() if hasattr(layout, 'id') else str(id(layout))
            
            self._layouts_by_key[layout_id] = layout
            self.layout_combo.addItem(layout_name, layout_id)
        
        self.layout_combo.setEnabled(True)
        self.layout_info.setText(f"✅ Layouts found: {len(layouts)}")
        
        if self.layout_combo.count() > 0:
            self.layout_combo.setCurrentIndex(0)

    def on_layout_selected(self, index):
        if index < 0:
            return
            
        layout_id = self.layout_combo.currentData()
        if not layout_id:
            self.current_layout = None
            self.layout_info.setText("❌ Invalid layout selected.")
            self.maps_list.clear()
            self.details_text.clear()
            self.continue_btn.setEnabled(False)
            self.apply_scale_btn.setEnabled(False)
            return
        
        self.current_layout = self._layouts_by_key.get(layout_id)
        
        if not self.current_layout:
            self.layout_info.setText("❌ This layout was not found in the project.")
            self.maps_list.clear()
            self.details_text.clear()
            self.continue_btn.setEnabled(False)
            self.apply_scale_btn.setEnabled(False)
            return
            
        self.analyze_layout()

    def analyze_layout(self):
        if not self.current_layout:
            return
            
        try:
            self.map_items = []
            self.maps_list.clear()
            self.details_text.clear()
            
            layout_info = self._get_layout_info()
            self.layout_info.setText(layout_info)
            
            self._extract_map_items()
            self._display_results()
            
            has_maps = len(self.map_items) > 0
            self.continue_btn.setEnabled(has_maps)
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error analyzing layout:\n{str(e)}")

    def _get_layout_info(self):
        info_lines = []
        
        if not self.current_layout:
            return "No layout selected"
        
        info_lines.append(f"📄 Layout Name: {self.current_layout.name()}")
        
        try:
            page_collection = self.current_layout.pageCollection()
            if page_collection and page_collection.pageCount() > 0:
                page = page_collection.page(0)
                page_size = page.pageSize()
                info_lines.append(f"📏 Page Size: {page_size.width():.1f} × {page_size.height():.1f} mm")
        except:
            pass
        
        try:
            items_count = len(self.current_layout.items())
            info_lines.append(f"🔢 Total Items: {items_count}")
        except:
            pass
        
        return "\n".join(info_lines)

    def _extract_map_items(self):
        if not self.current_layout:
            return
            
        try:
            for item in self.current_layout.items():
                if isinstance(item, QgsLayoutItemMap):
                    map_info = self._parse_map_item(item)
                    if map_info:
                        self.map_items.append(map_info)
        except Exception as e:
            print(f"Error extracting map items: {e}")

    def _parse_map_item(self, map_item):
        try:
            map_info = {
                'item_object': map_item,
                'id': '',
                'name': '',
                'type': 'QgsLayoutItemMap',
                'position': {'x': 0, 'y': 0},
                'size': {'width': 0, 'height': 0},
                'crs': 'Unknown',
                'extent': 'Unknown',
                'scale': 'Unknown',
                'locked': False,
                'visible': True,
                'has_data_defined_scale': False
            }
            
            try:
                if hasattr(map_item, 'id'):
                    map_info['id'] = map_item.id()
                else:
                    map_info['id'] = str(id(map_item))
            except:
                map_info['id'] = str(id(map_item))
            
            try:
                map_info['name'] = map_item.displayName()
            except:
                try:
                    map_info['name'] = getattr(map_item, 'name', '') or getattr(map_item, 'id', '')
                except:
                    map_info['name'] = 'Unknown'
            
            try:
                rect = map_item.sceneBoundingRect()
                map_info['position']['x'] = rect.x()
                map_info['position']['y'] = rect.y()
                map_info['size']['width'] = rect.width()
                map_info['size']['height'] = rect.height()
            except:
                pass
            
            try:
                crs = map_item.crs()
                if crs.isValid():
                    map_info['crs'] = crs.authid()
            except:
                pass
            
            try:
                extent = map_item.extent()
                map_info['extent'] = f"X: {extent.xMinimum():.2f}-{extent.xMaximum():.2f}, Y: {extent.yMinimum():.2f}-{extent.yMaximum():.2f}"
            except:
                pass
            
            try:
                scale = map_item.scale()
                map_info['scale'] = f"1:{scale:.0f}" if scale > 0 else 'Unknown'
            except:
                pass
            
            try:
                map_info['locked'] = map_item.isLocked()
                map_info['visible'] = map_item.isVisible()
            except:
                pass
            
            # Check for Data-defined override for scale
            try:
                data_defined_properties = map_item.dataDefinedProperties()
                if data_defined_properties.isActive(QgsLayoutObject.MapScale):
                    map_info['has_data_defined_scale'] = True
                    map_info['scale'] += " (has Data-defined override)"
            except:
                pass
            
            if not map_info['name'] or map_info['name'] == 'Unknown':
                map_info['name'] = f"Map {len(self.map_items) + 1}"
                
            return map_info
            
        except Exception as e:
            print(f"Error parsing map item: {e}")
            return None

    def _display_results(self):
        if not self.map_items:
            self.maps_list.addItem("❌ No map items found in this layout.")
            return
            
        for i, map_info in enumerate(self.map_items):
            icon = "🔒" if map_info['locked'] else "🗺️"
            visibility = "👁️" if map_info['visible'] else "🙈"
            data_defined_icon = "⚙️" if map_info['has_data_defined_scale'] else ""
            
            item_text = f"{icon} {visibility} {data_defined_icon} {map_info['name']}"
            item = QListWidgetItem(item_text)
            item.setData(Qt.UserRole, map_info)
            self.maps_list.addItem(item)
            
        current_info = self.layout_info.toPlainText()
        current_info += f"\n✅ Map items found: {len(self.map_items)}"
        self.layout_info.setText(current_info)

    def on_map_selected(self):
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            self.apply_scale_btn.setEnabled(False)
            return
            
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            self.apply_scale_btn.setEnabled(False)
            return
            
        details = self._format_map_details(map_info)
        self.details_text.setText(details)
        self.apply_scale_btn.setEnabled(True)

    def _format_map_details(self, map_info):
        details = []
        details.append("📋 Map Item Details:")
        details.append("─" * 40)
        details.append(f"🔷 Name: {map_info['name']}")
        details.append(f"🔷 ID: {map_info['id']}")
        details.append(f"🔷 Type: {map_info['type']}")
        details.append("")
        
        details.append("📍 Position (pixels):")
        details.append(f"   X: {map_info['position']['x']:.1f}")
        details.append(f"   Y: {map_info['position']['y']:.1f}")
        details.append("")
        
        details.append("📏 Size (pixels):")
        details.append(f"   Width: {map_info['size']['width']:.1f}")
        details.append(f"   Height: {map_info['size']['height']:.1f}")
        details.append("")
        
        details.append("⚙️ Status:")
        details.append(f"   Locked: {'✅ Yes' if map_info['locked'] else '❌ No'}")
        details.append(f"   Visible: {'✅ Yes' if map_info['visible'] else '❌ No'}")
        details.append(f"   Data-defined scale: {'✅ Yes' if map_info['has_data_defined_scale'] else '❌ No'}")
        details.append("")
        
        details.append(f"🗺️ Coordinate System: {map_info['crs']}")
        details.append(f"📐 Current Scale: {map_info['scale']}")
        details.append(f"🔭 Extent: {map_info['extent']}")
        
        return "\n".join(details)

    def apply_scale_to_map(self):
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "Warning", "Please select a map item from the list first.")
            return
        
        scale_text = self.scale_combo.currentText().strip()
        
        if scale_text.startswith("1:"):
            scale_value = scale_text[2:]
        else:
            scale_value = scale_text
        
        try:
            scale_num = float(scale_value)
            if scale_num <= 0:
                raise ValueError("Scale must be greater than zero")
        except ValueError:
            QMessageBox.warning(self, "Error", "Invalid scale entered. Please enter a valid number.")
            return
        
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            return
        
        try:
            map_item = map_info['item_object']
            
            # Remove Data-defined override for scale if exists
            self._remove_data_defined_scale(map_item)
            
            map_item.setScale(scale_num)
            
            self.on_map_selected()
            
            QMessageBox.information(self, "Success", f"Scale 1:{scale_num:.0f} applied successfully")
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error applying scale:\n{str(e)}")

    def _remove_data_defined_scale(self, map_item):
        """Remove Data-defined override for scale"""
        try:
            data_defined_properties = map_item.dataDefinedProperties()
            if data_defined_properties.isActive(QgsLayoutObject.MapScale):
                data_defined_properties.setProperty(QgsLayoutObject.MapScale, QgsProperty())
                return True
        except Exception as e:
            print(f"Error removing Data-defined scale: {e}")
        return False

    def on_continue_clicked(self):
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "Warning", "Please select a map item from the list first.")
            return
        
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            return

        scale_text = self.scale_combo.currentText().strip()
        if scale_text.startswith("1:"):
            scale_value = scale_text[2:]
        else:
            scale_value = scale_text
        
        try:
            scale_num = float(scale_value)
            if scale_num <= 0:
                raise ValueError("Scale must be greater than zero")
        except ValueError:
            QMessageBox.warning(self, "Error", "Invalid scale entered. Please enter a valid number.")
            return

        self.selected_map_item = map_info
        self.selected_scale = scale_num

        self.hide()
        self.activate_point_selection()

    def activate_point_selection(self):
        try:
            self.original_map_tool = iface.mapCanvas().mapTool()
            
            self.map_tool = PointSelectionTool(iface.mapCanvas(), self.on_point_selected)
            iface.mapCanvas().setMapTool(self.map_tool)
            
            QMessageBox.information(None, "Select Point", 
                                  "Please select the desired point on the map.\n\n")
            
        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error activating point selection tool:\n{str(e)}")
            self.show()

    def on_point_selected(self, point):
        try:
            if self.map_tool:
                iface.mapCanvas().unsetMapTool(self.map_tool)
                self.map_tool = None
            
            if self.original_map_tool:
                iface.mapCanvas().setMapTool(self.original_map_tool)
            
            self.process_selected_point(point)
            
        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error processing selected point:\n{str(e)}")
            self.show()

    def process_selected_point(self, point):
        try:
            iface.openLayoutDesigner(self.current_layout)

            map_item = self.selected_map_item['item_object']

            # Remove Data-defined override for scale if exists
            self._remove_data_defined_scale(map_item)
            
            # Set new scale
            map_item.setScale(self.selected_scale)

            # Calculate new extent with center at selected point
            current_extent = map_item.extent()
            
            width = current_extent.width()
            height = current_extent.height()
            
            new_extent = QgsRectangle(
                point.x() - width / 2,
                point.y() - height / 2,
                point.x() + width / 2,
                point.y() + height / 2
            )
            
            map_item.setExtent(new_extent)

            self.current_layout.refresh()

            QMessageBox.information(None, "Success", 
                                  f"Settings applied successfully to map item.\n"
                                  f"Name: {self.selected_map_item['name']}\n"
                                  f"Scale: 1:{self.selected_scale:.0f}\n"
                                  f"New Center: X={point.x():.2f}, Y={point.y():.2f}")

            self.close()

        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error applying settings to map item:\n{str(e)}")
            self.show()

    def closeEvent(self, event):
        if self.map_tool and iface.mapCanvas().mapTool() == self.map_tool:
            iface.mapCanvas().unsetMapTool(self.map_tool)
        
        if self.original_map_tool:
            iface.mapCanvas().setMapTool(self.original_map_tool)
        
        event.accept()