# new_layout_dialog.py
# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QListWidget, QListWidgetItem,
    QPushButton, QMessageBox, QTextEdit, QGroupBox, QComboBox, QWidget, QLineEdit, QFileDialog,
    QScrollArea, QSizePolicy
)
from qgis.PyQt.QtCore import Qt, QPointF
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtXml import QDomDocument
from qgis.core import (
    QgsProject, QgsLayoutItemMap, QgsLayoutPoint, QgsLayoutSize, QgsUnitTypes, QgsRectangle,
    QgsProperty, QgsPrintLayout, QgsReadWriteContext, QgsCoordinateTransform,
    QgsLayoutManager, QgsApplication, QgsLayoutObject
)
from qgis.gui import QgsMapToolEmitPoint
try:
    from qgis.utils import iface
except ImportError:
    iface = None
import os
import traceback


class PointSelectionTool(QgsMapToolEmitPoint):
    def __init__(self, canvas, callback):
        QgsMapToolEmitPoint.__init__(self, canvas)
        self.callback = callback
        self.canvas = canvas

    def canvasPressEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        self.callback(point)
        
    def activate(self):
        super().activate()
        self.canvas.setCursor(Qt.CrossCursor)
        
    def deactivate(self):
        super().deactivate()
        self.canvas.unsetCursor()


class LayoutMapViewer(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Create New Layout from Template")
        
        # Set optimal size and resizable capability
        self.setMinimumSize(600, 500)
        self.resize(700, 600)
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        
        self.template_path = None
        self.map_items = []
        self.selected_map_item = None
        self.selected_scale = None
        self.map_tool = None
        self.original_map_tool = None
        self.layout_name = "New Layout"
        self.created_layout = None
        
        self.setup_ui()
        self.load_templates()

    def setup_ui(self):
        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(10, 10, 10, 10)
        
        scroll_widget = QWidget()
        scroll_layout = QVBoxLayout(scroll_widget)
        scroll_layout.setContentsMargins(0, 0, 0, 0)
        
        scroll_layout.addWidget(self._create_name_input_group())
        scroll_layout.addWidget(self._create_template_selection_group())
        scroll_layout.addWidget(self._create_template_info_group())
        scroll_layout.addWidget(self._create_maps_list_group())
        scroll_layout.addWidget(self._create_details_group())
        scroll_layout.addWidget(self._create_scale_selection_group())
        scroll_layout.addStretch(1)
        
        scroll_area = QScrollArea()
        scroll_area.setWidget(scroll_widget)
        scroll_area.setWidgetResizable(True)
        scroll_area.setMinimumHeight(400)
        
        main_layout.addWidget(scroll_area)
        main_layout.addWidget(self._create_buttons())
        
        self.setLayout(main_layout)

    def _create_name_input_group(self):
        group = QGroupBox("Layout Name")
        group.setMaximumHeight(80)
        layout = QVBoxLayout()
        
        name_layout = QHBoxLayout()
        name_layout.addWidget(QLabel("Layout Name:"))
        
        self.name_input = QLineEdit()
        self.name_input.setText("New Layout")
        self.name_input.textChanged.connect(self.on_name_changed)
        name_layout.addWidget(self.name_input, 1)
        
        layout.addLayout(name_layout)
        group.setLayout(layout)
        return group

    def _create_template_selection_group(self):
        group = QGroupBox("Select Template")
        group.setMaximumHeight(100)
        layout = QVBoxLayout()
        
        selection_layout = QHBoxLayout()
        selection_layout.addWidget(QLabel("Template:"))
        
        self.template_combo = QComboBox()
        self.template_combo.setMaximumHeight(25)
        self.template_combo.currentIndexChanged.connect(self.on_template_selected)
        selection_layout.addWidget(self.template_combo, 1)
        
        button_layout = QVBoxLayout()
        self.browse_btn = QPushButton("Browse...")
        self.browse_btn.setMaximumHeight(25)
        self.browse_btn.clicked.connect(self.browse_template)
        button_layout.addWidget(self.browse_btn)
        
        self.refresh_btn = QPushButton("Refresh List")
        self.refresh_btn.setMaximumHeight(25)
        self.refresh_btn.clicked.connect(self.load_templates)
        button_layout.addWidget(self.refresh_btn)
        
        selection_layout.addLayout(button_layout)
        layout.addLayout(selection_layout)
        group.setLayout(layout)
        return group

    def _create_template_info_group(self):
        group = QGroupBox("Template Information")
        group.setMaximumHeight(100)
        layout = QVBoxLayout()
        
        self.template_info = QTextEdit()
        self.template_info.setMaximumHeight(80)
        self.template_info.setReadOnly(True)
        self.template_info.setPlaceholderText("Template information will be displayed after selection...")
        
        layout.addWidget(self.template_info)
        group.setLayout(layout)
        return group

    def _create_maps_list_group(self):
        group = QGroupBox("Map Items Found in Template")
        group.setMaximumHeight(150)
        layout = QVBoxLayout()
        
        self.maps_list = QListWidget()
        self.maps_list.setMaximumHeight(100)
        self.maps_list.itemSelectionChanged.connect(self.on_map_selected)
        layout.addWidget(self.maps_list)
        
        group.setLayout(layout)
        return group

    def _create_details_group(self):
        group = QGroupBox("Selected Map Item Details")
        group.setMaximumHeight(150)
        layout = QVBoxLayout()
        
        self.details_text = QTextEdit()
        self.details_text.setMaximumHeight(100)
        self.details_text.setReadOnly(True)
        self.details_text.setPlaceholderText("Select a map item from the list...")
        
        layout.addWidget(self.details_text)
        group.setLayout(layout)
        return group

    def _create_scale_selection_group(self):
        group = QGroupBox("Select Scale")
        group.setMaximumHeight(80)
        layout = QVBoxLayout()
        
        scale_layout = QHBoxLayout()
        scale_layout.addWidget(QLabel("Map Scale:"))
        
        self.scale_combo = QComboBox()
        self.scale_combo.setEditable(True)
        self.scale_combo.setMaximumHeight(25)
        self.scale_combo.setToolTip("You can select from the list or enter a custom scale (example: 1000)")
        
        common_scales = [
            "100", "200", "500", "1000", "2000", "5000", 
            "10000", "25000", "50000", "100000", "250000", 
            "500000", "1000000", "2000000", "5000000"
        ]
        
        for scale in common_scales:
            self.scale_combo.addItem(f"1:{scale}")
        
        self.scale_combo.setCurrentText("1:1000")
        
        scale_layout.addWidget(self.scale_combo, 1)
        
        self.apply_scale_btn = QPushButton("Apply Scale")
        self.apply_scale_btn.setMaximumHeight(25)
        self.apply_scale_btn.clicked.connect(self.apply_scale_to_map)
        self.apply_scale_btn.setEnabled(False)
        scale_layout.addWidget(self.apply_scale_btn)
        
        layout.addLayout(scale_layout)
        group.setLayout(layout)
        return group

    def _create_buttons(self):
        button_layout = QHBoxLayout()
        
        self.continue_btn = QPushButton("Continue and Select Location")
        self.continue_btn.setMinimumHeight(35)
        self.continue_btn.clicked.connect(self.on_continue_clicked)
        self.continue_btn.setEnabled(False)
        
        self.close_btn = QPushButton("Close")
        self.close_btn.setMinimumHeight(35)
        self.close_btn.clicked.connect(self.close)
        
        button_layout.addWidget(self.continue_btn)
        button_layout.addStretch()
        button_layout.addWidget(self.close_btn)
        
        widget = QWidget()
        widget.setLayout(button_layout)
        return widget

    def on_name_changed(self):
        self.layout_name = self.name_input.text().strip()
        if not self.layout_name:
            self.layout_name = "New Layout"

    def load_templates(self):
        self.template_combo.clear()
        self.maps_list.clear()
        self.details_text.clear()
        self.template_info.clear()
        self.map_items = []
        self.template_path = None
        
        self.continue_btn.setEnabled(False)
        self.apply_scale_btn.setEnabled(False)
        
        # Common QGIS template paths
        template_paths = [
            os.path.join(os.path.expanduser("~"), ".local", "share", "QGIS", "QGIS3", "profiles", "default", "composer_templates"),
            os.path.join(os.path.expanduser("~"), ".qgis3", "composer_templates"),
            os.path.join(os.path.expanduser("~"), "AppData", "Roaming", "QGIS", "QGIS3", "profiles", "default", "composer_templates"),
        ]
        
        templates_found = 0
        
        for template_path in template_paths:
            if os.path.exists(template_path):
                for file in os.listdir(template_path):
                    if file.lower().endswith('.qpt'):
                        full_path = os.path.join(template_path, file)
                        display_name = f"📄 {file}"
                        self.template_combo.addItem(display_name, full_path)
                        templates_found += 1
        
        if templates_found == 0:
            self.template_combo.addItem("No templates found", None)
            self.template_combo.setEnabled(False)
            self.template_info.setText("❌ No templates found in standard paths.")
        else:
            self.template_combo.setEnabled(True)
            self.template_info.setText(f"✅ Templates found: {templates_found}")
            
        if self.template_combo.count() > 0:
            self.template_combo.setCurrentIndex(0)

    def browse_template(self):
        file_path, _ = QFileDialog.getOpenFileName(
            self, 
            "Select Template File", 
            "", 
            "QGIS Template Files (*.qpt);;All Files (*)"
        )
        
        if file_path:
            display_name = f"📄 {os.path.basename(file_path)}"
            self.template_combo.addItem(display_name, file_path)
            self.template_combo.setCurrentIndex(self.template_combo.count() - 1)

    def on_template_selected(self, index):
        if index < 0:
            return
            
        self.template_path = self.template_combo.currentData()
        if not self.template_path or not os.path.exists(self.template_path):
            self.template_info.setText("❌ Invalid template selected.")
            self.maps_list.clear()
            self.details_text.clear()
            self.continue_btn.setEnabled(False)
            self.apply_scale_btn.setEnabled(False)
            return
            
        self.analyze_template()

    def analyze_template(self):
        if not self.template_path:
            return
            
        try:
            self.map_items = []
            self.maps_list.clear()
            self.details_text.clear()
            
            template_info = self._get_template_info()
            self.template_info.setText(template_info)
            
            self._extract_map_items()
            self._display_results()
            
            has_maps = len(self.map_items) > 0
            self.continue_btn.setEnabled(has_maps)
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error analyzing template:\n{str(e)}")

    def _get_template_info(self):
        if not self.template_path:
            return "No template selected"
        
        try:
            with open(self.template_path, 'r', encoding='utf-8') as file:
                template_content = file.read()
            
            doc = QDomDocument()
            if not doc.setContent(template_content):
                return "❌ Template file is not valid."
                
            layout_elem = doc.documentElement()
            layout_name = layout_elem.attribute("name", "Unknown")
            
            page_collection = layout_elem.firstChildElement("PageCollection")
            page_size = "Unknown"
            if not page_collection.isNull():
                page = page_collection.firstChildElement("ComposerPage")
                if not page.isNull():
                    width = page.attribute("width", "Unknown")
                    height = page.attribute("height", "Unknown")
                    page_size = f"{width} × {height} mm"
            
            info_lines = []
            info_lines.append(f"📄 Template Name: {layout_name}")
            info_lines.append(f"📏 Page Size: {page_size}")
            info_lines.append(f"📁 File Path: {os.path.basename(self.template_path)}")
            
            return "\n".join(info_lines)
            
        except Exception as e:
            return f"❌ Error reading template: {str(e)}"

    def _extract_map_items(self):
        if not self.template_path:
            return
            
        try:
            with open(self.template_path, 'r', encoding='utf-8') as file:
                template_content = file.read()
            
            doc = QDomDocument()
            if not doc.setContent(template_content):
                return
                
            items = doc.elementsByTagName("LayoutItem")
            
            for i in range(items.count()):
                item = items.at(i)
                if item.isElement():
                    elem = item.toElement()
                    item_type = elem.attribute("type", "")
                    
                    if item_type == "65639":  # QgsLayoutItemMap
                        map_info = {
                            'id': elem.attribute("id", ""),
                            'name': elem.attribute("name", f"Map {len(self.map_items) + 1}"),
                            'position': {'x': 0, 'y': 0},
                            'size': {'width': 0, 'height': 0},
                            'type': 'QgsLayoutItemMap',
                            'crs': 'Unknown',
                            'extent': 'Unknown',
                            'scale': 'Unknown',
                            'locked': False,
                            'visible': True,
                            'has_data_defined_scale': False
                        }
                        
                        # Position
                        pos_elem = elem.firstChildElement("Position")
                        if not pos_elem.isNull():
                            try:
                                map_info['position']['x'] = float(pos_elem.attribute("x", "0"))
                                map_info['position']['y'] = float(pos_elem.attribute("y", "0"))
                            except Exception:
                                pass
                        
                        # Size
                        size_elem = elem.firstChildElement("Size")
                        if not size_elem.isNull():
                            try:
                                map_info['size']['width'] = float(size_elem.attribute("width", "0"))
                                map_info['size']['height'] = float(size_elem.attribute("height", "0"))
                            except Exception:
                                pass
                        
                        # Try to detect data-defined for scale
                        dd_found = False
                        data_elems = elem.elementsByTagName("DataDefinedProperties")
                        if data_elems.count() > 0:
                            for j in range(data_elems.count()):
                                dd = data_elems.at(j).toElement()
                                if dd.toElement().text().find('scale') != -1:
                                    dd_found = True
                                    break
                        map_info['has_data_defined_scale'] = dd_found
                        
                        self.map_items.append(map_info)
                        
        except Exception as e:
            print(f"Error extracting map items: {e}")
            traceback.print_exc()

    def _display_results(self):
        if not self.map_items:
            self.maps_list.addItem("❌ No map items found in this template.")
            return
            
        for i, map_info in enumerate(self.map_items):
            icon = "🗺️"
            dd_flag = " (data-defined)" if map_info.get('has_data_defined_scale') else ""
            item_text = f"{icon} {map_info['name']}{dd_flag}"
            item = QListWidgetItem(item_text)
            item.setData(Qt.UserRole, map_info)
            self.maps_list.addItem(item)
            
        current_info = self.template_info.toPlainText()
        current_info += f"\n✅ Map items found: {len(self.map_items)}"
        self.template_info.setText(current_info)

    def on_map_selected(self):
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            self.apply_scale_btn.setEnabled(False)
            return
            
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            self.apply_scale_btn.setEnabled(False)
            return
            
        details = self._format_map_details(map_info)
        self.details_text.setText(details)
        self.apply_scale_btn.setEnabled(True)

    def _format_map_details(self, map_info):
        details = []
        details.append("📋 Map Item Details:")
        details.append("─" * 40)
        details.append(f"🔷 Name: {map_info['name']}")
        details.append(f"🔷 ID: {map_info['id']}")
        details.append(f"🔷 Type: {map_info['type']}")
        details.append("")
        
        details.append("📍 Position (millimeters):")
        details.append(f"   X: {map_info['position']['x']:.1f}")
        details.append(f"   Y: {map_info['position']['y']:.1f}")
        details.append("")
        
        details.append("📏 Size (millimeters):")
        details.append(f"   Width: {map_info['size']['width']:.1f}")
        details.append(f"   Height: {map_info['size']['height']:.1f}")
        details.append("")
        
        if map_info.get('has_data_defined_scale'):
            details.append("⚠️ This item may use data-defined override for scale.")
        
        return "\n".join(details)

    def apply_scale_to_map(self):
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "Warning", "Please select a map item from the list first.")
            return
        
        scale_text = self.scale_combo.currentText().strip()
        
        if scale_text.startswith("1:"):
            scale_value = scale_text[2:]
        else:
            scale_value = scale_text
        
        try:
            scale_num = float(scale_value)
            if scale_num <= 0:
                raise ValueError("Scale must be greater than zero")
        except ValueError:
            QMessageBox.warning(self, "Error", "Invalid scale entered. Please enter a valid number.")
            return
        
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            return
        
        try:
            map_info['scale'] = f"1:{scale_num:.0f}"
            self.on_map_selected()
            
            QMessageBox.information(self, "Success", f"Scale 1:{scale_num:.0f} saved for application in new layout")
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error saving scale:\n{str(e)}")

    def on_continue_clicked(self):
        if iface is None:
            QMessageBox.critical(self, "Error", "This plugin can only run in QGIS environment")
            return
            
        selected_items = self.maps_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "Warning", "Please select a map item from the list first.")
            return
        
        map_info = selected_items[0].data(Qt.UserRole)
        if not map_info:
            return

        scale_text = self.scale_combo.currentText().strip()
        if scale_text.startswith("1:"):
            scale_value = scale_text[2:]
        else:
            scale_value = scale_text
        
        try:
            scale_num = float(scale_value)
            if scale_num <= 0:
                raise ValueError("Scale must be greater than zero")
        except ValueError:
            QMessageBox.warning(self, "Error", "Invalid scale entered. Please enter a valid number.")
            return

        if not self.template_path:
            QMessageBox.warning(self, "Error", "Please select a template.")
            return

        self.selected_map_item = map_info
        self.selected_scale = scale_num
        self.layout_name = self.name_input.text().strip()
        if not self.layout_name:
            self.layout_name = "New Layout"

        self.hide()
        self.activate_point_selection()

    def activate_point_selection(self):
        try:
            if iface is None:
                QMessageBox.critical(self, "Error", "This plugin can only run in QGIS environment")
                self.show()
                return
                
            self.original_map_tool = iface.mapCanvas().mapTool()
            
            self.map_tool = PointSelectionTool(iface.mapCanvas(), self.on_point_selected)
            iface.mapCanvas().setMapTool(self.map_tool)
            
            QMessageBox.information(None, "Select Point", 
                                  "Please select the desired center point on the map.\n\n")
            
        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error activating point selection tool:\n{str(e)}")
            self.show()

    def on_point_selected(self, point):
        try:
            if self.map_tool:
                iface.mapCanvas().unsetMapTool(self.map_tool)
                self.map_tool = None
            
            if self.original_map_tool:
                iface.mapCanvas().setMapTool(self.original_map_tool)
            
            self.create_layout_with_point(point)
            
        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error processing selected point:\n{str(e)}")
            self.show()

    def generate_unique_layout_name(self, base_name):
        """Generate a unique layout name by adding numbers"""
        layout_manager = QgsProject.instance().layoutManager()
        existing_layouts = layout_manager.layouts()
        
        existing_names = [layout.name() for layout in existing_layouts]
        
        if base_name not in existing_names:
            return base_name
        
        counter = 1
        while True:
            new_name = f"{base_name} ({counter})"
            if new_name not in existing_names:
                return new_name
            counter += 1

    def _find_map_item_in_layout(self, created_layout):
        """Safely search for map item by checking object status"""
        try:
            # Check if layout object is still valid
            if created_layout is None or not hasattr(created_layout, 'items'):
                return None
                
            target_id = self.selected_map_item.get('id') if self.selected_map_item else None
            map_item = None
            
            # Use try-except for each item individually
            items = []
            try:
                items = created_layout.items()
            except RuntimeError:
                return None
                
            for item in items:
                try:
                    # Check each item's status before use
                    if item is None:
                        continue
                        
                    if isinstance(item, QgsLayoutItemMap):
                        # If we have target_id, try to match
                        if target_id and hasattr(item, 'id'):
                            try:
                                if item.id() == target_id:
                                    return item
                            except RuntimeError:
                                continue
                        
                        # Otherwise store the first map item
                        if not map_item:
                            map_item = item
                except RuntimeError:
                    continue
                    
            return map_item
            
        except Exception as e:
            print(f"Error searching for map item: {e}")
            return None

    def _clear_data_defined_scale(self, map_item):
        """Remove data-defined override for scale"""
        try:
            # Check object status before use
            if map_item is None:
                return
                
            try:
                map_item.setLocked(False)
            except Exception:
                pass

            # Different methods to remove data-defined properties
            methods_tried = False
            
            # Method 1: Use dataDefinedProperties
            try:
                if hasattr(map_item, 'dataDefinedProperties'):
                    ddp = map_item.dataDefinedProperties()
                    if ddp is not None:
                        if hasattr(ddp, 'hasProperty') and ddp.hasProperty(QgsLayoutObject.MapScale):
                            if hasattr(ddp, 'removeProperty'):
                                ddp.removeProperty(QgsLayoutObject.MapScale)
                                methods_tried = True
            except Exception:
                pass
                
            # Method 2: Set empty property
            try:
                if hasattr(map_item, 'dataDefinedProperties'):
                    ddp = map_item.dataDefinedProperties()
                    if ddp is not None and hasattr(ddp, 'setProperty'):
                        ddp.setProperty(QgsLayoutObject.MapScale, QgsProperty())
                        methods_tried = True
            except Exception:
                pass
                
            # Method 3: Use direct methods if available
            try:
                if hasattr(map_item, 'setDataDefinedProperty'):
                    map_item.setDataDefinedProperty(QgsLayoutObject.MapScale, None)
                    methods_tried = True
            except Exception:
                pass

        except Exception as e:
            print(f"Error clearing data-defined scale: {e}")

    def create_layout_with_point(self, center_point):
        """Create new layout from template and apply settings"""
        try:
            # Read template
            with open(self.template_path, 'r', encoding='utf-8') as file:
                template_content = file.read()
            
            doc = QDomDocument()
            if not doc.setContent(template_content):
                QMessageBox.critical(None, "Error", "Template file is not valid.")
                self.show()
                return
            
            # Create new layout
            project = QgsProject.instance()
            layout_manager = project.layoutManager()
            
            # Generate unique name - always use the name entered by user
            unique_layout_name = self.generate_unique_layout_name(self.layout_name)
            
            # If name changed, notify user
            if unique_layout_name != self.layout_name:
                user_choice = QMessageBox.question(
                    None, 
                    "Duplicate Name", 
                    f"A layout named '{self.layout_name}' already exists.\nWould you like to create the layout with name '{unique_layout_name}'?",
                    QMessageBox.Yes | QMessageBox.No,
                    QMessageBox.Yes
                )
                
                if user_choice == QMessageBox.No:
                    self.show()
                    return
            
            # Create layout
            layout = QgsPrintLayout(project)
            
            # Load template
            context = QgsReadWriteContext()
            success = layout.loadFromTemplate(doc, context)
            
            if not success:
                QMessageBox.warning(None, "Warning", "Some items may not have loaded correctly.")

            # Always set layout name to user's chosen name
            # This line is very important and should be after loadFromTemplate
            layout.setName(unique_layout_name)
            
            # Add layout to project
            layout_manager.addLayout(layout)
            
            # Find target map item
            map_item = self._find_map_item_in_layout(layout)
            
            if not map_item:
                QMessageBox.warning(None, "Error", "No map item found in template.")
                try:
                    layout_manager.removeLayout(layout)
                except Exception:
                    pass
                self.show()
                return

            # Remove data-defined scale override
            self._clear_data_defined_scale(map_item)

            # Apply scale
            try:
                if hasattr(map_item, 'setScale'):
                    map_item.setScale(self.selected_scale)
            except Exception as e:
                print(f"Error applying scale: {e}")

            # Set new center with coordinate transformation
            try:
                if hasattr(map_item, 'crs'):
                    map_crs = map_item.crs()
                    canvas_crs = iface.mapCanvas().mapSettings().destinationCrs()
                    
                    if map_crs != canvas_crs and map_crs.isValid() and canvas_crs.isValid():
                        transform = QgsCoordinateTransform(canvas_crs, map_crs, project.transformContext())
                        try:
                            center_point = transform.transform(center_point)
                        except Exception as e:
                            print(f"Error in coordinate transformation: {e}")
            except Exception as e:
                print(f"Error checking CRS: {e}")

            # Set map center
            try:
                if hasattr(map_item, 'setCenter'):
                    map_item.setCenter(center_point)
                elif hasattr(map_item, 'setExtent'):
                    # Alternative method: set extent
                    current_extent = map_item.extent()
                    width = current_extent.width()
                    height = current_extent.height()
                    
                    new_extent = QgsRectangle(
                        center_point.x() - width / 2,
                        center_point.y() - height / 2,
                        center_point.x() + width / 2,
                        center_point.y() + height / 2
                    )
                    map_item.setExtent(new_extent)
            except Exception as e:
                print(f"Error setting map center: {e}")

            # Open Layout Designer
            try:
                iface.openLayoutDesigner(layout)
            except Exception as e:
                print(f"Error opening layout designer: {e}")
            
            QMessageBox.information(None, "Success", 
                                  f"Layout '{unique_layout_name}' created successfully!\n"
                                  f"📍 Center location: X={center_point.x():.2f}, Y={center_point.y():.2f}\n"
                                  f"📐 Scale: 1:{self.selected_scale:.0f}")

            self.close()

        except Exception as e:
            QMessageBox.critical(None, "Error", f"Error creating layout:\n{str(e)}")
            traceback.print_exc()
            self.show()

    def closeEvent(self, event):
        if iface and self.map_tool and iface.mapCanvas().mapTool() == self.map_tool:
            iface.mapCanvas().unsetMapTool(self.map_tool)
        
        if iface and self.original_map_tool:
            iface.mapCanvas().setMapTool(self.original_map_tool)
        
        event.accept()