from qgis.PyQt.QtWidgets import (QWidget, QVBoxLayout, QHBoxLayout, QLabel, 
                                 QLineEdit, QPushButton, QMessageBox, QDockWidget)
from qgis.PyQt.QtGui import QColor
from qgis.gui import QgsRubberBand
from qgis.core import (QgsPointXY, QgsProject, QgsRectangle, QgsFeature, QgsGeometry, QgsWkbTypes,
                       QgsCoordinateReferenceSystem, QgsCoordinateTransform)

class LatLongDock(QDockWidget):
    def __init__(self, iface, parent=None):
        super(LatLongDock, self).__init__(parent)
        self.iface = iface 
        self.setWindowTitle("Lat/Long Digitizer")
        
        # Internal state
        self.vertices = []
        self.crs_input = QgsCoordinateReferenceSystem("EPSG:4326")
        
        # --- Rubber Band for Visual Feedback ---
        self.rubber_band = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PolygonGeometry)
        self.rubber_band.setColor(QColor(255, 0, 0, 100)) # Semi-transparent Red
        self.rubber_band.setWidth(2)

        # --- UI Setup ---
        self.main_widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel("Add Vertex by Coordinate (Lat, Lon):"))
        self.coord_input = QLineEdit()
        self.coord_input.setPlaceholderText("e.g., 40.7128, -74.0060")
        layout.addWidget(self.coord_input)
        
        self.btn_add_point = QPushButton("Add Point and Zoom")
        self.btn_add_point.setStyleSheet("font-weight: bold; padding: 5px;")
        layout.addWidget(self.btn_add_point)
        
        button_layout = QHBoxLayout()
        self.btn_undo_point = QPushButton("Undo Last Point")
        self.btn_undo_point.setStyleSheet("background-color: #ffc9c9;") 
        button_layout.addWidget(self.btn_undo_point)
        
        self.btn_close_polygon = QPushButton("Finish Polygon")
        self.btn_close_polygon.setStyleSheet("background-color: #a8ffb0;") 
        button_layout.addWidget(self.btn_close_polygon)
        
        layout.addLayout(button_layout)
        layout.addStretch()
        self.main_widget.setLayout(layout)
        self.setWidget(self.main_widget)
        
        # --- Connections ---
        self.btn_add_point.clicked.connect(self.add_point)
        self.btn_undo_point.clicked.connect(self.undo_point)
        self.btn_close_polygon.clicked.connect(self.finish_polygon)

    def update_rubber_band(self):
        """Redraws the temporary visual polygon on the map."""
        self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
        if len(self.vertices) > 0:
            for pt in self.vertices:
                self.rubber_band.addPoint(pt, False)
            self.rubber_band.show()

    def zoom_to_point(self, point):
        canvas = self.iface.mapCanvas()
        extent = QgsRectangle(point.x() - 0.001, point.y() - 0.001, 
                              point.x() + 0.001, point.y() + 0.001)
        canvas.setExtent(extent)
        canvas.refresh()

    def add_point(self):
        layer = self.iface.activeLayer()
        if not layer or not layer.isEditable():
            QMessageBox.warning(self, "Error", "Please enable editing (Pencil icon) first.")
            return

        lat, lon = self.parse_coords()
        if lat is None: return
            
        try:
            transform = QgsCoordinateTransform(self.crs_input, layer.crs(), QgsProject.instance())
            point = transform.transform(QgsPointXY(lon, lat))
            
            self.vertices.append(point)
            self.update_rubber_band()
            self.zoom_to_point(point)
            
            self.coord_input.clear()
            self.coord_input.setFocus()
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Could not add point: {e}")

    def undo_point(self):
        if self.vertices:
            self.vertices.pop()
            self.update_rubber_band()
        else:
            QMessageBox.information(self, "Undo", "No vertices to remove.")

    def finish_polygon(self):
        layer = self.iface.activeLayer()
        if not layer: return

        if len(self.vertices) < 3:
            QMessageBox.warning(self, "Error", "You need at least 3 points to make a polygon.")
            return
            
        try:
            polygon_ring = list(self.vertices)
            polygon_ring.append(self.vertices[0])
            
            geometry = QgsGeometry.fromPolygonXY([polygon_ring])
            feature = QgsFeature(layer.fields())
            feature.setGeometry(geometry)
            
            layer.addFeature(feature)
            self.iface.mapCanvas().refresh()
            
            # Reset visual and list
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            self.vertices = []
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Could not create polygon: {e}")

    def parse_coords(self):
        text = self.coord_input.text()
        try:
            parts = text.split(',')
            return float(parts[0].strip()), float(parts[1].strip())
        except:
            QMessageBox.warning(self, "Input Error", "Please use 'Lat, Lon' format.")
            return None, None